// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000, 2001 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

///////////////////////////////////////////////////////////////
// map.h
//
// Changes:
//  04-12-2000 - created
//  07-11-2001 - Unit and Building are now subclasses of the
//               new abstract MapObject class
///////////////////////////////////////////////////////////////

#ifndef _INCLUDE_MAP_H
#define _INCLUDE_MAP_H

#include "hexsup.h"
#include "rect.h"
#include "unit.h"
#include "building.h"
#include "list.h"
#include "misc.h"

// terrain types
#define TT_ROAD			0x0001
#define TT_PLAINS		0x0002
#define TT_FOREST		0x0004
#define TT_SWAMP		0x0008
#define TT_MOUNTAINS		0x0010
#define TT_WATER_SHALLOW	0x0020
#define TT_WATER_DEEP		0x0040
#define TT_BARRICADES		0x0080

#define TT_ENTRANCE		0x4000	// entrance to a building (also road)
#define TT_KEEP_OFF		0x8000	// no unit allowed to enter this (e.g. buildings)
					// the entrance to a building is of type (TT_ROAD|TT_ENTRANCE)


#define MCOST_STD	3	// cost to cross a plains hex for ground units
#define MCOST_MIN	2	// cost to cross a road hex for ground units
				// or any hex type for airborne units
#define MCOST_UNIT	30	// theoretical cost to cross a hex occupied by another unit
				// must be higher than the maximum unit speed

// these are the standard terrain types
#define TILE_PLAINS		0
#define TILE_FOREST		10
#define TILE_MOUNTAINS		9
#define TILE_SWAMP		36
#define TILE_ROAD		1
#define TILE_WATER_SHALLOW	18
#define TILE_WATER_DEEP		19


// definitions for the path finding algorithm
#define PATH_BEST	1
#define PATH_GOOD	4
#define PATH_FAST	10

class PathNode : public Node {
public:
  short x;
  short y;
  unsigned short eta;   // estimated time of arrival
  short cost;           // travelling cost so far
};

// terrain type definitions
class TerrainType {
public:
  int Load( SDL_RWops *file );

  unsigned short tt_type;       // one (or more) of the TT_ defined above
  unsigned short tt_image;
  char tt_att_mod;              // attack modifier (-20% to +20%)
  char tt_def_mod;              // defence modifier (-20% to +20%)
  char tt_move;                 // cost to cross hex type
  char reserved;
  unsigned long tt_color;       // color shown on the overview map display
};


class Map {
public:
  Map( void );
  ~Map( void );

  int Load( SDL_RWops *file );
  int Save( SDL_RWops *file ) const;
  int LoadTerrainTypes( SDL_RWops *file, unsigned short n );

  unsigned short MapWidth( void ) const { return m_w; }
  unsigned short MapHeight( void ) const { return m_h; }

  short AttackMod( short x, short y ) const
                 { return( HexType(x,y)->tt_att_mod ); }
  short DefenceMod( short x, short y ) const
                 { return( HexType(x,y)->tt_def_mod ); }
  unsigned long HexColor( unsigned short xy ) const
                 { return( m_terrains[m_data[xy]].tt_color ); }
  short HexImage( short x, short y ) const
                 { return( HexType(x,y)->tt_image ); }
  TerrainType *HexType( short x, short y ) const
                 { return( &m_terrains[m_data[XY2Index(x,y)]] ); }
  char MoveCost( short x, short y ) const
                 { return( HexType(x,y)->tt_move ); }
  unsigned short TerrainTypes( short x, short y ) const
                 { return( HexType(x,y)->tt_type ); }

  Unit *GetUnit( short x, short y ) const;
  void SetUnit( Unit *u, short x, short y );
  Building *GetBuilding( short x, short y ) const;
  void SetBuilding( Building *b, short x, short y )
               { m_objects[XY2Index(x,y)] = b; }

  void SetHexType( short x, short y, short type )
               { m_data[XY2Index(x,y)] = type; }

  void ClearPath( void ) const;
  int FindPath( const Unit *u, short dx, short dy, short qual,
                unsigned short deviate ) const;
  short *GetPath( void ) const { return m_path; }

  bool IsWater( short x, short y ) const
        { return (HexType(x,y)->tt_type & (TT_WATER_SHALLOW|TT_WATER_DEEP)) != 0; }
  bool IsGround( short x, short y ) const { return !IsWater( x, y ); }
  bool IsBuilding( short x, short y ) const
        { return (HexType(x,y)->tt_type & TT_ENTRANCE) != 0; }

  short GetNeighbors( short x, short y, Point *parray ) const;
  int XY2Index( short x, short y ) const { return( y * m_w + x ); }
  int Dir2XY( short x, short y, Direction dir, Point &dest ) const;
  bool Contains( short x, short y ) const;

protected:
  unsigned short m_w;
  unsigned short m_h;

  short *m_data;
  MapObject **m_objects;
  TerrainType *m_terrains;
  short *m_path;
};

#endif	/* _INCLUDE_MAP_H */

