
// Copyright (C) 2000, 2001 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////////
// surface.cpp
//
// History:
//  28-10-2000 - created
//  22-04-2001 - removed CopyPalette()
//  30-05-2001 - removed CopyToDisplay()
//  03-06-2001 - added SetClipRect() and GetClipRect()
//  28-07-2001 - added Colorize()
//  11-08-2001 - added FillPattern()
//  13-08-2001 - added FillRectAlpha()
////////////////////////////////////////////////////////////////////////////

#include "SDL_endian.h"

#include "surface.h"
#include "misc.h"
#include "globals.h"

////////////////////////////////////////////////////////////////////////
// NAME       : Surface::~Surface
// DESCRIPTION: Free the memory used by this surface.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

Surface::~Surface( void ) {
  if ( s_surface ) SDL_FreeSurface( s_surface );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Surface::Create
// DESCRIPTION: Allocate a buffer for this surface.
// PARAMETERS : w     - surface width
//              h     - surface height
//              bpp   - bits per pixel
//              flags - surface flags (see SDL_video.h for details)
// RETURNS    : 0 on success, -1 on error
//
// HISTORY
//  22-04-2001 - added bpp parameter
////////////////////////////////////////////////////////////////////////

int Surface::Create( short w, short h, int bpp, unsigned long flags ) {
  if ( s_surface ) SDL_FreeSurface( s_surface );
  s_surface = SDL_CreateRGBSurface( flags, w, h, bpp, 0, 0, 0, 0 );
  if ( s_surface == 0 ) return -1;
  this->w = s_surface->w;
  this->h = s_surface->h;
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Surface::SetColorKey
// DESCRIPTION: Set transparent color for this surface.
// PARAMETERS : col - transparent color
// RETURNS    : 0 on success, -1 on error
//
// HISTORY
////////////////////////////////////////////////////////////////////////

int Surface::SetColorKey( const Color &col ) {
  if ( s_surface == 0 ) return -1;
  if ( SDL_SetColorKey( s_surface, SDL_SRCCOLORKEY, MapRGB( col ) ) )
    return -1;
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Surface::DisplayFormat
// DESCRIPTION: Convert surface format to display format.
// PARAMETERS : -
// RETURNS    : 0 on success, -1 on error
//
// HISTORY
////////////////////////////////////////////////////////////////////////

int Surface::DisplayFormat( void ) {
  SDL_Surface *new_s;

  if ( s_surface == 0 ) return -1;
  if ( (new_s = SDL_DisplayFormat( s_surface )) == 0 ) return -1;
  SDL_FreeSurface( s_surface );
  s_surface = new_s;
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Surface::Blit
// DESCRIPTION: Blit parts of the surface onto another surface.
// PARAMETERS : dest - destination surface
//              from - source rectangle
//              dx   - left edge of destination rectangle
//              dy   - top edge of destination rectangle
// RETURNS    : 0 on success, -1 on error
//
// HISTORY
//   23-06-2001 - check for zero-sized rect before blitting
////////////////////////////////////////////////////////////////////////

int Surface::Blit( Surface *dest, const Rect &from, short dx, short dy ) {
  if ( (s_surface == 0) || (from.w == 0) || (from.h == 0) ) return -1;

  SDL_Rect src, dst;

  src.x = from.x;
  src.y = from.y;
  src.w = from.w;
  src.h = from.h;

  dst.x = dx;
  dst.y = dy;
  dst.w = from.w;
  dst.h = from.h;
  if ( SDL_BlitSurface( s_surface, &src, dest->s_surface, &dst ) ) return -1;
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Surface::LowerBlit
// DESCRIPTION: This function works exactly like Surface::Blit() but it
//              calls SDL_LowerBlit() instead of SDL_BitSurface().
//              SDL_LowerBlit does not clip the blitted area to the
//              surface, so it's faster, but you must make sure the
//              blitting area does not extend beyond the surface.
// PARAMETERS : dest - destination surface
//              from - source rectangle
//              dx   - left edge of destination rectangle
//              dy   - top edge of destination rectangle
// RETURNS    : 0 on success, -1 on error
//
// HISTORY
////////////////////////////////////////////////////////////////////////

int Surface::LowerBlit( Surface *dest, const Rect &from, short dx, short dy ) {
  if ( s_surface == 0 ) return -1;

  SDL_Rect src, dst;

  src.x = from.x;
  src.y = from.y;
  src.w = from.w;
  src.h = from.h;

  dst.x = dx;
  dst.y = dy;
  dst.w = from.w;
  dst.h = from.h;
  if ( SDL_LowerBlit( s_surface, &src, dest->s_surface, &dst ) ) return -1;
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Surface::SetClipRect
// DESCRIPTION: Set the clipping rectangle for the surface.
// PARAMETERS : clip - clipping rectangle
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Surface::SetClipRect( const Rect &clip ) const {
  SDL_Rect rect;
  rect.x = clip.LeftEdge();
  rect.y = clip.TopEdge();
  rect.w = clip.Width();
  rect.h = clip.Height();

  SDL_SetClipRect( s_surface, &rect );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Surface::GetClipRect
// DESCRIPTION: Get the current clipping rectangle for the surface.
// PARAMETERS : clip - rect to hold the current clipping rectangle
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Surface::GetClipRect( Rect &clip ) const {
  SDL_Rect rect;
  SDL_GetClipRect( s_surface, &rect );
  clip = Rect( rect.x, rect.y, rect.w, rect.h );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Surface::FillRect
// DESCRIPTION: Draw a solid rectangle.
// PARAMETERS : x   - left edge of rectangle
//              y   - top edge of rectangle
//              w   - width of rectangle
//              h   - height of rectangle
//              col - color of rectangle
// RETURNS    : 0 on success, -1 on error
//
// HISTORY
////////////////////////////////////////////////////////////////////////

int Surface::FillRect( short x, short y, unsigned short w, unsigned short h,
                       unsigned long col ) const {
  if ( s_surface == 0 ) return -1;
 
  SDL_Rect src;
  src.x = x;
  src.y = y;
  src.w = w;
  src.h = h;

  if ( SDL_FillRect( s_surface, &src, col ) ) return -1;
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Surface::FillRectAlpha
// DESCRIPTION: Draw a rectangle using alpha-blending.
// PARAMETERS : x     - left edge of rectangle
//              y     - top edge of rectangle
//              w     - width of rectangle
//              h     - height of rectangle
//              col   - color of rectangle
//              alpha - alpha value to use for blending (default 128)
// RETURNS    : 0 on success, -1 on error
//
// HISTORY
////////////////////////////////////////////////////////////////////////

int Surface::FillRectAlpha( short x, short y, unsigned short w, unsigned short h,
                            const Color &col, unsigned char alpha /* = 128 */ ) const {
  Uint8 *bits, bpp;
  SDL_PixelFormat *fmt = s_surface->format;

  // get offset into surface buffer
  bpp = fmt->BytesPerPixel;
  if ( bpp < 2 ) return FillRect( x, y, w, h, col );   // not supported
  if ( SDL_MUSTLOCK( s_surface ) && (SDL_LockSurface(s_surface) < 0) ) return -1;

  for ( int py = y; py < y + h; ++py ) {
    bits = ((Uint8 *)s_surface->pixels) + py * s_surface->pitch + x * bpp;

    for ( int px = x; px < x + w; ++px ) {
      unsigned char r, g, b;
      Uint32 pixel;

      // get pixel from surface
      switch ( bpp ) {
        case 2:
          pixel = *((Uint16 *)(bits));
          break;
        case 3:
          if ( SDL_BYTEORDER == SDL_LIL_ENDIAN ) {
            pixel = bits[0] + (bits[1] << 8) + (bits[2] << 16);
          } else pixel = (bits[0] << 16) + (bits[1] << 8) + bits[2];
          break;
        case 4:
          pixel = *((Uint32 *)(bits));
          break;
      }

      // get RGB values from pixel
      r = (((pixel&fmt->Rmask)>>fmt->Rshift)<<fmt->Rloss);
      g = (((pixel&fmt->Gmask)>>fmt->Gshift)<<fmt->Gloss);
      b = (((pixel&fmt->Bmask)>>fmt->Bshift)<<fmt->Bloss);

      // blend with alpha
      r = (((col.r-r)*alpha)>>8) + r;
      g = (((col.g-g)*alpha)>>8) + g;
      b = (((col.b-b)*alpha)>>8) + b;
      DrawPixel( px, py, Color( r, g, b ) );

      bits += bpp;
    }
  }
  if ( SDL_MUSTLOCK( s_surface ) ) SDL_UnlockSurface( s_surface );
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Surface::FillPattern
// DESCRIPTION: Fill an area of the surface with a graphical pattern.
// PARAMETERS : x       - left edge of rectangle
//              y       - top edge of rectangle
//              w       - width of rectangle
//              h       - height of rectangle
//              pattern - pattern image
//              dx      - left edge of the first pattern on the surface
//              dy      - top edge of the first pattern on the surface
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Surface::FillPattern( short x, short y, unsigned short w, unsigned short h,
                           const Image &pattern, short dx, short dy ) {
  Rect clip;
  GetClipRect( clip );
  SetClipRect( Rect( x, y, w, h ) );

  short cx, cy = y - (y - dy) % pattern.Height();

  do {
    cx = x - (x - dx) % pattern.Width();

    while ( cx < x + w ) {
      pattern.Draw( this, cx, cy );
      cx += pattern.Width();
    }

    cy += pattern.Height();
  } while ( cy < y + h );

  SetClipRect( clip );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Surface::Colorize
// DESCRIPTION: Change all non-transparent pixels in the specified part
//              of the surface to one color.
// PARAMETERS : x   - left edge of rectangle
//              y   - top edge of rectangle
//              w   - width of rectangle
//              h   - height of rectangle
//              col - color
// RETURNS    : -
//
// HISTORY
//   13-08-2001 - lock surface if necessary
////////////////////////////////////////////////////////////////////////

void Surface::Colorize( short x, short y, unsigned short w, unsigned short h,
                        const Color &col ) const {
  SDL_Surface *s = s_surface;
  Sint32 px, py, ypnt;
  Uint32 transp = GetColorKey();
  Uint32 color = MapRGB( col );

  if ( SDL_MUSTLOCK( s ) && (SDL_LockSurface( s ) < 0) ) return;

  switch ( s->format->BytesPerPixel ) {
    case 1: {
      Uint8 *pnt;
      for ( py = y; py < y + h; py++ ) {
        ypnt = py * s->pitch;
        for ( px = x; px < x + w; px++ ) {
          pnt = (Uint8 *)s->pixels + px + ypnt;
          if ( *pnt != transp ) *pnt= (Uint8)color;
        }
      }
      break;
    }
    case 2: {
      Uint16 *pnt;
      for ( py = y; py < y + h; py++ ) {
        ypnt = py * s->pitch/2;
        for ( px = x; px < x + w; px++ ) {
          pnt = (Uint16 *)s->pixels + px + ypnt;
          if ( *pnt != transp ) *pnt = (Uint16)color;
        }
      }
      break;
    }
    case 3:     // 24-bpp mode not supported
      break;

    case 4: {
      Uint32 *pnt;
      for ( py = y; py < y + h; py++ ) {
        ypnt = py * s->pitch/4;
        for ( px = x; px < x + w; px++ ) {
          pnt = (Uint32 *)s->pixels + px + ypnt;
          if ( *pnt != transp ) *pnt= (Uint32)color;
        }
      } 
      break;
    }
  }

  if ( SDL_MUSTLOCK( s ) ) SDL_UnlockSurface( s );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Surface::DrawBox
// DESCRIPTION: Draw a bevelled 3-dimensional box.
// PARAMETERS : box  - box position and dimensions
//              type - box type (see surface.h for details)
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Surface::DrawBox( const Rect &box, unsigned short type ) {
  int x1, y1, x2, y2;
  Color light, shadow;

  if ( type & BOX_RECESSED ) {
    light = Color(CF_COLOR_SHADOW);
    shadow = Color(CF_COLOR_HIGHLIGHT);
  } else {
    light = Color(CF_COLOR_HIGHLIGHT);
    shadow = Color(CF_COLOR_SHADOW);
  }

  if ( type & BOX_SOLID ) FillRect( box, Color(CF_COLOR_BACKGROUND) );

  x1 = box.x;
  y1 = box.y;
  x2 = box.x + box.w - 1;
  y2 = box.y + box.h - 1;

  if ( type & BOX_CARVED ) {
    FillRect( x1, y1, 1, y2-y1+1, shadow );
    FillRect( x1, y1, x2-x1+1, 1, shadow );
    FillRect( x2, y1, 1, y2-y1+1, light );
    FillRect( x1, y2, x2-x1+1, 1, light );

    x1++; y1++; x2--; y2--;

    FillRect( x1, y1, 1, y2-y1+1, light );
    FillRect( x1, y1, x2-x1+1, 1, light );
    FillRect( x2, y1, 1, y2-y1+1, shadow );
    FillRect( x1, y2, x2-x1+1, 1, shadow );
  } else {
    FillRect( x1, y1, 1, y2-y1, light );
    FillRect( x1, y1, x2-x1, 1, light );
    FillRect( x2, y1, 1, y2-y1+1, shadow );
    FillRect( x1, y2, x2-x1+1, 1, shadow );
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Surface::DrawLine
// DESCRIPTION: Draw a straight line to the surface.
// PARAMETERS : sx  - starting x coordinate
//              sy  - starting y coordinate
//              dx  - destination x coordinate
//              dy  - destination y coordinate
//              col - color
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

//void Surface::DrawLine( short sx, short sy, short dx, short dy, const Color &col ) {
//  if ( SDL_MUSTLOCK( s_surface ) && (SDL_LockSurface(s_surface) < 0) ) return;
//  if ( ABS(dx - sx) > ABS(dy - sy) ) {
//    float rise = (float)(dy - sy) / (float)(dx - sx);
//    unsigned short num = ABS( dx - sx );
//
//    for ( int i = 0; i < num; i++ )
//      DrawPixel( sx + i, sy + i * rise + 0.5, col );
//  } else {
//    float rise = (float)(dx - sx) / (float)(dy - sy);
//    unsigned short num = ABS( dy - sy );
//
//    for ( int i = 0; i < num; i++ )
//      DrawPixel( sx + i * rise + 0.5, sy + i, col );
//  }
//  if ( SDL_MUSTLOCK( s_surface ) ) SDL_UnlockSurface( s_surface );
//}

////////////////////////////////////////////////////////////////////////
// NAME       : Surface::DrawPixel
// DESCRIPTION: Draw a single pixel in the specified color. This
//              procedure does not check whether the surface needs to
//              be locked to modify it, so the caller must make sure
//              everything is properly set up.
// PARAMETERS : x   - horizontal position in surface
//              y   - vertical position in surface
//              col - color of pixel
// RETURNS    : -
//
// HISTORY
//   13-08-2001 - don't lock the surface; callers should do it
////////////////////////////////////////////////////////////////////////

void Surface::DrawPixel( short x, short y, const Color &col ) const {
  Uint32 pixel;
  Uint8 *bits, bpp;

  // map color to display
  pixel = MapRGB( col );

  // get offset into surface buffer
  bpp = s_surface->format->BytesPerPixel;
  bits = ((Uint8 *)s_surface->pixels) + y * s_surface->pitch + x * bpp;

  switch ( bpp ) {
    case 1:
      *((Uint8 *)(bits)) = (Uint8)pixel;
      break;
    case 2:
      *((Uint16 *)(bits)) = (Uint16)pixel;
      break;
    case 3:
      Uint8 r, g, b;

      r = (pixel>>s_surface->format->Rshift) & 0xFF;
      g = (pixel>>s_surface->format->Gshift) & 0xFF;
      b = (pixel>>s_surface->format->Bshift) & 0xFF;
      *((bits) + s_surface->format->Rshift / 8) = r;
      *((bits) + s_surface->format->Gshift / 8) = g;
      *((bits) + s_surface->format->Bshift / 8) = b;
      break;
    case 4:
      *((Uint16 *)(bits)) = (Uint16)pixel;
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Surface::LoadPalette
// DESCRIPTION: Load the color palette from a data file. The file is
//              supposed to be opened already.
// PARAMETERS : file - SDL_RWops data file descriptor
//              num  - number of colors to read
// RETURNS    : 0 on success, -1 on error
//
// HISTORY
////////////////////////////////////////////////////////////////////////

int Surface::LoadPalette( SDL_RWops *file, unsigned short num ) {
  unsigned char *pal, *col;

  // file contains num colors, first one is transparent
  pal = new unsigned char[num * 3];
  if ( !pal ) return -1;
  col = pal;

  SDL_RWread( file, pal, 1, num * 3 );

  for ( int i = 0; i < num; i++ ) {
    s_surface->format->palette->colors[i].r = *col++;
    s_surface->format->palette->colors[i].g = *col++;
    s_surface->format->palette->colors[i].b = *col++;
  }

  // set transparent color
  unsigned long ckey = SDL_MapRGB( s_surface->format, pal[0], pal[1], pal[2] );
  SDL_SetColorKey( s_surface, SDL_SRCCOLORKEY, ckey );

  delete [] pal;
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Surface::LoadImageData
// DESCRIPTION: Load image data from a data file. The file is supposed
//              to be opened already.
// PARAMETERS : file - SDL_RWops data file descriptor
// RETURNS    : 0 on success, -1 on error
//
// HISTORY
////////////////////////////////////////////////////////////////////////

int Surface::LoadImageData( SDL_RWops *file ) {
  unsigned short colors, width, height;
  unsigned long packed;

  colors = SDL_ReadLE16( file );
  width = SDL_ReadLE16( file );
  height = SDL_ReadLE16( file );
  packed = SDL_ReadLE32( file );

  if ( Create( width, height, DATA_BPP, 0 ) ) return -1;
  if ( LoadPalette( file, colors ) ) return -1;

  char *buf = new char[packed + width * height];
  if ( !buf ) return -1;
  char *unbuf = &buf[packed];

  SDL_RWread( file, buf, 1, packed );
  byterun_unpack( buf, unbuf, packed, width * height );

  for ( int y = 0; y < height; y++ ) {
    for ( int x = 0; x < width; x++ ) {
      *((char *)s_surface->pixels + x + y * s_surface->pitch) =
                                          unbuf[x + y * width];
    }
  }

  delete [] buf;
  return 0;
}


////////////////////////////////////////////////////////////////////////
// NAME       : Image::Image
// DESCRIPTION: Create an image from a surface. An image is simply a
//              rectangular portion of the surface.
// PARAMETERS : surface - image surface
//              x       - left edge of image
//              y       - zop edge of image
//              w       - image width
//              h       - image height
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

Image::Image( Surface *surface, short x, short y, unsigned short w,
              unsigned short h ) : Rect( x, y, w, h ) {
  this->surface = surface;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Image::Draw
// DESCRIPTION: Draw the image to a surface.
// PARAMETERS : dest - destination surface
//              x    - left edge of destination rectangle
//              y    - top edge of destination rectangle
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Image::Draw( Surface *dest, short x, short y ) const {
  surface->Blit( dest, *this, x, y );
}

