// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000, 2001 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

/////////////////////////////////////////////////////////////////////////
// unitwindow.cpp
//
// History:
//  17-12-2000 - created
//  03-04-2001 - list widget now answers double clicking and the space
//               bar, so there is no more need for the Select button
/////////////////////////////////////////////////////////////////////////

#include "unitwindow.h"
#include "extwindow.h"
#include "globals.h"
#include "game.h"

#define ICON_MINI_WIDTH		15
#define ICON_MINI_HEIGHT	15
#define ICON_MINI_REPAIR1_X	284
#define ICON_MINI_REPAIR1_Y	14
#define ICON_MINI_REPAIR2_X	284
#define ICON_MINI_REPAIR2_Y	14
#define ICON_MINI_BUILD1_X	299
#define ICON_MINI_BUILD1_Y	14
#define ICON_MINI_BUILD2_X	299
#define ICON_MINI_BUILD2_Y	14
#define ICON_MINI_TRANSFER1_X	329
#define ICON_MINI_TRANSFER1_Y	14
#define ICON_MINI_TRANSFER2_X	329
#define ICON_MINI_TRANSFER2_Y	14


// scrollable class used for the crystals exchange dialog
class TransferScrollable : public Scrollable {
public:
  TransferScrollable( short crystals1, short crystals2,
                      StringWidget *str1, StringWidget *str2,
                      UnitContainer *c1, UnitContainer *c2 ) :
          Scrollable( crystals1 )
  { csum = crystals1 + crystals2; widgets[0] = str1; widgets[1] = str2;
    container[0] = c1; container[1] = c2; }

  virtual void Set( short level );

private:
  short csum;              // sum of crystals in both exchange partners
  StringWidget *widgets[2];
  UnitContainer *container[2];
};

// node class used by the UnitListWidget
class ULWNode : public Node {
public:
  const UnitType *type;
  Unit *unit;
  unsigned short image;
  bool ok;
};

// list widget used to display units...
class UnitListWidget : public ListWidget {
public:
  UnitListWidget( short id, short x, short y, unsigned short w,
    unsigned short h, List *list, short selected,
    unsigned short flags, MapWindow *mwindow, Window *window );

  virtual void DrawNodes( void );
  virtual unsigned short ItemHeight( void ) const { return itemh; }

private:
  unsigned short itemh;
  MapWindow *mwin;
};


// button ID's for the container hook
#define CH_BUTTON_REPAIR	0
#define CH_BUTTON_BUILD		1
#define CH_BUTTON_TRANSFER	2

#define CH_BUTTON_REPAIR_CONFIRM 3



////////////////////////////////////////////////////////////////////////
// NAME       : TransferScrollable::Set
// DESCRIPTION: When a new slider level is set, adjust the TextWidgets
//              to display the correct amount of crystals to be
//              transferred.
// PARAMETERS : level - current slider level (= amount of crystals for
//                      the first exchange partner)
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void TransferScrollable::Set( short level ) {
  current = level;

  char buf[8];
  short current2 = csum - current;

  itoa( current, buf );
  widgets[0]->SetString( buf, true );
  itoa( current2, buf );
  widgets[1]->SetString( buf, true );

  container[0]->SetCrystals( current );
  container[1]->SetCrystals( current2 );
}


////////////////////////////////////////////////////////////////////////
// NAME       : UnitListWidget::UnitListWidget
// DESCRIPTION: Create a new unit list widget.
// PARAMETERS : id       - widget identifier
//              x        - left edge of widget
//              y        - top edge of widget
//              w        - widget width
//              h        - widget height
//              list     - list nodes
//              selected - node highlighted by default (-1 == none)
//              flags    - widget flags (see widget.h for details)
//              mwindow  - pointer to map window
//              window   - widget parent window
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

UnitListWidget::UnitListWidget( short id, short x, short y, unsigned short w,
    unsigned short h, List *list, short selected, unsigned short flags,
    MapWindow *mwindow, Window *window ) :
    ListWidget( id, x, y, w, h, list, selected, flags, NULL, window ) {
  mwin = mwindow;
  spacing = 0;
  itemh = ICON_HEIGHT + spacing;
  Update();
}

////////////////////////////////////////////////////////////////////////
// NAME       : UnitListWidget::DrawNodes
// DESCRIPTION: Draw the list nodes.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
//   13-08-2001 - use FillRectAlpha() for the selected item
////////////////////////////////////////////////////////////////////////

void UnitListWidget::DrawNodes( void ) {
  Rect box( x + 4, y + 1 + spacing, w - 8, h - 2 - 2 * spacing );
  Rect area( x + 1, y + 1, w - 2, h - 2 );
  short num = toprow / ItemHeight();                          // number of top node
  ULWNode *n = static_cast<ULWNode *>(list->GetNode( num ));  // top node
  short xoff = box.x, yoff = box.y + (num * ItemHeight()) - toprow;

  surface->DrawBack( area );

  while ( n ) {
    if ( num == current ) {
      Rect hilite( x + 2, yoff, w - 4, ItemHeight() );
      hilite.Clip( area );
      surface->FillRectAlpha( hilite, ColLight );
    }

    // draw unit icon
    Rect icon( *Images[ICON_HEX_BG] );
    Rect dest( xoff, yoff, ICON_WIDTH, ICON_HEIGHT );
    dest.ClipBlit( icon, area );
    Icons->Blit( surface, icon, dest.x, dest.y );
    mwin->DrawHex( n->image, surface, xoff + (ICON_WIDTH - GFX_WIDTH) / 2,
                   yoff + (ICON_HEIGHT - GFX_HEIGHT) / 2, box );
    if ( !n->ok || (n->unit && !n->unit->IsReady()) )
      mwin->DrawHex( IMG_NOT_AVAILABLE, surface,
                     xoff + (ICON_WIDTH - GFX_WIDTH) / 2,
                     yoff + (ICON_HEIGHT - GFX_HEIGHT) / 2, box );

    char buf[4];
    if ( n->unit ) {
      icon = *Images[ICON_XP_BASE + n->unit->XPLevel()];
      dest = Rect( xoff + ICON_WIDTH + 4, yoff + 2, XP_ICON_WIDTH, XP_ICON_HEIGHT );
      dest.ClipBlit( icon, area );
      Icons->Blit( surface, icon, dest.x, dest.y );
      font->Write( itoa( n->unit->GroupSize(), buf ), surface, xoff + ICON_WIDTH + 4,
                   yoff + ICON_HEIGHT - font->Height() - 2, box );
    } else
      font->Write( itoa( n->type->ut_build, buf ), surface, xoff + ICON_WIDTH + 4,
                   yoff + (ICON_HEIGHT - font->Height()) / 2, box );

    yoff += ItemHeight();
    if ( yoff >= box.y + box.h ) break;

    num++;
    n = static_cast<ULWNode *>(n->Next());
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : ContainerWindow::ContainerWindow
// DESCRIPTION: Create a new ContainerWindow for a transport or building.
// PARAMETERS : transport - transport to create the window for; if this
//                          is NULL (and only if it is) the building
//                          parameter must be given
//              building  - building for which to create the window; if
//                          it is NULL, transport must be non-NULL
//              view      - pointer to the view
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

ContainerWindow::ContainerWindow( Transport *transport, Building *building,
    View *view ) : Window( WIN_CLOSE_ESC|WIN_CENTER, view ) {
  int i;

  if ( transport ) {
    c = transport;
    unit = true;
  } else {
    c = building;
    unit = false;
  }
  mode = CW_MODE_NORMAL;
  transferscroll = NULL;

  // calculate window dimensions
  short width = GFX_WIDTH + XP_ICON_WIDTH + lfont->Width() * 20 + 60;
  short height = 8 * GFX_HEIGHT + 40 + sfont->Height();
  SetSize( width, height );

  width = GFX_WIDTH + XP_ICON_WIDTH + 25;   // listwidget width
  unitinfo = Rect( width + 20, h - 45 - (GFX_HEIGHT + ICON_HEIGHT + sfont->Height()) * 2,
                   w - width - 30, (GFX_HEIGHT + ICON_HEIGHT) * 2 + sfont->Height() + 25 );

  for ( i = 0; i < CH_NUM_BUTTONS; i++ ) buttons[i] = NULL;

  // create unit list
  for ( i = 0; i < c->FullSlots(); i++ ) {
    ULWNode *n = new ULWNode;

    n->unit = c->GetUnit( i );
    n->type = n->unit->Type();
    n->image = n->unit->BaseImage();
    n->ok = n->unit->IsReady();
    normal.AddTail( n );
  }

  // create list widget
  listwidget = new UnitListWidget( 0, 10, 10, width, 8 * GFX_HEIGHT + 10, &normal,
                            -1, WIDGET_VSCROLL|WIDGET_ARROWSCROLL, Gam, this );
  listwidget->SetHook( this );

  // create buttons
  new ButtonWidget( GUI_CLOSE, 10, h - sfont->Height() - 18,
      w - 20, sfont->Height() + 8, 'x', WIDGET_DEFAULT, "Exit", this );

  ButtonWidget *btn;
  unsigned short xoff = width + 20;
  if ( !unit ) {
    if ( building->IsWorkshop() ) {
      btn = new ButtonWidget( CH_BUTTON_REPAIR, xoff,
                      unitinfo.y - ICON_MINI_HEIGHT - 15, ICON_MINI_WIDTH + 4,
                      ICON_MINI_HEIGHT + 4, 'r', WIDGET_STYLE_GFX, NULL, this );
      btn->SetImage( Icons, Rect( ICON_MINI_REPAIR1_X, ICON_MINI_REPAIR1_Y,
                                  ICON_MINI_WIDTH, ICON_MINI_HEIGHT ),
                            Rect( ICON_MINI_REPAIR2_X, ICON_MINI_REPAIR2_Y,
                                  ICON_MINI_WIDTH, ICON_MINI_HEIGHT ) );
      btn->SetHook( this );
      xoff += btn->Width();
      buttons[CH_BUTTON_REPAIR] = btn;
    }

    if ( building->IsFactory() && (building->UnitProduction() != 0) ) {
      btn = new ButtonWidget( CH_BUTTON_BUILD, xoff,
                      unitinfo.y - ICON_MINI_HEIGHT - 15, ICON_MINI_WIDTH + 4,
                      ICON_MINI_HEIGHT + 4, 'b', WIDGET_STYLE_GFX, NULL, this );
      btn->SetImage( Icons, Rect( ICON_MINI_BUILD1_X, ICON_MINI_BUILD1_Y,
                                  ICON_MINI_WIDTH, ICON_MINI_HEIGHT ),
                            Rect( ICON_MINI_BUILD2_X, ICON_MINI_BUILD2_Y,
                                  ICON_MINI_WIDTH, ICON_MINI_HEIGHT ) );
      btn->SetHook( this );
      xoff += btn->Width();
      buttons[CH_BUTTON_BUILD] = btn;
    }
  }

  btn = new ButtonWidget( CH_BUTTON_TRANSFER, xoff,
               unitinfo.y - ICON_MINI_HEIGHT - 15, ICON_MINI_WIDTH + 4,
               ICON_MINI_HEIGHT + 4, 't', WIDGET_STYLE_GFX, NULL, this );
  btn->SetImage( Icons, Rect( ICON_MINI_TRANSFER1_X, ICON_MINI_TRANSFER1_Y,
                              ICON_MINI_WIDTH, ICON_MINI_HEIGHT ),
                        Rect( ICON_MINI_TRANSFER2_X, ICON_MINI_TRANSFER2_Y,
                              ICON_MINI_WIDTH, ICON_MINI_HEIGHT ) );
  btn->SetHook( this );
  buttons[CH_BUTTON_TRANSFER] = btn;

  Draw();
  if ( !normal.IsEmpty() ) listwidget->Select( 0 );
  Show();
}

////////////////////////////////////////////////////////////////////////
// NAME       : ContainerWindow::~ContainerWindow
// DESCRIPTION: Destroy a ContainerWindow instance.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

ContainerWindow::~ContainerWindow( void ) {
  delete transferscroll;
}

////////////////////////////////////////////////////////////////////////
// NAME       : ContainerWindow::Draw
// DESCRIPTION: Draw the container window.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
//   03-06-2001 - don't use clipping when writing; this is done anyway
////////////////////////////////////////////////////////////////////////

void ContainerWindow::Draw( void ) {
  Window::Draw();
  Gam->DrawUnitInfo( NULL, this, unitinfo );

  // draw a separator bar
  unsigned short sep = 45 + GFX_WIDTH + XP_ICON_WIDTH;

  // show the name of the unit/building
  DrawBox( Rect( sep, 10, w - sep - 10, lfont->Height() + 20 ), BOX_RAISED );
  const char *name;
  if ( unit ) name = static_cast<Transport *>(c)->Name();
  else name = static_cast<Building *>(c)->Name();

  lfont->Write( name, this,
                sep + 3 + (w - sep - 10 - lfont->TextWidth(name)) / 2, 23,
                ColDark );
  lfont->Write( name, this,
                sep + (w - sep - 10 - lfont->TextWidth(name)) / 2, 20,
                ColLight );

  // show crystal storage
  Images[ICON_CRYSTALS]->Draw( this, w - sfont->Width() * 10 - ICON_WIDTH - 10, lfont->Height() + 40 );
  DrawCrystals();
}

////////////////////////////////////////////////////////////////////////
// NAME       : ContainerWindow::DrawCrystals
// DESCRIPTION: Show the crystals storage of the container and its
//              change per turn.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void ContainerWindow::DrawCrystals( void ) {
  char buf[16];
  unsigned short stock = 0, change = 0;
  Rect rect( w - sfont->Width() * 10 - 10,
             lfont->Height() + 40 + (ICON_HEIGHT - sfont->Height()) / 2,
             sfont->Width() * 10, sfont->Height() );

  if ( !unit ) {
    Building *b = static_cast<Building *>(c);
    stock = b->Crystals();
    change = b->CrystalProduction();
  }

  if ( change > 0 ) sprintf( buf, "%d (+%d)", stock, change );
  else itoa( stock, buf );

  DrawBack( rect );
  sfont->Write( buf, this, rect.x, rect.y );
} 

////////////////////////////////////////////////////////////////////////
// NAME       : ContainerWindow::HandleEvent
// DESCRIPTION: React to system events.
// PARAMETERS : event - event received by the event handler
// RETURNS    : GUI status
//
// HISTORY
////////////////////////////////////////////////////////////////////////

GUI_Status ContainerWindow::HandleEvent( const SDL_Event &event ) {
  GUI_Status rc;

  if ( (event.type == SDL_KEYDOWN) && (event.key.keysym.sym == SDLK_SPACE) &&
       listwidget->Selected() ) {
    // activate the currently selected node in the list widget
    // => either select the unit for movement (NORMAL mode) or
    //    build the selected unit type (PRODUCTION mode)
    rc = SelectNode( static_cast<ULWNode *>( listwidget->Selected() ) );
  } else rc = Window::HandleEvent( event );
  return rc;
} 

////////////////////////////////////////////////////////////////////////
// NAME       : ContainerWindow::SwitchMode
// DESCRIPTION: Enter or leave build mode. This function creates or
//              destroys the unit production list for the unit list
//              widget. This function may only be used if the object
//              is a building (more specifically: a factory)
// PARAMETERS : newmode - mode to switch to
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void ContainerWindow::SwitchMode( unsigned short newmode ) {

  if ( newmode == CW_MODE_PRODUCTION ) {             // only buildings, current
                                                     // mode must be CW_MODE_NORMAL
    // create production list
    Building *b = static_cast<Building *>(c);
    unsigned long prod = b->UnitProduction();

    if ( prod ) {
      unsigned short crystals = b->Crystals();

      for ( int i = 0; i < 32; i++ ) {
        if ( prod & (1 << i) ) {        // this unit type can be built
          ULWNode *n = new ULWNode;
          n->type = Gam->GetUnitType( i );
          n->unit = NULL;
          n->image = n->type->ut_image + b->Owner()->ID() * 6;
          n->ok = (n->type->ut_build <= crystals);
          build.AddTail( n );
        }
      }

      // disable transfer and repair buttons
      if ( buttons[CH_BUTTON_REPAIR] ) buttons[CH_BUTTON_REPAIR]->Disable();
      buttons[CH_BUTTON_TRANSFER]->Disable();

      // display new list in list widget
      DrawBack( unitinfo );
      listwidget->SwitchList( &build, 0 );
      Gam->DrawUnitInfo( static_cast<ULWNode *>(build.Head())->type, this, unitinfo );
      Show();
    }
  } else if ( newmode == CW_MODE_TRANSFER ) {       // must be in CW_MODE_NORMAL
    // disable repair and build buttons and list widget
    if ( buttons[CH_BUTTON_REPAIR] ) buttons[CH_BUTTON_REPAIR]->Disable();
    if ( buttons[CH_BUTTON_BUILD] ) buttons[CH_BUTTON_BUILD]->Disable();
    listwidget->Disable();

    DrawBack( unitinfo );
    Gam->DrawUnitInfo( NULL, this, unitinfo );     // clear unit info area

    // create new widgets
    Transport *t = static_cast<Transport *>( static_cast<ULWNode *>(listwidget->Selected())->unit );
    char buf[8];
    short maxhost, maxclient, fullhost, fullclient, min, max;
    maxhost = (unit ? c->Slots() * c->MaxWeight() : static_cast<Building *>(c)->CrystalStore());
    maxclient = t->Slots() * t->MaxWeight();
    fullhost = c->Crystals();
    fullclient = t->Crystals();
    min = MAX( 0, fullhost - (maxclient - fullclient) );
    max = MIN( maxhost, fullhost + fullclient );

    slider = new SliderWidget( 0, unitinfo.x + GFX_WIDTH + 20,
                               unitinfo.y + (unitinfo.h - DEFAULT_SLIDER_SIZE) / 2,
                               unitinfo.w - 2 * ICON_WIDTH - 40, DEFAULT_SLIDER_SIZE,
                               min, max, fullhost, max - min + 1, 0,
                               WIDGET_HSCROLL|WIDGET_ARROWSCROLL, NULL, this );
    stringwidgets[0] = new StringWidget( 0, unitinfo.x + 10,
                               unitinfo.y + unitinfo.h / 2, ICON_WIDTH, sfont->Height() + 8,
                               0, itoa( fullclient, buf ), 4,
                               WIDGET_ALIGN_CENTER|WIDGET_STR_CONST, NULL, this );
    stringwidgets[1] = new StringWidget( 0, slider->LeftEdge() + slider->Width() + 10,
                               unitinfo.y + unitinfo.h / 2, ICON_WIDTH, sfont->Height() + 8,
                               0, itoa( fullhost, buf ), 4,
                               WIDGET_ALIGN_CENTER|WIDGET_STR_CONST, NULL, this );
    transferscroll = new TransferScrollable( fullhost, fullclient, stringwidgets[1],
                               stringwidgets[0], c, t );
    slider->SetScrollable( transferscroll );

    slider->Draw();
    stringwidgets[0]->Draw();
    stringwidgets[1]->Draw();

    // draw images
    Gam->DrawHex( t->BaseImage(), this, stringwidgets[0]->LeftEdge(),
                   stringwidgets[0]->TopEdge() - ICON_HEIGHT - 2, unitinfo );
    Images[ICON_CRYSTALS]->Draw( this, stringwidgets[1]->LeftEdge(),
                     stringwidgets[1]->TopEdge() - ICON_HEIGHT - 2 );
    Show();
  } else {                                          // switch back to CW_MODE_NORMAL...
    if ( mode == CW_MODE_PRODUCTION ) {             // ...from CW_MODE_PRODUCTION
      // destroy production list
      while ( !build.IsEmpty() ) delete build.RemHead();

      // re-enable repair and transfer buttons
      if ( buttons[CH_BUTTON_REPAIR] ) buttons[CH_BUTTON_REPAIR]->Enable();
      buttons[CH_BUTTON_TRANSFER]->Enable();

      // display normal list in widget
      DrawBack( unitinfo );
      listwidget->SwitchList( &normal, -1 );
      Gam->DrawUnitInfo( NULL, this, unitinfo );
      Show();
    } else if ( mode == CW_MODE_TRANSFER ) {          // ...from CW_MODE_TRANSFER
      // delete transfer stuff
      RemoveWidget( slider );
      RemoveWidget( stringwidgets[0] );
      RemoveWidget( stringwidgets[1] );
      delete slider;
      delete stringwidgets[0];
      delete stringwidgets[1];
      delete transferscroll;
      transferscroll = NULL;

      // redraw unit info
      DrawBack( unitinfo );
      Gam->DrawUnitInfo( static_cast<ULWNode *>(listwidget->Selected())->type,
                          this, unitinfo );
      DrawCrystals();

      // re-enable repair and build buttons and list widget
      if ( buttons[CH_BUTTON_REPAIR] ) buttons[CH_BUTTON_REPAIR]->Enable();
      if ( buttons[CH_BUTTON_BUILD] ) buttons[CH_BUTTON_BUILD]->Enable();
      listwidget->Enable();
      Show();
    }
  }

  mode = newmode;
}

////////////////////////////////////////////////////////////////////////
// NAME       : ContainerWindow::SelectNode
// DESCRIPTION: When a node from the list widget is selected, either try
//              to activate that unit (if in normal mode) or try to
//              build it (if in production mode).
// PARAMETERS : node - selected node from the list widget
// RETURNS    : GUI status
//
// HISTORY
////////////////////////////////////////////////////////////////////////

GUI_Status ContainerWindow::SelectNode( ULWNode *node ) {
  GUI_Status rc = GUI_OK;

  if ( node ) {
    if ( mode == CW_MODE_NORMAL ) {          // normal mode
      if ( node->ok ) {                      // unit is ready
        Gam->SelectUnit( node->unit );
        rc = GUI_CLOSE;
      }
    } else {                        // production mode
      if ( node->ok ) {             // sufficient amount of crystals
        Building *b = static_cast<Building *>(c);
        Unit *u = Gam->CreateUnit( node->type, b->Position().x,
                                    b->Position().y, b->Owner() );

        // add unit to the widget's list
        ULWNode *ulw = new ULWNode;

        ulw->unit = u;
        ulw->type = u->Type();
        ulw->image = u->BaseImage();
        ulw->ok = false;
        normal.AddTail( ulw );

	b->SetCrystals( b->Crystals() - u->BuildCost() );
        DrawCrystals();
        Show();
        SwitchMode( CW_MODE_NORMAL );
      } else new NoteWindow( "Insufficient supplies", "You cannot build this unit.", 0, view );
    }
  }
  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : ContainerWindow::Activate
// DESCRIPTION: Call the appropriate functions when buttons are pressed
//              in the ContainerWindow or associated windows.
// PARAMETERS : button - button that called the function
//              win    - pointer to the window the button belongs to
// RETURNS    : GUI status
//
// HISTORY
////////////////////////////////////////////////////////////////////////

GUI_Status ContainerWindow::Activate( ButtonWidget *button, Window *win ) {
  GUI_Status rc = GUI_OK;

  switch ( button->ID() ) {
  case CH_BUTTON_REPAIR: {
    ULWNode *n = static_cast<ULWNode *>(listwidget->Selected());
    if ( n ) {
      Unit *u = n->unit;
      if ( u->GroupSize() < MAX_GROUP_SIZE ) {
        Building *b = static_cast<Building *>(c);
        unsigned short cost = u->RepairCost();
        char repmsg[64];
        sprintf( repmsg, "Repairing the %s requires %d crystals.", u->Name(), cost );

        if ( cost <= b->Crystals() ) {
          RequestWindow *reqwin = new RequestWindow( repmsg, "rRepair", "cCancel",
                                                     1, view );
          reqwin->button1->SetID( CH_BUTTON_REPAIR_CONFIRM );
          reqwin->button1->SetHook( this );
        } else new NoteWindow( "Insufficient stocks!", repmsg, WIN_CLOSE_ESC, view );
      }
    }
    break; }
  case CH_BUTTON_BUILD:
    SwitchMode( (Mode() == CW_MODE_NORMAL) ? CW_MODE_PRODUCTION : CW_MODE_NORMAL );
    break;
  case CH_BUTTON_TRANSFER: {
    ULWNode *n = static_cast<ULWNode *>(listwidget->Selected());
    if ( n ) {
      Unit *u = n->unit;
      if ( u->IsTransport() ) {
        SwitchMode( (Mode() == CW_MODE_NORMAL) ? CW_MODE_TRANSFER : CW_MODE_NORMAL );
      } else new NoteWindow( "Transfer", "Unit cannot transport crystals",
                             WIN_CLOSE_ESC, view );
    }
    break; }

  case CH_BUTTON_REPAIR_CONFIRM: {
    Unit *u = static_cast<ULWNode *>(listwidget->Selected())->unit;
    Building *b = static_cast<Building *>(c);
    b->SetCrystals( u->Repair( b->Crystals() ) );
    listwidget->Draw();					// update list display
    DrawCrystals();					// update crystals
    Show();
    play_audio( SND_GAM_REPAIR, 0 );
    rc = GUI_CLOSE;					// close the request window
    break; }
  }

  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : ContainerWindow::Activate
// DESCRIPTION: Display information about the currently selected unit
//              or, if the node has been selected twice (e.g. by double
//              clicking), select the respective unit.
// PARAMETERS : widget - calling list widget
//              node   - pointer to selected node (may be NULL)
//              win    - pointer to the window the list belongs to
// RETURNS    : GUI status
//
// HISTORY
////////////////////////////////////////////////////////////////////////

GUI_Status ContainerWindow::Activate( ListWidget *widget, Node *node, Window *win ) {
  static Node *last = NULL;
  GUI_Status rc = GUI_OK;

  if ( node != last ) {
    const UnitType *type = NULL;
    if ( node ) type = static_cast<ULWNode *>(node)->type;
    DrawBack( unitinfo );
    Gam->DrawUnitInfo( type, this, unitinfo );
    win->Show( unitinfo );
    last = node;
  } else if ( node ) rc = SelectNode( static_cast<ULWNode *>(node) );
  return rc;
}


////////////////////////////////////////////////////////////////////////
// NAME       : UnitInfoWindow::UnitInfoWindow
// DESCRIPTION: Pop up a window with information about the given unit
//              type.
// PARAMETERS : type   - unit type
//              mapwin - pointer to map window
//              view   - view the window will be attached to
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

UnitInfoWindow::UnitInfoWindow( const UnitType *type, MapWindow *mapwin, View *view ) :
                Window( WIN_CLOSE_ESC|WIN_CENTER, view ) {
  // calculate window dimensions
  SetSize( sfont->Width() * 24 + ICON_WIDTH * 2 + 30,
           GFX_HEIGHT + GFX_OVERLAP_Y + (ICON_HEIGHT + sfont->Height()) * 2 + 40 );

  // create button
  ButtonWidget *bw = new ButtonWidget( GUI_CLOSE, 1, h - sfont->Height() - 8,
                                       w - 2, sfont->Height() + 8,
                                       'o', WIDGET_DEFAULT, "OK", this );
  Draw();
  mapwin->DrawUnitInfo( type, this, Rect( 5, 5, w - 10, h - bw->Height() - 10 ) );
  Show();
}

