# movingarray.rb, copyright (c) 2007 by Vincent Fourmond: 
# Moving arrays
  
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
  
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details (in the COPYING file).

require 'Dobjects/Dvector'
require 'Tioga/FigureConstants'


module CTioga

  Version::register_svn_info('$Revision: 854 $', '$Date: 2008-11-27 23:21:51 +0100 (Thu, 27 Nov 2008) $')

  # A helper function that converts user input into a Tioga
  # color.
  #
  # TODO: allow designs such as in LaTeX xcolor: Red!10!Blue !
  def self.get_tioga_color(str)
    if str =~ /([\d.]+),([\d.]+),([\d.]+)/ # Array notation
      color = Dobjects::Dvector.new([$1.to_f,$2.to_f,$3.to_f])
      if color.max > 1
        color *= 1.0/255.0
      end
      return color.to_a
    elsif str =~ /#([0-9a-fA-F]{6})/ # HTML notation
      return $1.scan(/../).map { |i| i.to_i(16)/255.0 }
    else
      return Tioga::FigureConstants.const_get(str) 
    end
  end

  # This class is just a means to make an instance of Array stand out
  class SpecialArray < Array
  end


  # The MovingArray class provides a way to scroll through an array,
  # such as a set of colors, markers, linstyles, and so on. The class
  # provides a way to set the value directly as well, and to forget it.
  class MovingArray 
    include Tioga

    def initialize(sets, current = nil, startpoint = 0)
      @sets = sets

      # Pick the first key as a default
      current = @sets.keys.first unless current
      choose_current_set(current) 

      @current_point = startpoint # the current point in the current set
    end

    def value
      @current_point += 1
      if @current_point >= @current_set.length
        @current_point %= @current_set.length
      end
      return @current_set[@current_point - 1]
    end

    # Chooses the current set. If the name refers to an existing set,
    # this one is chosen. If not, and if the name looks like a gradient
    # specification, we interpret it as such. In any other case,
    # the chosen set is the set containing only the argument. This way,
    # to get a set containing a single color, you can use
    #
    #  choose_current_set(Pink)
    def choose_current_set(name)
      @current_set_name = name
      if @sets.key?(name)
        @current_set_name = name
        @current_set = @sets[name]
      elsif name =~ /gradient:(.+)--(.+),(\d+)/
        # Special case for color gradients
        s = CTioga.get_tioga_color($1)
        e = CTioga.get_tioga_color($2)
        nb = $3.to_i
        fact = if nb > 1
                 1.0/(nb - 1)     # The famous off-by one...
               else
                 warn "Incorrect gradient number: #{nb}"
                 1.0
               end
        array = []
        nb.times do |i|
          a = []
          f = i * fact
          e.each_index do |c|
            a << s[c] * (1 - f) + e[c] * f
          end
          array << a
        end
        @current_set = array
      else
        if name.is_a? SpecialArray
          @current_set = name
        else
          @current_set = [name]
        end
      end
    end

    # Returns the available sets.
    def available_sets
      return @sets.keys
    end

    # Tells whether the given name is a valid set. Takes care of the special
    # case of gradients.
    #
    # TODO: this looks hackish, somehow, but I don't see a way to make
    # that less so...
    def valid_set?(name)
      if name =~ /^gradient:/
        return true
      else
        return @sets.key? name
      end
    end

  end

end
