
/*
 * Copyright (c) 2000 David Stes.
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published 
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: coroute.m,v 1.14 2001/03/14 19:17:56 stes Exp $
 */

#include <assert.h>
#include <stdlib.h>
#include <signal.h>
#include <Object.h>
#include <ocstring.h>
#include <ordcltn.h>
#include "cursel.h"
#include "process.h"
#include "coroute.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <curses.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include "frame.h"

static id coroutines;
static id okvsigpath;

void deletefifos()
{
  [coroutines elementsPerform:@selector(destroy)];
  [coroutines elementsPerform:@selector(unlink)];
  unlink([okvsigpath str]);
}

void makeokvsigpath()
{
  char *t = getenv("TMPDIR");
  if (!t) t = P_tmpdir;
  okvsigpath = [String sprintf:"%s/okvsig.%d",t,getpid()];
  mkfifo([okvsigpath str],0666);
  atexit(deletefifos);
}

static int vsiglock;

void startvsig(void)
{
  if (okvsigpath != nil && vsiglock == 0) {
    vsiglock = open([okvsigpath str],O_RDONLY | O_NONBLOCK);
  }
}

void stopvsig(void)
{
  if (vsiglock>0) {
    close(vsiglock);
    vsiglock = 0;
  } 
}

@implementation Coroutine 

#define NARG ((i++<n)?[tokens at:i]:nil)
#define FIFOMODS (S_IWUSR|S_IRUSR|S_IROTH|S_IWOTH|S_IRGRP|S_IWGRP)

- cocreate:tokens
{
  int ok;
  int fd;
  int i,j,n;
  char **argv;

  if (!okvsigpath) makeokvsigpath();
  if (!coroutines) coroutines = [OrdCltn new];

  for(i=1,n=[tokens size];i<n;i++) {
    char *s = [[tokens at:i] str];
    if (!strcmp(s,"-r")) { rpath=NARG;continue; }
    if (!strcmp(s,"-w")) { wpath=NARG;continue; }
    if (!strcmp(s,"-i")) { identifier=NARG;continue; }
    if (!strcmp(s,"-R")) { refname=NARG;continue; }
    if (!strcmp(s,"-e")) { expectstring=NARG;continue; }
    if (!strcmp(s,"-s")) { sendstring=NARG;continue; }
    break;
  }

  if (i < n) {
    char *t = getenv("TMPDIR");
    if (!t) t = P_tmpdir;
    if (!identifier) identifier=[[tokens at:i] copy];
    if (!rpath) rpath=[String sprintf:"%s/r%s.%d",t,[identifier str],getpid()];
    if (!wpath) wpath=[String sprintf:"%s/w%s.%d",t,[identifier str],getpid()];

    mkfifo([rpath str],0666);
    mkfifo([wpath str],0666);

    [coroutines add:self]; /* to unlink at exit */

    pid = fork();
    if (pid) {
       rfd = open([rpath str],O_RDONLY,FIFOMODS);
       if (rfd == -1) {
         [self errwrite:"cursel: "];
         [self errwrite:strerror(errno)];
       }
       wfd = open([wpath str],O_WRONLY|O_CREAT|O_TRUNC,FIFOMODS);
       if (wfd == -1) {
         [self errwrite:"cursel: "];
         [self errwrite:strerror(errno)];
       }
    } else {
       /* alternatively could set close-on-exec via fcntl */
       close(srcfd);
       if (dstfd>0) close(dstfd);

       n -= i;
       argv = (char**)malloc(sizeof(char*) * (n+1));
       for(j=0;j<n;j++) {
          argv[j]  = [[tokens at:i+j] str];
       }
       argv[n] = NULL;

       fd=open([rpath str],O_WRONLY|O_CREAT|O_TRUNC,FIFOMODS);
       if (fd == -1) {
         [self errwrite:"cursel: "];
         [self errwrite:strerror(errno)];
       } else {
         dup2(fd,1);
       }
       fd=open([wpath str],O_RDONLY,FIFOMODS);
       if (fd == -1) {
         [self errwrite:"cursel: "];
         [self errwrite:strerror(errno)];
       } else {
         dup2(fd,0);
       }

       if (errfd>0) close(errfd);
       dup2(1,2);

       [Frame unblockalarm];
       signal(SIGINT,(u_interrupt)?SIG_DFL:SIG_IGN);
       ok = execvp(argv[0],argv);
       if (ok == -1) {
         sfwrite(2,"cursel: ");
         sfwrite(2,argv[0]);
         sfwrite(2,": ");
         sfwrite(2,strerror(errno));
       }
       _exit(0);
    }

    return self;
  } else {
    return nil;
  }
}

- identifier
{
  return identifier;
}

+ findidentifier:tokens
{
  int i,n;
  id search = nil;
  blockoption = YES;
  for(i=1;i<n;i++) {
    char *t = [[tokens at:i] str];
    if (!strcmp(t,"-n")) { blockoption=NO; continue; }
    break;
  }
  if (i<n) search = [tokens at:i]; /* identifier */
  for(i=0,n=[coroutines size];i<n;i++) {
    id c = [coroutines at:i];
    if ([search isEqual:[c identifier]]) return c;
  }
  return nil;
}

- cosend:tokens
{
  id s;
  int n;
  char *e = "\n";
  if (sendstring) e = [sendstring str];
  s = [tokens lastElement];
  if (![s isEqual:identifier]) {
    dbg("cosend %s\n",[s str]);
    n = write(wfd,[s str],[s size]);
    assert(n == [s size]);
    n = write(wfd,e,strlen(e));
    assert(n == strlen(e));
    return (blockoption)?[self coreceive:nil]:self;
  } else {
    return self;
  }
}

- coreceive:tokens
{
  int n;
  id res;
  char buf[2];
  char *e = "\n";

  buf[1] = '\0';
  res = [String new];
  if (expectstring) e = [expectstring str];
  while ((1 == read(rfd,buf,1))) {
    if (!strcmp(buf,e)) break;
    [res concatSTR:buf];
  }
  
  [self dstwrite:[res str]];
  dbg("coreceive %s\n",[res str]);

  return self;
}

- cocheck:tokens
{
  int n;
  char *res;
  fd_set pending;
  struct timeval t;

  FD_ZERO(&pending);
  FD_SET(rfd,&pending);  
  
  t.tv_sec = 0; /* dont block */
  t.tv_usec = 0; /* dont block */
  n = select(rfd + 1,&pending,NULL,NULL,&t);

  [self dstwrite:(n>0)?"true":"false"];
  return self;
}

- destroy
{
  char *e;
  int status;

  /* install our SIGCLD handler */
  [self setsigchldhandler];

  /* close our read end to trigger a SIGPIPE for child when it's writing*/
  if (rfd>0) rfd=close(rfd);

  /* write something to child to trigger it to write to our (closed) read end */
  e = (sendstring)?[sendstring str]:"\n";
  if (wfd>0) {
     write(wfd,e,strlen(e));
     wfd = close(wfd);
  }

  /* tell child this is a good time to wake up */
  startvsig();
  stopvsig();

  return self;
}

- codestroy:tokens
{
  dbg("codestroy\n");
  [self destroy];
  [self unlink];
  [coroutines remove:self];
  return self;
}

- unlink
{
  unlink([rpath str]);
  unlink([wpath str]);
  return self;
}

@end
 
