/***************************************************************************
                          knoten.h  -  description
                             -------------------
    begin                : Sun Jul 1 2001
    copyright            : (C) 2001 by Immi
    email                : cuyo@karimmi.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/* Um Cuyo mit stl_map zu testen die nachfolgende Zeile einkommentieren. */
//#define map_stl 1


#ifndef KNOTEN_H
#define KNOTEN_H

#include <qstring.h>


#include "inkompatibel.h"


/* Wichtig: inkompatibel.h muss _vorher_ included werden! */
#if map_stl
/* Bei QT1 gibt's kein qmap.h; also stl-map verwenden. */
#include <map>
#else
#include <qmap.h>
#endif


#include <qlist.h>
#include <qarray.h>

#include "fehler.h"
#include "definition.h"


#define type_egal 0  // wird von DatenDatei::getEintragKnoten() benutzt
#define type_DefKnoten 1
#define type_ListenKnoten 2
#define type_WortKnoten 3

#define namespace_prozedur 0
#define namespace_variable 1


/* Die Tiefen der DefKnoten. */
#define tiefe_global 0
#define tiefe_level 1
#define tiefe_sorte 2



class Code;





/***********************************************************************/
/* Knoten */


/** Wenn die Level.descr geparst wird, wird das Ergebnis als ein
    Baum von Knoten gespeichert. Fr den Baum gilt (im Moment) folgendes:
    - Die Wurzel ist ein DefKnoten.
    - Kinder von DefKnoten sind DefKnoten oder ListenKnoten.
    - Kinder von ListenKnoten sind WortKnoten.
*/
class Knoten {


  /** Datei, in der dieser Knoten definiert wurde (fr Fehlermeldungen) */
  __String mDateiName;
  /** Zeilen-Nr, in der dieser Knoten definiert wurde (fr Fehlermeldungen) */
  int mZeilenNr;

public:
  Knoten(__String datna, int znr): mDateiName(datna), mZeilenNr(znr) {}
  
  virtual ~Knoten() {}
  
  virtual int type() const = 0;
  
  virtual __String toString() const = 0;

  /** Liefert einen String zurck, der angibt, wo dieser Code
      definiert wurde (fr Fehlermeldungen) */
  __String getDefString() const;
};






/***********************************************************************/
/* DefKnoten */



#if map_stl
/* Bei QT1 gibt's kein qmap.h; also stl-map verwenden. */


/* Auf QT-Srings gibt's keinen STL-kompatiblen <-Operator */
class less__String {
public:
  bool operator()(const QString & a, const QString & b) {
    return a < b.data();
  }
};


typedef map<__String, Knoten *, less__String> tKnotenMap;
typedef map<__String, Definition *, less__String> tCodeSpeicher;

#else 


typedef QMap<__String, Knoten *> tKnotenMap;
typedef QMap<__String, Definition *> tCodeSpeicher;

#endif



/** Knoten der Form
    bla=...
    blub=...
    <<
    bild=...
    gras=...
    >>
    Das was rechts von bla und blub steht, sind Kinder:
    Entweder DefKnoten oder ListenKnoten.
    Das, was in <<...>> steht, ist Code. (Neu: Code ist auch Knoten.)
*/
class DefKnoten: public Knoten {

  /** Die Knoten-Kinder dieses Knotens */
  tKnotenMap mKinder;
  
  /** Die Code-"Kinder" dieses Knotens */
  tCodeSpeicher mCodeSpeicher[2];
  
  /** Wenn man von hier aus im Baum nach oben luft der nchste
      DefKnoten, den man trifft. (Wird bentigt, um auf den
      CodeSpeicher vom Vater zuzufreifen.) */
  DefKnoten * mVater;
  
  /** 0 bei der Wurzel, 1 bei Kindern der Wurzel (Level-DefKnoten), etc. */
  int mTiefe;
  
  /** Nur fr den Top-Knoten: 1, wenn schon eine Level-Defintion
      gespeichert ist. Wenn danach noch Cual-Code kommt, wird eine Warnung
      ausgeben.
      3, wenn die Warnung schon ausgegeben wurde. */
  int mErstLevelDannCual;


  /** Variablen werden intern durchnummeriert. Hier steht die
      nchste freie Nummer (bzw. die Anzahl der belegten Nummern).
      
      Bei Sorten-Defknoten sind diese Variablen nicht von Bedeutung;
      um die Nummerierung der Sorten-Variablen kmmert sich auch der
      Level-DefKnoten. So hat intern jede Sorte alle Variablen.
      (Die Variablen-Definitionen stehen allerdings trotzdem im
      Code-Speicher der Sorten, so dass die Sortenvariablen-Namespace
      getrennt ist.
      
      berblick ber die ganzen Speizal-Variablen-Nummern siehe
      definition.h, bei class VarDefinition. */
  int mVarNrBei;
  /** Nummer der nchsten freien Bool-Variable. -1, wenn's grad
      keine freien Bool-Variablen gibt. */
  int mBoolNrBei;
  
  /** Die Default-Werte der Variablen, nochmal nach Nummer aufgelistet.
      Wird von den Blops bentigt, wenn die Variablen am Anfang
      initialisiert werden. Allerdings nur Default-Werte von echten
      Variablen (d. h. von Variablen, die Speicher belegen.) */
  QArray<int> mDefaultWerte;



  
public:
  
  /** Erzeugt den Top-Knoten. */
  DefKnoten();
  
  /** Erzeugt einen Unter-Knoten. */
  DefKnoten(__String datna, int znr, DefKnoten * vater);
  
  virtual ~DefKnoten();
  
  virtual int type() const {return type_DefKnoten;}

  virtual __String toString() const;
  
  void fuegeEin(const __String & na, Knoten * wert);
  
  /** Lscht alle Kinder raus, die DefKnoten sind und nicht
      "Title" heien.
      Wird von LevelDaten::ladLevelConf() gebraucht. */
  void loeschAlleLevel();
  
  bool enthaelt(__String na) {
     return mKinder.find(na) != mKinder.end();
  }
  
  Knoten * getKind(__String na) {
    CASSERT(mKinder.find(na) != mKinder.end());
    return mKinder[na];
  }
  
  
  /***** Methoden fr den Codespeicher *****/


  /** Speichert alle vordefinierten Variablen in den
      Namespace, auer die pics-Konstanten. Wird vom Constructor
      des WurzelKnotens aufgerufen. */
  void speicherGlobaleVordefinierte();

  /** Speichert die Pics-Konstanten. (picsliste sollte der pics-Knoten sein.)
      Wird von fuegeEin(...) aufgerufen, wenn es die pics bekommt. */
  void speicherPicsConst(Knoten * picsliste);

  /** Speichert eine Konstante mit dem Namen, der in nameKnoten steht und
      dem angegebenen Wert. nameKnoten ist hoffentlich ein ListenKnoten
      mit genau einem Eintrag. Wird von fuegeEin() aufgerufen, um die
      Gras-, die Grau- und die nix-Konstante abzuspeichern, wenn es die
      bekommt. */
  void speicherKnotenConst(Knoten * nameKnoten, int wert);

  
  /* Erzeugt eine neue Var-Definition und speichert sie ab. Dabei
     bekommt sie auch gleich eine Nummer. (Aufzurufen, wenn eine
     VarDefinition geparst wurde.) def ist der Default-Wert. */  
  void neueVarDefinition(const __String & na, int def);

  /* Speichert eine neue Definition - Code oder Variable. Noch unschn:
     Sollte von auen nur fr Code aufgerufen werden. Bei Variablen immer
     neueVarDefinition verwenden! */
  void speicherDefinition(int ns, const __String & na, Definition * f);
  
  /** Liefert eine Code-Definition aus diesem Speicher oder von
      weiter oben. Throwt bei nichtexistenz.
      Achtung: Behlt den Besitz an der Defintion. */
  Definition * getDefinition(int ns, __String na);
  


  
  /***** Variablen-Nummern-Verwaltung *****/


  
  /** Erzeugt eine unbenannte Variable und liefert die Nummer zurck.
      def wird als default-Wert in mDefaultWerte gespeichert. */
  int neueVariable(int def);

  /** Erzeugt eine unbenannte Bool-Variable und liefert
      die Nummer zurck. */
  int neueBoolVariable();
  
  
  /** Liefert zurck, wie viel Speicher fr Variablen jeder Blop
      reservieren muss. Nur auf Level-Ebene aufrufen. */
  int getDatenLaenge() const;
  
  /** Liefert den Default-Wert der Variable mit Nummer nr. Es
      muss aber eine richtige Variable sein, die echten Blop-
      Speicherplatz verbraucht. (Sonst soll man sich den Default-
      Wert aus der VarDefinition holen. Das hier ist nur fr
      Variablen-Anfangs-Initialisierung.) */
  int getDefaultWert(int nr) const;
  
  
};



/***********************************************************************/
/* ListenKnoten */


typedef QList<Knoten> tKnotenListe;


/** Knoten der Form bla1, bla2, bla3. Kinder sind WortKnoten. */
class ListenKnoten: public Knoten {

  /** mutable ist ntig, da die aktuelle Pos. gendert wird */
  mutable tKnotenListe mKinder;

public:

  ListenKnoten(__String datna, int znr): Knoten(datna, znr) {
    mKinder.setAutoDelete(true);
  }
  
  virtual int type() const {return type_ListenKnoten;}

  virtual __String toString() const;

  void fuegeEin(Knoten * wert) {
    mKinder.append(wert);
  }
  
  
  int getLaenge() const {
    return mKinder.count();
  }
  
  
  Knoten * getKind(int nr) {
    return mKinder.at(nr);
  }

  __String getWort(int nr);


  

};





/***********************************************************************/
/* WortKnoten */


/** Ein Wort-Knoten. "bla" oder bla. */
class WortKnoten: public Knoten {
  __String mWort;
  
public:

  WortKnoten(__String datna, int znr, __String w):
    Knoten(datna, znr), mWort(w) {}

  virtual int type() const {return type_WortKnoten;}

  virtual __String toString() const;

  __String getWort() {return mWort;}
  
};




#endif

