/* PlaylistController.m - this file is part of Cynthiune
 *
 * Copyright (C) 2002-2004  Wolfgang Sourdeau
 *
 * Author: Wolfgang Sourdeau <wolfgang@contre.com>
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#import <AppKit/AppKit.h>

#import "NSNumberExtensions.h"

#import "FormatTester.h"
#import "Preference.h"
#import "GeneralPreference.h"
#import "InfoDisplayController.h"
#import "Player.h"
#import "Playlist.h"
#import "PlaylistController.h"
#import "Song.h"
#import "TableViewController.h"
#import "utils.h"

#define LOCALIZED(X) NSLocalizedString (X, nil)

@implementation PlaylistController : NSObject

- (id) init
{
  if ((self = [super init]))
    {
      playlist = [Playlist new];
      [playlist setDelegate: self];
      player = [Player new];
      [player setDelegate: self];
      [player setMuted: [muteButton state]];
      playlistFilename = nil;
      timer = nil;
      timerShouldBeReset = NO;
      currentPlayerSong = nil;
    }

  return self;
}

- (void) _initButtonImages
{
  [previousButton setImage: [NSImage imageNamed: @"previous"]];
  [previousButton setAlternateImage: [NSImage imageNamed: @"previous-pushed"]];
  [playButton setImage: [NSImage imageNamed: @"play"]];
  [playButton setAlternateImage: [NSImage imageNamed: @"play-pushed"]];
  [pauseButton setImage: [NSImage imageNamed: @"pause"]];
  [pauseButton setAlternateImage: [NSImage imageNamed: @"pause-pushed"]];
  [stopButton setImage: [NSImage imageNamed: @"stop-pushed"]];
  [stopButton setAlternateImage: [NSImage imageNamed: @"stop-pushed"]];
  [nextButton setImage: [NSImage imageNamed: @"next"]];
  [nextButton setAlternateImage: [NSImage imageNamed: @"next-pushed"]];
  [ejectButton setImage: [NSImage imageNamed: @"eject"]];
  [ejectButton setAlternateImage: [NSImage imageNamed: @"eject-pushed"]];
  [muteButton setImage: [NSImage imageNamed: @"mute-pushed"]];
  [muteButton setAlternateImage: [NSImage imageNamed: @"mute"]];
  [repeatButton setImage: [NSImage imageNamed: @"repeat"]];
  [repeatButton setAlternateImage: [NSImage imageNamed: @"repeat-pushed"]];
  [shuffleButton setImage: [NSImage imageNamed: @"shuffle"]];
  [shuffleButton setAlternateImage: [NSImage imageNamed: @"shuffle-pushed"]];
}

- (void) _initPlaylistPopups
{
  NSRect frameBefore, frameAfter;
  NSMenu *menu;
  float delta;

  frameBefore = [addPopup frame];
  [addPopup removeAllItems];
  menu = [addPopup menu];
#ifdef __MACOSX__
  [menu addItemWithTitle: LOCALIZED (@"Add") action: NULL keyEquivalent: @""];
#endif
  [menu addItemWithTitle: LOCALIZED (@"Add") action: NULL keyEquivalent: @""];
  [menu addItemWithTitle: LOCALIZED (@"+ dir") action: NULL keyEquivalent: @""];
  [addPopup selectItemAtIndex: 0];
  [addPopup sizeToFit];
  frameAfter = [addPopup frame];
  delta = frameAfter.size.width - frameBefore.size.width;

  frameBefore = [removePopup frame];
  [removePopup removeAllItems];
  menu = [removePopup menu];
#ifdef __MACOSX__
  [menu addItemWithTitle: LOCALIZED (@"Remove") action: NULL keyEquivalent: @""];
#endif
  [menu addItemWithTitle: LOCALIZED (@"Remove") action: NULL keyEquivalent: @""];
  [menu addItemWithTitle: LOCALIZED (@"- all") action: NULL keyEquivalent: @""];
  [menu addItemWithTitle: LOCALIZED (@"Cleanup") action: NULL keyEquivalent: @""];
  [removePopup selectItemAtIndex: 0];
  [removePopup sizeToFit];
  frameAfter = [removePopup frame];
  frameAfter.origin.x += delta;
  [removePopup setFrame: frameAfter];
  delta += frameAfter.size.width - frameBefore.size.width;

  frameBefore = [savePopup frame];
  [savePopup removeAllItems];
  menu = [savePopup menu];
#ifdef __MACOSX__
  [menu addItemWithTitle: LOCALIZED (@"Save")
        action: NULL
        keyEquivalent: @""];
#endif
  [menu addItemWithTitle: LOCALIZED (@"Save")
        action: NULL
        keyEquivalent: @""];
  [menu addItemWithTitle: LOCALIZED (@"... as...")
        action: NULL
        keyEquivalent: @""];
  [savePopup selectItemAtIndex: 0];
  [savePopup sizeToFit];
  frameAfter = [savePopup frame];
  frameAfter.origin.x += delta;
  [savePopup setFrame: frameAfter];

  [[addPopup superview] setNeedsDisplay: YES];
}

- (void) awakeFromNib
{
  [repeatMenuItem setState: NSOffState];
  [shuffleMenuItem setState: NSOffState];
  [muteMenuItem setState: NSOffState];

  [playlist loadFromDefaults];
  [tableViewController setDelegate: self];
  [tableViewController setPlaylist: playlist];
}

- (void) initializeWidgets
{
  [progressSlider setEnabled: NO];
  [progressSlider setMinValue: 0.0];
  [progressSlider setIntValue: 0];

  [self _initButtonImages];
  [self _initPlaylistPopups];

  [pauseButton setEnabled: NO];
  [stopButton setEnabled: NO];

  [infoDisplayController initializeWidgets];
}

- (void) toggleMute: (id) sender
{
  BOOL newState;

  newState = ![player muted];
  [player setMuted: newState];
  if (newState)
    {
      [muteMenuItem setState: NSOnState];
      [muteButton setState: NSOnState];
    }
  else
    {
      [muteMenuItem setState: NSOffState];
      [muteButton setState: NSOffState];
    }
}

- (void) toggleShuffle: (id) sender
{
  BOOL newState;

  newState = ![playlist shuffle];
  [playlist setShuffle: newState];
  if (newState)
    {
      [shuffleMenuItem setState: NSOnState];
      [shuffleButton setState: NSOnState];
    }
  else
    {
      [shuffleMenuItem setState: NSOffState];
      [shuffleButton setState: NSOffState];
    }
}

- (void) toggleRepeat: (id) sender
{
  BOOL newState;

  newState = ![playlist repeat];
  [playlist setRepeat: newState];
  if (newState)
    {
      [repeatMenuItem setState: NSOnState];
      [repeatButton setState: NSOnState];
    }
  else
    {
      [repeatMenuItem setState: NSOffState];
      [repeatButton setState: NSOffState];
    }
}

- (void) _initProgressSliderFromSong: (Song *) aSong
{
  [progressSlider setIntValue: 0];
  [progressSlider setMaxValue: [[aSong duration] doubleValue]];
  [progressSlider setEnabled: [aSong isSeekable]];
}

- (void) _updatePlayer: (BOOL) force
{
  Song *currentSong;
  int currentSongNumber;

  currentSongNumber = [playlist currentSongNumber];
  currentSong = [playlist songAtIndex: currentSongNumber];
  if (force || currentSong != currentPlayerSong)
    {
      [self _initProgressSliderFromSong: currentSong];
      [player setStream: [currentSong openStreamForSong]];
      currentPlayerSong = currentSong;
      if ([player paused])
        [player setPaused: NO];
    }
  currentSong = currentPlayerSong;
  if ([player playing] && currentSong)
    {
      [infoDisplayController updateInfoFieldsFromSong: currentSong];
      [infoDisplayController setTimerFromSeconds: 0];
    }
  [tableViewController updateView];
}

- (void) _addFileIfAcceptable: (NSString *) filename
{
  FormatTester *formatTester;
  Song *newSong;

  if (fileIsAcceptable (filename))
    {
      if ([filename hasSuffix: @"pls"]
          || [filename hasSuffix: @"m3u"])
        [playlist loadFromFile: filename];
      else
        {
          formatTester = [FormatTester formatTester];
          if ([formatTester formatNumberForFile: filename] > -1)
            {
              newSong = [Song songWithFilename: filename];
              [playlist appendSong: newSong];
            }
        }
    }
}

- (void) _addSongsFromArray: (NSArray *) songArray
                    withDir: (NSString *) aDirectory
{
  NSString *filename, *songFilename;
  int max, count;

  max = [songArray count];
  for (count = 0; count < max; count++)
    {
      songFilename = [songArray objectAtIndex: count];
      filename = (aDirectory)
        ? [aDirectory stringByAppendingPathComponent: songFilename]
        : songFilename;
      [self _addFileIfAcceptable: filename];
    }
}

- (void) _addSongsFromOPanel: (NSOpenPanel*) oPanel
{
  [self _addSongsFromArray: [oPanel filenames] withDir: nil];
  [tableViewController updateView];
  [playlist saveToDefaults];
}

- (void) _oPanelDidEnd: (NSOpenPanel *) oPanel
            returnCode: (int) result
           contextInfo: (void *) contextInfo
{
  NSUserDefaults *userDefaults;

  if (result == NSOKButton)
    {
      userDefaults = [NSUserDefaults standardUserDefaults];
      [userDefaults setObject: [oPanel directory]
                    forKey: @"NSDefaultOpenDirectory"];
      [userDefaults synchronize];
      [self _addSongsFromOPanel: oPanel];
    }
}

- (void) _addDirSongsFromOPanel: (NSOpenPanel *) oPanel
{
  int count, max;
  NSArray *dirs;
  NSDirectoryEnumerator *dirEnum;
  NSString *dir;

  dirs = [oPanel filenames];
  max = [dirs count];

  for (count = 0; count < max; count++)
    {
      dir = [dirs objectAtIndex: count];
#ifdef GNUSTEP
      dirEnum = [[NSDirectoryEnumerator alloc]
                  initWithDirectoryPath: dir
                  recurseIntoSubdirectories: YES
                  followSymlinks: NO
                  justContents: NO];
#else
      dirEnum = [[NSFileManager defaultManager] enumeratorAtPath: dir];
#endif
      [self _addSongsFromArray: [dirEnum allObjects]
            withDir: dir];
    }

  [tableViewController updateView];
  [playlist saveToDefaults];
}

- (void) _dirOPanelDidEnd: (NSOpenPanel *) oPanel
               returnCode: (int) result
              contextInfo: (void *) contextInfo
{
  NSUserDefaults *userDefaults;

  if (result == NSOKButton)
    {
      userDefaults = [NSUserDefaults standardUserDefaults];
      [userDefaults setObject: [oPanel directory]
                    forKey: @"NSDefaultOpenDirectory"];
      [userDefaults synchronize];
      [self _addDirSongsFromOPanel: oPanel];
    }
}

- (void) _runOpenPanelWithDidEndSelector: (SEL) selector
{
  NSOpenPanel *oPanel;
  NSString *openDir;

  openDir = [[NSUserDefaults standardUserDefaults]
              stringForKey: @"NSDefaultOpenDirectory"];

  oPanel = [NSOpenPanel openPanel];
  [oPanel setAllowsMultipleSelection: YES];
  [oPanel setCanChooseDirectories: NO];
  [oPanel setTitle: LOCALIZED (@"Add music files...")];
#ifdef GNUSTEP
  [oPanel setDirectory: openDir];
#endif
  [oPanel setDelegate: self];
  [oPanel beginSheetForDirectory: openDir
          file: nil
          types: nil
          modalForWindow: [NSApp mainWindow]
          modalDelegate: self
          didEndSelector: selector
          contextInfo: nil];
}

- (void) addSongs: (id) sender
{
  [self _runOpenPanelWithDidEndSelector: @selector(_oPanelDidEnd:returnCode:contextInfo:)];
}

- (void) addSongsFolders: (id) sender
{
  NSString *openDir;
  NSOpenPanel *oPanel;

  openDir = [[NSUserDefaults standardUserDefaults]
              stringForKey: @"NSDefaultOpenDirectory"];

  oPanel = [NSOpenPanel openPanel];
  [oPanel setAllowsMultipleSelection: YES];
  [oPanel setCanChooseDirectories: YES];
  [oPanel setCanChooseFiles: NO];
  [oPanel setTitle: LOCALIZED (@"Add a music folder...")];
#ifdef GNUSTEP
  [oPanel setDirectory: openDir];
#endif
  [oPanel beginSheetForDirectory: openDir
          file: nil
          types: nil
          modalForWindow: [NSApp mainWindow]
          modalDelegate: self
          didEndSelector: @selector(_dirOPanelDidEnd:returnCode:contextInfo:)
          contextInfo: nil];
}

- (void) removeSelectedSongs: (id) sender
{
  NSArray *selection;
  Song *currentSong;
  int currentSongNumber;

  selection = [tableViewController getSelectedSongs];
  if ([selection count])
    {
      currentSong = ((player && [player playing])
                     ? currentPlayerSong
                     : [selection objectAtIndex: 0]);
      currentSongNumber = [playlist songIndex: currentSong];
      [playlist selectSongNumber: currentSongNumber];
      if ([selection containsObject: currentSong])
        [playlist nextSong];
      [playlist removeArrayOfSongs: selection];
    }
}

- (void) removeAllSongs: (id) sender
{
  [playlist removeAll];
}

- (void) cleanupPlaylist: (id) sender
{
  [playlist removeBadEntries];
}

- (void) _sPanelDidEnd: (NSSavePanel *) sPanel
            returnCode: (int) result
           contextInfo: (NSString *) extension
{
  NSString *saveDir, *filename;

  if (result == NSOKButton)
    {
      saveDir = [sPanel directory];
      [[NSUserDefaults standardUserDefaults]
        setObject: saveDir
        forKey: @"NSDefaultSaveDirectory"];

      if (playlistFilename)
        [playlistFilename release];
      playlistFilename = [[[sPanel filename] lastPathComponent]
                           stringByDeletingPathExtension];
      [playlistFilename retain];

      filename = [playlistFilename stringByAppendingPathExtension: extension];
      [playlist
        saveToFile: [saveDir stringByAppendingPathComponent: filename]];
    }
}

- (void) _runSaveListPanelWithExtension: (NSString *) extension
                             andSaveDir: (NSString *) saveDir
{
  NSSavePanel *sPanel;

  sPanel = [NSSavePanel savePanel];
  [sPanel setTitle: LOCALIZED (@"Save playlist as...")];
  [sPanel setRequiredFileType: extension];
  [sPanel setExtensionHidden: YES];
  [sPanel beginSheetForDirectory: saveDir
          file: playlistFilename
          modalForWindow: [NSApp mainWindow]
          modalDelegate: self
          didEndSelector: @selector (_sPanelDidEnd:returnCode:contextInfo:)
          contextInfo: extension];
}

- (void) saveList: (id) sender
{
  NSString *extension, *filename, *saveDir;
  NSUserDefaults *userDefaults;

  if (!playlistFilename)
    [self saveListAs: sender];
  else
    {
      extension = [[GeneralPreference instance] preferredPlaylistFormat];
      userDefaults = [NSUserDefaults standardUserDefaults];
      saveDir = [userDefaults stringForKey: @"NSDefaultSaveDirectory"];
      if (!saveDir || [saveDir isEqualToString: @""])
        saveDir = [userDefaults stringForKey: @"NSDefaultOpenDirectory"];

      filename = [playlistFilename stringByAppendingPathExtension: extension];
      [playlist
        saveToFile: [saveDir stringByAppendingPathComponent: filename]];
    }
}

- (void) saveListAs: (id) sender
{
  NSString *extension, *saveDir;
  NSUserDefaults *userDefaults;

  extension = [[GeneralPreference instance] preferredPlaylistFormat];
  userDefaults = [NSUserDefaults standardUserDefaults];
  saveDir = [userDefaults stringForKey: @"NSDefaultSaveDirectory"];
  if (!saveDir || [saveDir isEqualToString: @""])
    saveDir = [userDefaults stringForKey: @"NSDefaultOpenDirectory"];

  [self _runSaveListPanelWithExtension: extension andSaveDir: saveDir];
}

/* Popup menus */
- (void) addSong: (id) sender
{
  NSString *title;

  title = [sender titleOfSelectedItem];

  if ([title isEqualToString: LOCALIZED (@"Add")])
    [self addSongs: sender];
  else if ([title isEqualToString: LOCALIZED (@"+ dir")])
    [self addSongsFolders: sender];
  else
    NSLog (@"undefined action in 'Add' menu for entry '%@'", title);
}

- (void) removeSong: (id) sender
{
  NSString *title;

  title = [sender titleOfSelectedItem];

  if ([title isEqualToString: LOCALIZED (@"Remove")])
    [self removeSelectedSongs: sender];
  else if ([title isEqualToString: LOCALIZED (@"- all")])
    [self removeAllSongs: sender];
  else if ([title isEqualToString: LOCALIZED (@"Cleanup")])
    [self cleanupPlaylist: sender];
  else
    NSLog (@"undefined action in 'Remove' menu for entry '%@'", title);
}

- (void) saveListFromPopupMenu: (id) sender
{
  NSString *title;

  title = [sender titleOfSelectedItem];

  if ([title isEqualToString: LOCALIZED (@"Save")])
    [self saveList: sender];
  else if ([title isEqualToString: LOCALIZED (@"... as...")])
    [self saveListAs: sender];
  else
    NSLog (@"undefined action in 'Save' menu for entry '%@'", title);
}

- (void) addSongFromNSApp: (id) sender
{
  [self _runOpenPanelWithDidEndSelector:
          @selector(_oPanelDidEnd:returnCode:contextInfo:)];
}

- (BOOL) openSongFromNSApp: (Song *) aSong
{
  BOOL result;
  FormatTester *formatTester;

  formatTester = [FormatTester formatTester];
  if ([formatTester formatNumberForFile: [aSong filename]])
    {
      [playlist appendSong: aSong];
      [tableViewController updateView];
      result = YES;
    }
  else
    result = NO;

  return result;
}

- (void) _realStartPlaying
{
  if ([playlist count])
    {
      if ([playlist currentSongNumber] < 0)
        [playlist firstSong];
      if ([playlist shuffle])
        [playlist reShuffle];
      [self _updatePlayer: NO];
      [player play];
    }
}

- (void) _startOPanelDidEnd: (NSOpenPanel *) oPanel
                 returnCode: (int) result
                contextInfo: (void *) contextInfo
{
  NSUserDefaults *userDefaults;

  if (result == NSOKButton)
    {
      userDefaults = [NSUserDefaults standardUserDefaults];
      [userDefaults setObject: [oPanel directory]
                       forKey: @"NSDefaultOpenDirectory"];
      [userDefaults synchronize];
      [self _addSongsFromOPanel: oPanel];
      [self _realStartPlaying];
    }
}

- (void) startPlayer: (id) sender
{
  if (![playlist count])
    [self _runOpenPanelWithDidEndSelector: @selector(_startOPanelDidEnd:returnCode:contextInfo:)];
  [self _realStartPlaying];
}

- (void) pausePlayer: (id) sender
{
  if ([player playing])
    [player setPaused: ![player paused]];
}

- (void) previousSong: (id) sender
{
  int currentSongNumber;

  currentSongNumber = [playlist currentSongNumber];
  [playlist prevSong];
  if (currentSongNumber != [playlist currentSongNumber])
    [self _updatePlayer: NO];
  else
    [tableViewController updateView];
}

- (void) nextSong: (id) sender
{
  int currentSongNumber;

  currentSongNumber = [playlist currentSongNumber];
  [playlist nextSong];
  if (currentSongNumber != [playlist currentSongNumber])
    [self _updatePlayer: NO];
  else
    [tableViewController updateView];
}

- (void) stopPlayer: (id) sender
{
  [player stop];
}

- (void) _ejectOPanelDidEnd: (NSOpenPanel *) oPanel
                 returnCode: (int) result
                contextInfo: (void *) contextInfo
{
  NSUserDefaults *userDefaults;

  if (result == NSOKButton)
    {
      userDefaults = [NSUserDefaults standardUserDefaults];
      [userDefaults setObject: [oPanel directory]
                    forKey: @"NSDefaultOpenDirectory"];
      [userDefaults synchronize];
      [userDefaults synchronize];
      if ([player playing])
        [player stop];
      [playlist removeAll];
      [self _addSongsFromOPanel: oPanel];
      [self _realStartPlaying];
    }
}

- (void) eject: (id) sender
{
  [self _runOpenPanelWithDidEndSelector: @selector(_ejectOPanelDidEnd:returnCode:contextInfo:)];
}

/* when the timer field is used to display the total amount of seconds in the
   selection... */
- (void) _resetTimeDisplay: (NSTimer *) aTimer
{
  BOOL *passedOnce;

  passedOnce = [[aTimer userInfo] pointerValue];
  if (!*passedOnce)
    *passedOnce = YES;
  else
    {
      [timer invalidate];
      [timer release];
      timer = nil;
      [infoDisplayController resetTimerField];
      timerShouldBeReset = NO;
    }
}

- (void) _updateDisplayResetTimer
{
  static BOOL passedOnce;

  passedOnce = NO;
  if (timer)
    {
      [timer invalidate];
      [timer release];
    }
  timer = [NSTimer scheduledTimerWithTimeInterval: 1.0
                   target: self
                   selector: @selector (_resetTimeDisplay:)
                   userInfo: [NSValue valueWithPointer: &passedOnce]
                   repeats: YES];
  [timer fire];
  [timer retain];
}

- (void) _updateSelectionTimeDisplay
{
  NSArray *selection;
  NSNumber *duration;
  unsigned int totalSeconds, count, max;

  totalSeconds = 0;
  selection = [tableViewController getSelectedSongs];
  max = [selection count];

  for (count = 0; count < max; count++)
    {
      duration = [[selection objectAtIndex: count] duration];
      totalSeconds += [duration unsignedIntValue];
    }

  [infoDisplayController setTimerFromTotalSeconds: totalSeconds];
  timerShouldBeReset = YES;

  if (![player playing])
    [self _updateDisplayResetTimer];
}

/* tableview delegate */
- (void) tableDoubleClick: (NSNotification*) aNotification
{
  [self _updatePlayer: YES];
  if (![player playing])
    [self startPlayer: self];
}

- (void) tableSelectionDidChange: (NSNotification*) aNotification
{
  if (![player playing])
    [playlist selectSongNumber: [tableViewController getFirstSelectedRow]];
}

- (void) tableSelectionIsChanging: (NSNotification*) aNotification
{
  [self _updateSelectionTimeDisplay];
}

- (void) songCursorChange: (id) sender
{
  unsigned int time;

  time = [sender intValue];
  [infoDisplayController setTimerFromSeconds: time];
  [player seek: time];
}

- (void) changeTimeDisplay: (id) sender
{
  unsigned int time;

  if ([player playing])
    {
      time = [player timer];
      [infoDisplayController
        setReverseTimer: ![infoDisplayController timerIsReversed]];
      [infoDisplayController setTimerFromSeconds: time];
    }
}

- (void) _updateTimeDisplay
{
  unsigned int time;

  if (timerShouldBeReset)
    {
      [infoDisplayController resetTimerField];
      timerShouldBeReset = NO;
    }
  time = [player timer];
  [infoDisplayController setTimerFromSeconds: time];
  [progressSlider setIntValue: time];
}

- (void) _startTimeTimer
{
  NSRunLoop *runLoop;

  if (timer)
    [timer invalidate];
  timer = [NSTimer scheduledTimerWithTimeInterval: 1
                   target: self
                   selector: @selector (_updateTimeDisplay)
                   userInfo: nil
                   repeats: YES];
  runLoop = [NSRunLoop currentRunLoop];
  [runLoop addTimer: timer forMode: NSDefaultRunLoopMode];
  [runLoop addTimer: timer forMode: NSEventTrackingRunLoopMode];
  [timer retain];
}

- (void) _stopTimeTimer
{
  [timer invalidate];
  [timer release];
  timer = nil;
}

/* as a player delegate... */
- (void) playerPlaying: (NSNotification*) aNotification
{
  int currentSongNumber;
  double duration;
  Song *currentSong;

  currentSongNumber = [playlist currentSongNumber];
  currentSong = [playlist songAtIndex: currentSongNumber];
  [self _startTimeTimer];
  [infoDisplayController show];
  [infoDisplayController updateInfoFieldsFromSong: currentSong];
  [infoDisplayController setTimerFromSeconds: 0];
  duration = [[currentSong duration] doubleValue];
  [progressSlider setMaxValue: duration];

  [playButton setImage: [NSImage imageNamed: @"play-pushed"]];
  [playButton setEnabled: NO];

  [pauseButton setImage: [NSImage imageNamed: @"pause"]];
  [pauseButton setEnabled: YES];

  [stopButton setImage: [NSImage imageNamed: @"stop"]];
  [stopButton setAlternateImage: [NSImage imageNamed: @"stop-pushed"]];
  [stopButton setEnabled: YES];

  [tableViewController selectRow: [playlist songIndex: currentSong]];
}

- (void) playerPaused: (NSNotification*) aNotification
{
  [self _stopTimeTimer];
  [pauseButton setImage: [NSImage imageNamed: @"pause-pushed"]];
}

- (void) playerResumed: (NSNotification*) aNotification
{
  [self _startTimeTimer];
  [pauseButton setImage: [NSImage imageNamed: @"pause"]];
}

- (void) playerStopped: (NSNotification*) aNotification
{
  [self _stopTimeTimer];

  [playButton setImage: [NSImage imageNamed: @"play"]];
  [playButton setAlternateImage: [NSImage imageNamed: @"play-pushed"]];
  [playButton setEnabled: YES];
  [pauseButton setImage: [NSImage imageNamed: @"pause"]];
  [pauseButton setEnabled: NO];
  [stopButton setImage: [NSImage imageNamed: @"stop-pushed"]];
  [stopButton setEnabled: NO];

  [progressSlider setEnabled: NO];
  [progressSlider setMaxValue: 0];
  [infoDisplayController hide];
  [tableViewController updateView];

  currentPlayerSong = nil;
}

- (void) playerSongEnded: (NSNotification*) aNotification
{
  int currentSongNumber;

  currentSongNumber = [playlist currentSongNumber];
  [playlist nextSong];
  if ([playlist repeat])
    [self _updatePlayer: YES];
  else
    {
      if (currentSongNumber == [playlist currentSongNumber])
        {
          [playlist firstSong];
          [player stop];
        }
      else
        [self _updatePlayer: NO];
    }
}

/* Playlist delegate */
- (void) playlistChanged: (NSNotification *) aNotification
{
  [playlist saveToDefaults];

  [self _updatePlayer: NO];
}

/* NSOpenPanel delegate */
-       (BOOL) panel: (id) sender
  shouldShowFilename: (NSString *) fileName
{
  NSFileManager *fileManager;
  FormatTester *formatTester;
  BOOL isDir;

  fileManager = [NSFileManager defaultManager];
  formatTester = [FormatTester formatTester];

  return (([fileManager fileExistsAtPath: fileName isDirectory: &isDir]
           && isDir)
           || [formatTester extensionIsSupported: [fileName pathExtension]]);
}

/* Services */
- (id) validRequestorForSendType: (NSString *)sendType
                      returnType: (NSString *)returnType
{
  return ((sendType
           && [sendType isEqualToString: NSFilenamesPboardType]
           && ([tableViewController getFirstSelectedRow] > -1))
          ? self
          : nil);
}

- (BOOL) writeSelectionToPasteboard: (NSPasteboard*) pboard
                              types: (NSArray*) types
{
  NSArray *declaredTypes, *filenames;
  BOOL answer;

  if ([types containsObject: NSFilenamesPboardType])
    {
      declaredTypes = [NSArray arrayWithObject: NSFilenamesPboardType];
      [pboard declareTypes: declaredTypes owner: self];
      filenames = [tableViewController getSelectedSongsAsFilenames];
      answer = [pboard setPropertyList: filenames
                       forType: NSFilenamesPboardType];
    }
  else
    answer = NO;

  return answer;
}

@end
