/* Song.m - this file is part of Cynthiune
 *
 * Copyright (C) 2002, 2003 Wolfgang Sourdeau
 *
 * Author: Wolfgang Sourdeau <wolfgang@contre.com>
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#import <Foundation/Foundation.h>

#import "Format.h"
#import "FormatTester.h"
#import "Song.h"
#import "utils.h"

#define LOCALIZED(X) NSLocalizedString (X, nil)

@implementation Song : NSObject

+ (Song *) songWithFilename: (NSString *) aFilename
{
  Song *newSong;

  newSong = [[self alloc] initWithFilename: aFilename];
  [newSong autorelease];

  return newSong;
}

- (id) init
{
  if ((self = [super init]))
    {
      filename = nil;
      status = SongFileNotFound;
      isSeekable = NO;
      title = nil;
      artist = nil;
      album = nil;
      genre = nil; 
      trackNumber = nil;
      formatClass = nil;
      duration = nil;
      size = 0;
      date = nil;
    }

  return self;
}

- (id) initWithFilename: (NSString *) aFilename
{
  self = [self init];
  [self setFilename: aFilename];

  return self;
}

- (id) openStreamForSong
{
  id <Format> stream;
  FormatTester *formatTester;
  
  if (filename
      && [[NSFileManager defaultManager] fileExistsAtPath: filename])
    {
      if (!formatClass)
        {
          formatTester = [FormatTester formatTester];
          formatClass = [formatTester formatClassForFile: filename];
        }

      if (formatClass)
        {
          stream = [formatClass new];
          if ([stream streamOpen: filename])
            [stream autorelease];
          else
            {
              [stream release];
              stream = nil;
              status = SongStreamError;
            }
        }
      else
        {
          stream = nil;
          status = SongFormatNotRecognized;
        }
    }
  else
    {
      stream = nil;
      status = SongFileNotFound;
    }

  return stream;
}

- (BOOL) _fileWasModified
{
  NSFileManager *fileManager;
  NSDictionary *attributes;
  NSDate *newDate;
  unsigned long long newSize;
  BOOL modified;

  modified = NO;

  fileManager = [NSFileManager defaultManager];
  attributes = [fileManager fileAttributesAtPath: filename
                            traverseLink: YES];
  if (attributes)
    {
      newDate = [attributes fileModificationDate];
      newSize = [attributes fileSize];

      if (!(date && [date isEqualToDate: newDate]))
        {
          modified = YES;
          SET (date, newDate);
        }

      if (size != newSize)
        {
          modified = YES;
          size = newSize;
        }
    }
  else
    {
      status = SongFileNotFound;
      modified = YES;
    }

  return modified;
}

- (void) _refreshSongInfosFromStream: (id <Format>) stream
{
  isSeekable = [stream isSeekable];

  SET (title, [stream readTitle]);
  if ([title length] == 0)
    {
      title = [NSString stringWithFormat: @"[%@]",
                        [filename lastPathComponent]];
      [title retain];
    }
  SET (genre, [stream readGenre]);
  SET (artist, [stream readArtist]);
  SET (album, [stream readAlbum]);
  SET (trackNumber, [stream readTrackNumber]);
  SET (duration, [NSNumber numberWithUnsignedInt: [stream readDuration]]);
}

- (void) _readInfos
{
  id <Format> stream;

  if (fileIsAcceptable (filename))
    {
      if ([self _fileWasModified])
        {
          stream = [self openStreamForSong];

          if (stream)
            {
              [self _refreshSongInfosFromStream: stream];
              [stream streamClose];
              status = SongOK;
            }
        }
    }
  else
    status = SongFileNotFound;

  if (status != SongOK)
    {
      if (date)
        {
          [date release];
          date = nil;
        }
      size = 0;
    }
}

- (void) setFilename: (NSString *) aFilename
{
  SET (filename, aFilename);
  if (!fileIsAcceptable (filename))
    status = SongFileNotFound;
  else
    status = SongOK;
}

- (NSString *) filename
{
  RETURNSTRING (filename);
}

- (NSString *) shortFilename
{
  return ((filename)
          ? [NSString stringWithString: [filename lastPathComponent]]
          : @"");
}

- (NSNumber *) duration
{
  [self _readInfos];

  return duration;
}

- (NSString *) title
{
  [self _readInfos];

  RETURNSTRING (title);
}

- (NSString *) artist
{
  [self _readInfos];

  RETURNSTRING (artist);
}

- (NSString *) album
{
  [self _readInfos];

  RETURNSTRING (album);
}

- (NSString *) genre
{
  [self _readInfos];

  RETURNSTRING (genre);
}

- (NSString *) trackNumber
{
  [self _readInfos];

  RETURNSTRING (trackNumber);
}

- (NSString *) playlistRepresentation
{
  NSMutableString *string;

  string = nil;
  [self _readInfos];

  switch (status)
    {
    case SongOK:
      if ([trackNumber length])
        {
          string = [NSMutableString stringWithFormat: @"%@.", trackNumber];
          if ([artist length])
            [string appendString: @" "];
        }
      else
        string = [NSMutableString string];

      [string appendFormat: (([artist length] && [title length])
                             ? @"%@ - %@"
                             : @"%@%@"),
              title, artist];

      if ([album length])
        {
          if ([title length])
            [string appendString: @" "];
          [string appendFormat: @"(%@)", album];
        }
      break;
    case SongFileNotFound:
      string =
        [NSString stringWithFormat: LOCALIZED (@"[not found: %@]"), filename];
      break;
    case SongFormatNotRecognized:
      string =
        [NSString stringWithFormat:
                    LOCALIZED (@"[unknown format: %@]"), filename];
      break;
    case SongStreamError:
      string =
        [NSString stringWithFormat:
                    LOCALIZED (@"[format error: %@]"), filename];
      break;
    }

  return string;
}

- (SongStatus) status
{
  [self _readInfos];

  return status;
}

- (BOOL) isSeekable
{
  [self _readInfos];

  return isSeekable;
}

- (NSComparisonResult) compareByPlaylistRepresentation: (Song *) aSong
{
  NSComparisonResult result;

  result = [artist caseInsensitiveCompare: [aSong artist]];
  if (result == NSOrderedSame)
    {
      result = [album caseInsensitiveCompare: [aSong album]];
      if (result == NSOrderedSame)
        {
          result = compareStringsNumerically (trackNumber,
                                              [aSong trackNumber]);
          if (result == NSOrderedSame)
            {
              result = [title caseInsensitiveCompare: [aSong title]];
              if (result == NSOrderedSame)
                result = [[filename lastPathComponent]
                           caseInsensitiveCompare: [aSong filename]];
            }
        }
    }

  return result;
}

- (NSComparisonResult) reverseCompareByPlaylistRepresentation: (Song *) aSong
{
  return
    reverseComparisonResult ([self compareByPlaylistRepresentation: aSong]);
}

- (NSComparisonResult) compareByDuration: (Song *) aSong
{
  return [duration compare: [aSong duration]];
}

- (NSComparisonResult) reverseCompareByDuration: (Song *) aSong
{
  return reverseComparisonResult ([duration compare: [aSong duration]]);
}

- (void) dealloc
{
  RELEASEIFSET (filename);
  RELEASEIFSET (title);
  RELEASEIFSET (artist);
  RELEASEIFSET (album);
  RELEASEIFSET (genre); 
  RELEASEIFSET (trackNumber);
  RELEASEIFSET (duration);
  RELEASEIFSET (date);

  [super dealloc];
}

/* NSCoding protocol */
- (void) encodeWithCoder: (NSCoder *) encoder
{
  [self _readInfos];

  [encoder encodeObject: filename forKey: @"filename"];
  [encoder encodeObject: title forKey: @"title"];
  [encoder encodeObject: artist forKey: @"artist"];
  [encoder encodeObject: album forKey: @"album"];
  [encoder encodeObject: genre forKey: @"genre"];
  [encoder encodeObject: trackNumber forKey: @"trackNumber"];
  [encoder encodeObject: duration forKey: @"duration"];
  [encoder encodeObject: date forKey: @"date"];
  [encoder encodeBool: isSeekable forKey: @"isSeekable"];
  [encoder encodeInt: status forKey: @"status"];
  [encoder encodeInt64: size forKey: @"size"];
}

- (id) initWithCoder: (NSCoder *) decoder
{
  if ((self = [self init]))
    {
      SET (filename, [decoder decodeObjectForKey: @"filename"]);
      SET (title, [decoder decodeObjectForKey: @"title"]);
      SET (artist, [decoder decodeObjectForKey: @"artist"]);
      SET (album, [decoder decodeObjectForKey: @"album"]);
      SET (genre, [decoder decodeObjectForKey: @"genre"]);
      SET (trackNumber, [decoder decodeObjectForKey: @"trackNumber"]);
      SET (duration, [decoder decodeObjectForKey: @"duration"]);
      SET (date, [decoder decodeObjectForKey: @"date"]);
      isSeekable = [decoder decodeBoolForKey: @"isSeekable"];
      status = [decoder decodeIntForKey: @"status"];
      size = [decoder decodeIntForKey: @"size"];
    }

  return self;
}

@end
