
/* darkstat: a network traffic analyzer
 * (c) 2001-2003, Emil Mikulic.
 */

#include "spylog.h"
#include "port_db.h"

#include <pthread.h>
#include <stdio.h>
#include <time.h>
#include <pcap.h>
#include <string.h>
#include <netinet/in.h>

FILE *spylog = NULL;
time_t next_rotate = 0;
int spy_linktype = 0;
pcap_t *spy_cap = NULL;

#define FAIL() { up_spy = 0; pthread_exit( (void*)EXIT_FAIL ); }
#define SUCCEED() { up_spy = 0; pthread_exit( (void*)EXIT_SUCCESS ); }



static void spylog_rotate(time_t now)
{
	struct tm *tm = localtime(&now);
	char *fname;

	fname = build_str("%s" SPYLOG_FILE_NAME ".%04d%02d%02d",
			fprefix, 1900+tm->tm_year, 1+tm->tm_mon, tm->tm_mday);

	if (spylog) fclose(spylog);
	spylog = fopen(fname, "a");
	if (!spylog)
	{
		printf("Error: (spylog) can't append to %s\n", fname);
		freakout("fatal log rotation error");
	}
	free(fname);

	tm->tm_sec = tm->tm_min = tm->tm_hour = 0;
	next_rotate = mktime(tm) + 86400; /* tomorrow */
}



#define DISTANCE 12

static void spylog_packet(byte *packet, word length,
		dword srcip, dword destip, word destport)
{
	char method[6], *url, *host = NULL;
	int start = 0, end, found = 0;
	time_t now;
	struct tm *tm;

	if (length < 6) return; /* strlen("get /") */

	/*{int i;for(i=0;i<length;i++)if(isalnum(packet[i]))putchar(packet[i]
	  );else putchar('.');putchar(10);return;}*/

	/* it has to be HTTP, allow for <DISTANCE> bytes of garbage */
	while (start + 5 < length && start <= DISTANCE && !found)
	{
		if (strncasecmp(packet+start,  "GET ", 4) == 0 ||
		    strncasecmp(packet+start, "HEAD ", 5) == 0 ||
		    strncasecmp(packet+start, "POST ", 5) == 0)
			found = 1;
		else
			start++;
	}

	if (!found) return;

	/* -- find method */
	end = start;
	while (end < length && packet[end] != ' ') end++;
	memcpy(method, packet+start, end-start);
	method[end-start] = 0;

	/* -- find url */
	start = ++end;
	while (end < length && packet[end] != ' '
			    && packet[end] != '\r'
			    && packet[end] != '\n') end++;
	url = (char*)malloc(end - start + 1);
	memcpy(url, packet+start, end-start);
	url[end-start] = 0;

	/* -- find host (optional) */
	start = end+1;
	while (start + 6 < length && memcmp(packet+start, "Host: ", 6) != 0)
		start++;

	if (start + 6 < length)
	{
		/* it exists - so extract it */
		start += 6;
		end = start;
		while (end<length && packet[end] != '\r'
				  && packet[end] != '\n') end++;

		host = (char*)malloc(end - start + 1);
		memcpy(host, packet+start, end-start);
		host[end-start] = 0;
	}
	else if (destport == 80)
		host = build_str("%d.%d.%d.%d",
				(destip >> 24) & 255,
				(destip >> 16) & 255,
				(destip >>  8) & 255,
				(destip      ) & 255);
	else
		host = build_str("%d.%d.%d.%d:%d",
				(destip >> 24) & 255,
				(destip >> 16) & 255,
				(destip >>  8) & 255,
				(destip      ) & 255,
				destport);

	now = time(NULL);
	if (now > next_rotate) spylog_rotate(now);
	tm = localtime(&now);

	fprintf(spylog, "%04d-%02d-%02d %02d:%02d:%02d %d.%d.%d.%d %s "
			"http://%s%s\n",

			1900+tm->tm_year, 1+tm->tm_mon, tm->tm_mday,
			tm->tm_hour, tm->tm_min, tm->tm_sec,

			(srcip >> 24) & 255,
			(srcip >> 16) & 255,
			(srcip >>  8) & 255,
			(srcip      ) & 255,

			method, host, url);
	fflush(spylog);

	if (verbose) printf("(SPY) http://%s%s\n", host, url);

	free(url);
	free(host);
}



/* disassemble a packet and look for an HTTP request */
static void spy_pkt(byte *user unused,
	const struct pcap_pkthdr *pheader, byte *pdata)
{
	byte		*ptr;
	word		pkt_type;
	dword		pkt_srcip, pkt_destip;
	byte		*tcp_ptr;
	word		destport;

	/* assert(strcmp(user, "def") == 0); */

	interpret_linktype(spy_linktype, pdata, &ptr, &pkt_type);
	ptr = eth_pkt_strip(ptr);

	pkt_srcip = ip_pkt_srcip(ptr);
	pkt_destip = ip_pkt_destip(ptr);

	tcp_ptr = ip_pkt_strip(ptr);
	destport = tcp_pkt_destport(tcp_ptr);

	spylog_packet(tcp_ptr + 20,
		      pdata + pheader->len - tcp_ptr - 20,
		      pkt_srcip, pkt_destip, destport);
}



void spylog_main(void *spydev)
{
	struct bpf_program spyprog;
	char *prog;
	dword spyip;
	char err[PCAP_ERRBUF_SIZE];

	spy_cap = pcap_open_live((char*)spydev, 500, promisc,
		PCAP_TIMEOUT, err);

	if (!spy_cap)
	{
		printf("Error: pcap_open_live(%s): %s\n",
			(char*)spydev, err);
		FAIL();
	}

	spy_linktype = pcap_datalink(spy_cap);

	spyip = get_local_ip((char*)spydev);
	prog = build_str("ip proto \\tcp and not "
		"(dst host %d.%d.%d.%d and dst port %u)",
			(spyip >> 24) & 255,
			(spyip >> 16) & 255,
			(spyip >> 8) & 255,
			 spyip       & 255,
			 webport);

	if (pcap_compile(spy_cap, &spyprog, prog, 1, 0) == -1)
	{
		printf("Error: pcap_compile(): %s\n", err);
		FAIL();
	}

	if (pcap_setfilter(spy_cap, &spyprog) == -1)
	{
		printf("Error: pcap_setfilter(): %s\n", err);
		FAIL();
	}

#ifdef HAVE_FREECODE
	pcap_freecode(&spyprog);
#endif
	free(prog);

	/* spylogging loop */
	printf("SPY: Thread is up and logging.\n");
	up_spy = 1;

	while (!shutting_down)
	{
		struct pcap_stat ps;

		if (pcap_dispatch(spy_cap, -1,
			(pcap_handler)spy_pkt, NULL) == -1)
		{
			printf("Error: pcap_dispatch(): %s\n",
				pcap_geterr(spy_cap));
			FAIL();
		}

		if (verbose)
		{
			pcap_stats(spy_cap, &ps);
			printf("Packets: received %d, dropped %d (spylog)\n",
				ps.ps_recv, ps.ps_drop);
		}

	}

	pcap_close(spy_cap);

	printf("SPY: Thread down\n");
	SUCCEED(); /* implies up_spy = 0 */
}

