/* -*- mode:C; c-file-style:"linux"; tab-width:8; -*- */
/* 
 *  Dates - An electronic calendar optimised for embedded devices.
 *
 *  Principal author	: Chris Lord <chris@o-hand.com>
 *  Maemo port		: Tomas Frydrych <tf@o-hand.com>
 *
 *  Copyright (c) 2005 - 2006 OpenedHand Ltd - http://o-hand.com
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 */

#include <string.h>
#include <math.h>
#include <gtk/gtk.h>
#include <glade/glade.h>
#include <libecal/e-cal-time-util.h>
#include <libical/icaltime.h>
#include <gconf/gconf-client.h>
#include <gdk/gdkkeysyms.h>

#ifdef WITH_HILDON
#include <libosso.h>
#include <hildon-widgets/hildon-window.h>
#include <hildon-widgets/hildon-program.h>
#include <hildon-widgets/hildon-time-editor.h>
#include <hildon-widgets/hildon-date-editor.h>
#include <glade/glade-build.h>
#endif

#include "dates_view.h"
#include "gconf-bridge.h"

#define XML_FILE PKGDATADIR "/dates.glade"
#define CALENDAR_GCONF_PREFIX   "/apps/evolution/calendar"
#define CALENDAR_GCONF_SOURCES CALENDAR_GCONF_PREFIX "/sources"
#define CALENDAR_GCONF_SELECTED_CALENDARS CALENDAR_GCONF_PREFIX \
	"/display/selected_calendars"
#define TIME_MARKUP "%k:%M\n<small>%d %b %Y</small>"
#define DATES_GCONF_PREFIX	"/apps/dates"
#define DATES_GCONF_ZOOM	DATES_GCONF_PREFIX "/zoom"

#define CONVERT_CHILD_ITER(_filter,_child,_iter)\
	(gtk_tree_model_filter_convert_child_iter_to_iter (\
	GTK_TREE_MODEL_FILTER (_filter), _child, _iter))

/* TODO remove this logging stuff once hibernation fully debugged */
#undef MY_LOG

#ifdef MY_LOG
FILE * my_log = NULL;

#define LOG(x, a...) \
 fprintf(my_log, __FILE__ ":%d,%s() " x "\n", __LINE__, __func__, ##a); \
 fflush (my_log)

#undef g_debug
#define g_debug LOG

#undef g_message
#define g_message LOG
#endif

#ifdef DEBUG
typedef enum {
	DATES_DEBUG_HIBERNATE	= 1 << 0,
	DATES_DEBUG_EDIT	= 1 << 1,
	DATES_DEBUG_XEMBED	= 1 << 2,
	DATES_DEBUG_CALENDAR	= 1 << 3,
} DatesDebugFlag;

static const GDebugKey dates_debug_keys[] = {
	{ "hibernate", DATES_DEBUG_HIBERNATE },
	{ "edit", DATES_DEBUG_EDIT },
	{ "xembed", DATES_DEBUG_XEMBED },
	{ "calendar", DATES_DEBUG_CALENDAR },
};
#endif

enum {
	COL_CALNAME,
	COL_CALPTR,
	COL_NOTREADONLY,
	COL_LAST
};

typedef enum {
	NONE = 0,
	PENDING_CREATE,
	PENDING_DELETE,
	PENDING_CHANGE,
	PENDING_CAL_CHANGE,
} DatesPending;

typedef enum {
	NORMAL,
	NEW_EVENT,
} DatesEventType;

enum {
	DAY = 1,
	WEEK,
	FORTNIGHT,
	MONTH,
	YEAR,
	OTHER
};

typedef struct {
	DatesView *view;
	guint zoom;
	GladeXML *xml;
	ECalComponent *comp;
	ECal *cal;
	gboolean read_only;
	gboolean first_load;
	GtkListStore *cal_list_store;
	gchar *uri_uid;
	GList *widgets;
	DatesPending waiting;
	DatesEventType event_type;

	GtkWidget * TBNew;
	GtkWidget * TBOpen;
	GtkWidget * TBDelete;
	GtkWidget * TBBack;
	GtkWidget * TBToday;
	GtkWidget * TBForward;
	GtkWidget * TBZoomIn;
	GtkWidget * TBZoomOut;
	
#ifdef WITH_HILDON
	HildonProgram  *program;
	osso_context_t *osso_context;
	gboolean        init_done;
	gboolean        restored;
#endif
	
#ifdef DEBUG
	guint debug;
#endif

} DatesData;

static void dates_zoom_change (guint pos, DatesView *dates_view);
static void dates_update_calendars (ESourceList *cal_list, DatesData *d);
static gboolean dates_select_event_idle_cb (gpointer data);
void dates_save_changes (DatesData *d);

GtkWidget *
create_dates_view ()
{
	GtkWidget *widget;
	
	widget = dates_view_new ();
	/* TODO: This is temporary until the widget sizes itself properly */
	gtk_widget_set_size_request (widget, 240, 240);
	gtk_widget_show (widget);
	
	return widget;
}

/* From contacts src/contacts-utils.c */
static GList *
contacts_set_widget_desensitive_recurse (GtkWidget *widget, GList **widgets)
{
	if (GTK_IS_WIDGET (widget)) {
		if (GTK_WIDGET_SENSITIVE (widget) &&
		    GTK_WIDGET_VISIBLE (widget)) {
			gtk_widget_set_sensitive (widget, FALSE);
			*widgets = g_list_append (*widgets, widget);
		}
		
		if (GTK_IS_TABLE (widget) || GTK_IS_HBOX (widget) ||
		    GTK_IS_VBOX (widget) || GTK_IS_EVENT_BOX (widget)) {
			GList *c, *children = gtk_container_get_children (
				GTK_CONTAINER (widget));
			
			for (c = children; c; c = c->next) {
				contacts_set_widget_desensitive_recurse (
					c->data, widgets);
			}
			g_list_free (children);
		}
	}
	
	return *widgets;
}

GList *
contacts_set_widgets_desensitive (GtkWidget *widget)
{
	GList *list = NULL;
	
	contacts_set_widget_desensitive_recurse (widget, &list);
	
	return list;
}

void
contacts_set_widgets_sensitive (GList *widgets)
{
	GList *w;
	
	for (w = widgets; w; w = w->next) {
		gtk_widget_set_sensitive (GTK_WIDGET (w->data), TRUE);
	}
}

static void
dates_date_change (DatesData *data, int direction)
{
	icaltimetype *time, *oldtime;
	
	oldtime = (icaltimetype *)dates_view_get_date (data->view);
	time = g_new (icaltimetype, 1);
	g_memmove (time, oldtime, sizeof (icaltimetype));

	if (data->zoom == 16)
		time->year += direction;
	else if (data->zoom >= 11) {
		int days;
		time->month += direction;
		if (time->month > 12) {
			time->month = 1;
			time->year++;
		} else if (time->month < 1) {
			time->month = 12;
			time->year--;
		}
		days = icaltime_days_in_month (time->month, time->year);
		if (time->day > days) time->day = days;
	} else if (data->zoom >= 8)
		icaltime_adjust (time, direction * 7, 0, 0, 0);
	else
		icaltime_adjust (time, direction, 0, 0, 0);
	
	dates_view_set_date (data->view, time);
	
	g_free (time);
}

void
dates_back_cb (GtkButton *button, DatesData *data)
{
	dates_date_change (data, -1);
}

void
dates_forward_cb (GtkButton *button, DatesData *data)
{
	dates_date_change (data, 1);
}

void
dates_today_cb (GtkButton *button, DatesData *data)
{
	icaltimetype today = icaltime_today ();
	dates_view_set_date (data->view, &today);
}

void
dates_zoom_in_cb (GtkButton *button, DatesData *data)
{
	switch (data->zoom) {
		case 6:
			gtk_widget_set_sensitive (data->TBZoomIn, FALSE);
			gtk_widget_set_sensitive (glade_xml_get_widget (
				data->xml, "zoom_in_menuitem"), FALSE);
			data->zoom = 3;
			break;
		case 8:
			data->zoom = 6;
			break;
/*		case 9:
			data->zoom = 6;
			break;
		case 10:
			data->zoom = 9;
			break;*/
		case 11:
			data->zoom = 8;
			break;
/*		case 14:
			data->zoom = 11;
			break;*/
		case 16:
			gtk_widget_set_sensitive (data->TBZoomOut, TRUE);
			gtk_widget_set_sensitive (glade_xml_get_widget (
				data->xml, "zoom_out_menuitem"), TRUE);
			data->zoom = 11;
			break;
		default:
			return;
	}
	
/*	if (data->zoom > 1) {
		data->zoom--;
		dates_zoom_change (data->zoom, data->view);
	}*/
	
	dates_zoom_change (data->zoom, data->view);
}

void
dates_zoom_out_cb (GtkButton *button, DatesData *data)
{
	switch (data->zoom) {
		case 3:
			gtk_widget_set_sensitive (data->TBZoomIn, TRUE);
			gtk_widget_set_sensitive (glade_xml_get_widget (
				data->xml, "zoom_in_menuitem"), TRUE);
			data->zoom = 6;
			break;
		case 6:
			data->zoom = 8;
			break;
		case 8:
			data->zoom = 11;
			break;
/*		case 10:
			data->zoom = 11;
			break;*/
		case 11:
			gtk_widget_set_sensitive (data->TBZoomOut, FALSE);
			gtk_widget_set_sensitive (glade_xml_get_widget (
				data->xml, "zoom_out_menuitem"), FALSE);
			data->zoom = 16;
			break;
/*		case 14:
			gtk_widget_set_sensitive (GTK_WIDGET(button), FALSE);
			data->zoom = 16;
			break;*/
		default:
			return;
	}

/*	if (data->zoom < 16) {
		data->zoom++;
		dates_zoom_change (data->zoom, data->view);
	}*/

	dates_zoom_change (data->zoom, data->view);
}

static void
dates_zoom_change (guint pos, DatesView *dates_view)
{
	GConfClient *client;
	
	switch (pos) {
		case 1 :
			dates_view_set_visible_months (dates_view, 0);
			dates_view_set_visible_days (dates_view, 1);
			dates_view_set_visible_hours (dates_view, 4);
			break;
		case 2 :
			dates_view_set_visible_months (dates_view, 0);
			dates_view_set_visible_days (dates_view, 1);
			dates_view_set_visible_hours (dates_view, 8);
			break;
		case 3 :
			dates_view_set_visible_months (dates_view, 0);
			dates_view_set_visible_days (dates_view, 1);
			dates_view_set_visible_hours (dates_view, 12);
			break;
		case 4 :
			dates_view_set_visible_months (dates_view, 0);
			dates_view_set_visible_days (dates_view, 1);
			dates_view_set_visible_hours (dates_view, 12);
			break;
		case 5 :
			dates_view_set_visible_months (dates_view, 0);
			dates_view_set_visible_days (dates_view, 2);
			dates_view_set_visible_hours (dates_view, 12);
			break;
		case 6 :
			dates_view_set_visible_months (dates_view, 0);
			dates_view_set_visible_days (dates_view, 1);
			dates_view_set_visible_hours (dates_view, 24);
			break;
		case 7 :
			dates_view_set_visible_months (dates_view, 0);
			dates_view_set_visible_days (dates_view, 5);
			dates_view_set_visible_hours (dates_view, 12);
			break;
		case 8 :
			dates_view_set_visible_months (dates_view, 0);
			dates_view_set_visible_days (dates_view, 5);
			dates_view_set_visible_hours (dates_view, 24);
			break;
		case 9 :
			dates_view_set_visible_months (dates_view, 0);
			dates_view_set_visible_days (dates_view, 7);
			dates_view_set_visible_hours (dates_view, 12);
			break;
		case 10 :
			dates_view_set_visible_months (dates_view, 0);
			dates_view_set_visible_days (dates_view, 7);
			dates_view_set_visible_hours (dates_view, 24);
			break;
		case 11 :
			dates_view_set_visible_months (dates_view, 1);
			dates_view_set_visible_days (dates_view, 7);
			dates_view_set_visible_hours (dates_view, 24);
			break;
		case 12 :
			dates_view_set_visible_months (dates_view, 2);
			dates_view_set_visible_days (dates_view, 7);
			dates_view_set_visible_hours (dates_view, 24);
			break;
		case 13 :
			dates_view_set_visible_months (dates_view, 3);
			dates_view_set_visible_days (dates_view, 7);
			dates_view_set_visible_hours (dates_view, 24);
			break;
		case 14 :
			dates_view_set_visible_months (dates_view, 4);
			dates_view_set_visible_days (dates_view, 7);
			dates_view_set_visible_hours (dates_view, 24);
			break;
		case 15 :
			dates_view_set_visible_months (dates_view, 8);
			dates_view_set_visible_days (dates_view, 7);
			dates_view_set_visible_hours (dates_view, 24);
			break;
		case 16 :
			dates_view_set_visible_months (dates_view, 12);
			dates_view_set_visible_days (dates_view, 7);
			dates_view_set_visible_hours (dates_view, 24);
			break;
	}
	
	client = gconf_client_get_default ();
	gconf_client_set_int (client, DATES_GCONF_ZOOM, pos, NULL);
}

void
dates_date_changed_cb (DatesView *view, DatesData *d)
{
	struct tm timem;
	gchar buffer[256];
	const icaltimetype *date = dates_view_get_date (view);
#ifndef WITH_HILDON
	gchar *text;
#endif
	
	timem = icaltimetype_to_tm ((icaltimetype *)date);
	strftime (buffer, 255, "%a, %F", &timem);

#ifdef WITH_HILDON
	gtk_window_set_title (GTK_WINDOW (
		glade_xml_get_widget (d->xml, "main_window")), buffer);
#else
	text = g_strdup_printf ("%s - %s", buffer, _("Dates"));
	gtk_window_set_title (GTK_WINDOW (
		glade_xml_get_widget (d->xml, "main_window")), text);
	g_free (text);
#endif

	strftime (buffer, 255, "<big><b>%A %d %B, %Y</b></big>", &timem);
	gtk_label_set_markup (GTK_LABEL (
		glade_xml_get_widget (d->xml, "header_label")), buffer);
}


void
dates_about_cb (GtkWidget *widget, DatesData *d)
{
	GdkPixbuf *pixb;
	GError *error = NULL;

	/* NOTE: The hildon version of the about menu does not handle the
	 * credits button well (it is misplaced and not thematised), so we add
	 * the authors to the (c) string.
	 */
	#ifndef WITH_HILDON
	const gchar *authors[] = {
		"Principal author: Chris Lord <chris@o-hand.com>",
		"Maemo port: Tomas Frydrych <tf@o-hand.com>",
		NULL
	};

	const gchar *licence =
	"This program is free software; you can redistribute it and/or modify\n"
	"it under the terms of the GNU General Public License as published by\n"
	"the Free Software Foundation; either version 2, or (at your option)\n"
	"any later version.\n"
	"\n"
	"This program is distributed in the hope that it will be useful,\n"
	"but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
	"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
	"GNU General Public License for more details.";
	#endif

	pixb = gdk_pixbuf_new_from_file (
		PKGDATADIR G_DIR_SEPARATOR_S "oh-about-logo.png", &error);  

	if (!pixb) {
		g_assert (error);
		g_warning (error->message);
	}

	gtk_show_about_dialog (
		GTK_WINDOW (glade_xml_get_widget (d->xml, "main_window")),
		#ifndef WITH_HILDON
			"copyright",
			"(c) 2006 OpenedHand Ltd",

			"authors",
			authors,
			
			"license",
			licence,
		#else
			"copyright",
			"(c) 2006 OpenedHand Ltd\n"
			"Authors: Chris Lord <chris@o-hand.com>,\n"
			"Tomas Frydrych <tf@o-hand.com>\n",
		#endif

		"logo",
		pixb,

		"name",
		PACKAGE_NAME,

		"version",
		PACKAGE_VERSION,

		"website",
		"http://projects.o-hand.com/dates",

		NULL);
}

static void
dates_fill_details_dialog (DatesView *view, DatesData *d)
{
	const gchar *location;
	GSList *text_list;
	ECalComponentDateTime time;
	struct tm timem;
	gchar time_text[256];
	GtkTextBuffer *buffer;
	ECalComponentText summary;
	GtkWidget *widget;

	/* Set summary entry */
	e_cal_component_get_summary (d->comp, &summary);
	widget = glade_xml_get_widget (d->xml, "details_summary_entry");
	if (summary.value)
		gtk_entry_set_text (GTK_ENTRY (widget), summary.value);
	else if (summary.altrep)
		gtk_entry_set_text (GTK_ENTRY (widget), summary.altrep);
	else
		gtk_entry_set_text (GTK_ENTRY (widget), "");
	
	/* Set location entry */
	e_cal_component_get_location (d->comp, &location);
	if (location)
		gtk_entry_set_text (GTK_ENTRY (glade_xml_get_widget (
			d->xml, "details_location_entry")), location);
	else
		gtk_entry_set_text (GTK_ENTRY (glade_xml_get_widget (
			d->xml, "details_location_entry")), "");
	
	/* Set event description textview */
	/* Note: Docs say that only journal entries can have more than one
	 * description, so just use text_list->data
	 */
	e_cal_component_get_description_list (d->comp, &text_list);
	buffer = gtk_text_view_get_buffer (
		GTK_TEXT_VIEW (glade_xml_get_widget (
			d->xml, "details_textview")));
	gtk_text_buffer_set_text (buffer, "", -1);
	if (text_list) {
		ECalComponentText *desc = text_list->data;
		if (desc->value)
			gtk_text_buffer_set_text (buffer, desc->value, -1);
		e_cal_component_free_text_list (text_list);
	}
	
	/* Set time buttons and dialog */
	e_cal_component_get_dtstart (d->comp, &time);
	timem = icaltimetype_to_tm (time.value);
	strftime (time_text, 256, TIME_MARKUP, &timem);
	gtk_label_set_markup (GTK_LABEL (
		glade_xml_get_widget (d->xml, "details_start_label")),
		time_text);
	e_cal_component_free_datetime (&time);

	e_cal_component_get_dtend (d->comp, &time);
	timem = icaltimetype_to_tm (time.value);
	strftime (time_text, 256, TIME_MARKUP, &timem);
	gtk_label_set_markup (GTK_LABEL (
		glade_xml_get_widget (d->xml, "details_end_label")),
		time_text);
	e_cal_component_free_datetime (&time);
}

void
dates_event_selected_cb (DatesView *view, DatesData *d)
{
	GtkWidget *widget;

	if (d->comp) g_object_unref (d->comp);
	d->comp = dates_view_get_selected_event (d->view);

	if (d->comp) {
		GtkTreeIter iter;
		gboolean read_only = TRUE, cal_found = FALSE;
		GtkTreeModel *model;

/*		g_debug ("Selected event RID: %s",
			e_cal_component_get_recurid_as_string (d->comp));*/

		d->cal = dates_view_get_selected_event_cal (d->view);
			
/*		widget = glade_xml_get_widget (d->xml, "header_label");
		gtk_label_set_markup (GTK_LABEL (widget),
			"<big><b>Enter details</b></big>");*/
		
		/* Set the details calendar combo-box */
		widget = glade_xml_get_widget (
			d->xml, "details_calendar_combobox");
		model = GTK_TREE_MODEL (d->cal_list_store);
		if (gtk_tree_model_get_iter_first (model, &iter)) {
			do {
				ECal *c;

				gtk_tree_model_get (
					model, &iter, COL_CALPTR, &c, -1);

				if (c != d->cal) continue;
				cal_found = TRUE;
				
				gtk_tree_model_get (model, &iter,
					COL_NOTREADONLY, &read_only,
					-1);
				read_only = !read_only;
				d->read_only = read_only;
				if (!read_only) {
					GtkTreeIter filter_iter;
					
					model = gtk_combo_box_get_model (
						GTK_COMBO_BOX (widget));
					CONVERT_CHILD_ITER (model, &filter_iter,
						&iter);
					
					gtk_combo_box_set_active_iter (
						GTK_COMBO_BOX (widget),
						&filter_iter);

					gtk_widget_set_sensitive (widget, TRUE);
				} else {
#ifdef DEBUG
					if (d->debug & DATES_DEBUG_EDIT)
						g_debug ("Read-only event"
							" selected");
#endif
				}
				break;
			} while (gtk_tree_model_iter_next (model, &iter));
			if (!cal_found) {
				gtk_widget_set_sensitive (widget, FALSE);
				g_warning ("Event selected with "
					"unknown calendar");
			}
		} else {
			g_warning ("Error accessing calendar list");
		}
				
		gtk_widget_set_sensitive (d->TBOpen, !read_only);		
		gtk_widget_set_sensitive (d->TBDelete, !read_only);
	} else {
		d->cal = NULL;
		if (!d->waiting)
			gtk_widget_hide (glade_xml_get_widget (
				d->xml, "details_dialog"));

		gtk_widget_set_sensitive (d->TBOpen, FALSE);
		gtk_widget_set_sensitive (d->TBDelete, FALSE);
	}
}

static void
dates_commit_event_cb (GtkWidget *source, DatesData *data,
		       CalObjModType type)
{
	/* Commit changes */
#ifdef DEBUG
	if (data->debug & DATES_DEBUG_EDIT)
		g_debug ("Committing event (%p) changes", data->comp);
#endif
	e_cal_component_commit_sequence (data->comp);
	e_cal_modify_object (data->cal,
		e_cal_component_get_icalcomponent (data->comp),
		type, NULL);
}

void
dates_event_moved_cb (DatesView *view, ECalComponent *comp, DatesData *d)
{
	ECalComponentDateTime start, end;
	struct icaldurationtype duration = icaldurationtype_null_duration ();

	if (d->comp) g_object_unref (d->comp);
	d->comp = g_object_ref (comp);

	e_cal_component_get_dtstart (d->comp, &start);
	if (!start.value->is_date) {
		e_cal_component_get_dtend (d->comp, &end);
		duration = icaltime_subtract (*end.value, *start.value);
		*end.value = icaltime_add (*start.value, duration);
		e_cal_component_set_dtstart (d->comp, &start);
		e_cal_component_set_dtend (d->comp, &end);
		e_cal_component_free_datetime (&end);
	}
	e_cal_component_free_datetime (&start);
	dates_commit_event_cb (NULL, d, CALOBJ_MOD_THIS);
}

void
dates_event_sized_cb (DatesView *view, ECalComponent *comp, DatesData *d)
{
	if (d->comp) g_object_unref (d->comp);
	d->comp = g_object_ref (comp);

	dates_commit_event_cb (NULL, d, CALOBJ_MOD_THIS);
}

void
dates_details_time_entry_changed (GtkEditable *entry, gchar *new_text,
				  gint new_text_length,
				  gint *position,
				  DatesData *d)
{
	gint i;
	gchar *c;
	
	for (i = 0, c = new_text; c;
	     c = g_utf8_find_next_char (c, (new_text_length != -1) ?
	     	(new_text + new_text_length) : NULL)) {
		if (*c < '0' || *c > '9') {
			g_signal_stop_emission (entry, g_signal_lookup (
				"insert-text", GTK_TYPE_EDITABLE), 0);
			gdk_beep ();
			break;
		}
	}
}

static gboolean
dates_details_time_entry_alt (GtkArrow *source, const gchar *entry,
			      DatesData *d, gint limit)
{
	gboolean overflow = FALSE;
	GtkEntry *widget;
	gchar buffer[33];
	gint hour;
	GtkArrowType direction;
	
	widget = GTK_ENTRY (glade_xml_get_widget (d->xml, entry));
	gtk_widget_grab_focus (GTK_WIDGET (widget));
	hour = atoi (gtk_entry_get_text (widget));
	g_object_get (G_OBJECT (source), "arrow-type", &direction, NULL);
	if (direction == GTK_ARROW_UP)
		hour ++;
	else
		hour --;
	if (hour < 0) { hour = limit; overflow = TRUE; }
	if (hour > limit) { hour = 0; overflow = TRUE; }
	snprintf (buffer, 33, "%d", hour);
	gtk_entry_set_text (widget, buffer);
	
	return overflow;
}

void
dates_details_time_hour_cb (GtkButton *source, DatesData *d)
{
	dates_details_time_entry_alt (GTK_ARROW (gtk_bin_get_child (
		GTK_BIN (source))), "hour_entry", d, 23);
}

void
dates_details_time_lminute_cb (GtkButton *source, DatesData *d)
{
	dates_details_time_entry_alt (GTK_ARROW (gtk_bin_get_child (
		GTK_BIN (source))), "lminute_entry", d, 5);
}

void
dates_details_time_rminute_cb (GtkButton *source, DatesData *d)
{
	if (dates_details_time_entry_alt (GTK_ARROW (gtk_bin_get_child (
	    GTK_BIN (source))), "rminute_entry", d, 9))
		dates_details_time_lminute_cb (source, d);
}
#if 0
static void
dates_repeats_cb (GtkWidget *source, DatesData *d)
{
	GtkWidget *widget;
	
	gtk_window_set_transient_for (
		GTK_WINDOW (glade_xml_get_widget (d->xml, "time_dialog")),
		GTK_WINDOW (glade_xml_get_widget (d->xml, "repeats_dialog")));

	/* Run dialog */
	widget = glade_xml_get_widget (d->xml, "repeats_dialog");
	gtk_dialog_run (GTK_DIALOG (widget));
	gtk_widget_hide (widget);

	gtk_window_set_transient_for (
		GTK_WINDOW (glade_xml_get_widget (d->xml, "time_dialog")),
		GTK_WINDOW (glade_xml_get_widget (d->xml, "details_dialog")));
}

static void
dates_details_repeats_cb (GtkComboBox *combobox, DatesData *d)
{
	gint option = gtk_combo_box_get_active (combobox);
	
	if (option == OTHER)
		dates_repeats_cb (GTK_WIDGET (combobox), d);
	else {
		/* Ask for time */
		gtk_widget_show (glade_xml_get_widget (
			d->xml, "time_forever_checkbutton"));
		gtk_widget_hide (glade_xml_get_widget (d->xml, "time_dialog"));
	}
}
#endif
static void
dates_details_time_cb (DatesData *d, ECalComponentDateTime *time)
{
	GtkWidget *widget;
#ifndef WITH_HILDON
	gint i, hour, minute;
	gchar buffer[33];
	GtkCalendar *calendar;
	
	calendar = GTK_CALENDAR (
		glade_xml_get_widget (d->xml, "time_calendar"));
	
	/* Set time widgets */	
	gtk_calendar_select_month (calendar, time->value->month - 1,
		time->value->year);
	gtk_calendar_select_day (calendar, time->value->day);

	g_snprintf (buffer, 33, "%d", time->value->hour);
	gtk_entry_set_text (GTK_ENTRY (
		glade_xml_get_widget (d->xml, "hour_entry")), buffer);
	i = time->value->minute / 10;
	g_snprintf (buffer, 33, "%d", i);
	gtk_entry_set_text (GTK_ENTRY (
		glade_xml_get_widget (d->xml, "lminute_entry")), buffer);
	i = time->value->minute - (i * 10);
	g_snprintf (buffer, 33, "%d", i);
	gtk_entry_set_text (GTK_ENTRY (
		glade_xml_get_widget (d->xml, "rminute_entry")), buffer);
#else
	HildonDateEditor *date_edit = HILDON_DATE_EDITOR (glade_xml_get_widget (
		d->xml, "time_date_editor"));
	HildonTimeEditor *time_edit = HILDON_TIME_EDITOR (glade_xml_get_widget (
		d->xml, "time_time_editor"));
	
	/* Set date */
	hildon_date_editor_set_year (date_edit, time->value->year);
	hildon_date_editor_set_month (date_edit, time->value->month);
	hildon_date_editor_set_day (date_edit, time->value->day);
	
	/* Set time */
	hildon_time_editor_set_time (time_edit, time->value->hour,
		time->value->minute, 0);
	hildon_time_editor_show_seconds (time_edit, FALSE);
#endif
	
	gtk_widget_hide (glade_xml_get_widget (
		d->xml, "time_forever_checkbutton"));
	gtk_widget_show (glade_xml_get_widget (d->xml, "time_table"));

	/* Run dialog */
	widget = glade_xml_get_widget (d->xml, "time_dialog");
	gtk_dialog_run (GTK_DIALOG (widget));
	gtk_widget_hide (widget);
	
	/* Retrieve time */
#ifndef WITH_HILDON
	gtk_calendar_get_date (calendar,
		(guint *)(&time->value->year),
		(guint *)(&time->value->month),
		(guint *)(&time->value->day));
	time->value->month += 1;
	
	hour = atoi (gtk_entry_get_text (GTK_ENTRY (
		glade_xml_get_widget (d->xml, "hour_entry"))));
	if ((hour >= 0) && (hour <= 23))
		time->value->hour = hour;
	minute = (atoi (gtk_entry_get_text (GTK_ENTRY (
		glade_xml_get_widget (d->xml, "lminute_entry")))) * 10) +
		atoi (gtk_entry_get_text (GTK_ENTRY (
		glade_xml_get_widget (d->xml, "rminute_entry"))));
	if ((minute >= 0) && (minute <= 59))
		time->value->minute = minute;
#else
	time->value->year = hildon_date_editor_get_year (date_edit);
	time->value->month = hildon_date_editor_get_month (date_edit);
	time->value->day = hildon_date_editor_get_day (date_edit);
	
	hildon_time_editor_get_time (time_edit,
		(guint *)(&time->value->hour),
		(guint *)(&time->value->minute),
		(guint *)(&time->value->second));
#endif
}

void
dates_details_update_time_label (DatesData *d, const char *label,
				 struct icaltimetype *time)
{
	struct tm timem;
	gchar time_text[256];

	timem = icaltimetype_to_tm (time);
	strftime (time_text, 255, TIME_MARKUP, &timem);
	gtk_label_set_markup (GTK_LABEL (
		glade_xml_get_widget (d->xml, label)),
		time_text);
}

void
dates_details_time_start_cb (GtkWidget *source, DatesData *d)
{
	ECalComponentDateTime start, end;
	struct icaltimetype ostart;
	
	e_cal_component_get_dtstart (d->comp, &start);
	ostart = *start.value;
	dates_details_time_cb (d, &start);
	
	/* If start >= end, adjust end */
	e_cal_component_get_dtend (d->comp, &end);
	if (icaltime_compare (*start.value, *end.value) >= 0) {
		struct icaldurationtype duration =
			icaltime_subtract (*start.value, ostart);
		*end.value = icaltime_add (*end.value, duration);
		e_cal_component_set_dtend (d->comp, &end);

		dates_details_update_time_label (d, "details_end_label",
			end.value);
	}
	
	e_cal_component_set_dtstart (d->comp, &start);
	dates_details_update_time_label (d, "details_start_label",
		start.value);

	e_cal_component_free_datetime (&start);
	e_cal_component_free_datetime (&end);
	dates_view_redraw (d->view, FALSE);
}

void
dates_details_time_end_cb (GtkWidget *source, DatesData *d)
{
	ECalComponentDateTime start, end;
	struct icaltimetype oend;
	
	e_cal_component_get_dtend (d->comp, &end);
	oend = *end.value;
	dates_details_time_cb (d, &end);
	
	/* If end <= start, adjust start */
	e_cal_component_get_dtstart (d->comp, &start);
	if (icaltime_compare (*end.value, *start.value) <= 0) {
		struct icaldurationtype duration =
			icaltime_subtract (*end.value, oend);
		*start.value = icaltime_add (*end.value, duration);
		e_cal_component_set_dtstart (d->comp, &start);

		dates_details_update_time_label (d, "details_start_label",
			start.value);
	}
	
	e_cal_component_set_dtend (d->comp, &end);
	dates_details_update_time_label (d, "details_end_label",
		end.value);

	e_cal_component_free_datetime (&end);
	e_cal_component_free_datetime (&start);
	dates_view_redraw (d->view, FALSE);
}

void
dates_open_cb (GtkWidget *source, DatesData *d)
{
	if (d->read_only) return;
	/* TODO: Handle recurring events */
	if (e_cal_component_has_recurrences (d->comp)) return;
	dates_fill_details_dialog (d->view, d);
	gtk_widget_show (glade_xml_get_widget (d->xml, "details_dialog"));
}

void
dates_details_ok_cb (GtkWidget *source, DatesData *d)
{
	if (d->waiting == NONE) {
		d->event_type = NORMAL;
		d->waiting = NONE;
		dates_save_changes (d);
		gtk_widget_hide (
			glade_xml_get_widget (d->xml, "details_dialog"));
	} else {
		d->widgets = contacts_set_widgets_desensitive (
			glade_xml_get_widget (d->xml, "details_dialog"));
		d->waiting = PENDING_CHANGE;
	}
}

void
dates_delete_cb (GtkWidget *source, DatesData *d)
{
	ECalComponentText summary;
	GtkWidget *widget;
	const char *value;

	e_cal_component_get_summary (d->comp, &summary);
	widget = glade_xml_get_widget (d->xml, "details_dialog");
	if (GTK_WIDGET_VISIBLE (widget))
		value = gtk_entry_get_text (GTK_ENTRY (glade_xml_get_widget (
			d->xml, "details_summary_entry")));
	else if (summary.value)
		value = summary.value;
	else if (summary.altrep)
		value = summary.altrep;
	else {
		g_warning ("Deleting event with no summary");
		value = "Unknown event";
	}
		
	widget = gtk_message_dialog_new (
		GTK_WINDOW (glade_xml_get_widget (d->xml, "main_window")),
		GTK_DIALOG_MODAL,
		GTK_MESSAGE_QUESTION,
		GTK_BUTTONS_NONE,
		_("Are you sure you want to delete event '%s'?"),
		value);
	gtk_dialog_add_buttons (GTK_DIALOG (widget),
		_("Keep event"), GTK_RESPONSE_NO,
		_("Delete event"), GTK_RESPONSE_YES,
		NULL);
	
	if (gtk_dialog_run (GTK_DIALOG (widget)) == GTK_RESPONSE_YES) {
		/* Reset event type, in case this was a new event */
		d->event_type = NORMAL;
		
		if (d->waiting == NONE) {
			const char *uid = NULL;
			e_cal_component_get_uid (d->comp, &uid);
			e_cal_remove_object (d->cal, uid, NULL);

			/* Hide the details dialog, in case we deleted
			 * from there.
			 */
			gtk_widget_hide (glade_xml_get_widget (
				d->xml, "details_dialog"));
		} else {
			d->widgets = contacts_set_widgets_desensitive (
				glade_xml_get_widget (
					d->xml, "details_dialog"));
			d->waiting = PENDING_DELETE;
		}
	}
	
	gtk_widget_destroy (widget);
}

void
dates_details_cancel_cb (GtkWidget *source, DatesData *d)
{
	GtkWidget *widget;

	if (d->event_type == NEW_EVENT) {
		if (d->waiting == NONE) {
			const char *uid = NULL;
			d->event_type = NORMAL;
			e_cal_component_get_uid (d->comp, &uid);
			e_cal_remove_object (d->cal, uid, NULL);
		} else {
			d->widgets = contacts_set_widgets_desensitive (
				glade_xml_get_widget (
					d->xml, "details_dialog"));
			d->waiting = PENDING_DELETE;
			return;
		}
	}

	widget = glade_xml_get_widget (d->xml, "details_dialog");
	gtk_widget_hide (widget);
}

gboolean
dates_details_close_cb (GtkWidget *widget, GdkEvent *event, DatesData *d)
{
	dates_details_cancel_cb (widget, d);
	return TRUE;
}

void
dates_save_changes (DatesData *d)
{
	ECalComponentText text;
	gchar *desc;
	GSList *desc_list;
	GtkTextIter start, end;
	GtkTextBuffer *buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (
		glade_xml_get_widget (d->xml, "details_textview")));
	GtkWidget *widget;
	ECalComponentText summary, new_summary;

	gtk_text_buffer_get_start_iter (buffer, &start);
	gtk_text_buffer_get_end_iter (buffer, &end);
	desc = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);
	text.value = desc ? desc : "";
	text.altrep = NULL;
	desc_list = g_slist_prepend (NULL, &text);
	e_cal_component_set_description_list (d->comp, desc_list);
	if (desc) g_free (desc);
	g_slist_free (desc_list);
	
	e_cal_component_set_location (d->comp, gtk_entry_get_text (GTK_ENTRY (
		glade_xml_get_widget (d->xml, "details_location_entry"))));
	
	widget = glade_xml_get_widget (d->xml, "details_summary_entry");
	e_cal_component_get_summary (d->comp, &summary);
	new_summary.altrep = summary.altrep;
	new_summary.value = g_strdup (gtk_entry_get_text (GTK_ENTRY (widget)));
	e_cal_component_set_summary (d->comp, &new_summary);

	dates_commit_event_cb (NULL, d, CALOBJ_MOD_ALL);
}

static gboolean
dates_select_event_idle_cb (gpointer data)
{
	DatesData *d = data;
	gboolean selected =
		dates_view_set_selected_event (d->view, d->uri_uid, NULL);
	
	if (selected) {
		const char *uid = NULL;
		g_free (d->uri_uid);
#ifndef WITH_HILDON
		gtk_widget_hide (
			glade_xml_get_widget (d->xml, "details_message_hbox"));
#endif
		if (d->widgets) {
			contacts_set_widgets_sensitive (d->widgets);
			g_list_free (d->widgets);
			d->widgets = NULL;
		}
		switch (d->waiting) {
			case NONE :
			case PENDING_CREATE :
				break;
			case PENDING_DELETE :
				e_cal_component_get_uid (d->comp, &uid);
				e_cal_remove_object (d->cal, uid, NULL);
				gtk_widget_hide (glade_xml_get_widget (d->xml,
					"details_dialog"));
				break;
			case PENDING_CHANGE :
				dates_save_changes (d);
				gtk_widget_hide (glade_xml_get_widget (d->xml,
					"details_dialog"));
				break;
			case PENDING_CAL_CHANGE :
				break;
			default:
				g_warning ("Unknown pending state");
		}
		d->waiting = NONE;
	}
	
	return !selected;
}

void
dates_new_cb (GtkWidget *source, DatesData *d)
{
	GtkTreeIter iter;
	icalcomponent *comp;
	char *uid = NULL;
	struct icalperiodtype period;
	ECal *c;
	GError *error = NULL;

	/* FIXME: Use default calendar, not first */
	if (!gtk_tree_model_get_iter_first (GTK_TREE_MODEL (d->cal_list_store),
	    &iter)) {
		g_warning ("gtk_tree_model_get_iter_first failed");
		return;
	}

	if (!dates_view_get_selected_period (d->view, &period)) {
		struct icaltimetype date, now;

		/* Create a default time event */
		date = *(dates_view_get_date (d->view));
		now = icaltime_current_time_with_zone (date.zone);
		date.is_date = FALSE;
		now.is_date = FALSE;
		period.duration = icaldurationtype_null_duration ();
		period.duration.minutes = 0;

		/* New events default to starting at 9am */
		period.duration.hours = 9;
		period.start = icaltime_add (
			date, period.duration);
		
		/* NOTE: 11 is the magic number where we're viewing a 
		 * month or more. See dates_zoom_change.
		 */
		period.duration.hours = (d->zoom >= 11) ? 8 : 2;
		period.end = icaltime_add (period.start, period.duration);
	}
	
	comp = icalcomponent_new_vevent ();
	icalcomponent_set_dtstart (comp, period.start);
	icalcomponent_set_dtend (comp, period.end);
	icalcomponent_set_summary (comp, _("New event"));
	gtk_tree_model_get (GTK_TREE_MODEL (d->cal_list_store), &iter,
		COL_CALPTR, &c, -1);
	if (e_cal_create_object (c, comp, (char **)&uid, &error)) {
		d->event_type = NEW_EVENT;

		/* Bring up details dialog ASAP */
		d->comp = e_cal_component_new ();
		e_cal_component_set_icalcomponent (d->comp, comp);
		dates_fill_details_dialog (d->view, d);
		gtk_widget_show (glade_xml_get_widget (
			d->xml, "details_dialog"));

		/* Disable calendar combo-box */
		gtk_widget_set_sensitive (glade_xml_get_widget (
			d->xml, "details_calendar_combobox"), FALSE);

		/* Select new event */
		d->waiting = PENDING_CREATE;
		d->uri_uid = g_strconcat (e_cal_get_uri (c), uid, NULL);
		g_idle_add (dates_select_event_idle_cb, d);
	} else {
		g_warning ("Failed to create calendar object: %s",
			error->message);
		g_error_free (error);
	}
}

void
dates_calendar_combo_changed_cb (GtkComboBox *widget, DatesData *d)
{
	ECal *c;
	ECalComponent *comp = d->comp;
	GtkTreeIter iter;
	
	gtk_combo_box_get_active_iter (widget, &iter);
	gtk_tree_model_get (gtk_combo_box_get_model (widget),
		&iter, COL_CALPTR, &c, -1);
	
	if (c != d->cal) {
		/* Move event between calendars */
		const char *uid = NULL;
		
		g_object_ref (comp);
		if (!d->waiting)
			d->waiting = PENDING_CAL_CHANGE;
		e_cal_component_get_uid (comp, &uid);
		e_cal_remove_object (d->cal, uid, NULL);
		
		/* TODO: Error checking */
		e_cal_create_object (c,
			e_cal_component_get_icalcomponent (comp),
			(char **)&uid, NULL);
		g_object_unref (comp);
		d->uri_uid = g_strconcat (e_cal_get_uri (c), uid, NULL);
		d->widgets = contacts_set_widgets_desensitive (
			glade_xml_get_widget (d->xml, "details_dialog"));
		/* TODO: Show the saving note when the save takes > 0.5s? */
/*		gtk_widget_show (glade_xml_get_widget (
			d->xml, "details_message_hbox"));*/
		g_idle_add (dates_select_event_idle_cb, d);
	}
}

/* The following function taken from gnome clock-applet
 * (gnome-panel/applets/clock/calendar-sources.c)
 */
static ECal *
load_esource (ESource        *esource,
	      ECalSourceType  source_type,
	      GSList         *existing_clients,
	      DatesData      *d)
{
  ECal   *retval;
  GError *error;


  if (existing_clients)
    {
      GSList *l;

      for (l = existing_clients; l; l = l->next)
	{
	  ECal *client = E_CAL (l->data);

	  if (e_source_equal (esource, e_cal_get_source (client)))
	    {
#ifdef DEBUG
	      if (d->debug & DATES_DEBUG_CALENDAR)
	        g_debug ("load_esource: found existing source, returning that");
#endif
	      return g_object_ref (client);
	    }
	}
    }

  retval = e_cal_new (esource, source_type);
  if (!retval)
    {
      g_warning ("Could not load source '%s' from '%s'\n",
		 e_source_peek_name (esource),
		 e_source_get_uri (esource));
      return NULL;
    }

  error = NULL;
  if (!e_cal_open (retval, TRUE, &error))
    {
      g_assert (error != NULL);
      g_warning ("Cannot open calendar from uri '%s': %s\n",
		 e_cal_get_uri (retval), error->message);
      g_error_free (error);
      g_object_unref (retval);
      return NULL;
    }
#ifdef DEBUG
  if (d->debug & DATES_DEBUG_CALENDAR)
      g_debug ("Loaded calendar from uri '%s'",
	   e_cal_get_uri (retval));
#endif
  return retval;
}

static void
dates_backend_died_cb (ECal *client, DatesData *d)
{
#ifdef DEBUG
	if (d->debug & DATES_DEBUG_CALENDAR)
		g_debug ("Calendar died, removing");
#endif
	dates_view_remove_calendar (d->view, client);
}

static void
dates_sources_changed_cb (ESourceList *cal_list, DatesData *d)
{
#ifdef DEBUG
	if (d->debug & DATES_DEBUG_CALENDAR)
		g_debug ("Sources changed, reloading calendars");
#endif
	dates_update_calendars (cal_list, d);
}

static void
dates_cal_open_cb (ECal *ecal, ECalendarStatus status, DatesData *d)
{
	gboolean read_only;
	GError *error = NULL;
	const char *cal_name;
	ESource *source;
	GtkTreeIter iter;

#ifdef DEBUG
	if (d->debug & DATES_DEBUG_CALENDAR)
		g_debug ("Calendar opened, setting up...");
#endif
	
	if (status != E_CALENDAR_STATUS_OK) {
		g_warning ("Error '%d' opening ecal", status);
		g_object_unref (ecal);
		if (d->first_load) {
			GtkWidget *dialog = gtk_message_dialog_new (
				GTK_WINDOW (glade_xml_get_widget (
					d->xml, "main_window")),
				GTK_DIALOG_MODAL | GTK_DIALOG_NO_SEPARATOR,
				GTK_MESSAGE_ERROR,
				GTK_BUTTONS_CLOSE,
				"Dates failed to open the system calendar. "
				"You will not be able to view or create "
				"events; this may be remedied by restarting.");
			gtk_dialog_run (GTK_DIALOG (dialog));
		}
		return;
	}
	
	source = e_cal_get_source (ecal);

	/* Create the system calendar when it isn't found */
	if (d->first_load) {
		gchar *group_uri;
		ESourceGroup *group;
		ESourceList *cal_list;
		const gchar *cal_uri = e_cal_get_uri (ecal);
		gint size;
		GError *error = NULL;
		
#ifdef DEBUG
		if (d->debug & DATES_DEBUG_CALENDAR) {
			g_debug ("First time run, "
				"creating new system calendar");
			g_debug ("New system calendar URI: %s", cal_uri);
		}
#endif
		
		/* FIXME: This whole section is a little hacky, do it better */
		
		/* Create group by chopping off last bit of system calendar uri
		 */
		size = g_strrstr (cal_uri, "/system") - cal_uri;
		group_uri = g_new (gchar, size);
		strncpy (group_uri, cal_uri, size + 1);
		group_uri[size] = '\0';
#ifdef DEBUG
		if (d->debug & DATES_DEBUG_CALENDAR)
			g_debug ("Creating new group at %s", group_uri);
#endif
		group = e_source_group_new ("On This Computer", group_uri);
		g_free (group_uri);
		
		/* Name system calendar and add it to group */
		e_source_set_absolute_uri (source, cal_uri);
		/* NOTE: Default Evolution colour */
		e_source_set_color (source, 0xBECEDD);
		e_source_set_name (source, "Personal");
		e_source_group_add_source (group, source, 0);
		
		/* Add group to default list */
		cal_list = e_source_list_new_for_gconf_default (
			CALENDAR_GCONF_SOURCES);
		e_source_list_add_group (cal_list, group, 0);
		if (!e_source_list_sync (cal_list, &error)) {
			g_warning ("Error syncing ESourceList: %s",
				error->message);
			g_error_free (error);
			return;
		}
		
		g_signal_connect (G_OBJECT (cal_list), "changed",
			G_CALLBACK (dates_sources_changed_cb), d);
		d->first_load = FALSE;
		
		/* Return, the sources will change immediately after this */
#ifdef DEBUG
		if (d->debug & DATES_DEBUG_CALENDAR)
			g_debug ("Created new system calendar and group");
#endif
		return;
	}

	cal_name = e_source_peek_name (source);
	
	if (!e_cal_is_read_only (ecal, &read_only, &error)) {
		g_warning ("Error querying calendar's read-only status: %s",
			error->message);
		g_error_free (error);
		g_object_unref (ecal);
		return;
	}
			
	gtk_list_store_insert_with_values (
		d->cal_list_store, &iter, G_MAXINT,
		COL_CALNAME, cal_name ? cal_name : "Unknown",
		COL_CALPTR, ecal,
		COL_NOTREADONLY, !read_only, -1);
	dates_view_add_calendar (d->view, ecal);
		
	g_signal_connect (G_OBJECT (ecal), "backend_died",
		G_CALLBACK (dates_backend_died_cb), d);

#ifdef DEBUG
	if (d->debug & DATES_DEBUG_CALENDAR)
		g_debug ("Calendar added.");
#endif
}

/* snippet from evolution/calendar/gui/calendar-component.c:294 */
static gboolean
is_in_uids (GSList *uids, ESource *source)
{
	GSList *l;

	for (l = uids; l; l = l->next) {
		const char *uid = l->data;

		if (strcmp (uid, e_source_peek_uid (source)) == 0)
			return TRUE;
	}

	return FALSE;
}

static void
dates_update_calendars (ESourceList *cal_list, DatesData *d)
{
	GSList *groups, *g, *selected;

	selected = gconf_client_get_list (gconf_client_get_default (),
		CALENDAR_GCONF_SELECTED_CALENDARS, GCONF_VALUE_STRING, NULL);

	/* TODO: Don't reload *all* calendars on update */
	gtk_list_store_clear (d->cal_list_store);
	dates_view_remove_all_calendars (d->view);
	groups = e_source_list_peek_groups (cal_list);

	g_message ("Updating Calendars");
    
	for (g = groups; g; g = g->next) {
		GSList *sources, *s;
#ifdef DEBUG
		if (d->debug & DATES_DEBUG_CALENDAR)
			g_debug ("Group %s (%s)",
				e_source_group_peek_name (
					E_SOURCE_GROUP (g->data)),
				e_source_group_peek_base_uri (
					E_SOURCE_GROUP (g->data)));
#endif
		sources = e_source_group_peek_sources (
			E_SOURCE_GROUP (g->data));
		for (s = sources; s; s = s->next) {
			ESource *source = E_SOURCE (s->data);
			ECal *ecal;
			
			if (selected && !is_in_uids (selected, source))
				continue;
			
			ecal = load_esource (source,
				E_CAL_SOURCE_TYPE_EVENT, NULL, d);
			if (!ecal) {
				g_message ("No EVENTS in this source");
				continue;
			}

#ifdef DEBUG
			if (d->debug & DATES_DEBUG_CALENDAR)
				g_debug ("Found EVENTS in this source");
#endif

			g_signal_connect (G_OBJECT (ecal), "cal_opened",
				G_CALLBACK (dates_cal_open_cb), d);

			e_cal_open_async (ecal, TRUE);
		}
	}	
}

static void
dates_gconf_selected_cb (GConfClient *client, guint cnxn_id, GConfEntry *entry,
			 gpointer user_data)
{
	ESourceList *cal_list;
	DatesData *d = (DatesData *)user_data;
	
	cal_list = e_source_list_new_for_gconf_default (CALENDAR_GCONF_SOURCES);
	
	dates_update_calendars (cal_list, d);
}

static gboolean
dates_load_calendars (DatesData *d)
{
	ESourceList *cal_list;

	/* Load calendars */
	cal_list = e_source_list_new_for_gconf_default (CALENDAR_GCONF_SOURCES);

	if (!cal_list) {
		g_error ("Error loading ESource list");
		return FALSE;
	}

	if (!e_source_list_peek_groups (cal_list)) {
		g_warning ("No groups found");
		g_object_unref (cal_list);
		return FALSE;
	}

	g_signal_connect (G_OBJECT (cal_list), "changed",
		G_CALLBACK (dates_sources_changed_cb), d);
	dates_update_calendars (cal_list, d);
	
	return TRUE;
}

void
dates_window_state_cb (GtkWidget *widget, GdkEventVisibility *event,
		       DatesData *d)
{
	GtkWidget *header = glade_xml_get_widget (d->xml, "header_eventbox");
	if (gdk_window_get_state (widget->window) &
	    GDK_WINDOW_STATE_FULLSCREEN) {
		gtk_widget_show (header);
	} else {
		gtk_widget_hide (header);
	}
}

gboolean
dates_cal_key_press_cb (GtkWidget *widget, GdkEventKey *event, DatesData *d)
{
	switch (event->keyval) {
		case GDK_3270_Enter :
		case GDK_ISO_Enter :
		case GDK_KP_Enter :
		case GDK_Return :
			dates_new_cb (widget, d);
			return TRUE;
	}
	return FALSE;
}

static gboolean
dates_key_press_cb (GtkWidget *widget, GdkEventKey *event, DatesData *d)
{
	switch (event->keyval) {
#ifdef WITH_HILDON
		case GDK_F6 : {	/* Full-screen key */
			GtkWidget *window = glade_xml_get_widget (
				d->xml, "main_window");
			GtkWidget *header = glade_xml_get_widget (
				d->xml, "header_eventbox");
			if (!(gdk_window_get_state (window->window) &
			    GDK_WINDOW_STATE_FULLSCREEN)) {
				gtk_window_fullscreen (GTK_WINDOW (widget));
				gtk_widget_show (header);
			} else {
				gtk_window_unfullscreen (GTK_WINDOW (widget));
				gtk_widget_hide (header);
			}
			return TRUE;
		}
		case GDK_F8 :	/* Zoom out key */
			dates_zoom_out_cb (NULL, d);
			return TRUE;
		case GDK_F7 :	/* Zoom in key */
			dates_zoom_in_cb (NULL, d);
			return TRUE;
#else
		case GDK_KP_Subtract :
		case GDK_minus :
			if (event->state & GDK_CONTROL_MASK) {
				dates_zoom_out_cb (NULL, d);
				return TRUE;
			} else
				return FALSE;
		case GDK_KP_Add :
		case GDK_plus :
			if (event->state & GDK_CONTROL_MASK) {
				dates_zoom_in_cb (NULL, d);
				return TRUE;
			} else
				return FALSE;
#endif
	}
	
	return FALSE;
}

#ifdef WITH_HILDON
GtkWidget *
glade_hildon_window_new (GladeXML *xml, GType widget_type,
			 GladeWidgetInfo *info)
{
	return hildon_window_new ();
}

GtkWidget *
glade_hildon_date_editor_new (GladeXML *xml, GType widget_type,
			      GladeWidgetInfo *info)
{
	return hildon_date_editor_new ();
}

GtkWidget *
glade_hildon_time_editor_new (GladeXML *xml, GType widget_type,
			      GladeWidgetInfo *info)
{
	return hildon_time_editor_new ();
}

/* functions for hildon hibernation */

/* the osso state data size has to be fixed; this defines the space
 * we need for storing the time stamp (UTC string is at most 20 bytes)
 */
#define ICALTIME_MAX_SIZE 21 

/* Wrapper around the data we save by libosso API
 *
 * For now just a timestamp, but will need to save scroll later
 */
typedef struct 
{
	char t[ICALTIME_MAX_SIZE];
} StateData;

/* saves state data via libosso API
 *
 * The saved data survives application exit until next reboot
 */
gboolean
dates_save_state (DatesData *d)
{
	const icaltimetype * t;
	const char         * t_sz;
	osso_state_t         osd;
	StateData            sd;

#ifdef DEBUG
	if (d->debug & DATES_DEBUG_HIBERNATE)
		g_debug ("entered %s", G_STRFUNC);
#endif
	
	g_return_val_if_fail (d && d->view && d->osso_context, FALSE);

	t = dates_view_get_date (d->view);
	g_return_val_if_fail (t, FALSE);
	
	t_sz = icaltime_as_ical_string (*t);
	g_return_val_if_fail (t_sz, FALSE);

	memset (&sd, 0, sizeof (sd));
	strncpy (sd.t, t_sz, sizeof (sd.t));
	
	osd.state_size = sizeof(StateData);
	osd.state_data = (gpointer)&sd;

	if (osso_state_write (d->osso_context, &osd) == OSSO_OK)
	{
#ifdef DEBUG
		if (d->debug & DATES_DEBUG_HIBERNATE)
			g_debug ("success [%s]", sd.t);
#endif
		return TRUE;
	}

#ifdef DEBUG
	if (d->debug & DATES_DEBUG_HIBERNATE)
		g_debug ("saving state failed");
#endif
	return FALSE;
}

/* Load our state data from the libosso storage
 * 
 * We have a problem -- the system uses a dbus message to let us know that we
 * are comming out of hibernation, rather than starting afresh, but
 * unfortunately, it does not work (I think that message is dispatched *before*
 * we had a chance to register our callback, so we never see it. The problem is
 * cause by the fact that the state data is persistent, so we cannot load it
 * on every startup, but only when weaking up from hibernation.
 *
 * We work around this problem by replacing the state data with a single byte
 * placeholder everytime we get topped; we then use the size of the save data to
 * differentiate between real data to use for restore, and phony one to ignore.
 *
 * This hack will not work if the application is closed in some other way than
 * being background killed while in the background, but this should not be
 * happening in Maemo 2, as there is no way to close the up from the TN, but
 * might become issue in the future (hopefully, the will fix the API by then).
 */
gboolean
dates_restore_state (DatesData *d)
{
	icaltimetype t;
	osso_state_t osd;
	StateData    sd;
	char         c = 0;
	
#ifdef DEBUG
	if (d->debug & DATES_DEBUG_HIBERNATE)
		g_debug ("entered %s", G_STRFUNC);
#endif
	
	g_return_val_if_fail (d && d->view && d->osso_context, FALSE);

	osd.state_size = sizeof(c);
	osd.state_data = (gpointer)& c;
	
	if (osso_state_read (d->osso_context, & osd) == OSSO_OK) {
#ifdef DEBUG
		if (d->debug & DATES_DEBUG_HIBERNATE)
			g_debug ("state date contains place-holder only");
#endif
		d->restored = TRUE;
		return FALSE;
	}
	
	osd.state_size = sizeof(StateData);
	osd.state_data = (gpointer)& sd;
	
	if (osso_state_read (d->osso_context, & osd) != OSSO_OK) {
#ifdef DEBUG
		if (d->debug & DATES_DEBUG_HIBERNATE)
			g_debug ("Could not read saved state");
#endif
		return FALSE;
	}

#ifdef DEBUG
	if (d->debug & DATES_DEBUG_HIBERNATE)
		g_debug ("Retrieved saved date [%s]", sd.t);
#endif
	
	/* I hope this does not leak anything */
	t = icaltime_from_string (sd.t);
	dates_view_set_date (d->view, &t);

	d->restored = TRUE;
	
	return TRUE;
}

/* this is a hack working around the broken hibernation in hildon
 *
 * Basically, this function replaces any previously saved state data with data
 * of size sizeof(char) -- we then use the size of the saved data to
 * differentiate between real saved state and a mere placeholder
 */
gboolean
dates_clear_state (DatesData *d)
{
	osso_state_t         osd;
	char                 t = 0;

#ifdef DEBUG
	if (d->debug & DATES_DEBUG_HIBERNATE)
		g_debug ("entered %s", G_STRFUNC);
#endif
	
	g_return_val_if_fail (d && d->view && d->osso_context, FALSE);

	osd.state_size = sizeof(t);
	osd.state_data = (gpointer)&t;

	if (osso_state_write (d->osso_context, &osd) == OSSO_OK)
	{
#ifdef DEBUG
		if (d->debug & DATES_DEBUG_HIBERNATE)
			g_debug ("success");
#endif
		return TRUE;
	}
	
#ifdef DEBUG
	if (d->debug & DATES_DEBUG_HIBERNATE)
		g_debug ("clearing state failed");
#endif
	return FALSE;
}

static void
dates_is_topmost_notify (GObject *self,
			 GParamSpec *property_param,
			 DatesData * d)
{
	HildonProgram *program = HILDON_PROGRAM (self);
	if (hildon_program_get_is_topmost (program)){
#ifdef DEBUG
		if (d->debug & DATES_DEBUG_HIBERNATE)
			g_debug ("dates_is_topmost_notify: TRUE");
#endif

		/* if this is not the initial topping of the application,
		 * clear the state data
		 */
		if (d->restored) {
			dates_clear_state(d);
		}
		
		hildon_program_set_can_hibernate (program, FALSE);
	} else {
#ifdef DEBUG
		if (d->debug & DATES_DEBUG_HIBERNATE)
			g_debug ("dates_is_topmost_notify: FALSE");
#endif
		dates_save_state(d);
		hildon_program_set_can_hibernate (program, TRUE);
	}
}

static gboolean
dates_restore_timeout_cb (gpointer p)
{
	DatesData * d = (DatesData*)p;

#ifdef DEBUG
	if (d->debug & DATES_DEBUG_HIBERNATE)
		g_debug ("Entered restore timeout CB");
#endif

	if (d && d->restored) {
#ifdef DEBUG
		if (d->debug & DATES_DEBUG_HIBERNATE)
			g_debug ("State already restored");
#endif
		return FALSE;
	}
	
	if (!d || !d->init_done) {
#ifdef DEBUG
		if (d->debug & DATES_DEBUG_HIBERNATE)
			g_debug ("Initialisation not completed, yet ...");
#endif
		return TRUE;
	}

#ifdef DEBUG
	if (d->debug & DATES_DEBUG_HIBERNATE)
		g_debug ("restoring state");
#endif
	dates_restore_state (d);
	return FALSE;
}


static gint
dates_osso_rpc_event_cb (const gchar     *interface,
			 const gchar     *method,
			 GArray          *arguments,
			 gpointer         p,
			 osso_rpc_t      *retval)
{
	DatesData * d = (DatesData*) d;
	
#ifdef DEBUG
	if (d->debug & DATES_DEBUG_HIBERNATE)
		g_debug("osso_rpc_event_cb() interface [%s], method [%s]",
			interface, method);
#endif
	
	g_return_val_if_fail (d, OSSO_ERROR);

	
	/* the restored message can arrive *before* the app finished loading --
	 * if that happens we cannot retrive the state, so we install a
	 * callback to be called in 300ms intervals until the init is finished
	 */
	   
	if(!strcmp(method, "restored"))
	{
		if(d->init_done)
		{
#ifdef DEBUG
			if (d->debug & DATES_DEBUG_HIBERNATE)
				g_debug ("restoring state");
#endif
			dates_restore_state (d);
		}
		else
			g_timeout_add (300, dates_restore_timeout_cb, d);
	}

	return OSSO_OK;
}

static GtkToolItem *
load_tb_item (GtkWidget * toolbar,
	      const char * name, const char * fallback,
	      GCallback cb, gpointer data)
{
	GtkToolItem * tb;
	GtkWidget *icon;

	icon = gtk_image_new_from_icon_name (
		name, GTK_ICON_SIZE_SMALL_TOOLBAR);

	if (!icon) {
#ifdef DEBUG
		g_debug ("could not load theme icon [%s] -- using stock [%s]",
			 name, fallback);
#endif
		tb = gtk_tool_button_new_from_stock (fallback);
	} else
		tb = gtk_tool_button_new (icon, NULL);

	gtk_toolbar_insert (GTK_TOOLBAR (toolbar), tb, -1);
	
	g_signal_connect (G_OBJECT (tb), "clicked",
			  cb, data);
	
	return tb;
}

static GtkWidget *
create_toolbar (gpointer d)
{
	DatesData * data = (DatesData *)d;
	
	/* Create new GTK toolbar */
	GtkWidget * toolbar = gtk_toolbar_new ();

	/* Set toolbar properties */
	gtk_toolbar_set_orientation( GTK_TOOLBAR(toolbar),
				     GTK_ORIENTATION_HORIZONTAL);
	gtk_toolbar_set_style( GTK_TOOLBAR(toolbar),
			       GTK_TOOLBAR_BOTH_HORIZ);

	/* Make menus and buttons to toolbar: */
	/* Create toolitems using defined items from stock */
	data->TBNew = load_tb_item (toolbar,
		"qgn_list_messagin_editor", GTK_STOCK_NEW,
		G_CALLBACK(dates_new_cb), d);
	data->TBOpen = load_tb_item (toolbar,
		"qgn_list_gene_fldr_opn", GTK_STOCK_OPEN,
		G_CALLBACK(dates_open_cb), d);
	data->TBDelete = load_tb_item (toolbar,
		"qgn_list_gene_invalid",
		GTK_STOCK_DELETE,
		G_CALLBACK(dates_delete_cb), d);
		
        gtk_toolbar_insert (GTK_TOOLBAR(toolbar),
		GTK_TOOL_ITEM (GTK_SEPARATOR_TOOL_ITEM (
			gtk_separator_tool_item_new ())), -1);
		
	data->TBBack = load_tb_item (toolbar,
		"qgn_list_gene_back", GTK_STOCK_GO_BACK,
		G_CALLBACK(dates_back_cb), d);
	data->TBToday = load_tb_item (toolbar,
		"qgn_toolb_browser_home", GTK_STOCK_HOME,
		G_CALLBACK(dates_today_cb), d);
	data->TBForward = load_tb_item (toolbar,
		"qgn_list_gene_forward", GTK_STOCK_GO_FORWARD,
		G_CALLBACK(dates_forward_cb), d);
		
        gtk_toolbar_insert (GTK_TOOLBAR(toolbar),
		GTK_TOOL_ITEM (GTK_SEPARATOR_TOOL_ITEM (
			gtk_separator_tool_item_new ())), -1);
			
	data->TBZoomOut = load_tb_item (toolbar,
		"qgn_toolb_gene_zoomout", GTK_STOCK_ZOOM_OUT,
		G_CALLBACK(dates_zoom_out_cb), d);
	data->TBZoomIn = load_tb_item (toolbar,
		"qgn_toolb_gene_zoomin", GTK_STOCK_ZOOM_IN,
		G_CALLBACK(dates_zoom_in_cb), d);

	gtk_widget_show_all (GTK_WIDGET (toolbar));
	gtk_widget_hide (data->TBOpen);
	gtk_widget_hide (data->TBDelete);

	return toolbar;
}
#endif

gboolean
dates_button_press_cb (GtkWidget	*widget,
		       GdkEventButton	*event,
		       DatesData	*d)
{
	if ((event->button != 1) || (event->type != GDK_2BUTTON_PRESS) ||
	    (d->comp))
		return FALSE;
	
	dates_new_cb (widget, d);
	
	return TRUE;
}

/* Thanks to GnuCash to find out how to do this */
static void
dates_autoconnect (const gchar *handler_name, GObject *object,
			const gchar *signal_name, const gchar *signal_data,
			GObject *connect_object, gboolean after,
			gpointer user_data)
{
	static GModule *symbols = NULL;
	GCallback func;
	GCallback *pfunc = &func;
	
	if (!symbols) {
		symbols = g_module_open(NULL, 0);
	}
	
	if (!g_module_symbol (symbols, handler_name, (gpointer *)pfunc)) {
		g_warning ("Handler '%s' not found.", handler_name);
		return;
	}

	if (connect_object) {
		if (after)
			g_signal_connect_object (object, signal_name,
				func, connect_object, G_CONNECT_AFTER);
		else
			g_signal_connect_object (object, signal_name,
				func, connect_object, 0);
	} else {
		if (after)
			g_signal_connect_after(object, signal_name,
				func, user_data);
		else
			g_signal_connect(object, signal_name, func, user_data);
	}
}

int
main (int argc, char **argv)
{
	GtkWidget *widget;
	GladeXML *xml;
	DatesData data;
	GtkCellRenderer *renderer;
	GtkTreeModel *filter;
	GConfClient *client;
	GConfBridge *bridge;
	GOptionContext *context;
	static gint plug = 0;
#ifdef DEBUG
	const gchar *debug;
#endif
	
	static GOptionEntry entries[] = {
		{ "plug", 'p', 0, G_OPTION_ARG_INT, &plug,
			"Socket ID of an XEmbed socket to plug into", NULL },
		{ NULL }
	};

	/* Initialise the i18n support code */
	bindtextdomain (GETTEXT_PACKAGE, DATES_LOCALE_DIR);;
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);

	context = g_option_context_new (" - A light-weight, zooming calendar");
	g_option_context_add_main_entries (context, entries, GETTEXT_PACKAGE);
	g_option_context_add_group (context, gtk_get_option_group (TRUE));
	g_option_context_parse (context, &argc, &argv, NULL);

	/* Standard initialisation for gtk and glade */
	/* NOTE: No need to call gtk_init after the g_option_context_parse */
	/* gtk_init (&argc, &argv); */
	glade_init ();

#ifdef DEBUG
	debug = g_getenv ("DATES_DEBUG");
	if (debug)
		data.debug = g_parse_debug_string (debug,
			dates_debug_keys, G_N_ELEMENTS (dates_debug_keys));
#endif

#ifdef MY_LOG
	my_log = fopen ("/home/tomas/dates.log", "w");
#endif
	
#ifdef WITH_HILDON
	
	data.init_done = FALSE;
	data.restored  = FALSE;
	
	g_message ("Creating HildonProgram");
	g_set_application_name (_("Dates"));
	data.program = hildon_program_new ();

	g_message ("Initialising HildonProgram");
	data.osso_context = osso_initialize ("dates", "0.1", TRUE, NULL);

	if (!data.osso_context) {
		g_critical("Could not initialize libosso\n");
		return 1;
	}

	if (osso_rpc_set_default_cb_f(data.osso_context,
				      dates_osso_rpc_event_cb,
				      &data) != OSSO_OK)
	{
		g_warning ("Could not register rpc callback");
	}
	
	
	glade_register_widget (HILDON_TYPE_WINDOW,
		glade_hildon_window_new,
		glade_standard_build_children,
		NULL);
	glade_register_widget (HILDON_TYPE_DATE_EDITOR,
		glade_hildon_date_editor_new,
		NULL, NULL);
	glade_register_widget (HILDON_TYPE_TIME_EDITOR,
		glade_hildon_time_editor_new,
		NULL, NULL);

#endif
	
	/* Set critical errors to close application */
	g_log_set_always_fatal (G_LOG_LEVEL_CRITICAL);

	/* Set default icon name */
#ifndef WITH_HILDON
	gtk_window_set_default_icon_name ("stock_calendar");
#endif

	/* Load up main_window from interface xml, quit if xml file not found */
	xml = glade_xml_new (XML_FILE, NULL, NULL);
	if (!xml)
		g_critical ("Could not find interface XML file '%s'",
			    XML_FILE);
	
	widget = glade_xml_get_widget (xml, "dates_view");
	g_assert(widget);
	
#ifdef WITH_HILDON
	g_object_set(G_OBJECT(gtk_widget_get_settings(glade_xml_get_widget(xml,
							"main_window"))),
			      "gtk-button-images",
			      FALSE, NULL );
	
	hildon_program_set_common_menu(data.program,
		GTK_MENU(glade_xml_get_widget (xml, "main_menubar")));
	hildon_program_set_common_toolbar (data.program,
					GTK_TOOLBAR (create_toolbar (&data)));
	
	dates_view_set_use_dragbox (DATES_VIEW (widget), FALSE);
#endif
	client       = gconf_client_get_default ();
	data.view    = DATES_VIEW (widget);
	data.zoom    = gconf_client_get_int (client, DATES_GCONF_ZOOM, NULL);
	data.xml     = xml;
	data.comp    = NULL;
	data.widgets = NULL;
	data.waiting = NONE;

#ifndef WITH_HILDON
	data.TBZoomOut = glade_xml_get_widget (xml, "zoom_out_button");
	data.TBZoomIn  = glade_xml_get_widget (xml, "zoom_in_button");
	data.TBBack    = glade_xml_get_widget (xml, "back_button");
	data.TBForward = glade_xml_get_widget (xml, "forward_button");
	data.TBToday   = glade_xml_get_widget (xml, "today_button");
	data.TBOpen    = glade_xml_get_widget (xml, "open_button");
	data.TBDelete  = glade_xml_get_widget (xml, "delete_button");
	data.TBNew     = glade_xml_get_widget (xml, "new_button");
#endif
	
	if (data.zoom == 0) {
		data.zoom = 3;
		gtk_widget_set_sensitive (data.TBZoomIn, FALSE);
	} else if (data.zoom == 3) {
		gtk_widget_set_sensitive (data.TBZoomIn, FALSE);
	} else if (data.zoom == 16) {
		gtk_widget_set_sensitive (data.TBZoomOut, FALSE);
	}

	gtk_widget_set_sensitive (data.TBDelete, FALSE);
	
/* Perhaps in a future version? */
/*	dates_view_set_use_list (data.view, TRUE);*/

	dates_zoom_change (data.zoom, data.view);

	/* TODO: UI for multiple calendars */
	/* Load calendars */
	data.cal_list_store = gtk_list_store_new (COL_LAST, G_TYPE_STRING,
		G_TYPE_POINTER, G_TYPE_BOOLEAN);
	data.first_load = FALSE;
	if (!dates_load_calendars (&data)) {
		/* Ensure system calendar exists */
		data.first_load = TRUE;
		ECal *ecal = e_cal_new_system_calendar ();

#ifdef DEBUG
		if (data.debug & DATES_DEBUG_CALENDAR)
			g_debug ("Trying to create/open system calendar...");
#endif
		g_assert (ecal);

		g_signal_connect (G_OBJECT (ecal), "cal_opened",
				  G_CALLBACK (dates_cal_open_cb), &data);

		e_cal_open_async (ecal, FALSE);
	}
	
	/* Set transient parent for dialogs */
	gtk_window_set_transient_for (
		GTK_WINDOW (glade_xml_get_widget (xml, "details_dialog")),
		GTK_WINDOW (glade_xml_get_widget (xml, "main_window")));
	gtk_window_set_transient_for (
		GTK_WINDOW (glade_xml_get_widget (xml, "time_dialog")),
		GTK_WINDOW (glade_xml_get_widget (xml, "details_dialog")));
	gtk_window_set_transient_for (
		GTK_WINDOW (glade_xml_get_widget (xml, "exceptions_dialog")),
		GTK_WINDOW (glade_xml_get_widget (xml, "details_dialog")));
	
#ifdef WITH_HILDON
	/* HildonTime/DateEditor are broken, show_all them */
	gtk_widget_show_all (glade_xml_get_widget (xml, "time_date_editor"));
	gtk_widget_show_all (glade_xml_get_widget (xml, "time_time_editor"));
#endif
	
	/* Create tree model filter to filter out read-only calendars */
	filter = gtk_tree_model_filter_new (
		GTK_TREE_MODEL (data.cal_list_store), NULL);
	gtk_tree_model_filter_set_visible_column (
		GTK_TREE_MODEL_FILTER (filter), COL_NOTREADONLY);
	
	/* Set tree model for details dialog calendar selector */
	widget = glade_xml_get_widget (xml, "details_calendar_combobox");
	gtk_combo_box_set_model (GTK_COMBO_BOX (widget), filter);
	renderer = gtk_cell_renderer_text_new ();
	gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (widget), renderer, TRUE);
	gtk_cell_layout_add_attribute (GTK_CELL_LAYOUT (widget),
		renderer, "text", COL_CALNAME);
	
	/* Set nice colours for full-screen date header */
	gtk_widget_set_state (glade_xml_get_widget (xml, "header_eventbox"),
		GTK_STATE_SELECTED);
	gtk_widget_set_state (glade_xml_get_widget (xml, "header_label"),
		GTK_STATE_SELECTED);
	dates_date_changed_cb (data.view, &data);

	/* Bind widgets to gconf properties */
	bridge = gconf_bridge_get ();
	gconf_bridge_bind_window (bridge, DATES_GCONF_PREFIX, GTK_WINDOW (
		glade_xml_get_widget (xml, "main_window")), TRUE, TRUE);
/*	gconf_bridge_bind_property (bridge, DATES_GCONF_PREFIX,
		G_OBJECT (data.view), "week_start");*/

	/* Connect signals */
	glade_xml_signal_autoconnect_full (xml, dates_autoconnect, &data);

	/* Calendar selection signals */
	gconf_client_add_dir (client, CALENDAR_GCONF_PREFIX,
		GCONF_CLIENT_PRELOAD_NONE, NULL);
	gconf_client_notify_add (client, CALENDAR_GCONF_SELECTED_CALENDARS,
		dates_gconf_selected_cb, &data, NULL, NULL);
	
	widget = glade_xml_get_widget (xml, "main_window");
#ifdef WITH_HILDON
	hildon_program_add_window (data.program, HILDON_WINDOW (widget));
	g_signal_connect (G_OBJECT (widget), "key_press_event",
			  G_CALLBACK (dates_key_press_cb), &data);

	g_signal_connect (G_OBJECT (data.program), "notify::is-topmost",
			  G_CALLBACK (dates_is_topmost_notify), &data);

	data.init_done = TRUE;

	if (!data.restored)
		dates_restore_state (&data);	
#else
	g_signal_connect (G_OBJECT (widget), "key_press_event",
			  G_CALLBACK (dates_key_press_cb), &data);
#endif

	if (plug > 0) {
		GtkWidget *plug_widget;
		GtkWidget *contents;
		
#ifdef DEBUG
		if (data.debug & DATES_DEBUG_XEMBED)
			g_debug ("Plugging into socket %d", plug);
#endif
		plug_widget = gtk_plug_new (plug);
		contents = g_object_ref (gtk_bin_get_child (GTK_BIN (widget)));
		gtk_container_remove (GTK_CONTAINER (widget), contents);
		gtk_container_add (GTK_CONTAINER (plug_widget), contents);
		g_object_unref (contents);
		g_signal_connect (G_OBJECT (plug_widget), "destroy",
				  G_CALLBACK (gtk_main_quit), NULL);
		widget = glade_xml_get_widget (xml, "main_menubar");
		gtk_widget_hide (widget);
		widget = glade_xml_get_widget (xml, "header_eventbox");
		gtk_widget_show (widget);
		gtk_widget_show (plug_widget);
	} else {
		gtk_widget_show (widget);
	}

	gtk_main ();

#ifdef MY_LOG
	fclose (my_log);
#endif
	return 0;
}
