/* gtkcellrendererpixbuf.c
 * Copyright (C) 2000  Red Hat, Inc.,  Jonathan Blandford <jrb@redhat.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <gnome.h>
#include "gtkcellrendererpixbufx.h"
//#include "gtkintl.h"

static void gtk_cell_renderer_pixbufx_get_property  (GObject                    *object,
						    guint                       param_id,
						    GValue                     *value,
						    GParamSpec                 *pspec);
static void gtk_cell_renderer_pixbufx_set_property  (GObject                    *object,
						    guint                       param_id,
						    const GValue               *value,
						    GParamSpec                 *pspec);
static void gtk_cell_renderer_pixbufx_init       (GtkCellRendererPixbufx      *celltext);
static void gtk_cell_renderer_pixbufx_class_init (GtkCellRendererPixbufxClass *class);
static void gtk_cell_renderer_pixbufx_finalize   (GObject                    *object);
static void gtk_cell_renderer_pixbufx_create_stock_pixbuf (GtkCellRendererPixbufx *cellpixbuf,
							  GtkWidget             *widget);
static void gtk_cell_renderer_pixbufx_get_size   (GtkCellRenderer            *cell,
						 GtkWidget                  *widget,
						 GdkRectangle               *rectangle,
						 gint                       *x_offset,
						 gint                       *y_offset,
						 gint                       *width,
						 gint                       *height);
static void gtk_cell_renderer_pixbufx_render     (GtkCellRenderer            *cell,
						 GdkWindow                  *window,
						 GtkWidget                  *widget,
						 GdkRectangle               *background_area,
						 GdkRectangle               *cell_area,
						 GdkRectangle               *expose_area,
						 GtkCellRendererState        flags);


enum {
	PROP_ZERO,
	PROP_PIXBUF,
	PROP_PIXBUF_EXPANDER_OPEN,
	PROP_PIXBUF_EXPANDER_CLOSED,
	PROP_STOCK_ID,
	PROP_STOCK_SIZE,
	PROP_STOCK_DETAIL,
	PROP_ROWSPACING,
	PROP_BACKGROUND,
	PROP_BACKGROUND_GDK,
	PROP_BACKGROUND_SET,
};

static gpointer parent_class;

#define CELLINFO_KEY "gtk-cell-renderer-pixbuf-info"

typedef struct _GtkCellRendererPixbufxInfo GtkCellRendererPixbufxInfo;
struct _GtkCellRendererPixbufxInfo
{
  gchar *stock_id;
  GtkIconSize stock_size;
  gchar *stock_detail;
};

GType
gtk_cell_renderer_pixbufx_get_type (void)
{
  static GType cell_pixbuf_type = 0;

  if (!cell_pixbuf_type)
    {
      static const GTypeInfo cell_pixbuf_info =
      {
	sizeof (GtkCellRendererPixbufxClass),
	NULL,		/* base_init */
	NULL,		/* base_finalize */
	(GClassInitFunc) gtk_cell_renderer_pixbufx_class_init,
	NULL,		/* class_finalize */
	NULL,		/* class_data */
	sizeof (GtkCellRendererPixbufx),
	0,              /* n_preallocs */
	(GInstanceInitFunc) gtk_cell_renderer_pixbufx_init,
      };

      cell_pixbuf_type =
	g_type_register_static (GTK_TYPE_CELL_RENDERER, "GtkCellRendererPixbufx",
			        &cell_pixbuf_info, 0);
    }

  return cell_pixbuf_type;
}

static void
gtk_cell_renderer_pixbufx_init (GtkCellRendererPixbufx *cellpixbuf)
{
  GtkCellRendererPixbufxInfo *cellinfo;

  cellinfo = g_new0 (GtkCellRendererPixbufxInfo, 1);
  cellinfo->stock_size = GTK_ICON_SIZE_MENU;
  g_object_set_data (G_OBJECT (cellpixbuf), CELLINFO_KEY, cellinfo);
}

static void
gtk_cell_renderer_pixbufx_class_init (GtkCellRendererPixbufxClass *class)
{
  GObjectClass *object_class = G_OBJECT_CLASS (class);
  GtkCellRendererClass *cell_class = GTK_CELL_RENDERER_CLASS (class);

  parent_class = g_type_class_peek_parent (class);

  object_class->finalize = gtk_cell_renderer_pixbufx_finalize;

  object_class->get_property = gtk_cell_renderer_pixbufx_get_property;
  object_class->set_property = gtk_cell_renderer_pixbufx_set_property;

  cell_class->get_size = gtk_cell_renderer_pixbufx_get_size;
  cell_class->render = gtk_cell_renderer_pixbufx_render;

  g_object_class_install_property (object_class,
				   PROP_PIXBUF,
				   g_param_spec_object ("pixbuf",
							_("Pixbuf Object"),
							_("The pixbuf to render"),
							GDK_TYPE_PIXBUF,
							G_PARAM_READABLE |
							G_PARAM_WRITABLE));

  g_object_class_install_property (object_class,
				   PROP_PIXBUF_EXPANDER_OPEN,
				   g_param_spec_object ("pixbuf_expander_open",
							_("Pixbuf Expander Open"),
							_("Pixbuf for open expander"),
							GDK_TYPE_PIXBUF,
							G_PARAM_READABLE |
							G_PARAM_WRITABLE));

  g_object_class_install_property (object_class,
				   PROP_PIXBUF_EXPANDER_CLOSED,
				   g_param_spec_object ("pixbuf_expander_closed",
							_("Pixbuf Expander Closed"),
							_("Pixbuf for closed expander"),
							GDK_TYPE_PIXBUF,
							G_PARAM_READABLE |
							G_PARAM_WRITABLE));

  g_object_class_install_property (object_class,
				   PROP_STOCK_ID,
				   g_param_spec_string ("stock_id",
							_("Stock ID"),
							_("The stock ID of the stock icon to render"),
							NULL,
							G_PARAM_READWRITE));

  g_object_class_install_property (object_class,
				   PROP_STOCK_SIZE,
				   g_param_spec_enum ("stock_size",
						      _("Size"),
						      _("The size of the rendered icon"),
						      GTK_TYPE_ICON_SIZE,
						      GTK_ICON_SIZE_MENU,
						      G_PARAM_READWRITE));

  g_object_class_install_property (object_class,
				   PROP_STOCK_DETAIL,
				   g_param_spec_string ("stock_detail",
							_("Detail"),
							_("Render detail to pass to the theme engine"),
							NULL,
							G_PARAM_READWRITE));

  g_object_class_install_property (object_class,
				   PROP_ROWSPACING,
				   g_param_spec_boolean ("rowspacing",
							_("Rowspacing"),
							_("Put an empty line at the bottom of the row"),
							TRUE,
							G_PARAM_READABLE | G_PARAM_WRITABLE));

  g_object_class_install_property (object_class,
				   PROP_BACKGROUND,
				   g_param_spec_string ("background",
							_("Background color name"),
							_("Background color as a string"),
							NULL,
							G_PARAM_WRITABLE));
  g_object_class_install_property (object_class,
                                   PROP_BACKGROUND_GDK,
                                   g_param_spec_boxed ("background_gdk",
                                                       _("Background color"),
                                                       _("Background color as a GdkColor"),
                                                       GDK_TYPE_COLOR,
                                                       G_PARAM_READABLE | G_PARAM_WRITABLE));



#define ADD_SET_PROP(propname, propval, nick, blurb) g_object_class_install_property (object_class, propval, g_param_spec_boolean (propname, nick, blurb, FALSE, G_PARAM_READABLE | G_PARAM_WRITABLE))

  ADD_SET_PROP ("background_set", PROP_BACKGROUND_SET,
                _("Background set"),
                _("Whether this tag affects the background color"));



}

static void
gtk_cell_renderer_pixbufx_finalize (GObject *object)
{
  GtkCellRendererPixbufx *cellpixbuf = GTK_CELL_RENDERER_PIXBUFX (object);
  GtkCellRendererPixbufxInfo *cellinfo = g_object_get_data (object, CELLINFO_KEY);

  if (cellpixbuf->pixbuf && cellinfo->stock_id)
    g_object_unref (cellpixbuf->pixbuf);

  if (cellinfo->stock_id)
    g_free (cellinfo->stock_id);

  if (cellinfo->stock_detail)
    g_free (cellinfo->stock_detail);

  g_free (cellinfo);
  g_object_set_data (object, CELLINFO_KEY, NULL);

  (* G_OBJECT_CLASS (parent_class)->finalize) (object);
}

static void
gtk_cell_renderer_pixbufx_get_property (GObject        *object,
				       guint           param_id,
				       GValue         *value,
				       GParamSpec     *pspec)
{
  GtkCellRendererPixbufx *cellpixbuf = GTK_CELL_RENDERER_PIXBUFX (object);
  GtkCellRendererPixbufxInfo *cellinfo = g_object_get_data (object, CELLINFO_KEY);
  
  switch (param_id)
    {
    case PROP_PIXBUF:
      g_value_set_object (value,
                          cellpixbuf->pixbuf ? G_OBJECT (cellpixbuf->pixbuf) : NULL);
      break;
    case PROP_PIXBUF_EXPANDER_OPEN:
      g_value_set_object (value,
                          cellpixbuf->pixbuf_expander_open ? G_OBJECT (cellpixbuf->pixbuf_expander_open) : NULL);
      break;
    case PROP_PIXBUF_EXPANDER_CLOSED:
      g_value_set_object (value,
                          cellpixbuf->pixbuf_expander_closed ? G_OBJECT (cellpixbuf->pixbuf_expander_closed) : NULL);
      break;
    case PROP_STOCK_ID:
      g_value_set_string (value, cellinfo->stock_id);
      break;
    case PROP_STOCK_SIZE:
      g_value_set_enum (value, cellinfo->stock_size);
      break;
    case PROP_STOCK_DETAIL:
      g_value_set_string (value, cellinfo->stock_detail);
      break;

	 case PROP_ROWSPACING:
		g_value_set_boolean (value, cellpixbuf->rowspacing);
		break;

	 case PROP_BACKGROUND_GDK:
      {
        GdkColor color;

        color.red = cellpixbuf->background.red;
        color.green = cellpixbuf->background.green;
        color.blue = cellpixbuf->background.blue;

        g_value_set_boxed (value, &color);
      }
      break;


	 case PROP_BACKGROUND_SET:
		g_value_set_boolean(value, cellpixbuf->background_set);
		break;

		case PROP_BACKGROUND:
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
      break;
    }
}


static void
set_bg_color (GtkCellRendererPixbufx *celltext,
              GdkColor            *color)
{
  if (color)
    {
      if (!celltext->background_set)
        {
          celltext->background_set = TRUE;
          g_object_notify (G_OBJECT (celltext), "background_set");
        }

      celltext->background.red = color->red;
      celltext->background.green = color->green;
      celltext->background.blue = color->blue;
    }
  else
    {
      if (celltext->background_set)
        {
          celltext->background_set = FALSE;
          g_object_notify (G_OBJECT (celltext), "background_set");
        }
    }
}


static void
gtk_cell_renderer_pixbufx_set_property (GObject      *object,
				       guint         param_id,
				       const GValue *value,
				       GParamSpec   *pspec)
{
  GdkPixbuf *pixbuf;
  GtkCellRendererPixbufx *cellpixbuf = GTK_CELL_RENDERER_PIXBUFX (object);
  GtkCellRendererPixbufxInfo *cellinfo = g_object_get_data (object, CELLINFO_KEY);
  
  switch (param_id)
    {
    case PROP_PIXBUF:
      pixbuf = (GdkPixbuf*) g_value_get_object (value);
      if (pixbuf)
        g_object_ref (pixbuf);
      if (cellpixbuf->pixbuf)
	g_object_unref (cellpixbuf->pixbuf);
      cellpixbuf->pixbuf = pixbuf;
      break;
    case PROP_PIXBUF_EXPANDER_OPEN:
      pixbuf = (GdkPixbuf*) g_value_get_object (value);
      if (pixbuf)
        g_object_ref (pixbuf);
      if (cellpixbuf->pixbuf_expander_open)
	g_object_unref (cellpixbuf->pixbuf_expander_open);
      cellpixbuf->pixbuf_expander_open = pixbuf;
      break;
    case PROP_PIXBUF_EXPANDER_CLOSED:
      pixbuf = (GdkPixbuf*) g_value_get_object (value);
      if (pixbuf)
        g_object_ref (pixbuf);
      if (cellpixbuf->pixbuf_expander_closed)
	g_object_unref (cellpixbuf->pixbuf_expander_closed);
      cellpixbuf->pixbuf_expander_closed = pixbuf;
      break;
    case PROP_STOCK_ID:
      if (cellinfo->stock_id)
        g_free (cellinfo->stock_id);
      cellinfo->stock_id = g_strdup (g_value_get_string (value));
      break;
    case PROP_STOCK_SIZE:
      cellinfo->stock_size = g_value_get_enum (value);
      break;
    case PROP_STOCK_DETAIL:
      if (cellinfo->stock_detail)
        g_free (cellinfo->stock_detail);
      cellinfo->stock_detail = g_strdup (g_value_get_string (value));
      break;
	 case PROP_ROWSPACING:
	   cellpixbuf->rowspacing = g_value_get_boolean(value);
		break;

	 case PROP_BACKGROUND:
      {
        GdkColor color;

        if (!g_value_get_string (value))
          set_bg_color (cellpixbuf, NULL);       /* reset to backgrounmd_set to FALSE */
        else if (gdk_color_parse (g_value_get_string (value), &color))
          set_bg_color (cellpixbuf, &color);
        else
          g_warning ("Don't know color `%s'", g_value_get_string (value));
   
        g_object_notify (object, "background_gdk");
      }
		break;
    case PROP_BACKGROUND_GDK:
      /* This notifies the GObject itself. */
      set_bg_color (cellpixbuf, g_value_get_boxed (value));
      break;


    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
      break;
    }

  if (cellpixbuf->pixbuf && cellinfo->stock_id)
    {
      g_object_unref (cellpixbuf->pixbuf);
      cellpixbuf->pixbuf = NULL;
    }
}

/**
 * gtk_cell_renderer_pixbufx_new:
 * 
 * Creates a new #GtkCellRendererPixbufx. Adjust rendering
 * parameters using object properties. Object properties can be set
 * globally (with g_object_set()). Also, with #GtkTreeViewColumn, you
 * can bind a property to a value in a #GtkTreeModel. For example, you
 * can bind the "pixbuf" property on the cell renderer to a pixbuf value
 * in the model, thus rendering a different image in each row of the
 * #GtkTreeView.
 * 
 * Return value: the new cell renderer
 **/
GtkCellRenderer *
gtk_cell_renderer_pixbufx_new (void)
{
  return g_object_new (GTK_TYPE_CELL_RENDERER_PIXBUFX, NULL);
}

static void
gtk_cell_renderer_pixbufx_create_stock_pixbuf (GtkCellRendererPixbufx *cellpixbuf,
					      GtkWidget             *widget)
{
  GtkCellRendererPixbufxInfo *cellinfo = g_object_get_data (G_OBJECT (cellpixbuf), CELLINFO_KEY);

  if (cellpixbuf->pixbuf)
    g_object_unref (cellpixbuf->pixbuf);

  cellpixbuf->pixbuf = gtk_widget_render_icon (widget,
					       cellinfo->stock_id,
					       cellinfo->stock_size,
					       cellinfo->stock_detail);
}

static void
gtk_cell_renderer_pixbufx_get_size (GtkCellRenderer *cell,
				   GtkWidget       *widget,
				   GdkRectangle    *cell_area,
				   gint            *x_offset,
				   gint            *y_offset,
				   gint            *width,
				   gint            *height)
{
  GtkCellRendererPixbufx *cellpixbuf = (GtkCellRendererPixbufx *) cell;
  GtkCellRendererPixbufxInfo *cellinfo = g_object_get_data (G_OBJECT (cell), CELLINFO_KEY);
  gint pixbuf_width = 0;
  gint pixbuf_height = 0;
  gint calc_width;
  gint calc_height;

  if (!cellpixbuf->pixbuf && cellinfo->stock_id)
    gtk_cell_renderer_pixbufx_create_stock_pixbuf (cellpixbuf, widget);

  if (cellpixbuf->pixbuf)
    {
      pixbuf_width = gdk_pixbuf_get_width (cellpixbuf->pixbuf);
      pixbuf_height = gdk_pixbuf_get_height (cellpixbuf->pixbuf);
    }
  if (cellpixbuf->pixbuf_expander_open)
    {
      pixbuf_width = MAX (pixbuf_width, gdk_pixbuf_get_width (cellpixbuf->pixbuf_expander_open));
      pixbuf_height = MAX (pixbuf_height, gdk_pixbuf_get_height (cellpixbuf->pixbuf_expander_open));
    }
  if (cellpixbuf->pixbuf_expander_closed)
    {
      pixbuf_width = MAX (pixbuf_width, gdk_pixbuf_get_width (cellpixbuf->pixbuf_expander_closed));
      pixbuf_height = MAX (pixbuf_height, gdk_pixbuf_get_height (cellpixbuf->pixbuf_expander_closed));
    }
  
  calc_width  = (gint) cell->xpad * 2 + pixbuf_width;
  calc_height = (gint) cell->ypad * 2 + pixbuf_height;
  
  if (x_offset) *x_offset = 0;
  if (y_offset) *y_offset = 0;

  if (cell_area && pixbuf_width > 0 && pixbuf_height > 0)
    {
      if (x_offset)
	{
	  *x_offset = (cell->xalign *
                       (cell_area->width - calc_width - 2 * cell->xpad));
	  *x_offset = MAX (*x_offset, 0) + cell->xpad;
	}
      if (y_offset)
	{
	  *y_offset = (cell->yalign *
                       (cell_area->height - calc_height - 2 * cell->ypad));
	  *y_offset = MAX (*y_offset, 0) + cell->ypad;
	}
    }

  if (width)
    *width = calc_width;
  
  if (height)
    *height = calc_height;
}

static void
gtk_cell_renderer_pixbufx_render (GtkCellRenderer      *cell,
				 GdkWindow            *window,
				 GtkWidget            *widget,
				 GdkRectangle         *background_area,
				 GdkRectangle         *cell_area,
				 GdkRectangle         *expose_area,
				 GtkCellRendererState  flags)

{
  GtkCellRendererPixbufx *cellpixbuf = (GtkCellRendererPixbufx *) cell;
  GtkCellRendererPixbufxInfo *cellinfo = g_object_get_data (G_OBJECT (cell), CELLINFO_KEY);
  GdkPixbuf *pixbuf;
  GdkRectangle pix_rect;
  GdkRectangle draw_rect;
  gboolean stock_pixbuf = FALSE;
	GtkStateType state;


  pixbuf = cellpixbuf->pixbuf;
  if (cell->is_expander)
    {
      if (cell->is_expanded &&
	  cellpixbuf->pixbuf_expander_open != NULL)
	pixbuf = cellpixbuf->pixbuf_expander_open;
      else if (! cell->is_expanded &&
	       cellpixbuf->pixbuf_expander_closed != NULL)
	pixbuf = cellpixbuf->pixbuf_expander_closed;
    }

	/* ------------ background --------------- */
  if ((flags & GTK_CELL_RENDERER_SELECTED) == GTK_CELL_RENDERER_SELECTED)
    {
      if (GTK_WIDGET_HAS_FOCUS (widget))
   state = GTK_STATE_SELECTED;
      else
   state = GTK_STATE_ACTIVE;
    }
  else
    {
      if (GTK_WIDGET_STATE (widget) == GTK_STATE_INSENSITIVE)
   state = GTK_STATE_INSENSITIVE;
      else
   state = GTK_STATE_NORMAL;
    }

  if (cellpixbuf->background_set && (state != GTK_STATE_SELECTED && state != GTK_STATE_ACTIVE))
    {
      GdkColor color;
      GdkGC *gc;

      color.red = cellpixbuf->background.red;
      color.green = cellpixbuf->background.green;
      color.blue = cellpixbuf->background.blue;

      gc = gdk_gc_new (window);

      gdk_gc_set_rgb_fg_color (gc, &color);

      gdk_draw_rectangle (window,
                          gc,
                          TRUE,
#if 1
                          background_area->x,
                          background_area->y,
                          background_area->width+cell_area->x,
                          background_area->height);
#else
                          cell_area->x,
                          cell_area->y,
                          cell_area->width,
                          cell_area->height);
#endif

      g_object_unref (gc);
    }

	/* ------------ end of background --------------- */

  if (!pixbuf && !cellinfo->stock_id)
    return;
  else if (!pixbuf && cellinfo->stock_id)
    stock_pixbuf = TRUE;

  gtk_cell_renderer_pixbufx_get_size (cell, widget, cell_area,
				     &pix_rect.x,
				     &pix_rect.y,
				     &pix_rect.width,
				     &pix_rect.height);

  if (stock_pixbuf)
    pixbuf = cellpixbuf->pixbuf;
  
  pix_rect.x += cell_area->x;
  pix_rect.y += cell_area->y;
  pix_rect.width  -= cell->xpad * 2;
  pix_rect.height -= cell->ypad * 2;

  if (gdk_rectangle_intersect (cell_area, &pix_rect, &draw_rect) &&
      gdk_rectangle_intersect (expose_area, &draw_rect, &draw_rect))
    gdk_draw_pixbuf (window,
		     widget->style->black_gc,
		     pixbuf,
		     /* pixbuf 0, 0 is at pix_rect.x, pix_rect.y */
		     draw_rect.x - pix_rect.x,
		     draw_rect.y - pix_rect.y,
		     draw_rect.x,
		     draw_rect.y,
		     draw_rect.width,
		     draw_rect.height,
		     GDK_RGB_DITHER_NORMAL,
		     0, 0);
	if(cellpixbuf->rowspacing)
	{
	   GdkColor color;
      GdkGC *gc;

      color.red = 0xF800;
      color.green = 0xF800;
      color.blue = 0xF800;

      gc = gdk_gc_new (window);

      gdk_gc_set_rgb_fg_color (gc, &color);

      gdk_draw_line (window,
                          gc,
                          background_area->x,
                          background_area->y+background_area->height-1,
                          background_area->x+background_area->width-1+cell_area->x,
                          background_area->y+background_area->height-1);

      g_object_unref (gc);
	}
}
