/* file.c - Functions to read the status file for deborphan.
   Copyright (C) 2000 Cris van Pelt

   Distributed under the terms of the Artistic License.
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>

#include <sys/types.h>
#include <sys/stat.h>

#include <config.h>
#include <deborphan.h>
#include <xalloc.h>

#ifdef LOW_MEM
/* Be nice. This uses a little more memory than the longest line in your
 * status file (which probably means some 100 bytes).
 *
 * This is more or less how everything was handled before version 0.1.20.
 */
FILE *
statusopen(const char *file)
{
    FILE *f;

    f = fopen(file, "r");

    return f ? f : NULL;
}

char *
nextline(FILE **fp)
{
    char t[32];
    static int i =  32;
    static char *s;
    if (!s)
	s = xmalloc (i * sizeof(char));

    fgets(s, i-1, *fp);
    if (!s)
	return NULL;

    while (!strchr(s, '\n')) {
	if (!fgets(t, 31, *fp))
	    return NULL;
	s = xrealloc(s, i+=32);
	strcat(s, t);
    }
    *(strchr(s, '\n')) = '\0';

    return s;
}

#else
/* Don't be nice. This uses up as much memory as your status file (near a
 * megabyte). It is considerably faster, though.
 */
char *
statusopen(const char *file)
{
    int fd;
    char *buf;
    struct stat statbuf;

    fd = open(file, O_RDONLY);
    if (fd < 0)
	return NULL;

    fstat(fd, &statbuf);
    buf = (char *) xmalloc(statbuf.st_size + 1 * sizeof(char));

    if (read(fd, buf, statbuf.st_size) < statbuf.st_size) {
	close(fd);
	return NULL;
    }

    *(buf + statbuf.st_size + 1) = '\0';
    close(fd);
    return buf;
}

/* This function is similar to strsep(s, "\n"), but I have the feeling this
 * is faster, because it only checks for '\n'. Well, that, and I just like 
 * writing this *(s+1) stuff.
 */
char *
nextline(char **s)
{
    char *a = *s;
    char *n;
    
    if (!a)
	return NULL;
    for (n = *s; *n && *n != '\n'; n++);

    if (*n == '\0')
	return NULL;

    *n = '\0';
    *s = n + 1;

    return a;
}
#endif /* ! LOW_MEM */
