# Written by Bram Cohen
# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: Choker.py 266 2007-08-18 02:06:35Z camrdale-guest $

"""Contains the Choker class."""

from random import randrange, shuffle
from DebTorrent.clock import clock

class Choker:
    """Manages the choking and unchoking of other downloaders.
    
    @type config: C{dictonary}
    @ivar config: the configuration variables
    @type round_robin_period: C{int}
    @ivar round_robin_period: the number of seconds between the client's 
        switching of upload targets
    @type schedule: C{method}
    @ivar schedule: the method to call to schedule future events
    @type picker: L{PiecePicker}
    @ivar picker: the PiecePicker to get connection information from
    @type connections: C{list} of L{Connecter.Connection}
    @ivar connections: the connections from peers to the client
    @type last_preferred: C{int}
    @ivar last_preferred: the number of preferred connections found in the
        last examination
    @type last_round_robin: C{long}
    @ivar last_round_robin: the last time the connections were examined
    @type done: C{Event}
    @ivar done: flag to indicate when the download is complete
    @type super_seed: C{boolean}
    @ivar super_seed: whether super-seeding is enabled
    @type paused: C{Event}
    @ivar paused: flag to indicate when the download is paused
    
    """
    
    def __init__(self, config, schedule, picker, done = lambda: False):
        """Initialize the Choker instance.
        
        @type config: C{dictonary}
        @param config: the configuration variables
        @type schedule: C{method}
        @param schedule: the method to call to schedule future events
        @type picker: L{PiecePicker}
        @param picker: the piece picker to use to 
        @type done: C{Event}
        @param done: flag to indicate when the download is complete
        
        """
        
        self.config = config
        self.round_robin_period = config['round_robin_period']
        self.schedule = schedule
        self.picker = picker
        self.connections = []
        self.last_preferred = 0
        self.last_round_robin = clock()
        self.done = done
        self.super_seed = False
        self.paused = False
        schedule(self._round_robin, 5)

    def set_round_robin_period(self, x):
        """Set a new round-robin period.
        
        @type x: C{int}
        @param x: the new round-robin period
        @see: L{Choker.round_robin_period}
        
        """
        
        self.round_robin_period = x

    def _round_robin(self):
        """Periodically determine the ordering for connections and call the choker."""
        self.schedule(self._round_robin, 5)
        if self.super_seed:
            cons = range(len(self.connections))
            to_close = []
            count = self.config['min_uploads']-self.last_preferred
            if count > 0:   # optimization
                shuffle(cons)
            for c in cons:
                i = self.picker.next_have(self.connections[c], count > 0)
                if i is None:
                    continue
                if i < 0:
                    to_close.append(self.connections[c])
                    continue
                self.connections[c].send_have(i)
                count -= 1
            for c in to_close:
                c.close()
        if self.last_round_robin + self.round_robin_period < clock():
            self.last_round_robin = clock()
            for i in xrange(1, len(self.connections)):
                c = self.connections[i]
                u = c.get_upload()
                if u.is_choked() and u.is_interested():
                    self.connections = self.connections[i:] + self.connections[:i]
                    break
        self._rechoke()

    def _rechoke(self):
        """Unchoke some connections.
        
        Reads the current upload and download rates from the connections, 
        as well as the connection state, and unchokes the most preferable
        ones.
        
        """
        preferred = []
        maxuploads = self.config['max_uploads']
        if self.paused:
            for c in self.connections:
                c.get_upload().choke()
            return
        if maxuploads > 1:
            for c in self.connections:
                u = c.get_upload()
                if not u.is_interested():
                    continue
                if self.done():
                    r = u.get_rate()
                else:
                    d = c.get_download()
                    r = d.get_rate()
                    if r < 1000 or d.is_snubbed():
                        continue
                preferred.append((-r, c))
            self.last_preferred = len(preferred)
            preferred.sort()
            del preferred[maxuploads-1:]
            preferred = [x[1] for x in preferred]
        count = len(preferred)
        hit = False
        to_unchoke = []
        for c in self.connections:
            u = c.get_upload()
            if c in preferred:
                to_unchoke.append(u)
            else:
                if count < maxuploads or not hit:
                    to_unchoke.append(u)
                    if u.is_interested():
                        count += 1
                        hit = True
                else:
                    u.choke()
        for u in to_unchoke:
            u.unchoke()

    def connection_made(self, connection, p = None):
        """Adds a new connection to the list.
        
        @type connection: L{Connecter.Connection}
        @param connection: the connection to the client from the peer
        @type p: C{int}
        @param p: the location to insert the new connection into the list
            (optional, default is to choose a random location)
        
        """
        
        if p is None:
            p = randrange(-2, len(self.connections) + 1)
        self.connections.insert(max(p, 0), connection)
        self._rechoke()

    def connection_lost(self, connection):
        """Removes a lost connection from the list.
        
        @type connection: L{Connecter.Connection}
        @param connection: the connection to the client from the peer
        
        """
        
        self.connections.remove(connection)
        self.picker.lost_peer(connection)
        if connection.get_upload().is_interested() and not connection.get_upload().is_choked():
            self._rechoke()

    def interested(self, connection):
        """Indicate the connection is now interesting.
        
        @type connection: L{Connecter.Connection}
        @param connection: the connection to the client from the peer
        
        """
        
        if not connection.get_upload().is_choked():
            self._rechoke()

    def not_interested(self, connection):
        """Indicate the connection is no longer interesting.
        
        @type connection: L{Connecter.Connection}
        @param connection: the connection to the client from the peer
        
        """
        
        if not connection.get_upload().is_choked():
            self._rechoke()

    def set_super_seed(self):
        """Change to super seed state."""
        while self.connections:             # close all connections
            self.connections[0].close()
        self.picker.set_superseed()
        self.super_seed = True

    def pause(self, flag):
        """Pause the choker.
        
        @type flag: C{Event}
        @param flag: flag to indicate when pausing is finished
        
        """
        
        self.paused = flag
        self._rechoke()
