# Written by John Hoffman and Uoti Urpala
# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: parsedir.py 266 2007-08-18 02:06:35Z camrdale-guest $

"""Parse a directory for torrent files.

@type logger: C{logging.Logger}
@var logger: the logger to send all log messages to for this module

"""

from bencode import bencode, bdecode
from download_bt1 import get_response
from os.path import exists, isfile
from sha import sha
import sys, os, logging

logger = logging.getLogger('DebTorrent.parsedir')

def parsedir(directory, parsed, files, blocked,
             exts = ['dtorrent'], 
             return_metainfo = False):
    """Parse a directory for torrent files.
    
    Only works for .dtorrent files, it will not find or properly parse
    Packages files to extract the torrent from them.
    
    @type directory: C{string}
    @param directory: the directory to parse (somewhat recursively)
    @type parsed: C{dictionary}
    @param parsed: the cache of all torrent files that were ever found
    @type files: C{dictionary}
    @param files: the files that were previously found
    @type blocked: C{dictionary}
    @param blocked: the files that were previously blocked
    @type exts: C{list} of C{string}
    @param exts: the extensions to look for torrent files in
        (optional, defaults to 'dtorrent')
    @type return_metainfo: C{boolean}
    @param return_metainfo: whether to return the torrent metainfo 
        (optional, defaults to False)
    @rtype: (C{dictionary}, C{dictionary}, C{dictionary}, C{dictionary}, C{dictionary})
    @return: the cache of all torrents ever found, all the files found, all 
        the files blocked, the new torrents that were found, the torrents 
        that are now missing
    
    """
    
    logger.info('checking dir')
    dirs_to_check = [directory]
    new_files = {}
    new_blocked = {}
    torrent_type = {}
    while dirs_to_check:    # first, recurse directories and gather torrents
        directory = dirs_to_check.pop()
        newtorrents = False
        for f in os.listdir(directory):
            newtorrent = None
            for ext in exts:
                if f.endswith(ext):
                    newtorrent = ext[:]
                    break
            if newtorrent:
                newtorrents = True
                p = os.path.join(directory, f)
                new_files[p] = [(os.path.getmtime(p), os.path.getsize(p)), 0]
                torrent_type[p] = newtorrent
        if not newtorrents:
            for f in os.listdir(directory):
                p = os.path.join(directory, f)
                if os.path.isdir(p):
                    dirs_to_check.append(p)

    new_parsed = {}
    to_add = []
    added = {}
    removed = {}
    # files[path] = [(modification_time, size), hash], hash is 0 if the file
    # has not been successfully parsed
    for p,v in new_files.items():   # re-add old items and check for changes
        oldval = files.get(p)
        if not oldval:          # new file
            to_add.append(p)
            continue
        h = oldval[1]
        if oldval[0] == v[0]:   # file is unchanged from last parse
            if h:
                if blocked.has_key(p):  # parseable + blocked means duplicate
                    to_add.append(p)    # other duplicate may have gone away
                else:
                    new_parsed[h] = parsed[h]
                new_files[p] = oldval
            else:
                new_blocked[p] = 1  # same broken unparseable file
            continue
        if parsed.has_key(h) and not blocked.has_key(p):
            logger.debug('removing '+p+' (will re-add)')
            removed[h] = parsed[h]
        to_add.append(p)

    to_add.sort()
    for p in to_add:                # then, parse new and changed torrents
        new_file = new_files[p]
        v,h = new_file
        if new_parsed.has_key(h): # duplicate
            if not blocked.has_key(p) or files[p][0] != v:
                logger.warning(p+' is a duplicate torrent for '+new_parsed[h]['path'])
            new_blocked[p] = 1
            continue
                
        logger.debug('adding '+p)
        try:
            (d, garbage) = get_response(p, '')
            h = sha(bencode(d['info'])).digest()
            new_file[1] = h
            if new_parsed.has_key(h):
                logger.warning(p+' is a duplicate torrent for '+new_parsed[h]['path'])
                new_blocked[p] = 1
                continue

            a = {}
            a['path'] = p
            f = os.path.basename(p)
            a['file'] = f
            a['type'] = torrent_type[p]
            i = d['info']
            l = 0
            nf = 0
            if i.has_key('length'):
                l = i.get('length',0)
                nf = 1
            elif i.has_key('files'):
                for li in i['files']:
                    nf += 1
                    if li.has_key('length'):
                        l += li['length']
            a['numfiles'] = nf
            a['length'] = l
            a['name'] = d.get('name', f)
            
            def setkey(k, d = d, a = a):
                """Set the key of a dictionary with the value from another dictionary.
                
                @type k: unknown
                @param k: the key to set
                @type d: C{dictionary}
                @param d: the dictionary to get the value from
                @type a: C{dictionary}
                @param a: the dictionary to set the {k: d[k]} value in
                
                """
                
                if d.has_key(k):
                    a[k] = d[k]

            setkey('failure reason')
            setkey('warning message')
            setkey('announce-list')
            if return_metainfo:
                a['metainfo'] = d
        except:
            logger.exception(p+' has errors')
            new_blocked[p] = 1
            continue
        logger.debug('... successful')
        new_parsed[h] = a
        added[h] = a

    for p,v in files.items():       # and finally, mark removed torrents
        if not new_files.has_key(p) and not blocked.has_key(p):
            logger.info('removing '+p)
            removed[v[1]] = parsed[v[1]]

    logger.info('done checking')
    return (new_parsed, new_files, new_blocked, added, removed)

