;;; dh-elpa.el --- package.el style packages for Debian  -*- lexical-binding:t -*-

;; Copyright (C) 2015 David Bremner & contributors
;; Portions Copyright 2007-2015 The Free Software Foundation

;; Author: David Bremner <bremner@debian.org>
;; Created: 11 July 2015
;; Version: 0.0.2
;; Keywords: tools
;; Package-Requires: ((tabulated-list "1.0"))

;; This file is NOT part of GNU Emacs.

;; dh-elpa  is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; dh-elpa is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with dh-elpa.  If not, see <http://www.gnu.org/licenses/>.

(require 'package)
(require 'cl-lib)

;; Originally package-unpack from package.el in Emacs 24.5
(defun dhelpa-unpack (pkg-desc destdir &optional epoch-time)
  "Install the contents of the current buffer into DESTDIR as a package.
Optional argument EPOCH-TIME specifies time (as a string or
number) to use in autoload files; if unspecifed or nil the
current time is used."
  (let* ((name (package-desc-name pkg-desc))
         (dirname (package-desc-full-name pkg-desc))
	 (pkg-dir (expand-file-name dirname destdir))
	 (pkg-time (if epoch-time (seconds-to-time
				   (if (stringp epoch-time)
				       (string-to-number epoch-time)
				     epoch-time))
		     (current-time)))
	 (backup-inhibited t))
    (make-directory pkg-dir t)
    (pcase (package-desc-kind pkg-desc)
      (`tar
       (let ((default-directory (file-name-as-directory destdir)))
	 (package-untar-buffer dirname)))
      (`single
       (let ((el-file (expand-file-name (format "%s.el" name) pkg-dir)))
         (package--write-file-no-coding el-file)))
      (kind (error "Unknown package kind: %S" kind)))
    (defun dhelpa-autoload-insert-section-header (real-fun outbuf autoloads load-name file time)
      (funcall real-fun outbuf autoloads load-name file  pkg-time))
    (advice-add #'autoload-insert-section-header
		:around #'dhelpa-autoload-insert-section-header)
    (package--make-autoloads-and-stuff (dhelpa-filter-pkg-desc pkg-desc) pkg-dir)
    (advice-remove #'autoload-insert-section-header
		   #'dhelpa-autoload-insert-section-header)
    pkg-dir))

(defun dhelpa-filter-deps-for-debian (deps)
  "Filter a list of package.el deps DEPS for Debian.

Remove packages that are maintained outside of the elpa-*
namespace in Debian, plus Emacs itself."
  (let ((non-elpa (list 'emacs 's 'dash 'yasnippet)))
    (cl-remove-if (lambda (dep)
                    (or (memq (car dep) non-elpa)
                        (package-built-in-p (car dep))))
                  deps)))

(defun dhelpa-filter-pkg-desc (desc)
  "Filter the dependencies of package description DESC for Debian."
  (let ((our-desc (copy-package-desc desc)))
    (cl-callf dhelpa-filter-deps-for-debian (package-desc-reqs our-desc))
    our-desc))

(defun dhelpa-debianise-deps (deps)
  "Convert a list of package.el deps DEPS to debian/control format."
  (mapconcat
   (lambda (dep)
     (let ((pkg-name (format "elpa-%s" (car dep)))
           (pkg-ver (mapconcat 'number-to-string (car (cdr dep)) ".")))
       (concat pkg-name " (>= " pkg-ver ")")))
   deps ", "))

;; Write out (partial) package description in a form easily parsed by
;; non-lisp tools.
(defun dhelpa-write-desc (desc dest)
  (let* ((name (package-desc-name desc))
	 (version (package-version-join (package-desc-version desc)))
	 (deps (dhelpa-debianise-deps (package-desc-reqs (dhelpa-filter-pkg-desc desc))))
	 (desc-file (expand-file-name (format "%s.desc" name) dest)))
    (with-temp-file desc-file
      (insert (format "ELPA-Name: %s\n" name))
      (insert (format "ELPA-Version: %s\n" version))
      (insert (format "ELPA-Requires: %s\n" deps)))))

;;;###autoload
(defun dhelpa-install-from-buffer (destdir &optional epoch-time)
  "Install a package from the current buffer into DESTDIR.
The current buffer is assumed to be a single .el or .tar file
that follows the packaging guidelines; see info
node `(elisp)Packaging'. If EPOCH-TIME is non-nil, it specifies
the time (in seconds since the epoch) to be used in the generated
autoload files."
  (interactive "D")
  (let ((pkg-desc (if (derived-mode-p 'tar-mode)
                      (package-tar-file-info)
                    (package-buffer-info))))
    (dhelpa-unpack pkg-desc destdir epoch-time)
    pkg-desc))

;;;###autoload
(defun dhelpa-batch-install-file ()
  "Install third command line argument (an emacs lisp file or tar
file) into second command line argument (a directory). The
optional fourth argument specifies a destination for a package
description file."
  (apply #'dhelpa-install-file command-line-args-left))

;;;###autoload
(defun dhelpa-batch-install-directory ()
  "Install third command line argument (a directory containing a
multifile elpa package) into second command line argument (a
directory). An optional third command line argument specifies
where to make temporary files and write a descriptor file."
  (apply #'dhelpa-install-directory command-line-args-left))

;;;###autoload
(defun dhelpa-install-file (dest el-file &optional desc-dir epoch-time)
  "Install EL-FILE (an emacs lisp file or tar file) into DEST (a directory).
Optional DESC-DIR specifies where to write a simplified package description file.
Optional EPOCH-TIME specifies time to use in the generated autoload files."
  (with-temp-buffer
    (insert-file-contents-literally el-file)
    (when (string-match "\\.tar\\'" el-file) (tar-mode))
    (let ((desc (dhelpa-install-from-buffer (expand-file-name dest) epoch-time)))
      (when desc-dir
	(dhelpa-write-desc desc desc-dir)))))

;;;###autoload
(defun dhelpa-install-directory (dest elpa-dir &optional work-dir epoch-time)
  "Install ELPA-DIR (an unpacked elpa tarball) into DEST (a directory).
The directory must either be named `package' or
`package-version'. If a working directory WORK-DIR is specified,
cleaning up is the caller's responsibility. Optional EPOCH-TIME
specifies time to use in generated autoloads files."
  (unless (file-exists-p
           (expand-file-name (package--description-file elpa-dir) elpa-dir))
    (dhelpa-generate-pkg-file elpa-dir))
  (let ((desc (package-load-descriptor elpa-dir)))
    (if (not desc)
	(message "Could not compute version from directory %s" elpa-dir)
      (let* ((canonical-dir (package-desc-full-name desc))
	     (base-dir (file-name-nondirectory elpa-dir))
	     (parent-dir (file-name-directory elpa-dir))
	     (temp-dir (or work-dir (make-temp-file nil t)))
	     (tar-file (concat (expand-file-name canonical-dir temp-dir) ".tar"))
	     ;; this relies on GNU tar features.
	     (transform-command (concat "--transform=s/"
					(regexp-quote base-dir) "/" canonical-dir "/")))
	(call-process "tar" nil nil nil "--create" "-C" parent-dir transform-command
		      "--file" tar-file base-dir)
	(dhelpa-install-file dest tar-file work-dir epoch-time)
	(unless work-dir
	  (delete-file tar-file)
	  (delete-directory temp-dir))))))

(defun dhelpa-generate-pkg-file (pkg-dir)
  "Generate PKG-DIR/foo-pkg.el by consulting PKG-DIR/foo.el."
  (let* ((pkg-file (expand-file-name (package--description-file pkg-dir) pkg-dir))
         (root-file (replace-regexp-in-string "-pkg" "" pkg-file))
         (pkg-desc
          (condition-case nil
              (with-temp-buffer
                (find-file root-file)
                (package-buffer-info))
            (error (progn
                     (message "dh_elpa: couldn't generate -pkg file; please write one")
                     (kill-emacs -1)))))
         (filtered-pkg-desc (dhelpa-filter-pkg-desc pkg-desc)))
    ;; although the docstring for `package-generate-description-file'
    ;; says that it generates a description for single-file packages,
    ;; there is in fact no difference between the descriptions for
    ;; single-file and multifile packages
    (package-generate-description-file filtered-pkg-desc pkg-file)))

;;; dh-elpa.el ends here
