/*   Copyright (C) 1995-2000 Simon G. Vogl

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or (at
your option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
------------------------------------------------------------------------- */

/* With some changes from Kysti Mlkki <kmalkki@cc.hut.fi> and even
   Frodo Looijaard <frodol@dds.nl>. Then patched for the DMX4Linux
   drivers. */

#include <linux/version.h>
#include <linux/config.h>

#if !defined(CONFIG_I2C) && !(defined(CONFIG_I2C_MODULE) && CONFIG_I2C_MODULE==1)
#error DMX4Linux I2C devices need I2C support in your kernel
#endif

#include <linux/module.h>
#include <linux/init.h>
#include <linux/parport.h>

#include <linux/i2c-algo-bit.h>

static int type=0;
static int parport_nr=0;

struct i2c_par
{
  struct pardevice *pdev;
  struct i2c_adapter adapter;
  struct i2c_algo_bit_data bit_lp_data;
  struct i2c_par *next;
};

static struct i2c_par *adapter_list=NULL;

#define MODULE_NAME "i2c-simple-par"

/* ----- printer port defines ------------------------------------------*/
#ifdef LINUX_I2C_ORIGINAL_DRIVER
/* Pin Port  Inverted	name	*/
#define I2C_ON		0x20		/* 12 status N	paper		*/
					/* ... only for phil. not used  */
#define I2C_SDA		0x80		/*  9 data   N	data7		*/
#define I2C_SCL		0x08		/* 17 ctrl   N	dsel		*/

#define I2C_SDAIN	0x80		/* 11 stat   Y	busy		*/
#define I2C_SCLIN	0x08		/* 15 stat   Y	enable		*/

#define I2C_DMASK	0x7f
#define I2C_CMASK	0xf7
#elif defined(LINUX_I2C_QLC_DRIVER)
#define I2C_SDA		0x08		/*  5 data   N  D1              */
#define I2C_SCL		0x02		/*  3 data   N  D3              */

#define I2C_SDAIN	0x80		/* 11 stat   Y  busy            */
#define I2C_SCLIN	0x80		/* 11 stat   Y  busy            */

#define I2C_DMASK	(0xff-I2C_SDA)
#define I2C_CMASK	(0xff-I2C_SCL)
#else /* we really use this! */

#define I2C_SDA         0x02            /*  3 data   N  D1              */
#define I2C_SCL         0x80            /*  9 data   N  D7              */

#define I2C_SDAIN       0x08            /* 15 stat   Y  enable          */
#define I2C_SCLIN       0x80            /* 11 stat   Y  busy (inv)      */

#define I2C_DMASK	(0xff-I2C_SDA)
#define I2C_CMASK	(0xff-I2C_SCL)

#define I2C_PULLUPS     0x41            /* pull up resistors */
/*
 *   parport     ___            I2C bus
 *    pin2  o---[___]---.
 *    pin3  o----|<|----+-------> SDA
 *    pin15 o-----------'
 *               ___
 *    pin8  o---[___]---.
 *    pin9  o----|<|----+-------> SCL
 *    pin11 o-----------'
 *
 *    pin25 o-------------------> GND
 */
#endif

/* ----- local functions ----------------------------------------------	*/

/*
 * write to the data-port with only affekting bits that are set in the mask.
 */
void static parport_frob_data (struct parport *port,
			       unsigned char mask,
			       unsigned char val)
{
  static unsigned data_shadow_register = 0;
  data_shadow_register = (data_shadow_register & ~mask) | (val & mask);
  parport_write_data(port, data_shadow_register);
}



static void bit_lp_setscl(void *data, int state)
{
#if 0
  /*be cautious about state of the control register -
    touch only the one bit needed*/
  if (state)
    parport_write_control((struct parport *) data, parport_read_control((struct parport *) data)|I2C_SCL);
  else
    parport_write_control((struct parport *) data, parport_read_control((struct parport *) data)&I2C_CMASK);

#else
  parport_frob_data ((struct parport *) data, I2C_SCL, state?I2C_SCL:0);
#endif
}

static void bit_lp_setsda(void *data, int state)
{
#if 0
  if (state)
    parport_write_data((struct parport *) data, I2C_DMASK);
  else
    parport_write_data((struct parport *) data, I2C_SDA);
#else
  parport_frob_data ((struct parport *) data, I2C_SDA, state?I2C_SDA:0);
#endif
}

static int bit_lp_getscl(void *data)
{
  return (parport_read_status((struct parport *) data) & I2C_SCLIN)?0:I2C_SCLIN;
}

static int bit_lp_getsda(void *data)
{
  return parport_read_status((struct parport *) data) & I2C_SDAIN;
}

static void bit_lp_setscl2(void *data, int state)
{
  if (state)
    parport_write_data((struct parport *) data, parport_read_data((struct parport *) data)|0x1);
  else
    parport_write_data((struct parport *) data, parport_read_data((struct parport *) data)&0xfe);
}

static void bit_lp_setsda2(void *data, int state)
{
  if (state)
    parport_write_data((struct parport *) data, parport_read_data((struct parport *) data)|0x2);
  else
    parport_write_data((struct parport *) data, parport_read_data((struct parport *) data)&0xfd);
}

static int bit_lp_getsda2(void *data)
{
  return (parport_read_status((struct parport *) data) & PARPORT_STATUS_BUSY) ? 0 : 1;
}

static int bit_lp_reg(struct i2c_client *client)
{
  return 0;
}

static int bit_lp_unreg(struct i2c_client *client)
{
  return 0;
}

static void bit_lp_inc_use(struct i2c_adapter *adap)
{
  MOD_INC_USE_COUNT;
}

static void bit_lp_dec_use(struct i2c_adapter *adap)
{
  MOD_DEC_USE_COUNT;
}

/* ------------------------------------------------------------------------
 * Encapsulate the above functions in the correct operations structure.
 * This is only done when more than one hardware adapter is supported.
 */

static struct i2c_algo_bit_data bit_lp_data = {
 setsda: bit_lp_setsda,
 setscl: bit_lp_setscl,
 getsda: bit_lp_getsda,
 getscl: bit_lp_getscl,
 udelay: 80,
 mdelay: 80,
 timeout: 100,
};

static struct i2c_algo_bit_data bit_lp_data2 = {
 setsda: bit_lp_setsda2,
 setscl: bit_lp_setscl2,
 getsda: bit_lp_getsda2,
 udelay: 80,
 mdelay: 80,
 timeout: 100,
};

static struct i2c_adapter bit_lp_ops = {
  name: "I2C Parallel Port Adapter",
  id: I2C_HW_B_LP,
  inc_use: bit_lp_inc_use,
  dec_use: bit_lp_dec_use,
  client_register: bit_lp_reg,
  client_unregister: bit_lp_unreg,
};

static void i2c_parport_attach (struct parport *port)
{
  if(port->number == parport_nr)
    {
      struct i2c_par *adapter = kmalloc(sizeof(struct i2c_par), GFP_KERNEL);
      if (!adapter)
        {
          printk(KERN_ERR MODULE_NAME ": could not alloc.\n");
          return;
        }

      printk(KERN_INFO MODULE_NAME ": attaching to %s:%i\n", port->name, port->number);

      adapter->pdev = parport_register_device(port, MODULE_NAME,
					      NULL, NULL, NULL,
					      PARPORT_FLAG_EXCL,
					      NULL);
      if (!adapter->pdev)
	{
	  printk(KERN_ERR MODULE_NAME ": Unable to register parport.\n");
	  return;
	}

      adapter->adapter = bit_lp_ops;
      adapter->adapter.algo_data = &adapter->bit_lp_data;
      adapter->bit_lp_data = type ? bit_lp_data2 : bit_lp_data;
      adapter->bit_lp_data.data = port;

      /* reset hardware to sane state */
      parport_claim_or_block(adapter->pdev);
#ifdef I2C_PULLUPS
      parport_frob_data (port, 0xff, I2C_PULLUPS);
#endif
      bit_lp_setsda(port, 1);
      bit_lp_setscl(port, 1);
      parport_release(adapter->pdev);

      if (i2c_bit_add_bus(&adapter->adapter) < 0)
	{
	  printk(KERN_ERR MODULE_NAME ": Unable to register I2C.\n");
	  parport_unregister_device(adapter->pdev);
	  kfree(adapter);
	  return;		/* No good */
	}

      adapter->next = adapter_list;
      adapter_list = adapter;
    }
}

static void i2c_parport_detach (struct parport *port)
{
  struct i2c_par *adapter, *prev = NULL;

  for (adapter = adapter_list; adapter; adapter = adapter->next)
    {
      if (adapter->pdev->port == port)
	{
	  parport_unregister_device(adapter->pdev);
	  i2c_bit_del_bus(&adapter->adapter);
	  if (prev)
	    prev->next = adapter->next;
	  else
	    adapter_list = adapter->next;
	  kfree(adapter);
	  return;
	}
      prev = adapter;
    }
}

static struct parport_driver i2c_driver = {
  name: MODULE_NAME,
  attach: i2c_parport_attach,
  detach: i2c_parport_detach,
};

int __init i2c_bitlp_init(void)
{
  parport_register_driver(&i2c_driver);
  return 0;
}

void __exit i2c_bitlp_exit(void)
{
  parport_unregister_driver(&i2c_driver);
}

EXPORT_NO_SYMBOLS;

MODULE_AUTHOR("Simon G. Vogl <simon@tk.uni-linz.ac.at>, DMX4Linux team");
MODULE_DESCRIPTION("I2C-Bus adapter routines for simple parallel port adapter");
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,17)
MODULE_LICENSE("GPL");
#endif

MODULE_PARM(type, "i");
MODULE_PARM_DESC(type, "I2C access method (0,1)");

MODULE_PARM(parport_nr, "i");
MODULE_PARM_DESC(parport_nr, "parport number to use");

#ifdef MODULE
module_init(i2c_bitlp_init);
module_exit(i2c_bitlp_exit);
#endif
