/* Drip - a transcoder for Unix
 * Copyright (C) 2001-2003 Jarl van Katwijk
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#include <sys/types.h>
#include <glib.h>
#include "spu.h"

//#define DEBUG
#undef DEBUG
#define OVERLAY_RANGE 1000


static gfloat div2[OVERLAY_RANGE];


/* Proto's */
void overlay(guint8 *background[3],gint xb,gint yb,gint ob,struct spu_img_struct *spu,gdouble o1);
void overlay_init(void);


/* Mixing function for YV12 mode */
static inline void display_mix_function_yuv(guint32 color, guint32 contrast, guint8 *y_pixel, gint field, guint8 *u_pixel, guint8 *v_pixel) {
    register gint n = 0;
    register guint32 invcontrast = 256 - contrast;
    register guint8 y, u, v;
    register guint32 py;
    static gint odd;
    static guint32 pu, pv;

    u = (color >> 16) & 0xff;
    v = (color >> 8) & 0xff;
    y = color & 0xff;

    /* if no transparancy just overwrite */
    if (contrast == (0xf<<4)) {
        y_pixel[0] = y;
        /* only write uv on even columns and rows */
        if (!field) {
            gint odd = (gint)y_pixel & 1;
            for(n = odd; n < (2 - odd)>>1; n++) {
                u_pixel[n] = u;
                v_pixel[n] = v;
            }
        } 
    } else {
        py = y_pixel[0];
        py = (py * invcontrast + y * contrast) >> 8;
        y_pixel[0] = py;
        /* only write uv on even columns and rows */
        if (!field) {
            odd = (gint)y_pixel & 1;
            for (n = odd; n < (2 - odd)>>1; n++) {
                pu = u_pixel[n];
                pu = (pu * invcontrast + u * contrast) >> 8;
                u_pixel[n] = pu;
                pv = v_pixel[n];
                pv = (pv * invcontrast + v * contrast) >> 8;
                v_pixel[n] = pv;
            }
        }  
    }
    return;
}

/* Build some lookup tables */
void overlay_init() {
    gint i;
    for (i=0;i<OVERLAY_RANGE;i++) {
        div2[i]     = (gfloat)(i/2);
    }
    g_log(SPU_LD,G_LOG_LEVEL_DEBUG,"SPU: Overlaying initialised");
    return;
}


/* Main loop for overlaying SPU's onto unscaled frames */
void overlay(guint8 *background[3],gint xb,gint yb,gint ob,struct spu_img_struct *spu,gdouble o1) {
    static glong x,y,y2;
    static guint32 colour,contrast;
    static guchar *spu_data_ptr;
    static glong line_y,line_y2,offset,offset2;
    static gchar *addr,*addr_u, *addr_v;
    static guint16 field = 0;

    #ifdef DEBUG
    g_log(SPU_LD,G_LOG_LEVEL_DEBUG,"SPU: Overlaying x=%i, y=%i, spu->x=%i, spu->y=%i, spu->height=%i, spu->width=%i, spu->ybase=%i,  background[0] = %p, background[1] = %p, background[2] = %p",xb,yb,spu->x,spu->y,spu->height,spu->width,spu->ybase,background[0],background[1],background[2]);
    #endif
    spu_data_ptr = (guchar*)((glong)spu->data + spu->height * spu->width);
    addr = background[0];
    addr_u = background[1];
    addr_v = background[2];

    /* SPU render offset */
    y2 = spu->y - spu->cliptop;                       // Move subs upward
    if ((y2+spu->height) > (yb-spu->clipbottom)) 
        y2 -= (y2+spu->height)-(yb-spu->clipbottom);  // If subs fall of on bottom on frame, extra move upward

    for (y=spu->height-1; y>-1; y--) {
        line_y  = (y + y2) * xb;
        line_y2 = (glong)((div2[y + y2]) * div2[xb]);
        offset = line_y + spu->x + div2[spu->clipleft];
        for (x=spu->width-1; x>-1; x--) {
            if (line_y>-1) {                          // Anything to draw in visible area?
                /* Define colour and contrast */
                colour = (*spu_data_ptr)&0xF;
                contrast = (guint32)(((*spu_data_ptr)&0xF0)*o1);
                /* Overlay if not completely transparant */
                if (contrast != 0) {
                    addr = background[0] + offset + x;
                    offset2 = line_y2 + div2[x + spu->x + (glong)div2[spu->clipleft]];
                    addr_u = background[1] + offset2;
                    addr_v = background[2] + offset2;
                    #ifdef DEBUG
                    g_log(SPU_LD,G_LOG_LEVEL_DEBUG,"SPU: colour=%u (COL=%i), contrast=%u",spu_clut[colour],(*spu_data_ptr)&0xF,contrast);
                    #endif
                    display_mix_function_yuv(spu_clut[colour], contrast, addr, field, addr_u, addr_v);
                }
            }
            spu_data_ptr--;
        }
        field = 1-field;                              // even\odd fields 
    }
    return;
}

