/*
 *  ---------
 * |.**> <**.|  CardContact
 * |*       *|  Software & System Consulting
 * |*       *|  Minden, Germany
 * |**> <**|  Copyright (c) 1999. All rights reserved
 *  --------- 
 *
 * See file LICENSE for details on licensing
 *
 * Abstract :       Defines procedures for CTBCS commands
 *
 * Author :         Frank Thater (FTH)
 *
 * Last modified:   10/08/1999
 *
 *****************************************************************************/
 
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <string.h>
#include "eco5000.h" 
#include "ctbcs.h"
#include "ctapi.h"
#include "atr.h"
#include "defines.h"
#include "ecoT0.h"
#include "ecoT1.h"
#include "mc_2wbp.h"
#include "mc_3wbp.h"
#include "mc_sdap.h"


 
int setResponse (struct eco5000_t *ctx, 
                 unsigned char *cmd,
                 unsigned int *lr, 
                 unsigned char *rsp)

{
    unsigned char index = 0;
    unsigned char what = cmd[3] & 0x0F;
  
 
    switch(what) {
        case 0x01:   /* complete ATR     */
            if(*lr < ctx->LenOfATR + 2)
                return ERR_MEMORY;
            memcpy(rsp, ctx->ATR, ctx->LenOfATR);
            index = ctx->LenOfATR;
            rsp[index] = HIGH(SMARTCARD_SUCCESS);
            rsp[index+1] = ctx->Type == ASYNC_ICC ? 0x01 : 0x00;
            *lr = ctx->LenOfATR+2;
            break;
        case 0x02:   /* historical Bytes */
            if(*lr < ctx->NumOfHB + 2)
                return ERR_MEMORY;
            memcpy(rsp, ctx->HCC, ctx->NumOfHB);
            index = ctx->NumOfHB;
            rsp[index] = HIGH(SMARTCARD_SUCCESS);
            rsp[index+1] = ctx->Type == ASYNC_ICC ? 0x01 : 0x00;
            *lr = ctx->NumOfHB+2;
            break;
        default:     /* nothing          */
            memset(rsp, 0, sizeof(rsp));
            if(*lr < 2)
                return ERR_MEMORY;
            rsp[0] = HIGH(SMARTCARD_SUCCESS);
            rsp[1] = ctx->Type == ASYNC_ICC ? 0x01 : 0x00;
            *lr = 2;
    }
    
    return OK;
}


int InstallHandler (struct eco5000_t *ctx)

{
    switch(ctx->Protocol)   {
        case MC_2WBP:
            MC_2WBPInit (ctx);
            break;
     
        case MC_3WBP:
            MC_3WBPInit (ctx);
            break;

        case MC_SDAP:
            MC_SDAPInit (ctx);
            break;

        case 0x01:
            ecoT1Init (ctx); 
            break;
     
        default:
            ecoT0Init (ctx);
    }
   
    return OK;
}



int ResetTerminal (struct eco5000_t *ctx, 
                   unsigned int *lr, 
                   unsigned char *rsp)

{
    int response = 0;

    if(rs232Mode(ctx->fh, 9600, 'E', -1, -1, -1) < 0)
        return ERR_CT;
   
    rs232Flush(ctx->fh);
   
    if((response = ecoCommand(ctx, RESET, 0, NULL, 0, NULL)) < 0) {
        rsp[0] = HIGH(NOT_SUCCESSFUL);
        rsp[1] = LOW(NOT_SUCCESSFUL);
        *lr = 2;
        return ERR_TRANS;
    }

    ctx->Baud = 9600;
    ctx->Indirect = FALSE;

    rsp[0] = HIGH(SMARTCARD_SUCCESS);
    rsp[1] = LOW(SMARTCARD_SUCCESS);
    *lr = 2;
   
    return OK;
}



int ResetCard (struct eco5000_t *ctx,
               unsigned int lc,
               unsigned char *cmd, 
               unsigned int *lr, 
               unsigned char *rsp)

{
    int response = 0;
    int llr = 0;

    llr = *lr;

    response = ResetTerminal(ctx, lr, rsp);

    if (response < 0)
        return response;
   
    *lr = llr;

    if((GetATR(ctx)) < 0) {
        rsp[0] = HIGH(NOT_SUCCESSFUL);
        rsp[1] = LOW(NOT_SUCCESSFUL);
        *lr = 2;
        return OK;
    }
       
    /* check if explicit ATR was set */
    if ((lc > 4) && (cmd[4] >= 2))
        memcpy(ctx->ATR, (unsigned char *) &cmd[5], cmd[4]);

    if((response = InstallHandler(ctx)) < 0)
        return response;
   
    if((response = setResponse(ctx, cmd, lr, rsp)) < 0)
        return response;

    return OK;
}



int RequestICC (struct eco5000_t *ctx, 
                unsigned int lc,
                unsigned char *cmd, 
                unsigned int *lr, 
                unsigned char *rsp)

{
    unsigned char value;
    int status, timeout;
    
    if ((lc > 4) && (cmd[4] == 1))
        timeout = cmd[5];
    else
        timeout = 0;

    if((status = ecoCommand(ctx, READ_CARD_IN, 0, NULL, 1, &value)) < 0) {
        rsp[0] = HIGH(NOT_SUCCESSFUL);         /* Card present ?*/
        rsp[1] = LOW(NOT_SUCCESSFUL);      
        *lr = 2;
        return ERR_CT;
    }
   
    if(value == 0x01)   {       /* card already inserted ! */
        if((status = ecoCommand(ctx, READ_POWER, 0, NULL, 1, &value)) < 0) {
            rsp[0] = HIGH(NOT_SUCCESSFUL);         /* Card present ?*/
            rsp[1] = LOW(NOT_SUCCESSFUL);      
            *lr = 2;
            return ERR_CT;
        }
        if(value == 0x00)   {   /* card already powered !! */
            rsp[0] = HIGH(W_ICC_ALREADY_PRESENT);
            rsp[1] = LOW(W_ICC_ALREADY_PRESENT);
            *lr = 2;
            return OK;
        }
    }
    
#ifdef DEBUG
    printf("\nWaiting for %u seconds!\n", timeout);
#endif
    
    timeout*=4;
    
    do {
        if((status = ecoCommand(ctx, READ_CARD_IN, 0, NULL, 1, &value)) < 0) {
            rsp[0] = HIGH(NOT_SUCCESSFUL);
            rsp[1] = LOW(NOT_SUCCESSFUL);
            *lr = 2;
            return ERR_CT;
        }
      
        if ((value==0x01) || !timeout) {
            break;
        }

        usleep(250000);
        timeout--;
    } while(timeout);
   
    if(!timeout && (value != 0x01)) {
        rsp[0] = HIGH(W_NO_CARD_PRESENTED);
        rsp[1] = LOW(W_NO_CARD_PRESENTED);
        *lr = 2;
        return OK;
    }

#ifdef DEBUG   
    printf("\nCard detected!\n");
#endif   

    if((status = ResetCard(ctx, lc, cmd, lr, rsp)) < 0)
        return status;
         
    return OK;
}



int EjectICC (struct eco5000_t *ctx,
              unsigned int lc,
              unsigned char *cmd, 
              unsigned int *lr,
              unsigned char *rsp)
{
    int response;
    unsigned char value, save_timeout;
    unsigned char timeout;

    /* ECO 5000 has no display or other goodies, so check for correct P2 parameter */

    if (cmd[3] != 0x00) {
        rsp[0] = HIGH(WRONG_PARAMETERS_P1_P2);
        rsp[1] = LOW(WRONG_PARAMETERS_P1_P2);
        *lr = 2;
        return OK;
    }

    if ((lc > 4) && (cmd[4] > 0))
        timeout = cmd[5];
    else
        timeout = 0;

    save_timeout = timeout;

    if ((response = Async_PowerOff(ctx)) < 0) {
        rsp[0] = HIGH(NOT_SUCCESSFUL);
        rsp[1] = LOW(NOT_SUCCESSFUL);
        *lr = 2;
        return ERR_TRANS;
    }

    ctx->CTModFunc = NULL;
    ctx->LenOfATR = 0;
    ctx->Type = UNKNOWN;
    ctx->NumOfHB = 0;
    ctx->Protocol = 0;
    ctx->Indirect = FALSE;
    
    save_timeout *= 4;
   
    if(save_timeout > 0)  {
        do  {
        
            if((response = ecoCommand(ctx, READ_CARD_IN, 0, NULL, 1, &value)) < 0) {
                rsp[0] = HIGH(NOT_SUCCESSFUL);
                rsp[1] = LOW(NOT_SUCCESSFUL);
                *lr = 2;
                return ERR_CT;
            }
            
            if(value==0x00)   {
                break;
            }
            
            usleep(250000);
            
        } while(--save_timeout); 
        
    } else {                            /* Command OK,no timeout specified   */
        rsp[0] = HIGH(SMARTCARD_SUCCESS);
        rsp[1] = LOW(SMARTCARD_SUCCESS);
        *lr = 2;
        return OK;
    }
    
    if(save_timeout)  {                 /* Command OK, card removed          */
        rsp[0] = HIGH(SMARTCARD_SUCCESS);
        rsp[1] = LOW(SMARTCARD_SUCCESS);
        *lr = 2;
        return OK;
    }
    
    if((!save_timeout) && (timeout > 0)) {   /* warning: card not removed */
        rsp[0] = HIGH(W_NO_CARD_PRESENTED);
        rsp[1] = LOW(W_NO_CARD_PRESENTED);
        *lr = 2;
    }
    
    return OK;
}




int GetICCStatus (struct eco5000_t *ctx,
                  unsigned int *lr,
                  unsigned char *rsp)
{
    unsigned char response;
    int status;
      
    if((status = ecoCommand(ctx, READ_POWER, 0, NULL, 1, &response)) < 0) {
        rsp[0] = HIGH(NOT_SUCCESSFUL);
        rsp[1] = LOW(NOT_SUCCESSFUL);
        *lr = 2;
        return ERR_CT;
    }

    if(*lr < 5)
        return ERR_MEMORY;
   
    rsp[0] = 0x80;
    rsp[1] = 0x01;
    rsp[2] = 0x00;                        /* Get ICC Status DO                 */
      
    if(status==ACK_CVCC_OFF)
        rsp[2] |= 0x03;                   /* card in, no CVCC                  */

    if(status==ACK_CVCC_ON)
        rsp[2] |= 0x05;                   /* card in, CVCC on                  */

    rsp[3]=HIGH(SMARTCARD_SUCCESS);
    rsp[4]=LOW(SMARTCARD_SUCCESS);
    *lr=5;

    return OK;
}



int GetStatus (struct eco5000_t *ctx,
               unsigned char *cmd,
               unsigned int *lr,
               unsigned char *rsp)
{
    int response;
    unsigned char func_unit = cmd[2];
    unsigned char what = cmd[3];

#ifdef DEBUG   
    printf("\n%02x   %02x\n", func_unit, what);
#endif
   
    if(func_unit == 0x00)  {

        switch(what)    {       

            case 0x46:  
        
                if(*lr < 19)
                    return ERR_MEMORY;
                memcpy(rsp, "\x46\x0F" "DEORGECO50", 12);
                memcpy(rsp+12, ctx->Firmware, sizeof(ctx->Firmware));
                rsp[17] = HIGH(SMARTCARD_SUCCESS);
                rsp[18] = LOW(SMARTCARD_SUCCESS);
                *lr = 17+2; 
        
                break;
        
            case 0x80:  
                if((response = GetICCStatus(ctx, lr, rsp)) < 0)
                    return response;
                break;
        
            case 0x81:  
                if(*lr < 5)
                    return ERR_MEMORY;
        
                /* Get FU object */

                rsp[0] = 0x81;           /* TAG */
                rsp[1] = 0x01;           /* Length of following data */
                rsp[2] = 0x01;           /* Status for slot 1 */
                rsp[3] = HIGH(SMARTCARD_SUCCESS);
                rsp[4] = LOW(SMARTCARD_SUCCESS);
                *lr = 5;
                break;
        }
      
    } else {
        if((response = GetICCStatus(ctx, lr, rsp)) < 0) {
            return response;

        }
    }
    
    return OK;
}              





