/*
***************************************************************************
*
* Author: Teunis van Beelen
*
* Copyright (C) 2010 Teunis van Beelen
*
* teuniz@gmail.com
*
***************************************************************************
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation version 2 of the License.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License along
* with this program; if not, write to the Free Software Foundation, Inc.,
* 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*
***************************************************************************
*
* This version of GPL is at http://www.gnu.org/licenses/old-licenses/gpl-2.0.txt
*
***************************************************************************
*/



#include "export_annotations.h"



UI_ExportAnnotationswindow::UI_ExportAnnotationswindow(QWidget *parent)
{
  int i;

  mainwindow = (UI_Mainwindow *)parent;

  ExportAnnotsDialog = new QDialog;

  ExportAnnotsDialog->setMinimumSize(QSize(800, 225));
  ExportAnnotsDialog->setMaximumSize(QSize(800, 225));
  ExportAnnotsDialog->setWindowTitle("Export annotations");
  ExportAnnotsDialog->setModal(TRUE);
  ExportAnnotsDialog->setAttribute(Qt::WA_DeleteOnClose, TRUE);

  filelist = new QListWidget(ExportAnnotsDialog);
  filelist->setGeometry(QRect(10, 10, 780, 75));
  filelist->setSelectionBehavior(QAbstractItemView::SelectRows);
  filelist->setSelectionMode(QAbstractItemView::SingleSelection);
  for(i=0; i<mainwindow->files_open; i++)
  {
    new QListWidgetItem(mainwindow->edfheaderlist[i]->filename, filelist);
  }

  formatGroupBox = new QGroupBox("output format", ExportAnnotsDialog);
  formatGroupBox->setGeometry(10, 120, 200, 90);

  CVSRadioButton = new QRadioButton("comma separated text");
  CVSRadioButton->setChecked(TRUE);

  EDFplusRadioButton = new QRadioButton("EDFplus");

  formatVBoxLayout = new QVBoxLayout;
  formatVBoxLayout->addWidget(CVSRadioButton);
  formatVBoxLayout->addWidget(EDFplusRadioButton);
  formatGroupBox->setLayout(formatVBoxLayout);

  fileGroupBox = new QGroupBox("export annotations", ExportAnnotsDialog);
  fileGroupBox->setGeometry(240, 120, 200, 90);

  selectRadioButton = new QRadioButton("from selected file");
  selectRadioButton->setChecked(TRUE);

  mergeRadioButton = new QRadioButton("from all files (merge)");

  fileVBoxLayout = new QVBoxLayout;
  fileVBoxLayout->addWidget(selectRadioButton);
  fileVBoxLayout->addWidget(mergeRadioButton);
  fileGroupBox->setLayout(fileVBoxLayout);

  ExportButton = new QPushButton(ExportAnnotsDialog);
  ExportButton->setGeometry(QRect(500, 185, 100, 25));
  ExportButton->setText("Export");

  CloseButton = new QPushButton(ExportAnnotsDialog);
  CloseButton->setGeometry(QRect(690, 185, 100, 25));
  CloseButton->setText("Close");

  QObject::connect(CloseButton,  SIGNAL(clicked()), ExportAnnotsDialog, SLOT(close()));
  QObject::connect(ExportButton, SIGNAL(clicked()), this,               SLOT(ExportButtonClicked()));

  filelist->setCurrentRow(mainwindow->files_open - 1);

  ExportAnnotsDialog->exec();
}



void UI_ExportAnnotationswindow::ExportButtonClicked()
{
  int i, j, n,
      len,
      cvs_format,
      hdl,
      annot_cnt;

  char path[1024],
       str[1024];

  FILE *annotationfile;

  struct annotationblock *annot, *annot_list;

  struct edfhdrblock *hdr;

  struct date_time_struct tm;


  if(CVSRadioButton->isChecked() == TRUE)
  {
    cvs_format = 1;
  }
  else
  {
    cvs_format = 0;
  }

  if(!mainwindow->files_open)
  {
    ExportAnnotsDialog->close();
    return;
  }

  if(filelist->count() < 1)
  {
    ExportAnnotsDialog->close();
    return;
  }

  ExportButton->setEnabled(FALSE);
  CloseButton->setEnabled(FALSE);

  for(i=0; i<mainwindow->files_open; i++)
  {
    if(!strcmp(mainwindow->edfheaderlist[i]->filename, filelist->item(filelist->currentRow())->text().toLatin1().data()))
    {
      break;
    }
  }

  if(i==mainwindow->files_open)
  {
    ExportAnnotsDialog->close();
    return;
  }

  n = i;

  if(mergeRadioButton->isChecked() == TRUE)
  {
    n = mainwindow->sel_viewtime;
  }

  hdr = mainwindow->edfheaderlist[n];

  QFileDialog fchooser;
  fchooser.setFileMode(QFileDialog::AnyFile);
  fchooser.setAcceptMode(QFileDialog::AcceptSave);
  fchooser.setWindowTitle("Export annotations");
  fchooser.setLabelText(QFileDialog::FileName, "File");
  if(cvs_format)
  {
    fchooser.setDefaultSuffix("txt");
    fchooser.setFilter("Text files (*.txt *.TXT)");
  }
  else
  {
    fchooser.setDefaultSuffix("edf");
    fchooser.setFilter("EDF files (*.edf *.EDF)");
  }

  get_filename_from_path(path, mainwindow->edfheaderlist[n]->filename, 1024);
  remove_extension_from_filename(path);
  if(cvs_format)
  {
    strcat(path, "_annotations.txt");
  }
  else
  {
    strcat(path, "_annotations.edf");
  }
  fchooser.selectFile(path);
  if(mainwindow->recent_savedir[0] != 0)
  {
    fchooser.setDirectory(mainwindow->recent_savedir);
  }

  if(!(fchooser.exec() == QDialog::Accepted))
  {
    ExportAnnotsDialog->close();
    return;
  }

  strcpy(mainwindow->recent_savedir, fchooser.directory().absolutePath().toLatin1().data());

  strcpy(path, fchooser.selectedFiles().at(0).toLatin1().data());

  if(mainwindow->file_is_opened(path))
  {
    UI_Messagewindow popuperror("Export annotations", "Error, selected file is in use.");
    return;
  }

  annot_list = edfplus_annotation_copy_list(&mainwindow->annotationlist[n]);

  if(mergeRadioButton->isChecked() == TRUE)
  {
    for(i=0; i < mainwindow->files_open; i++)
    {
      if(i != mainwindow->sel_viewtime)
      {
        annot_cnt = edfplus_annotation_count(&mainwindow->annotationlist[i]);

        for(j=0; j < annot_cnt; j++)
        {
          edfplus_annotation_add_copy(&annot_list, edfplus_annotation_item(&mainwindow->annotationlist[i], j));

          annot = edfplus_annotation_item(&annot_list, edfplus_annotation_count(&annot_list) - 1);

          annot->onset -= (mainwindow->edfheaderlist[i]->viewtime - mainwindow->edfheaderlist[mainwindow->sel_viewtime]->viewtime);
        }
      }
    }

    edfplus_annotation_sort(&annot_list);
  }

  annot_cnt = edfplus_annotation_count(&annot_list);

///////////////////////////// CVS (text) export //////////////////////////////////////

  if(cvs_format)
  {
    annotationfile = fopen64(path, "wb");
    if(annotationfile==NULL)
    {
      UI_Messagewindow popuperror("Error", "Error, can not open annotationfile for writing.");
      ExportAnnotsDialog->close();
      edfplus_annotation_delete_list(&annot_list);
      return;
    }

    fprintf(annotationfile, "Onset,Duration,Annotation\n");

    for(j=0; j < annot_cnt; j++)
    {
      annot = edfplus_annotation_item(&annot_list, j);
      if(annot == NULL)
      {
        break;
      }
      strncpy(str, annot->annotation, 1024);
      str[1023] = 0;
      utf8_to_latin1(str);

      len = strlen(str);
      for(i=0; i<len; i++)
      {
        if((((unsigned char *)str)[i] < 32) || (((unsigned char *)str)[i] == ','))
        {
          str[i] = '.';
        }
      }

      fprintf(annotationfile, "%+.7f,%s,%s\n", (double)annot->onset / TIME_DIMENSION, annot->duration, str);
    }

    fclose(annotationfile);

    edfplus_annotation_delete_list(&annot_list);

    ExportAnnotsDialog->close();

    return;
  }

///////////////////////////// EDFplus export //////////////////////////////////////

  hdl = edfopen_file_writeonly(path, EDFLIB_FILETYPE_EDFPLUS, 0);
  if(hdl < 0)
  {
    switch(hdl)
    {
      case EDFLIB_MALLOC_ERROR : strcpy(str, "EDFlib: malloc error");
                                 break;
      case EDFLIB_NO_SUCH_FILE_OR_DIRECTORY : strcpy(str, "EDFlib: no such file or directory");
                                 break;
      case EDFLIB_MAXFILES_REACHED : strcpy(str, "EDFlib: maximum files reached");
                                 break;
      case EDFLIB_FILE_ALREADY_OPENED : strcpy(str, "EDFlib: file already opened");
                                 break;
      case EDFLIB_NUMBER_OF_SIGNALS_INVALID : strcpy(str, "EDFlib: number of signals is invalid");
                                 break;
      default : strcpy(str, "EDFlib: unknown error");
                                 break;
    }

    UI_Messagewindow popuperror("Error", str);
    edfplus_annotation_delete_list(&annot_list);
    ExportAnnotsDialog->close();
    return;
  }

  utc_to_date_time(hdr->utc_starttime, &tm);
  edf_set_startdatetime(hdl, tm.year, tm.month, tm.day, tm.hour, tm.minute, tm.second);

  if((hdr->edfplus) || (hdr->bdfplus))
  {
    edf_set_patientname(hdl, hdr->plus_patient_name);
    edf_set_patientcode(hdl, hdr->plus_patientcode);
    if(!(strcmp(hdr->plus_gender, "Male")))
    {
      edf_set_gender(hdl, 1);
    }
    if(!(strcmp(hdr->plus_gender, "Female")))
    {
      edf_set_gender(hdl, 0);
    }
    if(hdr->plus_birthdate[0] != 0)
    {
      if(!(strncmp(hdr->plus_birthdate + 3, "jan", 3)))  n = 1;
      if(!(strncmp(hdr->plus_birthdate + 3, "feb", 3)))  n = 2;
      if(!(strncmp(hdr->plus_birthdate + 3, "mar", 3)))  n = 3;
      if(!(strncmp(hdr->plus_birthdate + 3, "apr", 3)))  n = 4;
      if(!(strncmp(hdr->plus_birthdate + 3, "may", 3)))  n = 5;
      if(!(strncmp(hdr->plus_birthdate + 3, "jun", 3)))  n = 6;
      if(!(strncmp(hdr->plus_birthdate + 3, "jul", 3)))  n = 7;
      if(!(strncmp(hdr->plus_birthdate + 3, "aug", 3)))  n = 8;
      if(!(strncmp(hdr->plus_birthdate + 3, "sep", 3)))  n = 9;
      if(!(strncmp(hdr->plus_birthdate + 3, "oct", 3)))  n = 10;
      if(!(strncmp(hdr->plus_birthdate + 3, "nov", 3)))  n = 11;
      if(!(strncmp(hdr->plus_birthdate + 3, "dec", 3)))  n = 12;
      edf_set_birthdate(hdl, atoi(hdr->plus_birthdate + 7), n, atoi(hdr->plus_birthdate));
    }
    edf_set_patient_additional(hdl, hdr->plus_patient_additional);
    edf_set_admincode(hdl, hdr->plus_admincode);
    edf_set_technician(hdl, hdr->plus_technician);
    edf_set_equipment(hdl, hdr->plus_equipment);
    edf_set_recording_additional(hdl, hdr->plus_recording_additional);
  }
  else
  {
    edf_set_patientname(hdl, hdr->patient);
    edf_set_recording_additional(hdl, hdr->recording);
  }

  int hasdot,
      decimals;

  for(j=0; j < annot_cnt; j++)
  {
    annot = edfplus_annotation_item(&annot_list, j);
    if(annot == NULL)
    {
      break;
    }

    if(annot->duration[0] == 0)
    {
      edfwrite_annotation_utf8(hdl, annot->onset / 1000, -1, annot->annotation);
    }
    else
    {
      strcpy(str, annot->duration);

      len = strlen(str);

      hasdot = 0;

      for(i=0; i<len; i++)
      {
        if(str[i] == '.')
        {
          hasdot = 1;

          for(decimals=0; decimals<4; decimals++)
          {
            str[i] = str[i+1];

            if(str[i] == 0)
            {
              for( ; decimals<4; decimals++)
              {
                str[i] = '0';

                i++;
              }

              i--;
            }

            i++;
          }

          str[i] = 0;

          break;
        }
      }

      if(!hasdot)
      {
        strcat(str, "0000");
      }

      edfwrite_annotation_utf8(hdl, annot->onset / 1000, atoi(str), annot->annotation);
    }
  }

  if(edfclose_file(hdl) != 0)
  {
    UI_Messagewindow popuperror("Error", "An error occurred: edfclose_file()");
  }

  edfplus_annotation_delete_list(&annot_list);

  ExportAnnotsDialog->close();
}















