/* Copyright (C) 2001 Chris Vine

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/


#include <vector>

#include <cstdlib>
#include <iostream>
#include <fstream>
#include <strstream>
#include <iomanip>
#include <list>
#include <cstring>

#include <unistd.h>
#include <limits.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <dirent.h>

//#include <gtk--/scrollbar.h>
//#include <gtk--/adjustment.h>
#include <gtk--/pixmap.h>
#include <gdk/gdkkeysyms.h> // the key codes are here

#include "fax_list.h"
#include "dialogs.h"
#include "fax_list_icons.h"

int FaxListDialog::is_fax_received_list = 0;
int FaxListDialog::is_fax_sent_list = 0;


FaxListDialog::FaxListDialog(Mode mode_, const int standard_size_):
                              Gtk::Window(GTK_WINDOW_DIALOG),
			      mode(mode_), standard_size(standard_size_),
			      fax_list_box(false, 0), table(2, 1, false),
                              tool_bar(GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_ICONS),
                              close_button("Close"), fax_list(2) {

  // notify the existence of this object
  if (mode == received) is_fax_received_list++;
  else is_fax_sent_list++;

  fax_list_scroll_window.set_usize(standard_size * 13, standard_size * 7);

  fax_list_scroll_window.set_policy(GTK_POLICY_ALWAYS, GTK_POLICY_ALWAYS);
  // use Gtk::ScrolledWindow::add() not Gtk::ScrolledWindow::add_with_viewport()
  // to prevent the vertical scroll scrolling the titles
  fax_list_scroll_window.add(fax_list);

  // set up the fax list
  {
    using namespace Gtk::CList_Helpers;

    fax_list.set_column_title(0, "Fax");
    fax_list.set_column_title(1, "Description");
    fax_list.column_title_passive(0);
    //fax_list.column_title_passive(1);
    fax_list.column_title_active(1);
    fax_list.column_titles_show();
    fax_list.set_column_visibility(0, true);
    fax_list.set_column_visibility(1, true);
    fax_list.click_column.connect(SigC::slot(this, &FaxListDialog::describe_fax_prompt_witharg));

    // get the row list
    get_fax_list_rows();

    fax_list.set_selection_mode(GTK_SELECTION_SINGLE);
  }

  // set up the tool bar
  {
    using namespace Gtk::Toolbar_Helpers;
    
    // first make the pixmaps
    Gtk::Pixmap* printIcon_p = manage(new Gtk::Pixmap(print_xpm));
    Gtk::Pixmap* viewIcon_p = manage(new Gtk::Pixmap(view_xpm));
    Gtk::Pixmap* describeIcon_p = manage(new Gtk::Pixmap(describe_xpm));
    Gtk::Pixmap* deleteIcon_p = manage(new Gtk::Pixmap(delete_xpm));
    Gtk::Pixmap* refreshIcon_p = manage(new Gtk::Pixmap(refresh_xpm));

    tool_bar.set_tooltips(true);

    ToolList& tool_list = tool_bar.tools();
    tool_list.push_back(ButtonElem(*printIcon_p, SigC::slot(this, &FaxListDialog::print_fax_prompt),
				   "Print selected fax"));
    print_button_p = static_cast<Gtk::Button*>(tool_list.back()->get_widget());
    print_button_p->set_relief(GTK_RELIEF_NONE);
    print_button_p->set_sensitive(false);

    tool_list.push_back(ButtonElem(*viewIcon_p, SigC::slot(this, &FaxListDialog::view_fax),
				   "View selected fax"));
    view_button_p = static_cast<Gtk::Button*>(tool_list.back()->get_widget());
    view_button_p->set_relief(GTK_RELIEF_NONE);
    view_button_p->set_sensitive(false);

    tool_list.push_back(ButtonElem(*describeIcon_p, SigC::slot(this, &FaxListDialog::describe_fax_prompt),
				   "Add/amend description for selected fax"));
    describe_button_p = static_cast<Gtk::Button*>(tool_list.back()->get_widget());
    describe_button_p->set_relief(GTK_RELIEF_NONE);
    describe_button_p->set_sensitive(false);

    tool_list.push_back(ButtonElem(*deleteIcon_p, SigC::slot(this, &FaxListDialog::delete_fax_prompt),
				   "Delete selected fax"));
    delete_button_p = static_cast<Gtk::Button*>(tool_list.back()->get_widget());
    delete_button_p->set_relief(GTK_RELIEF_NONE);
    delete_button_p->set_sensitive(false);

    tool_list.push_back(ButtonElem(*refreshIcon_p, SigC::slot(this, &FaxListDialog::refresh_slot),
				   "Refresh fax list"));
    Gtk::Button* refresh_button_p = static_cast<Gtk::Button*>(tool_list.back()->get_widget());
    refresh_button_p->set_relief(GTK_RELIEF_NORMAL);
    refresh_button_p->set_sensitive(true);
  }


  table.attach(fax_list_scroll_window, 0, 1, 0, 1, GTK_EXPAND | GTK_FILL,
         GTK_EXPAND | GTK_FILL, standard_size/3, standard_size/3);
  table.attach(close_button, 0, 1, 1, 2, GTK_EXPAND,
	 0, standard_size/3, standard_size/3);

  fax_list_box.pack_start(tool_bar, false, false);
  fax_list_box.pack_start(table, true, true);

  close_button.clicked.connect(SigC::slot(this, &FaxListDialog::close_slot));
  close_button.set_usize(standard_size * 4, standard_size);

  fax_list.select_row.connect(SigC::slot(this, &FaxListDialog::set_buttons_slot));

  table.set_border_width(standard_size/3);

  if (mode == received) set_title("efax-gtk: Received fax list");
  else set_title("efax-gtk: Sent fax list");
  set_position(GTK_WIN_POS_CENTER);
  add(fax_list_box);

  show_all();
  // get the column headings to resize properly
  fax_list.columns_autosize();
}

FaxListDialog::~FaxListDialog(void) {
  // notify the destruction of this object
  if (mode == received) is_fax_received_list--;
  else is_fax_sent_list--;
}

void FaxListDialog::get_fax_list_rows(void) {

  using namespace Gtk::CList_Helpers;

  string dir(prog_config.homedir);
  if (mode == received) dir += "/faxin";
  else dir += "/faxsent";

  chdir(dir.c_str());

  DIR* dir_p;
  if ((dir_p = opendir(dir.c_str())) == 0) {
    string msg("Can't open directory ");
    msg += dir;
    msg += '\n';
    write_error(msg.c_str());
  }
  else {

    struct dirent* direntry;
    struct stat statinfo;
    list<string> dir_list;

    // first populate dir_list with the directory names in $HOME/faxin
    while ((direntry = readdir(dir_p)) != 0) {
      stat(direntry->d_name, &statinfo);
      if (S_ISDIR(statinfo.st_mode)
	  && strcmp(direntry->d_name, "oldfax")
	  && strcmp(direntry->d_name, ".")
	  && strcmp(direntry->d_name, "..")
	  && (mode == sent
	      || strcmp(direntry->d_name, prog_config.receive_dirname))) {
	dir_list.push_back(direntry->d_name);
      }
    }
    closedir(dir_p);

    // now insert the directory names in fax_list, with description (if any)
    // first get the row list (and clear it) -- we need to clear it here before
    // testing for an empty dir_list, or on deleting the last fax in the list,
    // the clear won't take out the former last entry
    RowList& row_list = fax_list.rows();
    row_list.clear();

    if (!dir_list.empty()) {
      // first sort them
      dir_list.sort();
      
      // prepare to get the fax description (if any)
      vector<string> list_entry;
      string filename;
      string line;
      ifstream file;
      list<string>::iterator iter;

      for (iter = dir_list.begin(); iter!= dir_list.end(); ++iter) {
      
	list_entry.push_back(*iter);
	filename = dir + '/';
	filename += *iter;
	filename += "/Description";

#ifdef HAVE_IOS_NOCREATE
	file.open(filename.c_str(), ios::in | ios::nocreate);
#else
	// we must have Std C++ so we probably don't need a ios::nocreate
	// flag on a read open to ensure uniqueness
	file.open(filename.c_str(), ios::in);
#endif

	if (file) {
	  while (getline(file, line) && line.empty());
        }
	list_entry.push_back(line);

	row_list.push_back(list_entry);

	list_entry.clear();
	filename = "";
	line = "";
	file.close();
	file.clear();
      }
    }
    fax_list.columns_autosize();
  }
  // reset current directory
  string temp(prog_config.homedir + "/faxin");
  chdir(temp.c_str());
}

void FaxListDialog::close_slot(void) {
  hide_all();
  delete this; // this is completely safe as the dialog is self-owning and modeless
}


gint FaxListDialog::delete_event_impl(GdkEventAny*) {
  close_slot();
  return true; // returning true prevents destroy sig being emitted
}

void FaxListDialog::set_buttons_slot(gint, gint, GdkEvent*) {

  if (!fax_list.selection().empty()) {
    print_button_p->set_relief(GTK_RELIEF_NORMAL);
    print_button_p->set_sensitive(true);

    view_button_p->set_relief(GTK_RELIEF_NORMAL);
    view_button_p->set_sensitive(true);

    describe_button_p->set_relief(GTK_RELIEF_NORMAL);
    describe_button_p->set_sensitive(true);

    delete_button_p->set_relief(GTK_RELIEF_NORMAL);
    delete_button_p->set_sensitive(true);
  }
    
  else {
    print_button_p->set_relief(GTK_RELIEF_NONE);
    print_button_p->set_sensitive(false);

    view_button_p->set_relief(GTK_RELIEF_NONE);
    view_button_p->set_sensitive(false);

    describe_button_p->set_relief(GTK_RELIEF_NONE);
    describe_button_p->set_sensitive(false);

    delete_button_p->set_relief(GTK_RELIEF_NONE);
    delete_button_p->set_sensitive(false);
  }
}

void FaxListDialog::describe_fax_prompt(void) {

  using namespace Gtk::CList_Helpers;

  selection = fax_list.selection();
  if (!selection.empty()) {
    string description;
    if (selection.front().size() > 1) description = selection.front()[1].get_text();
    
    DescriptionDialog* dialog_p = new DescriptionDialog(standard_size, description, *this);
    if (!dialog_p) {
      cerr << "Memory allocation error in FaxListDialog::init_describe_slot()" << endl;
      exit(MEM_ERROR);
    }
    dialog_p->accepted.connect(SigC::slot(this, &FaxListDialog::describe_fax));
    // there is no memory leak -- the memory will be deleted when DescriptionDialog closes
  }
}

void FaxListDialog::describe_fax(const string& description) {

  fax_list.cell(selection.front().get_row_num(), 1).set_text(description);

  string filename(prog_config.homedir);
  if (mode == received) filename += "/faxin/";
  else filename += "/faxsent/";

  filename += selection.front()[0].get_text();
  filename += "/Description";
  ofstream file(filename.c_str(), ios::out);
  if (file) file << description;
  else {
    string msg("Can't open file ");
    msg += filename;
    msg += '\n';
    write_error(msg.c_str());
  }
}

void FaxListDialog::print_fax_prompt(void) {

  using namespace Gtk::CList_Helpers;

  selection = fax_list.selection();
  if (!selection.empty() && !prog_config.print_cmd.empty()) {
    string faxnumber(selection.front()[0].get_text());
    string msg("Print fax ");
    msg += faxnumber + '?';
    
    PromptDialog* dialog_p = new PromptDialog(msg.c_str(), "Print fax", standard_size, *this);
    if (!dialog_p) {
      cerr << "Memory allocation error in FaxListDialog::print_fax_prompt()" << endl;
      exit(MEM_ERROR);
    }
    dialog_p->accepted.connect(SigC::slot(this, &FaxListDialog::print_fax));
    // there is no memory leak -- the memory will be deleted when PromptDialog closes
  }
}

void FaxListDialog::print_fax(void) {

  // get the base name of the fax to be printed
  // do this before we fork to avoid a possible race with another
  // FaxListDialog function resetting the value of FaxListDialog::selection
  string basename(prog_config.homedir);
  if (mode == received) basename += "/faxin/";
  else basename += "/faxsent/";

  basename += selection.front()[0].get_text();
  basename += '/';
  basename += selection.front()[0].get_text();
  basename += '.';

  // now launch a new process to control the printing process
  // the main program process needs to continue while the printing
  // is going on
  pid_t pid = fork();

  if (pid == -1) {
    write_error("Fork error - exiting\n");
    exit(FORK_ERROR);
  }
  if (!pid) { // child print process
      
    connect_to_stderr();

    // first set up the parms for efix
    vector<string> efix_parms;
    string temp;
    
    efix_parms.push_back("efix");
    efix_parms.push_back("-ve");
    efix_parms.push_back("-r300");
    efix_parms.push_back("-ops");
    temp = "-p";
    temp += prog_config.page_dim;
    efix_parms.push_back(temp);

    if (prog_config.print_shrink.compare("100")) { // if print_shrink is not 100
      temp = "-s0.";
      temp += prog_config.print_shrink;
      efix_parms.push_back(temp);
      
      ostrstream strm;
      if (!prog_config.page_size.compare("a4")) {

	strm << "-d";
	float val = 21.0 * (100 - atoi(prog_config.print_shrink.c_str()))/200.0;
	strm << val;
	val = 29.7 * (100 - atoi(prog_config.print_shrink.c_str()))/200.0;
	strm << ',' << val << "cm" << ends;
	
	const char* dim = strm.str();
	efix_parms.push_back(dim);
	delete[] dim;
      }

      else if (!prog_config.page_size.compare("letter")) {

	strm << "-d";
	float val = 8.465 * (100 - atoi(prog_config.print_shrink.c_str()))/200.0;
	strm << val;
	val = 11.0 * (100 - atoi(prog_config.print_shrink.c_str()))/200.0;
	strm << ',' << val << "in" << ends;
	
	const char* dim = strm.str();
	efix_parms.push_back(dim);
	delete[] dim;
      }

      else if (!prog_config.page_size.compare("legal")) {

	strm << "-d";
	float val = 8.465 * (100 - atoi(prog_config.print_shrink.c_str()))/200.0;
	strm << val;
	val = 14.0 * (100 - atoi(prog_config.print_shrink.c_str()))/200.0;
	strm << ',' << val << "in" << ends;
	
	const char* dim = strm.str();
	efix_parms.push_back(dim);
	delete[] dim;
      }
    }

    // now set up the parms for the print program
    vector<string> print_parms;
    string print_name;
    string::size_type end_pos;
    
    if ((end_pos = prog_config.print_cmd.find_first_of(' ')) != string::npos) { // we have parms
      print_name.assign(prog_config.print_cmd, 0, end_pos);
      print_parms.push_back(print_name);
      // find start of next parm
      string::size_type start_pos = prog_config.print_cmd.find_first_not_of(' ', end_pos);
      while (start_pos != string::npos) {
	end_pos = prog_config.print_cmd.find_first_of(' ', start_pos);
	if (end_pos != string::npos) {
	  print_parms.push_back(prog_config.print_cmd.substr(start_pos, end_pos - start_pos));
	  start_pos = prog_config.print_cmd.find_first_not_of(' ', end_pos); // prepare for next interation
	}
	else {
	  print_parms.push_back(prog_config.print_cmd.substr(start_pos, 
	     				   prog_config.print_cmd.size() - start_pos));
	  start_pos = end_pos;
	}
      }
    }

    else { // just a print command without parameters to be passed
      print_name = prog_config.print_cmd;
      print_parms.push_back(print_name);
    }

    int partnumber = 1;
    ostrstream strm;
    strm << basename.c_str() << setfill('0') << setw(3) << partnumber << ends;
    const char* file_name = strm.str();
    int result = access(file_name, R_OK);
    
    while (!result) {  // file OK -- now print it!
      pid_t pid = fork();   // make a process we will exec() to the print program from

      if (pid == -1) {
	write_error("Fork error\n");
	_exit(FORK_ERROR); // we have already forked, so use _exit() not exit()
      }
      if (!pid) {  // child process for printing
	Pipe_fifo fork_pipe(Pipe_fifo::block);
	
	pid_t pid = fork();  // make a process we will exec() to efix from

	if (pid == -1) {
	  write_error("Fork error\n");
	  _exit(FORK_ERROR); // we have already forked, so use _exit() not exit()
	}
	if (!pid) {  // child process for efix
	  fork_pipe.connect_to_stdout();
	  efix_parms.push_back(file_name);
	  char** exec_parms = new char*[efix_parms.size() + 1]; // this does not create a leak
                                                           // it will be deleted by the system
                                                           // when the child process terminates
	  char**  temp_pp = exec_parms;
	  vector<string>::iterator iter;
	  for (iter = efix_parms.begin(); iter != efix_parms.end(); ++iter, ++temp_pp) {
	    *temp_pp = new char[iter->size() + 1]; // this does not create a leak
	                                           // it will be deleted by the system
	                                           // when the child process terminates
	    strcpy(*temp_pp, iter->c_str());
	  }
	    
	  *temp_pp = 0;
	  execvp("efix", exec_parms);

	  // if we reached this point, then the execvp() call must have failed
	  write_error("Can't find the efix program - please check your installation\n"
		      "and the PATH environmental variable\n");
	  usleep(500000);
	  // use _exit(), not exit()
	  _exit(0);
	} // end of efix child process

	// this is the print program process
	fork_pipe.connect_to_stdin();

	char** exec_parms = new char*[print_parms.size() + 1]; // this does not create a leak
                                                           // it will be deleted by the system
                                                           // when the child process terminates
	char**  temp_pp = exec_parms;
	vector<string>::iterator iter;
	for (iter = print_parms.begin(); iter != print_parms.end(); ++iter, ++temp_pp) {
	  *temp_pp = new char[iter->size() + 1]; // this does not create a leak
	                                         // it will be deleted by the system
	                                         // when the child process terminates
	  strcpy(*temp_pp, iter->c_str());
	}
	    
	*temp_pp = 0;

	execvp(print_name.c_str(), exec_parms);

	// if we reached this point, then the execvp() call must have failed
	write_error("Can't find the print program - please check your installation\n"
		    "and the PATH environmental variable\n");
	usleep(500000);
	// use _exit(), not exit()
	_exit(0);
      } // end of print program process
	
      wait(0);

      delete[] file_name;
	
      partnumber++;
      ostrstream strm;
      strm << basename.c_str() << setfill('0') << setw(3) << partnumber << ends;
      file_name = strm.str();
      result = access(file_name, R_OK);
    }
    delete[] file_name;
      
    // this child process must end here - use _exit() not exit()
    _exit(1); 
  } // end of child print-control process
  // parent process -- continue servicing the main program
}

void FaxListDialog::view_fax(void) {

  selection = fax_list.selection();
  if (!selection.empty() && !prog_config.view_cmd.empty()) {
    string basename(prog_config.homedir);
    if (mode == received) basename += "/faxin/";
    else basename += "/faxsent/";

    basename += selection.front()[0].get_text();
    basename += '/';
    basename += selection.front()[0].get_text();
    basename += '.';

    if (prog_config.view_cmd[0] == '$') view_fax_multi_page(basename);
    else view_fax_single_page(basename);
  }
}

void FaxListDialog::view_fax_multi_page(const string& basename) {

  pid_t pid = fork();

  if (pid == -1) {
    write_error("Fork error\n");
    exit(FORK_ERROR);
  }
  if (!pid) { // child view-control process

    connect_to_stderr();

    // first set up the parms for efix
    vector<string> efix_parms;
    string temp;
    
    efix_parms.push_back("efix");
    efix_parms.push_back("-ve");
    efix_parms.push_back("-opgm");
    temp = "-r";
    temp += prog_config.view_res;
    efix_parms.push_back(temp);
    temp = "-p";
    temp += prog_config.page_dim;
    efix_parms.push_back(temp);

    // now set up the parms for the view program
    vector<string> view_parms;
    string view_name;
    string::size_type end_pos;
    
    if ((end_pos = prog_config.view_cmd.find_first_of(' ')) != string::npos) { // we have parms
      view_name.assign(prog_config.view_cmd, 1, end_pos);
      view_parms.push_back(view_name);
      // find start of next parm
      string::size_type start_pos = prog_config.view_cmd.find_first_not_of(' ', end_pos);
      while (start_pos != string::npos) {
	end_pos = prog_config.view_cmd.find_first_of(' ', start_pos);
	if (end_pos != string::npos) {
	  view_parms.push_back(prog_config.view_cmd.substr(start_pos, end_pos - start_pos));
	  start_pos = prog_config.view_cmd.find_first_not_of(' ', end_pos); // prepare for next interation
	}
	else {
	  view_parms.push_back(prog_config.view_cmd.substr(start_pos, 
						   prog_config.view_cmd.size() - start_pos));
	  start_pos = end_pos;
	}
      }
    }

    else { // just a print command without parameters to be passed
      view_name.assign(prog_config.view_cmd, 1, prog_config.view_cmd.size() - 1);
      view_parms.push_back(view_name);
    }

    vector<string> pgm_list;
    int partnumber = 1;
    ostrstream strm;
    strm << basename.c_str() << setfill('0') << setw(3) << partnumber << ends;
    const char* file_name = strm.str();
    int result = access(file_name, R_OK);

    while (!result) {  // file OK -- now add it to the efix and view parameters
      
      temp = file_name;
      temp += ".pgm";
      pgm_list.push_back(temp);
      view_parms.push_back(temp);

      pid_t pid = fork(); // create a process to exec() to efix from to create the PGM file for this file_name

      if (pid == -1) {
	write_error("Fork error\n");
	_exit(FORK_ERROR); // we have already forked, so use _exit() not exit()
      }
      if (!pid) {  // child process for efix
	
	temp = "-n";
	temp += pgm_list.back();
	efix_parms.push_back(temp);
	efix_parms.push_back(file_name);

	char** exec_parms = new char*[efix_parms.size() + 1]; // this does not create a leak
                                                              // it will be deleted by the system
                                                              // when the child process terminates
	char**  temp_pp = exec_parms;
	vector<string>::iterator iter;
	for (iter = efix_parms.begin(); iter != efix_parms.end(); ++iter, ++temp_pp) {
	  *temp_pp = new char[iter->size() + 1]; // this does not create a leak
	                                         // it will be deleted by the system
	                                         // when the child process terminates
	  strcpy(*temp_pp, iter->c_str());
	}
	    
	*temp_pp = 0;
	execvp("efix", exec_parms);

	// if we reached this point, then the execvp() call must have failed
	write_error("Can't find the efix program - please check your installation\n"
		    "and the PATH environmental variable\n");
	usleep(500000);
	// use _exit(), not exit()
	_exit(0);
      } // end of efix child process
      
      wait(0); // wait for the PGM file to be created

      delete[] file_name;
      partnumber++;
      ostrstream strm;
      strm << basename.c_str() << setfill('0') << setw(3) << partnumber << ends;
      file_name = strm.str();
      result = access(file_name, R_OK);
    }
    // we have finished creating the view program paramters and creating the PGM files
    // now clean up
    delete[] file_name;

    pid_t pid = fork(); // now create a process to exec() to the view program from

    if (pid == -1) {
      write_error("Fork error\n");
      _exit(FORK_ERROR); // we have already forked, so use _exit() not exit()
    }
    if (!pid) {  // child process for view program

      char** exec_parms = new char*[view_parms.size() + 1]; // this does not create a leak
                                                            // it will be deleted by the system
                                                            // when the child process terminates
      char**  temp_pp = exec_parms;
      vector<string>::iterator iter;
      for (iter = view_parms.begin(); iter != view_parms.end(); ++iter, ++temp_pp) {
	*temp_pp = new char[iter->size() + 1]; // this does not create a leak
	                                       // it will be deleted by the system
	                                       // when the child process terminates
	strcpy(*temp_pp, iter->c_str());
      }
      
      *temp_pp = 0;
      execvp(view_name.c_str(), exec_parms);

      // if we reached this point, then the execvp() call must have failed
      write_error("Can't find the fax view program - please check your installation\n"
		  "and the PATH environmental variable\n");
      usleep(500000);
      // use _exit(), not exit()
      _exit(0);
    } // end of view program process
    wait(0);
    // the view program has closed -- now delete the *.pgm files
    vector<string>::iterator iter;
    for (iter = pgm_list.begin(); iter != pgm_list.end(); ++iter) {
      unlink(iter->c_str());
    }
    // this child process must end here - use _exit() not exit()
    _exit(0);
  } // end of view-control process
  // parent process -- continue servicing the main program
}

void FaxListDialog::view_fax_single_page(const string& basename) {

  pid_t pid = fork();

  if (pid == -1) {
    write_error("Fork error\n");
    exit(FORK_ERROR);
  }
  else if (!pid) { // child view-control process
      
    connect_to_stderr();

    // first set up the parms for efix
    vector<string> efix_parms;
    string temp;
    
    efix_parms.push_back("efix");
    efix_parms.push_back("-ve");
    efix_parms.push_back("-opgm");
    temp = "-r";
    temp += prog_config.view_res;
    efix_parms.push_back(temp);
    temp = "-p";
    temp += prog_config.page_dim;
    efix_parms.push_back(temp);

    // now set up the parms for the view program
    vector<string> view_parms;
    string view_name;
    string::size_type end_pos;
    
    if ((end_pos = prog_config.view_cmd.find_first_of(' ')) != string::npos) { // we have parms
      view_name.assign(prog_config.view_cmd, 0, end_pos);
      view_parms.push_back(view_name);
      // find start of next parm
      string::size_type start_pos = prog_config.view_cmd.find_first_not_of(' ', end_pos);
      while (start_pos != string::npos) {
	end_pos = prog_config.view_cmd.find_first_of(' ', start_pos);
	if (end_pos != string::npos) {
	  view_parms.push_back(prog_config.view_cmd.substr(start_pos, end_pos - start_pos));
	  start_pos = prog_config.view_cmd.find_first_not_of(' ', end_pos); // prepare for next interation
	}
	else {
	  view_parms.push_back(prog_config.view_cmd.substr(start_pos, 
	     				   prog_config.view_cmd.size() - start_pos));
	  start_pos = end_pos;
	}
      }
    }

    else { // just a print command without parameters to be passed
      view_name = prog_config.view_cmd;
      view_parms.push_back(view_name);
    }

    int partnumber = 1;
    ostrstream strm;
    strm << basename.c_str() << setfill('0') << setw(3) << partnumber << ends;
    const char* file_name = strm.str();
    int result = access(file_name, R_OK);
    
    while (!result) {  // file OK -- now print it!
      pid_t pid = fork();  // make a process we will exec() to the view program from

      if (pid == -1) {
	write_error("Fork error\n");
	_exit(FORK_ERROR); // we have already forked, so use _exit() not exit()
      }
      if (!pid) {  // child process for view program
	Pipe_fifo fork_pipe(Pipe_fifo::block);
	
	pid_t pid = fork();  // make a process we will exec() to efix from

	if (pid == -1) {
	  write_error("Fork error\n");
	  _exit(FORK_ERROR); // we have already forked, so use _exit() not exit()
	}
	if (!pid) {  // child process for efix
	  fork_pipe.connect_to_stdout();
	  efix_parms.push_back(file_name);
	  char** exec_parms = new char*[efix_parms.size() + 1]; // this does not create a leak
                                                           // it will be deleted by the system
                                                           // when the child process terminates
	  char**  temp_pp = exec_parms;
	  vector<string>::iterator iter;
	  for (iter = efix_parms.begin(); iter != efix_parms.end(); ++iter, ++temp_pp) {
	    *temp_pp = new char[iter->size() + 1]; // this does not create a leak
	                                           // it will be deleted by the system
	                                           // when the child process terminates
	    strcpy(*temp_pp, iter->c_str());
	  }
	    
	  *temp_pp = 0;
	  execvp("efix", exec_parms);

	  // if we reached this point, then the execvp() call must have failed
	  write_error("Can't find the efix program - please check your installation\n"
		      "and the PATH environmental variable\n");
	  usleep(500000);
	  // use _exit(), not exit()
	  _exit(0);
	} // end of efix child process

	// this is the view program process
	fork_pipe.connect_to_stdin();
	
	char** exec_parms = new char*[view_parms.size() + 1]; // this does not create a leak
                                                              // it will be deleted by the system
                                                              // when the child process terminates
	char**  temp_pp = exec_parms;
	vector<string>::iterator iter;
	for (iter = view_parms.begin(); iter != view_parms.end(); ++iter, ++temp_pp) {
	  *temp_pp = new char[iter->size() + 1]; // this does not create a leak
	                                         // it will be deleted by the system
	                                         // when the child process terminates
	  strcpy(*temp_pp, iter->c_str());
	}
	    
	*temp_pp = 0;
	execvp(view_name.c_str(), exec_parms);

	// if we reached this point, then the execvp() call must have failed
	write_error("Can't find the fax view program - please check your installation\n"
		    "and the PATH environmental variable\n");
	usleep(500000);
	// use _exit(), not exit()
	_exit(0);
      } // end of view program process
	
      wait(0);

      delete[] file_name;
	
      partnumber++;
      ostrstream strm;
      strm << basename.c_str() << setfill('0') << setw(3) << partnumber << ends;
      file_name = strm.str();
      result = access(file_name, R_OK);
    }

    delete[] file_name;
      
    // this child process must end here - use _exit() not exit()
    _exit(1); 
  } // end of child view-control process
  // parent process -- continue servicing the main program
}

void FaxListDialog::delete_fax_prompt(void) {

  using namespace Gtk::CList_Helpers;

  selection = fax_list.selection();
  if (!selection.empty()) {
    string faxnumber(selection.front()[0].get_text());
    string msg("Delete fax ");
    msg += faxnumber + '?';
    
    PromptDialog* dialog_p = new PromptDialog(msg.c_str(), "Delete fax", standard_size, *this);
    if (!dialog_p) {
      cerr << "Memory allocation error in FaxListDialog::delete_fax_prompt()" << endl;
      exit(MEM_ERROR);
    }
    dialog_p->accepted.connect(SigC::slot(this, &FaxListDialog::delete_fax));
    // there is no memory leak -- the memory will be deleted when PromptDialog closes
  }
}

void FaxListDialog::delete_fax(void) {

  // get the name of the fax to be moved and the new name
  string old_dirname(prog_config.homedir);
  string new_dirname(prog_config.homedir);
  if (mode == received) {
    old_dirname += "/faxin/";
    new_dirname += "/faxin/oldfax/";
  }
  else {
    old_dirname += "/faxsent/";
    new_dirname += "/faxsent/oldfax/";
  }
  old_dirname += selection.front()[0].get_text();
  new_dirname += selection.front()[0].get_text();

  // make a new directory to copy the files into
  mkdir(new_dirname.c_str(), S_IRUSR | S_IWUSR | S_IXUSR);

  // it should be possible to use link()/unlink() as the target and source
  // are sub-directories of $HOME/faxin or $HOME/faxsent.  If because of a
  // very odd filesystem arrangement this won't work, the method will do nothing

  vector<string> filelist;
  struct dirent* direntry;
  struct stat statinfo;

  DIR* dir_p;
  if ((dir_p = opendir(old_dirname.c_str())) == 0) {
    string msg("Can't open directory ");
    msg += old_dirname;
    msg += '\n';
    write_error(msg.c_str());
  }

  else {
    chdir(old_dirname.c_str());
    while ((direntry = readdir(dir_p)) != 0) {
      stat(direntry->d_name, &statinfo);
      if (S_ISREG(statinfo.st_mode)) {
	filelist.push_back(direntry->d_name);
      }
    }

    closedir(dir_p);

    bool failure = false;
    vector<string>::iterator iter;
    string old_filename;
    string new_filename;

    for (iter = filelist.begin(); iter != filelist.end(); ++iter) {
      old_filename = old_dirname + '/';
      old_filename += *iter;
      new_filename = new_dirname + '/';
      new_filename += *iter;
      if (link(old_filename.c_str(), new_filename.c_str())) {
	failure = true;
	break;
      }
    }

    if (failure) { // recover position
      vector<string>::iterator recover_iter;
      for (recover_iter = filelist.begin(); recover_iter != iter; ++recover_iter) {
	new_filename = new_dirname + '/';
	new_filename += *recover_iter;
	unlink(new_filename.c_str());
      }
      rmdir(new_dirname.c_str());
      string msg("Can't move directory ");
      msg += old_dirname;
      msg += " to ";
      msg += new_dirname;
      msg += '\n';
      write_error(msg.c_str());
    }
    else {
      for (iter = filelist.begin(); iter != filelist.end(); ++iter) {
	old_filename = old_dirname + '/';
	old_filename += *iter;
	unlink(old_filename.c_str());
      }
      if (rmdir(old_dirname.c_str())) {
	string msg("Can't delete directory ");
	msg += old_dirname;
	msg += "\nThe contents should have been moved to ";
	msg += new_dirname;
	msg += "\nand it should now be empty -- please check\n";
	write_error(msg.c_str());
      }
      else get_fax_list_rows();
    }
    // reset current directory
    string temp(prog_config.homedir + "/faxin");
    chdir(temp.c_str());
  }
}

void FaxListDialog::refresh_slot(void) {

  get_fax_list_rows();
  set_buttons_slot(0, 0, 0);
}

DescriptionDialog::DescriptionDialog(const int standard_size, const string& text, Gtk::Window& window):
                             Gtk::Window(GTK_WINDOW_DIALOG), in_run_loop(false),
			     ok_button("OK"), cancel_button("Cancel"),
			     label("Fax description?"), table(3, 2, false),
                             parent(window) {

  table.attach(label, 0, 2, 0, 1, GTK_FILL | GTK_EXPAND,
	 GTK_FILL | GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(entry, 0, 2, 1, 2, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(ok_button, 0, 1, 2, 3, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(cancel_button, 1, 2, 2, 3, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  ok_button.clicked.connect(SigC::bind(SigC::slot(this, &DescriptionDialog::selected), true));
  cancel_button.clicked.connect(SigC::bind(SigC::slot(this, &DescriptionDialog::selected), false));

  add(table);
  
  entry.set_text(text);
  set_title("Fax description");
  set_transient_for(parent);
  parent.set_sensitive(false);
  set_modal(true);

  ok_button.set_usize(standard_size * 3, standard_size);
  cancel_button.set_usize(standard_size * 3, standard_size);
  entry.set_usize(standard_size * 8, standard_size);
  set_border_width(standard_size/2);

  entry.grab_focus();

  set_position(GTK_WIN_POS_CENTER);
  set_policy(false, false, false);

  show_all();
}

void DescriptionDialog::run(void) {
  in_run_loop = true;
  Gtk::Main::run();
}

void DescriptionDialog::selected(bool accept) {
  parent.set_sensitive(true); // do this before we emit accepted()
  hide_all();
  if (accept) accepted(entry.get_text());
  if (in_run_loop) Gtk::Main::quit();
  // if we have not called run(), then this dialog is self-owning and it is safe to call `delete this'
  else delete this;
}

gint DescriptionDialog::delete_event_impl(GdkEventAny*) {
  selected(false);
  return true; // returning true prevents destroy sig being emitted
}

gint DescriptionDialog::key_press_event_impl(GdkEventKey* event_p) {

  if (event_p->keyval == GDK_Escape) selected(false);
  else if (event_p->keyval == GDK_Return && !cancel_button.has_focus()) selected(true);
  else Gtk::Window::key_press_event_impl(event_p);
  return true; // processing ends here
}
  
