#------------------------------------------------------------------------------
# Copyright (c) 2005, Enthought, Inc.
# All rights reserved.
# 
# This software is provided without warranty under the terms of the BSD
# license included in enthought/LICENSE.txt and may be redistributed only
# under the conditions described in the aforementioned license.  The license
# is also available online at http://www.enthought.com/licenses/BSD.txt
# Thanks for using Enthought open source!
# 
# Author: Enthought, Inc.
# Description: <Enthought pyface package component>
#------------------------------------------------------------------------------
""" A dialog that prompts the user for confirmation. """


# Major package imports.
import wx

# Enthought library imports.
from enthought.traits.api import Bool, Enum, Instance, Str

# Local imports.
from constant import CANCEL, NO, YES
from dialog import Dialog
from image_resource import ImageResource
from image_widget import ImageWidget


def confirm(parent, message, title=None, cancel=False, default=NO):
    """ Convenience function to show a confirmation dialog. """
    
    if title is None:
        title = "Confirmation"

    dialog = ConfirmationDialog(
        parent  = parent,
        message = message,
        cancel  = cancel,
        default = default,
        title   = title
    )

    return dialog.open()


class ConfirmationDialog(Dialog):
    """ A dialog that prompts the user for confirmation. """

    DEFAULT_TO_STYLE_MAP = {
        YES : wx.YES_DEFAULT,
        NO  : wx.NO_DEFAULT
    }

    #### 'Dialog' interface ###################################################

    # Is the dialog resizeable?
    resizeable = False

    #### 'ConfirmationDialog' interface #######################################

    # Should the cancel button be displayed?
    cancel = Bool(False)
    
    # The default button.
    default = Enum(NO, YES, CANCEL)
    
    # The image displayed to the left of the message.
    image = Instance(ImageResource, ImageResource('warning.png'))

    # The message displayed in the body of the dialog (use the inherited
    # 'title' trait to set the title of the dialog itself).
    message = Str

    # Button labels.
    yes_label    = Str('Yes')
    no_label     = Str('No')
    cancel_label = Str('Cancel')
    
    ###########################################################################
    # Protected 'Dialog' interface.
    ###########################################################################

    def _create_dialog_area(self, parent):
        """ Creates the main content of the dialog. """

        panel = wx.Panel(parent, -1)
        sizer = wx.BoxSizer(wx.HORIZONTAL)
        panel.SetSizer(sizer)
        panel.SetAutoLayout(True)

        # The image.
        image = ImageWidget(panel, bitmap=self.image.create_bitmap())
        sizer.Add(image.control, 0, wx.EXPAND)

        # The message.
        message = wx.StaticText(panel, -1, self.message)
        sizer.Add(message, 1, wx.EXPAND | wx.TOP, 15)

        # Resize the panel to match the sizer.
        sizer.Fit(panel)
        
        return panel

    def _create_buttons(self, parent):
        """ Creates the buttons. """

        sizer = wx.BoxSizer(wx.HORIZONTAL)

        # 'YES' button.
        self._yes = yes = wx.Button(parent, wx.ID_YES, self.yes_label)
        if self.default == YES:
            yes.SetDefault()
        wx.EVT_BUTTON(parent, wx.ID_YES, self._on_yes)
        sizer.Add(yes)

        # 'NO' button.
        self._no = no = wx.Button(parent, wx.ID_NO, self.no_label)
        if self.default == NO:
            no.SetDefault()
        wx.EVT_BUTTON(parent, wx.ID_NO, self._on_no)
        sizer.Add(no, 0, wx.LEFT, 10)

        if self.cancel:
            # 'Cancel' button.
            self._cancel = cancel = wx.Button(
                parent, wx.ID_CANCEL, self.cancel_label
            )
            if self.default == CANCEL:
                cancel.SetDefault()
                
            wx.EVT_BUTTON(parent, wx.ID_CANCEL, self._on_cancel)
            sizer.Add(cancel, 0, wx.LEFT, 10)

        return sizer

    ###########################################################################
    # 'Private' interface.
    ###########################################################################

    #### wx event handlers ####################################################

    def _on_yes(self, event):
        """ Called when the 'Yes' button is pressed. """

        self.control.EndModal(wx.ID_YES)
        
        return

    def _on_no(self, event):
        """ Called when the 'No' button is pressed. """

        self.control.EndModal(wx.ID_NO)

        return

#### EOF ######################################################################
