#------------------------------------------------------------------------------
# Copyright (c) 2005, Enthought, Inc.
# All rights reserved.
# 
# This software is provided without warranty under the terms of the BSD
# license included in enthought/LICENSE.txt and may be redistributed only
# under the conditions described in the aforementioned license.  The license
# is also available online at http://www.enthought.com/licenses/BSD.txt
# Thanks for using Enthought open source!
# 
# Author: David C. Morrill
# Date: 09/19/2003
# Description: Simple program for interactively testing the various standard
#  traits.
# Bugs: 
#  - Setting a 'simple' ImageEnum value does not always update the image 
#    displayed.  
#  - The 'readonly' editor for an ImageEnum appears to be an active image 
#    button.  
#------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
#  Imports:
#-------------------------------------------------------------------------------

from string   import capwords
from wxPython import wx
app = wx.wxApp()

import enthought.chaco.plot_container as plot_traits

import os.path
import cPickle

from enthought.traits.api                 import *
from enthought.traits.trait_handlers  import TraitListObject
from enthought.traits.wxmenu          import MakeMenu

import enthought.traits.standard      as standard_traits
import enthought.traits.wxtrait_sheet as trait_sheet

#=============================================================================== 
#  Constants:
#=============================================================================== 

DEBUG = True

# Application title:
app_title = 'Standard Traits Explorer'

# Screen size (initialized in 'OnInit' method of 'TraitExplorerApp'):
screen_dx    = wx.wxSystemSettings_GetMetric( wx.wxSYS_SCREEN_X ) or 1024
screen_dy    = wx.wxSystemSettings_GetMetric( wx.wxSYS_SCREEN_Y ) or 768
scrollbar_dx = wx.wxSystemSettings_GetMetric( wx.wxSYS_VSCROLL_X )

#=============================================================================== 
#  Main Menu Description:
#=============================================================================== 

trait_explorer_menu = """
    &File
        Exit | Ctrl-Q: self.on_exit()
"""

#-------------------------------------------------------------------------------
#  'TraitExplorer' class:
#-------------------------------------------------------------------------------

class TraitExplorer ( wx.wxFrame ):

    #---------------------------------------------------------------------------
    #  Initialize object: 
    #---------------------------------------------------------------------------
 
    def __init__ ( self ):
        wx.wxFrame.__init__( self, None, -1, app_title, pos = (-1,-1),
                             size = (-1,-1) )
 
        # Get the saved state dictionary (if any):
        self.dic = dic = self.get_state()
 
        # Create a status bar for displaying messages:
        self.CreateStatusBar( 4 )
        self.SetStatusWidths( [ 40, -1, 50, -1 ] )
        self.SetStatusText( 'Value:',  0 )
        self.SetStatusText( 'Value_:', 2 )
 
        # Create the menu bar:
        self.menu = MakeMenu( trait_explorer_menu, self )
        
        # Get the set of standard traits:
        self.traits        = {}
        self.trait_objects = {}
        self.add_traits( standard_traits )
        self.add_traits( plot_traits )
        
        # Create the window controls:
        panel       = wx.wxPanel( self, -1 )
        sizervl     = wx.wxBoxSizer( wx.wxVERTICAL )
        sizervr     = wx.wxBoxSizer( wx.wxVERTICAL )
        sizerh      = wx.wxBoxSizer( wx.wxHORIZONTAL )
        trait_names = self.traits.keys()
        trait_names.sort()
        self.trait_list = tlist = wx.wxListBox( panel, -1, 
                                                choices = trait_names )
        self.mode_list  = mlist = wx.wxListBox( panel, -1, choices = [ 
                                      'Custom', 'Simple', 'Text', 'ReadOnly' ] )
        tlist.SetSelection( 0 )
        mlist.SetSelection( 0 )
        wx.EVT_CLOSE(  self, self.on_close )
        wx.EVT_LISTBOX( panel, tlist.GetId(), self.on_trait_selected )
        wx.EVT_LISTBOX( panel, mlist.GetId(), self.on_trait_selected )
        sizervl.Add( wx.wxStaticText( panel, -1, 'Trait', 
                                   style = wx.wxALIGN_CENTRE ), 0, wx.wxEXPAND )
        sizervl.Add( tlist,  4, wx.wxEXPAND | wx.wxALL, 3 )
        sizervl.Add( wx.wxStaticText( panel, -1, 'Style', 
                                   style = wx.wxALIGN_CENTRE ), 0, wx.wxEXPAND )
        sizervl.Add( mlist,  1, wx.wxEXPAND | wx.wxALL, 3 )
        sizerh.Add( sizervl, 0, wx.wxEXPAND )
        self.trait_panel = tpanel = wx.wxScrolledWindow( panel, -1 )
        sizervr.Add( wx.wxStaticText( panel, -1, 'Current Trait Editor', 
                                   style = wx.wxALIGN_CENTRE ), 0, wx.wxEXPAND )
        sizervr.Add( wx.wxStaticLine( panel, -1 ), 0, 
                     wx.wxTOP | wx.wxBOTTOM | wx.wxEXPAND, 4 )                                   
        sizervr.Add( tpanel, 5, wx.wxEXPAND )
        sizerh.Add( sizervr, 5, wx.wxEXPAND | wx.wxLEFT | wx.wxRIGHT, 8 )
        
        # Finish setting up the window:
        panel.SetAutoLayout( True )
        panel.SetSizer( sizerh )
 
        # Restore user preference items:
        self.restore_state()
        
        # Create a starting trait editor:
        self.on_trait_selected()
       
    #---------------------------------------------------------------------------
    #  Add all of the traits in the specified module to the traits dictionary: 
    #---------------------------------------------------------------------------
    
    def add_traits ( self, module ):
        traits = self.traits
        for name in dir( module ):
            value = getattr( module, name )
            if isinstance( value, CTrait ):
                traits[ capwords( name.replace( '_', ' ' ) ) ] = value  
       
    #---------------------------------------------------------------------------
    #  Handle the user selecting a new trait:
    #---------------------------------------------------------------------------
 
    def on_trait_selected ( self, event = None ):
        mode        = self.mode_list.GetStringSelection().lower()
        trait_name  = self.trait_list.GetStringSelection()
        test_object = self.trait_objects.get( trait_name )
        if test_object is None:
            test_object = HasDynamicTraits()
            self.trait_objects[ trait_name ] = test_object
            test_object.add_trait( 'value', self.traits[ trait_name ] )
            test_object.on_trait_change( self.trait_changed, 'value' )
            if isinstance( test_object.value, TraitListObject ):
                test_object.on_trait_change( self.trait_changed, 'value_items' )
        self.test_object = test_object
        tpanel = self.trait_panel
        tpanel.SetSizer( None )
        tpanel.DestroyChildren()
        sizer = wx.wxBoxSizer( wx.wxVERTICAL )
        sheet = trait_sheet.TraitSheet( tpanel, test_object,
                                        TraitGroup( 'value', style = mode ) )
        sizer.Add( sheet )
        tpanel.SetAutoLayout( True )
        tpanel.SetSizer( sizer )
        tpanel.SetScrollRate( 0, 16 )
        width, height = sheet.GetSize()
        tpanel.SetSize( wx.wxSize( width + scrollbar_dx, height ) )
        tpanel.GetParent().Layout()
        self.trait_changed()
       
    #---------------------------------------------------------------------------
    #  Handle the user changing the trait value:
    #---------------------------------------------------------------------------
    
    def trait_changed ( self ):
        self.SetStatusText( str( self.test_object.value ), 1 )
        try:
            value = str( self.test_object.value_ )
        except:
            value = '<undefined>'
        self.SetStatusText( value, 3 )
        
    #---------------------------------------------------------------------------
    #  Handle application start-up:
    #---------------------------------------------------------------------------
 
    def start ( self ):
        self.Show()
           
    #---------------------------------------------------------------------------
    #  Handle the user exiting the plot test window:
    #---------------------------------------------------------------------------
 
    def on_exit ( self ):
        self.on_close()
 
    #---------------------------------------------------------------------------
    #  Handle the user closing the plot test window:
    #---------------------------------------------------------------------------
 
    def on_close ( self, event = None ):
        self.save_state()
        self.Destroy()
        
    #---------------------------------------------------------------------------
    #  Open the file used to save the user preference 'pickles' in:
    #---------------------------------------------------------------------------
 
    def open_state ( self, mode ):
        try:
            path = os.environ[ 'HOME' ]
        except:
            path = ''
        return open( os.path.join( path, '.trait_explorer.save' ), mode )
 
    #---------------------------------------------------------------------------
    #  Pickle all of the user preference items:
    #---------------------------------------------------------------------------
 
    def save_state ( self ):
        
        # Try to save the session state:
        fd = None
        try:
            fd  = self.open_state( 'w' )
            p   = cPickle.Pickler( fd )
            dic = self.dic
          
            # Save the window size and position:
            dic[ 'window' ] = self.GetPositionTuple() + self.GetSizeTuple()
          
            # Save the dictionary:
            p.dump( dic )
        except:
            pass
 
        # Close the file (if necessary):
        if fd:
            fd.close()

    #---------------------------------------------------------------------------
    #  Return the dictionary of user preference items:
    #---------------------------------------------------------------------------
 
    def get_state ( self ):
 
        # Assume that the restore will fail:
        dic = {}
        fd  = None
 
        # Try to restore the previous session state:
        try:
            fd = self.open_state( 'r' )
         
            # Restore the dictionary of saved values:
            dic = cPickle.Unpickler( fd ).load()
        except:
           pass
 
        # Close the file (if necessary):
        if fd:
            fd.close()
 
        # Return the saved state dictionary:
        return dic
 
    #---------------------------------------------------------------------------
    #  Restore the user preference items:
    #---------------------------------------------------------------------------
 
    def restore_state ( self ):
        dic_get = self.dic.get
        
        # Restore the window size and position:
        x, y, width, height = dic_get( 'window',
                                       ( screen_dx / 3, screen_dy / 3, 
                                         screen_dx / 3, screen_dy / 3 ) )
        self.SetDimensions( x, y, width, height )

#-------------------------------------------------------------------------------
#  Program Start-up:
#-------------------------------------------------------------------------------

wx.wxInitAllImageHandlers()
if __name__ == '__main__':
    trait_explorer = TraitExplorer()
    app.SetTopWindow( trait_explorer )
    trait_explorer.start()
    app.MainLoop()
   
