#include <gtk/gtk.h>
#include "entity.h"
#include "gtk-common.h"
#include <string.h>

typedef struct {
    gdouble x;
    gdouble y;
    guint button;
    guint32 time;
    guint id;
} button_press_timeout_t;

/* DOUBLECLICK: second click must occur within 1/4 s * * order for double
 * click events * 1. GDK_BUTTON_PRESS * 2. GDK_BUTTON_RELEASE * 3.
 * GDK_BUTTON_PRESS * 4. GDK_2BUTTON_PRESS * 5. GDK_BUTTON_RELEASE * * *
 * TRIPLECLICK: second click must occur within 1/2 s * * order for triple
 * click events * 1. GDK_BUTTON_PRESS * 2. GDK_BUTTON_RELEASE * 3.
 * GDK_BUTTON_PRESS * 4. GDK_2BUTTON_PRESS * 5. GDK_BUTTON_RELEASE * 6.
 * GDK_BUTTON_PRESS * 7. GDK_3BUTTON_PRESS * 8. GDK_BUTTON_RELEASE */

static gint
button_press_timeout_callback (gpointer callbackData)
{
    ENode *node = (ENode *) callbackData;
    button_press_timeout_t *data = enode_get_kv (node, "buttonpress-timeout");
    gchar *function = enode_attrib_str (node, "onbuttonpress", NULL);

    enode_call_ignore_return (node, function, "idd", data->button, data->x,
			      data->y);
    return FALSE;		/* don't call again timeout callback */
}

static gint
doubleclick_timeout_callback (gpointer callbackData)
{
    ENode *node = (ENode *) callbackData;
    button_press_timeout_t *data = enode_get_kv (node, "buttonpress-timeout");
    gchar *function = enode_attrib_str (node, "ondoubleclick", NULL);

    enode_call_ignore_return (node, function, "idd", data->button, data->x,
			      data->y);
    return FALSE;		/* don't call again timeout callback */
}

static gint
button_press_event_callback (GtkWidget * widget,
			     GdkEventButton * event, gpointer user_data)
{
    ENode *node = user_data;
    EBuf *multiclick;

    if (GDK_BUTTON_PRESS != event->type) {
	return (TRUE);
    }

    multiclick = enode_attrib_quiet (node, "ondoubleclick", NULL);

    if (!multiclick) {
	multiclick = enode_attrib_quiet (node, "ontripleclick", NULL);
    }

    if (multiclick && multiclick->str && strlen (multiclick->str)) {
	/* we've also a multiclick function */

	button_press_timeout_t *data =
	    enode_get_kv (node, "buttonpress-timeout");

	if (!data) {
	    data = g_new0 (button_press_timeout_t, 1);
	    data->time = event->time;
	    enode_set_kv (node, "buttonpress-timeout", data);
	} else {
	    if (event->time - data->time > 250) {
		data->time = event->time;
	    }
	    gtk_timeout_remove (data->id);
	}

	data->x = event->x;
	data->y = event->y;
	data->button = event->button;
	data->id = gtk_timeout_add (250, button_press_timeout_callback, node);

    } else {
	/* no doubleclick event, so call  the function immediately */
	/* Arguments will be: node path, button number, x pos, y pos. Other
	 * possibles include xtilt, ytilt, and pressure. */
	gchar *function = enode_attrib_str (node, "onbuttonpress", NULL);
	enode_call_ignore_return (node, function, "idd", event->button,
				  event->x, event->y);
    }

    /* don't propogate events to parent windows */
    return (TRUE);
}

static gint
doubleclick_event_callback (GtkWidget * widget,
			    GdkEventButton * event, gpointer user_data)
{
    ENode *node = user_data;
    button_press_timeout_t *data;
    EBuf *tripleclick;

    if (GDK_2BUTTON_PRESS != event->type) {
	return (TRUE);
    }

    data = enode_get_kv (node, "buttonpress-timeout");
    gtk_timeout_remove (data->id);

    tripleclick = enode_attrib_quiet (node, "ontripleclick", NULL);
    if (ebuf_not_empty (tripleclick)) {

	guint32 timediff = event->time - data->time;
	data->x = event->x;
	data->y = event->y;
	data->button = event->button;
	data->id =
	    gtk_timeout_add (timediff < 500 ? 500 - timediff : 500,
			     doubleclick_timeout_callback, node);

    } else {
	/* no tripleclick event, so call  the function immediately */
	/* Arguments will be: node path, button number, x pos, y pos. Other
	 * possibles include xtilt, ytilt, and pressure. */
	gchar *function = enode_attrib_str (node, "ondoubleclick", NULL);
	enode_call_ignore_return (node, function, "idd", event->button,
				  event->x, event->y);
    }

    /* don't propogate events to parent windows */
    return (TRUE);
}

static gint
tripleclick_event_callback (GtkWidget * widget,
			    GdkEventButton * event, gpointer user_data)
{
    ENode *node = user_data;
    gchar *function;
    button_press_timeout_t *data;

    if (GDK_3BUTTON_PRESS != event->type) {
	return (TRUE);
    }

    data = enode_get_kv (node, "buttonpress-timeout");
    gtk_timeout_remove (data->id);

    function = enode_attrib_str (node, "ontripleclick", NULL);

    /* Arguments will be: node path, button number, x pos, y pos. Other
     * possibles include xtilt, ytilt, and pressure. */
    enode_call_ignore_return (node, function, "idd", event->button, event->x,
			      event->y);

    /* don't propogate events to parent windows */
    return (TRUE);
}


static void
button_release_event_callback (GtkWidget * widget,
			       GdkEventButton * event, gpointer user_data)
{
    ENode *node = user_data;
    gchar *function = NULL;

    function = enode_attrib_str (node, "onbuttonrelease", NULL);

    /* Arguments will be: node path, button number, x pos, y pos. Other
     * possibles include xtilt, ytilt, and pressure. */
    /* g_print ("executing perl function %s\n", function); */

    enode_call_ignore_return (node, function, "idd",
			      event->button, event->x, event->y);
}

static void
mousemotion_event_callback (GtkWidget * widget,
			    GdkEventMotion * event, gpointer user_data)
{
    ENode *node = user_data;
    gchar *function = NULL;

    function = enode_attrib_str (node, "onmousemotion", NULL);

    /* Arguments will be: node path, x pos, y pos. */
    /* note: event->button is not valid for motion events */
    /* note: event->state will contain the sum of: *        256 (button 1
     * pressed) *        512 (button 2 pressed) *       1024 (button 3
     * pressed) */

    enode_call_ignore_return (node, function, "idd", event->state >> 8,
			      event->x, event->y);
}

static gint
keypress_event_callback (GtkWidget * widget,
			 GdkEventKey * event, gpointer user_data)
{
    ENode *node = user_data;
    gchar *function = NULL;
    EBuf *ret;

    function = enode_attrib_str (node, "onkeypress", NULL);

    /* Arguments will be: node path, key string, key number */
    ret = enode_call (node, function, "si", gdk_keyval_name (event->keyval),
		      event->keyval);


    if (ret && ebuf_equal_strcase (ret, "stop")) {
	EDEBUG (("gtk-widget-attr", "ret = %s", ret->str));
	gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), "key_press_event");
	return TRUE;
    }
    return FALSE;
}

static gint
keyrelease_event_callback (GtkWidget * widget,
			   GdkEventKey * event, gpointer user_data)
{
    ENode *node = user_data;
    gchar *function = NULL;
    EBuf *ret;

    function = enode_attrib_str (node, "onkeyrelease", NULL);

    /* Arguments will be: node path, key number, key string */
    ret = enode_call (node, function, "si", gdk_keyval_name (event->keyval),
		      event->keyval);

    if (ret && ebuf_equal_strcase (ret, "stop")) {
	EDEBUG (("gtk-widget-attr", "ret = %s", ret->str));
	gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), "key_release_event");
	return TRUE;
    }
    return FALSE;
}

static void
focused_event_callback (GtkWidget * widget, gpointer user_data)
{
    ENode *node = user_data;
    gchar *function = NULL;

    if (!GTK_WIDGET_HAS_FOCUS (widget))
	return;

    function = enode_attrib_str (node, "onfocus", NULL);

    enode_call_ignore_return (node, function, "");


}


/*****************************************************************
 *    container widget section 					 *
 *****************************************************************/

static gint
rendgtk_widget_container_border_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *widget = enode_get_kv (node, "top-widget");

    if (widget)
	gtk_container_set_border_width (GTK_CONTAINER (widget),
					erend_get_integer (value));

    return (TRUE);
}

static void
rendgtk_type_container_attr_register (Element * element)
{
    ElementAttr *e_attr;

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "border";
    e_attr->description = "Set size of border.";
    e_attr->value_desc = "integer";
    e_attr->possible_values = "0,*";
    e_attr->set_attr_func = rendgtk_widget_container_border_set;
    element_register_attrib (element, e_attr);
}

/*****************************************************************
 *    toggle widget section 					 *
 *****************************************************************/

static gint
rendgtk_widget_toggle_button_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *toggle;

    toggle = enode_get_kv (node, "top-widget");

    EDEBUG (
	    ("gtk-widget-attr", "toggle button value being set to %s\n",
	     value));

    if (toggle)
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (toggle),
				      erend_value_is_true (value));

    return (TRUE);
}

static void
rendgtk_type_toggle_button_attr_register (Element * element)
{
    ElementAttr *e_attr;
    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "value";
    e_attr->description = "Set if the button is toggled to true.";
    e_attr->value_desc = "boolean";
    e_attr->possible_values = "false,true";
    e_attr->set_attr_func = rendgtk_widget_toggle_button_attr_set;
    element_register_attrib (element, e_attr);
}

/*****************************************************************
 *    editable widgets section   				 *
 *****************************************************************/

static int
rendgtk_widget_editable_edit_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *editable;

    editable = enode_get_kv (node, "top-widget");

    if (editable)
	gtk_editable_set_editable (GTK_EDITABLE (editable),
				   erend_value_is_true (value));
    return (TRUE);
}

static void
rendgtk_type_editable_attr_register (Element * element)
{
    ElementAttr *e_attr;

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "editable";
    e_attr->description = "Set if the user can type into the widget.";
    e_attr->value_desc = "boolean";
    /* e_attr->possible_values = ""; */
    e_attr->set_attr_func = rendgtk_widget_editable_edit_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onchange";
    e_attr->description = "Sets the function to call when text is changed.";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(calling_node)";
    e_attr->set_attr_func = NULL;
    element_register_attrib (element, e_attr);
}

/*****************************************************************
 *    box widget section    					 *
 *****************************************************************/

static gint
rendgtk_widget_box_homogenous_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *widget = enode_get_kv (node, "top-widget");

    if (widget)
	gtk_box_set_homogeneous (GTK_BOX (widget), erend_value_is_true (value));

    return (TRUE);
}

static gint
rendgtk_widget_box_spacing_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *widget = enode_get_kv (node, "top-widget");

    if (widget)
	gtk_box_set_spacing (GTK_BOX (widget), erend_get_integer (value));

    return (TRUE);
}

static void
rendgtk_type_box_attr_register (Element * element)
{
    ElementAttr *e_attr;

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "homogenous";
    e_attr->description = "Set all contained widgets to be the same size.";
    e_attr->value_desc = "boolean";
    e_attr->possible_values = "false,true";
    e_attr->set_attr_func = rendgtk_widget_box_homogenous_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "spacing";
    e_attr->description = "Set spacing between child nodes.";
    e_attr->value_desc = "integer";
    e_attr->possible_values = "0,*";
    e_attr->set_attr_func = rendgtk_widget_box_spacing_set;
    element_register_attrib (element, e_attr);
}

/*****************************************************************
 *     widget section    					 *
 *****************************************************************/

static gint
rendgtk_widget_widget_focused_attr_set_idle (gpointer user_data)
{
    GtkWidget *widget;
    ENode *node = user_data;

    widget = enode_get_kv (node, "top-widget");
    if (!widget)
	return (FALSE);

    EDEBUG (("gtk-widget-attr", "going to draw the focus in idle callback"));

    GTK_WIDGET_SET_FLAGS (widget, GTK_CAN_FOCUS);
    gtk_widget_grab_focus (widget);

    return (FALSE);
}

static gint
rendgtk_widget_default_widget_attr_set_idle (gpointer user_data)
{
    ENode *node = user_data;
    GtkWidget *widget;
    EBuf *value;

    widget = enode_get_kv (node, "top-widget");
    if (!widget)
	return (FALSE);

    EDEBUG (
	    ("gtk-widget-attr",
	     "going to set widget as default in idle callback"));

    value = enode_attrib (node, "default-widget", NULL);
    if (ebuf_not_empty (value) && erend_value_is_true (value)) {
	GTK_WIDGET_SET_FLAGS (widget, GTK_CAN_DEFAULT);
	gtk_widget_grab_default (widget);
    }
    return (FALSE);
}


static void
rendgtk_widget_widget_focused_attr_get (ENode * node, gchar * attr)
{
    GtkWidget *widget;
    EBuf *focus;

    widget = enode_get_kv (node, "top-widget");
    if (!widget)
	return;

    if (GTK_WIDGET_HAS_FOCUS (widget))
	focus = ebuf_new_with_true ();
    else
	focus = ebuf_new_with_false ();

    enode_attrib_quiet (node, "focus", focus);
}

static gint
rendgtk_widget_widget_focused_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *widget;

    widget = enode_get_kv (node, "top-widget");
    if (!widget)
	return (TRUE);

    gtk_idle_add (rendgtk_widget_widget_focused_attr_set_idle, node);

    return (TRUE);
}

static gint
rendgtk_widget_default_widget_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *widget;

    widget = enode_get_kv (node, "top-widget");
    if (!widget)
	return (TRUE);

    gtk_idle_add (rendgtk_widget_default_widget_attr_set_idle, node);

    return (TRUE);
}


static gint
rendgtk_widget_widget_tooltip_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkTooltips *tips = NULL;
    GtkWidget *widget;

    if (!node->parent)
	return (TRUE);

    widget = enode_get_kv (node, "top-widget");
    if (!widget)
	return (TRUE);

    if (!tips) {
	tips = gtk_tooltips_new ();
    }

    if (value->len > 0)
	gtk_tooltips_set_tip (tips, widget, value->str, NULL);
    else
	gtk_tooltips_set_tip (tips, widget, NULL, NULL);


    return (TRUE);
}

static gint
rendgtk_widget_widget_usize_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *widget;
    gfloat xival;
    gfloat yival;
    EBuf *xval;
    EBuf *yval;

    widget = enode_get_kv (node, "top-widget");
    if (!widget)
	return (TRUE);

    yval = enode_attrib (node, "height", NULL);
    if (ebuf_not_empty (yval))
	yival = erend_get_integer (yval);
    else
	yival = -1;

    xval = enode_attrib (node, "width", NULL);
    if (ebuf_not_empty (xval))
	xival = erend_get_integer (xval);
    else
	xival = -1;

    gtk_widget_set_usize (GTK_WIDGET (widget), xival, yival);
    return (TRUE);
}

static gint
rendgtk_widget_widget_sensitive_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *widget = enode_get_kv (node, "top-widget");

    if (widget) {
	gint sensitive;
	if (ebuf_empty (value)) {
	    sensitive = TRUE;
	} else {
	    sensitive = erend_value_is_true (value);
	}
	
	gtk_widget_set_sensitive (GTK_WIDGET (widget), sensitive);
    }

    return (TRUE);
}

static gint
rendgtk_widget_widget_visible_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *widget = enode_get_kv (node, "top-widget");

    if (widget) {
	if (erend_value_is_true (value))
	    gtk_widget_show (widget);
	else
	    gtk_widget_hide (widget);
    }

    return (TRUE);
}

static gint
rendgtk_widget_widget_dragdata_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *widget = enode_get_kv (node, "top-widget");

    if (widget)
	rendgtk_dnd_dragable_set (node, widget);

    return (TRUE);
}

static gint
rendgtk_widget_widget_ondrop_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *widget;

    widget = enode_get_kv (node, "top-widget");

    if (widget)
	rendgtk_dnd_target_create (node, widget);

    return (TRUE);
}

static gint
rendgtk_widget_widget_font_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *widget;
    GtkStyle *style;
    GdkFont *font;

    widget = enode_get_kv (node, "top-widget");
    if (!widget)
	return (TRUE);

    font = gdk_font_load (value->str);

    if (font) {
	gtk_widget_ensure_style (widget);
	style = gtk_style_copy (widget->style);
	style->font = font;
	gtk_widget_set_style (widget, style);
    }
    return (TRUE);
}

static void
rendgtk_widget_widget_height_get (ENode *node, gchar *value)
{
    GtkWidget *widget;
    EBuf *height;

    widget = enode_get_kv (node, "top-widget");
    if (!widget)
	return;

    height = ebuf_new_with_int (widget->allocation.height);

    enode_attrib_quiet (node, "_height", height);
}

static void
rendgtk_widget_widget_width_get (ENode *node, gchar *value)
{
    GtkWidget *widget;
    EBuf *width;

    widget = enode_get_kv (node, "top-widget");
    if (!widget)
	return;

    width = ebuf_new_with_int (widget->allocation.width);

    enode_attrib_quiet (node, "_width", width);
}

#if 0
static gushort *
rendgtk_multiclick_get (ENode * node)
{
    gushort *value = enode_get_kv (node, "multiple-click-actions");
    if (!value) {
	value = g_new0 (gushort, 1);
	g_message ("creating multiple-click-actions entry for node %p", node);
    }
    return value;
}

static gushort
rendgtk_multiclick_add (ENode * node, gushort flag)
{
    gushort *value = rendgtk_multiclick_get (node);
    *value |= flag;
    enode_set_kv (node, "multiclick", (gpointer) value);
    g_message ("(rendgtk_multiclick_get) multiclick entry for node %p --> %d",
	       node, *value);
    return *value;
}

static gushort
rendgtk_multiclick_remove (ENode * node, gushort flag)
{
    gushort *value = rendgtk_multiclick_get (node);
    *value &= ~flag;
    enode_set_kv (node, "multiclick", (gpointer) value);
    g_message ("(rendgtk_multiclick_get) multiclick entry for node %p --> %d",
	       node, *value);
    return *value;
}
#endif


static gint
rendgtk_widget_widget_event_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *widget;

    widget = enode_get_kv (node, "top-widget");
    if (!widget)
	return (TRUE);

    if (!GTK_WIDGET_NO_WINDOW (widget)) {
	gtk_widget_set_events (widget, gtk_widget_get_events (widget) |
			       GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
			       GDK_BUTTON_MOTION_MASK | GDK_POINTER_MOTION_MASK
			       | GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
			       GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK);
    }

    if (ebuf_equal_str (attr, "onbuttonpress")) {
	if (!strlen (value->str)) {
	    gtk_signal_disconnect_by_func (GTK_OBJECT (widget),
					   GTK_SIGNAL_FUNC
					   (button_press_event_callback), node);
	} else {
	    gtk_signal_connect (GTK_OBJECT (widget), "button_press_event",
				GTK_SIGNAL_FUNC (button_press_event_callback),
				node);
	}
	return (TRUE);
    }

    else if (ebuf_equal_str (attr, "ondoubleclick")) {
	if (!strlen (value->str)) {
	    gtk_signal_disconnect_by_func (GTK_OBJECT (widget),
					   GTK_SIGNAL_FUNC
					   (doubleclick_event_callback), node);
	} else {
	    gtk_signal_connect (GTK_OBJECT (widget), "button_press_event",
				GTK_SIGNAL_FUNC (doubleclick_event_callback),
				node);
	}
	return (TRUE);
    }

    else if (ebuf_equal_str (attr, "ontripleclick")) {
	if (!strlen (value->str)) {
	    gtk_signal_disconnect_by_func (GTK_OBJECT (widget),
					   GTK_SIGNAL_FUNC
					   (tripleclick_event_callback), node);
	} else {
	    gtk_signal_connect (GTK_OBJECT (widget), "button_press_event",
				GTK_SIGNAL_FUNC (tripleclick_event_callback),
				node);
	}
	return (TRUE);
    }

    else if (ebuf_equal_str (attr, "onbuttonrelease")) {
	if (!strlen (value->str)) {
	    gtk_signal_disconnect_by_func (GTK_OBJECT (widget),
					   button_release_event_callback, node);
	} else {
	    gtk_signal_connect (GTK_OBJECT (widget), "button_release_event",
				button_release_event_callback, node);
	}
	return (TRUE);
    }

    else if (ebuf_equal_str (attr, "onmousemotion")) {
	if (!strlen (value->str)) {
	    gtk_signal_disconnect_by_func (GTK_OBJECT (widget),
					   mousemotion_event_callback, node);
	} else {
	    gtk_signal_connect (GTK_OBJECT (widget), "motion_notify_event",
				mousemotion_event_callback, node);
	}
	return (TRUE);
    }

    else if (ebuf_equal_str (attr, "onkeypress")) {
	if (!value->len) {
	    gtk_signal_disconnect_by_func (GTK_OBJECT (widget),
					   GTK_SIGNAL_FUNC
					   (keypress_event_callback), node);
	} else {
	    gtk_signal_connect (GTK_OBJECT (widget), "key-press-event",
				GTK_SIGNAL_FUNC (keypress_event_callback),
				node);
	}
	return (TRUE);
    } else if (ebuf_equal_str (attr, "onkeyrelease")) {
	if (!value->len) {
	    gtk_signal_disconnect_by_func (GTK_OBJECT (widget),
					   GTK_SIGNAL_FUNC
					   (keyrelease_event_callback), node);
	} else {
	    gtk_signal_connect (GTK_OBJECT (widget), "key-release-event",
				GTK_SIGNAL_FUNC (keyrelease_event_callback),
				node);
	}
	return (TRUE);
    } else if (ebuf_equal_str (attr, "onfocus")) {
	if (!strlen (value->str)) {
	    gtk_signal_disconnect_by_func (GTK_OBJECT (widget),
					   focused_event_callback, node);
	} else {
	    gtk_signal_connect (GTK_OBJECT (widget), "draw-focus",
				focused_event_callback, node);
	}
	return (TRUE);
    }


    /* should not be reached */
    return (FALSE);
}

static void
rendgtk_type_widget_attr_register (Element * element)
{
    ElementAttr *e_attr;

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "width";
    e_attr->description = "Set total width of widget in pixels."
	" -1 specifies natural width.";
    e_attr->value_desc = "integer";
    e_attr->possible_values = "-1,*";
    e_attr->set_attr_func = rendgtk_widget_widget_usize_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "_width";
    e_attr->description = "Get total width of widget in pixels.";
    e_attr->value_desc = "integer";
    e_attr->get_attr_func = rendgtk_widget_widget_width_get;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "height";
    e_attr->description = "Set total height of widget in pixels."
	" -1 specifies natural height.";
    e_attr->value_desc = "integer";
    e_attr->possible_values = "-1,*";
    e_attr->set_attr_func = rendgtk_widget_widget_usize_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "_height";
    e_attr->description = "Get total height of widget in pixels.";
    e_attr->value_desc = "integer";
    e_attr->get_attr_func = rendgtk_widget_widget_height_get;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "sensitive";
    e_attr->description = "Toggle widget sensitivity."
	"  Will propogate through widget tree.";
    e_attr->value_desc = "boolean";
    e_attr->possible_values = "true,false";
    e_attr->set_attr_func = rendgtk_widget_widget_sensitive_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "visible";
    e_attr->description =
	"Toggle if the widget is to be visible on the screen.";
    e_attr->value_desc = "boolean";
    e_attr->possible_values = "true,false";
    e_attr->set_attr_func = rendgtk_widget_widget_visible_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "dragdata-text";
    e_attr->description = "Specify the text data to be given as drag data.";
    e_attr->value_desc = "string";
    e_attr->set_attr_func = rendgtk_widget_widget_dragdata_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "dragdata-url";
    e_attr->description = "Specify the URL to be given as drag data.";
    e_attr->value_desc = "string";
    e_attr->set_attr_func = rendgtk_widget_widget_dragdata_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "ondrop";
    e_attr->description = "Makes widget accept drag events, and"
	" calls specified function on a drop.";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(target_node, drag_data, suggested_action)";
    e_attr->set_attr_func = rendgtk_widget_widget_ondrop_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "ondrag";
    e_attr->description = "Makes widget a drag source, and calls"
	" specified function when drag starts";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(source_node)";
    e_attr->set_attr_func = NULL;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "font";
    e_attr->description = "Specify font used in widget.";
    e_attr->value_desc = "font";
    e_attr->set_attr_func = rendgtk_widget_widget_font_set;
    element_register_attrib (element, e_attr);

    /* Various signals to connect to arbitrary widgets * (only with windows
     * though really..  may want to * check that in the future */
    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onbuttonpress";
    e_attr->description = "Specify function called on a button press event.";
    e_attr->value_desc = "function";
    e_attr->possible_values =
	"(node, button_number, x_coordinate, y_coordinate)";
    e_attr->set_attr_func = rendgtk_widget_widget_event_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "ondoubleclick";
    e_attr->description = "Specify function called on a doubleclick event.";
    e_attr->value_desc = "function";
    e_attr->possible_values =
	"(node, button_number, x_coordinate, y_coordinate)";
    e_attr->set_attr_func = rendgtk_widget_widget_event_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "ontripleclick";
    e_attr->description = "Specify function called on a tripleclick event.";
    e_attr->value_desc = "function";
    e_attr->possible_values =
	"(node, button_number, x_coordinate, y_coordinate)";
    e_attr->set_attr_func = rendgtk_widget_widget_event_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onbuttonrelease";
    e_attr->description = "Specify function called on a button release event.";
    e_attr->value_desc = "function";
    e_attr->possible_values =
	"(node, button_number, x_coordinate, y_coordinate)";
    e_attr->set_attr_func = rendgtk_widget_widget_event_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onmousemotion";
    e_attr->description = "Specify function called when the mouse"
	" is moved on this widget.";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(node, button_mask, x_coordinate, y_coordinate)";
    e_attr->set_attr_func = rendgtk_widget_widget_event_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onkeypress";
    e_attr->description = "Specify function called on a keypress event.";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(event_node, key_name, key_value)";
    e_attr->set_attr_func = rendgtk_widget_widget_event_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onkeyrelease";
    e_attr->description = "Specify function called on a keyrelease event.";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(event_node, key_name, key_value)";
    e_attr->set_attr_func = rendgtk_widget_widget_event_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "tooltip";
    e_attr->description = "Specify tooltip for widget.";
    e_attr->value_desc = "string";
    e_attr->set_attr_func = rendgtk_widget_widget_tooltip_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "focused";
    e_attr->description = "Whether this widget has focus or not.";
    e_attr->value_desc = "boolean";
    e_attr->set_attr_func = rendgtk_widget_widget_focused_attr_set;
    e_attr->get_attr_func = rendgtk_widget_widget_focused_attr_get;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "default-widget";
    e_attr->description =
	"Whether this widget is the 'default' for the window.";
    e_attr->value_desc = "boolean";
    e_attr->set_attr_func = rendgtk_widget_default_widget_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onfocus";
    e_attr->description = "Specify function called on a focus in event.";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(event_node)";
    e_attr->set_attr_func = rendgtk_widget_widget_event_set;
    element_register_attrib (element, e_attr);

    rendgtk_type_widget_style_attr_register (element);
}



/*****************************************************************
 *     misc widget section    *
 *****************************************************************/

static gint
rendgtk_widget_misc_align_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *widget;
    gfloat xfval;
    gfloat yfval;
    EBuf *xval;
    EBuf *yval;

    widget = enode_get_kv (node, "top-widget");
    if (!widget)
	return (TRUE);

    /* FIXME: These need to use enode_attrib_str_str proper. */
    yval = enode_attrib (node, "yalign", NULL);
    if (ebuf_not_empty (yval))
	yfval = erend_get_percentage (yval);
    else
	yfval = 0.5;

    xval = enode_attrib (node, "xalign", NULL);
    if (ebuf_not_empty (xval))
	xfval = erend_get_percentage (xval);
    else
	xfval = 0.5;

    gtk_misc_set_alignment (GTK_MISC (widget), xfval, yfval);
    return (TRUE);
}

static gint
rendgtk_widget_misc_pad_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *widget;
    gint xival;
    gint yival;
    EBuf *xval;
    EBuf *yval;

    widget = enode_get_kv (node, "top-widget");
    if (!widget)
	return (TRUE);

    /* FIXME: These need to use enode_attrib_str_str too */
    yval = enode_attrib (node, "yalign", NULL);
    if (ebuf_not_empty (yval))
	yival = erend_get_integer (yval);
    else
	yival = 1;

    xval = enode_attrib (node, "xalign", NULL);
    if (ebuf_not_empty (xval))
	xival = erend_get_integer (xval);
    else
	xival = 1;

    gtk_misc_set_padding (GTK_MISC (widget), xival, yival);
    return (TRUE);
}

static void
rendgtk_type_misc_attr_register (Element * element)
{
    ElementAttr *e_attr;

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "yalign";
    e_attr->description = "Virtical alignment.";
    e_attr->value_desc = "percentage";
    e_attr->set_attr_func = rendgtk_widget_misc_align_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "xalign";
    e_attr->description = "Horizontal alignment.";
    e_attr->value_desc = "percentage";
    e_attr->set_attr_func = rendgtk_widget_misc_align_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "xpad";
    e_attr->description = "Horizontal padding in pixels.";
    e_attr->value_desc = "integer";
    e_attr->set_attr_func = rendgtk_widget_misc_pad_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "ypad";
    e_attr->description = "Vertical padding in pixels.";
    e_attr->value_desc = "integer";
    e_attr->set_attr_func = rendgtk_widget_misc_pad_set;
    element_register_attrib (element, e_attr);
}

static gint
rendgtk_button_relief_style_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *widget;
    GtkReliefStyle style;
    
    widget = enode_get_kv (node, "top-widget");
    if (!widget)
	return (TRUE);

    if (ebuf_equal_str (value, "none")) {
	style = GTK_RELIEF_NONE;
    } else if (ebuf_equal_str (value, "half")) {
	style = GTK_RELIEF_HALF;
    } else {
	style = GTK_RELIEF_NORMAL;
    }

    gtk_button_set_relief (GTK_BUTTON(widget), style);

    return (TRUE);
}

static void
rendgtk_type_button_attr_register (Element * element)
{
    ElementAttr *e_attr;

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "relief-style";
    e_attr->description = "Set style of button border.";
    e_attr->value_desc = "choice";
    e_attr->possible_values = "normal,half,none";
    e_attr->set_attr_func = rendgtk_button_relief_style_set;
    element_register_attrib (element, e_attr);
}



/*****************************************************************
 *     general widget attribute registration			 *
 *****************************************************************/


void
rendgtk_widget_attr_register (Element * element, GtkType widget_type)
{
    GtkObjectClass *klass;
    GtkType type;

    /* Ensure the class is initialized */
    klass = gtk_type_class (widget_type);

    type = widget_type;

    /* ITERATE through types until either you find the attr your looking *
     * for your you end up at GTK_OBJECT! */
    while ((type != GTK_TYPE_NONE) && (type != GTK_TYPE_OBJECT)) {

	if (type == GTK_TYPE_WIDGET) {
	    rendgtk_type_widget_attr_register (element);
	}

	else if (type == GTK_TYPE_CONTAINER) {
	    rendgtk_type_container_attr_register (element);
	}

	else if (type == GTK_TYPE_BUTTON) {
	    rendgtk_type_button_attr_register (element);
	}

	else if (type == GTK_TYPE_BOX) {
	    rendgtk_type_box_attr_register (element);
	}

	else if (type == GTK_TYPE_MISC) {
	    rendgtk_type_misc_attr_register (element);
	}

	else if (type == GTK_TYPE_TOGGLE_BUTTON) {
	    rendgtk_type_toggle_button_attr_register (element);
	}

	else if (type == GTK_TYPE_EDITABLE) {
	    rendgtk_type_editable_attr_register (element);
	}

	type = gtk_type_parent (type);
    }
}
