# -*- coding: utf-8 -*-

# Copyright (c) 2002 - 2006 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a configuration dialog for the tools menu.
"""

import sys
import os
import copy

from qt import *

from KdeQt import KQFileDialog, KQMessageBox

from ToolConfigurationForm import ToolConfigurationForm
import Utilities

class ToolConfigurationDialog(ToolConfigurationForm):
    """
    Class implementing a configuration dialog for the tools menu.
    """
    def __init__(self, toollist, parent=None):
        """
        Constructor
        
        @param toollist list of configured tools
        @param parent parent widget (QWidget)
        """
        ToolConfigurationForm.__init__(self, parent, None, 1)
        
        self.toollist = copy.deepcopy(toollist)
        for tool in toollist:
            self.toolsListBox.insertItem(tool[0])
            
        if len(toollist):
            self.toolsListBox.setCurrentItem(0)
            
        t = QWhatsThis.textFor(self.argumentsEdit)
        if not t.isEmpty():
            t = t.append(Utilities.getPercentReplacementHelp())
            QWhatsThis.add(self.argumentsEdit, t)
        
    def handleNew(self):
        """
        Private slot to clear all entry fields.
        """
        self.executableEdit.clear()
        self.menuEdit.clear()
        self.argumentsEdit.clear()
        self.redirectCheckBox.setChecked(1)
        
    def handleAdd(self):
        """
        Private slot to add a new entry.
        """
        executable = self.executableEdit.text()
        menutext = self.menuEdit.text()
        arguments = self.argumentsEdit.text()
        redirect = self.redirectCheckBox.isChecked()
        
        if executable.isEmpty():
            KQMessageBox.critical(self,
                self.trUtf8("Add tool entry"),
                self.trUtf8("You have to set an executable to add to the"
                    " Tools-Menu first."),
                self.trUtf8("OK"))
            return
            
        if menutext.isEmpty():
            KQMessageBox.critical(self,
                self.trUtf8("Add tool entry"),
                self.trUtf8("You have to insert a menuentry text to add the"
                    " selected program to the Tools-Menu first."),
                self.trUtf8("OK"))
            return
            
        executable = unicode(executable)
        if not Utilities.isinpath(executable):
            KQMessageBox.critical(self,
                self.trUtf8("Add tool entry"),
                self.trUtf8("The selected file could not be found or"
                    " is not an executable."
                    " Please choose an executable filename."),
                self.trUtf8("OK"))
            return
            
        self.toolsListBox.insertItem(menutext)
        self.toollist.append((unicode(menutext), executable, 
            unicode(arguments), redirect))
        
    def handleChange(self):
        """
        Private slot to change an entry.
        """
        itm = self.toolsListBox.currentItem()
        if itm == -1:
            return
            
        executable = self.executableEdit.text()
        menutext = self.menuEdit.text()
        arguments = self.argumentsEdit.text()
        redirect = self.redirectCheckBox.isChecked()
        
        if executable.isEmpty():
            KQMessageBox.critical(self,
                self.trUtf8("Change tool entry"),
                self.trUtf8("You have to set an executable to change the"
                    " Tools-Menu entry."),
                self.trUtf8("OK"))
            return
            
        if menutext.isEmpty():
            KQMessageBox.critical(self,
                self.trUtf8("Change tool entry"),
                self.trUtf8("You have to insert a menuentry text to change the"
                    " selected Tools-Menu entry."),
                self.trUtf8("OK"))
            return
            
        executable = unicode(executable)
        if not Utilities.isinpath(executable):
            KQMessageBox.critical(self,
                self.trUtf8("Change tool entry"),
                self.trUtf8("The selected file could not be found or"
                    " is not an executable."
                    " Please choose an existing executable filename."),
                self.trUtf8("OK"))
            return
            
        self.toollist[itm] = \
            (unicode(menutext), executable, unicode(arguments), redirect)
        self.toolsListBox.changeItem(menutext, itm)
        
    def handleDelete(self):
        """
        Private slot to delete the selected entry.
        """
        itm = self.toolsListBox.currentItem()
        if itm == -1:
            return
            
        del self.toollist[itm]
        self.toolsListBox.removeItem(itm)
        
    def handleDown(self):
        """
        Private slot to move an entry down in the list.
        """
        curr = self.toolsListBox.currentItem()
        self.swap(curr, curr+1)
        self.toolsListBox.clear()
        for tool in self.toollist:
            self.toolsListBox.insertItem(tool[0])
        self.toolsListBox.setCurrentItem(curr+1)
        if curr+1 == len(self.toollist):
            self.downButton.setEnabled(0)
        self.upButton.setEnabled(1)
        
    def handleUp(self):
        """
        Private slot to move an entry up in the list.
        """
        curr = self.toolsListBox.currentItem()
        self.swap(curr-1, curr)
        self.toolsListBox.clear()
        for tool in self.toollist:
            self.toolsListBox.insertItem(tool[0])
        self.toolsListBox.setCurrentItem(curr-1)
        if curr-1 == 0:
            self.upButton.setEnabled(0)
        self.downButton.setEnabled(1)
        
    def handleExeSelect(self):
        """
        Private slot to handle the executable selection via a file selection dialog.
        """
        execfile = KQFileDialog.getOpenFileName()
        if not execfile.isNull() and not execfile.isEmpty():
            execfile = unicode(QDir.convertSeparators(execfile))
            if not Utilities.isinpath(execfile):
                KQMessageBox.critical(self,
                    self.trUtf8("Select executable"),
                    self.trUtf8("The selected file is not an executable."
                        " Please choose an executable filename."),
                    self.trUtf8("OK"))
                return
            
            self.executableEdit.setText(execfile)
        
    def handleToolEntrySelected(self, index):
        """
        Private slot to set the lineedits depending on the selected entry.
        
        @param index the index of the selected entry (integer)
        """
        if index >= 0:
            tool = self.toollist[index]
            self.menuEdit.setText(tool[0])
            self.executableEdit.setText(tool[1])
            self.argumentsEdit.setText(tool[2])
            self.redirectCheckBox.setChecked(tool[3])
            
            self.deleteButton.setEnabled(1)
            self.changeButton.setEnabled(1)
            
            if index != 0:
                self.upButton.setEnabled(1)
            else:
                self.upButton.setEnabled(0)
                
            if index+1 != len(self.toollist):
                self.downButton.setEnabled(1)
            else:
                self.downButton.setEnabled(0)
        else:
            self.executableEdit.clear()
            self.menuEdit.clear()
            self.argumentsEdit.clear()
            self.downButton.setEnabled(0)
            self.upButton.setEnabled(0)
            self.deleteButton.setEnabled(0)
            self.changeButton.setEnabled(0)
        
    def getToollist(self):
        """
        Public method to retrieve the tools list. 
        
        @return a list of tuples containing the menu text, the executable, 
            the executables arguments and a redirection flag
        """
        return self.toollist
        
    def swap(self, itm1, itm2):
        """
        Private method used two swap two list entries given by their index.
        
        @param itm1 index of first entry (int)
        @param itm2 index of second entry (int)
        """
        tmp = self.toollist[itm1]
        self.toollist[itm1] = self.toollist[itm2]
        self.toollist[itm2] = tmp
