# -*- coding: utf-8 -*-

# Copyright (c) 2005 - 2006 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the VCS project browser helper for subversion.
"""

import os

from qt import *

from VCS.ProjectBrowserHelper import VcsProjectBrowserHelper
from UI.Browser import BrowserFile
from Project.ProjectBrowser import ProjectBrowserSimpleDir, ProjectOthersBrowser

import Preferences

class SvnProjectBrowserHelper(VcsProjectBrowserHelper):
    """
    Class implementing the VCS project browser helper for subversion.
    """
    def __init__(self, vcsObject, browserObject, projectObject, isTranslationsBrowser,
        parent=None, name=None):
        """
        Constructor
        
        @param vcsObject reference to the vcs object
        @param browserObject reference to the project browser object
        @param projectObject reference to the project object
        @param isTranslationsBrowser flag indicating, the helper is requested for the
            translations browser (this needs some special treatment)
        @param parent parent widget (QWidget)
        @param name name of this object (string or QString)
        """
        VcsProjectBrowserHelper.__init__(self, vcsObject, browserObject, projectObject,
            isTranslationsBrowser, parent, name)
    
    def handleShowPopupMenu(self, menu, standardItems):
        """
        Slot called before the context menu is shown. 
        
        It enables/disables the VCS menu entries depending on the overall 
        VCS status and the file status.
        
        @param menu reference to the menu to be shown
        @param standardItems array of standard items that need activation/deactivation
            depending on the overall VCS status
        """
        if unicode(self.browser.currentItem().text(1)) == self.vcs.vcsName():
            for itm in self.vcsMenuItems:
                menu.setItemEnabled(itm, 1)
            for itm in self.vcsAddMenuItems:
                menu.setItemEnabled(itm, 0)
            for itm in standardItems:
                menu.setItemEnabled(itm, 0)
        else:
            for itm in self.vcsMenuItems:
                menu.setItemEnabled(itm, 0)
            for itm in self.vcsAddMenuItems:
                menu.setItemEnabled(itm, 1)
            if 1 in self.browser.specialMenuEntries and \
               not os.path.isdir(unicode(self.browser.currentItem().fileName())):
                menu.setItemEnabled(self.vcsMenuAddTree, 0)
            for itm in standardItems:
                menu.setItemEnabled(itm, 1)
    
    def handleShowPopupMenuMulti(self, menu, standardItems):
        """
        Slot called before the context menu (multiple selections) is shown. 
        
        It enables/disables the VCS menu entries depending on the overall 
        VCS status and the files status.
        
        @param menu reference to the menu to be shown
        @param standardItems array of standard items that need activation/deactivation
            depending on the overall VCS status
        """
        vcsName = self.vcs.vcsName()
        items = self.browser.getSelectedItems()
        vcsItems = 0
        # determine number of selected items under VCS control
        for itm in items:
            if unicode(itm.text(1)) == vcsName:
                vcsItems += 1
        
        if vcsItems > 0:
            if vcsItems != len(items):
                for itm in self.vcsMultiMenuItems:
                    menu.setItemEnabled(itm, 0)
            else:
                for itm in self.vcsMultiMenuItems:
                    menu.setItemEnabled(itm, 1)
            for itm in self.vcsAddMultiMenuItems:
                menu.setItemEnabled(itm, 0)
            for itm in standardItems:
                menu.setItemEnabled(itm, 0)
        else:
            for itm in self.vcsMultiMenuItems:
                menu.setItemEnabled(itm, 0)
            for itm in self.vcsAddMultiMenuItems:
                menu.setItemEnabled(itm, 1)
            if 1 in self.browser.specialMenuEntries and \
               self.__itemsHaveFiles(items):
                menu.setItemEnabled(self.vcsMultiMenuAddTree, 0)
            for itm in standardItems:
                menu.setItemEnabled(itm, 1)
    
    def handleShowPopupMenuDir(self, menu, standardItems):
        """
        Slot called before the context menu is shown. 
        
        It enables/disables the VCS menu entries depending on the overall 
        VCS status and the directory status.
        
        @param menu reference to the menu to be shown
        @param standardItems array of standard items that need activation/deactivation
            depending on the overall VCS status
        """
        if unicode(self.browser.currentItem().text(1)) == self.vcs.vcsName():
            for itm in self.vcsDirMenuItems:
                menu.setItemEnabled(itm, 1)
            for itm in self.vcsAddDirMenuItems:
                menu.setItemEnabled(itm, 0)
            for itm in standardItems:
                menu.setItemEnabled(itm, 0)
        else:
            for itm in self.vcsDirMenuItems:
                menu.setItemEnabled(itm, 0)
            for itm in self.vcsAddDirMenuItems:
                menu.setItemEnabled(itm, 1)
            for itm in standardItems:
                menu.setItemEnabled(itm, 1)
    
    def handleShowPopupMenuDirMulti(self, menu, standardItems):
        """
        Slot called before the context menu is shown. 
        
        It enables/disables the VCS menu entries depending on the overall 
        VCS status and the directory status.
        
        @param menu reference to the menu to be shown
        @param standardItems array of standard items that need activation/deactivation
            depending on the overall VCS status
        """
        vcsName = self.vcs.vcsName()
        items = self.browser.getSelectedItems()
        vcsItems = 0
        # determine number of selected items under VCS control
        for itm in items:
            if unicode(itm.text(1)) == vcsName:
                vcsItems += 1
        
        if vcsItems > 0:
            if vcsItems != len(items):
                for itm in self.vcsDirMultiMenuItems:
                    menu.setItemEnabled(itm, 0)
            else:
                for itm in self.vcsDirMultiMenuItems:
                    menu.setItemEnabled(itm, 1)
            for itm in self.vcsAddDirMultiMenuItems:
                menu.setItemEnabled(itm, 0)
            for itm in standardItems:
                menu.setItemEnabled(itm, 0)
        else:
            for itm in self.vcsDirMultiMenuItems:
                menu.setItemEnabled(itm, 0)
            for itm in self.vcsAddDirMultiMenuItems:
                menu.setItemEnabled(itm, 1)
            for itm in standardItems:
                menu.setItemEnabled(itm, 1)

    ############################################################################
    # Private menu generation methods below
    ############################################################################

    def addVCSMenu(self, menu):
        """
        Public method used to add the VCS menu to all project browsers.
        
        @param menu reference to the menu to be amended
        """
        self.vcsMenuItems = []
        self.vcsAddMenuItems = []
        
        lbl = QLabel(self.vcs.vcsName(), menu)
        lbl.setFrameStyle( QFrame.Panel | QFrame.Sunken )
        lbl.setAlignment(Qt.AlignHCenter)
        font = lbl.font()
        font.setBold(1)
        lbl.setFont(font)
        menu.insertItem(lbl)
        
        itm = menu.insertItem(self.trUtf8('Update from repository'), self.handleVCSUpdate)
        self.vcsMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Commit changes to repository...'), self.handleVCSCommit)
        self.vcsMenuItems.append(itm)
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Add to repository'), self.handleVCSAdd)
        self.vcsAddMenuItems.append(itm)
        if 1 in self.browser.specialMenuEntries:
            self.vcsMenuAddTree = menu.insertItem(self.trUtf8('Add tree to repository'), self.handleVCSAddTree)
            self.vcsAddMenuItems.append(self.vcsMenuAddTree)
        itm = menu.insertItem(self.trUtf8('Remove from repository (and disk)'), self.handleVCSRemove)
        self.vcsMenuItems.append(itm)
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Copy in repository'), self.handleSVNCopy)
        self.vcsMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Move in repository'), self.handleSVNMove)
        self.vcsMenuItems.append(itm)
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Show log'), self.handleVCSLog)
        self.vcsMenuItems.append(itm)
        if self.vcs.versionStr >= '1.2.0':
            itm = menu.insertItem(self.trUtf8('Show limited log'), self.handleSVNLogLimited)
            self.vcsMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Show status'), self.handleVCSStatus)
        self.vcsMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Show difference'), self.handleVCSDiff)
        self.vcsMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Show difference (extended)'), self.handleSVNExtendedDiff)
        self.vcsMenuItems.append(itm)
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Revert changes'), self.handleVCSRevert)
        self.vcsMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Merge changes'), self.handleVCSMerge)
        self.vcsMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Resolve conflict'), self.handleSVNResolve)
        self.vcsMenuItems.append(itm)
        if self.vcs.versionStr >= '1.2.0':
            menu.insertSeparator()
            itm = menu.insertItem(self.trUtf8('Lock'), self.handleSVNLock)
            self.vcsMenuItems.append(itm)
            itm = menu.insertItem(self.trUtf8('Unlock'), self.handleSVNUnlock)
            self.vcsMenuItems.append(itm)
            itm = menu.insertItem(self.trUtf8('Break Lock'), self.handleSVNBreakLock)
            self.vcsMenuItems.append(itm)
            itm = menu.insertItem(self.trUtf8('Steal Lock'), self.handleSVNStealLock)
            self.vcsMenuItems.append(itm)
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Set Property'), self.handleSVNSetProp)
        self.vcsMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('List Properties'), self.handleSVNListProps)
        self.vcsMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Delete Property'), self.handleSVNDelProp)
        self.vcsMenuItems.append(itm)
        menu.insertSeparator()
        menu.insertItem(self.trUtf8('Select all local file entries'),
                        self.browser.handleSelectLocalEntries)
        menu.insertItem(self.trUtf8('Select all versioned file entries'),
                        self.browser.handleSelectVCSEntries)
        menu.insertItem(self.trUtf8('Select all local directory entries'),
                        self.browser.handleSelectLocalDirEntries)
        menu.insertItem(self.trUtf8('Select all versioned directory entries'),
                        self.browser.handleSelectVCSDirEntries)
        
    def addVCSMenuMulti(self, menu):
        """
        Public method used to add the VCS menu for multi selection to all project browsers.
        
        @param menu reference to the menu to be amended
        """
        self.vcsMultiMenuItems = []
        self.vcsAddMultiMenuItems = []
        
        lbl = QLabel(self.vcs.vcsName(), menu)
        lbl.setFrameStyle( QFrame.Panel | QFrame.Sunken )
        lbl.setAlignment(Qt.AlignHCenter)
        font = lbl.font()
        font.setBold(1)
        lbl.setFont(font)
        menu.insertItem(lbl)
        
        itm = menu.insertItem(self.trUtf8('Update from repository'), self.handleVCSUpdate)
        self.vcsMultiMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Commit changes to repository...'), self.handleVCSCommit)
        self.vcsMultiMenuItems.append(itm)
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Add to repository'), self.handleVCSAdd)
        self.vcsAddMultiMenuItems.append(itm)
        if 1 in self.browser.specialMenuEntries:
            self.vcsMultiMenuAddTree = menu.insertItem(self.trUtf8('Add tree to repository'), self.handleVCSAddTree)
            self.vcsAddMultiMenuItems.append(self.vcsMultiMenuAddTree)
        itm = menu.insertItem(self.trUtf8('Remove from repository (and disk)'), self.handleVCSRemove)
        self.vcsMultiMenuItems.append(itm)
        self.vcsRemoveMultiMenuItem = itm
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Show status'), self.handleVCSStatus)
        self.vcsMultiMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Show difference'), self.handleVCSDiff)
        self.vcsMultiMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Show difference (extended)'), self.handleSVNExtendedDiff)
        self.vcsMultiMenuItems.append(itm)
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Revert changes'), self.handleVCSRevert)
        self.vcsMultiMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Resolve conflict'), self.handleSVNResolve)
        self.vcsMultiMenuItems.append(itm)
        if self.vcs.versionStr >= '1.2.0':
            menu.insertSeparator()
            itm = menu.insertItem(self.trUtf8('Lock'), self.handleSVNLock)
            self.vcsMultiMenuItems.append(itm)
            itm = menu.insertItem(self.trUtf8('Unlock'), self.handleSVNUnlock)
            self.vcsMultiMenuItems.append(itm)
            itm = menu.insertItem(self.trUtf8('Break Lock'), self.handleSVNBreakLock)
            self.vcsMultiMenuItems.append(itm)
            itm = menu.insertItem(self.trUtf8('Steal Lock'), self.handleSVNStealLock)
            self.vcsMultiMenuItems.append(itm)
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Set Property'), self.handleSVNSetProp)
        self.vcsMultiMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('List Properties'), self.handleSVNListProps)
        self.vcsMultiMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Delete Property'), self.handleSVNDelProp)
        self.vcsMultiMenuItems.append(itm)
        menu.insertSeparator()
        menu.insertItem(self.trUtf8('Select all local file entries'),
                        self.browser.handleSelectLocalEntries)
        menu.insertItem(self.trUtf8('Select all versioned file entries'),
                        self.browser.handleSelectVCSEntries)
        menu.insertItem(self.trUtf8('Select all local directory entries'),
                        self.browser.handleSelectLocalDirEntries)
        menu.insertItem(self.trUtf8('Select all versioned directory entries'),
                        self.browser.handleSelectVCSDirEntries)
        
    def addVCSMenuBack(self, menu):
        """
        Public method used to add the VCS menu to all project browsers.
        
        @param menu reference to the menu to be amended
        """
        lbl = QLabel(self.vcs.vcsName(), menu)
        lbl.setFrameStyle( QFrame.Panel | QFrame.Sunken )
        lbl.setAlignment(Qt.AlignHCenter)
        font = lbl.font()
        font.setBold(1)
        lbl.setFont(font)
        menu.insertItem(lbl)
        
        menu.insertItem(self.trUtf8('Select all local file entries'),
                        self.browser.handleSelectLocalEntries)
        menu.insertItem(self.trUtf8('Select all versioned file entries'),
                        self.browser.handleSelectVCSEntries)
        menu.insertItem(self.trUtf8('Select all local directory entries'),
                        self.browser.handleSelectLocalDirEntries)
        menu.insertItem(self.trUtf8('Select all versioned directory entries'),
                        self.browser.handleSelectVCSDirEntries)
        
    def addVCSMenuDir(self, menu):
        """
        Public method used to add the VCS menu to all project browsers.
        
        @param menu reference to the menu to be amended
        """
        if menu is None:
            return
        
        self.vcsDirMenuItems = []
        self.vcsAddDirMenuItems = []
        
        lbl = QLabel(self.vcs.vcsName(), menu)
        lbl.setFrameStyle( QFrame.Panel | QFrame.Sunken )
        lbl.setAlignment(Qt.AlignHCenter)
        font = lbl.font()
        font.setBold(1)
        lbl.setFont(font)
        menu.insertItem(lbl)
        
        itm = menu.insertItem(self.trUtf8('Update from repository'), self.handleVCSUpdate)
        self.vcsDirMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Commit changes to repository...'), self.handleVCSCommit)
        self.vcsDirMenuItems.append(itm)
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Add to repository'), self.handleVCSAdd)
        self.vcsAddDirMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Remove from repository (and disk)'), self.handleVCSRemove)
        self.vcsDirMenuItems.append(itm)
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Copy in repository'), self.handleSVNCopy)
        self.vcsDirMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Move in repository'), self.handleSVNMove)
        self.vcsDirMenuItems.append(itm)
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Show log'), self.handleVCSLog)
        self.vcsDirMenuItems.append(itm)
        if self.vcs.versionStr >= '1.2.0':
            itm = menu.insertItem(self.trUtf8('Show limited log'), self.handleSVNLogLimited)
            self.vcsDirMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Show status'), self.handleVCSStatus)
        self.vcsDirMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Show difference'), self.handleVCSDiff)
        self.vcsDirMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Show difference (extended)'), self.handleSVNExtendedDiff)
        self.vcsDirMenuItems.append(itm)
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Revert changes'), self.handleVCSRevert)
        self.vcsDirMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Merge changes'), self.handleVCSMerge)
        self.vcsDirMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Resolve conflict'), self.handleSVNResolve)
        self.vcsDirMenuItems.append(itm)
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Set Property'), self.handleSVNSetProp)
        self.vcsDirMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('List Properties'), self.handleSVNListProps)
        self.vcsDirMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Delete Property'), self.handleSVNDelProp)
        self.vcsDirMenuItems.append(itm)
        menu.insertSeparator()
        menu.insertItem(self.trUtf8('Select all local file entries'),
                        self.browser.handleSelectLocalEntries)
        menu.insertItem(self.trUtf8('Select all versioned file entries'),
                        self.browser.handleSelectVCSEntries)
        menu.insertItem(self.trUtf8('Select all local directory entries'),
                        self.browser.handleSelectLocalDirEntries)
        menu.insertItem(self.trUtf8('Select all versioned directory entries'),
                        self.browser.handleSelectVCSDirEntries)
        
    def addVCSMenuDirMulti(self, menu):
        """
        Public method used to add the VCS menu to all project browsers.
        
        @param menu reference to the menu to be amended
        """
        if menu is None:
            return
        
        self.vcsDirMultiMenuItems = []
        self.vcsAddDirMultiMenuItems = []
        
        lbl = QLabel(self.vcs.vcsName(), menu)
        lbl.setFrameStyle( QFrame.Panel | QFrame.Sunken )
        lbl.setAlignment(Qt.AlignHCenter)
        font = lbl.font()
        font.setBold(1)
        lbl.setFont(font)
        menu.insertItem(lbl)
        
        itm = menu.insertItem(self.trUtf8('Update from repository'), self.handleVCSUpdate)
        self.vcsDirMultiMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Commit changes to repository...'), self.handleVCSCommit)
        self.vcsDirMultiMenuItems.append(itm)
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Add to repository'), self.handleVCSAdd)
        self.vcsAddDirMultiMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Remove from repository (and disk)'), self.handleVCSRemove)
        self.vcsDirMultiMenuItems.append(itm)
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Show status'), self.handleVCSStatus)
        self.vcsDirMultiMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Show difference'), self.handleVCSDiff)
        self.vcsDirMultiMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Show difference (extended)'), self.handleSVNExtendedDiff)
        self.vcsDirMultiMenuItems.append(itm)
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Revert changes'), self.handleVCSRevert)
        self.vcsDirMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Merge changes'), self.handleVCSMerge)
        self.vcsDirMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Resolve conflict'), self.handleSVNResolve)
        self.vcsDirMultiMenuItems.append(itm)
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Set Property'), self.handleSVNSetProp)
        self.vcsDirMultiMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('List Properties'), self.handleSVNListProps)
        self.vcsDirMultiMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Delete Property'), self.handleSVNDelProp)
        self.vcsDirMultiMenuItems.append(itm)
        menu.insertSeparator()
        menu.insertItem(self.trUtf8('Select all local file entries'),
                        self.browser.handleSelectLocalEntries)
        menu.insertItem(self.trUtf8('Select all versioned file entries'),
                        self.browser.handleSelectVCSEntries)
        menu.insertItem(self.trUtf8('Select all local directory entries'),
                        self.browser.handleSelectLocalDirEntries)
        menu.insertItem(self.trUtf8('Select all versioned directory entries'),
                        self.browser.handleSelectVCSDirEntries)
        
    ############################################################################
    # Menu handling methods below
    ############################################################################
    
    def handleSVNCopy(self):
        """
        Private slot called by the context menu to copy the selected file.
        """
        itm = self.browser.currentItem()
        fn = unicode(itm.fileName())
        self.vcs.svnCopy(fn, self.project)
        
    def handleSVNMove(self):
        """
        Private slot called by the context menu to move the selected file.
        """
        itm = self.browser.currentItem()
        fn = unicode(itm.fileName())
        movefiles = self.browser.project.getFiles(fn)
        if self.vcs.svnMove(fn, self.project):
            self.browser.removeNode(itm)
            try:
                self.browser.children.remove(itm)
            except ValueError:
                pass
            del itm
            if isinstance(self.browser, ProjectOthersBrowser):
                self.browser.rebuildTree()
            if os.path.isfile(fn):
                self.browser.emit(PYSIGNAL('closeSourceWindow'), (fn,))
            else:
                for mf in movefiles:
                    self.browser.emit(PYSIGNAL('closeSourceWindow'), (mf,))
        
    def handleSVNResolve(self):
        """
        Private slot called by the context menu to resolve conflicts of a file.
        """
        names = [unicode(itm.fileName()) for itm in self.browser.getSelectedItems()]
        self.vcs.svnResolve(names)
        
    def handleSVNListProps(self):
        """
        Private slot called by the context menu to list the subversion properties of a file.
        """
        if self.isTranslationsBrowser:
            items = self.browser.getSelectedItems([ProjectBrowserSimpleDir])
            if items:
                names = [unicode(itm.fileName()) for itm in items]
            else:
                items = self.browser.getSelectedItems([BrowserFile])
                names = [unicode(itm.fileName()) for itm in items]
                names += ['%s.qm' % os.path.splitext(name)[0] for name in names]
        else:
            names = [unicode(itm.fileName()) for itm in self.browser.getSelectedItems()]
        self.vcs.svnListProps(names)
        
    def handleSVNSetProp(self):
        """
        Private slot called by the context menu to set a subversion property of a file.
        """
        if self.isTranslationsBrowser:
            items = self.browser.getSelectedItems([ProjectBrowserSimpleDir])
            if items:
                names = [unicode(itm.fileName()) for itm in items]
            else:
                items = self.browser.getSelectedItems([BrowserFile])
                names = [unicode(itm.fileName()) for itm in items]
                names += ['%s.qm' % os.path.splitext(name)[0] for name in names]
        else:
            names = [unicode(itm.fileName()) for itm in self.browser.getSelectedItems()]
        self.vcs.svnSetProp(names)
        
    def handleSVNDelProp(self):
        """
        Private slot called by the context menu to delete a subversion property of a file.
        """
        if self.isTranslationsBrowser:
            items = self.browser.getSelectedItems([ProjectBrowserSimpleDir])
            if items:
                names = [unicode(itm.fileName()) for itm in items]
            else:
                items = self.browser.getSelectedItems([BrowserFile])
                names = [unicode(itm.fileName()) for itm in items]
                names += ['%s.qm' % os.path.splitext(name)[0] for name in names]
        else:
            names = [unicode(itm.fileName()) for itm in self.browser.getSelectedItems()]
        self.vcs.svnDelProp(names)
        
    def handleSVNExtendedDiff(self):
        """
        Private slot called by the context menu to show the difference of a file to the repository.
        
        This gives the chance to enter the revisions to compare.
        """
        names = [unicode(itm.fileName()) for itm in self.browser.getSelectedItems()]
        self.vcs.svnExtendedDiff(names)
        
    def handleSVNLogLimited(self):
        """
        Private slot called by the context menu to show the limited log of a file.
        """
        itm = self.browser.currentItem()
        fn = unicode(itm.fileName())
        self.vcs.svnLogLimited(fn)
        
    def handleSVNLock(self):
        """
        Private slot called by the context menu to lock files in the repository.
        """
        names = [unicode(itm.fileName()) for itm in self.browser.getSelectedItems()]
        self.vcs.svnLock(names)
        
    def handleSVNUnlock(self):
        """
        Private slot called by the context menu to unlock files in the repository.
        """
        names = [unicode(itm.fileName()) for itm in self.browser.getSelectedItems()]
        self.vcs.svnUnlock(names)
        
    def handleSVNBreakLock(self):
        """
        Private slot called by the context menu to break lock files in the repository.
        """
        names = [unicode(itm.fileName()) for itm in self.browser.getSelectedItems()]
        self.vcs.svnUnlock(names, breakIt=True)
        
    def handleSVNStealLock(self):
        """
        Private slot called by the context menu to steal lock files in the repository.
        """
        names = [unicode(itm.fileName()) for itm in self.browser.getSelectedItems()]
        self.vcs.svnLock(names, stealIt=True)

    ############################################################################
    # Some private utility methods below
    ############################################################################
    
    def __itemsHaveFiles(self, items):
        """
        Private method to check, if items contain file type items.
        
        @param items items to check (list of QListViewItems)
        @return flag indicating items contain file type items (boolean)
        """
        for itm in items:
            if isinstance(itm, BrowserFile):
                return 1
        return 0
