/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: shrink.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

#include <fullengine.h>

#include "shrink.h"
#include "engine.h"
#include "handlemgr.h"
#include "common.h"
#include "commit.h"
#include "discover.h"
#include "object.h"
#include "memman.h"
#include "internalAPI.h"
#include "message.h"


static inline int get_object_shrink_points(storage_object_t * obj, u_int64_t * max_delta_size, dlist_t shrink_points) {

    int rc = 0;

    LOG_PROC_ENTRY();

    rc = obj->plugin->functions.plugin->can_shrink(obj, max_delta_size, shrink_points);

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


static inline int get_volume_shrink_points(logical_volume_t * volume, dlist_t shrink_points) {

    int rc = 0;
    u_int64_t max_delta_size = -1;

    LOG_PROC_ENTRY();

    /*
     * If the volume has an FSIM, gets its maximum delta shrink size.  We reget
     * the current size and limits because the file system could be mounted and
     * be active.
     */
    if (volume->file_system_manager != NULL) {
        rc = volume->file_system_manager->functions.fsim->get_fs_size(volume, &volume->fs_size);
        if (rc == 0) {
            rc = volume->file_system_manager->functions.fsim->get_fs_limits(volume,
                                                                            &volume->min_fs_size,
                                                                            &volume->max_fs_size,
                                                                            &volume->max_vol_size);
            if (rc == 0) {
                max_delta_size = volume->fs_size - volume->min_fs_size;
            }
        }
    }

    if (rc == 0) {
        if (!(volume->flags & VOLFLAG_MKFS)) {
            rc = get_object_shrink_points(volume->object, &max_delta_size, shrink_points);

        } else {
            LOG_ERROR("Volume \"%s\" cannot be shrunk because it is scheduled to have a file system installed on it.\n", volume->name);
            rc = EINVAL;
        }
    }

    if (rc == 0) {
        /*
         * If the volume has an FSIM and there are shrink points, check if the
         * FSIM can handle the shrink.
         */
        if (volume->file_system_manager != NULL) {
            if (!ListEmpty(shrink_points)) {
                rc = volume->file_system_manager->functions.fsim->can_shrink_by(volume, &max_delta_size);

                /*
                 * The FSIM cannot handle the shrink.  Delete all the shrink
                 * points from the list.
                 */
                if (rc != 0) {
                    DeleteAllItems(shrink_points, TRUE);
                }
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Find out if an object can be shrunk.  The "thing" must be either a volume
 * or a top level object.  What we'll do is call the internal
 * get_shrink_points() and see if any shrink candidates show up.  If we get
 * shrink points then return 0 to say that the object can be shrunk.  Else,
 * return an error.
 */
int evms_can_shrink(object_handle_t thing) {

    int rc = 0;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {
        rc = translate_handle(thing,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {
            dlist_t shrink_point_list = CreateList();

            if (shrink_point_list != NULL) {
                switch (type) {
                    case DISK:
                    case SEGMENT:
                    case REGION:
                    case EVMS_OBJECT:
                        {
                            storage_object_t * obj = object;

                            /*
                             * Shrinks can only be done on top level objects.
                             */
                            if (is_top_object(obj)) {
                                dlist_t shrink_points = CreateList();

                                if (shrink_points != NULL) {
                                    u_int64_t max_delta_size = obj->size;

                                    rc = get_object_shrink_points(obj, &max_delta_size, shrink_points);

                                    if (rc == 0) {
                                        uint count = 0;

                                        GetListSize(shrink_points, &count);

                                        if (count == 0) {
                                            rc = ENOENT;
                                        }
                                    }

                                    DestroyList(&shrink_points, FALSE);

                                } else {
                                    LOG_CRITICAL("Error allocating memory for a shrink point list.\n");
                                    rc = ENOMEM;
                                }

                            } else {
                                LOG_ERROR("Object %s is not a top level object.  Shrinks can only be done on volumes or top level objects.\n", obj->name);
                                rc = EINVAL;
                            }
                        }
                        break;

                    case VOLUME:
                        {
                            logical_volume_t * volume = (logical_volume_t *) object;
                            dlist_t shrink_points = CreateList();

                            if (shrink_points != NULL) {
                                rc = get_volume_shrink_points(volume, shrink_points);

                                if (rc == 0) {
                                    uint count = 0;

                                    GetListSize(shrink_points, &count);

                                    if (count == 0) {
                                        rc = ENOENT;
                                    }
                                }

                                DestroyList(&shrink_points, FALSE);

                            } else {
                                LOG_CRITICAL("Error allocating memory for a shrink point list.\n");
                                rc = ENOMEM;
                            }
                        }
                        break;

                    default:
                        rc = EINVAL;
                        break;
                }

                DestroyList(&shrink_point_list, FALSE );

            } else {
                rc = ENOMEM;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Given a shrink_info_t, add the app_handle for the object and the size to an
 * shrink_handle_array_t.
 * The parameters of this function are structured so that it can be called by
 * ForEachItem().
 */
static int make_shrink_handle_entry(ADDRESS object,
                                    TAG     object_tag,
                                    uint    object_size,
                                    ADDRESS object_handle,
                                    ADDRESS parameters) {
    int rc = 0;

    shrink_object_info_t * object_info = (shrink_object_info_t *) object;
    shrink_handle_array_t * sha = (shrink_handle_array_t *) parameters;

    LOG_PROC_ENTRY();

    LOG_DEBUG("Current number of entries in handle array:  %d.\n", sha->count);

    if (object_tag == SHRINK_OBJECT_TAG) {
        LOG_DEBUG("Add entry for storage object %s.\n", object_info->object->name);

        rc = ensure_app_handle(object_info->object,
                               object_info->object->object_type,
                               &(object_info->object->app_handle));
        if (rc == 0) {
            sha->shrink_point[sha->count].object = object_info->object->app_handle;
            sha->shrink_point[sha->count].max_shrink_size = object_info->max_shrink_size;
            sha->count++;
        }

    } else {
        LOG_WARNING("Cannot make a shrink handle entry from an object with tag %ld.\n", object_tag);
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Make an array of shrink handles (shrink_handle_array_t) for the objects in
 * a dlist_t.
 */
static int make_shrink_handle_array(dlist_t list, shrink_handle_array_t * * sha) {
    int rc = 0;
    uint count;
    uint size;

    LOG_PROC_ENTRY();

    rc = GetListSize(list, &count);

    if (rc == 0) {
        LOG_DEBUG("Number of objects in the list:  %d\n", count);
        if (count > 1) {
            size = sizeof(shrink_handle_array_t) + ((count -1) * sizeof(shrink_handle_t));
        } else {
            size = sizeof(shrink_handle_array_t);
        }

        *sha = alloc_app_struct(size, NULL);
        if (*sha != NULL) {
            rc = ForEachItem(list,
                             make_shrink_handle_entry,
                             *sha,
                             TRUE);
        } else {
            rc = ENOMEM;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Find out which objects in the stack can shrink.  evms_get_shrink_points()
 * must be targeted at either a volume or a top level object.  Any plug-in
 * that can shrink will add a shrink_object_info_t to the shrink_points
 * dlist.  The dlist is then converted into a shrink_handle_array_t for return
 * to the user.
 */
int evms_get_shrink_points(object_handle_t thing, shrink_handle_array_t * * shrink_points) {

    int rc = 0;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {

        rc = translate_handle(thing,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {

            if (shrink_points != NULL) {
                dlist_t shrink_point_list = CreateList();

                if (shrink_point_list != NULL) {

                    switch (type) {
                        case VOLUME:
                            {
                                logical_volume_t * volume = (logical_volume_t *) object;

                                rc = get_volume_shrink_points(volume, shrink_point_list);
                            }
                            break;

                        case EVMS_OBJECT:
                        case REGION:
                        case SEGMENT:
                        case DISK:
                            {
                                storage_object_t * obj = (storage_object_t *) object;
                                /*
                                 * Shrinks can only be done on top level objects.
                                 */
                                if (is_top_object(obj)) {
                                    u_int64_t max_delta_size = obj->size;

                                    rc = get_object_shrink_points(obj, &max_delta_size, shrink_point_list);

                                } else {
                                    LOG_ERROR("Object %s is not a top level object.  Shrinks can only be done on volumes or top level objects.\n", obj->name);
                                    rc = EINVAL;
                                }
                            }
                            break;

                        default:
                            LOG_ERROR("An object of type %d cannot be shrunk.\n", type);
                            rc = EINVAL;
                            break;
                    }

                    if (rc == 0) {
                        rc = make_shrink_handle_array(shrink_point_list, shrink_points);
                    }

                    DestroyList(&shrink_point_list, FALSE);
                }

            } else {
                LOG_ERROR("The pointer to the shrink points list cannot be NULL.\n");
                rc = EINVAL;
            }

        } else {
            LOG_CRITICAL("Error when allocating memory for a shrink point list.\n");
            rc = ENOMEM;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Determine if the given object is a storage object.  Return 0 if it is, else
 * an error code.
 * The parameters of this function are structured so that it can be called by
 * the ForEachItem() dlist processor.
 */
static int is_storage_object(ADDRESS object,
                             TAG     object_tag,
                             uint    object_size,
                             ADDRESS object_handle,
                             ADDRESS parameters) {
    int rc = 0;

    LOG_PROC_ENTRY();

    if ((object_tag != EVMS_OBJECT_TAG) &&
        (object_tag != REGION_TAG) &&
        (object_tag != SEGMENT_TAG) &&
        (object_tag != DISK_TAG)) {

        switch (object_tag) {
            case PLUGIN_TAG:
                LOG_ERROR("Object is a plug-in.\n");
                break;
            case CONTAINER_TAG:
                LOG_ERROR("Object is a container.\n");
                break;
            case VOLUME_TAG:
                LOG_ERROR("Object is a volume.\n");
                break;
            default:
                LOG_ERROR("Object is of unknown type %d.\n", object_tag);
                break;
        }
        rc = EINVAL;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * If the object is part of a volume and the volume has no FSIM, warn the
 * user that there is no FSIM to handle the shrink.  The user has the
 * choice of continuing or aborting.
 * Returns 0 if the user wants to continue, EINTR if the user wants to abort.
 */
static int warn_if_no_fsim(storage_object_t * obj) {

    int rc = 0;

    LOG_PROC_ENTRY();

    if (obj->volume != NULL) {
        if (obj->volume->file_system_manager == NULL) {
            char * choices[] = {"Continue", "Abort", NULL};
            int answer = 0;     /* Initialize to "Continue" */

            engine_user_message(&answer,
                                choices,
                                "WARNING: Volume \"%s\" does not have an associated File System Interface Module which could coordinate the shrinking of the file system on the volume.  "
                                "The file system will not be shrunk.  "
                                "You may need to run a separate utility to shrink the file system before this operation starts.  "
                                "Shrinking the volume may make the file system unusable.  "
                                "Do you want to continue with the shrink or abort?\n",
                                obj->volume->name);

            if (answer != 0) {
                rc = EINTR;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Shrink the size of an object.  The shrink gets targeted at the top object of
 * the feature stack.  The plug-ins pass it on down until it hits the object
 * that is to do the real shrink.
 */
int evms_shrink(object_handle_t thing, handle_array_t * input_objects, option_array_t * options) {

    int rc = 0;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_write_access();

    if (rc == 0) {

        rc = translate_handle(thing,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {

            if ((type == EVMS_OBJECT) ||
                (type == REGION) ||
                (type == SEGMENT) ||
                (type == DISK)) {

                storage_object_t * obj = (storage_object_t *) object;

                if (obj->volume != NULL) {
                    if (obj->volume->flags & VOLFLAG_MKFS) {
                        LOG_ERROR("Volume \"%s\" cannot be shrunk because it is scheduled to have a file system installed on it.\n", obj->volume->name);
                        rc = EINVAL;
                    }
                }

                if (rc == 0) {
                    dlist_t input_object_list = CreateList();

                    if (input_object_list != NULL) {
                        rc = make_dlist(input_objects, input_object_list);

                        if (rc == 0) {

                            /*
                             * Make sure all of the input objects are storage
                             * objects.  (The caller could have supplied handles
                             * for bogus items such as plug-ins or volumes.)
                             */
                            rc = ForEachItem(input_object_list,
                                             is_storage_object,
                                             NULL,
                                             TRUE);

                            if (rc == DLIST_SUCCESS) {

                                /*
                                 * Find the top object in the feature stack.
                                 */
                                storage_object_t * top_object = obj;
                                uint parent_count = 0;

                                rc = GetListSize(top_object->parent_objects, &parent_count);

                                while ((rc == DLIST_SUCCESS) && (parent_count > 0)) {
                                    rc = GetObject(top_object->parent_objects,
                                                   sizeof(storage_object_t),
                                                   EVMS_OBJECT_TAG,
                                                   NULL,
                                                   FALSE,
                                                   (ADDRESS *) &top_object);

                                    if (rc == DLIST_SUCCESS) {
                                        rc = GetListSize(top_object->parent_objects, &parent_count);
                                    }
                                }

                                /* Clean up error codes that are OK. */
                                if ((rc == DLIST_EMPTY) || (rc == DLIST_END_OF_LIST)) {
                                    rc = DLIST_SUCCESS;
                                }

                                if (rc == DLIST_SUCCESS) {
                                    /*
                                     * Warn the user if there is no FSIM to
                                     * handle the shrink.
                                     */
                                    rc = warn_if_no_fsim(obj);

                                    if (rc == 0) {
                                        /*
                                         * Start the shrink at the top level
                                         * object.
                                         */
                                        rc = top_object->plugin->functions.plugin->shrink(top_object, obj, input_object_list, options);

                                        if (rc == 0) {

                                            /*
                                             * Check if the input object(s)
                                             * is/are different from the object
                                             * that was shrunk.
                                             */
                                            if ((input_objects->count > 1) ||
                                                ((input_objects->count == 1) &&
                                                 (input_objects->handle[0] != obj->app_handle))) {

                                                /*
                                                 * Make sure the child objects
                                                 * are marked as not being part
                                                 * of a volume.
                                                 */
                                                ForEachItem(input_object_list,
                                                            set_volume_in_object,
                                                            NULL,
                                                            TRUE);
                                            }

                                            if (obj->object_type == EVMS_OBJECT) {
                                                /*
                                                 * Mark all the child feature
                                                 * headers dirty to ensure that
                                                 * they get the correct depth
                                                 * and get written in the right
                                                 * place.
                                                 */
                                                ForEachItem(obj->child_objects,
                                                            mark_feature_headers_dirty,
                                                            NULL,
                                                            TRUE);
                                            }

                                            /*
                                             * If the object is part of a
                                             * volume...
                                             */
                                            if (obj->volume != NULL) {

                                                /* Mark it for rediscovery. */
                                                engine_rediscover_volume(obj->volume, FALSE);

                                                /*
                                                 * Mark it dirty so that the
                                                 * feature headers get written
                                                 * to their new locations.
                                                 */
                                                obj->volume->flags |= VOLFLAG_DIRTY;

                                                /*
                                                 * Update the volume size.  It
                                                 * may not be the resulting
                                                 * volume size after the FSIM
                                                 * does its shrink, but that
                                                 * will be updated after the
                                                 * commit.
                                                 */
                                                obj->volume->vol_size = top_object->size;
                                                if (top_object->feature_header != NULL) {
                                                    /*
                                                     * Object has EVMS volume
                                                     * feature headers.
                                                     * Subtract them from the
                                                     * size.
                                                     */
                                                    obj->volume->vol_size -= FEATURE_HEADER_SECTORS * 2;
                                                }
                                            }
                                        }
                                    }

                                } else {
                                    LOG_WARNING("Error code %d encountered when trying to find the top level object.\n", rc);
                                }

                            } else {
                                LOG_ERROR("One or more items in the input object list is not a storage object.\n");
                                rc = EINVAL;
                            }

                        } else {
                            LOG_ERROR("Error code %d when making a dlist from the input_objects handle array.\n", rc);
                        }

                        DestroyList(&input_object_list, FALSE);

                    } else {
                        LOG_CRITICAL("Error allocating memory to create the input object list.\n");
                        rc = ENOMEM;
                    }
                }

            } else {
                LOG_ERROR("Object of type %d cannot be shrunk.\n", type);
                rc = EINVAL;
            }
        }

        if (rc == 0) {
            changes_pending = TRUE;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * engine_can_shrink_by() is a service that the Engine provides for the
 * plug-ins.  A plug-in calls engine_can_shrink_by() to find out if all of its
 * parent objects, including the volume, would approve of a shrink by the
 * specified amount.  The Engine walks up the object tree calling each
 * plug-in's can_shrink_by() function to see if it would approve of the
 * shrink.  Each plug-in can update the "size" if the resulting size of its
 * object would have a different delta size than the delta size of its child
 * object.  If the top of the object stack is reached and the object is part of
 * a volume, the Engine then checks the resulting delta size against any limits
 * that an FSIM may have imposed on the volume.
 */
int engine_can_shrink_by(storage_object_t * object, sector_count_t * delta_size) {

    int rc = 0;
    storage_object_t * curr_object = object;
    uint parent_count = 0;
    sector_count_t original_delta_size = *delta_size;

    LOG_PROC_ENTRY();

    rc = GetListSize(object->parent_objects, &parent_count);

    while ((rc == DLIST_SUCCESS) && (parent_count > 0)) {
        rc = GetObject(curr_object->parent_objects,
                       sizeof(storage_object_t),
                       EVMS_OBJECT_TAG,
                       NULL,
                       FALSE,
                       (ADDRESS *) &curr_object);

        if (rc == DLIST_SUCCESS) {
            rc = curr_object->plugin->functions.plugin->can_shrink_by(curr_object, delta_size);

            if (rc == 0) {
                rc = GetListSize(curr_object->parent_objects, &parent_count);
            }
        }
    }

    /* Clean up error codes that are OK. */
    if ((rc == DLIST_EMPTY) || (rc == DLIST_END_OF_LIST)) {
        rc = DLIST_SUCCESS;
    }

    if (rc == 0) {
        /*
         * If this object is part of a volume and the volume has an FSIM, check
         * to make sure the final size does not go below the FSIM's minimum size
         * for the file system.
         */
        if (object->volume != NULL) {
            if (object->volume->file_system_manager != NULL) {
                rc = object->volume->file_system_manager->functions.fsim->can_shrink_by(object->volume, delta_size);
            }
        }
    }

    if (rc == 0) {
        if (*delta_size != original_delta_size) {
            rc = EAGAIN;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}

