/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: volume.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <ctype.h>

#include <fullengine.h>
#include "volume.h"
#include "engine.h"
#include "handlemgr.h"
#include "discover.h"
#include "common.h"
#include "internalAPI.h"


/*
 * Utility to check if a volume is mounted.  Update the mount_point
 * in the volume structure.
 */
BOOLEAN is_volume_mounted(logical_volume_t * vol) {

    BOOLEAN result = FALSE;

    LOG_PROC_ENTRY();

    if (vol->mount_point != NULL) {
        free(vol->mount_point);
        vol->mount_point = NULL;
    }

    result = is_mounted(vol->name, &vol->mount_point);

    LOG_PROC_EXIT_BOOLEAN(result);
    return result;
}


/*
 * Can this object be made into an EVMS volume?
 */
int evms_can_create_volume(object_handle_t object_handle) {
    int rc = 0;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();
    if (rc == 0) {
        rc = translate_handle(object_handle,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {
            if ((type == DISK) ||
                (type == SEGMENT) ||
                (type == REGION) ||
                (type == EVMS_OBJECT)) {
                storage_object_t * obj = (storage_object_t *) object;

                /* Object must be a data object. */
                if (obj->data_type == DATA_TYPE) {

                    /* Object must not be corrupt. */
                    if (!(obj->flags & SOFLAG_CORRUPT)) {

                        /* Only top level objects can be made into volumes. */
                        if (is_top_object(obj)) {

                            /*
                             * Ask the object if it can handle being made into a
                             * volume.  Right now we know of no reason why any
                             * plug-in should fail this call.
                             */
                            rc = obj->plugin->functions.plugin->can_set_volume(obj, TRUE);

                        } else {
                            /* The object is not a top level object. */
                            rc = EINVAL;
                        }

                    } else {
                        /* Object is corrupt. */
                        rc = EINVAL;
                    }

                } else {
                    /* Object is not a data object. */
                    rc = EINVAL;
                }

            } else {
                /* The handle is not for a storage object. */
                rc = EINVAL;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Can this object be made into a compatibility volume?
 */
int evms_can_create_compatibility_volume(object_handle_t object_handle) {
    int rc = 0;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();
    if (rc == 0) {
        storage_object_t * obj;

        rc = translate_handle(object_handle,
                              &object,
                              &type);

        obj = (storage_object_t *) object;

        if (rc == HANDLE_MANAGER_NO_ERROR) {
            switch (type) {
                case DISK:
                case SEGMENT:
                case REGION:
                    /*
                     * Object must be a data object, i.e., not meta data, not
                     * free space
                     */
                    if (obj->data_type == DATA_TYPE) {

                        /* Object must not be corrupt. */
                        if (!(obj->flags & SOFLAG_CORRUPT)) {

                            /* Object must be a top object. */
                            if (is_top_object(obj)) {
                                /*
                                 * Ask the object if it can handle being made
                                 * into a volume.  Right now we know of no
                                 * reason why any plug-in should fail this call.
                                 */
                                rc = obj->plugin->functions.plugin->can_set_volume(obj, TRUE);

                            } else {
                                /* The object is not a top level object. */
                                rc = EINVAL;
                            }

                        } else {
                            /* The object is corrupt. */
                            rc = EINVAL;
                        }

                    } else {
                        rc = EINVAL;
                    }
                    break;

                default:
                    rc = EINVAL;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Can this volume have its name changed?
 */
int evms_can_set_volume_name(engine_handle_t volume_handle) {
    int rc = 0;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_write_access();

    if (rc == 0) {
        rc = translate_handle(volume_handle,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {
            if (type == VOLUME) {
                logical_volume_t * vol = (logical_volume_t *) object;

                /* Only EVMS volumes can have their name changed. */
                if (!(vol->flags & VOLFLAG_COMPATIBILITY)) {

                    /* A volume that is mounted cannot have its name changed. */
                    if (!is_volume_mounted(vol)) {
                        /*
                         * Ask the object below the volume if it can have the
                         * volume changed.
                         */
                        storage_object_t * vol_obj = (storage_object_t *) vol->object;

                        rc = vol_obj->plugin->functions.plugin->can_set_volume(vol_obj, TRUE);

                        if (rc != 0) {
                            LOG_ERROR("Object %s will not let the name of volume %s be changed.\n", vol_obj->name, vol->name);
                        }

                    } else{
                        LOG_ERROR("Volume \"%s\" is currently mounted on %s and cannot have its name changed.\n", vol->name, vol->mount_point);
                        rc = EBUSY;
                    }

                } else {
                    LOG_ERROR("Volume \"%s\" is not an EVMS volume.  Only EVMS volumes can have their names changed.\n", vol->name);
                    rc = EINVAL;
                }

            } else {
                /* The handle is not for a logical volume. */
                LOG_ERROR("Handle %d is not for a volume.\n", volume_handle);
                rc = EINVAL;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * find_free_evms_minor compares the current minor number in question with the
 * minor number of a given volume.  If the minor number matches the minor number
 * of the volume, the minor number is decremented for comparison with the next
 * volume in the list.  Since the VolumeList is sorted, we know we have found a
 * free minor number when the current minor number doesn't match the minor
 * number of the volume.
 */
static int find_free_evms_minor(ADDRESS Object,
                                TAG     ObjectTag,
                                uint    ObjectSize,
                                ADDRESS ObjectHandle,
                                ADDRESS Parameters) {

    int rc = 0;
    uint * pMinor = (uint *) Parameters;

    LOG_PROC_ENTRY();

    if (ObjectTag == VOLUME_TAG) {
        logical_volume_t * volume = (logical_volume_t *) Object;

        if (volume->minor_number != *pMinor) {
            /*
             * Tell the ForEachItem processor that our search is over.
             * We have found a free minor number.
             */
            rc = DLIST_SEARCH_COMPLETE;

        } else {
            /*
             * The minor number is not free.  Decrement the minor number (we
             * search for EVMS minor numbers from the top, down) for the
             * comparison with the next volume structure in the list.
             */
            (*pMinor)--;

            if (*pMinor == 0) {
                /*
                 * Safety check.  If we are down to minor number 0, then the
                 * search is over.
                 */
                rc = DLIST_SEARCH_COMPLETE;
            }
        }

    } else {
        rc = DLIST_ITEM_TAG_WRONG;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Get a minor number for an EVMS volume.  Search for a free minor number
 * starting with the highest minor number and moving down.
 */
int get_evms_minor_number(uint * minor) {
    int rc = 0;

    /* Start searching at the highest minor number. */
    uint new_minor = MAX_EVMS_VOLUMES - 1;

    LOG_PROC_ENTRY();

    rc = ForEachItem(VolumeList,
                     find_free_evms_minor,
                     &new_minor,
                     FALSE);  /* Go backwards in the list; from the top, down.*/

    if ((rc == DLIST_SUCCESS) && (new_minor != 0)) {
        *minor = new_minor;

    } else {
        /* We could not find a unique minor number. */
        rc = ENOENT;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * find_free_compatibility_minor compares the current minor number in question
 * with the minor number of a given volume.  If the minor number matches the
 * minor number of the volume, the minor number is decremented for comparison
 * with the next volume in the list.  Since the VolumeList is sorted, we know we
 * have found a free minor number when the current minor number doesn't match
 * the minor number of the volume.
 */
static int find_free_compatibility_minor(ADDRESS Object,
                                         TAG     ObjectTag,
                                         uint    ObjectSize,
                                         ADDRESS ObjectHandle,
                                         ADDRESS Parameters) {

    int rc = 0;
    uint * pMinor = (uint *) Parameters;

    LOG_PROC_ENTRY();

    if (ObjectTag == VOLUME_TAG) {
        logical_volume_t * volume = (logical_volume_t *) Object;

        if (volume->minor_number != *pMinor) {
            /*
             * Tell the ForEachItem processor that our search is over.
             * We have found a free minor number.
             */
            rc = DLIST_SEARCH_COMPLETE;

        } else {
            /*
             * The minor number is not free.  Increment the minor number (we
             * search for compatibility minor numbers from the bottom, up) for
             * the comparison with the next volume structure in the list.
             */
            (*pMinor)++;

            if (*pMinor == MAX_EVMS_VOLUMES) {
                /*
                 * Safety check.  If we are down to minor number 0, then the
                 * search is over.
                 */
                rc = DLIST_SEARCH_COMPLETE;
            }
        }

    } else {
        rc = DLIST_ITEM_TAG_WRONG;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Get a minor number for a compatibility volume.  Search for a free minor
 * number starting with the lowest minor number and moving up.
 */
int get_compatibility_minor_number(uint * minor) {
    int rc = 0;

    /* Start searching at the lowest minor number. */
    uint new_minor = 1;

    LOG_PROC_ENTRY();

    rc = ForEachItem(VolumeList,
                     find_free_compatibility_minor,
                     &new_minor,
                     TRUE);

    if ((rc == DLIST_SUCCESS) && (new_minor != 0)) {
        *minor = new_minor;

    } else {
        /* We could not find a unique minor number. */
        rc = ENOENT;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Check to see if a volume's serial number matches a given serial number.
 * This function has its parameters structured so that it can be called by the
 * DLIST ForEachItem() processor.  The "parameters" parameter is a pointer to
 * the given serial number.
 * Return 0 if the serial numbers don't match, error if they do match.
 */
int check_volume_serial(ADDRESS object,
                        TAG     object_tag,
                        uint    object_size,
                        ADDRESS object_handle,
                        ADDRESS parameters) {

    int rc = 0;
    logical_volume_t * volume = (logical_volume_t *) object;
    u_int32_t * pSerial = (u_int32_t *) parameters;

    LOG_PROC_ENTRY();

    if (object_tag == VOLUME_TAG) {
        if (volume->serial_number == *pSerial) {
            rc = EINVAL;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Generate a 64-bit volume serial number.  The random number generator is used
 * to make up a serial number.  Zero is not a valid serial number.
 */
u_int64_t generate_volume_serial() {

    u_int64_t serial = 0;
    u_int32_t * p_int32 = (u_int32_t *) &serial;

    LOG_PROC_ENTRY();

    while (serial == 0) {
        p_int32[0] = rand();
        p_int32[1] = rand();
    }

    LOG_DEBUG("Recommended serial number is %016llx.\n", serial);

    LOG_PROC_EXIT_VOID();
    return serial;
}


/*
 * Add a volume feature header to an object.
 */
static int add_volume_feature_header_to_object(storage_object_t * object) {

    int rc = 0;
    evms_feature_header_t * fh;

    LOG_PROC_ENTRY();

    fh = calloc(1, sizeof(evms_feature_header_t));
    if (fh != NULL) {
        fh->feature_id = EVMS_VOLUME_FEATURE_ID;
        fh->sequence_number = 1;

        object->feature_header = fh;

    } else {
        LOG_CRITICAL("Error allocating memory for a feature header for object %s.\n", object->name);
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * This routine takes the input string, dups it, and strips leading and
 * trailing spaces from the dup'd string. If the resultant string is
 * empty we return EINVAL. If we are unable to alloc the memory for the
 * duplicate string, we return ENOMEM.
 */
static int strip_evms_volume_name(char *name, char * * stripped_name) {
    int rc = 0;

    if (*name != '\0') {
        char *wrk_str;
        char *leading;
        char *trailer;

        wrk_str = leading = strdup (name);

        if (leading) {

            /*
             * Remove leading and trailing space. If at the end, the
             * string is not empty dup the trimmed string and free
             * the work string.
             */
            while (isspace(*leading)) leading++;

            trailer = leading + strlen(leading) - 1;

            while (isspace(*trailer)) trailer--;

            *(trailer+1) = '\0';

            if (*leading != '\0') {
                *stripped_name = strdup(leading);
            } else {
                LOG_ERROR("Resultant string was just white space.\n");
                rc = EINVAL;
            }

            free(wrk_str);
        } else {
            LOG_ERROR("Unable to allocate memory to duplicate volume name string.\n");
            rc = ENOMEM;
        }
    } else {
        LOG_ERROR("An empty volume name string is not acceptable.\n");
        rc = EINVAL;
    }

    return rc;
}


/*
 * make_evms_volume_name checks to make sure the volume name has the
 * EVMS_DEV_NODE_PATH prefix.  If not, it builds a volume name with the
 * EVMS_DEV_NODE_PATH prefix prepended to the name given and changes the
 * original name pointer to point to the buffer.  The buffer must be at least
 * EVMS_VOLUME_NAME_SIZE+1 in size.
 */
static int make_evms_volume_name(char * * name, char * buffer) {

    int   rc = 0;
    char *stripped_name;

    LOG_PROC_ENTRY();

    rc = strip_evms_volume_name (*name, &stripped_name);

    if (rc == 0) {
        /*
         * Prepend the EVMS_DEV_NODE_PATH to the name if the caller didn't do it
         * already.
         */
        if (strncmp(stripped_name, EVMS_DEV_NODE_PATH, strlen(EVMS_DEV_NODE_PATH)) != 0) {
            if (strlen(EVMS_DEV_NODE_PATH) + strlen(stripped_name) <= EVMS_VOLUME_NAME_SIZE) {
                strcpy(buffer, EVMS_DEV_NODE_PATH);
                strcat(buffer, stripped_name);
                *name = buffer;

            } else {
                LOG_ERROR("Volume name \"%s%s\" is too long.  It must be %d bytes or fewer.\n", EVMS_DEV_NODE_PATH, stripped_name, EVMS_VOLUME_NAME_SIZE);
                rc = EOVERFLOW;
            }

        } else {
            int name_len;

            /*
             * Check to see that the name isn't just the /dev/evms/ prefix.
             * If not, copy the entire stripped name to the supplied buffer.
             */

            name_len = strlen(stripped_name);

            if (strncmp(stripped_name, EVMS_DEV_NODE_PATH, name_len) == 0) {
                LOG_ERROR("Volume name is made of just the %s prefix.\n", EVMS_DEV_NODE_PATH);
                rc = EINVAL;
            } else {
                if (name_len <= EVMS_VOLUME_NAME_SIZE) {
                    strcpy(buffer, stripped_name);
                    *name = buffer;
                } else {
                    LOG_ERROR("Volume name \"%s\" is too long.  It must be %d bytes or fewer.\n", stripped_name, EVMS_VOLUME_NAME_SIZE);
                    rc = EOVERFLOW;
                }
            }

        }

        free(stripped_name);
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Make an EVMS volume for an object.
 */
int evms_create_volume(object_handle_t object_handle,
                       char          * name) {
    int rc = 0;
    void * object;
    object_type_t type;
    char new_name[EVMS_VOLUME_NAME_SIZE+1];

    LOG_PROC_ENTRY();

    rc = check_engine_write_access();

    if (rc == 0) {

        rc = translate_handle(object_handle,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {

            switch (type) {
                case DISK:
                case SEGMENT:
                case REGION:
                case EVMS_OBJECT:
                    {
                        storage_object_t * obj = (storage_object_t *) object;

                        /* Object must be a data object */
                        if (obj->data_type == DATA_TYPE) {

                            /* Object must not be corrupt. */
                            if (!(obj->flags & SOFLAG_CORRUPT)) {

                                /*
                                 * Only top level objects can be made into
                                 * volumes.
                                 */
                                if (is_top_object(obj)) {

                                    /*
                                     * Ask the object if it can handle being
                                     * made into a volume.  Right now we know of
                                     * no reason why any plug-in should fail
                                     * this call.
                                     */
                                    rc = obj->plugin->functions.plugin->can_set_volume(obj, TRUE);

                                    if (rc == 0) {

                                        LOG_DEBUG("Request to make object %s into volume \"%s\".\n", obj->name, name);

                                        rc = make_evms_volume_name(&name, new_name);

                                        if (rc == 0) {
                                            /*
                                             * Make sure this volume name does
                                             * not already exist.
                                             */

                                            rc = engine_validate_name(name);

                                            if (rc == 0) {
                                                /*
                                                 * This is not an EVMS object.
                                                 * It must have a feature header
                                                 * applied to it in order to
                                                 * become an EVMS volume.
                                                 */
                                                if (type != EVMS_OBJECT) {
                                                    rc = add_volume_feature_header_to_object(obj);

                                                    if (rc == 0) {
                                                        obj->flags |= SOFLAG_FEATURE_HEADER_DIRTY;
                                                    } else {
                                                        LOG_WARNING("Failed to add a feature header to object %s/\n", ((storage_object_t *) object)->name);
                                                    }
                                                }

                                                if (rc == 0) {
                                                    u_int32_t new_minor;
                                                    rc = get_evms_minor_number(&new_minor);

                                                    if (rc == 0) {
                                                        u_int64_t new_serial;

                                                        /*
                                                         * Get a serial number for the
                                                         * volume.
                                                         */
                                                        do {
                                                            new_serial = generate_volume_serial();
                                                            rc = ForEachItem(VolumeList,
                                                                             check_volume_serial,
                                                                             &new_serial,
                                                                             TRUE);
                                                        } while (rc != 0);

                                                        rc = make_volume(obj,
                                                                         name,
                                                                         new_minor,
                                                                         VOLFLAG_NEW | VOLFLAG_DIRTY | VOLFLAG_NEEDS_DEV_NODE,
                                                                         new_serial);

                                                    } else {
                                                        LOG_WARNING("Unable to allocate a minor number for the new volume.\n");
                                                    }
                                                }
                                            }
                                        }

                                    } else {
                                        LOG_ERROR("Object %s refuses to be made into a volume.\n", obj->name);
                                    }

                                } else {
                                    LOG_ERROR("Object %s cannot be made into a volume because it is not a top level object.\n", obj->name);
                                    rc = EINVAL;
                                }

                            } else {
                                /* Object is corrupt. */
                                LOG_ERROR("Object %s is corrupt.\n", obj->name);
                                rc = EINVAL;
                            }

                        } else {
                            /* Object is not a data object. */
                            LOG_ERROR("Object %s is not a data object.\n", obj->name);
                            rc = EINVAL;
                        }
                    }
                    break;

                default:
                    LOG_ERROR("Handle %d is not for a storage object.\n", object_handle);
                    rc = EINVAL;
            }
        }

        if (rc == 0) {
            changes_pending = TRUE;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Change the name of an EVMS volume.
 */
int evms_set_volume_name(engine_handle_t volume_handle,
                         char          * name) {
    int rc = 0;
    void * object;
    object_type_t type;
    char new_name[EVMS_VOLUME_NAME_SIZE+1];

    LOG_PROC_ENTRY();

    rc = check_engine_write_access();

    if (rc == 0) {
        rc = translate_handle(volume_handle,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {
            if (type == VOLUME) {
                logical_volume_t * vol = (logical_volume_t *) object;

                /* Only EVMS volumes can have their name changed. */
                if (!(vol->flags & VOLFLAG_COMPATIBILITY)) {

                    /* Make sure the volume is not mounted. */
                    if (!is_volume_mounted(vol)) {
                        /*
                         * Ask the object below the volume if it can have the
                         * volume changed.
                         */
                        storage_object_t * vol_obj = (storage_object_t *) vol->object;

                        rc = vol_obj->plugin->functions.plugin->can_set_volume(vol_obj, TRUE);

                        if (rc == 0) {
                            rc = make_evms_volume_name(&name, new_name);

                            if (rc == 0) {
                                /*
                                 * Make sure this volume name does not already
                                 * exist.
                                 */
                                rc = engine_validate_name(name);

                                if (rc == 0) {
                                    ADDRESS trash;

                                    /* Remove the volume from the VolumeList. */
                                    DeleteObject(VolumeList, vol);

                                    /*
                                     * If the volume is not new, put the old
                                     * volume on the HardVolumeDeleteList so
                                     * that the kernel will remove it.
                                     */
                                    if (!(vol->flags & VOLFLAG_NEW)) {
                                        rc = InsertObject(HardVolumeDeleteList,
                                                          sizeof(logical_volume_t),
                                                          vol,
                                                          VOLUME_TAG,
                                                          NULL,
                                                          AppendToList,
                                                          FALSE,
                                                          &trash);
                                    }

                                    if (rc == 0) {
                                        /*
                                         * Make sure the volume is not on the
                                         * SoftVolumeDeleteList from some prior
                                         * operation.  It can't be both hard
                                         * deleted and soft deleted.
                                         */
                                        DeleteObject(SoftVolumeDeleteList, vol);

                                        /*
                                         * Make a new volume based on the old
                                         * one.
                                         */
                                        rc = make_volume(vol_obj,
                                                         name,
                                                         vol->minor_number,
                                                         vol->flags | VOLFLAG_NEW | VOLFLAG_DIRTY,
                                                         vol->serial_number);

                                        if (rc == 0) {

                                            /*
                                             * Copy fields from the old volume
                                             * structure to the new one.
                                             */
                                            vol_obj->volume->file_system_manager = vol->file_system_manager;
                                            vol_obj->volume->original_fsim       = vol->original_fsim;
                                            vol_obj->volume->mount_point         = vol->mount_point;
                                            vol_obj->volume->original_vol_size   = vol->original_vol_size;
                                            vol_obj->volume->fs_size             = vol->fs_size;
                                            vol_obj->volume->min_fs_size         = vol->min_fs_size;
                                            vol_obj->volume->max_fs_size         = vol->max_fs_size;
                                            vol_obj->volume->max_vol_size        = vol->max_vol_size;
                                            vol_obj->volume->associated_volume   = vol->associated_volume;
                                            vol_obj->volume->mkfs_options        = vol->mkfs_options;
                                            vol_obj->volume->fsck_options        = vol->fsck_options;
                                            vol_obj->volume->defrag_options      = vol->defrag_options;
                                            vol_obj->volume->private_data        = vol->private_data;

                                            /*
                                             * Check if the new volume needs a
                                             * dev node.
                                             */
                                            rc = hasa_dev_node(vol->object->volume->name,
                                                               vol->object->volume->minor_number);
                                            if ((rc == ENOENT) ||
                                                (rc == EEXIST)) {
                                                vol_obj->volume->flags |= VOLFLAG_NEEDS_DEV_NODE;
                                            }

                                            /* Reset the error code. */
                                            rc = 0;

                                            /*
                                             * Free up the old volume's app
                                             * handle.
                                             */
                                            destroy_handle(vol->app_handle);

                                            /* We have changes pending. */
                                            changes_pending = TRUE;

                                        } else {
                                            LOG_WARNING("Error code %d when making volume \"%s\".\n", rc, name);
                                        }

                                    } else {
                                        LOG_WARNING("Error code %d when putting volume %s on the HardVolumeDeleteList.\n", rc, vol->name);
                                    }

                                    if (rc != 0) {
                                        /*
                                         * Something went wrong either in
                                         * putting the old volume on the
                                         * HardVolumeDeleteList or in making the
                                         * new volume.  Put the old volume back
                                         * on the VolumeList and return the
                                         * error.
                                         */
                                        insert_new_volume_into_volume_list(vol);
                                    }
                                }
                            }

                        } else {
                            LOG_ERROR("Object %s will not let the name of volume %s be changed.\n", vol_obj->name, vol->name);
                        }

                    } else {
                        LOG_ERROR("Volume \"%s\" is currently mounted and cannot have its name changed.\n", vol->name);
                        rc = EBUSY;
                    }

                } else {
                    LOG_ERROR("Volume \"%s\" is not an EVMS volume.  Only EVMS volumes can have their names changed.\n", vol->name);
                    rc = EINVAL;
                }

            } else {
                /* The handle is not for a logical volume. */
                LOG_ERROR("Handle %d is not for a volume.\n", volume_handle);
                rc = EINVAL;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Make a compatibility volume for an object.
 */
int evms_create_compatibility_volume(object_handle_t object_handle) {
    int rc = 0;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_write_access();

    if (rc == 0) {
        rc = translate_handle(object_handle,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {
            switch (type) {
                case DISK:
                case SEGMENT:
                case REGION:
                    {
                        storage_object_t * obj = (storage_object_t *) object;
                        /*
                         * Object must be a data object, i.e., not meta data,
                         * not free space
                         */
                        if (obj->data_type == DATA_TYPE) {

                            /* Object must not be corrupt. */
                            if (!(obj->flags & SOFLAG_CORRUPT)) {

                                /* Object must be a top object. */
                                if (is_top_object(obj)) {

                                    /*
                                     * Ask the object if it can handle being
                                     * made into a volume.  Right now we know of
                                     * no reason why any plug-in should fail
                                     * this call.
                                     */
                                    rc = obj->plugin->functions.plugin->can_set_volume(obj, TRUE);

                                    if (rc == 0) {
                                        char * pVolName = obj->name;

                                        /*
                                         * Build the compatibility volume name
                                         * by prepending the EVMS dev node
                                         * prefix to the object name if it isn't
                                         * already there.
                                         */
                                        if (strncmp(pVolName, EVMS_DEV_NODE_PATH, strlen(EVMS_DEV_NODE_PATH)) != 0) {
                                            pVolName = malloc(strlen(pVolName) + strlen(EVMS_DEV_NODE_PATH) + 1);
                                            if (pVolName != NULL) {
                                                strcpy(pVolName, EVMS_DEV_NODE_PATH);
                                                strcat(pVolName, obj->name);
                                            } else {
                                                LOG_CRITICAL("Could not get memory for building a volume name for object %s.\n", obj->name);
                                                rc = ENOMEM;
                                            }
                                        }

                                        if (rc == 0) {
                                            /*
                                             * Reserve a minor number for the
                                             * volume.  This may not be the
                                             * volume's minor number when the
                                             * kernel discovers it.  This just
                                             * helps the Engine avoid collisions
                                             * with minor numbers.
                                             */
                                            u_int32_t new_minor;

                                            rc = get_compatibility_minor_number(&new_minor);

                                            if (rc == 0) {
                                                /*
                                                 * Kill any stop data sectors
                                                 * that might be on the object.
                                                 */
                                                rc = obj->plugin->functions.plugin->add_sectors_to_kill_list(obj, obj->size - (FEATURE_HEADER_SECTORS * 2), FEATURE_HEADER_SECTORS * 2);

                                                if (rc == 0) {
                                                    rc = make_volume(obj,
                                                                     pVolName,
                                                                     new_minor,
                                                                     VOLFLAG_NEW | VOLFLAG_DIRTY | VOLFLAG_COMPATIBILITY | VOLFLAG_NEEDS_DEV_NODE,
                                                                     0);

                                                    if (rc == 0) {
                                                        /*
                                                         * Remove any feature
                                                         * header that might be
                                                         * hanging off the
                                                         * object (which would
                                                         * make us think this
                                                         * should be an EVMS
                                                         * volume).
                                                         */
                                                        if (obj->feature_header != NULL) {
                                                            free(obj->feature_header);
                                                            obj->feature_header = NULL;
                                                        }

                                                    } else {
                                                        LOG_WARNING("Could not make volume %s for object %s.  Return code was %d.\n", pVolName, obj->name, rc);
                                                    }

                                                } else {
                                                    LOG_DEBUG("Wipe out stop data.  Return code from add_sectors_to_kill_list() was %d.\n", rc);
                                                }

                                            } else {
                                                LOG_WARNING("Unable to allocate a minor number for the new volume.\n");
                                            }
                                        }

                                    } else {
                                        LOG_ERROR("Object %s refuses to be made into a volume.\n", obj->name);
                                    }

                                } else {
                                    LOG_ERROR("Object %s cannot be made into a volume because it is not a top level object.\n", obj->name);
                                    rc = EINVAL;
                                }

                            } else {
                                LOG_ERROR("Object %s is corrupt.\n", obj->name);
                            }

                        } else {
                            LOG_ERROR("Object %s cannot be made into a volume because it is not a data object.\n", obj->name);
                            rc = EINVAL;
                        }
                    }
                    break;

                default:
                    LOG_ERROR("Handle %d is not for a storage object.\n", object_handle);
                    rc = EINVAL;
            }
        }

        if (rc == 0) {
            changes_pending = TRUE;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Can this file system be installed on this volume?
 */
int evms_can_mkfs(object_handle_t volume_handle,
                  plugin_handle_t fsim_handle) {

    int rc = 0;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {
        rc = translate_handle(volume_handle,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {
            if (type == VOLUME) {
                logical_volume_t * vol = (logical_volume_t *) object;

                if (!(vol->flags & VOLFLAG_READ_ONLY)) {
                    if (vol->file_system_manager == NULL) {
                        rc = translate_handle(fsim_handle,
                                              &object,
                                              &type);

                        if (rc == HANDLE_MANAGER_NO_ERROR) {
                            if (type == PLUGIN) {

                                plugin_record_t * fsim = (plugin_record_t *) object;

                                if (GetPluginType(fsim->id) == EVMS_FILESYSTEM_INTERFACE_MODULE) {

                                    rc = fsim->functions.fsim->can_mkfs(vol);

                                } else {
                                    LOG_ERROR("Handle %d is for a plug-in %s which is not a File System Interface Module.\n", fsim_handle, fsim->short_name);
                                    rc = EINVAL;
                                }

                            } else {
                                LOG_ERROR("Handle %d is not for a plug-in.\n", fsim_handle);
                                rc = EINVAL;
                            }
                        }

                    } else {
                        LOG_ERROR("Volume \"%s\" already has file system %s installed on it.  The file system must be removed (unmkfs) before a new file system can be installed.\n", vol->name, vol->file_system_manager->short_name);
                        rc = EINVAL;
                    }

                } else {
                    LOG_ERROR("Volume \"%s\" is read only.\n", vol->name);
                    rc = EINVAL;
                }

            } else {
                LOG_ERROR("Handle %d is not for a volume.\n", volume_handle);
                rc = EINVAL;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Can the file system be removed from the volume.
 */
int evms_can_unmkfs(object_handle_t volume_handle) {

    int rc = 0;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {
        rc = translate_handle(volume_handle,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {
            if (type == VOLUME) {

                logical_volume_t * vol = (logical_volume_t *) object;
                plugin_record_t * fsim = vol->file_system_manager;

                if (!(vol->flags & VOLFLAG_READ_ONLY)) {
                    if (fsim != NULL) {
                        rc = fsim->functions.fsim->can_unmkfs(vol);

                    } else {
                        LOG_ERROR("Volume \"%s\" does not have a File System Interface Module associated with it.\n", vol->name);
                        rc = ENOSYS;
                    }

                } else {
                    LOG_ERROR("Volume \"%s\" is read only.\n", vol->name);
                    rc = EINVAL;
                }

            } else {
                LOG_ERROR("Handle %d is not for a volume.\n", volume_handle);
                rc = EINVAL;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Can fsck be run on the volume?
 */
int evms_can_fsck(object_handle_t volume_handle) {

    int rc = 0;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {
        rc = translate_handle(volume_handle,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {
            if (type == VOLUME) {

                logical_volume_t * vol = (logical_volume_t *) object;
                plugin_record_t * fsim = vol->file_system_manager;

                if (fsim != NULL) {
                    rc = fsim->functions.fsim->can_fsck(vol);

                } else {
                    LOG_ERROR("Volume \"%s\" does not have a File System Interface Module associated with it.\n", vol->name);
                    rc = ENOSYS;
                }

            } else {
                LOG_ERROR("Handle %d is not for a volume.\n", volume_handle);
                rc = EINVAL;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Can the volume be defragmented?
 */
int evms_can_defrag(object_handle_t  volume_handle) {

    int rc = 0;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {
        rc = translate_handle(volume_handle,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {
            if (type == VOLUME) {

                logical_volume_t * vol = (logical_volume_t *) object;
                plugin_record_t * fsim = vol->file_system_manager;

                if (!(vol->flags & VOLFLAG_READ_ONLY)) {
                    if (fsim != NULL) {
                        rc = fsim->functions.fsim->can_defrag(vol);

                    } else {
                        LOG_ERROR("Volume \"%s\" does not have a File System Interface Module associated with it.\n", vol->name);
                        rc = ENOSYS;
                    }

                } else {
                    LOG_ERROR("Volume \"%s\" is read only.\n", vol->name);
                    rc = EINVAL;
                }

            } else {
                LOG_ERROR("Handle %d is not for a volume.\n", volume_handle);
                rc = EINVAL;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/* Free an option array and all its associated memory. */
void free_option_array(option_array_t * options) {

    int i;

    for (i = 0; i < options->count; i++) {
        if (options->option[i].name != NULL) {
            free(options->option[i].name);
        }

        if (options->option[i].type == EVMS_Type_String) {
            if (options->option[i].value.s != NULL) {
                free(options->option[i].value.s);
            }
        }
    }

    free(options);
}


/*
 * A utility function for doing a deep copy of an option array.
 */
static int copy_option_array(option_array_t * source, option_array_t * * target) {

    int rc = 0;
    option_array_t * copy;

    LOG_PROC_ENTRY();

    *target = NULL;

    if (source != NULL) {
        int size;

        if (source->count > 1) {
            size = sizeof(option_array_t) + source->count * sizeof(key_value_pair_t);
        } else {
            size = sizeof(option_array_t);
        }

        copy = (option_array_t *) calloc(1, size);

        if (copy != NULL) {
            int i;

            copy->count = source->count;

            for (i = 0; (rc == 0) && (i < copy->count); i++) {

                /* Copy the fixed fields */
                copy->option[i].number          = source->option[i].number;
                copy->option[i].is_number_based = source->option[i].is_number_based;
                copy->option[i].type            = source->option[i].type;

                /* If the option has a name string, copy it. */
                if (source->option[i].name != NULL) {
                    copy->option[i].name = strdup(source->option[i].name);

                    if (copy->option[i].name == NULL) {
                        LOG_CRITICAL("Error allocating memory for a key_value_pair name.\n");
                        rc = ENOMEM;
                    }
                }

                if (rc == 0) {
                    /* If the value is a string, make a copy of it. */
                    if (copy->option[i].type == EVMS_Type_String) {
                        if (copy->option[i].value.s != NULL) {

                            copy->option[i].value.s = strdup(source->option[i].value.s);

                            if (copy->option[i].value.s == NULL) {
                                LOG_CRITICAL("Error allocating memory for a key_value_pair string value.\n");
                                rc = ENOMEM;
                            }
                        }

                    } else {
                        /* It's not a string.  Just copy the value. */
                        copy->option[i].value = source->option[i].value;
                    }
                }
            }

            if (rc == 0) {
                *target = copy;

            } else {
                /* Something failed.  Free up all the allocated memory. */
                free_option_array(copy);
            }

        } else {
            LOG_CRITICAL("Error allocating memory for a copy of an option array.\n");
            rc = ENOMEM;
        }

    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Install a file system on a volume.
 */
int evms_mkfs(object_handle_t  volume_handle,
              plugin_handle_t  fsim_handle,
              option_array_t * options) {

    int rc = 0;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_write_access();

    if (rc == 0) {
        rc = translate_handle(volume_handle,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {
            if (type == VOLUME) {
                logical_volume_t * vol = (logical_volume_t *) object;

                if (!(vol->flags & VOLFLAG_READ_ONLY)) {
                    if (vol->file_system_manager == NULL) {
                        rc = translate_handle(fsim_handle,
                                              &object,
                                              &type);

                        if (rc == HANDLE_MANAGER_NO_ERROR) {
                            if (type == PLUGIN) {

                                plugin_record_t * fsim = (plugin_record_t *) object;

                                if (GetPluginType(fsim->id) == EVMS_FILESYSTEM_INTERFACE_MODULE) {

                                    /* Check if FSIM can handle mkfs. */
                                    rc = fsim->functions.fsim->can_mkfs(vol);
                                    if (rc == 0) {

                                        rc = copy_option_array(options, &vol->mkfs_options);

                                        if (rc == 0) {
                                            vol->file_system_manager = fsim;
                                            vol->flags |= VOLFLAG_MKFS;
                                        }

                                    } else {
                                        LOG_ERROR("The FSIM %d cannot run mkfs on volume %s.  Error code is %d.\n", fsim->short_name, vol->name, rc);
                                    }

                                } else {
                                    LOG_ERROR("Handle %d is for a plug-in %s which is not a File System Interface Module.\n", fsim_handle, fsim->short_name);
                                    rc = EINVAL;
                                }

                            } else {
                                LOG_ERROR("Handle %d is not for a plug-in.\n", fsim_handle);
                                rc = EINVAL;
                            }
                        }

                    } else {
                        LOG_ERROR("Volume \"%s\" already has file system %s installed on it.  The file system must be removed (unmkfs) before a new file system can be installed.\n", vol->name, vol->file_system_manager->short_name);
                        rc = EINVAL;
                    }

                } else {
                    LOG_ERROR("Volume \"%s\" is read only.\n", vol->name);
                    rc = EINVAL;
                }

            } else {
                LOG_ERROR("Handle %d is not for a volume.\n", volume_handle);
                rc = EINVAL;
            }
        }
    }

    if (rc == 0) {
        changes_pending = TRUE;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Remove a file system from a volume.
 */
int evms_unmkfs(object_handle_t volume_handle) {

    int rc = 0;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_write_access();

    if (rc == 0) {
        rc = translate_handle(volume_handle,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {
            if (type == VOLUME) {

                logical_volume_t * vol = (logical_volume_t *) object;
                plugin_record_t * fsim = vol->file_system_manager;

                if (!(vol->flags & VOLFLAG_READ_ONLY)) {
                    if (fsim != NULL) {

                        /* Check if the FSIM can unmkfs. */
                        rc = fsim->functions.fsim->can_unmkfs(vol);

                        if (rc == 0) {
                            /* Mark that this volume has no FSIM. */
                            vol->file_system_manager = NULL;

                            /*
                             * Since it has no FSIM any pending mkfs can't be
                             * run.
                             */
                            vol->flags &= ~VOLFLAG_MKFS;

                            /*
                             * If during discovery an FSIM claimed the volume,
                             * turn on the VOLFLAG_UNMKFS flag so that unmkfs
                             * will be run at commit time.
                             */
                            if (vol->original_fsim != NULL) {
                                vol->flags |= VOLFLAG_UNMKFS;
                            }

                        } else {
                            LOG_ERROR("The %s FSIM cannot run unmkfs on volume %s.  Error code is %d.\n", fsim->short_name, vol->name, rc);
                        }

                    } else {
                        LOG_ERROR("Volume \"%s\" does not have a File System Interface Module associated with it.\n", vol->name);
                        rc = ENOSYS;
                    }

                } else {
                    LOG_ERROR("Volume \"%s\" is read only.\n", vol->name);
                    rc = EINVAL;
                }

            } else {
                LOG_ERROR("Handle %d is not for a volume.\n", volume_handle);
                rc = EINVAL;
            }
        }
    }

    if (rc == 0) {
        changes_pending = TRUE;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Run fsck on a volume.
 */
int evms_fsck(object_handle_t  volume_handle,
              option_array_t * options) {

    int rc = 0;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_write_access();

    if (rc == 0) {
        rc = translate_handle(volume_handle,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {
            if (type == VOLUME) {

                logical_volume_t * vol = (logical_volume_t *) object;
                plugin_record_t * fsim = vol->file_system_manager;

                if (fsim != NULL) {

                    /* Check if FSIM can handle fsck. */
                    rc = fsim->functions.fsim->can_fsck(vol);
                    if (rc == 0) {
                        rc = copy_option_array(options, &vol->fsck_options);

                        if (rc == 0) {
                            vol->flags |= VOLFLAG_FSCK;
                        }

                    } else {
                        LOG_ERROR("The FSIM %d cannot run fsck on volume %s.  Error code is %d.\n", fsim->short_name, vol->name, rc);
                    }

                } else {
                    LOG_ERROR("Volume \"%s\" does not have a File System Interface Module associated with it.\n", vol->name);
                    rc = ENOSYS;
                }

            } else {
                LOG_ERROR("Handle %d is not for a volume.\n", volume_handle);
                rc = EINVAL;
            }
        }
    }

    if (rc == 0) {
        changes_pending = TRUE;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Defragment a volume.
 */
int evms_defrag(object_handle_t  volume_handle,
                option_array_t * options) {

    int rc = 0;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_write_access();

    if (rc == 0) {
        rc = translate_handle(volume_handle,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {
            if (type == VOLUME) {

                logical_volume_t * vol = (logical_volume_t *) object;
                plugin_record_t * fsim = vol->file_system_manager;

                if (!(vol->flags & VOLFLAG_READ_ONLY)) {
                    if (fsim != NULL) {

                        /* Check if FSIM can handle defrag. */
                        rc = fsim->functions.fsim->can_defrag(vol);
                        if (rc == 0) {
                            rc = copy_option_array(options, &vol->defrag_options);

                            if (rc == 0) {
                                vol->flags |= VOLFLAG_DEFRAG;
                            }

                        } else {
                            LOG_ERROR("The FSIM %d cannot run defrag on volume %s.  Error code is %d.\n", fsim->short_name, vol->name, rc);
                        }

                    } else {
                        LOG_ERROR("Volume \"%s\" does not have a File System Interface Module associated with it.\n", vol->name);
                        rc = ENOSYS;
                    }

                } else {
                    LOG_ERROR("Volume \"%s\" is read only.\n", vol->name);
                    rc = EINVAL;
                }

            } else {
                LOG_ERROR("Handle %d is not for a volume.\n", volume_handle);
                rc = EINVAL;
            }
        }
    }

    if (rc == 0) {
        changes_pending = TRUE;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * evms_get_volume_list returns a pointer to a handle_array_t with handles for
 * logical volumes, optionally filtering on the FSIM that manages the volume.
 * If the object handle for the FSIM is 0, handles for all of the
 * logical volumes will be returned.
 */
int evms_get_volume_list(object_handle_t    fsim_handle,
                         handle_array_t * * volume_handle_list) {
    int rc = 0;
    ADDRESS object = NULL;
    object_type_t type;
    plugin_record_t * fsim = NULL;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {
        if (fsim_handle != 0) {
            /*
             * Translate the handle for the FSIM to make sure it is valid and to
             * get the plugin_record_t for the FSIM.
             */
            rc = translate_handle(fsim_handle,
                                  &object,
                                  &type);

            if (rc == HANDLE_MANAGER_NO_ERROR) {
                if (type == PLUGIN) {
                    plugin_record_t * fsim = (plugin_record_t *) object;
                    if (GetPluginType(fsim->id) != EVMS_FILESYSTEM_INTERFACE_MODULE) {
                        rc = EINVAL;
                    }

                } else {
                    rc = EINVAL;
                }
            }
        }

        if (rc == 0) {
            dlist_t volume_list;

            /*
             * Call the internal version of GetVolumeList.  "FSIM" will be NULL
             * if the caller did not specify a FSIM, else it will be a pointer
             * to the FSIM's plugin_record_t.
             */
            rc = engine_get_volume_list(fsim, &volume_list);

            if (rc == 0) {
                rc = make_user_handle_array(volume_list, volume_handle_list);

                /*
                 * We are finished with the list that was returned by
                 * engine_get_volume_list.
                 */
                DestroyList(&volume_list, FALSE);
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}

