/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: checks.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>
#include <linux/evms/evms_user.h>

#include "ptables.h"
#include "defsegmgr.h"
#include "checks.h"
#include "display.h"
#include "segs.h"
#include "seg_geometry.h"




BOOLEAN isa_linux_raid_partition_record(struct partition *p)
{
    if (  SYS_IND(p) == LINUX_RAID_PARTITION )
        return TRUE;
    else
        return FALSE;
}

BOOLEAN isa_data_partition_record(struct partition *p)
{

    if ( (isa_null_partition_record(p) == FALSE) &&
         (isa_ebr_partition_record(p)  == FALSE)) {
        return TRUE;
    }

    return FALSE;
}

BOOLEAN isa_null_partition_record(struct partition *p)
{
    int          i;
    u_int32_t   *uip = (u_int32_t *) p;

    for (i=0; i<4; i++) {
        if (*uip!=0x00) return FALSE;
    }

    return TRUE;
}


BOOLEAN isa_zero_length_partition_record(struct partition *p)
{

    if ( NR_SECTS(p) == 0 )
        return TRUE;
    else
        return FALSE;

}

// test on size field only in case there is garbage in ptable area
BOOLEAN ptable_has_partition_records( LOGICALDISK *ld, Master_Boot_Record *mbr )
{

    if ( (isa_zero_length_partition_record(&mbr->Partition_Table[0])==TRUE) &&
         (isa_zero_length_partition_record(&mbr->Partition_Table[1])==TRUE) &&
         (isa_zero_length_partition_record(&mbr->Partition_Table[2])==TRUE) &&
         (isa_zero_length_partition_record(&mbr->Partition_Table[3])==TRUE) ) {
        return FALSE;
    }

    return TRUE;
}

BOOLEAN ptable_has_data_partition_record( Master_Boot_Record *mbr )
{

    if ( (isa_data_partition_record(&mbr->Partition_Table[0])==TRUE) ||
         (isa_data_partition_record(&mbr->Partition_Table[1])==TRUE) ||
         (isa_data_partition_record(&mbr->Partition_Table[2])==TRUE) ||
         (isa_data_partition_record(&mbr->Partition_Table[3])==TRUE) ) {
        return TRUE;
    }

    return FALSE;
}

BOOLEAN isa_ebr_partition_record(struct partition *p)
{
    if ( (SYS_IND(p) == DOS_EXTENDED_PARTITION) ||
         (SYS_IND(p) == WIN98_EXTENDED_PARTITION) ||
         (SYS_IND(p) == LINUX_EXTENDED_PARTITION) ) {
        return TRUE;
    }

    return FALSE;
}

BOOLEAN isa_mbr_partition_record(struct partition *p)
{
    if (  SYS_IND(p) == MBR_PARTITION  )   // our own internal define used by build_mbr()
        return TRUE;                       // which is found in segs.c
    else
        return FALSE;
}

BOOLEAN isa_linux_swap_partition_record(LOGICALDISK *ld, Partition_Record *p, lba_t extd_partition_lba )
{
    if ( SYS_IND(p) == LINUX_SWAP_PARTITION ) {  // actually Solaris uses the same IND
        return TRUE;
    }
    else {
        return FALSE;
    }
}


// Test if disk has a GUID Partition Table by testing for a protective MBR.
// A protective MBR has a single partition record of type 0xEE.
BOOLEAN has_guid_partition_table( Master_Boot_Record *mbr )
{
    int      i;
    int      pcount=0;
    BOOLEAN  found_gpt_partition=FALSE;

    for (i=0; i<4; i++) {

        // test size field to see if partition record is valid
        if ( DISK_TO_CPU32(NR_SECTS(((Partition_Record *)&mbr->Partition_Table[i]))) != 0 ) {

            // test type field to see if it is a GPT record
            if ( SYS_IND(((Partition_Record *)&mbr->Partition_Table[i]) )==GPT_PARTITION) {
                found_gpt_partition = TRUE;
            }

            // inc partition counter
            ++pcount;
        }
    }

    // if 1 and only 1 partition record and its a GPT record then this disk
    // has a protective MBR and is using a GUID Partition Table.
    if (pcount==1 && found_gpt_partition==TRUE) {
        return TRUE;
    }
    else {
        return FALSE;
    }

}


// Signature is found on AIX IPL Records which are always the 1st sector
// of an AIX physical volume.
#define AIX_IPL_REC_ID  0xc9c2d4c1  // Value is EBCIDIC 'IBMA'


BOOLEAN has_msdos_signature( Master_Boot_Record *mbr )
{

    if  ( ( (*( (unsigned int *) mbr )) != AIX_IPL_REC_ID ) &&
          ( DISK_TO_CPU16(mbr->Signature) == MSDOS_DISKMAGIC )) {
        return TRUE;
    }
    else {
        return FALSE;
    }

}



/*
 *  Validates a partition record by running basic tests against it.
 */
static int  isa_valid_partition_record( LOGICALDISK      *ld,
                                        struct partition *part,
                                        lba_t             ebr_lba,
                                        lba_t             extd_partition_lba,
                                        BOOLEAN           ebr_chaining )
{
    chs_t              chs;
    chs_t              start_chs;
    chs_t              end_chs;

    lba_t              lba;
    lba_t              start_lba;
    lba_t              end_lba;

    sector_count_t     size;

    DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);


    LOGENTRY();

    /*
     *  Parm checks
     */
    if ( disk_pdata == NULL || part == NULL) {
        LOG_ERROR("error, invalid parms passed.\n");
        LOGEXIT();
        return EINVAL;
    }


    /*
     *  Null records cant have any errors ...
     */
    if (  isa_null_partition_record( part ) == TRUE ) {
        LOGEXIT();
        return 0;
    }

    size               =  DISK_TO_CPU32(NR_SECTS(part));

    start_chs.cylinder =  ((START_SECT(part)&0xC0)<<2)  +  START_CYL(part);
    start_chs.head     =  START_HEADS(part);
    start_chs.sector   =  START_SECT(part)&0x3F;

    end_chs.cylinder   =  ((END_SECT(part)&0xC0)<<2)  +  END_CYL(part);
    end_chs.head       =  END_HEADS(part);
    end_chs.sector     =  END_SECT(part)&0x3F;

    if (isa_ebr_partition_record(part)==TRUE) {
        start_lba      =  DISK_TO_CPU32(START_LBA(part)) + extd_partition_lba;
    }
    else {
        start_lba      =  DISK_TO_CPU32(START_LBA(part)) + ebr_lba;
    }
    end_lba            =  (lba_t) start_lba + size - 1;


/* Uncomment for debugging ...
        LOG_DEBUG("         sys_ind : %d\n", SYS_IND(part) );
        LOG_DEBUG(" start cylinders : %08d\n", start_chs.cylinder );
        LOG_DEBUG("           heads : %08d\n", start_chs.head );
        LOG_DEBUG("         sectors : %08d\n\n", start_chs.sector );
        LOG_DEBUG("   end cylinders : %08d\n", end_chs.cylinder );
        LOG_DEBUG("           heads : %08d\n", end_chs.head );
        LOG_DEBUG("         sectors : %08d\n\n", end_chs.sector );
        LOG_DEBUG("       start lba : %08d\n", (u_int32_t) start_lba );
        LOG_DEBUG("         end lba : %08d\n\n", (u_int32_t) end_lba );

        lba = 0;
        CHStoLBA( ld,  &start_chs, &lba );
        LOG_DEBUG("       start lba : %08d (calculated using CHStoLBA)\n", (u_int32_t) lba );

        lba = 0;
        CHStoLBA( ld,  &end_chs, &lba );
        LOG_DEBUG("         end lba : %08d (calculated using CHStoLBA)\n\n", (u_int32_t) lba );
        LOG_DEBUG("       disk size : %08d (sectors)\n", ld->size );
        LOG_DEBUG("       disk name : %s\n", ld->name );
*/


    /*
     *  Test partition record's sector offset field
     */
    if ( start_lba > ld->size ) {
        LOG_ERROR("error, partition starting LBA > size of disk\n");
        LOGEXIT();
        return EOVERFLOW;
    }

    if ( (start_lba+size) > ld->size)  {
        LOG_ERROR("error, partition starting LBA + partition size > size of disk\n");
        LOGEXIT();
        return EOVERFLOW;
    }

    /*
     * If using LBA addressing and we made it past size tests
     * then there is no need to run tests on CHS fields
     */
    if ( disk_pdata->flags & DISK_USES_LBA_ADDRESSING ) {
        LOG_DEBUG("disk is using lba addressing ... skipping CHS tests on this partition record\n");
        LOGEXIT();
        return 0;
    }


    /*
     * Test Conversions between CHS and LBA of the starting LBA of the partition
     */
    if ( below_1024_cylinder_limit(ld,start_lba ) == TRUE ) {

        /* convert LBA to CHS */
        if ( LBAtoCHS( ld, start_lba, &chs ) ) {
            LOG_ERROR("error, below 1024 cyls, partition starting LBA to CHS conversion failed\n");
            LOGEXIT();
            return EINVAL;
        }

        if ( ( chs.cylinder != start_chs.cylinder) ||
             ( chs.head     != start_chs.head ) ||
             ( chs.sector   != start_chs.sector ) ) {
            LOG_ERROR("error, below 1024 cyls, partition starting LBA != calculated starting CHS.\n");
            LOG_ERROR("       partition  CHS = %d:%d:%d\n",
                       start_chs.cylinder,
                       start_chs.head,
                       start_chs.sector);
            LOG_ERROR("       calculated CHS = %d:%d:%d\n",
                       chs.cylinder,
                       chs.head,
                       chs.sector);
            LOGEXIT();
            return EPROTO;
        }

        /* convert CHS to LBA */
        if ( CHStoLBA( ld,  &start_chs, &lba ) ) {
            LOG_ERROR("error, below 1024 cyls, partition starting CHS to LBA conversion call failed\n");
            LOGEXIT();
            return EINVAL;
        }
        if ( lba != start_lba ) {
            LOG_ERROR("error, below 1024 cyls, partition starting LBA != calculated starting LBA\n");
            LOG_ERROR("       partition  LBA = %lld\n", start_lba );
            LOG_ERROR("       calculated LBA = %lld\n", lba );
            LOGEXIT();
            return EPROTO;
        }
    }


    /*
     * Test Conversions between CHS and LBA of the ending LBA of the partition
     */
    if ( below_1024_cylinder_limit(ld, end_lba ) == TRUE ) {

        if ( LBAtoCHS( ld, end_lba, &chs ) ) {
            LOG_ERROR("error, below 1024 cyls, partition ending LBA to CHS conversion call failed\n");
            LOGEXIT();
            return EINVAL;
        }
        if ( ( chs.cylinder != end_chs.cylinder) ||
             ( chs.head     != end_chs.head ) ||
             ( chs.sector   != end_chs.sector ) ) {
            LOG_ERROR("error, below 1024 cyls, partition ending LBA != calculated ending CHS\n");
            LOG_ERROR("       partition  CHS = %d:%d:%d\n",
                       end_chs.cylinder,
                       end_chs.head,
                       end_chs.sector);
            LOG_ERROR("       calculated CHS = %d:%d:%d\n",
                       chs.cylinder,
                       chs.head,
                       chs.sector);
            LOGEXIT();
            return EPROTO;
        }

        if ( CHStoLBA( ld,  &end_chs, &lba ) ) {
            LOG_ERROR( "error, below 1024 cyls, partition ending CHS to LBA conversion call failed\n");
            LOGEXIT();
            return EINVAL;
        }

        if ( lba != end_lba ) {
            LOG_ERROR( "error, below 1024 cyls, partition ending CHS != ending LBA (START LBA + SIZE)\n");
            LOG_ERROR("       partition  LBA = %lld\n", end_lba );
            LOG_ERROR("       calculated LBA = %lld\n", lba );
            LOGEXIT();
            return EPROTO;
        }
    }


    /*
     * Test for max geometry values in the partition record for those
     * partitions above the 1024 cylinder limit.
     */
    if ( below_1024_cylinder_limit(ld,start_lba) == FALSE ) {  // partition entirely above 1024 cyls

       // only check primary partitions because logical partitions
       // will have CHS values that are relative to the start
       // of the logical drive and may not be MAX geometry even
       // if the logical drive is above 1024 cylinders.
       if (ebr_chaining == FALSE) {

           if ( (start_chs.cylinder != MAX_CYLINDERS ) ||
                (start_chs.head     != disk_pdata->geometry.heads-1 ) ||
                (start_chs.sector   != disk_pdata->geometry.sectors_per_track ) ||
                (end_chs.cylinder   != MAX_CYLINDERS ) ||
                (end_chs.head       != disk_pdata->geometry.heads-1 ) ||
                (end_chs.sector     != disk_pdata->geometry.sectors_per_track )) {

              LOG_ERROR("error, partition is above 1024 cyls, but start/end CHS fields are not MAX values\n");
              LOG_ERROR("       partition  start CHS = %d:%d:%d\n",
                          start_chs.cylinder,
                          start_chs.head,
                          start_chs.sector);
              LOG_ERROR("       partition    end CHS = %d:%d:%d\n",
                          end_chs.cylinder,
                          end_chs.head,
                          end_chs.sector);
              LOG_ERROR("            max values  CHS = %d:%d:%d\n",
                          MAX_CYLINDERS,
                          disk_pdata->geometry.heads-1,
                          disk_pdata->geometry.sectors_per_track);
              LOGEXIT();
              return EPROTO;

          }

       }

    }

    if ( ( below_1024_cylinder_limit(ld,start_lba) == TRUE ) &&    // partition starts below 1024 cyls
         ( below_1024_cylinder_limit(ld,end_lba) == FALSE )) {     // but ends above it.

       if ( (end_chs.cylinder != MAX_CYLINDERS ) ||
            (end_chs.head     != disk_pdata->geometry.heads-1 ) ||
            (end_chs.sector   != disk_pdata->geometry.sectors_per_track )) {
           LOG_ERROR("error, partition starts below 1024 cyls but ends above it and ending CHS field is not MAX values\n");
           LOG_ERROR("       partition    end CHS = %d:%d:%d\n",
                       end_chs.cylinder,
                       end_chs.head,
                       end_chs.sector);
           LOG_ERROR("            max values  CHS = %d:%d:%d\n",
                       MAX_CYLINDERS,
                       disk_pdata->geometry.heads-1,
                       disk_pdata->geometry.sectors_per_track);

           LOGEXIT();
           return EPROTO;
       }

    }


    LOGEXIT();
    return 0;
}


static char
bad_partition_msg[]= "\nA partition record, describing partition %s, is invalid.\nSkipping segment discovery on this drive.\n";
static char
bad_ebr_chain_msg[]= "\nA partition record, chaining logical drives in the extended partition on drive %s, is invalid.\nSkipping segment discovery on this drive.\nSee /var/log/evmsEngine.log for more info.\n";
static char
bad_extd_partition_msg[]= "\nThe partition record, describing the extended partition on drive %s, is invalid.\nSkipping segment discovery on this drive.\nSee /var/log/evmsEngine.log for more info.\n";

/*
 *  Called to follow the MBR/EBR chain, seeing if we can support the
 *  partitioning scheme found on the drive.  This routine is called with
 *  the mbr sector and will follow the ebr chain if an extended partition
 *  exists.
 */
int      isa_valid_partition_table_chain( LOGICALDISK        *ld,                 // disk
                                          Master_Boot_Record *mbr,                // sector buffer - mbr or ebr
                                          lba_t               mbr_lba,            // LBA of sector
                                          lba_t               extd_partition_lba, // LBA of extd part on this disk
                                          BOOLEAN             ebr_chaining,       // TRUE if following ebr chain
                                          int                 logical_count,      // where we are in the ebr chain
                                          BOOLEAN             final_call  )       // if this is the final attempt
{                                                                                 // to validate the chain.
    int                         i;
    int                         extended_partition_count = 0;
    int                         partition_count  = 0;
    lba_t                       ebr_lba;
    Extended_Boot_Record       *ebr = NULL;
    Partition_Record           *part;
    struct plugin_functions_s  *dft;
    int                         rc;
    char                        pname[EVMS_VOLUME_NAME_SIZE+1];
    DISK_PRIVATE_DATA          *disk_pdata = get_disk_private_data(ld);

    LOGENTRY();
    LOG_DEBUG("validating ... Partition Table ... at LBA %08d \n", (u_int32_t) mbr_lba );
    LOG_DEBUG("using geometry:  Cylinders= %lld  Heads= %d  Sectors= %d\n",
               disk_pdata->geometry.cylinders,
               disk_pdata->geometry.heads,
               disk_pdata->geometry.sectors_per_track);

    if (ebr_chaining == TRUE)
        DisplayPartitionTable(ld, &mbr->Partition_Table[0], FALSE );
    else
        DisplayPartitionTable(ld, &mbr->Partition_Table[0], TRUE );


    /*
     *  First, validate any Primary/Logical partitions found in the partition table
     */
    for (i=0; i<4; i++) {

        part = &mbr->Partition_Table[i];

        rc = isa_valid_partition_record( ld, part, mbr_lba, extd_partition_lba, ebr_chaining );
        if (rc != 0) {

            // dont give up and display any error messages unless we cant continue ...
            if ( ( rc == EOVERFLOW ) ||    // something wrong with LBA and SIZE fields
                 ( rc == EINVAL ) ||       // something wrong with conversion calls
                 ( final_call == TRUE )) {  // not gunna make any further attempts ...
                                            // then display error message of some sort.

                if (isa_ebr_partition_record(part)==TRUE) {
                    if (ebr_chaining == TRUE) {
                        POPUP_MSG( NULL, NULL, bad_ebr_chain_msg, ld->name );
                    }
                    else {
                        POPUP_MSG( NULL, NULL, bad_extd_partition_msg, ld->name );
                    }
                }
                else {
                    if (ebr_chaining == TRUE) {
                        sprintf(pname, "%s%d", ld->name, logical_count+5 );
                    }
                    else {
                        sprintf(pname, "%s%d", ld->name, partition_count+1 );
                    }
                    POPUP_MSG( NULL, NULL, bad_partition_msg, pname, ld->name );
                }

                rc = ENOSYS; // cant support this disk
            }
            else {
                rc = EAGAIN;      // error ... but try again
            }


            LOGEXIT();
            return rc;  // return non-fatal error ... let caller try diff geometry
        }
        else if ( isa_ebr_partition_record(part) == TRUE ) {
            ++extended_partition_count;
        }
        else if ( isa_null_partition_record(part) == FALSE ) {
            ++partition_count;
        }

    }

    /*
     *  Test - Looking at an MBR with more than 1 extended partition defined ?
     */
    if ( (extended_partition_count > 1) &&
         (ebr_chaining == FALSE)) {
        LOG_ERROR("error, found more than 1 extended partition on disk %s\n", ld->name);
        POPUP_MSG( NULL, NULL, "\nFound more than 1 extended partition on disk %s.\nSkipping segment discovery on this disk.\n", ld->name);
        LOGEXIT();
        return ENOSYS;  // will not support this disk
    }

    /*
     *  Test - Looking at an EBR with more than 1 EBR chained to it ?
     */
    if ( (extended_partition_count > 1) &&
         (ebr_chaining == TRUE)) {
        LOG_ERROR("error, found more than 1 extended partition record in an EBR partition table on disk %s.\n", ld->name);
        POPUP_MSG( NULL, NULL, "\nFound more than 1 extended partition record in an EBR partition table on disk %s.\nSkipping segment discovery on this disk.\n", ld->name);
        LOGEXIT();
        return ENOSYS;  // will not support this disk
    }

    /*
     *  Test - Looking at an EBR with more than 1 logical partition ?
     */
    if ( ( partition_count > 1) &&
         ( ebr_chaining == TRUE ) ) {
        LOG_ERROR("error, found more than 1 logical partition in an EBR partition table on disk %s\n", ld->name);
        POPUP_MSG( NULL, NULL, "\nFound more than 1 logical partition in an EBR partition table on disk %s.\nSkipping segment discovery on this disk.\n", ld->name);
        LOGEXIT();
        return ENOSYS;  // will not support this disk
    }


    /*
     * This partition table is Ok ...  Now follow EBR chain if it exists...
     */
    for (i=0; i<4; i++) {
        part = &mbr->Partition_Table[i];
        if ( isa_ebr_partition_record(part) == TRUE ) break;
    }

    if ( isa_ebr_partition_record(part) == TRUE ) {

        ebr_lba    = DISK_TO_CPU32(START_LBA(part)) + extd_partition_lba;

        ebr = (Extended_Boot_Record *) malloc(disk_pdata->geometry.bytes_per_sector);
        if (ebr==NULL) {
            LOG_DEBUG("error, malloc of sector sized buffer (%d bytes) failed\n", disk_pdata->geometry.bytes_per_sector);
            LOGEXIT();
            return ENOSYS;  // will not support this disk
        }

        dft = (struct plugin_functions_s *)ld->plugin->functions.plugin;
        if ( dft->read( ld, ebr_lba, 1,(void *) ebr ) ) {
            LOG_DEBUG("error, I/O error while trying to read partition table at LBA %d off disk %s\n", ebr_lba, ld->name);
            LOGEXIT();
            free(ebr);
            return ENOSYS;  // will not support this disk
        }

        if (extd_partition_lba == 0) {
            rc = isa_valid_partition_table_chain( ld, ebr, ebr_lba, ebr_lba, TRUE, ++logical_count, final_call );
        }
        else {
            rc = isa_valid_partition_table_chain( ld, ebr, ebr_lba, extd_partition_lba, TRUE, ++logical_count, final_call );
        }

        free(ebr);

        LOGEXIT();
        return rc;

    }

    LOG_DEBUG("table (lba %08d) is valid\n", (u_int32_t)mbr_lba);
    LOGEXIT();
    return 0;
}


BOOLEAN partition_record_uses_lba_addressing( LOGICALDISK *ld, struct partition *part )
{
    u_int32_t  start_cylinders, start_heads, start_sectors;
    u_int32_t  end_cylinders, end_heads, end_sectors;
    u_int32_t  partition_size;

    LOGENTRY();

    // look for caller passing NULL partition record
    if (isa_null_partition_record(part)==TRUE) {
        LOGEXIT();
        return FALSE;
    }

    // get info from partition record
    start_cylinders = ((START_SECT(part)&0xC0)<<2) + START_CYL(part);
    start_heads     =  START_HEADS(part);
    start_sectors   =  START_SECT(part) & 0x3F;

    end_cylinders   = ((END_SECT(part)&0xC0)<<2) + END_CYL(part);
    end_heads       =  END_HEADS(part);
    end_sectors     =  END_SECT(part) & 0x3F;

    partition_size  =  DISK_TO_CPU32( NR_SECTS(part) );

LOG_DEBUG("  sys: 0x%x\n", SYS_IND(part));
LOG_DEBUG(" boot: 0x%x\n", BOOT_IND(part));
LOG_DEBUG("  lba: %d\n", START_LBA(part));
LOG_DEBUG(" size: %d\n", NR_SECTS(part));
LOG_DEBUG("start: C= %d   H= %d  S=%d\n", start_cylinders, start_heads, start_sectors );
LOG_DEBUG("  end: C= %d   H= %d  S=%d\n", end_cylinders, end_heads, end_sectors );



    //
    // if   START_CHS == END_CHS
    // then  the partition says it is 1 sector in size
    //
    if  ( ( start_cylinders != end_cylinders ) ||
          ( start_heads     != end_heads ) ||
          ( start_sectors   != end_sectors )) {
        LOGEXIT();
        return FALSE;
    }


    //
    // and if the partition size is > 1 sector
    // then ... it must be using lba addressing !!
    //
    if ( partition_size > 1) {
        LOG_DEBUG("partition record is using LBA addressing\n");
        LOGEXIT();
        return TRUE;
    }
    else {
        LOGEXIT();
        return FALSE;
    }


}


/*
 *  Called to test if the disk is using LBA addressing mode
 *  for partition records. If all records are using max
 *  geometry for CHS values, regardless of the LBA, then it
 *  is using LBA addressing mode.
 */
BOOLEAN     disk_uses_lba_addressing( LOGICALDISK *ld )
{
    int                         i,rc;
    Master_Boot_Record          mbr;
    Partition_Record           *part;
    struct plugin_functions_s  *dft;


    LOGENTRY();

    // check parms
    if ( ld == NULL ) {
        LOGEXIT();
        return FALSE;
    }

    // get mbr sector
    dft = (struct plugin_functions_s *)ld->plugin->functions.plugin;
    rc  = dft->read( ld, 0, 1,(void *) &mbr );
    if ( rc ){
        LOGEXIT();
        return FALSE;
    }

    // only works if disk has an msdos partition scheme that forces
    // cylinder allignment.
    if (has_msdos_signature( &mbr ) == FALSE) {
        LOGEXIT();
        return FALSE;
    }

    // need data partition records to figure things out
    if ( ptable_has_data_partition_record( &mbr )==FALSE) {
        LOGEXIT();
        return FALSE;
    }

    // examine each record
    for (i=0; i<4; i++) {

        part = &mbr.Partition_Table[i];

        if (isa_null_partition_record( part ) == TRUE) {
            continue;
        }

        if (partition_record_uses_lba_addressing(ld, part)==FALSE) {
            return FALSE;
        }

    }

    LOG_DEBUG("returning true\n");
    LOGEXIT();
    return TRUE;
}


