/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: embedded.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>
#include <linux/evms/evms_user.h>

#include "ptables.h"
#include "segs.h"
#include "embedded.h"
#include "unixware.h"


/*
 * Builds a disk segment matching the unixware partition info
 */
static DISKSEG * build_unixware_segment( LOGICALDISK       *ld,
                                         DISKSEG           *msdos_seg,
                                         u_int32_t          start,
                                         u_int32_t          size,
                                         u_int32_t          sys_id,
                                         u_int32_t          ptable_index,
                                         u_int32_t          minor,
                                         u_int32_t          tag,
                                         u_int32_t          flags,
                                         dlist_t            recovery_list )
{
    DISKSEG           *seg     = NULL;
    SEG_PRIVATE_DATA  *pdata   = NULL;
    void              *handle  = NULL;
    int                rc;

    LOGENTRY();

    seg = build_segment_for_embedded_partition( ld, msdos_seg, start, size, sys_id, ptable_index, minor );
    if (seg) {

        pdata = (SEG_PRIVATE_DATA *) seg->private_data;

        pdata->flags = SEG_IS_UNIXWARE_PARTITION | SEG_IS_EMBEDDED;

        pdata->tag         = tag;
        pdata->permissions = flags;

        rc = InsertObject ( (dlist_t)          recovery_list,
                            (int)              sizeof(DISKSEG),
                            (void *)           seg,
                            (TAG)              SEGMENT_TAG,
                            (void *)           NULL,
                            (Insertion_Modes)  InsertBefore,
                            (BOOLEAN)          TRUE,
                            (void **)         &handle );

        if (rc != DLIST_SUCCESS) {
            free_disk_segment( seg );
            seg = NULL;
        }

    }


    LOGEXIT();
    return seg;
}



int do_unixware_partition_discover( LOGICALDISK *ld, Partition_Record *part )
{
    struct unixware_disklabel  *l;
    struct unixware_slice      *p;
    char                        data[EVMS_VSECTOR_SIZE];
    int                         rc = 0;
    struct plugin_functions_s  *dft;
    DISK_PRIVATE_DATA          *disk_pdata;
    DISKSEG                    *seg;
    DISKSEG                    *uw_seg;
    int                         uw_partition_count=0;
    u_int32_t                   ptable_index=0;
    u_int32_t                   minor;
    dlist_t                     recovery_list;


    LOGENTRY();

    // get disk info
    disk_pdata = get_disk_private_data(ld);
    if (disk_pdata == NULL) {
        LOGEXIT();
        return ENODEV;
    }

     // get disk manager function table
    dft = (struct plugin_functions_s *)ld->plugin->functions.plugin;
    if (dft==NULL) {
        LOGEXIT();
        return ENODEV;
    }

    // read UnixWare info
    rc = dft->read( ld,
                    DISK_TO_CPU32( START_LBA(part) ) + UNIXWARE_PART_TABLE_SECTOR_OFFSET,
                    1,
                    (void *) data );

    if (rc) {
        LOGEXIT();
        return rc;
    }

    // check unixware magic number
    l = (struct unixware_disklabel *) data;

    if ( ( DISK_TO_CPU32(l->d_magic)      != UNIXWARE_DISKMAGIC ) ||
         ( DISK_TO_CPU32(l->vtoc.v_magic) != UNIXWARE_DISKMAGIC2 ))  {
        LOGEXIT();
        return 0;
    }

    // create a dlist
    recovery_list = CreateList();
    if (recovery_list == NULL) {
        LOGEXIT();
        return ENOMEM;
    }

    // get evms segment storage object from disk segment list
    // and remove it from the list, making room for bsd partitions.
    seg = get_matching_segment( (dlist_t)        ld->parent_objects,
                                (lba_t)          DISK_TO_CPU32( START_LBA(part) ),
                                (sector_count_t) DISK_TO_CPU32( NR_SECTS(part)  ));

    if (seg) {
        rc = remove_diskseg_from_list( ld->parent_objects, seg );
        if (rc) {
            DestroyList(&recovery_list, FALSE);
            LOGEXIT();
            return rc;
        }
    }
    else {
        DestroyList(&recovery_list, FALSE);
        LOGEXIT();
        return ENODEV;
    }


    // calculate the first minor to be used by UnixWare embedded partitions
    minor = disk_pdata->logical_drive_count + disk_pdata->embedded_partition_count + 5;


LOG_DEBUG("UnixWare Info:\n");
LOG_DEBUG("     geometry:  C= %d   H= %d  S= %d\n", DISK_TO_CPU32(l->d_ncylinders), DISK_TO_CPU32(l->d_ntracks), DISK_TO_CPU32(l->d_nsectors));
LOG_DEBUG("     sector size = %d\n", DISK_TO_CPU32(l->d_secsize) );
LOG_DEBUG("     number of unixware partition table entries: %d\n", DISK_TO_CPU16(l->vtoc.v_nslices) );


    // walk through the partition table

    p            = &l->vtoc.v_slice[0];
    ptable_index = 0;
    rc           = 0;

    while ( p - &l->vtoc.v_slice[0] < DISK_TO_CPU16(l->vtoc.v_nslices) ) {

        if ( ( DISK_TO_CPU16(p->s_flags)  &  UW_FLAG_VALID ) &&
             ( DISK_TO_CPU16(p->s_label) !=  UW_TAG_ENTIRE_DISK )) {

LOG_DEBUG("Slice %d: start=%d  size=%d  label=%d  flags=%d\n", ptable_index, DISK_TO_CPU32( p->start_sect ), DISK_TO_CPU32( p->nr_sects ), DISK_TO_CPU16(p->s_label), DISK_TO_CPU16(p->s_flags));

            uw_seg = build_unixware_segment( ld,
                                             seg,
                                             DISK_TO_CPU32( p->start_sect ),
                                             DISK_TO_CPU32( p->nr_sects ),
                                             UNIXWARE_PARTITION,
                                             ptable_index,
                                             minor,
                                             (u_int32_t) DISK_TO_CPU16(p->s_label),
                                             (u_int32_t) DISK_TO_CPU16(p->s_flags),
                                             recovery_list );

            if (uw_seg) {

                if ( insert_diskseg_into_list( ld->parent_objects, uw_seg) ){
                    ++minor;
                    ++uw_partition_count;
                    ++disk_pdata->embedded_partition_count;
                }
                else {
                    rc = DLIST_CORRUPTED;
                }
            }
            else {
                rc = ENOMEM;
            }
        }

        if (rc) break;

        ++p;
        ++ptable_index;

    }

    if (rc) {
        LOG_ERROR("error, problems adding unixware partitions for disk %s.", ld->name );
        remove_embedded_partitions_from_disk( ld, recovery_list );
        insert_diskseg_into_list( ld->parent_objects, seg);
        POPUP_MSG(NULL,NULL,"\nAbandoning effort with embedded unixware partitions found in %s\n", seg->name);
        rc = 0;
    }
    else {

        // if we produced -ANY- unixware segments ... then consume the container segment
        if ( uw_partition_count > 0 ) {
            diskseg_to_container_segment( seg );
            CopyList(seg->parent_objects, recovery_list, AppendToList);
            LOG_DEBUG("Info, found %d embedded unixware partitions in %s\n", uw_partition_count, seg->name);
        }
        else {
            insert_diskseg_into_list( ld->parent_objects, seg);
        }

    }

    DestroyList(&recovery_list, FALSE);
    LOGEXIT();
    return rc;
}


int do_unixware_partition_commit( LOGICALDISK *ld, DISKSEG *seg )
{
    struct unixware_disklabel  *l;
    struct unixware_slice      *p;
    DISKSEG                    *tseg;
    char                        data[EVMS_VSECTOR_SIZE];
    int                         rc = 0;
    struct plugin_functions_s  *dft;
    DISK_PRIVATE_DATA          *disk_pdata;
    SEG_PRIVATE_DATA           *pdata;


    LOGENTRY();

    // get disk info
    disk_pdata = get_disk_private_data(ld);
    if (disk_pdata == NULL) {
        LOGEXIT();
        return ENODEV;
    }

     // get disk manager function table
    dft = (struct plugin_functions_s *)ld->plugin->functions.plugin;
    if (dft==NULL) {
        LOGEXIT();
        return ENODEV;
    }

    // read UnixWare info
    rc = dft->read( ld,
                    seg->start + UNIXWARE_PART_TABLE_SECTOR_OFFSET,
                    1,
                    (void *) data );

    if (rc) {
        LOGEXIT();
        return rc;
    }

    // check unixware magic number
    l = (struct unixware_disklabel *) data;

    if ( ( DISK_TO_CPU32(l->d_magic)      != UNIXWARE_DISKMAGIC ) ||
         ( DISK_TO_CPU32(l->vtoc.v_magic) != UNIXWARE_DISKMAGIC2 ))  {
        LOGEXIT();
        return 0;
    }

    // walk through the partition table, removing all the data
    // partitions we find.
    p = &l->vtoc.v_slice[0];
    while ( p - &l->vtoc.v_slice[0] < DISK_TO_CPU16(l->vtoc.v_nslices) ) {

        if ( ( DISK_TO_CPU16(p->s_flags)  &  UW_FLAG_VALID ) &&
             ( DISK_TO_CPU16(p->s_label) !=  UW_TAG_ENTIRE_DISK )) {

           memset( p, 0, sizeof(struct unixware_slice) );

        }

        ++p;
    }

    // now walk through the segment list on this drive and find all
    // the unixware partitions that are embedded in this container seg.
    // and add them to the unixware partition table.
    rc = GoToStartOfList( ld->parent_objects );
    if (rc == DLIST_SUCCESS) {

        rc = GetObject( ld->parent_objects,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE, (void **) &tseg );
        while (rc == DLIST_SUCCESS) {

            pdata = (SEG_PRIVATE_DATA *)tseg->private_data;

            if ( ( pdata->flags & SEG_IS_UNIXWARE_PARTITION ) &&
                 ( only_child( tseg ) == seg ) ) {

                p = &l->vtoc.v_slice[pdata->ptable_index];

                p->start_sect = CPU_TO_DISK32( tseg->start );
                p->nr_sects   = CPU_TO_DISK32( tseg->size );
                p->s_label    = CPU_TO_DISK16( (u_int16_t) pdata->tag );
                p->s_flags    = CPU_TO_DISK16( (u_int16_t) pdata->permissions );

            }

            if (rc == DLIST_SUCCESS) {
                rc = GetNextObject( ld->parent_objects,sizeof(DISKSEG),SEGMENT_TAG, (ADDRESS *) &tseg );
            }

        }
    }


    // update UnixWare info
    rc = dft->write( ld,
                     seg->start + UNIXWARE_PART_TABLE_SECTOR_OFFSET,
                     1,
                     (void *) data );



    LOGEXIT();
    return rc;
}

