/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: lvmregmgr
 * File: lvmregmgr.h
 */ 

#ifndef _LVM_REGION_MANAGER_H_
#define _LVM_REGION_MANAGER_H_ 1


// Datatypes and Defs used in lvmregmgr.c
#define SECTOR_SIZE_BITS		9
#define PAGE_SIZE_SECTORS		16	// for 8k pages
#define PE_SIZE_ALIGNMENT_SECTORS	128	// for 64k aligned PEs
#define LVM_DEV_DIRECTORY		"lvm/"
#define EVMS_LVM_PV_REMOVE_IOCTL	0x01
#define EVMS_LVM_SNAPSHOT_STAT_IOCTL	0x02

typedef u_int8_t boolean;
typedef u_int8_t uint8_t;
typedef u_int16_t uint16_t;
typedef u_int32_t uint32_t;
typedef u_int64_t uint64_t;


// The following definitions and data structures are from lvm.h and liblvm.h
// LVM 0.9.1beta8 distribution (Copyright Heinz Mauelshagen and Sistina
// Software (www.sistina.com)). Since the metadata format changed in beta8,
// lvm.h (in the Linux kernel source) changed significantly enough that this
// module would no longer compile. Instead of requiring EVMS users to install
// the latest lvm release, the required definitions and data structures will
// now be included in this header file.


// absolute limits for VGs, PVs per VG and LVs per VG
#ifndef	SECTOR_SIZE
#define SECTOR_SIZE		512
#endif
#ifndef PAGE_SIZE
#define PAGE_SIZE		4096
#endif
#define MAX_VG  		99
#define MAX_LV			256	// due to 8 bit minor
#define	MAX_PV			256	
#define	NAME_LEN		128	// don't change!!!
#define	UUID_LEN		32	// don't change!!!

// The following are all sector-based
#define	LVM_METADATA_VERSION	1
#define	LVM_PE_T_MAX		((1 << (sizeof(uint16_t)* 8)) - 2)
#define	LVM_LV_SIZE_MAX(pe_size) ((long long)LVM_PE_T_MAX * pe_size > (long long) 1024*1024/SECTOR_SIZE*1024*1024 ? (long long) 1024*1024/SECTOR_SIZE*1024*1024 : (long long) LVM_PE_T_MAX * pe_size)
#define	LVM_MIN_PE_SIZE		(8L * 1024L / SECTOR_SIZE)		// 8 kB in sectors
#define	LVM_MAX_PE_SIZE		(16L * 1024L * 1024L / SECTOR_SIZE * 1024L)	// 16 GB in sectors
#define	LVM_DEFAULT_PE_SIZE	(4L * 1024L * 1024L / SECTOR_SIZE)	// 4 MB in sectors
#define	LVM_DEFAULT_STRIPE_SIZE	(16L * 1024L / SECTOR_SIZE)		// 16 kB  in sectors
#define	LVM_MIN_STRIPE_SIZE	(PAGE_SIZE / SECTOR_SIZE)		// PAGESIZE in sectors 
#define	LVM_MAX_STRIPE_SIZE	(512L * 1024 / SECTOR_SIZE)		// 512 kB in sectors
#define	LVM_MAX_STRIPES		128					// max # of stripes
#define	LVM_MAX_SIZE            (1024L * 1024L * 1024L * 1024L / SECTOR_SIZE)	// 1 TB in sectors
#define	LVM_PE_SIZE_PV_SIZE_REL	5					// max relation PV size and PE size
#define	LVM_SNAPSHOT_MAX_CHUNK	(1024L * 1024L / SECTOR_SIZE)		// 1 MB in sectors
#define	LVM_SNAPSHOT_DEF_CHUNK	(64L * 1024L / SECTOR_SIZE)		// 64 kB in sectors
#define	LVM_SNAPSHOT_MIN_CHUNK	(PAGE_SIZE / SECTOR_SIZE)		// PAGESIZE in sectors
#define	LVM_MAX_READ_AHEAD	120					// maximum read ahead sectors

// The following are all byte-based
#define LVM_VGDA_ALIGN		4096UL			// some metadata on the disk need to be aligned
#define	LVM_PV_DISK_BASE	0L			// base of PV structure in disk partition
#define	LVM_PV_DISK_SIZE	1024L			// size reserved for PV structure on disk
#define	LVM_VG_DISK_BASE	round_up(LVM_PV_DISK_BASE + LVM_PV_DISK_SIZE, LVM_VGDA_ALIGN)
							// base of VG structure in disk partition
#define	LVM_VG_DISK_SIZE  	(8*512L)		// size reserved for VG structure
#define	LVM_PV_UUIDLIST_DISK_BASE round_up(LVM_VG_DISK_BASE + LVM_VG_DISK_SIZE, LVM_VGDA_ALIGN)
							// name list of physical volumes on disk
#define	LVM_LV_DISK_OFFSET(a,b)	((a)->lv_on_disk.base + sizeof(lv_disk_t) * b)
#define	LVM_VGDA_SIZE(pv)	((pv)->pe_on_disk.base + (pv)->pe_on_disk.size)
#define	LVM_PE_ALIGN		(65536UL / SECTOR_SIZE)

// Status flags
//
// vg_disk_t
#define	VG_ACTIVE            0x01	// vg_status
#define	VG_EXPORTED          0x02
#define	VG_EXTENDABLE        0x04

#define	VG_READ              0x01	// vg_access
#define	VG_WRITE             0x02
#define	VG_CLUSTERED         0x04
#define	VG_SHARED            0x08

// lv_disk_t
#define	LV_ACTIVE            0x01	// lv_status
#define	LV_SPINDOWN          0x02

#define	LV_READ              0x01	// lv_access
#define	LV_WRITE             0x02
#define	LV_SNAPSHOT          0x04
#define	LV_SNAPSHOT_ORG      0x08

#define	LV_BADBLOCK_ON       0x01	// lv_badblock

#define	LV_STRICT            0x01	// lv_allocation
#define	LV_CONTIGUOUS        0x02

// pv_disk_t
#define	PV_ACTIVE            0x01	// pv_status
#define	PV_ALLOCATABLE       0x02	// pv_allocatable

// misc
#define LVM_SNAPSHOT_DROPPED_SECTOR 1

// copy on write tables in disk format
typedef struct lv_COW_table_disk_v1 {
	uint64_t pv_org_number;
	uint64_t pv_org_rsector;
	uint64_t pv_snap_number;
	uint64_t pv_snap_rsector;
} lv_COW_table_disk_t;

// disk stored pe information
typedef struct {
	uint16_t lv_num;
	uint16_t le_num;
} pe_disk_t;

// disk stored PV, VG, LV and PE size and offset information
typedef struct {
	uint32_t base;
	uint32_t size;
} lvm_disk_data_t;

// PV disk structure
typedef struct pv_disk_v2 {
	uint8_t id[2];			// Identifier
	uint16_t version;		// LVM version
	lvm_disk_data_t pv_on_disk;
	lvm_disk_data_t vg_on_disk;
	lvm_disk_data_t pv_uuidlist_on_disk;
	lvm_disk_data_t lv_on_disk;
	lvm_disk_data_t pe_on_disk;
	uint8_t pv_uuid[NAME_LEN];
	uint8_t vg_name[NAME_LEN];
	uint8_t system_id[NAME_LEN];	// for vgexport/vgimport
	uint32_t pv_major;
	uint32_t pv_number;
	uint32_t pv_status;
	uint32_t pv_allocatable;
	uint32_t pv_size;
	uint32_t lv_cur;
	uint32_t pe_size;
	uint32_t pe_total;
	uint32_t pe_allocated;
	
	// new in struct version 2
	uint32_t pe_start;	        // in sectors

} pv_disk_t;

// LV disk structure
typedef struct lv_disk_v3 {
	uint8_t lv_name[NAME_LEN];
	uint8_t vg_name[NAME_LEN];
	uint32_t lv_access;
	uint32_t lv_status;
	uint32_t lv_open;	
	uint32_t lv_dev;
	uint32_t lv_number;
	uint32_t lv_mirror_copies;	// for future use
	uint32_t lv_recovery;		//       "
	uint32_t lv_schedule;		//       "
	uint32_t lv_size;
	uint32_t lv_snapshot_minor;	// minor number of original
	uint16_t lv_chunk_size;		// chunk size of snapshot
	uint16_t dummy;
	uint32_t lv_allocated_le;
	uint32_t lv_stripes;
	uint32_t lv_stripesize;
	uint32_t lv_badblock;		// for future use
	uint32_t lv_allocation;
	uint32_t lv_io_timeout;		// for future use
	uint32_t lv_read_ahead;
} lv_disk_t;

// VG disk structure
typedef struct vg_disk_v2 {
	uint8_t vg_uuid[UUID_LEN];	// volume group UUID
	uint8_t vg_name_dummy[NAME_LEN-UUID_LEN];	// rest of v1 VG name
	uint32_t vg_number;		// volume group number
	uint32_t vg_access;		// read/write
	uint32_t vg_status;		// active or not
	uint32_t lv_max;		// maximum logical volumes
	uint32_t lv_cur;		// current logical volumes
	uint32_t lv_open;		// open    logical volumes
	uint32_t pv_max;		// maximum physical volumes
	uint32_t pv_cur;		// current physical volumes FU
	uint32_t pv_act;		// active physical volumes
	uint32_t dummy;
	uint32_t vgda;			// volume group descriptor arrays FU
	uint32_t pe_size;		// physical extent size in sectors 
	uint32_t pe_total;		// total of physical extents
	uint32_t pe_allocated;		// allocated physical extents
	uint32_t pvg_total;		// physical volume groups FU
} vg_disk_t;

// Useful inlines
// The "size" paramaters MUST be power-of-2.
static inline ulong round_down(ulong n, ulong size) {
	size--;
	return(n & ~size);
}

static inline ulong round_up(ulong n, ulong size) {
	size--;
	return((n + size) & ~size);
}

static inline ulong div_up(ulong n, ulong size) {
	return(round_up(n, size) / size);
}

// End of lvm.h/liblvm.h imported data structures


// Many of the following data structures were originally defined in evms_lvm.h
// in the kernel-side code. They have evolved enough from their definitions in
// the kernel that it is just simpler to redefine them here.


// Structure for doing PV remove ioctls
typedef struct lvm_pv_remove_ioctl_s {
	unsigned char			vg_uuid[UUID_LEN];
	int				pv_number;
	struct lvm_pv_remove_ioctl_s	* next;
} lvm_pv_remove_ioctl_t;


// Structure for doing snapshot stat ioctls
typedef struct lvm_snapshot_stat_ioctl_s {
	unsigned char	vg_uuid[UUID_LEN];
	int		lv_number;
	u_int64_t	next_free_chunk;
	u_int32_t	lv_status;
} lvm_snapshot_stat_ioctl_t;


// Entries in the array of physical volumes (PV)
// in a volume group (VG)

#define LVM_PV_FLAG_INVALID_UUID	(1 << 0)
#define LVM_PV_FLAG_LV_CUR_UPDATED	(1 << 1)

typedef struct lvm_physical_volume_s {
	pv_disk_t		* pv;		// Copy of LVM on-disk PV struct
	storage_object_t	* segment;	// EVMS segment representing this PV
	struct lvm_volume_group_s * group;	// Pointer back to the parent volume group.
	pe_disk_t		* pe_map;	// Entire mapping of PEs on this PV
	u_int64_t		pe_map_sectors;	// Number of sectors needed for the PE map.
	long			number;
	long			flags;       
} lvm_physical_volume_t;


// Table for mapping logical extents (LE) to physical extents (PE)
// This structure is used within the lvm_logical_volume_t
typedef struct le_table_entry_s {
	lvm_physical_volume_t	* owning_pv;		// The PV that this LE is on
	u_int32_t		pe_number;		// The PE that this LE maps to
	u_int64_t		pe_sector_offset;	// The LSN of that PE
} le_table_entry_t;


// Logical volumes (LV) in a volume group (VG)
// A lot of the fields that were used in the kernel have been removed, since
// we will now be keeping a copy of the lv_disk_t structure.

#define LVM_LV_FLAG_EXPORTED		(1 << 0)
#define LVM_LV_FLAG_INCOMPLETE		(1 << 1)
#define LVM_LV_FLAG_DIRTY		(1 << 2)
#define LVM_LV_FLAG_ZERO_FIRST_1K	(1 << 3)

typedef struct lvm_logical_volume_s {
	lv_disk_t		* lv;		// The corresponding entry in the group's lv_array
	storage_object_t	* region;	// The EVMS region representing this volume
	struct lvm_volume_group_s * group;	// Pointer back to parent volume group
	le_table_entry_t	* le_map;	// Mapping of logical to physical extents
	int			number;		// lv->lv_number + 1
	int			minor;		// MINOR(lv->lv_dev)
	int			flags;		// LVM_LV_*
	u_int64_t		next_free_chunk;	// Logical starting LBA of next available chunk
	struct lvm_logical_volume_s * snapshot_next;	// Linked list of volumes snapshotting the original
	struct lvm_logical_volume_s * snapshot_org;	// Pointer to volume being snapshotted
} lvm_logical_volume_t;


// Volume groups (VG)

#define LVM_VG_FLAG_UUID_LIST_PRESENT	(1 << 0)
#define LVM_VG_FLAG_LV_LIST_PRESENT	(1 << 1)
#define LVM_VG_FLAG_INVALID_VG_NUMBER	(1 << 2)

typedef struct lvm_volume_group_s {
	vg_disk_t		* vg;			// Copy of LVM on-disk VG struct
	storage_container_t	* container;		// The EVMS storage container representing this group
	lvm_physical_volume_t	* pv_list[MAX_PV+1];	// Array of physical volumes that make up this group
	unsigned char		* uuid_list[MAX_PV+1];	// Array of UUIDs of the PVs in this group
	lvm_logical_volume_t	* volume_list[MAX_LV+1];// Array of logical volumes exported from this group
	lv_disk_t		* lv_array;		// The entire array of LVs stored on each PV in this VG
	lvm_logical_volume_t	* freespace;		// The volume representing the freespace in this group
	int			pv_count;		// Number of PVs/segments found on this VG
	int			volume_count;		// Number of LVs found on this VG
	int			flags;
} lvm_volume_group_t;


// Global symbols exported from lvmregmgr.c
extern plugin_record_t * evms_plugin_records[];


// Global data used by lvmregmgr
extern engine_functions_t		* lvm_engine;
extern plugin_record_t			LVM_Plugin;
extern plugin_record_t			* lvm_plugin;
extern struct lvm_volume_group_s	* lvm_unassigned_group;
extern dlist_t				lvm_group_list;
extern struct lvm_pv_remove_ioctl_s	* lvm_pv_remove_list;


// FOR_EACH: Iterate through each item in a dlist.
//           Requires an "int rc" variable in the calling function.
//           The "list" variable must be a dlist_t type.
//           The "item" variable can be any pointer type.
//           No items may be removed from the list during this loop.
#define FOR_EACH(item, list)	for ( rc = GoToStartOfList(list); \
					!rc && ((item) = lvm_get_list_item(list)); \
					rc = NextItem(list) )


// Log message macros
#define MESSAGE(msg, args...)		lvm_engine->user_message(lvm_plugin, NULL, NULL, msg, ## args)
#define LOG_CRITICAL(msg, args...)	MESSAGE("%s: " msg, __FUNCTION__ , ## args)
#define LOG_SERIOUS(msg, args...)	MESSAGE("%s: " msg, __FUNCTION__ , ## args)
#define LOG_ERROR(msg, args...)		MESSAGE("%s: " msg, __FUNCTION__ , ## args)
#define LOG_WARNING(msg, args...)	lvm_engine->write_log_entry(WARNING,	lvm_plugin, "%s: " msg, __FUNCTION__ , ## args)
#define LOG(msg, args...)		lvm_engine->write_log_entry(DEFAULT,	lvm_plugin, "%s: " msg, __FUNCTION__ , ## args)
#define LOG_DETAILS(msg, args...)	lvm_engine->write_log_entry(DETAILS,	lvm_plugin, "%s: " msg, __FUNCTION__ , ## args)
#define LOG_DEBUG(msg, args...)		lvm_engine->write_log_entry(DEBUG,	lvm_plugin, "%s: " msg, __FUNCTION__ , ## args)
#define LOG_EXTRA(msg, args...)		lvm_engine->write_log_entry(EXTRA,	lvm_plugin, "%s: " msg, __FUNCTION__ , ## args)
#define LOG_PROC(msg, args...)		lvm_engine->write_log_entry(ENTRY_EXIT,	lvm_plugin, "%s: " msg, __FUNCTION__ , ## args)
#define LOG_EVERYTHING(msg, args...)	lvm_engine->write_log_entry(EVERYTHING,	lvm_plugin, "%s: " msg, __FUNCTION__ , ## args)

#define	LOG_ENTRY			LOG_PROC("Entering\n")
#define LOG_EXIT(x) 			LOG_PROC("Exiting: rc = %d\n", x)
#define RETURN(x)			do { LOG_EXIT(x); return(x); } while (0)

// These need to follow all of the structure definitions.
#include "lvm_discover.h"
#include "lvm_dlist.h"
#include "lvm_groups.h"
#include "lvm_info.h"
#include "lvm_io.h"
#include "lvm_names.h"
#include "lvm_options.h"
#include "lvm_pv.h"
#include "lvm_snapshots.h"
#include "lvm_uuid.h"
#include "lvm_volumes.h"


#endif

