/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: parser.c
 */

/*
 * Change History:
 *
 * 6/2001  B. Rafanello  Initial version.
 *
 */

/*
 *
 */

/* Identify this file. */
#define PARSER_C

/*--------------------------------------------------
 * Necessary include files
 --------------------------------------------------*/

#include <ctype.h>    /* toupper */
#include <stdlib.h>   /* malloc */
#include <string.h>   /* malloc */
#include "token.h"    /* TokenType, Token, MaxIdentifierLength */
#include "scanner.h"
#include "screener.h" /* GetScreenedToken , and all of the predefined strings for keywords. */
#include "dlist.h"
#include "error.h"
/* include "evms_appAPI.h" */
#include "parser.h"


/*--------------------------------------------------
 * Private constants
 --------------------------------------------------*/
#define HANDLE_TAG  39392

/*--------------------------------------------------
 * Private types
 --------------------------------------------------*/
typedef enum
{
  Try_Again,
  Error,
  Token_Accepted
} Parse_Results;

/*--------------------------------------------------
 * Private variables
 --------------------------------------------------*/
static TokenType *    Current_Token = NULL;                    /* Token being parsed. */
static TokenType      Null_Token = {EofToken,0,0,0,NULL,TRUE}; /* Used to represent an empty token. */
static char *         Command_Filename = NULL;                 /* Name of command file, if any, being parsed. */
static dlist_t        Tokens = NULL;                           /* Tokens being stored during when lookahead is in operation. */
static uint           LookAhead_Active = 0;                    /* Used to indicate whether or not Look Ahead is active for parsing. */
static dlist_t        Lookahead_Handles = NULL;                /* Used to hold the handles of tokens to return to in the Tokens list. */

/*--------------------------------------------------
 * Private functions
 --------------------------------------------------*/

/* Set up token handling functions so that if we need to go to a lookahead scheme, it won't
   affect any existing code.                                                                 */
static void               Consume_Token( void );                                 /* Free's the current token and sets CurrentToken to NULL. */
static void               Next_Token( void );                                    /* Gets the next token to be parsed and sets CurrentToken to point to it. */
static Executable_Node *  Activate_Lookahead(void);                              /* Start Lookahead mode. */
static BOOLEAN            Restore_Current_Token( Executable_Node * Temp_Node);   /* Restores Current_Token to the value it was at before any Lookahead calls were made. */
static BOOLEAN            Terminate_Lookahead(Executable_Node * Current_Node,
                                              Executable_Node * Temp_Node,
                                              BOOLEAN           Keep_Children);  /* End lookahead mode. */
static void               Assign_Text( Executable_Node * Node, 
                                       TokenType * Token);                       /* Moves Token->Token_Text to Node->Node_Value taking into account whether or not
                                                                                    the Token is in the Lookahead buffer (i.e. the Tokens list. )                   */

/* Private Utility Functions */
static void Free_Command_List(dlist_t Command_List); /* Deletes all entries in the Command_List and frees their memory. */
static BOOLEAN Kill_Command_List_Entries (ADDRESS   Object,
                                          TAG       ObjectTag,
                                          uint      ObjectSize,
                                          ADDRESS   ObjectHandle,
                                          ADDRESS   Parameters,
                                          BOOLEAN * FreeMemory,
                                          uint    * Error);        /* Used with the PruneList function to delete entries in the Command_List. */

static BOOLEAN Kill_Tokens_List_Entries (ADDRESS   Object,
                                         TAG       ObjectTag,
                                         uint      ObjectSize,
                                         ADDRESS   ObjectHandle,
                                         ADDRESS   Parameters,
                                         BOOLEAN * FreeMemory,
                                         uint    * Error);


/* Private functions corresponding to non-terminals in the grammar. */
static Parse_Results     Command(Executable_Node * Current_Node, BOOLEAN Optional);
static Parse_Results     Allocation_Parameters(Executable_Node * Current_Node, BOOLEAN Optional);
static Parse_Results     Assign_Parameters(Executable_Node * Current_Node, BOOLEAN Optional);
static Parse_Results     Creation_Parameters(Executable_Node * Current_Node, BOOLEAN Optional);
static Parse_Results     Check_For_Compatibility_Keyword(Executable_Node * Current_Node, BOOLEAN Optional);
static Parse_Results     Expand_Shrink_Parameters(Executable_Node * Current_Node, BOOLEAN Optional);
static Parse_Results     Format_Parameters(Executable_Node * Current_Node, BOOLEAN Optional);
static Parse_Results     Set_Parameters(Executable_Node * Current_Node, NodeTypes Node_To_Create, BOOLEAN Optional);
static Parse_Results     Check_Parameters(Executable_Node * Current_Node, BOOLEAN Optional);
static Parse_Results     Defrag_Parameters(Executable_Node * Current_Node, BOOLEAN Optional);
static Parse_Results     Query_Type(Executable_Node * Current_Node, BOOLEAN Optional);
static Parse_Results     Plugin_Types(Executable_Node * Current_Node, BOOLEAN Use_Current_Token, BOOLEAN Optional);
static Parse_Results     Help_Type(Executable_Node * Current_Node, BOOLEAN Optional);
static Parse_Results     Feature_Identifier(Executable_Node * Current_Node, BOOLEAN Optional, BOOLEAN Parameters_Follow, BOOLEAN Objects_Are_Optional);
static Parse_Results     Plugin_Identifier(Executable_Node * Current_Node, BOOLEAN Parameters_Follow, BOOLEAN Get_Object_List, BOOLEAN Optional);
static Parse_Results     Size_Parameters(Executable_Node * Current_Node, BOOLEAN Filter_On_Freespace, BOOLEAN Optional);
static Parse_Results     Specify_Object(Executable_Node * Current_Node,  NodeTypes Node_To_Create, BOOLEAN Use_Current_Token, BOOLEAN Optional);
static Parse_Results     Size(Executable_Node * Current_Node, BOOLEAN Optional);
static Parse_Results     Precision(Executable_Node * Current_Node, BOOLEAN Optional);
static Parse_Results     Accept_Name(Executable_Node * Current_Node, BOOLEAN Name_Optional);              /* Used for Name = "string" */
static Parse_Results     Name(Executable_Node * Current_Node, BOOLEAN Field_Name, BOOLEAN Optional);                          /* Used for Name = Value pairs */
static Parse_Results     Parse_Name_Value_Pair(Executable_Node * Current_Node, BOOLEAN Optional);
static Parse_Results     Value(Executable_Node * Current_Node, BOOLEAN Optional);
static Parse_Results     Parse_Object_List(Executable_Node * Current_Node, BOOLEAN Optional);
static Parse_Results     Parse_Query_Plugins_Parameters(Executable_Node * Current_Node, BOOLEAN Optional);
static Parse_Results     Parse_Query_List_Options(Executable_Node * Current_Node, BOOLEAN Optional);
static Parse_Results     Parse_Query_Volumes_Parameters(Executable_Node * Current_Node, BOOLEAN Optional);
static Parse_Results     Parse_Query_Regions_Parameters(Executable_Node * Current_Node, BOOLEAN Optional);
static Parse_Results     Parse_Query_Disks_Parameters(Executable_Node * Current_Node, BOOLEAN Optional);
static Parse_Results     Parse_Query_Segments_Parameters(Executable_Node * Current_Node, BOOLEAN Optional);
static Parse_Results     Parse_Query_Objects_Parameters(Executable_Node * Current_Node, BOOLEAN Optional);
static Parse_Results     Parse_Query_Acceptable_Parameters(Executable_Node * Current_Node, BOOLEAN Optional);
static Parse_Results     Parse_Query_Acceptable_Create(Executable_Node * Current_Node, BOOLEAN Optional);
static Parse_Results     Parse_Query_Containers_Parameters(Executable_Node * Current_Node, BOOLEAN Optional);
static Parse_Results     Parse_Query_Extended_Info_Parameters(Executable_Node * Current_Node, BOOLEAN Optional);
static Parse_Results     Parse_Units(Executable_Node * Current_Node, BOOLEAN Allow_Time_Units, BOOLEAN Optional);
static Parse_Results     GT_LT_Parameter_Check(Executable_Node * Current_Node, BOOLEAN Filter_On_Freespace, BOOLEAN Optional);
static Parse_Results     Get_Volume(Executable_Node * Current_Node, BOOLEAN Use_Current_Token, BOOLEAN Optional);
static Parse_Results     Get_Object(Executable_Node * Current_Node, BOOLEAN Use_Current_Token, BOOLEAN Optional);
static Parse_Results     Get_Plugin(Executable_Node * Current_Node, BOOLEAN Use_Current_Token, BOOLEAN Optional);
static Parse_Results     Get_Region(Executable_Node * Current_Node, BOOLEAN Use_Current_Token, BOOLEAN Optional);
static Parse_Results     Get_Container(Executable_Node * Current_Node, BOOLEAN Use_Current_Token, BOOLEAN Optional);
static Parse_Results     Get_Disk(Executable_Node * Current_Node, BOOLEAN Use_Current_Token, BOOLEAN Optional);
static Parse_Results     Get_Expand_Shrink(Executable_Node * Current_Node, BOOLEAN Use_Current_Token, BOOLEAN Optional);
static Parse_Results     Get_Unclaimed(Executable_Node * Current_Node, BOOLEAN Use_Current_Token, BOOLEAN Optional);
static Parse_Results     Get_Freespace(Executable_Node * Current_Node, BOOLEAN Use_Current_Token, BOOLEAN Optional);
static Parse_Results     Check_For_Colon(BOOLEAN Optional);
static Parse_Results     Check_For_Comma(BOOLEAN  Comma_Is_Optional);
static Parse_Results     Check_For_EqualSign(BOOLEAN  EqualSign_Is_Optional);
static Parse_Results     Check_For_Open_Brace(BOOLEAN  Optional);
static Parse_Results     Check_For_Closing_Brace(BOOLEAN  Optional);
static Parse_Results     Check_For_Open_Paren(BOOLEAN  Optional);
static Parse_Results     Check_For_Closing_Paren(BOOLEAN  Optional);
static Executable_Node * Make_Child_Node(Executable_Node * Current_Node, BOOLEAN Add_To_Current_Node);



/*--------------------------------------------------
 * Public Functions
 --------------------------------------------------*/

dlist_t Parse_Command_Line(void)
{
  Parse_Results     Command_Found = FALSE;      /* Used to hold the return value from lower level parsing functions. */
  dlist_t             Command_List = NULL;        /* List of commands parsed thus far. */
  Executable_Node * Current_Node = NULL;        /* The node used to represent the current command being parsed. */
  ADDRESS           Unneeded;                   /* Used as a parameter for parameters we don't care about. */

  /* Clear out the lookahead system in case a previous command failed while lookahead was active. */
  LookAhead_Active = 0;
  if ( Tokens != NULL)
  {

    /* Remove any left over tokens from the Tokens list. */
    if (PruneList( Tokens,&Kill_Tokens_List_Entries,NULL) != DLIST_SUCCESS)
    {
      Report_Parser_Error(Internal_Error_Encountered,NULL,NULL);
      return NULL;
    }

    /* Now destroy the Tokens list. */
    if (DestroyList( &Tokens, FALSE) != DLIST_SUCCESS)
    {
      Report_Parser_Error(Internal_Error_Encountered,NULL,NULL);
      return NULL;
    }


  }

  if ( Lookahead_Handles != NULL )
  {

    /* Clear any handles from the handles list. */
    if (DestroyList( &Lookahead_Handles,TRUE) )
    {
      Report_Parser_Error(Internal_Error_Encountered,NULL,NULL);
      return NULL;
    }

  }

  /* Create the dlist_t we are to return. */
  Command_List = CreateList();
  if ( Command_List == NULL )
  {

    Report_Parser_Error(Out_Of_Memory, NULL, NULL);

    return NULL;

  }

  /* Begin parsing! */

  /* Get the first token. */
  Next_Token();

  /* We will loop until we are out of tokens. */
  while ( ( Current_Token != NULL ) &&
          ( Current_Token->Characterization != EofToken ) &&
          ( Current_Token->Characterization != InvalidCharacter )
        )
  {

    /* We must create a command node. */
    Current_Node = Make_Child_Node(NULL,FALSE);
    if ( Current_Node == NULL )
    {
      Report_Parser_Error(Out_Of_Memory, NULL, NULL);

      Free_Command_List(Command_List);
      Command_List = NULL;

      return NULL;

    }

    /* Check for EOF or InvalidCharacter */
    if ( ( Current_Token == NULL ) || ( Current_Token->Characterization == EofToken ) || ( Current_Token->Characterization == InvalidCharacter ) )
    {
      Report_Parser_Error(Expecting_Command_Start, Command_Filename, Current_Token);

      /* Free the Current_Node. */
      DestroyList(&Current_Node->Children, TRUE);
      free(Current_Node);
      Current_Node = NULL;

      /* We must free all of the nodes in the Command_List. */
      Free_Command_List(Command_List);
      Command_List = NULL;

      break;
    }

    Command_Found = Command( Current_Node, FALSE );   /* Attempt to parse a command. */

    /* We should have a complete command.  Put the command into the Command_List.
       If the command is not complete, when the Command_List is destroyed using the Free_Command_List
       function, then the node we just added to the command list will be disposed of also.             */
    if ( InsertObject(Command_List, sizeof(Executable_Node), Current_Node, STACK_NODE, NULL, AppendToList, TRUE, &Unneeded) != DLIST_SUCCESS )
    {
      /* We must be out of memory! */
      Report_Parser_Error(Out_Of_Memory, NULL, NULL);

      /* Free the Current_Node. */
      DestroyList(&Current_Node->Children, TRUE);
      free(Current_Node);
      Current_Node = NULL;

      Free_Command_List(Command_List);
      Command_List = NULL;

      break;

    }

    /* Did we successfully parse a commmand? */
    if ( Command_Found != Token_Accepted )
    {
      /* No commands were recognized or an error occurred. */
      if ( Command_Found == Try_Again )
        Report_Parser_Error( Expecting_Command_Name, NULL, Current_Token);

      /* Any error should have been reported by whoever found it. */

      /* Free any commands in the Command_List. */
      Free_Command_List(Command_List);
      Command_List = NULL;

      break;

    }

    /* Since we completed this command, lets look for another. */

    /* Get the next token. */
    Next_Token();

    /* Are we done? */
    if ( ( Current_Token == NULL ) || ( Current_Token->Characterization == EofToken ) )
    {
      Consume_Token();
      break;
    }

    /* Check for InvalidCharacter */
    if ( Current_Token->Characterization == InvalidCharacter )
    {

      Report_Parser_Error(Expecting_Command_Separator, Command_Filename, Current_Token);

      /* We must free all of the nodes in the Command_List. */
      Free_Command_List(Command_List);
      Command_List = NULL;

      break;

    }

    /* Check for the command separator - the ':' */
    if ( ( Current_Token->Characterization != Separator ) ||
         ( Current_Token->TokenText[0] != ':' )
       )
    {

      Report_Parser_Error(Expecting_Command_Separator, Command_Filename, Current_Token);

      /* We must free all of the nodes in the Command_List. */
      Free_Command_List(Command_List);
      Command_List = NULL;

      break;

    }

    /* We have a command separator.  Consume the token and start the while loop again. */
    Consume_Token();
    Next_Token();

  } /* while */

  return Command_List;

}


/*******************************************************************
 *  Private Functions
 *******************************************************************/

static Parse_Results Command(Executable_Node * Current_Node, BOOLEAN Optional)
{

  Executable_Node * Temp_Node = NULL;  /* Used for lookahead operations. */
  Parse_Results     Return_Value;      /* Used to track the return value from lower level parsing functions. */

  /* At this point, we know that '-' or '--' has been encountered.
     Also, we know that Current_Token has been loaded for us and
     has already been checked for NULL, EOF, and InvalidCharacter
     tokens.  We also know that Current_Node has been allocated
     but its fields have not yet been initialized.                 */

  /* A command is always a KeyWord. */
  if ( Current_Token->Characterization != KeyWord )
  {
    if ( Optional)
      return Try_Again;
    else
    {
      Report_Parser_Error( Expecting_Command_Name, NULL, Current_Token);
      return Error;
    }

  }

  /* Which command do we have? */
  switch ( Current_Token->TokenText[0] )
  {
    case 'A' : /* Allocate, Assign */
      /* Do we have an allocate command? */
      if ( ( Current_Token->TokenText == AllocateStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {

        /* Mark Current_Node as being an Allocate command. */
        Current_Node->NodeType = Allocate_Node;

        /* We have an allocate command. Lets consume the token, check for the
           colon which should follow the allocate keyword, and then call
           the Allocation_Parameters function.                                */
        Consume_Token();

        Return_Value = Check_For_Colon(Optional);
        if (Return_Value != Token_Accepted)
          return Return_Value;

        return Allocation_Parameters(Current_Node, Optional);

      }

      /* Do we have an assign command? */
      if ( Current_Token->TokenText == AssignStr )
      {

        /* Mark Current_Node as being an Assign command. */
        Current_Node->NodeType = Assign_Node;

        /* We have an assign command. Lets consume the token, check for the
           colon which should follow the assign keyword, and then call
           the Assign_Parameters function.                                */
        Consume_Token();

        Return_Value = Check_For_Colon(Optional);
        if (Return_Value != Token_Accepted)
          return Return_Value;

        return Assign_Parameters(Current_Node, Optional);

      }

      break;

    case 'C' : /* Check, Create */
      /* Do we have a check command? */
      if ( Current_Token->TokenText == CheckStr )
      {

        /* Mark Current_Node as being a Check command. */
        Current_Node->NodeType = Check_Node;

        Consume_Token();

        Return_Value = Check_For_Colon(Optional);
        if (Return_Value != Token_Accepted)
          return Return_Value;

        return Check_Parameters(Current_Node, Optional);
      }

      /* Do we have a create command? */
      if ( ( Current_Token->TokenText == CreateStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {

        /* Mark Current_Node as being a Create command. */
        Current_Node->NodeType = Create_Node;

        /* We have a create command. Lets consume the token, check for the
           colon which should follow the create keyword, and then call
           the Creation_Parameters function.                                */
        Consume_Token();

        Return_Value = Check_For_Colon(Optional);
        if (Return_Value != Token_Accepted)
          return Return_Value;

        return Creation_Parameters(Current_Node, Optional);
      
      }

      break;

    case 'D' : /* Defragment, Delete */
      /* Do we have a defragment command? */
      if ( Current_Token->TokenText == DefragmentStr )
      {

        /* Mark Current_Node as being a Defragment command. */
        Current_Node->NodeType = Defragment_Node;

        Consume_Token();

        Return_Value = Check_For_Colon(Optional);
        if (Return_Value != Token_Accepted)
          return Return_Value;

        return Defrag_Parameters(Current_Node, Optional);
      }

      /* Do we have a delete command? */
      if ( ( Current_Token->TokenText == DeleteStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {

        /* Mark Current_Node as being a Delete command. */
        Current_Node->NodeType = Delete_Node;

        /* We have a delete command. Lets consume the token, check for the
           colon which should follow the delete keyword, and then call
           the Specify_Object function.                                     */
        Consume_Token();

        Return_Value = Check_For_Colon(Optional);
        if (Return_Value != Token_Accepted )
          return Return_Value;

        return Specify_Object(Current_Node, Translate_Name_To_Handle, FALSE, Optional);

      }

      break;

    case 'E' : /* Expand */
      /* Do we have an expand command? */
      if ( (Current_Token->TokenText == ExpandStr) ||
           (Current_Token->TokenLength == 1)
         )
      {
        /* Mark Current_Node as being a Expand command. */
        Current_Node->NodeType = Expand_Node;

        /* We have an expand command! */
        Consume_Token();

        Return_Value = Check_For_Colon(Optional);
        if (Return_Value != Token_Accepted)
          return Return_Value;

        return Expand_Shrink_Parameters(Current_Node, Optional);

      }

      break;

    case 'F' : /* Format */
      /* Do we have a format command? */
      if ( (Current_Token->TokenText == FormatStr) ||
           (Current_Token->TokenLength == 1)
         )
      {

        /* Mark Current_Node as being a Format command. */
        Current_Node->NodeType = Format_Node;

        /* We have an format command! */
        Consume_Token();

        Return_Value = Check_For_Colon(Optional);
        if (Return_Value != Token_Accepted)
          return Return_Value;

        return Format_Parameters(Current_Node, Optional);

      }

      break;

    case 'H' : /* Help */
      /* Do we have a help command? */
      if ( ( Current_Token->TokenText == HelpStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        /* Mark Current_Node as being a Help command. */
        Current_Node->NodeType = Help_Node;

        /* We have a help command. Lets consume the token, check for the
           colon which should follow the help keyword, and then call
           the Help_Type function.                                        */
        Consume_Token();

        /* Since this is optional, activate Lookahead mode. */
        Temp_Node = Activate_Lookahead();
        if ( Temp_Node == NULL)
          return Error;

        Return_Value = Check_For_Colon(TRUE);
        if (Return_Value == Token_Accepted )
          Return_Value = Help_Type(Current_Node, TRUE);

        /* Restore the current token so that parsing can continue. */
        if ( ( Return_Value != Token_Accepted ) &&
             ( !Restore_Current_Token( Temp_Node ) )
           )
          return Error;

        /* Leave lookahead mode. */
        if (! Terminate_Lookahead(Current_Node, Temp_Node, (Return_Value == Token_Accepted) ) )
          Return_Value = Error;

        /* Since these parameters were optional, indicate success unless Error was returned. */
        if ( Return_Value == Try_Again )
          Return_Value = Token_Accepted;

        return Return_Value;

      }

      break;

    case 'P' : /* Probe */
      /* Do we have a Probe command? */
      if ( ( Current_Token->TokenText == ProbeStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        /* Mark Current_Node as being a Probe command. */
        Current_Node->NodeType = Probe_Node;

        /* We have a Probe command. Lets consume the token.  This command is complete.*/
        Consume_Token();

        return Token_Accepted;

      }

      break;

    case 'Q' : /* Query */
      /* Do we have a query command? */
      if ( ( Current_Token->TokenText == QueryStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {

        /* Mark Current_Node as being a Query command. */
        Current_Node->NodeType = Query_Node;

        /* We have a query command. Lets consume the token, check for the
           colon which should follow the query keyword, and then call
           the Query_Type function.                                        */
        Consume_Token();

        Return_Value = Check_For_Colon(Optional);
        if (Return_Value != Token_Accepted )
          return Return_Value;

        return Query_Type(Current_Node, Optional);

      }

      break;

    case 'R' : /* Rename, Remove, or Revert */

      /* Do we have revert? */
      if ( Current_Token->TokenText == RevertStr )
      {

        /* We have a revert command. Lets consume the token, check for the
           colon which should follow the revert keyword, and then call
           the Specify_Object function.                                */
        Consume_Token();

        /* Mark Current_Node as being a Revert command. */
        Current_Node->NodeType = Revert_Node;

        Return_Value = Check_For_Colon(Optional);
        if (Return_Value != Token_Accepted )
          return Return_Value;

        return Specify_Object(Current_Node, Translate_Name_To_Handle, FALSE, Optional);

      }

      if ( Current_Token->TokenText == RenameStr )
      {
        /* Mark Current_Node as being a Rename command. */
        Current_Node->NodeType = Rename_Node;

        Consume_Token();

        Return_Value = Check_For_Colon(Optional);
        if (Return_Value != Token_Accepted )
          return Return_Value;

        Return_Value = Specify_Object(Current_Node, Translate_Name_To_Handle, FALSE, Optional);
        if (Return_Value != Token_Accepted )
          return Return_Value;

        Return_Value = Check_For_Comma(Optional);
        if (Return_Value != Token_Accepted)
          return Return_Value;

        return Accept_Name(Current_Node, Optional);

      }

      if ( Current_Token->TokenText == RemoveStr )
      {
        /* Mark Current_Node as being a Remove command. */
        Current_Node->NodeType = Remove_Node;

        Consume_Token();

        Return_Value = Check_For_Colon(Optional);
        if (Return_Value != Token_Accepted )
          return Return_Value;

        return Specify_Object(Current_Node, Translate_Name_To_Handle, FALSE, Optional);

      }

      break;

    case 'S' : /* Set, Shrink */
      /* Do we have a shrink command? */
      if ( (Current_Token->TokenText == ShrinkStr) ||
           (Current_Token->TokenLength == 1)
         )
      {
        /* Mark Current_Node as being a Shrink command. */
        Current_Node->NodeType = Shrink_Node;

        Consume_Token();

        Return_Value = Check_For_Colon(Optional);
        if (Return_Value != Token_Accepted)
          return Return_Value;

        return Expand_Shrink_Parameters(Current_Node, Optional);

      }

      /* Do we have a Set command? */
      if ( Current_Token->TokenText == SetStr )
      {
        /* Mark Current_Node as being a Set command. */
        Current_Node->NodeType = Set_Node;

        Consume_Token();

        Return_Value = Check_For_Colon(Optional);
        if (Return_Value != Token_Accepted)
          return Return_Value;

        return Set_Parameters(Current_Node, Translate_Name_To_Handle, Optional);

      }

      break;
    case 'U' :  /* Unformat */
      /* Do we have an Unformat command? */
      if ( (Current_Token->TokenText == UnformatStr) ||
           (Current_Token->TokenLength == 1)
         )
      {
        /* Mark Current_Node as being an Unformat command. */
        Current_Node->NodeType = Unformat_Node;

        Consume_Token();

        Return_Value = Check_For_Colon(Optional);
        if (Return_Value != Token_Accepted)
          return Return_Value;

        return Specify_Object(Current_Node, Name_To_Volume_Handle, FALSE, Optional);

      }
      break;
    default:   /* Error! Unrecognized command! */
      break;

  }

  /* No command found. */
  if (Optional)
    return Try_Again;

  Report_Parser_Error( Expecting_Command_Name, NULL, Current_Token);

  return Error;

}

static Parse_Results Assign_Parameters(Executable_Node * Current_Node, BOOLEAN Optional)
{
  Parse_Results     Return_Value = Error;  /* Used to track the return value from other functions called by this function. */

  /* Get the segment manager to assign, along with its options. */
  Return_Value = Plugin_Identifier(Current_Node, TRUE, TRUE, Optional);

  return Return_Value;

}


static Parse_Results Allocation_Parameters(Executable_Node * Current_Node, BOOLEAN Optional)
{
  Executable_Node * Temp_Node = NULL;      /* Used for lookahead operations. */
  Executable_Node * Child = NULL;          /* Used for the "dummy" object list node.  This object list node will have just one child. */
  Parse_Results     Return_Value = Error;  /* Used to track the return value from other functions called by this function. */

  /* Create the child node and add it to the children list of Current_Node.  This child node will be an object list node. */
  Child = Make_Child_Node(Current_Node, TRUE);
  if ( Child == NULL )
    return Error;

  /* To make the Interpreter code simpler, we will create an object list node to place the single object used by this command in. */
  Child->NodeType = ObjectList;

  /* Get the container/region we are to allocate from. */
  Return_Value = Specify_Object(Child, Translate_Name_To_Handle, FALSE, Optional);
  if (Return_Value == Token_Accepted )
  {
    Return_Value = Check_For_Comma(Optional);
    if ( Return_Value == Token_Accepted )
    {
      /* Get the parameters which tell us how to allocate from the container. */

      Temp_Node = Activate_Lookahead();
      if ( Temp_Node == NULL)
        return Error;

      do
      {
        Return_Value = Parse_Name_Value_Pair(Temp_Node, TRUE);

        if ( Return_Value == Try_Again)
          if ( !Restore_Current_Token( Temp_Node ) )
            return Error;

        if ( ! Terminate_Lookahead(Current_Node, Temp_Node, (Return_Value == Token_Accepted) ) )
          return Error;

        if ( Return_Value == Token_Accepted )
        {
          /* We have just parsed a name-value pair.  Look for another one. */

          Temp_Node = Activate_Lookahead();
          if ( Temp_Node == NULL)
            return Error;

          Return_Value = Check_For_Comma(TRUE);
          if ( Return_Value == Try_Again)
          {
            if ( !Restore_Current_Token( Temp_Node ) )
              return Error;

          }

          if ( ( Return_Value != Token_Accepted ) && ( ! Terminate_Lookahead(Current_Node, Temp_Node, FALSE ) ) )
            return Error;

        }

      } while ( Return_Value == Token_Accepted );

      if ( Return_Value != Error )
        Return_Value = Token_Accepted;

    }

  }

  return Return_Value;

}


static Parse_Results Creation_Parameters(Executable_Node * Current_Node, BOOLEAN Optional)
{
  Executable_Node * Temp_Node = NULL;      /* Used for lookahead operations. */
  Executable_Node * Child = NULL;          /* Used for the "dummy" object list node.  This object list node will have just one child. */
  Parse_Results     Return_Value = Error;  /* Used to track the return value from other functions called by this function. */

  /* Get the token to parse. */
  Next_Token();

  /* Check for EOF or InvalidCharacter */
  if ( ( Current_Token == NULL ) || ( Current_Token->Characterization == EofToken ) || ( Current_Token->Characterization == InvalidCharacter ) )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Creation_Parameters, Command_Filename, Current_Token);

    return Error;

  }

  /* The parameter we are looking for is characterized as a KeyWord.  */
  if ( Current_Token->Characterization != KeyWord )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Creation_Parameters, Command_Filename, Current_Token);

    return Error;

  }

  /* Lets see which creation parameter we have. */
  switch ( Current_Token->TokenText[0] )
  {
    case 'V' : /* Volume */
      if ( ( Current_Token->TokenText == VolumeStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        /* We are going to create a volume.  Mark Current_Node. */
        Current_Node->NodeValue = (void *) VolumeStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        /* The volume keyword should be followed by a comma. */
        Return_Value = Check_For_Comma(Optional);
        if ( Return_Value == Token_Accepted )
        {
          Return_Value = Specify_Object(Current_Node, Translate_Name_To_Handle, FALSE, Optional);   /* Get the storage object which is to become a volume. */

          if ( Return_Value == Token_Accepted )
          {
            Return_Value = Check_For_Comma(Optional);
            if ( Return_Value == Token_Accepted )
            {

              /* We have two possible choices here - we need lookahead to decide which we have. */
              Temp_Node = Activate_Lookahead();
              if ( Temp_Node == NULL)
                return Error;

              Return_Value = Accept_Name(Temp_Node, TRUE);  /* Get the name for the volume. */
              if ( Return_Value == Try_Again)
              {

                /* Restore the current token so that parsing can continue. */
                if ( !Restore_Current_Token( Temp_Node ) )
                  return Error;

                Return_Value = Check_For_Compatibility_Keyword(Temp_Node, TRUE);
                if (Return_Value == Token_Accepted)
                  Current_Node->NodeValue = (void *) CompatibilityStr;
                else if ( Return_Value != Error)
                  Report_Parser_Error(Expecting_Name_Or_Compatibility, Command_Filename, Current_Token);

              }

              /* Exit Lookahead mode. */
              if ( ! Terminate_Lookahead(Current_Node, Temp_Node, (Return_Value == Token_Accepted) ) )
                return Error;

            }

          }

        }

      }
      else
      {
        if (Optional)
          return Try_Again;

        Report_Parser_Error(Expecting_Volume_Parameter,Command_Filename,Current_Token);
        Return_Value = Error;
      }

      break;
    case 'O': /* Object */
      if ( ( Current_Token->TokenText == ObjectStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        /* We are going to create a storage object.  Mark Current_Node. */
        Current_Node->NodeValue = (void *) ObjectStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        /* Check for the comma which follows the Object keyword. */
        Return_Value = Check_For_Comma(Optional);
        if ( Return_Value == Token_Accepted )
        {

          Return_Value = Feature_Identifier(Current_Node, Optional, TRUE, FALSE);

        }

      }
      else
      {
        if (Optional)
          return Try_Again;

        Report_Parser_Error(Expecting_Object_Parameter,Command_Filename,Current_Token);
        Return_Value = Error;
      }

      break;

    case 'C':
      if ( ( Current_Token->TokenText == ContainerStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        /* We are going to create a container.  Mark Current_Node. */
        Current_Node->NodeValue = (void *) ContainerStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        Return_Value = Check_For_Comma(Optional);
        if ( Return_Value == Token_Accepted)
        {

          Return_Value = Plugin_Identifier(Current_Node, TRUE, TRUE, Optional);

        }

      }
      else
      {
        if (Optional)
          return Try_Again;

        Report_Parser_Error(Expecting_Container_Parameter,Command_Filename,Current_Token);
        Return_Value = Error;
      }

      break;

    case 'R' : /* Region */
      if ( ( Current_Token->TokenText == RegionStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        /* We are going to create a region.  Mark Current_Node. */
        Current_Node->NodeValue = (void *) RegionStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        Return_Value = Check_For_Comma(Optional);
        if ( Return_Value == Token_Accepted)
        {

          Return_Value = Plugin_Identifier(Current_Node, TRUE, TRUE, Optional);

        }

      }
      else
      {
        if (Optional)
          return Try_Again;

        Report_Parser_Error(Expecting_Volume_Parameter,Command_Filename,Current_Token);
        Return_Value = Error;
      }

      break;

    case 'S' : /* Segment */
      if ( ( Current_Token->TokenText == SegmentStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        /* We are going to allocate a segment from a block of freespace.  Mark Current_Node. */
        Current_Node->NodeType = Allocate_Node;
        Current_Node->NodeValue = (void *) SegmentStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        Return_Value = Check_For_Comma(Optional);
        if ( Return_Value == Token_Accepted )
        {
          /* Create the child node and add it to the children list of Current_Node.  This child node will be an object list node. */
          Child = Make_Child_Node(Current_Node, TRUE);
          if ( Child == NULL )
            return Error;

          /* To make the Interpreter code simpler, we will create an object list node to place the single object used by this command in. */
          Child->NodeType = ObjectList;

          /* Get the block of freespace to allocate from. */
          Return_Value = Specify_Object(Child, Name_To_Segment_Handle, FALSE, Optional);
          if (Return_Value == Token_Accepted )
          {
            Return_Value = Check_For_Comma(Optional);
            if ( Return_Value == Token_Accepted )
            {
              /* Get the parameters which tell us how to allocate from the block of freespace. */

              Temp_Node = Activate_Lookahead();
              if ( Temp_Node == NULL)
                return Error;

              do
              {
                Return_Value = Parse_Name_Value_Pair(Temp_Node, TRUE);

                if ( Return_Value == Try_Again)
                  if ( !Restore_Current_Token( Temp_Node ) )
                    return Error;

                if ( ! Terminate_Lookahead(Current_Node, Temp_Node, (Return_Value == Token_Accepted) ) )
                  return Error;

                if ( Return_Value == Token_Accepted )
                {
                  /* We have just parsed a name-value pair.  Look for another one. */

                  Temp_Node = Activate_Lookahead();
                  if ( Temp_Node == NULL)
                    return Error;

                  Return_Value = Check_For_Comma(TRUE);
                  if ( Return_Value == Try_Again)
                  {
                    if ( !Restore_Current_Token( Temp_Node ) )
                      return Error;

                  }

                  if ( ( Return_Value != Token_Accepted ) && ( ! Terminate_Lookahead(Current_Node, Temp_Node, FALSE ) ) )
                    return Error;

                }

              } while ( Return_Value == Token_Accepted );

              if ( Return_Value != Error )
                Return_Value = Token_Accepted;

            }

          }

        }

      }
      else
      {
        if (Optional)
          return Try_Again;

        Report_Parser_Error(Expecting_Volume_Parameter,Command_Filename,Current_Token);
        Return_Value = Error;
      }

      break;
    default:
      Report_Parser_Error(Expecting_Creation_Parameters, Command_Filename, Current_Token);

      return Error;

      break;

  }

  return Return_Value;

}


static Parse_Results Check_For_Compatibility_Keyword(Executable_Node * Current_Node, BOOLEAN Optional)
{
  /* Get the token to parse. */
  Next_Token();

  /* Check for EOF or InvalidCharacter */
  if ( ( Current_Token == NULL ) ||
       ( Current_Token->Characterization == EofToken ) ||
       ( Current_Token->Characterization == InvalidCharacter ) ||
       ( Current_Token->Characterization != KeyWord )
     )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Compatibility_Keyword, Command_Filename, Current_Token);

    return Error;

  }

  if ( ( Current_Token->TokenText == CompatibilityStr ) ||
       ( ( Current_Token->TokenLength == 1 ) &&
         ( Current_Token->TokenText[0] == 'C' )
       )
     )
  {

    /* We are through with this token, so consume it. */
    Consume_Token();

    return Token_Accepted;

  }
  else
  {
    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Compatibility_Keyword,Command_Filename,Current_Token);
  }

  return Error;

}


static Parse_Results Check_Parameters(Executable_Node * Current_Node, BOOLEAN Optional)
{
  return Set_Parameters(Current_Node, Name_To_Volume_Handle, Optional);
}


static Parse_Results Defrag_Parameters(Executable_Node * Current_Node, BOOLEAN Optional)
{
  return Set_Parameters(Current_Node, Name_To_Volume_Handle, Optional);
}


static Parse_Results Format_Parameters(Executable_Node * Current_Node, BOOLEAN Optional)
{
  Parse_Results     Return_Value = Error;  /* Used to track the return value from other functions called by this function. */

  /* Get the segment manager to assign, along with its options. */
  Return_Value = Plugin_Identifier(Current_Node, TRUE, TRUE, Optional);

  return Return_Value;

}


static Parse_Results Set_Parameters(Executable_Node * Current_Node, NodeTypes Node_To_Create, BOOLEAN Optional)
{
  Executable_Node * Temp_Node = NULL;     /* Used for lookahead operations. */
  Parse_Results     Return_Value = Error; /* Used to track the return value from other functions called by this function. */

  Return_Value = Specify_Object(Current_Node, Node_To_Create, FALSE, Optional);
  if (Return_Value != Token_Accepted)
    return Return_Value;

  /* Get however many name-value pairs there may be. */
  do
  {
    /* The following name-value pairs are optional.*/
    Temp_Node = Activate_Lookahead();
    if ( Temp_Node == NULL)
      return Error;

    /* Check for a comma - if found, a name-value pair may follow. */
    Return_Value = Check_For_Comma(TRUE);
    if ( Return_Value == Token_Accepted )
      Return_Value = Parse_Name_Value_Pair(Temp_Node, TRUE);

    if ( Return_Value == Try_Again)
    {
      /* Restore the current token so that parsing can continue. */
      if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }

    /* Exit Lookahead mode. */
    if ( ! Terminate_Lookahead(Current_Node, Temp_Node, (Return_Value == Token_Accepted) ) )
      return Error;

  } while ( Return_Value == Token_Accepted );

  if (Return_Value == Error )
    return Error;
  else
    return Token_Accepted;

}

static Parse_Results Expand_Shrink_Parameters(Executable_Node * Current_Node, BOOLEAN Optional)
{
  Executable_Node * Temp_Node = NULL;     /* Used for lookahead operations. */
  Parse_Results     Return_Value = Error; /* Used to track the return value from other functions called by this function. */

  Return_Value = Specify_Object(Current_Node, Translate_Name_To_Handle, FALSE, Optional);
  if (Return_Value != Token_Accepted)
    return Return_Value;

  /* The expansion/shrink point may be followed by a series of Name-Value pairs. */

  /* Get however many name-value pairs there may be. */
  do
  {
    /* The following name-value pairs are optional.*/
    Temp_Node = Activate_Lookahead();
    if ( Temp_Node == NULL)
      return Error;

    /* Check for a comma - if found, a name-value pair may follow. */
    Return_Value = Check_For_Comma(TRUE);
    if ( Return_Value == Token_Accepted )
      Return_Value = Parse_Name_Value_Pair(Temp_Node, TRUE);

    if ( Return_Value == Try_Again)
    {
      /* Restore the current token so that parsing can continue. */
      if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }

    /* Exit Lookahead mode. */
    if ( ! Terminate_Lookahead(Current_Node, Temp_Node, (Return_Value == Token_Accepted) ) )
      return Error;

  } while ( Return_Value == Token_Accepted );

  if ( Return_Value == Error )
    return Return_Value;

  /* Now we must get any objects to be used for the expansion. */
  Temp_Node = Activate_Lookahead();
  if ( Temp_Node == NULL)
    return Error;

  Return_Value = Parse_Object_List(Temp_Node,TRUE);

  if ( Return_Value == Try_Again)
  {
    /* Restore the current token so that parsing can continue. */
    if ( !Restore_Current_Token( Temp_Node ) )
      return Error;
  }

  /* Exit Lookahead mode. */
  if ( ! Terminate_Lookahead(Current_Node, Temp_Node, (Return_Value == Token_Accepted) ) )
    return Error;

  /* Since the specification of storage objects or name value pairs is optional in some circumstances, 
     a Return_Value of Try_Again is acceptable and we will thus translate it to Token_Accepted.         */
  if ( Return_Value != Error )
    Return_Value = Token_Accepted;

  return Return_Value;

}


static Parse_Results Query_Type(Executable_Node * Current_Node, BOOLEAN Optional)
{
  Executable_Node * Temp_Node = NULL;  /* Used for lookahead operations. */
  Parse_Results     Return_Value;      /* Used to track the return value from other functions called by this function. */

  /* Get the token to parse. */
  Next_Token();

  /* Check for EOF or InvalidCharacter */
  if ( ( Current_Token == NULL ) || ( Current_Token->Characterization == EofToken ) || ( Current_Token->Characterization == InvalidCharacter ) )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Query_Parameters, Command_Filename, Current_Token);

    return Error;

  }

  /* The parameter we are looking for is characterized as a KeyWord.  */
  if ( Current_Token->Characterization != KeyWord )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Query_Parameters, Command_Filename, Current_Token);

    return Error;

  }

  /* Lets see which query parameter we have. */
  switch ( Current_Token->TokenText[0] )
  {
    case 'P' : /* Plugins or Parent */
      if ( ( Current_Token->TokenText == PluginsStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        /* We are going to produce a query about plugins.  Mark Current_Node. */
        Current_Node->NodeValue = (void *) PluginsStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        /* The remaining parameters are optional.  Lets see if any were specified. */
        Temp_Node = Activate_Lookahead();
        if ( Temp_Node == NULL)
          return Error;

        /* Check for a comma - if found, optional parameters should follow. */
        Return_Value = Check_For_Comma(TRUE);
        if ( Return_Value == Token_Accepted )
          Return_Value = Parse_Query_Plugins_Parameters(Temp_Node, TRUE);

        if ( Return_Value == Error )
          break;

        if ( Return_Value == Try_Again)
        {

          /* Restore the current token so that parsing can continue. */
          if ( !Restore_Current_Token( Temp_Node ) )
            return Error;

        }

        /* Exit Lookahead mode. */
        if ( ! Terminate_Lookahead(Current_Node, Temp_Node, (Return_Value == Token_Accepted) ) )
          return Error;

        if ( Return_Value != Error)
        {

          /* Check for the List Options parameter, which is optional. */
          Temp_Node = Activate_Lookahead();
          if ( Temp_Node == NULL)
            return Error;

          /* Check for a comma - if found, optional parameters should follow. */
          Return_Value = Check_For_Comma(TRUE);
          if ( Return_Value == Token_Accepted )
            Return_Value = Parse_Query_List_Options(Temp_Node, TRUE);

          if ( Return_Value == Error )
            break;

          if ( Return_Value == Try_Again)
          {

            /* Restore the current token so that parsing can continue. */
            if ( !Restore_Current_Token( Temp_Node ) )
              return Error;

          }

          /* Exit Lookahead mode. */
          if ( ! Terminate_Lookahead(Current_Node, Temp_Node, (Return_Value == Token_Accepted) ) )
            return Error;


          Return_Value = Token_Accepted;   /* The command is complete. */

        }

      }
      else
      {

        if ( Current_Token->TokenText == ParentStr )
        {
          /* We are going to produce a query about an objects parent.  Mark Current_Node. */
          Current_Node->NodeValue = (void *) ParentStr;
          Current_Node->NodeType = Query_Node_Type_2;

          /* We are through with this token, so consume it. */
          Consume_Token();

          /* Check for a comma. */
          Return_Value = Check_For_Comma(Optional);

          if ( Return_Value == Error )
            break;

          if ( Return_Value == Try_Again )
          {

            if ( Optional )
              return Try_Again;

            Report_Parser_Error(Expecting_Region_Or_Object, Command_Filename, Current_Token);
            Return_Value = Error;

          }
          else
            Return_Value = Specify_Object(Current_Node, Translate_Name_To_Handle, FALSE, Optional);

        }
        else
        {

          if (Optional)
            return Try_Again;

          Report_Parser_Error(Expecting_Plugins_Or_Parent_Parameter,Command_Filename,Current_Token);
          Return_Value = Error;

        }

      }

      break;

    case 'V' : /* Volume */
      if ( ( Current_Token->TokenText == VolumesStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        /* We are going to produce a query about volumes.  Mark Current_Node. */
        Current_Node->NodeValue = (void *) VolumesStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        /* The remaining parameters are optional.  Lets see if any were specified. */
        Temp_Node = Activate_Lookahead();
        if ( Temp_Node == NULL)
          return Error;

        /* Check for a comma - if found, optional parameters should follow. */
        Return_Value = Check_For_Comma(TRUE);
        if ( Return_Value == Token_Accepted )
          Return_Value = Parse_Query_Volumes_Parameters(Temp_Node, TRUE);

        if ( Return_Value == Error )
          break;

        if ( Return_Value == Try_Again)
        {

          /* Restore the current token so that parsing can continue. */
          if ( !Restore_Current_Token( Temp_Node ) )
            return Error;

        }

        /* Exit Lookahead mode. */
        if ( ! Terminate_Lookahead(Current_Node, Temp_Node, (Return_Value == Token_Accepted) ) )
          return Error;

        Return_Value = Token_Accepted;   /* The command is complete. */

      }
      else
      {

        if (Optional)
          return Try_Again;

        Report_Parser_Error(Expecting_Volumes_Parameter,Command_Filename,Current_Token);
        Return_Value = Error;

      }

      break;
    case 'R' : /* Region */
      if ( ( Current_Token->TokenText == RegionsStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        /* We are going to produce a query about regions.  Mark Current_Node. */
        Current_Node->NodeValue = (void *) RegionsStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        /* The remaining parameters are optional.  Lets see if any were specified. */
        Temp_Node = Activate_Lookahead();
        if ( Temp_Node == NULL)
          return Error;

        /* Check for a comma - if found, optional parameters should follow. */
        Return_Value = Check_For_Comma(TRUE);
        if ( Return_Value == Token_Accepted )
          Return_Value = Parse_Query_Regions_Parameters(Temp_Node, TRUE);

        if ( Return_Value == Error )
          break;

        if ( Return_Value == Try_Again)
        {

          /* Restore the current token so that parsing can continue. */
          if ( !Restore_Current_Token( Temp_Node ) )
            return Error;

        }

        /* Exit Lookahead mode. */
        if ( ! Terminate_Lookahead(Current_Node, Temp_Node, (Return_Value == Token_Accepted) ) )
          return Error;

        Return_Value = Token_Accepted;   /* The command is complete. */

      }
      else
      {

        if (Optional)
          return Try_Again;

        Report_Parser_Error(Expecting_Regions_Parameter,Command_Filename,Current_Token);
        Return_Value = Error;

      }

      break;
    case 'D' : /* Disks */
      if ( ( Current_Token->TokenText == DisksStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        /* We are going to produce a query about the disks in the system.  Mark Current_Node. */
        Current_Node->NodeValue = (void *) DisksStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        /* The remaining parameters are optional.  Lets see if any were specified. */
        Temp_Node = Activate_Lookahead();
        if ( Temp_Node == NULL)
          return Error;

        /* Check for a comma - if found, optional parameters should follow. */
        Return_Value = Check_For_Comma(TRUE);
        if ( Return_Value == Token_Accepted )
          Return_Value = Parse_Query_Disks_Parameters(Temp_Node, TRUE);

        if ( Return_Value == Error )
          break;

        if ( Return_Value == Try_Again)
        {

          /* Restore the current token so that parsing can continue. */
          if ( !Restore_Current_Token( Temp_Node ) )
            return Error;

        }

        /* Exit Lookahead mode. */
        if ( ! Terminate_Lookahead(Current_Node, Temp_Node, (Return_Value == Token_Accepted) ) )
          return Error;

        Return_Value = Token_Accepted;   /* The command is complete. */

      }
      else
      {

        if (Optional)
          return Try_Again;

        Report_Parser_Error(Expecting_Disks_Parameter,Command_Filename,Current_Token);
        Return_Value = Error;

      }

      break;
    case 'S' : /* Segments, Shrink Points */
      if ( ( Current_Token->TokenText == SegmentsStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        /* We are going to produce a query about the disk segments in the system.  Mark Current_Node. */
        Current_Node->NodeValue = (void *) SegmentsStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        /* The remaining parameters are optional.  Lets see if any were specified. */
        Temp_Node = Activate_Lookahead();
        if ( Temp_Node == NULL)
          return Error;

        /* Check for a comma - if found, optional parameters should follow. */
        Return_Value = Check_For_Comma(TRUE);
        if ( Return_Value == Token_Accepted )
          Return_Value = Parse_Query_Segments_Parameters(Temp_Node, TRUE);

        if ( Return_Value == Error )
          break;

        if ( Return_Value == Try_Again)
        {

          /* Restore the current token so that parsing can continue. */
          if ( !Restore_Current_Token( Temp_Node ) )
            return Error;

        }

        /* Exit Lookahead mode. */
        if ( ! Terminate_Lookahead(Current_Node, Temp_Node, (Return_Value == Token_Accepted) ) )
          return Error;

        Return_Value = Token_Accepted;   /* The command is complete. */

      }
      else if ( Current_Token->TokenText == SPStr )
      {
        /* We are going to query the shrink points of a volume or object. */
        Current_Node->NodeValue = (void *) SPStr;
        Current_Node->NodeType = Query_Node_Type_2;
        Consume_Token();

        /* Check for a comma. */
        Return_Value = Check_For_Comma(Optional);

        if ( Return_Value == Error )
          break;

        if ( Return_Value == Try_Again )
        {

          if ( Optional )
            return Try_Again;

          Report_Parser_Error(Expecting_Volume_Or_Object, Command_Filename, Current_Token);
          Return_Value = Error;

        }
        else
          Return_Value = Specify_Object(Current_Node, Translate_Name_To_Handle, FALSE, Optional);

      }
      else if ( Current_Token->TokenText == ShrinkStr )
      {

        Consume_Token();
        Next_Token();

        if ( ( Current_Token != NULL ) &&
             ( Current_Token->Characterization == KeyWord ) &&
             ( Current_Token->TokenText == PointsStr )
           )
        {
          /* We are going to query the shrink points of a volume or object. */
          Current_Node->NodeValue = (void *) SPStr;
          Current_Node->NodeType = Query_Node_Type_2;
          Consume_Token();

          /* Check for a comma. */
          Return_Value = Check_For_Comma(Optional);

          if ( Return_Value == Error )
            break;

          if ( Return_Value == Try_Again )
          {

            if ( Optional )
              return Try_Again;

            Report_Parser_Error(Expecting_Volume_Or_Object, Command_Filename, Current_Token);
            Return_Value = Error;

          }
          else
            Return_Value = Specify_Object(Current_Node, Translate_Name_To_Handle, FALSE, Optional);

        }
        else
        {
          if (Optional)
            return Try_Again;

          Report_Parser_Error(Expecting_Query_Shrink_Points_Keywords, Command_Filename, Current_Token);

          return Error;

        }

      }
      else
      {

        if (Optional)
          return Try_Again;

        Report_Parser_Error(Expecting_Query_Parameters, Command_Filename, Current_Token);

        return Error;

      }


      break;

    case 'A' : /* Acceptable */
      if ( ( Current_Token->TokenText == AcceptableStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        /* We are going to produce a query about what is acceptable to a plugin.  Mark Current_Node. */
        Current_Node->NodeValue = (void *) AcceptableStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        /* Check for a comma. */
        Return_Value = Check_For_Comma(Optional);

        if ( Return_Value == Error )
          break;

        if ( Return_Value == Try_Again)
        {

          if (Optional)
            return Try_Again;

          Report_Parser_Error(Expecting_Acceptable_Parameter,Command_Filename,Current_Token);
          Return_Value = Error;

        }

        /* Now we know that a parameter should follow. */
        Return_Value = Parse_Query_Acceptable_Parameters(Current_Node, Optional);

      }
      else
      {

        if (Optional)
          return Try_Again;

        Report_Parser_Error(Expecting_Acceptable_Parameter,Command_Filename,Current_Token);
        Return_Value = Error;

      }

      break;
    case 'E' : /* Expand Points, Extended Info */
      if ( Current_Token->TokenText == EPStr )
      {
        /* We are going to query the expansion points of a volume or object. */
        Current_Node->NodeValue = (void *) EPStr;
        Current_Node->NodeType = Query_Node_Type_2;
        Consume_Token();

        /* Check for a comma. */
        Return_Value = Check_For_Comma(Optional);

        if ( Return_Value == Error )
          break;

        if ( Return_Value == Try_Again )
        {

          if ( Optional )
            return Try_Again;

          Report_Parser_Error(Expecting_Volume_Or_Object, Command_Filename, Current_Token);
          Return_Value = Error;

        }
        else
          Return_Value = Specify_Object(Current_Node, Translate_Name_To_Handle, FALSE, Optional);

      }
      else if ( Current_Token->TokenText == ExpandStr )
      {

        Consume_Token();
        Next_Token();

        if ( ( Current_Token != NULL ) &&
             ( Current_Token->Characterization == KeyWord ) &&
             ( Current_Token->TokenText == PointsStr )
           )
        {
          /* We are going to query the expansion points of a volume or object. */
          Current_Node->NodeValue = (void *) EPStr;
          Current_Node->NodeType = Query_Node_Type_2;
          Consume_Token();

          /* Check for a comma. */
          Return_Value = Check_For_Comma(Optional);

          if ( Return_Value == Error )
            break;

          if ( Return_Value == Try_Again )
          {

            if ( Optional )
              return Try_Again;

            Report_Parser_Error(Expecting_Volume_Or_Object, Command_Filename, Current_Token);
            Return_Value = Error;

          }
          else
            Return_Value = Specify_Object(Current_Node, Translate_Name_To_Handle, FALSE, Optional);

        }
        else
        {
          if (Optional)
            return Try_Again;

          Report_Parser_Error(Expecting_Query_Expand_Points_Keywords, Command_Filename, Current_Token);

          return Error;

        }

      }
      else if ( Current_Token->TokenText == EIStr )
      {
        /* We are going to query the extended info of something. */
        Current_Node->NodeValue = (void *) NULL;
        Current_Node->NodeType = Query_Extended_Info;
        Consume_Token();

        Return_Value = Parse_Query_Extended_Info_Parameters(Current_Node,Optional);

      }
      else if ( Current_Token->TokenText == ExtendedStr )
      {

        Consume_Token();
        Next_Token();

        if ( ( Current_Token != NULL ) &&
             ( Current_Token->Characterization == KeyWord ) &&
             ( Current_Token->TokenText == InfoStr )
           )
        {
          /* We are going to query the extended info of something. */
          Current_Node->NodeValue = (void *) NULL;
          Current_Node->NodeType = Query_Extended_Info;
          Consume_Token();

          Return_Value = Parse_Query_Extended_Info_Parameters(Current_Node,Optional);

        }
        else
        {
          if (Optional)
            return Try_Again;

          Report_Parser_Error(Expecting_Query_Extended_Info_Keywords, Command_Filename, Current_Token);

          return Error;

        }

      }
      else
      {

        if (Optional)
          return Try_Again;

        Report_Parser_Error(Expecting_Query_Parameters, Command_Filename, Current_Token);

        return Error;

      }

      break;
    case 'C' : /* Containers or Childeren. */
      if ( ( Current_Token->TokenText == ContainersStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        /* We are going to produce a query about containers.  Mark Current_Node. */
        Current_Node->NodeValue = (void *) ContainersStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        /* The remaining parameters are optional.  Lets see if any were specified. */
        Temp_Node = Activate_Lookahead();
        if ( Temp_Node == NULL)
          return Error;

        /* Check for a comma - if found, optional parameters should follow. */
        Return_Value = Check_For_Comma(TRUE);
        if ( Return_Value == Token_Accepted )
          Return_Value = Parse_Query_Containers_Parameters(Temp_Node, TRUE);

        if ( Return_Value == Error )
          break;

        if ( Return_Value == Try_Again)
        {

          /* Restore the current token so that parsing can continue. */
          if ( !Restore_Current_Token( Temp_Node ) )
            return Error;

        }

        /* Exit Lookahead mode. */
        if ( ! Terminate_Lookahead(Current_Node, Temp_Node, (Return_Value == Token_Accepted) ) )
          return Error;

        Return_Value = Token_Accepted;   /* The command is complete. */

      }
      else
      {

        if ( Current_Token->TokenText == ChildrenStr )
        {
          /* We are going to produce a query about the Children of an object.  Mark Current_Node. */
          Current_Node->NodeValue = (void *) ChildrenStr;
          Current_Node->NodeType = Query_Node_Type_2;

          /* We are through with this token, so consume it. */
          Consume_Token();

          /* Check for a comma. */
          Return_Value = Check_For_Comma(Optional);

          if ( Return_Value == Error )
            break;

          if ( Return_Value == Try_Again )
          {

            if ( Optional )
              return Try_Again;

            Report_Parser_Error(Expecting_Volume_Or_Object, Command_Filename, Current_Token);
            Return_Value = Error;

          }
          else
            Return_Value = Specify_Object(Current_Node, Translate_Name_To_Handle, FALSE, Optional);

        }
        else
        {

          if (Optional)
            return Try_Again;

          Report_Parser_Error(Expecting_Containers_Or_Children_Parameter,Command_Filename,Current_Token);
          Return_Value = Error;

        }

      }

      break;
    case 'O': /* Objects */
      if ( ( Current_Token->TokenText == ObjectsStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        /* We are going to produce a query about the storage objects in the system.  Mark Current_Node. */
        Current_Node->NodeValue = (void *) ObjectsStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        /* The remaining parameters are optional.  Lets see if any were specified. */
        Temp_Node = Activate_Lookahead();
        if ( Temp_Node == NULL)
          return Error;

        /* Check for a comma - if found, optional parameters should follow. */
        Return_Value = Check_For_Comma(TRUE);

        if ( Return_Value == Token_Accepted )
        {

          /* Now we know that an optional parameter follows. */
          Return_Value = Parse_Query_Objects_Parameters(Temp_Node, TRUE);

        }

        if ( Return_Value == Error )
          break;

        if ( Return_Value == Try_Again)
        {

          /* Restore the current token so that parsing can continue. */
          if ( !Restore_Current_Token( Temp_Node ) )
            return Error;

        }

        /* Exit Lookahead mode. */
        if ( ! Terminate_Lookahead(Current_Node, Temp_Node, (Return_Value == Token_Accepted) ) )
          return Error;

        Return_Value = Token_Accepted;   /* The command is complete. */

      }
      else
      {

        if (Optional)
          return Try_Again;

        Report_Parser_Error(Expecting_Objects_Parameter,Command_Filename,Current_Token);
        Return_Value = Error;

      }

      break;

    default:
      if (Optional)
        Return_Value = Try_Again;
      else
      {
        Return_Value = Error;
        Report_Parser_Error(Expecting_Query_Parameters,Command_Filename,Current_Token);
      }
      break;

  }

  return Return_Value;

}

static Parse_Results Help_Type(Executable_Node * Current_Node, BOOLEAN Optional)
{
  Parse_Results     Return_Value = Error;     /* Used to track the return value from other functions called by this function. */
  BOOLEAN           Help_Topic_Found = FALSE;

  /* Get the token to parse. */
  Next_Token();

  /* Check for EOF or InvalidCharacter */
  if ( ( Current_Token == NULL ) || ( Current_Token->Characterization == EofToken ) || ( Current_Token->Characterization == InvalidCharacter ) )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Help_Parameters, Command_Filename, Current_Token);

    return Error;

  }

  /* The parameter we are looking for is characterized as a KeyWord.  */
  if ( Current_Token->Characterization != KeyWord )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Help_Parameters, Command_Filename, Current_Token);

    return Error;

  }

  /* Lets see which expansion parameters we have. */
  switch ( Current_Token->TokenText[0] )
  {
    case 'A' :
      if ( ( Current_Token->TokenText == AllocateStr ) || ( Current_Token->TokenLength == 1 ) )
      {

        Help_Topic_Found = TRUE;

        /* We need help for the allocate command. */
        Current_Node->NodeValue = (void *) AllocateStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        Return_Value = Token_Accepted;

      }
      else
      {

        if ( Current_Token->TokenText == AssignStr )
        {
          
          Help_Topic_Found = TRUE;
          
          /* We need help for the assign command. */
          Current_Node->NodeValue = (void *) AssignStr;

          /* We are through with this token, so consume it. */
          Consume_Token();

          Return_Value = Token_Accepted;

        }

      }
      break;
    case 'C' : /* Check, Create */
      if ( ( Current_Token->TokenText == CreateStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        
        Help_Topic_Found = TRUE;

        /* We need help for the create command. */
        Current_Node->NodeValue = (void *) CreateStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        Return_Value = Token_Accepted;

      }
      else if ( Current_Token->TokenText == CheckStr ) 
      {

        Help_Topic_Found = TRUE;

        /* We need help for the check command. */
        Current_Node->NodeValue = (void *) CheckStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        Return_Value = Token_Accepted;

      }

      break;

    case 'D' : /* Defragment, Delete */
      if ( ( Current_Token->TokenText == DeleteStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        
        Help_Topic_Found = TRUE;
        
        /* We need help for the delete command. */
        Current_Node->NodeValue = (void *) DeleteStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        Return_Value = Token_Accepted;

      }
      else  if ( Current_Token->TokenText == DefragmentStr ) 
      {
        
        Help_Topic_Found = TRUE;
        
        /* We need help for the defragment command. */
        Current_Node->NodeValue = (void *) DefragmentStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        Return_Value = Token_Accepted;

      }

      break;
    case 'E' : /* Expand */
      if ( ( Current_Token->TokenText == ExpandStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        
        Help_Topic_Found = TRUE;
        
        /* We need help for the expand command. */
        Current_Node->NodeValue = (void *) ExpandStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        Return_Value = Token_Accepted;

      }

      break;
    case 'F' : /* Format */
      if ( ( Current_Token->TokenText == FormatStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        
        Help_Topic_Found = TRUE;
        
        /* We need help for the format command. */
        Current_Node->NodeValue = (void *) FormatStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        Return_Value = Token_Accepted;

      }

      break;
    case 'S' : /* Set, Shrink */
      if ( ( Current_Token->TokenText == ShrinkStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        
        Help_Topic_Found = TRUE;
        
        /* We need help for the create command. */
        Current_Node->NodeValue = (void *) ShrinkStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        Return_Value = Token_Accepted;

      }
      else if ( Current_Token->TokenText == SetStr ) 
      {
        
        Help_Topic_Found = TRUE;
        
        /* We need help for the set command. */
        Current_Node->NodeValue = (void *) SetStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        Return_Value = Token_Accepted;

      }

      break;
    case 'Q' : /* Query */
      if ( ( Current_Token->TokenText == QueryStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        
        Help_Topic_Found = TRUE;
        
        /* We need help for the create command. */
        Current_Node->NodeValue = (void *) QueryStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        Return_Value = Token_Accepted;

      }

      break;
    case 'P' :
      if ( ( Current_Token->TokenText == ProbeStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        
        Help_Topic_Found = TRUE;
        
        /* We need help for the create command. */
        Current_Node->NodeValue = (void *) ProbeStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        Return_Value = Token_Accepted;

      }

      break;
    case 'R' : /* Remove, Rename or Revert */
      if ( Current_Token->TokenText == RenameStr )
      {

        Help_Topic_Found = TRUE;
        
        /* We need help for the rename command. */
        Current_Node->NodeValue = (void *) RenameStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        Return_Value = Token_Accepted;

      }
      else
      {

        if ( Current_Token->TokenText == RevertStr )
        {
          
          Help_Topic_Found = TRUE;
          
          /* We need help for the revert command. */
          Current_Node->NodeValue = (void *) RevertStr;

          /* We are through with this token, so consume it. */
          Consume_Token();

          Return_Value = Token_Accepted;

        }
        else
        {

          if ( Current_Token->TokenText == RemoveStr )
          {
            
            Help_Topic_Found = TRUE;
            
            /* We need help for the remove command. */
            Current_Node->NodeValue = (void *) RemoveStr;

            /* We are through with this token, so consume it. */
            Consume_Token();

            Return_Value = Token_Accepted;

          }

        }

      }
      break;
    case 'U' : /* Unformat */
      if ( ( Current_Token->TokenText == UnformatStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        
        Help_Topic_Found = TRUE;
        
        /* We need help for the unformat command. */
        Current_Node->NodeValue = (void *) UnformatStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        Return_Value = Token_Accepted;

      }

      break;    
    default:
      break;

  }

  if ( ! Help_Topic_Found )
  {
    
    /* Use standard error handling */
    if (Optional)
      Return_Value = Try_Again;
    else
    {

      Report_Parser_Error(Expecting_Help_Parameters,Command_Filename,Current_Token);
      Return_Value = Error;

    }
    
  }

  return Return_Value;

}

static Parse_Results Plugin_Types(Executable_Node * Current_Node, BOOLEAN Use_Current_Token, BOOLEAN Optional)
{
  Executable_Node * Child = NULL;         /* The node used to represent the current command being parsed. */
  Parse_Results     Return_Value = Error; /* Used to track the return value from other functions called by this function. */

  /* Create the child node and add it to the children list of Current_Node.  We will fill in the contents of Child later. */
  Child = Make_Child_Node(Current_Node, TRUE);
  if ( Child == NULL )
    return Error;

  Child->NodeType = Plugin_Mask;

  /* Get the token to parse. */
  if ( ! Use_Current_Token )
    Next_Token();

  /* Check for EOF, InvalidCharacter, or bad token characterization. */
  if ( ( Current_Token == NULL ) ||
       ( Current_Token->Characterization == EofToken ) ||
       ( Current_Token->Characterization == InvalidCharacter ) ||
       ( Current_Token->Characterization != KeyWord )
     )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Plugin_Type_Parameter, Command_Filename, Current_Token);

    return Error;

  }

  /* Lets see which what kind of plugin we are after. */
  switch ( Current_Token->TokenText[0] )
  {
    case 'C' : /* Cluster Management, CM */
      if ( Current_Token->TokenText == CMStr )
      {
        /* We are after a Cluster Management plugin. */
        Child->NodeValue = (void *) CMStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        Return_Value = Token_Accepted;

        break;

      }

      if ( Current_Token->TokenText == ClusterStr )
      {

        /* We have the start of the Cluster Manager plugin type. */
        Consume_Token();

        /* Get the next token, which we hope will be the Management keyword. */
        Next_Token();

        /* Check for EOF or InvalidCharacter */
        if ( ( Current_Token == NULL ) ||
             ( Current_Token->Characterization == EofToken ) ||
             ( Current_Token->Characterization == InvalidCharacter )
           )
        {

          if (Optional)
            return Try_Again;

          Report_Parser_Error(Expecting_Plugin_Type_Parameter, Command_Filename, Current_Token);

          return Error;

        }

        if ( ( Current_Token->Characterization == KeyWord ) &&
             ( Current_Token->TokenText == ManagementStr )
           )
        {

          /* We are after a Cluster Management plugin. */
          Child->NodeValue = (void *) CMStr;

          /* We are through with this token, so consume it. */
          Consume_Token();

          Return_Value = Token_Accepted;

          break;

        }

      }

      /* No matching keyword. */

      if (Optional)
        return Try_Again;

      Report_Parser_Error(Expecting_Plugin_Type_Parameter,Command_Filename,Current_Token);
      Return_Value = Error;

      break;

    case 'D' : /* Device Manager, DLM, Distributed Lock Management */
      if ( Current_Token->TokenLength == 1 )
      {
        /* We are after a Device Manager plugin. */
        Child->NodeValue = (void *) DeviceStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        Return_Value = Token_Accepted;

        break;


      }

      if ( Current_Token->TokenText == DLMStr )
      {
        /* We are after a Distributed Lock Management plugin. */
        Child->NodeValue = (void *) DLMStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        Return_Value = Token_Accepted;

        break;

      }

      if ( Current_Token->TokenText == DeviceStr )
      {

        /* We have the start of the Device Manager plugin type. */
        Consume_Token();

        /* Get the next token, which we hope will be the Manager keyword. */
        Next_Token();

        if ( ( Current_Token->Characterization == KeyWord ) &&
             ( Current_Token->TokenText == ManagerStr )
           )
        {

          /* We are after a Device Manager plugin. */
          Child->NodeValue = (void *) DeviceStr;

          /* We are through with this token, so consume it. */
          Consume_Token();

          Return_Value = Token_Accepted;

          break;

        }

      }
      else
      {

        if ( Current_Token->TokenText == DistributedStr )
        {

          /* We have the start of the Distributed Lock Management plugin type. */
          Consume_Token();

          /* Get the next token, which we hope will be the Lock keyword. */
          Next_Token();

          /* Check for EOF or InvalidCharacter */
          if ( ( Current_Token == NULL ) || ( Current_Token->Characterization == EofToken ) || ( Current_Token->Characterization == InvalidCharacter ) )
          {

            if (Optional)
              return Try_Again;

            Report_Parser_Error(Expecting_Plugin_Type_Parameter, Command_Filename, Current_Token);

            return Error;

          }

          /* The parameter we are looking for is characterized as a KeyWord.  */
          if ( Current_Token->Characterization != KeyWord )
          {

            if (Optional)
              return Try_Again;

            Report_Parser_Error(Expecting_Plugin_Type_Parameter, Command_Filename, Current_Token);

            return Error;

          }

          if ( Current_Token->TokenText == LockStr )
          {

            /* We have two parts of the Distributed Lock Management plugin type. */
            Consume_Token();

            /* Get the next token, which we hope will be the Management keyword. */
            Next_Token();

            /* Check for EOF or InvalidCharacter */
            if ( ( Current_Token == NULL ) || ( Current_Token->Characterization == EofToken ) || ( Current_Token->Characterization == InvalidCharacter ) )
            {

              if (Optional)
                return Try_Again;

              Report_Parser_Error(Expecting_Plugin_Type_Parameter, Command_Filename, Current_Token);

              return Error;

            }

            /* The parameter we are looking for is characterized as a KeyWord.  */
            if ( Current_Token->Characterization != KeyWord )
            {

              if (Optional)
                return Try_Again;

              Report_Parser_Error(Expecting_Plugin_Type_Parameter, Command_Filename, Current_Token);

              return Error;

            }

            if ( Current_Token->TokenText == ManagerStr )
            {

              /* We are after a DLM plugin. */
              Child->NodeValue = (void *) DLMStr;

              /* We are through with this token, so consume it. */
              Consume_Token();

              Return_Value = Token_Accepted;

              break;

            }

          }

        }

      }

      /* No matching keyword. */

      if (Optional)
        return Try_Again;

      Report_Parser_Error(Expecting_Plugin_Type_Parameter,Command_Filename,Current_Token);
      Return_Value = Error;

      break;

    case 'R' : /* Region Manager */
      if ( Current_Token->TokenLength == 1 )
      {

        /* We are after a Segment Manager plugin. */
        Child->NodeValue = (void *) RegionStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        Return_Value = Token_Accepted;

        break;

      }

      if ( Current_Token->TokenText == RegionStr )
      {

        /* We have the start of the Region Manager plugin type. */
        Consume_Token();

        /* Get the next token, which we hope will be the Manager keyword. */
        Next_Token();

        if ( ( Current_Token->Characterization == KeyWord ) &&
             ( Current_Token->TokenText == ManagerStr )
           )
        {

          /* We are after a Region Manager plugin. */
          Child->NodeValue = (void *) RegionStr;

          /* We are through with this token, so consume it. */
          Consume_Token();

          Return_Value = Token_Accepted;

          break;

        }

      }
      /* No matching keyword. */

      if (Optional)
        return Try_Again;

      Report_Parser_Error(Expecting_Plugin_Type_Parameter,Command_Filename,Current_Token);
      Return_Value = Error;
      break;
    case 'S' : /* Segment Manager */
      if ( Current_Token->TokenLength == 1 )
      {

        /* We are after a Segment Manager plugin. */
        Child->NodeValue = (void *) SegmentStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        Return_Value = Token_Accepted;

        break;

      }

      if ( Current_Token->TokenText == SegmentStr )
      {

        /* We have the start of the Segment Manager plugin type. */
        Consume_Token();

        /* Get the next token, which we hope will be the Manager keyword. */
        Next_Token();

        if ( ( Current_Token->Characterization == KeyWord ) &&
             ( Current_Token->TokenText == ManagerStr )
           )
        {

          /* We are after a Segment Manager plugin. */
          Child->NodeValue = (void *) SegmentStr;

          /* We are through with this token, so consume it. */
          Consume_Token();

          Return_Value = Token_Accepted;

          break;

        }

      }
      /* No matching keyword. */

      if (Optional)
        return Try_Again;

      Report_Parser_Error(Expecting_Plugin_Type_Parameter,Command_Filename,Current_Token);
      Return_Value = Error;
      break;
    case 'F' : /* Feature, File System Interface Module, FIM */
      if ( ( Current_Token->TokenLength == 1 ) || ( Current_Token->TokenText == FeatureStr ) )
      {
        /* We are after a Feature plugin. */
        Child->NodeValue = (void *) FeatureStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        Return_Value = Token_Accepted;

        break;


      }

      if ( Current_Token->TokenText == FSIMStr )
      {
        /* We are after a Filesystem Interface Module plugin. */
        Child->NodeValue = (void *) FSIMStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        Return_Value = Token_Accepted;

        break;

      }

      if ( Current_Token->TokenText == FilesystemStr )
      {

        /* We have the start of the Filesystem Interface Module plugin type. */
        Consume_Token();

        /* Get the next token, which we hope will be the Interface keyword. */
        Next_Token();

        /* Check for error conditions. */
        if ( ( Current_Token == NULL ) ||
             ( Current_Token->Characterization == EofToken ) ||
             ( Current_Token->Characterization == InvalidCharacter ) ||
             ( Current_Token->Characterization != KeyWord )
           )
        {

          if (Optional)
            return Try_Again;

          Report_Parser_Error(Expecting_Plugin_Type_Parameter, Command_Filename, Current_Token);

          return Error;

        }

        if ( Current_Token->TokenText == InterfaceStr )
        {

          /* We have two parts of the Filesystem Interface Module plugin type. */
          Consume_Token();

          /* Get the next token, which we hope will be the Management keyword. */
          Next_Token();

          /* Check for error conditions. */
          if ( ( Current_Token == NULL ) ||
               ( Current_Token->Characterization == EofToken ) ||
               ( Current_Token->Characterization == InvalidCharacter ) ||
               ( Current_Token->Characterization != KeyWord )
             )
          {

            if (Optional)
              return Try_Again;

            Report_Parser_Error(Expecting_Plugin_Type_Parameter, Command_Filename, Current_Token);

            return Error;

          }

          if ( Current_Token->TokenText == ModuleStr )
          {

            /* We are after a FSIM plugin. */
            Child->NodeValue = (void *) FSIMStr;

            /* We are through with this token, so consume it. */
            Consume_Token();

            Return_Value = Token_Accepted;

            break;

          }

        }

      }

      /* No matching keyword. */

      if (Optional)
        return Try_Again;

      Report_Parser_Error(Expecting_Plugin_Type_Parameter,Command_Filename,Current_Token);
      Return_Value = Error;
      break;
    default:
      if (Optional)
        Return_Value = Try_Again;
      else
      {

        Report_Parser_Error(Expecting_Plugin_Type_Parameter,Command_Filename,Current_Token);
        Return_Value = Error;

      }

      break;

  }

  return Return_Value;

}

static Parse_Results Feature_Identifier(Executable_Node * Current_Node, BOOLEAN Optional, BOOLEAN Parameters_Follow, BOOLEAN Objects_Are_Optional)
{
  Executable_Node * Child = NULL;         /* The node used to represent the current command being parsed. */
  Executable_Node * Temp_Node = NULL;     /* Used for lookahead operations. */
  Parse_Results     Return_Value = Error; /* Used to track the return value from other functions called by this function. */

  /* Create the child node and add it to the children list of Current_Node.  We will fill in the contents of Child later. */
  Child = Make_Child_Node(Current_Node, TRUE);
  if ( Child == NULL )
    return Error;

  /* Get the token to parse. */
  Next_Token();

  /* Check for EOF or InvalidCharacter */
  if ( ( Current_Token == NULL ) || ( Current_Token->Characterization == EofToken ) || ( Current_Token->Characterization == InvalidCharacter ) )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Feature_Identifier, Command_Filename, Current_Token);

    return Error;

  }

  switch ( Current_Token->Characterization )
  {
    
    case Number: /* We should have a Feature ID which we will need to turn into a handle later. */

      if ( Parameters_Follow )
        Child->NodeType = Feature_ID;
      else
        Child->NodeType = Plugin_ID;
      Assign_Text(Child,Current_Token);

      /* We are through with this token, so consume it. */
      Consume_Token();

      Return_Value = Token_Accepted;

      break;
    case KeyWord:
    case String :
      if ( Parameters_Follow )
        Child->NodeType = Feature_Name;
      else
        Child->NodeType = Name_To_Plugin_Handle;

      Assign_Text(Child,Current_Token);      

      /* We are through with this token, so consume it. */
      Consume_Token();

      Return_Value = Token_Accepted;

      break;
    default:
      if (Optional)
        Return_Value = Try_Again;
      else
      {

        Report_Parser_Error(Expecting_Feature_Identifier,Command_Filename,Current_Token);
        Return_Value = Error;

      }

      break;
  }

  if ( Parameters_Follow &&
       ( Return_Value == Token_Accepted )
     )
  {

    /* If we accepted the token, then we must look for the parameters that go with the specified feature. */

    /* An equal sign must follow the feature identifier. */
    Return_Value = Check_For_EqualSign(Optional);
    if ( Return_Value == Token_Accepted )
    {
      Return_Value = Check_For_Open_Brace(Optional);
      if ( Return_Value == Token_Accepted )
      {
        /* All of the name value pairs between the braces are optional! */

        Temp_Node = Activate_Lookahead();
        if ( Temp_Node == NULL)
          return Error;

        do
        {
          Return_Value = Parse_Name_Value_Pair(Temp_Node, TRUE);

          if ( Return_Value == Try_Again)
            if ( !Restore_Current_Token( Temp_Node ) )
              return Error;

          if ( ! Terminate_Lookahead(Child, Temp_Node, (Return_Value == Token_Accepted) ) )
            return Error;

          if ( Return_Value == Token_Accepted )
          {
            /* We have just parsed a name-value pair.  Look for another one. */

            Temp_Node = Activate_Lookahead();
            if ( Temp_Node == NULL)
              return Error;

            Return_Value = Check_For_Comma(TRUE);
            if ( Return_Value == Try_Again)
            {
              if ( !Restore_Current_Token( Temp_Node ) )
                return Error;

              if ( ! Terminate_Lookahead(Child, Temp_Node, FALSE ) )
                return Error;
            }

          }

        } while ( Return_Value == Token_Accepted );


        if ( Return_Value != Error )
        {
          Return_Value = Check_For_Closing_Brace(Optional);
          if ( Return_Value == Token_Accepted )
          {
            /* Activate lookahead since the objects could be optional. */
            Temp_Node = Activate_Lookahead();
            if ( Temp_Node == NULL)
              return Error;

            /* Now we must check for the objects and regions to be used. */
            Return_Value = Parse_Object_List(Temp_Node, ( Optional || Objects_Are_Optional ) );

            if ( ( Return_Value == Try_Again ) && ( !Restore_Current_Token( Temp_Node ) ) )
              return Error;

            if ( ! Terminate_Lookahead(Child, Temp_Node, ( Return_Value == Token_Accepted ) ) )
              return Error;

            if ( Optional || Objects_Are_Optional )
              Return_Value = Token_Accepted;

          }

        }

      }

    }

  }

  return Return_Value;

}

static Parse_Results Plugin_Identifier(Executable_Node * Current_Node, BOOLEAN Parameters_Follow, BOOLEAN Get_Object_List, BOOLEAN Optional)
{
  Executable_Node * Child = NULL;         /* The node used to represent the current command being parsed. */
  Executable_Node * Temp_Node = NULL;     /* Used for lookahead operations. */
  Parse_Results     Return_Value = Error; /* Used to track the return value from other functions called by this function. */

  /* Create the child node and add it to the children list of Current_Node.  We will fill in the contents of Child later. */
  Child = Make_Child_Node(Current_Node, TRUE);
  if ( Child == NULL )
    return Error;

  /* Get the token to parse. */
  Next_Token();

  /* Check for EOF or InvalidCharacter */
  if ( ( Current_Token == NULL ) || ( Current_Token->Characterization == EofToken ) || ( Current_Token->Characterization == InvalidCharacter ) )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Plugin_Identifier, Command_Filename, Current_Token);

    return Error;

  }

  switch ( Current_Token->Characterization )
  {
    
    case Number: /* We should have a Plugin ID which we will need to turn into a handle later. */

      Child->NodeType = Plugin_ID;
      Assign_Text(Child,Current_Token);

      /* We are through with this token, so consume it. */
      Consume_Token();

      Return_Value = Token_Accepted;

      break;
    case KeyWord:
    case String :
      if ( Parameters_Follow )
        Child->NodeType = Plugin_Name;
      else
        Child->NodeType = Name_To_Plugin_Handle;

      Assign_Text(Child,Current_Token);      

      /* We are through with this token, so consume it. */
      Consume_Token();

      Return_Value = Token_Accepted;

      break;
    default:
      if (Optional)
        Return_Value = Try_Again;
      else
      {

        Report_Parser_Error(Expecting_Plugin_Identifier,Command_Filename,Current_Token);
        Return_Value = Error;

      }

      break;
  }

  /* If we accepted the token, then we must look for the parameters that go with the specified plug-in. */
  if ( Parameters_Follow && ( Return_Value == Token_Accepted ) )
  {

    /* An equal sign must follow the plug-in identifier. */
    Return_Value = Check_For_EqualSign(Optional);
    if ( Return_Value == Token_Accepted )
    {
      Return_Value = Check_For_Open_Brace(Optional);
      if ( Return_Value == Token_Accepted )
      {
        /* All of the name value pairs between the braces are optional! */

        Temp_Node = Activate_Lookahead();
        if ( Temp_Node == NULL)
          return Error;

        do
        {
          Return_Value = Parse_Name_Value_Pair(Temp_Node, TRUE);

          if ( Return_Value == Try_Again)
            if ( !Restore_Current_Token( Temp_Node ) )
              return Error;

          if ( ! Terminate_Lookahead(Child, Temp_Node, (Return_Value == Token_Accepted) ) )
            return Error;

          if ( Return_Value == Token_Accepted )
          {
            /* We have just parsed a name-value pair.  Look for another one. */

            Temp_Node = Activate_Lookahead();
            if ( Temp_Node == NULL)
              return Error;

            Return_Value = Check_For_Comma(TRUE);
            if ( Return_Value == Try_Again)
            {
              if ( !Restore_Current_Token( Temp_Node ) )
                return Error;

              if ( ! Terminate_Lookahead(Child, Temp_Node, FALSE ) )
                return Error;
            }

          }

        } while ( Return_Value == Token_Accepted );


        if ( Return_Value != Error )
        {
          Return_Value = Check_For_Closing_Brace(Optional);

          if ( ( Return_Value == Token_Accepted ) && Get_Object_List )
          {

            /* Now we must check for the segments/regions/disks to be used. */
            Return_Value = Parse_Object_List(Child, Optional);

          }

        }

      }

    }

  }

  return Return_Value;

}

static Parse_Results Specify_Object(Executable_Node * Current_Node, NodeTypes Node_To_Create, BOOLEAN Use_Current_Token, BOOLEAN Optional)
{
  Executable_Node * Child = NULL;         /* The node used to represent the current command being parsed. */
  Parse_Results     Return_Value = Error; /* Used to track the return value from other functions called by this function. */

  /* Get the token to parse. */
  if ( !Use_Current_Token )
    Next_Token();

  /* Check for EOF or InvalidCharacter */
  if ( ( Current_Token == NULL ) || ( Current_Token->Characterization == EofToken ) || ( Current_Token->Characterization == InvalidCharacter ) )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Object_Identifier, Command_Filename, Current_Token);

    return Error;

  }

  if ( Current_Token->Characterization == String )
  {

    /* Create the child node and add it to the children list of Current_Node. */
    Child = Make_Child_Node(Current_Node, TRUE);
    if ( Child == NULL )
      return Error;

    Child->NodeType = Node_To_Create;
    Assign_Text(Child,Current_Token);

    /* We are through with this token, so consume it. */
    Consume_Token();

    Return_Value = Token_Accepted;

  }
  else
  {
    if (Optional)
      Return_Value = Try_Again;
    else
    {

      Report_Parser_Error(Expecting_Object_Identifier,Command_Filename,Current_Token);
      Return_Value = Error;

    }

  }

  return Return_Value;

}

static Parse_Results Size(Executable_Node * Current_Node, BOOLEAN Optional)
{
  Executable_Node * Child = NULL;         /* The node used to represent the current command being parsed. */

  /* Create the child node and add it to the children list of Current_Node.  We will fill in the contents of Child later. */
  Child = Make_Child_Node(Current_Node, TRUE);
  if ( Child == NULL )
    return Error;

  /* Get the token to parse. */
  Next_Token();

  /* Check for EOF or InvalidCharacter */
  if ( ( Current_Token == NULL ) || ( Current_Token->Characterization == EofToken ) || ( Current_Token->Characterization == InvalidCharacter ) )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Size_Parameters, Command_Filename, Current_Token);

    return Error;

  }

  /* Check the characterization of the token. */
  if ( ( Current_Token->Characterization != Number ) &&
       ( Current_Token->Characterization != RealNumber )
     )
  {
    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Size_Value, Command_Filename, Current_Token);

    return Error;

  }

  /* Fill in the appropriate fields in Child. */
  if ( Current_Token->Characterization == RealNumber )
    Child->NodeType = RealNumberData;
  else
    Child->NodeType = IntegerData;

  Assign_Text(Child,Current_Token);  

  /* We are through with this token, so consume it. */
  Consume_Token();

  /* Now we must find out the unit associated with the number. */
  return Parse_Units(Child,FALSE, Optional);

}


static Parse_Results Precision(Executable_Node * Current_Node, BOOLEAN Optional)
{
  Executable_Node * Child = NULL;          /* The node used to represent the current command being parsed. */
  Executable_Node * Temp_Node = NULL;      /* Used for lookahead operations. */
  Parse_Results     Return_Value = Error;  /* Used to track the return value from other functions called by this function. */

  /* Create the child node and add it to the children list of Current_Node.  We will fill in the contents of Child later. */
  Child = Make_Child_Node(Current_Node, TRUE);
  if ( Child == NULL )
    return Error;

  /* Get the token to parse. */
  Next_Token();

  /* Check for EOF or InvalidCharacter */
  if ( ( Current_Token == NULL ) || ( Current_Token->Characterization == EofToken ) || ( Current_Token->Characterization == InvalidCharacter ) )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Size_Value, Command_Filename, Current_Token);

    return Error;

  }

  /* Check the characterization of the token. */
  if ( Current_Token->Characterization != Number )
  {
    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Size_Value, Command_Filename, Current_Token);

    return Error;

  }

  /* Fill in the appropriate fields in Child. */
  Child->NodeType = Precision_Node;
  Assign_Text(Child,Current_Token);

  /* We are through with this token, so consume it. */
  Consume_Token();

  /* There may be an optional unit (KB, MB, GB, TB) associated with this value.  Check for it. */
  Temp_Node = Activate_Lookahead();
  if ( Temp_Node == NULL)
    return Error;

  Return_Value = Parse_Units(Temp_Node, FALSE, TRUE);
  if ( Return_Value == Try_Again)
  {

    /* Restore the current token so that parsing can continue. */
    if ( !Restore_Current_Token( Temp_Node ) )
      return Error;

  }

  /* Exit Lookahead mode. */
  if ( ! Terminate_Lookahead(Child, Temp_Node, (Return_Value == Token_Accepted) ) )
    return Error;

  /* Since the unit was optional, the command is complete without it. Set Return_Value accordingly. */
  if ( Return_Value != Error )
    Return_Value = Token_Accepted;

  return Return_Value;
}


static Parse_Results Size_Parameters(Executable_Node * Current_Node, BOOLEAN Filter_On_Freespace, BOOLEAN Optional)
{

  Executable_Node * Child = NULL;      /* The node used to represent the current command being parsed. */
  Executable_Node * Temp_Node = NULL;  /* Used for lookahead operations. */
  Parse_Results     Return_Value;      /* Used to track the return value from other functions called by this function. */

  /* Create the child node and add it to the children list of Current_Node.  We will fill in the contents of Child later. */
  Child = Make_Child_Node(Current_Node, TRUE);
  if ( Child == NULL )
    return Error;

  /* Get the token to parse. */
  Next_Token();

  /* Check for EOF or InvalidCharacter */
  if ( ( Current_Token == NULL ) || ( Current_Token->Characterization == EofToken ) || ( Current_Token->Characterization == InvalidCharacter ) )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Size_Parameters, Command_Filename, Current_Token);

    return Error;

  }

  /* The parameter we are looking for is characterized as a KeyWord.  */
  if ( Current_Token->Characterization != KeyWord )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Size_Parameters, Command_Filename, Current_Token);

    return Error;

  }

  /* Set the type of filter to create. */
  if ( Filter_On_Freespace )
    Child->NodeType = Freespace_Size_Filter;
  else
    Child->NodeType = Filter;

  /* Lets see which size parameters we have. */
  switch ( Current_Token->TokenText[0] )
  {
    case 'E' : /* EQ */
      if ( Current_Token->TokenText == EQStr )
      {
        /* We are going to produce an EQ filter. */
        Child->NodeValue = (void *) EQStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        /* Now get the size. */
        Return_Value = Size(Child,Optional);
        if ( Return_Value == Token_Accepted )
        {
          /* Now get the precision.  The precision is a number representing how much of a "fudge" factor we should allow
             when determining if two values are equal.                                                                   */

          /* The precision is optional. */

          Temp_Node = Activate_Lookahead();
          if ( Temp_Node == NULL)
            return Error;

          Return_Value = Check_For_Comma(TRUE);
          if ( Return_Value == Token_Accepted )
            Return_Value = Precision(Temp_Node, TRUE);  /* Get the name for the volume. */

          if ( Return_Value == Try_Again)
          {

            /* Restore the current token so that parsing can continue. */
            if ( !Restore_Current_Token( Temp_Node ) )
              return Error;

          }

          /* Exit Lookahead mode. */
          if ( ! Terminate_Lookahead(Child, Temp_Node, (Return_Value == Token_Accepted) ) )
            return Error;

          /* Precision is optional, so we return Token_Accepted regardless of whether or not we found a Precision specified. 
             The only exception is if an error occurred.                                                                      */
          if (Return_Value != Error)
            Return_Value = Token_Accepted;

        }


      }
      else
      {

        if ( Optional )
          return Try_Again;

        Report_Parser_Error(Expecting_EQ_Keyword, Command_Filename, Current_Token);
        Return_Value = Error;

      }

      break;

    case 'L' : /* LT */
      if ( Current_Token->TokenText != LTStr )
      {

        if ( Optional )
          return Try_Again;

        Report_Parser_Error(Expecting_LT_Keyword, Command_Filename, Current_Token);
        Return_Value = Error;

        break;
      }

      /* We are going to produce an LT filter. */
      Child->NodeValue = (void *) LTStr;

      /* We are through with this token, so consume it. */
      Consume_Token();

      /* Now get the size. */
      Return_Value = Size(Child,Optional);

      if ( Return_Value != Token_Accepted )
        break;

      /* There may be an optional GT command following this LT command.  Lets check. */
      Temp_Node = Activate_Lookahead();
      if ( Temp_Node == NULL)
        return Error;

      /* Check for a comma - if found, optional parameters should follow. */
      Return_Value = Check_For_Comma(TRUE);

      if ( Return_Value == Token_Accepted )
      {

        Return_Value = GT_LT_Parameter_Check(Temp_Node,Filter_On_Freespace, TRUE);

      }

      if ( Return_Value == Error )
        return Error;

      if ( Return_Value == Try_Again)
      {

        /* Restore the current token so that parsing can continue. */
        if ( !Restore_Current_Token( Temp_Node ) )
          return Error;

      }

      /* Exit Lookahead mode. */
      if ( ! Terminate_Lookahead(Current_Node, Temp_Node, (Return_Value == Token_Accepted) ) )
        return Error;

      if ( Return_Value == Try_Again )
        Return_Value = Token_Accepted;  /* The command is complete - no optional parameters follow. */

      break;
    case 'G' : /* GT */
      if ( Current_Token->TokenText != GTStr )
      {
        if ( Optional )
          return Try_Again;

        Report_Parser_Error(Expecting_GT_Keyword, Command_Filename, Current_Token);
        Return_Value = Error;

        break;
      }

      /* We are going to produce an GT filter. */
      Child->NodeValue = (void *) GTStr;

      /* We are through with this token, so consume it. */
      Consume_Token();

      /* Now get the size. */
      Return_Value = Size(Child,Optional);

      if ( Return_Value != Token_Accepted )
      {

        if ( Return_Value == Error )
          return Error;

        if ( ! Optional )
        {

          Report_Parser_Error(Expecting_Size_Value, Command_Filename, Current_Token);
          return Error;

        }

        break;

      }

      /* There may be an optional LT command following this GT command.  Lets check. */
      Temp_Node = Activate_Lookahead();
      if ( Temp_Node == NULL)
        return Error;

      /* Check for a comma - if found, optional parameters should follow. */
      Return_Value = Check_For_Comma(TRUE);

      if ( Return_Value == Token_Accepted )
      {

        Return_Value = GT_LT_Parameter_Check(Temp_Node, Filter_On_Freespace, TRUE);

      }

      if ( Return_Value == Error )
        return Error;

      if ( Return_Value == Try_Again)
      {

        /* Restore the current token so that parsing can continue. */
        if ( !Restore_Current_Token( Temp_Node ) )
          return Error;

      }

      /* Exit Lookahead mode. */
      if ( ! Terminate_Lookahead(Current_Node, Temp_Node, (Return_Value == Token_Accepted) ) )
        return Error;

      if ( Return_Value == Try_Again )
        Return_Value = Token_Accepted;  /* The command is complete - no optional parameters follow. */

      break;

    default:
      if (Optional)
        Return_Value = Try_Again;
      else
      {
        Return_Value = Error;
        Report_Parser_Error(Expecting_Size_Parameters,Command_Filename,Current_Token);
      }
      break;

  }

  return Return_Value;

}

static Parse_Results Parse_Units(Executable_Node * Current_Node, BOOLEAN Allow_Time_Units, BOOLEAN Optional)
{
  Executable_Node * Child = NULL;         /* The node used to represent the current command being parsed. */
  Parser_Errors     Error_Code;

  /* Set the error message to use. */
  if (Allow_Time_Units)
    Error_Code = Expecting_Units_Parameter;
  else
    Error_Code = Expecting_Disk_Units_Parameter;

  /* Create the child node and add it to the children list of Current_Node.  We will fill in the contents of Child later. */
  Child = Make_Child_Node(Current_Node, TRUE);
  if ( Child == NULL )
    return Error;

  /* Get the token to parse. */
  Next_Token();

  /* Check for EOF or InvalidCharacter */
  if ( ( Current_Token == NULL ) || ( Current_Token->Characterization == EofToken ) || ( Current_Token->Characterization == InvalidCharacter ) )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Error_Code, Command_Filename, Current_Token);

    return Error;

  }

  /* We are expecting a key word. */
  if ( Current_Token->Characterization != KeyWord )
  {
    if (Optional)
      return Try_Again;

    Report_Parser_Error(Error_Code, Command_Filename, Current_Token);

    return Error;

  }

  switch ( Current_Token->TokenText[0] )
  {
    
    case 'M' :  /* MB, Minutes, Milliseconds, Microseconds */
      if ( Current_Token->TokenText == MegabyteStr)
      {
        /* Accept the token. */
        Child->NodeType = Units_Node;
        Child->NodeValue = (void *) MegabyteStr;

        /* We are through with this token, so consume it. */
        Consume_Token();
        break;
      }

      if ( ( Current_Token->TokenText == MinutesStr ) && Allow_Time_Units )
      {
        /* Accept the token. */
        Child->NodeType = Units_Node;
        Child->NodeValue = (void *) MinutesStr;

        /* We are through with this token, so consume it. */
        Consume_Token();
        break;
      }

      if ( ( Current_Token->TokenText == MillisecondsStr ) && Allow_Time_Units )
      {
        /* Accept the token. */
        Child->NodeType = Units_Node;
        Child->NodeValue = (void *) MillisecondsStr;

        /* We are through with this token, so consume it. */
        Consume_Token();
        break;
      }

      if ( ( Current_Token->TokenText == MicrosecondsStr ) && Allow_Time_Units )
      {
        /* Accept the token. */
        Child->NodeType = Units_Node;
        Child->NodeValue = (void *) MicrosecondsStr;

        /* We are through with this token, so consume it. */
        Consume_Token();
        break;
      }

      Report_Parser_Error(Error_Code, Command_Filename, Current_Token);

      return Error;

      break;  /* Keep the compiler happy. */
    case 'G' : /* GB */
      if ( Current_Token->TokenText != GigabyteStr)
      {
        Report_Parser_Error(Error_Code, Command_Filename, Current_Token);

        return Error;
      }

      /* Accept the token. */
      Child->NodeType = Units_Node;
      Child->NodeValue = (void *) GigabyteStr;

      /* We are through with this token, so consume it. */
      Consume_Token();

      break;
    case 'H' : /* Hours */
      if ( ( ( Current_Token->TokenText == HoursStr ) || 
             ( Current_Token->TokenLength == 1 )
           ) &&
           Allow_Time_Units
         )
      {
        /* Accept the token. */
        Child->NodeType = Units_Node;
        Child->NodeValue = (void *) HoursStr;

        /* We are through with this token, so consume it. */
        Consume_Token();
        break;
      }

      Report_Parser_Error(Error_Code, Command_Filename, Current_Token);

      return Error;

      break;  /* Keep the compiler happy. */
    case 'K' : /* KB */
      if ( Current_Token->TokenText != KilobyteStr)
      {
        Report_Parser_Error(Error_Code, Command_Filename, Current_Token);

        return Error;
      }

      /* Accept the token. */
      Child->NodeType = Units_Node;
      Child->NodeValue = (void *) KilobyteStr;

      /* We are through with this token, so consume it. */
      Consume_Token();

      break;
    case 'S' : /* Seconds, Sectors */
      if ( ( ( Current_Token->TokenText == SecondsStr ) ||
             ( Current_Token->TokenLength == 1 )
           ) &&
           Allow_Time_Units
         )
      {
        /* Accept the token. */
        Child->NodeType = Units_Node;
        Child->NodeValue = (void *) SecondsStr;

        /* We are through with this token, so consume it. */
        Consume_Token();
        break;
      }

      if ( Current_Token->TokenText == SectorsStr)
      {
        /* Accept the token. */
        Child->NodeType = Units_Node;
        Child->NodeValue = (void *) SectorsStr;

        /* We are through with this token, so consume it. */
        Consume_Token();
        break;
      }

      Report_Parser_Error(Error_Code, Command_Filename, Current_Token);

      return Error;

      break; /* Keep the compiler happy. */
    case 'T' : /* TB */
      if ( Current_Token->TokenText != TerrabyteStr)
      {
        Report_Parser_Error(Error_Code, Command_Filename, Current_Token);

        return Error;
      }

      /* Accept the token. */
      Child->NodeType = Units_Node;
      Child->NodeValue = (void *) TerrabyteStr;

      /* We are through with this token, so consume it. */
      Consume_Token();

      break;
    default:

      if (Optional)
        return Try_Again;

      Report_Parser_Error(Error_Code, Command_Filename, Current_Token);

      return Error;

      break;
  }

  return Token_Accepted;

}

static Parse_Results GT_LT_Parameter_Check(Executable_Node * Current_Node, BOOLEAN Filter_On_Freespace, BOOLEAN Optional)
{
  Executable_Node * Child = NULL;      /* The node used to represent the current command being parsed. */
  Parse_Results     Return_Value;      /* Used to track the return value from other functions called by this function. */

  /* Create the child node and add it to the children list of Current_Node.  We will fill in the contents of Child later. */
  Child = Make_Child_Node(Current_Node, TRUE);
  if ( Child == NULL )
    return Error;

  /* Get the token to parse. */
  Next_Token();

  /* Check for EOF or InvalidCharacter */
  if ( ( Current_Token == NULL ) || ( Current_Token->Characterization == EofToken ) || ( Current_Token->Characterization == InvalidCharacter ) )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Size_Parameters, Command_Filename, Current_Token);

    return Error;

  }

  /* The parameter we are looking for is characterized as a KeyWord.  */
  if ( Current_Token->Characterization != KeyWord )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Size_Parameters, Command_Filename, Current_Token);

    return Error;

  }

  /* Set the type of filter to create. */
  if ( Filter_On_Freespace )
    Child->NodeType = Freespace_Size_Filter;
  else
    Child->NodeType = Filter;

  /* Lets see which size parameters we have. */
  switch ( Current_Token->TokenText[0] )
  {
    case 'L' : /* LT */
      if ( Current_Token->TokenText != LTStr )
      {

        if ( Optional )
          return Try_Again;

        Report_Parser_Error(Expecting_LT_Keyword, Command_Filename, Current_Token);
        Return_Value = Error;

        break;
      }

      /* We are going to produce an LT filter. */
      Child->NodeValue = (void *) LTStr;

      /* We are through with this token, so consume it. */
      Consume_Token();

      Return_Value = Size(Child, Optional);

      break;
    case 'G' : /* GT */
      if ( Current_Token->TokenText != GTStr )
      {

        if ( Optional )
          return Try_Again;

        Report_Parser_Error(Expecting_GT_Keyword, Command_Filename, Current_Token);
        Return_Value = Error;

        break;
      }

      /* We are going to produce an GT filter. */
      Child->NodeValue = (void *) GTStr;

      /* We are through with this token, so consume it. */
      Consume_Token();

      Return_Value = Size(Child, Optional);

      break;

    default:
      if (Optional)
        Return_Value = Try_Again;
      else
      {
        Return_Value = Error;
        Report_Parser_Error(Expecting_GT_LT_Parameter,Command_Filename,Current_Token);
      }
      break;

  }

  return Return_Value;

}

static Parse_Results Value(Executable_Node * Current_Node, BOOLEAN Optional)
{
  Executable_Node * Child = NULL;         /* The node used to represent the current command being parsed. */
  Executable_Node * Temp_Node = NULL;      /* Used for lookahead operations. */
  Parse_Results     Return_Value = Error;  /* Used to track the return value from other functions called by this function. */

  /* Create the child node and add it to the children list of Current_Node.  We will fill in the contents of Child later. */
  Child = Make_Child_Node(Current_Node, TRUE);
  if ( Child == NULL )
    return Error;

  /* Get the token to parse. */
  Next_Token();

  /* Check for EOF or InvalidCharacter */
  if ( ( Current_Token == NULL ) || ( Current_Token->Characterization == EofToken ) || ( Current_Token->Characterization == InvalidCharacter ) )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Number_Or_String, Command_Filename, Current_Token);

    return Error;

  }

  /* We are expecting either a numeric or string type here. */
  switch ( Current_Token->Characterization )
  {
    
    case Number :
      Child->NodeType = IntegerData;
      Assign_Text(Child,Current_Token);      

      Consume_Token();

      /* There may be an optional unit (KB, MB, GB, TB) associated with this value.  Check for it. */
      Temp_Node = Activate_Lookahead();
      if ( Temp_Node == NULL)
        return Error;

      Return_Value = Parse_Units(Temp_Node,TRUE, TRUE);
      if ( Return_Value == Try_Again)
      {

        /* Restore the current token so that parsing can continue. */
        if ( !Restore_Current_Token( Temp_Node ) )
          return Error;

      }

      /* Exit Lookahead mode. */
      if ( ! Terminate_Lookahead(Child, Temp_Node, (Return_Value == Token_Accepted) ) )
        return Error;

      /* Since the unit was optional, the command is complete without it. Set Return_Value accordingly. */
      if ( Return_Value != Error )
        Return_Value = Token_Accepted;

      break;
    case RealNumber :
      Child->NodeType = RealNumberData;
      Assign_Text(Child,Current_Token);

      Consume_Token();

      /* There may be an optional unit (KB, MB, GB, TB) associated with this value.  Check for it. */
      Temp_Node = Activate_Lookahead();
      if ( Temp_Node == NULL)
        return Error;

      Return_Value = Parse_Units(Temp_Node, TRUE, TRUE);
      if ( Return_Value == Try_Again)
      {

        /* Restore the current token so that parsing can continue. */
        if ( !Restore_Current_Token( Temp_Node ) )
          return Error;

      }

      /* Exit Lookahead mode. */
      if ( ! Terminate_Lookahead(Child, Temp_Node, (Return_Value == Token_Accepted) ) )
        return Error;

      /* Since the unit was optional, the command is complete without it. Set Return_Value accordingly. */
      if ( Return_Value != Error )
        Return_Value = Token_Accepted;

      break;
    case String :
      Child->NodeType = StringData;
      Assign_Text(Child,Current_Token);
      Return_Value = Token_Accepted;
      break;
    case KeyWord :
      Child->NodeType = StringData;
      Child->NodeValue = strdup(Current_Token->TokenText);
      if ( Child->NodeValue == NULL )
      {
        Report_Parser_Error(Out_Of_Memory, Command_Filename, Current_Token);
        return Error;
      }
      Return_Value = Token_Accepted;
      break;
    default:
      if (Optional)
        return Try_Again;

      Report_Parser_Error(Expecting_Number_Or_String, Command_Filename, Current_Token);

      return Error;

      break;
  }

  return Return_Value;

}


static Parse_Results Name(Executable_Node * Current_Node, BOOLEAN Field_Name, BOOLEAN Optional)
{
  Executable_Node * Child = NULL;         /* The node used to represent the current command being parsed. */

  /* Create the child node and add it to the children list of Current_Node.  We will fill in the contents of Child later. */
  Child = Make_Child_Node(Current_Node, TRUE);
  if ( Child == NULL )
    return Error;

  /* Get the token to parse. */
  Next_Token();

  /* Check for EOF or InvalidCharacter or bad token characterization. */
  if ( ( Current_Token == NULL ) ||
       ( Current_Token->Characterization == EofToken ) ||
       ( Current_Token->Characterization == InvalidCharacter ) ||
       ( ( Current_Token->Characterization != String ) &&
         ( Current_Token->Characterization != KeyWord )
       )
     )
  {

    if (Optional)
      return Try_Again;

    if ( Field_Name )
      Report_Parser_Error(Expecting_Field_Name, Command_Filename, Current_Token);
    else
      Report_Parser_Error(Expecting_Name_Value, Command_Filename, Current_Token);

    return Error;

  }

  Child->NodeType = StringData;

  if ( Current_Token->Characterization == KeyWord)
  {
    Child->NodeValue = strdup(Current_Token->TokenText);
    if ( Child->NodeValue == NULL)
    {
      Report_Parser_Error(Out_Of_Memory, Command_Filename, Current_Token);
      return Error;
    }
  }
  else
    Assign_Text(Child,Current_Token);

  Consume_Token();

  return Token_Accepted;

}


static Parse_Results Accept_Name(Executable_Node * Current_Node, BOOLEAN Name_Optional)
{
  Executable_Node * Child = NULL;         /* The node used to represent the current command being parsed. */
  Parse_Results     Return_Value = Error; /* Used to track the return value from other functions called by this function. */

  Next_Token();

  /* Check for EOF, InvalidCharacter, or incorrect characterization */
  if ( ( Current_Token == NULL ) ||
       ( Current_Token->Characterization == EofToken ) ||
       ( Current_Token->Characterization == InvalidCharacter ) ||
       ( Current_Token->Characterization != KeyWord )
     )
  {

    if (Name_Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Name, Command_Filename, Current_Token);

    return Error;

  }

  /* Is the token the Name keyword? */
  if ( ( Current_Token->TokenText == NameStr ) ||
       ( ( Current_Token->TokenLength == 1 ) &&
         ( Current_Token->TokenText[0] == 'N' )
       )
     )
  {
    /* We have the Name keyword. */

    /* Create the child node and add it to the children list of Current_Node. */
    Child = Make_Child_Node(Current_Node, TRUE);
    if ( Child == NULL )
      return Error;

    Child->NodeType = StringData;

    Consume_Token();

    Return_Value = Check_For_EqualSign(Name_Optional);
    if ( Return_Value == Token_Accepted )
    {
      Next_Token();

      /* Check for EOF, InvalidCharacter, or incorrect characterization */
      if ( ( Current_Token == NULL ) ||
           ( Current_Token->Characterization == EofToken ) ||
           ( Current_Token->Characterization == InvalidCharacter ) ||
           ( Current_Token->Characterization != String )
         )
      {

        if (Name_Optional)
          Return_Value = Try_Again;
        else
        {
          Report_Parser_Error(Expecting_Name, Command_Filename, Current_Token);

          Return_Value =  Error;
        }

      }
      else
      {
        Assign_Text(Child,Current_Token);
        Return_Value = Token_Accepted;

        Consume_Token();
      }

    }

  }
  else
  {
    if (Name_Optional)
      Return_Value =  Try_Again;
    else
      Report_Parser_Error(Expecting_Object_Identifier, Command_Filename, Current_Token);

  }

  return Return_Value;

}


static Parse_Results Parse_Name_Value_Pair(Executable_Node * Current_Node, BOOLEAN Optional)
{
  Executable_Node * Child = NULL;           /* The node used to represent the current command being parsed. */
  Executable_Node * Temp_Node = NULL;       /* Used for Lookahead operations. */
  Parse_Results     Return_Value;           /* Used to track the return value from other functions called by this function. */
  BOOLEAN           List_Mode = FALSE;      /* Set to TRUE if we have Name = (value,value,...)  to parse. */

  /* Create the child node and add it to the children list of Current_Node.  We will fill in the contents of Child later. */
  Child = Make_Child_Node(Current_Node, TRUE);
  if ( Child == NULL )
    return Error;

  Child->NodeType = Name_Value_Pair;

  /* Get the name. */
  Return_Value = Name(Child, FALSE, Optional);
  if ( Return_Value == Token_Accepted )
  {
    /* We should have an equal sign next, followed by a value. */
    Return_Value = Check_For_EqualSign(Optional);
    if ( Return_Value == Token_Accepted )
    {

      /* Do we have a list of values? */
      Temp_Node = Activate_Lookahead();
      if ( Temp_Node == NULL)
        return Error;

      /* Check for open parenthesis */
      Return_Value = Check_For_Open_Paren(TRUE);

      if ( Return_Value == Try_Again )
      {

        /* Restore the current token so that parsing can continue. */
        if ( !Restore_Current_Token( Temp_Node ) )
          return Error;

      }
      else if ( Return_Value == Token_Accepted )
        List_Mode = TRUE;
      else
        return Error;

      /* Exit Lookahead mode. */
      if ( ! Terminate_Lookahead(Child, Temp_Node, (Return_Value == Token_Accepted) ) )
        return Error;

      do
      {

        /* Now lets get the value to go with the name. */
        Return_Value = Value(Child, Optional);

        if ( List_Mode )
        {

          /* Check for the end of the list. */
          Temp_Node = Activate_Lookahead();
          if ( Temp_Node == NULL)
            return Error;

          /* Check for closing parenthesis */
          Return_Value = Check_For_Closing_Paren(TRUE);

          if ( Return_Value == Try_Again )
          {

            /* Restore the current token so that parsing can continue. */
            if ( !Restore_Current_Token( Temp_Node ) )
              return Error;

            /* Since this was not the end of the list, a comma must follow. */
            Return_Value = Check_For_Comma(FALSE);
          }
          else if ( Return_Value == Token_Accepted )
            List_Mode = FALSE;

          /* Exit Lookahead mode. */
          if ( ! Terminate_Lookahead(Child, Temp_Node, (Return_Value == Token_Accepted) ) )
            return Error;

        }

      } while ( (Return_Value == Token_Accepted) && List_Mode );

    }

  }

  return Return_Value;

}


static Parse_Results Parse_Query_Volumes_Parameters(Executable_Node * Current_Node, BOOLEAN Optional)
{
  Executable_Node * Temp_Node = NULL;                  /* Used for lookahead operations. */
  Executable_Node * Temp2_Node = NULL;                 /* Used for lookahead operations. */
  Parse_Results     Return_Value;                      /* Used to track the return value from other functions called by this function. */
  BOOLEAN           Size_Parameters_Found = FALSE;
  BOOLEAN           Plugin_Parameters_Found = FALSE;
  BOOLEAN           Container_Parameters_Found = FALSE;
  BOOLEAN           Expand_Shrink_Found = FALSE;
  BOOLEAN           Region_Or_Object_Found = FALSE;
  BOOLEAN           Nested_Lookahead = FALSE;

  /* Since these parameters are optional, activate look ahead. */
  Temp_Node = Activate_Lookahead();
  if ( Temp_Node == NULL)
    return Error;

  /* Check for a region being specified. */
  Return_Value = Get_Region(Temp_Node, FALSE, TRUE);
  if ( Return_Value != Token_Accepted )
  {
    if ( !Restore_Current_Token( Temp_Node ) )
      return Error;
  }
  else
    Region_Or_Object_Found = TRUE;

  if ( Return_Value == Try_Again )
  {
    Return_Value = Get_Object(Temp_Node, FALSE, TRUE);

    if ( Return_Value != Token_Accepted )
    {
      if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }
    else
      Region_Or_Object_Found = TRUE;

  }

  /* Save any parameters we may have found. */
  if ( ! Terminate_Lookahead( Current_Node, Temp_Node, (Return_Value == Token_Accepted) ) )
    return Error;

  if ( Return_Value == Try_Again )
  {

    /* Since these parameters are optional, activate look ahead. */
    Temp_Node = Activate_Lookahead();
    if ( Temp_Node == NULL)
      return Error;

    /* Loop looking for each of the possible optional parameters.  Abort when a parameter other than one of the ones we
       are looking for is found.                                                                                        */
    for (;;)
    {

      /* Initialize Return_Value to Try_Again.  We will key off of this value later in the loop. */
      Return_Value = Try_Again;

      /* Check for optional parameters */
      if ( ! Size_Parameters_Found )
      {

        Return_Value = Size_Parameters(Temp_Node, FALSE, TRUE);
        if ( Return_Value == Token_Accepted )
          Size_Parameters_Found = TRUE;
        else
          if ( !Restore_Current_Token( Temp_Node ) )
          return Error;

      }

      if ( ( Return_Value == Try_Again ) && ( ! Plugin_Parameters_Found ) )
      {
        Return_Value = Get_Plugin(Temp_Node, FALSE, TRUE);
        if ( Return_Value == Token_Accepted )
          Plugin_Parameters_Found = TRUE;
        else
          if ( !Restore_Current_Token( Temp_Node ) )
          return Error;
      }

      if ( ( Return_Value == Try_Again ) && ( ! Container_Parameters_Found ) )
      {
        Return_Value = Get_Container(Temp_Node, FALSE, TRUE);
        if ( Return_Value == Token_Accepted )
          Container_Parameters_Found = TRUE;
        else
          if ( !Restore_Current_Token( Temp_Node ) )
          return Error;
      }

      if ( ( Return_Value == Try_Again ) && ( ! Expand_Shrink_Found  ) )
      {
        Return_Value = Get_Expand_Shrink(Temp_Node, FALSE, TRUE);
        if ( Return_Value == Token_Accepted )
          Expand_Shrink_Found = TRUE;
        else
          if ( !Restore_Current_Token( Temp_Node ) )
          return Error;
      }

      /* Save any parameters we may have found. */
      if ( ! Terminate_Lookahead( Current_Node, Temp_Node, ( Return_Value == Token_Accepted ) ) )
        return Error;

      if (Nested_Lookahead)
      {

        if ( Return_Value != Token_Accepted )
          if ( !Restore_Current_Token( Temp2_Node ) )
            return Error;

        if ( !Terminate_Lookahead( Current_Node, Temp2_Node, ( Return_Value == Token_Accepted ) ) )
          return Error;

        Nested_Lookahead = FALSE;
      }

      if ( Return_Value != Token_Accepted )
      {
        /* No parameter match has been found. */
        break;
      }

      /* We need to check for additional parameters. */

      /* Since this parameter is optional, activate look ahead. */
      Temp2_Node = Activate_Lookahead();
      if ( Temp_Node == NULL)
        return Error;

      /* Check for a comma which would indicate possible additional parameters. */
      Return_Value = Check_For_Comma(TRUE);
      if ( Return_Value != Token_Accepted )
      {
        if ( !Restore_Current_Token( Temp2_Node ) )
          return Error;

        if ( ! Terminate_Lookahead( Current_Node, Temp2_Node, FALSE ) )
          return Error;

        break;
      }

      /* We want a nested lookahead context here so that we don't have to test for the comma
         everytime, yet we still want to be able to go back to the comma should we not find
         the correct option after the comma.                                                  */
      Temp_Node = Activate_Lookahead();
      if ( Temp_Node == NULL)
        return Error;

      Nested_Lookahead = TRUE;
    }

  }

  if ( Return_Value != Error)
  {

    /* Check for the List Options parameter, which is optional. */
    Temp_Node = Activate_Lookahead();
    if ( Temp_Node == NULL)
      return Error;

    /* Check for a comma - if found, optional parameters should follow. */
    if (Size_Parameters_Found || Plugin_Parameters_Found || Container_Parameters_Found || Expand_Shrink_Found || Region_Or_Object_Found)
      Return_Value = Check_For_Comma(TRUE);
    else
      Return_Value = Token_Accepted;


    if ( Return_Value == Token_Accepted )
      Return_Value = Parse_Query_List_Options(Temp_Node, TRUE);

    if ( Return_Value == Error )
      return Error;

    if ( Return_Value == Try_Again)
    {

      /* Restore the current token so that parsing can continue. */
      if ( !Restore_Current_Token( Temp_Node ) )
        return Error;

    }

    /* Exit Lookahead mode. */
    if ( ! Terminate_Lookahead(Current_Node, Temp_Node, (Return_Value == Token_Accepted) ) )
      return Error;


    Return_Value = Token_Accepted;   /* The command is complete. */

  }

  /* Since all of the parameters we are looking for here are optional, it doesn't matter whether we found any or not. */
  if ( Return_Value != Error )
    return Token_Accepted;
  else
    return Error;

}


static Parse_Results Parse_Query_List_Options(Executable_Node * Current_Node, BOOLEAN Optional)
{
  Executable_Node * Child = NULL;         /* The node used to represent the current command being parsed. */

  /* Create the child node and add it to the children list of Current_Node.  We will fill in the contents of Child later. */
  Child = Make_Child_Node(Current_Node, TRUE);
  if ( Child == NULL )
    return Error;

  /* Get the token to parse. */
  Next_Token();

  /* Check for EOF or InvalidCharacter or wrong token characterization. */
  if ( ( Current_Token == NULL ) || ( Current_Token->Characterization == EofToken ) || ( Current_Token->Characterization == InvalidCharacter ) || ( Current_Token->Characterization != KeyWord ) )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_List_Options_Keywords, Command_Filename, Current_Token);

    return Error;

  }

  /* Check for the LO abbreviation. */
  if ( Current_Token->TokenText == LOStr )
  {
    Child->NodeType = Filter;
    Child->NodeValue = (void *) LOStr;

    /* We are through with this token, so consume it. */
    Consume_Token();

    return Token_Accepted;

  }

  if ( Current_Token->TokenText == ListStr )
  {
    /* We are through with this token, so consume it. */
    Consume_Token();

    /* Get the next token. */
    Next_Token();

    /* Check for EOF or InvalidCharacter or wrong token characterization. */
    if ( ( Current_Token == NULL ) || ( Current_Token->Characterization == EofToken ) || ( Current_Token->Characterization == InvalidCharacter ) || ( Current_Token->Characterization != KeyWord ) )
    {

      if (Optional)
        return Try_Again;

      Report_Parser_Error(Expecting_List_Options_Keywords, Command_Filename, Current_Token);

      return Error;

    }

    if ( Current_Token->TokenText == OptionsStr )
    {

      Child->NodeType = Filter;
      Child->NodeValue = (void *) LOStr;

      /* We are through with this token, so consume it. */
      Consume_Token();

      return Token_Accepted;

    }

  }

  if (Optional)
    return Try_Again;

  Report_Parser_Error(Expecting_List_Options_Keywords, Command_Filename, Current_Token);

  return Error;

}


static Parse_Results Parse_Query_Plugins_Parameters(Executable_Node * Current_Node, BOOLEAN Optional)
{
  Executable_Node * Child = NULL;         /* The node used to represent the current command being parsed. */
  Parse_Results     Return_Value = Error; /* Used to track the return value from other functions called by this function. */

  /* Create the child node and add it to the children list of Current_Node.  We will fill in the contents of Child later. */
  Child = Make_Child_Node(Current_Node, TRUE);
  if ( Child == NULL )
    return Error;

  /* Get the token to parse. */
  Next_Token();

  /* Check for EOF or InvalidCharacter */
  if ( ( Current_Token == NULL ) || ( Current_Token->Characterization == EofToken ) || ( Current_Token->Characterization == InvalidCharacter ) )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Query_Plugin_Parameters, Command_Filename, Current_Token);

    return Error;

  }

  /* The parameter we are looking for is characterized as a KeyWord.  */
  if ( Current_Token->Characterization != KeyWord )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Query_Plugin_Parameters, Command_Filename, Current_Token);

    return Error;

  }

  /* Lets see which what kind of plugin we are after. */
  switch ( Current_Token->TokenText[0] )
  {
    case 'C' : /* Container */
      if ( ( Current_Token->TokenText == ContainerStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        /* A container is to be specified. */
        Child->NodeType = Filter;
        Child->NodeValue = (void *) ContainerStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        /* Get the = */
        Return_Value = Check_For_EqualSign(Optional);

        if ( Return_Value == Token_Accepted )
          Return_Value = Specify_Object(Child, Name_To_Container_Handle, FALSE, Optional);

        break;

      }

      /* No matching keyword. */

      if (Optional)
        return Try_Again;

      Report_Parser_Error(Expecting_Query_Plugin_Parameters,Command_Filename,Current_Token);
      Return_Value = Error;

      break;

    case 'P' : /* Plug-in */
      /* Is the token the Plugin keyword? */
      if ( ( Current_Token->TokenText == PluginStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        /* We have the Plugin keyword. */
        Child->NodeType = Filter;
        Child->NodeValue = PluginStr;

        Consume_Token();

        Return_Value = Check_For_EqualSign(Optional);
        if ( Return_Value == Token_Accepted )
          Return_Value = Plugin_Identifier(Child,FALSE, FALSE, Optional);

        break;
      }

      /* No matching keyword. */
      if (Optional)
        return Try_Again;

      Report_Parser_Error(Expecting_Query_Plugin_Parameters,Command_Filename,Current_Token);
      Return_Value = Error;

      break;      
    case 'O' : /* Object */

      if ( ( Current_Token->TokenText == ObjectStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        /* An object  is to be specified. */
        Child->NodeType = Filter;
        Child->NodeValue = (void *) ObjectStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        /* Get the = */
        Return_Value = Check_For_EqualSign(Optional);

        if ( Return_Value == Token_Accepted )
          Return_Value = Specify_Object(Child, Name_To_Object_Handle, FALSE, Optional);

        break;

      }

      /* No matching keyword. */

      if (Optional)
        return Try_Again;

      Report_Parser_Error(Expecting_Query_Plugin_Parameters,Command_Filename,Current_Token);
      Return_Value = Error;

      break;

    case 'V' : /* Volume */
      if ( ( Current_Token->TokenText == VolumeStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        /* A volume is to be specified. */
        Child->NodeType = Filter;
        Child->NodeValue = (void *) VolumeStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        /* Get the = */
        Return_Value = Check_For_EqualSign(Optional);

        if ( Return_Value == Token_Accepted )
          Return_Value = Specify_Object(Child, Name_To_Volume_Handle, FALSE, Optional);

        break;

      }

      /* No matching keyword. */

      if (Optional)
        return Try_Again;

      Report_Parser_Error(Expecting_Query_Plugin_Parameters,Command_Filename,Current_Token);
      Return_Value = Error;

      break;

    case 'T' : /* Plugin type */

      if ( ( Current_Token->TokenText == TypeStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        /* An plug-in type is to be specified. */
        Child->NodeType = Filter;
        Child->NodeValue = (void *) TypeStr;

        /* We are through with this token, so consume it. */
        Consume_Token();

        /* Get the = */
        Return_Value = Check_For_EqualSign(Optional);

        if ( Return_Value == Token_Accepted )
          Return_Value = Plugin_Types(Child, FALSE, Optional);

        break;

      }

      /* No matching keyword. */

      if (Optional)
        return Try_Again;

      Report_Parser_Error(Expecting_Query_Plugin_Parameters,Command_Filename,Current_Token);
      Return_Value = Error;

      break;

    default:
      if (Optional)
        Return_Value = Try_Again;
      else
      {

        Report_Parser_Error(Expecting_Query_Plugin_Parameters,Command_Filename,Current_Token);
        Return_Value = Error;

      }

      break;

  }

  return Return_Value;

}

static Parse_Results Parse_Query_Objects_Parameters(Executable_Node * Current_Node, BOOLEAN Optional)
{
  Executable_Node * Temp_Node = NULL;                  /* Used for lookahead operations. */
  Executable_Node * Temp2_Node = NULL;                 /* Used for lookahead operations. */
  Parse_Results     Return_Value;                      /* Used to track the return value from other functions called by this function. */
  BOOLEAN           Size_Parameters_Found = FALSE;
  BOOLEAN           Plugin_Parameters_Found = FALSE;
  BOOLEAN           Volume_Parameters_Found = FALSE;
  BOOLEAN           Container_Parameters_Found = FALSE;
  BOOLEAN           Expand_Shrink_Found = FALSE;
  BOOLEAN           Nested_Lookahead = FALSE;
  BOOLEAN           Unclaimed_Found = FALSE;
  BOOLEAN           List_Options_Found = FALSE;
  BOOLEAN           Region_Found = FALSE;

  /* Since these parameters are optional, activate look ahead. */
  Temp_Node = Activate_Lookahead();
  if ( Temp_Node == NULL)
    return Error;

  /* Loop looking for each of the possible optional parameters.  Abort when a parameter other than one of the ones we
     are looking for is found.                                                                                        */
  for (;;)
  {

    /* Initialize Return_Value to Try_Again.  We will key off of this value later in the loop. */
    Return_Value = Try_Again;

    /* Check for optional parameters */
    if ( ! Size_Parameters_Found )
    {

      Return_Value = Size_Parameters(Temp_Node, FALSE, TRUE);
      if ( Return_Value == Token_Accepted )
        Size_Parameters_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;

    }

    if ( ( Return_Value == Try_Again ) && ( ! Region_Found ) )
    {
      Return_Value = Get_Region(Temp_Node, FALSE, TRUE);
      if ( Return_Value == Token_Accepted )
        Region_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }

    if ( ( Return_Value == Try_Again ) && ( ! Plugin_Parameters_Found ) )
    {
      Return_Value = Get_Plugin(Temp_Node, FALSE, TRUE);
      if ( Return_Value == Token_Accepted )
        Plugin_Parameters_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }

    if ( ( Return_Value == Try_Again ) && ( ! Volume_Parameters_Found ) )
    {
      Return_Value = Get_Volume(Temp_Node, FALSE, TRUE);
      if ( Return_Value == Token_Accepted )
        Volume_Parameters_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }

    if ( ( Return_Value == Try_Again ) && ( ! Container_Parameters_Found ) )
    {
      Return_Value = Get_Container(Temp_Node, FALSE, TRUE);
      if ( Return_Value == Token_Accepted )
        Container_Parameters_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }

    if ( ( Return_Value == Try_Again ) && ( ! Expand_Shrink_Found  ) )
    {
      Return_Value = Get_Expand_Shrink(Temp_Node, FALSE, TRUE);
      if ( Return_Value == Token_Accepted )
        Expand_Shrink_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }

    if ( ( Return_Value == Try_Again ) && ( ! List_Options_Found ) )
    {
      Return_Value = Parse_Query_List_Options(Temp_Node, TRUE);
      if ( Return_Value == Token_Accepted )
        List_Options_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }

    if ( ( Return_Value == Try_Again ) && ( ! Unclaimed_Found  ) )
    {
      Return_Value = Get_Unclaimed(Temp_Node, FALSE, TRUE);
      if ( Return_Value == Token_Accepted )
        Unclaimed_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }

    /* Save any parameters we may have found. */
    if ( ! Terminate_Lookahead( Current_Node, Temp_Node, ( Return_Value == Token_Accepted ) ) )
      return Error;

    if (Nested_Lookahead)
    {

      if ( Return_Value != Token_Accepted )
        if ( !Restore_Current_Token( Temp2_Node ) )
          return Error;

      if ( !Terminate_Lookahead( Current_Node, Temp2_Node, ( Return_Value == Token_Accepted ) ) )
        return Error;

      Nested_Lookahead = FALSE;
    }

    if ( Return_Value != Token_Accepted )
    {

      /* No parameter match has been found. */
      break;

    }

    /* We need to check for additional parameters. */

    /* Since this parameter is optional, activate look ahead. */
    Temp2_Node = Activate_Lookahead();
    if ( Temp2_Node == NULL)
      return Error;

    /* Check for a comma which would indicate possible additional parameters. */
    Return_Value = Check_For_Comma(TRUE);
    if ( Return_Value != Token_Accepted )
    {
      if ( !Restore_Current_Token( Temp2_Node ) )
        return Error;

      if ( ! Terminate_Lookahead( Current_Node, Temp2_Node, FALSE ) )
        return Error;

      break;
    }

    /* We want a nested lookahead context here so that we don't have to test for the comma
       everytime, yet we still want to be able to go back to the comma should we not find
       the correct option after the comma.                                                  */
    Temp_Node = Activate_Lookahead();
    if ( Temp_Node == NULL)
      return Error;

    Nested_Lookahead = TRUE;

  }

  /* Since all of the parameters we are looking for here are optional, it doesn't matter whether we found any or not. */
  if ( Return_Value != Error )
    return Token_Accepted;
  else
    return Error;

}

static Parse_Results Parse_Query_Containers_Parameters(Executable_Node * Current_Node, BOOLEAN Optional)
{
  Executable_Node * Temp_Node = NULL;                  /* Used for lookahead operations. */
  Executable_Node * Temp2_Node = NULL;                 /* Used for lookahead operations. */
  Parse_Results     Return_Value;                      /* Used to track the return value from other functions called by this function. */
  BOOLEAN           Size_Parameters_Found = FALSE;
  BOOLEAN           Plugin_Found = FALSE;
  BOOLEAN           Freespace_Found = FALSE;
  BOOLEAN           Region_Found = FALSE;
  BOOLEAN           List_Options_Found = FALSE;
  BOOLEAN           Nested_Lookahead = FALSE;

  /* Since these parameters are optional, activate look ahead. */
  Temp_Node = Activate_Lookahead();
  if ( Temp_Node == NULL)
    return Error;

  /* Loop looking for each of the possible optional parameters.  Abort when a parameter other than one of the ones we
     are looking for is found.                                                                                        */
  for (;;)
  {

    /* Initialize Return_Value to Try_Again.  We will key off of this value later in the loop. */
    Return_Value = Try_Again;

    /* Check for optional parameters */
    if ( ( ! Size_Parameters_Found ) && ( ! Region_Found ) )
    {

      Return_Value = Size_Parameters(Temp_Node, FALSE, TRUE);
      if ( Return_Value == Token_Accepted )
        Size_Parameters_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;

    }

    if ( ( Return_Value == Try_Again ) && ( ! List_Options_Found ) )
    {
      Return_Value = Parse_Query_List_Options(Temp_Node, TRUE);
      if ( Return_Value == Token_Accepted )
        List_Options_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }

    if ( ( Return_Value == Try_Again ) && 
         ( ! Region_Found ) &&
         ( ! Plugin_Found ) &&
         ( ! Freespace_Found ) &&
         ( ! Size_Parameters_Found )
       )
    {
      Return_Value = Get_Region(Temp_Node, FALSE, TRUE);
      if ( Return_Value == Token_Accepted )
        Region_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }

    if ( ( Return_Value == Try_Again ) && 
         ( ! Plugin_Found ) &&
         ( ! Region_Found )
       )
    {
      Return_Value = Get_Plugin(Temp_Node, FALSE, TRUE);
      if ( Return_Value == Token_Accepted )
        Plugin_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }

    if ( ( Return_Value == Try_Again ) && 
         ( ! Freespace_Found ) &&
         ( ! Region_Found ) 
       )
    {
      Return_Value = Get_Freespace(Temp_Node, FALSE, TRUE);
      if ( Return_Value == Token_Accepted )
      {
        Freespace_Found = TRUE;
        Size_Parameters_Found = FALSE;
      }
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }

    /* Save any parameters we may have found. */
    if ( ! Terminate_Lookahead( Current_Node, Temp_Node, ( Return_Value == Token_Accepted ) ) )
      return Error;

    if (Nested_Lookahead)
    {

      if ( Return_Value != Token_Accepted )
        if ( !Restore_Current_Token( Temp2_Node ) )
          return Error;

      if ( !Terminate_Lookahead( Current_Node, Temp2_Node, ( Return_Value == Token_Accepted ) ) )
        return Error;

      Nested_Lookahead = FALSE;
    }

    if ( Return_Value != Token_Accepted )
    {

      /* No parameter match has been found. */
      break;

    }

    /* We need to check for additional parameters. */

    /* Since this parameter is optional, activate look ahead. */
    Temp2_Node = Activate_Lookahead();
    if ( Temp2_Node == NULL)
      return Error;

    /* Check for a comma which would indicate possible additional parameters. */
    Return_Value = Check_For_Comma(TRUE);
    if ( Return_Value != Token_Accepted )
    {
      if ( !Restore_Current_Token( Temp2_Node ) )
        return Error;

      if ( ! Terminate_Lookahead( Current_Node, Temp2_Node, FALSE ) )
        return Error;

      break;
    }

    /* We want a nested lookahead context here so that we don't have to test for the comma
       everytime, yet we still want to be able to go back to the comma should we not find
       the correct option after the comma.                                                  */
    Temp_Node = Activate_Lookahead();
    if ( Temp_Node == NULL)
      return Error;

    Nested_Lookahead = TRUE;

  }

  /* Since all of the parameters we are looking for here are optional, it doesn't matter whether we found any or not. */
  if ( Return_Value != Error )
    return Token_Accepted;
  else
    return Error;

}


static Parse_Results Parse_Query_Regions_Parameters(Executable_Node * Current_Node, BOOLEAN Optional)
{
  Executable_Node * Temp_Node = NULL;                  /* Used for lookahead operations. */
  Executable_Node * Temp2_Node = NULL;                 /* Used for lookahead operations. */
  Parse_Results     Return_Value;                      /* Used to track the return value from other functions called by this function. */
  BOOLEAN           Size_Parameters_Found = FALSE;
  BOOLEAN           Expand_Shrink_Found = FALSE;
  BOOLEAN           List_Options_Found = FALSE;
  BOOLEAN           Plugin_Parameters_Found = FALSE;
  BOOLEAN           Volume_Found = FALSE;
  BOOLEAN           Object_Found = FALSE;
  BOOLEAN           Container_Found = FALSE;
  BOOLEAN           Unclaimed_Found = FALSE;
  BOOLEAN           Nested_Lookahead = FALSE;

  /* Since these parameters are optional, activate look ahead. */
  Temp_Node = Activate_Lookahead();
  if ( Temp_Node == NULL)
    return Error;

  /* Loop looking for each of the possible optional parameters.  Abort when a parameter other than one of the ones we
     are looking for is found.                                                                                        */
  for (;;)
  {

    /* Initialize Return_Value to Try_Again.  We will key off of this value later in the loop. */
    Return_Value = Try_Again;

    /* Check for optional parameters */
    if ( ! Size_Parameters_Found )
    {

      Return_Value = Size_Parameters(Temp_Node, FALSE, TRUE);
      if ( Return_Value == Token_Accepted )
        Size_Parameters_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;

    }

    if ( ( Return_Value == Try_Again ) && ( ! Expand_Shrink_Found  ) )
    {
      Return_Value = Get_Expand_Shrink(Temp_Node, FALSE, TRUE);
      if ( Return_Value == Token_Accepted )
        Expand_Shrink_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }

    if ( ( Return_Value == Try_Again ) && ( ! List_Options_Found ) )
    {
      Return_Value = Parse_Query_List_Options(Temp_Node, TRUE);
      if ( Return_Value == Token_Accepted )
        List_Options_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }

    if ( ( Return_Value == Try_Again ) && ( ! Plugin_Parameters_Found ) )
    {
      Return_Value = Get_Plugin(Temp_Node, FALSE, TRUE);
      if ( Return_Value == Token_Accepted )
        Plugin_Parameters_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }
    
    if ( ( Return_Value == Try_Again ) && ( !Volume_Found ) )
    {
      Return_Value = Get_Volume(Temp_Node, FALSE, TRUE);
      if ( Return_Value == Token_Accepted )
        Volume_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }

    if ( ( Return_Value == Try_Again ) && ( !Object_Found ) )
    {
      Return_Value = Get_Object(Temp_Node, FALSE, TRUE);
      if ( Return_Value == Token_Accepted )
        Object_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }

    if ( ( Return_Value == Try_Again ) && ( !Container_Found ) )
    {
      Return_Value = Get_Container(Temp_Node, FALSE, TRUE);
      if ( Return_Value == Token_Accepted )
        Container_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }

    if ( ( Return_Value == Try_Again ) && ( !Unclaimed_Found  ) )
    {
      Return_Value = Get_Unclaimed(Temp_Node, FALSE, TRUE);
      if ( Return_Value == Token_Accepted )
        Unclaimed_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }

    /* Save any parameters we may have found. */
    if ( ! Terminate_Lookahead( Current_Node, Temp_Node, ( Return_Value == Token_Accepted ) ) )
      return Error;

    if (Nested_Lookahead)
    {

      if ( Return_Value != Token_Accepted )
        if ( !Restore_Current_Token( Temp2_Node ) )
          return Error;

      if ( !Terminate_Lookahead( Current_Node, Temp2_Node, ( Return_Value == Token_Accepted ) ) )
        return Error;

      Nested_Lookahead = FALSE;
    }

    if ( Return_Value != Token_Accepted )
    {

      /* No parameter match has been found. */
      break;

    }

    /* We need to check for additional parameters. */

    /* Since this parameter is optional, activate look ahead. */
    Temp2_Node = Activate_Lookahead();
    if ( Temp2_Node == NULL)
      return Error;

    /* Check for a comma which would indicate possible additional parameters. */
    Return_Value = Check_For_Comma(TRUE);
    if ( Return_Value != Token_Accepted )
    {
      if ( !Restore_Current_Token( Temp2_Node ) )
        return Error;

      if ( ! Terminate_Lookahead( Current_Node, Temp2_Node, FALSE ) )
        return Error;

      break;
    }

    /* We want a nested lookahead context here so that we don't have to test for the comma
       everytime, yet we still want to be able to go back to the comma should we not find
       the correct option after the comma.                                                  */
    Temp_Node = Activate_Lookahead();
    if ( Temp_Node == NULL)
      return Error;

    Nested_Lookahead = TRUE;

  }

  /* Since all of the parameters we are looking for here are optional, it doesn't matter whether we found any or not. */
  if ( Return_Value != Error)
    Return_Value = Token_Accepted;   /* The command is complete. */

  return Return_Value;

}


static Parse_Results Parse_Query_Extended_Info_Parameters(Executable_Node * Current_Node, BOOLEAN Optional)
{
  Executable_Node * Temp_Node = NULL;                /* Used for lookahead operations. */
  Parse_Results     Return_Value;                    /* Used to track the return value from other functions called by this function. */

  /* Check for a comma. */
  Return_Value = Check_For_Comma(Optional);

  if ( Return_Value == Error )
    return Return_Value;

  if ( Return_Value == Try_Again )
  {

    if ( Optional )
      return Try_Again;

    Report_Parser_Error(Expecting_EVMS_Identifier, Command_Filename, Current_Token);
    Return_Value = Error;

  }
  else
  {
    /* Get the object we are to get the extended info for. */
    Return_Value = Specify_Object(Current_Node, Translate_Name_To_Handle, FALSE, Optional);
    if ( Return_Value == Token_Accepted )
    {
      /* Now get the optional field name. */

      /* Activate lookahead. */
      Temp_Node = Activate_Lookahead();
      if ( Temp_Node == NULL)
        return Error;

      /* Check for a comma. */
      Return_Value = Check_For_Comma(TRUE);
      if ( Return_Value == Token_Accepted )
        Return_Value = Name(Temp_Node, TRUE, TRUE);

      if ( ( Return_Value != Token_Accepted ) && 
           ( !Restore_Current_Token( Temp_Node ) )
         )
        return Error;


      /* Exit Lookahead mode. */
      if ( ! Terminate_Lookahead( Current_Node, Temp_Node, (Return_Value == Token_Accepted) ) )
        return Error;

      if ( Return_Value != Error )
        Return_Value = Token_Accepted;  /* Name was optional, so the command is complete with or without it! */

    }

  }

  return Return_Value;
}


static Parse_Results Parse_Object_List(Executable_Node * Current_Node, BOOLEAN Optional)
{
  Executable_Node * Child = NULL;                    /* The node used to represent the current command being parsed. */
  Executable_Node * Temp_Node = NULL;                /* Used for lookahead operations. */
  Parse_Results     Return_Value;                    /* Used to track the return value from other functions called by this function. */
  uint              Count = 0;                       /* Used to track how many items are in the list. */

  /* Create the child node and add it to the children list of Current_Node.  We will fill in the contents of Child later. */
  Child = Make_Child_Node(Current_Node, TRUE);
  if ( Child == NULL )
    return Error;

  Child->NodeType = ObjectList;

  /* We have a list of objects separated by commas.  A comma preceeds the list as well. */

  for (;;)
  {

    /* Activate lookahead. */
    Temp_Node = Activate_Lookahead();
    if ( Temp_Node == NULL)
      return Error;

    /* Do we have a comma? */
    Return_Value = Check_For_Comma(TRUE);
    if ( Return_Value != Token_Accepted )
      break;

    Return_Value = Specify_Object(Temp_Node, Translate_Name_To_Handle, FALSE, TRUE);
    if ( Return_Value == Token_Accepted )
      Count++;
    else
      break;

    /* Exit Lookahead mode. */
    if ( ! Terminate_Lookahead( Child, Temp_Node, TRUE ) )
      return Error;

  }

  /* Restore the current token so that whoever we return to can parse it. */
  if ( !Restore_Current_Token( Temp_Node ) )
    return Error;

  /* Exit Lookahead mode. */
  if ( ! Terminate_Lookahead( Child, Temp_Node, FALSE ) )
    return Error;

  /* Do we have any objects in the list? */
  if ( Count == 0 )
  {
    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Object_List, Command_Filename, NULL);

    return Error;
  }

  return Token_Accepted;

}

static Parse_Results Parse_Query_Disks_Parameters(Executable_Node * Current_Node, BOOLEAN Optional)
{
  Executable_Node * Temp_Node = NULL;                  /* Used for lookahead operations. */
  Executable_Node * Temp2_Node = NULL;                 /* Used for lookahead operations. */
  Parse_Results     Return_Value;                      /* Used to track the return value from other functions called by this function. */
  BOOLEAN           Size_Parameters_Found = FALSE;
  BOOLEAN           Plugin_Parameters_Found = FALSE;
  BOOLEAN           Unclaimed_Found = FALSE;
  BOOLEAN           Nested_Lookahead = FALSE;
  BOOLEAN           List_Options_Found = FALSE;
  BOOLEAN           Expand_Shrink_Found = FALSE;

  /* Since these parameters are optional, activate look ahead. */
  Temp_Node = Activate_Lookahead();
  if ( Temp_Node == NULL)
    return Error;

  /* Loop looking for each of the possible optional parameters.  Abort when a parameter other than one of the ones we
     are looking for is found.                                                                                        */
  for (;;)
  {

    /* Initialize Return_Value to Try_Again.  We will key off of this value later in the loop. */
    Return_Value = Try_Again;

    /* Check for optional parameters */
    if ( ! Size_Parameters_Found )
    {

      Return_Value = Size_Parameters(Temp_Node, FALSE, TRUE);
      if ( Return_Value == Token_Accepted )
        Size_Parameters_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;

    }

    if ( ( Return_Value == Try_Again ) && ( ! Expand_Shrink_Found  ) )
    {
      Return_Value = Get_Expand_Shrink(Temp_Node, FALSE, TRUE);
      if ( Return_Value == Token_Accepted )
        Expand_Shrink_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }

    if ( ( Return_Value == Try_Again ) && ( ! List_Options_Found ) )
    {
      Return_Value = Parse_Query_List_Options(Temp_Node, TRUE);
      if ( Return_Value == Token_Accepted )
        List_Options_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }

    if ( ( Return_Value == Try_Again ) && ( ! Plugin_Parameters_Found ) )
    {
      Return_Value = Get_Plugin(Temp_Node, FALSE, TRUE);
      if ( Return_Value == Token_Accepted )
        Plugin_Parameters_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }

    if ( ( Return_Value == Try_Again ) && ( ! Unclaimed_Found  ) )
    {
      Return_Value = Get_Unclaimed(Temp_Node, FALSE, TRUE);
      if ( Return_Value == Token_Accepted )
        Unclaimed_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }

    /* Save any parameters we may have found. */
    if ( ! Terminate_Lookahead( Current_Node, Temp_Node, ( Return_Value == Token_Accepted ) ) )
      return Error;

    if (Nested_Lookahead)
    {

      if ( Return_Value != Token_Accepted )
        if ( !Restore_Current_Token( Temp2_Node ) )
          return Error;

      if ( !Terminate_Lookahead( Current_Node, Temp2_Node, ( Return_Value == Token_Accepted ) ) )
        return Error;

      Nested_Lookahead = FALSE;
    }

    if ( Return_Value != Token_Accepted )
    {

      /* No parameter match has been found. */
      break;

    }

    /* We need to check for additional parameters. */

    /* Since this parameter is optional, activate look ahead. */
    Temp2_Node = Activate_Lookahead();
    if ( Temp2_Node == NULL)
      return Error;

    /* Check for a comma which would indicate possible additional parameters. */
    Return_Value = Check_For_Comma(TRUE);
    if ( Return_Value != Token_Accepted )
    {
      if ( !Restore_Current_Token( Temp2_Node ) )
        return Error;

      if ( ! Terminate_Lookahead( Current_Node, Temp2_Node, FALSE ) )
        return Error;

      break;
    }

    /* We want a nested lookahead context here so that we don't have to test for the comma
       everytime, yet we still want to be able to go back to the comma should we not find
       the correct option after the comma.                                                  */
    Temp_Node = Activate_Lookahead();
    if ( Temp_Node == NULL)
      return Error;

    Nested_Lookahead = TRUE;

  }

  /* Since all of the parameters we are looking for here are optional, it doesn't matter whether we found any or not. */
  return Token_Accepted;
}

static Parse_Results Parse_Query_Segments_Parameters(Executable_Node * Current_Node, BOOLEAN Optional)
{
  Executable_Node * Temp_Node = NULL;                  /* Used for lookahead operations. */
  Executable_Node * Temp2_Node = NULL;                 /* Used for lookahead operations. */
  Parse_Results     Return_Value;                      /* Used to track the return value from other functions called by this function. */
  BOOLEAN           Size_Parameters_Found = FALSE;
  BOOLEAN           Plugin_Parameters_Found = FALSE;
  BOOLEAN           Container_Parameters_Found = FALSE;
  BOOLEAN           Disk_Parameters_Found = FALSE;
  BOOLEAN           Unclaimed_Found = FALSE;
  BOOLEAN           Nested_Lookahead = FALSE;
  BOOLEAN           List_Options_Found = FALSE;
  BOOLEAN           Expand_Shrink_Found = FALSE;

  /* Since these parameters are optional, activate look ahead. */
  Temp_Node = Activate_Lookahead();
  if ( Temp_Node == NULL)
    return Error;

  /* Loop looking for each of the possible optional parameters.  Abort when a parameter other than one of the ones we
     are looking for is found.                                                                                        */
  for (;;)
  {

    /* Initialize Return_Value to Try_Again.  We will key off of this value later in the loop. */
    Return_Value = Try_Again;

    /* Check for optional parameters */
    if ( ! Size_Parameters_Found )
    {

      Return_Value = Size_Parameters(Temp_Node, FALSE, TRUE);
      if ( Return_Value == Token_Accepted )
        Size_Parameters_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;

    }

    if ( ( Return_Value == Try_Again ) && ( ! Plugin_Parameters_Found ) )
    {
      Return_Value = Get_Plugin(Temp_Node, FALSE, TRUE);
      if ( Return_Value == Token_Accepted )
        Plugin_Parameters_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }

    if ( ( Return_Value == Try_Again ) && ( ! Container_Parameters_Found ) )
    {
      Return_Value = Get_Container(Temp_Node, FALSE, TRUE);
      if ( Return_Value == Token_Accepted )
        Container_Parameters_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }

    if ( ( Return_Value == Try_Again ) && ( ! Disk_Parameters_Found ) )
    {
      Return_Value = Get_Disk(Temp_Node, FALSE, TRUE);
      if ( Return_Value == Token_Accepted )
        Disk_Parameters_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }

    if ( ( Return_Value == Try_Again ) && ( ! Expand_Shrink_Found  ) )
    {
      Return_Value = Get_Expand_Shrink(Temp_Node, FALSE, TRUE);
      if ( Return_Value == Token_Accepted )
        Expand_Shrink_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }

    if ( ( Return_Value == Try_Again ) && ( ! List_Options_Found ) )
    {
      Return_Value = Parse_Query_List_Options(Temp_Node, TRUE);
      if ( Return_Value == Token_Accepted )
        List_Options_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }

    if ( ( Return_Value == Try_Again ) && ( ! Unclaimed_Found  ) )
    {
      Return_Value = Get_Unclaimed(Temp_Node, FALSE, TRUE);
      if ( Return_Value == Token_Accepted )
        Unclaimed_Found = TRUE;
      else
        if ( !Restore_Current_Token( Temp_Node ) )
        return Error;
    }

    /* Save any parameters we may have found. */
    if ( ! Terminate_Lookahead( Current_Node, Temp_Node, ( Return_Value == Token_Accepted ) ) )
      return Error;

    if (Nested_Lookahead)
    {

      if ( Return_Value != Token_Accepted )
        if ( !Restore_Current_Token( Temp2_Node ) )
          return Error;

      if ( !Terminate_Lookahead( Current_Node, Temp2_Node, ( Return_Value == Token_Accepted ) ) )
        return Error;

      Nested_Lookahead = FALSE;
    }

    if ( Return_Value != Token_Accepted )
    {

      /* No parameter match has been found. */
      break;

    }

    /* We need to check for additional parameters. */

    /* Since this parameter is optional, activate look ahead. */
    Temp2_Node = Activate_Lookahead();
    if ( Temp2_Node == NULL)
      return Error;

    /* Check for a comma which would indicate possible additional parameters. */
    Return_Value = Check_For_Comma(TRUE);
    if ( Return_Value != Token_Accepted )
    {
      if ( !Restore_Current_Token( Temp2_Node ) )
        return Error;

      if ( ! Terminate_Lookahead( Current_Node, Temp2_Node, FALSE ) )
        return Error;

      break;
    }

    /* We want a nested lookahead context here so that we don't have to test for the comma
       everytime, yet we still want to be able to go back to the comma should we not find
       the correct option after the comma.                                                  */
    Temp_Node = Activate_Lookahead();
    if ( Temp_Node == NULL)
      return Error;

    Nested_Lookahead = TRUE;

  }

  /* Since all of the parameters we are looking for here are optional, it doesn't matter whether we found any or not. */
  return Token_Accepted;

}

static Parse_Results Parse_Query_Acceptable_Parameters(Executable_Node * Current_Node, BOOLEAN Optional)
{
  Executable_Node * Child = NULL;         /* The node used to represent the current command being parsed. */
  Parse_Results     Return_Value = Error; /* Used to track the return value from other functions called by this function. */

  /* Create the child node and add it to the children list of Current_Node.  We will fill in the contents of Child later. */
  Child = Make_Child_Node(Current_Node, TRUE);
  if ( Child == NULL )
    return Error;

  /* Get the token to parse. */
  Next_Token();

  /* Check for EOF or InvalidCharacter */
  if ( ( Current_Token == NULL ) || ( Current_Token->Characterization == EofToken ) || ( Current_Token->Characterization == InvalidCharacter ) )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Query_Acceptable_Parameters, Command_Filename, Current_Token);

    return Error;

  }

  /* The parameter we are looking for is characterized as a KeyWord.  */
  if ( Current_Token->Characterization != KeyWord )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Query_Acceptable_Parameters, Command_Filename, Current_Token);

    return Error;

  }

  switch ( Current_Token->TokenText[0] )
  {
    case 'C' : /* Create */
      if ( ( Current_Token->TokenText == CreateStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        Child->NodeType = Filter;
        Child->NodeValue = CreateStr;

        Consume_Token();

        Return_Value = Check_For_Comma(Optional);
        if ( Return_Value == Token_Accepted )
          Return_Value = Parse_Query_Acceptable_Create(Child, Optional);
      }
      else
        if (Optional)
        Return_Value = Try_Again;
      else
      {
        Return_Value = Error;
        Report_Parser_Error(Expecting_Query_Acceptable_Parameters, Command_Filename, Current_Token);
      }
      break;
    case 'E' : /* Expand */
      if ( ( Current_Token->TokenText == ExpandStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        Child->NodeType = Filter;
        Child->NodeValue = ExpandStr;

        Consume_Token();

        Return_Value = Check_For_Comma(Optional);
        if ( Return_Value == Token_Accepted )
        {
          Return_Value = Specify_Object(Child, Translate_Name_To_Handle, FALSE, Optional);
          if ( Return_Value == Token_Accepted )
          {
            Return_Value = Parse_Object_List(Child,TRUE);
            if ( Return_Value == Try_Again )
              Return_Value = Token_Accepted;   /* The object list is optional, so if we don't find one we don't care. */
          }

        }

      }
      else
        if (Optional)
        Return_Value = Try_Again;
      else
      {
        Return_Value = Error;
        Report_Parser_Error(Expecting_Query_Acceptable_Parameters, Command_Filename, Current_Token);
      }
      break;
    default:
      if ( Optional )
        Return_Value = Try_Again;
      else
      {
        Return_Value = Error;
        Report_Parser_Error(Expecting_Query_Acceptable_Parameters, Command_Filename, Current_Token);
      }
      break;
  }

  return Return_Value;

}


static Parse_Results Parse_Query_Acceptable_Create(Executable_Node * Current_Node, BOOLEAN Optional)
{
  Executable_Node * Child = NULL;             /* The node used to represent the current command being parsed. */
  Parse_Results     Return_Value = Error;     /* Used to track the return value from other functions called by this function. */
  BOOLEAN           Create_Volume = FALSE;
  BOOLEAN           Create_Container = FALSE;

  /* Get the token to parse. */
  Next_Token();

  /* Check for EOF or InvalidCharacter */
  if ( ( Current_Token == NULL ) || ( Current_Token->Characterization == EofToken ) || ( Current_Token->Characterization == InvalidCharacter ) )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Query_Acceptable_Create, Command_Filename, Current_Token);

    return Error;

  }

  /* The parameter we are looking for is characterized as a KeyWord.  */
  if ( Current_Token->Characterization != KeyWord )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Query_Acceptable_Create, Command_Filename, Current_Token);

    return Error;

  }

  /* Assume failure. */
  if (Optional)
    Return_Value = Try_Again;
  else
    Return_Value = Error;

  /* See what we have. */
  switch ( Current_Token->TokenText[0] )
  {
    case 'V' : /* Volume */
      if ( ( Current_Token->TokenText == VolumeStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {

        /* Create the child node and add it to the children list of Current_Node. */
        Child = Make_Child_Node(Current_Node, TRUE);
        if ( Child == NULL )
          return Error;

        /* Fill in the fields of Child */
        Child->NodeType = Topmost_Objects;
        Child->NodeValue = VolumeStr;

        Return_Value = Token_Accepted;
        Create_Volume = TRUE;
      }
      break;
    case 'O' : /* Object */
      if ( ( Current_Token->TokenText == ObjectStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        /* We don't need a new node for queries of this type. */
        Return_Value = Token_Accepted;
      }
      break;
    case 'R' : /* Region */
      if ( ( Current_Token->TokenText == RegionStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        /* We don't need a new node for queries of this type. */
        Return_Value = Token_Accepted;
      }
      break;
    case 'S' : /* Segment */
      if ( ( Current_Token->TokenText == SegmentStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        /* We don't need a new node for queries of this type. */
        Return_Value = Token_Accepted;
      }
      break;
    case 'C' : /* Container */
      if ( ( Current_Token->TokenText == ContainerStr ) ||
           ( Current_Token->TokenLength == 1 )
         )
      {
        /* Create the child node and add it to the children list of Current_Node. */
        Child = Make_Child_Node(Current_Node, TRUE);
        if ( Child == NULL )
          return Error;

        /* Fill in the fields of Child */
        Child->NodeType = Task_Adjust;
        Child->NodeValue = ContainerStr;

        Create_Container = TRUE;
        Return_Value = Token_Accepted;

      }
      break;
    default:
      break;
  }

  if ( Return_Value == Error )
    Report_Parser_Error(Expecting_Query_Acceptable_Create, Command_Filename, Current_Token);

  if ( Return_Value == Token_Accepted )
  {

    Consume_Token();

    if ( ! Create_Volume )
    {

      Return_Value = Check_For_Comma(Optional);
      if ( Return_Value == Token_Accepted )
      {
        if ( Create_Container )
          Return_Value = Feature_Identifier(Child, Optional, TRUE, TRUE);
        else
          Return_Value = Feature_Identifier(Current_Node, Optional, TRUE, TRUE);
      }

    }

  }

  return Return_Value;

}


static Parse_Results Get_Volume(Executable_Node * Current_Node, BOOLEAN Use_Current_Token, BOOLEAN Optional)
{
  Executable_Node * Child = NULL;         /* The node used to represent the current command being parsed. */
  Parse_Results     Return_Value = Error; /* Used to track the return value from other functions called by this function. */

  if ( ! Use_Current_Token )
    Next_Token();

  /* Check for EOF, InvalidCharacter, or incorrect characterization */
  if ( ( Current_Token == NULL ) ||
       ( Current_Token->Characterization == EofToken ) ||
       ( Current_Token->Characterization == InvalidCharacter ) ||
       ( Current_Token->Characterization != KeyWord )
     )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Volume_Identifier, Command_Filename, Current_Token);

    return Error;

  }

  /* Is the token of the Volume keyword? */
  if ( ( Current_Token->TokenText == VolumeStr ) ||
       ( ( Current_Token->TokenLength == 1 ) &&
         ( Current_Token->TokenText[0] == 'V' )
       )
     )
  {
    /* We have the Volume keyword. */

    /* Create the child node and add it to the children list of Current_Node. */
    Child = Make_Child_Node(Current_Node, TRUE);
    if ( Child == NULL )
      return Error;

    Child->NodeType = Filter;
    Child->NodeValue = VolumeStr;

    Consume_Token();

    Return_Value = Check_For_EqualSign(Optional);
    if ( Return_Value == Token_Accepted )
      Return_Value = Specify_Object(Child, Name_To_Volume_Handle, FALSE, Optional);

  }
  else
  {
    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Volume_Identifier, Command_Filename, Current_Token);

  }

  return Return_Value;

}

static Parse_Results Get_Object(Executable_Node * Current_Node, BOOLEAN Use_Current_Token, BOOLEAN Optional)
{
  Executable_Node * Child = NULL;         /* The node used to represent the current command being parsed. */
  Parse_Results     Return_Value = Error; /* Used to track the return value from other functions called by this function. */

  if ( ! Use_Current_Token )
    Next_Token();

  /* Check for EOF, InvalidCharacter, or incorrect characterization */
  if ( ( Current_Token == NULL ) ||
       ( Current_Token->Characterization == EofToken ) ||
       ( Current_Token->Characterization == InvalidCharacter ) ||
       ( Current_Token->Characterization != KeyWord )
     )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Object_Identifier, Command_Filename, Current_Token);

    return Error;

  }

  /* Is the token of the Volume keyword? */
  if ( ( Current_Token->TokenText == ObjectStr ) ||
       ( ( Current_Token->TokenLength == 1 ) &&
         ( Current_Token->TokenText[0] == 'O' )
       )
     )
  {
    /* We have the Object keyword. */

    /* Create the child node and add it to the children list of Current_Node. */
    Child = Make_Child_Node(Current_Node, TRUE);
    if ( Child == NULL )
      return Error;

    Child->NodeType = Filter;
    Child->NodeValue = ObjectStr;

    Consume_Token();

    Return_Value = Check_For_EqualSign(Optional);
    if ( Return_Value == Token_Accepted )
      Return_Value = Specify_Object(Child, Name_To_Object_Handle, FALSE, Optional);

  }
  else
  {
    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Object_Identifier, Command_Filename, Current_Token);

  }

  return Return_Value;

}

static Parse_Results Get_Plugin(Executable_Node * Current_Node, BOOLEAN Use_Current_Token, BOOLEAN Optional)
{
  Executable_Node * Child = NULL;         /* The node used to represent the current command being parsed. */
  Parse_Results     Return_Value = Error; /* Used to track the return value from other functions called by this function. */

  if ( ! Use_Current_Token )
    Next_Token();

  /* Check for EOF, InvalidCharacter, or incorrect characterization */
  if ( ( Current_Token == NULL ) ||
       ( Current_Token->Characterization == EofToken ) ||
       ( Current_Token->Characterization == InvalidCharacter ) ||
       ( Current_Token->Characterization != KeyWord )
     )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Plugin_Identifier, Command_Filename, Current_Token);

    return Error;

  }

  /* Is the token the Plugin keyword? */
  if ( ( Current_Token->TokenText == PluginStr ) ||
       ( ( Current_Token->TokenLength == 1 ) &&
         ( Current_Token->TokenText[0] == 'P' )
       )
     )
  {
    /* We have the Plugin keyword. */

    /* Create the child node and add it to the children list of Current_Node.  */
    Child = Make_Child_Node(Current_Node, TRUE);
    if ( Child == NULL )
      return Error;

    Child->NodeType = Filter;
    Child->NodeValue = PluginStr;

    Consume_Token();

    Return_Value = Check_For_EqualSign(Optional);
    if ( Return_Value == Token_Accepted )
      Return_Value = Plugin_Identifier(Child,FALSE, FALSE, Optional);

  }
  else
  {
    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Plugin_Identifier, Command_Filename, Current_Token);

  }

  return Return_Value;

}


static Parse_Results Get_Region(Executable_Node * Current_Node, BOOLEAN Use_Current_Token, BOOLEAN Optional)
{

  Executable_Node * Child = NULL;         /* The node used to represent the current command being parsed. */
  Parse_Results     Return_Value = Error; /* Used to track the return value from other functions called by this function. */

  if ( ! Use_Current_Token )
    Next_Token();

  /* Check for EOF, InvalidCharacter, or incorrect characterization */
  if ( ( Current_Token == NULL ) ||
       ( Current_Token->Characterization == EofToken ) ||
       ( Current_Token->Characterization == InvalidCharacter ) ||
       ( Current_Token->Characterization != KeyWord )
     )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Region_Identifier, Command_Filename, Current_Token);

    return Error;

  }

  /* Is the token of the Volume keyword? */
  if ( ( Current_Token->TokenText == RegionStr ) ||
       ( ( Current_Token->TokenLength == 1 ) &&
         ( Current_Token->TokenText[0] == 'R' )
       )
     )
  {
    /* We have the Region keyword. */

    /* Create the child node and add it to the children list of Current_Node. */
    Child = Make_Child_Node(Current_Node, TRUE);
    if ( Child == NULL )
      return Error;

    Child->NodeType = Filter;
    Child->NodeValue = RegionStr;

    Consume_Token();

    Return_Value = Check_For_EqualSign(Optional);
    if ( Return_Value == Token_Accepted )
      Return_Value = Specify_Object(Child, Name_To_Region_Handle, FALSE, Optional);

  }
  else
  {
    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Region_Identifier, Command_Filename, Current_Token);

  }

  return Return_Value;

}

static Parse_Results Get_Container(Executable_Node * Current_Node, BOOLEAN Use_Current_Token, BOOLEAN Optional)
{

  Executable_Node * Child = NULL;         /* The node used to represent the current command being parsed. */
  Parse_Results     Return_Value = Error; /* Used to track the return value from other functions called by this function. */

  if ( ! Use_Current_Token )
    Next_Token();

  /* Check for EOF, InvalidCharacter, or incorrect characterization */
  if ( ( Current_Token == NULL ) ||
       ( Current_Token->Characterization == EofToken ) ||
       ( Current_Token->Characterization == InvalidCharacter ) ||
       ( Current_Token->Characterization != KeyWord )
     )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Container_Identifier, Command_Filename, Current_Token);

    return Error;

  }

  /* Is the token of the Volume keyword? */
  if ( ( Current_Token->TokenText == ContainerStr ) ||
       ( ( Current_Token->TokenLength == 1 ) &&
         ( Current_Token->TokenText[0] == 'C' )
       )
     )
  {
    /* We have the Container keyword. */

    /* Create the child node and add it to the children list of Current_Node. */
    Child = Make_Child_Node(Current_Node, TRUE);
    if ( Child == NULL )
      return Error;

    Child->NodeType = Filter;
    Child->NodeValue = ContainerStr;

    Consume_Token();

    Return_Value = Check_For_EqualSign(Optional);
    if ( Return_Value == Token_Accepted )
      Return_Value = Specify_Object(Child, Name_To_Container_Handle, FALSE, Optional);

  }
  else
  {
    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Container_Identifier, Command_Filename, Current_Token);

  }

  return Return_Value;

}

static Parse_Results Get_Disk(Executable_Node * Current_Node, BOOLEAN Use_Current_Token, BOOLEAN Optional)
{
  Executable_Node * Child = NULL;         /* The node used to represent the current command being parsed. */
  Parse_Results     Return_Value = Error; /* Used to track the return value from other functions called by this function. */

  if ( ! Use_Current_Token )
    Next_Token();

  /* Check for EOF, InvalidCharacter, or incorrect characterization */
  if ( ( Current_Token == NULL ) ||
       ( Current_Token->Characterization == EofToken ) ||
       ( Current_Token->Characterization == InvalidCharacter ) ||
       ( Current_Token->Characterization != KeyWord )
     )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Disk_Identifier, Command_Filename, Current_Token);

    return Error;

  }

  /* Is the token of the Disk keyword? */
  if ( ( Current_Token->TokenText == DiskStr ) ||
       ( ( Current_Token->TokenLength == 1 ) &&
         ( Current_Token->TokenText[0] == 'D' )
       )
     )
  {
    /* We have the Disk keyword. */

    /* Create the child node and add it to the children list of Current_Node. */
    Child = Make_Child_Node(Current_Node, TRUE);
    if ( Child == NULL )
      return Error;

    Child->NodeType = Filter;
    Child->NodeValue = DiskStr;

    Consume_Token();

    Return_Value = Check_For_EqualSign(Optional);
    if ( Return_Value == Token_Accepted )
      Return_Value = Specify_Object(Child, Name_To_Disk_Handle, FALSE, Optional);

  }
  else
  {
    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Disk_Identifier, Command_Filename, Current_Token);

  }

  return Return_Value;
}

static Parse_Results Get_Expand_Shrink(Executable_Node * Current_Node, BOOLEAN Use_Current_Token, BOOLEAN Optional)
{

  Executable_Node * Child = NULL;         /* The node used to represent the current command being parsed. */

  if ( ! Use_Current_Token )
    Next_Token();

  /* Check for EOF, InvalidCharacter, or incorrect characterization */
  if ( ( Current_Token == NULL ) ||
       ( Current_Token->Characterization == EofToken ) ||
       ( Current_Token->Characterization == InvalidCharacter ) ||
       ( Current_Token->Characterization != KeyWord )
     )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Unclaimed_Identifier, Command_Filename, Current_Token);

    return Error;

  }

  /* Is the token the Expandable keyword? */
  if ( ( Current_Token->TokenText == ExpandableStr ) ||
       ( ( Current_Token->TokenLength == 1 ) &&
         ( Current_Token->TokenText[0] == 'E' )
       )
     )
  {
    /* We have the Expandable keyword. */

    /* Create the child node and add it to the children list of Current_Node. */
    Child = Make_Child_Node(Current_Node, TRUE);
    if ( Child == NULL )
      return Error;

    Child->NodeType = Filter;
    Child->NodeValue = ExpandableStr;

    Consume_Token();

  }
  else
    if ( ( Current_Token->TokenText == ShrinkableStr ) ||
         ( ( Current_Token->TokenLength == 1 ) &&
           ( Current_Token->TokenText[0] == 'S' )
         )
       )
  {
    /* Create the child node and add it to the children list of Current_Node. */
    Child = Make_Child_Node(Current_Node, TRUE);
    if ( Child == NULL )
      return Error;

    Child->NodeType = Filter;
    Child->NodeValue = ShrinkableStr;

    Consume_Token();

  }
  else
  {
    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Expandable_Shrinkable_Identifier, Command_Filename, Current_Token);
    return Error;

  }

  return Token_Accepted;

}

static Parse_Results Get_Unclaimed(Executable_Node * Current_Node, BOOLEAN Use_Current_Token, BOOLEAN Optional)
{

  Executable_Node * Child = NULL;         /* The node used to represent the current command being parsed. */

  if ( ! Use_Current_Token )
    Next_Token();

  /* Check for EOF, InvalidCharacter, or incorrect characterization */
  if ( ( Current_Token == NULL ) ||
       ( Current_Token->Characterization == EofToken ) ||
       ( Current_Token->Characterization == InvalidCharacter ) ||
       ( Current_Token->Characterization != KeyWord )
     )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Unclaimed_Identifier, Command_Filename, Current_Token);

    return Error;

  }

  /* Is the token of the Volume keyword? */
  if ( ( Current_Token->TokenText == UnclaimedStr ) ||
       ( ( Current_Token->TokenLength == 1 ) &&
         ( Current_Token->TokenText[0] == 'U' )
       )
     )
  {
    /* We have the Unclaimed keyword. */

    /* Create the child node and add it to the children list of Current_Node. */
    Child = Make_Child_Node(Current_Node, TRUE);
    if ( Child == NULL )
      return Error;

    Child->NodeType = Filter;
    Child->NodeValue = UnclaimedStr;

    Consume_Token();

  }
  else
  {
    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Unclaimed_Identifier, Command_Filename, Current_Token);

    return Error;

  }

  return Token_Accepted;

}

static Parse_Results Get_Freespace(Executable_Node * Current_Node, BOOLEAN Use_Current_Token, BOOLEAN Optional)
{

  Executable_Node * Child = NULL;         /* The node used to represent the current command being parsed. */

  if ( ! Use_Current_Token )
    Next_Token();

  /* Check for EOF, InvalidCharacter, or incorrect characterization */
  if ( ( Current_Token == NULL ) ||
       ( Current_Token->Characterization == EofToken ) ||
       ( Current_Token->Characterization == InvalidCharacter ) ||
       ( Current_Token->Characterization != KeyWord )
     )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Freespace_Identifier, Command_Filename, Current_Token);

    return Error;

  }

  /* Is the token the freespace keyword? */
  if ( ( Current_Token->TokenText == FreespaceStr ) ||
       ( ( Current_Token->TokenLength == 1 ) &&
         ( Current_Token->TokenText[0] == 'F' )
       )
     )
  {
    /* We have the Freespace keyword. */

    /* Create the child node and add it to the children list of Current_Node. */
    Child = Make_Child_Node(Current_Node, TRUE);
    if ( Child == NULL )
      return Error;

    Child->NodeType = Filter;
    Child->NodeValue = FreespaceStr;

    Consume_Token();

  }
  else
  {
    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Freespace_Identifier, Command_Filename, Current_Token);

    return Error;

  }

  return Token_Accepted;

}



/*******************************************
 * Utility Functions                       *
 *******************************************/

static void Assign_Text( Executable_Node * Node, TokenType * Token)
{
  if ( Token->In_Lookahead_Queue )
  {

    Node->NodeValue = strdup(Current_Token->TokenText);
    if ( Node->NodeValue == NULL )
    {
      Report_Parser_Error(Out_Of_Memory, Command_Filename, Current_Token);
      exit(ENOMEM);
    }

  }
  else
  {
    Node->NodeValue = Token->TokenText;
    Token->TokenText = NULL;
  }
  return;
}

static void Consume_Token( void )   /* Free's the current token and sets CurrentToken to NULL. */
{

  if ( ( Current_Token != NULL ) && ( !Current_Token->In_Lookahead_Queue ) )
  {

    if ( ( Current_Token->TokenText != NULL ) && 
         ( Current_Token->Characterization != KeyWord )
       )
      free(Current_Token->TokenText);

    free(Current_Token);

  }

  Current_Token = NULL;

}


static void Next_Token( void )      /* Gets the next token to be parsed and sets CurrentToken to point to it. */
{

  uint    Return_Value;
  ADDRESS Unneeded;

  if ( Tokens == NULL )
    Current_Token = GetScreenedToken();
  else
  {
    do
    {
      Return_Value = GetNextObject(Tokens,sizeof(TokenType),TOKEN_CODE,(void *) &Current_Token);
    } while ( ( Return_Value == DLIST_SUCCESS ) && 
              ( ( Current_Token == NULL ) ||
                ( Current_Token == &Null_Token)
              )
            );

    if (Return_Value != DLIST_SUCCESS )
    {

      if ( Return_Value == DLIST_END_OF_LIST )
      {

        /* There were no tokens remaining in the list, so get one. */
        Current_Token = GetScreenedToken();

        /* If lookahead is active, then we must add the new token to the Tokens list. */
        if (LookAhead_Active > 0)
        {

          /* Add Current_Token to the Tokens list. */
          if ( InsertObject(Tokens, sizeof(TokenType), Current_Token, TOKEN_CODE, NULL, AppendToList, TRUE, &Unneeded) != DLIST_SUCCESS )
          {
            /* We must be out of memory! */
            Report_Parser_Error(Out_Of_Memory, NULL, NULL);

            Current_Token = NULL;
            return;

          }

          Current_Token->In_Lookahead_Queue = TRUE;

        }
        else
        {

          /* Get rid of the Tokens list since Lookahead is not active and all of the
             tokens in it have been used -- it is not needed anymore.                 */
          PruneList(Tokens,&Kill_Tokens_List_Entries,NULL);

          DestroyList(&Tokens,TRUE);

        }

      }
      else
      {
        /* This error condition should only happen if there is an internal error of some sort! */
        Report_Parser_Error(Internal_Error_Encountered, NULL, NULL);

        exit(0xFF);

      }

    }

  }

  return;

}

static BOOLEAN Restore_Current_Token( Executable_Node * Temp_Node) /* Restores Current_Token to the value it was at before any Lookahead calls were made. */
{

  ADDRESS  Token_Handle;
  int      Error;

  /* Free Temp_Node */

  /* Free the NodeValue (if any ) */
  if ( Temp_Node->NodeValue != NULL )
    free(Temp_Node->NodeValue);

  /* If the Temp_Node has children, we must kill them also. */
  if ( Temp_Node->Children != NULL )
  {

    /* Kill the children. */
    Error = PruneList(Temp_Node->Children, Kill_Command_List_Entries, NULL);

    if ( Error != DLIST_SUCCESS )
    {
      Report_Parser_Error(Internal_Error_Encountered,NULL,NULL);
      return FALSE;
    }

  }

  /* Now we must deal with resetting the current token. */
  if ( Tokens == NULL )
  {
    Report_Parser_Error(Internal_Error_Encountered,NULL,NULL);
    return FALSE;
  }

  /* Get the handle of the Token to reset to. */
  if ( GoToEndOfList(Lookahead_Handles) != DLIST_SUCCESS)
  {
    Report_Parser_Error(Internal_Error_Encountered,NULL,NULL);
    return FALSE;
  }

  if (GetItem(Lookahead_Handles,sizeof(ADDRESS), &Token_Handle, HANDLE_TAG ,NULL,FALSE) != DLIST_SUCCESS)
  {
    Report_Parser_Error(Internal_Error_Encountered,NULL,NULL);
    return FALSE;
  }

  /* Reset the Tokens list based upon the handle from the Lookahead_Handles list. */
  if ( GoToSpecifiedItem(Tokens, Token_Handle) != DLIST_SUCCESS )
  {
    Report_Parser_Error(Internal_Error_Encountered,NULL,NULL);
    return FALSE;
  }

  /* Get the token from the tokens list. */
  if ( GetObject(Tokens,sizeof(TokenType), TOKEN_CODE, NULL, TRUE, (void*) &Current_Token) != DLIST_SUCCESS )
  {
    Report_Parser_Error(Internal_Error_Encountered,NULL,NULL);
    return FALSE;
  }

  if ( Current_Token == &Null_Token )
    Current_Token = NULL;

  return TRUE;

}

static Executable_Node * Activate_Lookahead(void)
{

  ADDRESS   Lookahead_Handle;
  ADDRESS   NotNeeded;

  if ( Tokens == NULL )
  {
    /* Create the Tokens list. */
    Tokens = CreateList();
    if ( Tokens == NULL )
    {

      /* Since we can't create the Tokens list, we can't do lookahead! */
      Report_Parser_Error(Out_Of_Memory, NULL, NULL);

      return FALSE;

    }

    if (Current_Token == NULL)
      Current_Token = &Null_Token;

    /* Add the Current_Token to the Tokens list. */
    if ( InsertObject(Tokens, sizeof(TokenType), Current_Token, TOKEN_CODE, NULL, AppendToList, FALSE, (void*) &Lookahead_Handle) != DLIST_SUCCESS )
    {
      /* We must be out of memory! */
      Report_Parser_Error(Out_Of_Memory, NULL, NULL);

      return FALSE;

    }

    Current_Token->In_Lookahead_Queue = TRUE;

  }

  if ( Lookahead_Handles == NULL )
  {
    /* Create the Lookahead_Handles list. */
    Lookahead_Handles = CreateList();
    if ( Lookahead_Handles == NULL )
    {

      /* Since we can't create the Lookahead_Handles list, we can't do lookahead! */
      Report_Parser_Error(Out_Of_Memory, NULL, NULL);

      return FALSE;

    }

  }

  /* Get the handle of the Current_Token in the Tokens List. */
  if ( GetHandle(Tokens, &Lookahead_Handle) != DLIST_SUCCESS)
  {
    Report_Parser_Error(Internal_Error_Encountered,NULL,NULL);
    return FALSE;
  }

  /* Add the handle of the Current_Token in the Tokens list to the Lookahead_Handles list. */
  if ( InsertItem(Lookahead_Handles, sizeof(ADDRESS), &Lookahead_Handle, HANDLE_TAG, NULL, AppendToList, FALSE, &NotNeeded) )
  {
    /* We must be out of memory! */
    Report_Parser_Error(Out_Of_Memory, NULL, NULL);

    return FALSE;
  }

  /* Enable Lookahead. */
  LookAhead_Active++;

  /* We need to return a temporary executable node to our caller for use while lookahead is active. */
  return Make_Child_Node(NULL, FALSE);

}


static BOOLEAN Terminate_Lookahead(Executable_Node * Current_Node,
                                   Executable_Node * Temp_Node,
                                   BOOLEAN           Keep_Children)
{
  ADDRESS  Current_Handle = NULL;   /* The handle of the current item in the Tokens list. */

  if (LookAhead_Active > 0)
    LookAhead_Active--;
  else
    return FALSE;

  if ( ( Tokens != NULL ) && (LookAhead_Active == 0) )
  {

    /* Delete all of the items in the Tokens list prior to the current one. */

    /* Get the handle of the current token in the Tokens list. */
    if ( GetHandle(Tokens, &Current_Handle) != DLIST_SUCCESS )
    {
      Report_Parser_Error(Internal_Error_Encountered,NULL,NULL);
      return FALSE;
    }

    if (PruneList( Tokens,&Kill_Tokens_List_Entries,Current_Handle) != DLIST_SUCCESS)
    {
      Report_Parser_Error(Internal_Error_Encountered,NULL,NULL);
      return FALSE;
    }

  }

  /* Remove the last handle from the Lookahead_Handles list. */
  if ( ( GoToEndOfList(Lookahead_Handles) != DLIST_SUCCESS ) ||
       ( DeleteItem(Lookahead_Handles,TRUE,NULL) != DLIST_SUCCESS ) )
    return FALSE;

  if ( Keep_Children )
  {

    /* Now we must append the children list of Temp_Node to the children list of Current_Node.
       Then we can free Temp_Node.                                                              */
    if (AppendList(Current_Node->Children,Temp_Node->Children) != DLIST_SUCCESS)
    {
      Report_Parser_Error(Internal_Error_Encountered,NULL,NULL);
      return FALSE;
    }

  }
  else
  {

    /* We don't need the children, so delete them. */
    if (PruneList(Temp_Node->Children, &Kill_Command_List_Entries, NULL) != DLIST_SUCCESS )
    {
      Report_Parser_Error(Internal_Error_Encountered,NULL,NULL);
      return FALSE;
    }

  }

  if (DestroyList(&Temp_Node->Children,TRUE) != DLIST_SUCCESS)
  {
    Report_Parser_Error(Internal_Error_Encountered,NULL,NULL);
    return FALSE;
  }

  if (Temp_Node->NodeValue != NULL)
    free(Temp_Node->NodeValue);

  free(Temp_Node);

  return TRUE;

}

static BOOLEAN Kill_Tokens_List_Entries (ADDRESS   Object,
                                         TAG       ObjectTag,
                                         uint      ObjectSize,
                                         ADDRESS   ObjectHandle,
                                         ADDRESS   Parameters,
                                         BOOLEAN * FreeMemory,
                                         uint    * Error)
{

  /* Establish easy access to the current token. */
  TokenType * Token_To_Kill = (TokenType *) Object;
  ADDRESS     Token_To_Match =  Parameters;

  /* Assume success. */
  *Error = DLIST_SUCCESS;

  /* Sanity check!  Is the node of the expected type? */
  if ( ObjectTag != TOKEN_CODE )
  {

    /* We have an illegal node in the list!  Abort! */
    *Error =  DLIST_ITEM_TAG_WRONG;
    return FALSE;

  }

  /* Should we kill the current token? */
  if ( ObjectHandle != Token_To_Match)
  {

    if ( ( Token_To_Kill != NULL ) &&
         ( Token_To_Kill != &Null_Token ) &&
         ( Token_To_Kill->TokenText != NULL )  &&
         ( Token_To_Kill->Characterization != KeyWord ) 
       )
      free(Token_To_Kill->TokenText);

    if ( ( Token_To_Kill == NULL ) ||
         ( Token_To_Kill == &Null_Token ) )
      *FreeMemory = FALSE;
    else
      *FreeMemory = TRUE;

    return TRUE;

  }

  /* We have found the current token.  We can stop deleting tokens. */
  *FreeMemory = FALSE;
  *Error = DLIST_SEARCH_COMPLETE;
  return FALSE;

}



static void Free_Command_List(dlist_t Command_List) /* Deletes all entries in the Command_List and frees their memory. */
{

  if ( Command_List != NULL )
  {

    /* Free all of the elements within the command list. */
    PruneList(Command_List, &Kill_Command_List_Entries, NULL);

    /* Now free the command list itself. */
    DestroyList(&Command_List, TRUE);

  }

  return;

}

static BOOLEAN Kill_Command_List_Entries (ADDRESS   Object,
                                          TAG       ObjectTag,
                                          uint      ObjectSize,
                                          ADDRESS   ObjectHandle,
                                          ADDRESS   Parameters,
                                          BOOLEAN * FreeMemory,
                                          uint    * Error)
{

  /* Establish easy access to the current node. */
  Executable_Node * Node_To_Kill = (Executable_Node *) Object;

  /* Assume success. */
  *Error = DLIST_SUCCESS;

  /* Sanity check!  Is the node of the expected type? */
  if ( ObjectTag != STACK_NODE )
  {

    /* We have an illegal node in the list!  Abort! */
    *Error =  DLIST_ITEM_TAG_WRONG;
    return FALSE;

  }

  /* Free the NodeValue (if any ) */
  /* if ( Node_To_Kill->NodeValue != NULL )
    free(Node_To_Kill->NodeValue); */

  /* If the node has children, we must kill them also. */
  if ( Node_To_Kill->Children != NULL )
  {

    /* Kill the children. */
    *Error = PruneList(Node_To_Kill->Children, Kill_Command_List_Entries, NULL);

    if ( *Error != DLIST_SUCCESS )
      return FALSE;

    *Error = DestroyList(&Node_To_Kill->Children,TRUE);
    if (*Error != DLIST_SUCCESS)
      return FALSE;

  }

  /* Now kill the node itself. */
  *FreeMemory = TRUE;

  return TRUE;

}

static Parse_Results Check_For_Colon(BOOLEAN Optional)
{

  /* Get the next token. */
  Next_Token();

  /* Check for EOF or InvalidCharacter */
  if ( ( Current_Token == NULL ) || ( Current_Token->Characterization == EofToken ) || ( Current_Token->Characterization == InvalidCharacter ) )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Colon_Separator, Command_Filename, Current_Token);

    return Error;

  }

  /* Check for the ':' separator. */
  if ( ( Current_Token->Characterization != Separator ) ||
       ( Current_Token->TokenText[0] != ':' )
     )
  {

    if (Optional)
      return Try_Again;

    Report_Parser_Error(Expecting_Colon_Separator, Command_Filename, Current_Token);

    return Error;

  }

  /* Since we found the ':', consume the token. */
  Consume_Token();

  return Token_Accepted;

}

static Parse_Results Check_For_Comma(BOOLEAN Comma_Is_Optional)
{

  /* Get the next token. */
  Next_Token();

  /* Check for EOF or InvalidCharacter */
  if ( ( Current_Token == NULL ) || ( Current_Token->Characterization == EofToken ) || ( Current_Token->Characterization == InvalidCharacter ) )
  {
    if ( Comma_Is_Optional )
      return Try_Again;

    Report_Parser_Error(Expecting_Comma_Separator, Command_Filename, Current_Token);

    return Error;

  }

  /* Check for the ',' separator. */
  if ( ( Current_Token->Characterization != Separator ) ||
       ( Current_Token->TokenText[0] != ',' )
     )
  {

    if ( Comma_Is_Optional )
      return Try_Again;

    Report_Parser_Error(Expecting_Comma_Separator, Command_Filename, Current_Token);

    return Error;

  }

  /* Since we found the ',', consume the token. */
  Consume_Token();

  return Token_Accepted;

}

static Parse_Results Check_For_EqualSign(BOOLEAN  EqualSign_Is_Optional)
{

  /* Get the next token. */
  Next_Token();

  /* Check for EOF or InvalidCharacter */
  if ( ( Current_Token == NULL ) || ( Current_Token->Characterization == EofToken ) || ( Current_Token->Characterization == InvalidCharacter ) )
  {

    if ( EqualSign_Is_Optional )
      return Try_Again;

    Report_Parser_Error(Expecting_Equal_Sign_Separator, Command_Filename, Current_Token);

    return Error;

  }

  /* Check for the '=' separator. */
  if ( ( Current_Token->Characterization != Separator ) ||
       ( Current_Token->TokenText[0] != '=' )
     )
  {

    if ( EqualSign_Is_Optional )
      return Try_Again;

    Report_Parser_Error(Expecting_Equal_Sign_Separator, Command_Filename, Current_Token);

    return Error;

  }

  /* Since we found the '=', consume the token. */
  Consume_Token();

  return Token_Accepted;

}

static Parse_Results Check_For_Open_Brace(BOOLEAN  Optional)
{

  /* Get the next token. */
  Next_Token();

  /* Check for EOF or InvalidCharacter */
  if ( ( Current_Token == NULL ) || ( Current_Token->Characterization == EofToken ) || ( Current_Token->Characterization == InvalidCharacter ) )
  {

    if ( Optional )
      return Try_Again;

    Report_Parser_Error(Expecting_Open_Brace_Separator, Command_Filename, Current_Token);

    return Error;

  }

  /* Check for the '{' separator. */
  if ( ( Current_Token->Characterization != Separator ) ||
       ( Current_Token->TokenText[0] != '{' )
     )
  {

    if ( Optional )
      return Try_Again;

    Report_Parser_Error(Expecting_Open_Brace_Separator, Command_Filename, Current_Token);

    return Error;

  }

  /* Since we found the '{', consume the token. */
  Consume_Token();

  return Token_Accepted;

}


static Parse_Results Check_For_Closing_Brace(BOOLEAN  Optional)
{

  /* Get the next token. */
  Next_Token();

  /* Check for EOF or InvalidCharacter */
  if ( ( Current_Token == NULL ) || ( Current_Token->Characterization == EofToken ) || ( Current_Token->Characterization == InvalidCharacter ) )
  {

    if ( Optional )
      return Try_Again;

    Report_Parser_Error(Expecting_Closing_Brace_Separator, Command_Filename, Current_Token);

    return Error;

  }

  /* Check for the '}' separator. */
  if ( ( Current_Token->Characterization != Separator ) ||
       ( Current_Token->TokenText[0] != '}' )
     )
  {

    if ( Optional )
      return Try_Again;

    Report_Parser_Error(Expecting_Closing_Brace_Separator, Command_Filename, Current_Token);

    return Error;

  }

  /* Since we found the '}', consume the token. */
  Consume_Token();

  return Token_Accepted;

}


static Parse_Results Check_For_Open_Paren(BOOLEAN  Optional)
{

  /* Get the next token. */
  Next_Token();

  /* Check for EOF or InvalidCharacter */
  if ( ( Current_Token == NULL ) || ( Current_Token->Characterization == EofToken ) || ( Current_Token->Characterization == InvalidCharacter ) )
  {

    if ( Optional )
      return Try_Again;

    Report_Parser_Error(Expecting_Open_Paren_Separator, Command_Filename, Current_Token);

    return Error;

  }

  /* Check for the '{' separator. */
  if ( ( Current_Token->Characterization != Separator ) ||
       ( Current_Token->TokenText[0] != '(' )
     )
  {

    if ( Optional )
      return Try_Again;

    Report_Parser_Error(Expecting_Open_Paren_Separator, Command_Filename, Current_Token);

    return Error;

  }

  /* Since we found the '(', consume the token. */
  Consume_Token();

  return Token_Accepted;

}


static Parse_Results Check_For_Closing_Paren(BOOLEAN  Optional)
{

  /* Get the next token. */
  Next_Token();

  /* Check for EOF or InvalidCharacter */
  if ( ( Current_Token == NULL ) || ( Current_Token->Characterization == EofToken ) || ( Current_Token->Characterization == InvalidCharacter ) )
  {

    if ( Optional )
      return Try_Again;

    Report_Parser_Error(Expecting_Closing_Paren_Separator, Command_Filename, Current_Token);

    return Error;

  }

  /* Check for the '}' separator. */
  if ( ( Current_Token->Characterization != Separator ) ||
       ( Current_Token->TokenText[0] != ')' )
     )
  {

    if ( Optional )
      return Try_Again;

    Report_Parser_Error(Expecting_Closing_Paren_Separator, Command_Filename, Current_Token);

    return Error;

  }

  /* Since we found the ')', consume the token. */
  Consume_Token();

  return Token_Accepted;

}


static Executable_Node * Make_Child_Node(Executable_Node * Current_Node, BOOLEAN Add_To_Current_Node)
{

  Executable_Node * Child = NULL;
  void *            Unneeded;

  /* Allocate the child node. */
  Child = (Executable_Node *) malloc( sizeof(Executable_Node) );
  if ( Child != NULL )
  {
    Child->NodeValue = NULL;
    Child->Children = CreateList();
    if ( Child->Children == NULL)
    {
      free(Child);
      Child = NULL;
    }

  }

  /* Did we succeed in creating the child node? */
  if ( Child == NULL )
  {

    Report_Parser_Error(Out_Of_Memory, NULL, NULL);

    return NULL;

  }

  if ( Add_To_Current_Node )
  {

    /* Add the child node to the Current_Node. */
    if ( InsertObject(Current_Node->Children, sizeof(Executable_Node), Child, STACK_NODE, NULL, AppendToList, TRUE, &Unneeded) != DLIST_SUCCESS )
    {
      /* We must be out of memory! */
      Report_Parser_Error(Out_Of_Memory, NULL, NULL);

      /* Free the Child node. */
      DestroyList(&Child->Children, TRUE);
      free(Child);
      Child = NULL;

    }

  }

  Child->NodeValue = NULL;
  return Child;

}
