/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: LvmUtils
 * File: evms_vgdisplay.c
 *
 *	Emulates LVM's 'vgdisplay' utility using the EVMS Engine. All options
 *	and several status messages are based on the original lvscan command
 *	from Heinz Mauelshagen and Sistina Software (www.sistina.com).
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <libgen.h>
#include <frontend.h>

typedef struct cmd_options_s {
	int debug;
	int help;
	int verbose;
	int version;
} cmd_options_t;

static char * cmd = NULL;
static cmd_options_t opts;

#include "helpers/get_container_handle.c"
#include "helpers/is_lvm_container.c"
#include "helpers/open_engine.c"
#include "helpers/print_ext_info_item.c"
#include "helpers/remove_duplicates.c"


static int showheader( void )
{
	// VERSION and DATE are defined in the top-level make.rules
	printf("Enterprise Volume Management System\n");
	printf("International Business Machines  %s\n", DATE);
	printf("LVM Emulation Utilities %s\n\n", VERSION);
	return 0;
}


static int showhelp( void )
{
	showheader();
	printf("\n");
	printf("%s - display volume group information\n\n", cmd);
	printf("%s [-c|--colon | -s|--short | -v[v]|--verbose [--verbose]]\n", cmd);
	printf("\t[-d|--debug] [-h|--help] [-V|--version]\n");
	printf("\t[-A|--activevolumegroups | [-D|--disk] [VolumeGroupName...] ]\n");
	return 0;
}


static int parse_options( int		argc,
			char		** argv )
{
	int		c;
	char		* short_opts = "AcdDh?svV";
	struct option	long_opts[] = {
				{ "activevolumegroups",	no_argument, NULL, 'A'},
				{ "colon",		no_argument, NULL, 'c'},
				{ "debug",		no_argument, NULL, 'd'},
				{ "disk",		no_argument, NULL, 'D'},
				{ "help",		no_argument, NULL, 'h'},
				{ "short",		no_argument, NULL, 's'},
				{ "verbose",		no_argument, NULL, 'v'},
				{ "version",		no_argument, NULL, 'V'},
				{ NULL, 0, NULL, 0} };

	while ( (c = getopt_long(argc, argv, short_opts,
				long_opts, NULL)) != EOF ) {
		switch (c) {
		case 'A':
			// -A is ignored by EVMS
			break;
		case 'c':
			// -c is ignored by EVMS
			break;
		case 'd':
			opts.debug++;
			opts.verbose++;
			break;
		case 'D':
			// -D is ignored by EVMS
			break;
		case 'h':
		case '?':
			opts.help++;
			break;
		case 's':
			// -s is ignored by EVMS
			break;
		case 'v':
			opts.verbose++;
			break;
		case 'V':
			opts.version++;
			break;
		default:
			printf("%s -- unrecognized option \"%c\"\n\n", cmd, c);
			return EINVAL;
		}
	}

	return 0;
}


int main( int argc, char * argv[] )
{
	object_handle_t			container_handle;
	extended_info_array_t		* container_ext_info = NULL;
	extended_info_array_t		* pv_name_info = NULL;
	extended_info_array_t		* lv_name_info = NULL;
	char				* vg_names[256] = {NULL};
	int				number_of_vgs;
	int				log_level = DEFAULT;
	int				i, j, rc;

	memset(&opts, 0, sizeof(cmd_options_t));
	cmd = basename(argv[0]);

	// Get the command line options.
	rc = parse_options(argc, argv);
	if (rc) {
		showhelp();
		return rc;
	}
	if ( opts.help ) {
		showhelp();
		return 0;
	}
	if ( opts.version ) {
		showheader();
		return 0;
	}
	if ( opts.verbose ) {
		log_level = DEBUG;
	}
	if ( opts.debug ) {
		log_level = ENTRY_EXIT;
	}

	// Check for VG names
	if ( optind == argc ) {
		printf("%s -- please enter a volume group name\n\n", cmd);
		return EINVAL;
	}
	number_of_vgs = argc - optind;

	// Copy the VG names from the command line, detecting any duplicates.
	remove_duplicates( &argv[optind], vg_names, number_of_vgs);

	// Open the engine.
	rc = open_engine(ENGINE_READONLY, log_level);
	if (rc) {
		return rc;
	}

	// Examine each volume group
	for ( i = 0; i < number_of_vgs; i++ ) {

		// Get the handle for this LV.
		rc = get_container_handle(vg_names[i], &container_handle);
		if (rc) {
			continue;
		}

		// Does this region belong to LVM?
		if ( ! is_lvm_container(container_handle) ) {
			printf("%s -- Container %s does not belong to LVM.\n", cmd, vg_names[i]);
			continue;
		}

		// Get the extended info for the VG.
		rc = evms_get_extended_info(container_handle, NULL, &container_ext_info);
		if (rc) {
			printf("%s -- Error getting extended info for container %s (%d)\n", cmd, vg_names[i], rc);
			continue;
		}

		// Display all available information
		printf("--- Volume Group ---\n");
		for ( j = 0; j < container_ext_info->count; j++ ) {
			print_ext_info_item(&(container_ext_info->info[j]));
		}
		printf("\n");
		evms_free(container_ext_info);

		if ( opts.verbose ) {
			// Get the list of LV names
			rc = evms_get_extended_info(container_handle, "Current_LVs", &lv_name_info);
			if (rc) {
				printf("--- No Logical Volume Information ---\n\n");
			}
			else {
				printf("--- Logical Volumes ---\n");
				for ( j = 0; j < lv_name_info->count; j++ ) {
					print_ext_info_item(&(lv_name_info->info[j]));
				}
				printf("\n");
				evms_free(lv_name_info);
			}

			// Get the list of PV names
			rc = evms_get_extended_info(container_handle, "Current_PVs", &pv_name_info);
			if (rc) {
				printf("--- No Physical Volume Information ---\n\n");
			}
			else {
				printf("--- Physical Volumes ---\n");
				for ( j = 0; j < pv_name_info->count; j++ ) {
					print_ext_info_item(&(pv_name_info->info[j]));
				}
				printf("\n");
				evms_free(pv_name_info);
			}
		}
	}

	evms_close_engine();
	return 0;
}

