/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libgpt.so
 *
 *   File: helpers.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "gpt.h"
#include "checks.h"
#include "helpers.h"

// list for keeping disk private data areas
static list_anchor_t  Disk_PrivateData_List=NULL;
#define DISK_PDATA_TAG  0x08080808


/*
 *  Called to allocate a GPT_DISK_PRIVATE_DATA struct and link it into the
 *  list we maintain for disks we manage disk segments on.
 */
int create_gpt_disk_private_data( LOGICALDISK *ld )
{
        int rc=ENOMEM;
        DISK_PRIVATE_DATA *disk_pdata;


        LOG_ENTRY();

        // make sure list is created
        if (Disk_PrivateData_List == NULL) {

                Disk_PrivateData_List = EngFncs->allocate_list();

                if (Disk_PrivateData_List == NULL) {
                        LOG_EXIT_INT(rc);
                        return rc;
                }

        }

        if ( get_gpt_disk_private_data(ld ) == NULL ) {

                disk_pdata = (DISK_PRIVATE_DATA *) calloc(1, sizeof(DISK_PRIVATE_DATA) );

                if ( disk_pdata ) {
                        list_element_t e;

                        disk_pdata->signature          = GPT_SEG_MGR_PDATA_SIGNATURE;
                        disk_pdata->key                = ld;
                        disk_pdata->vsectors_per_block = ld->geometry.bytes_per_sector >> EVMS_VSECTOR_SIZE_SHIFT;

                        e = EngFncs->insert_thing( Disk_PrivateData_List, disk_pdata, INSERT_AFTER, NULL );
                        if (e != NULL) {
                                rc = 0;
                        }
                        else {
                                rc = EPERM;
                                free(disk_pdata);
                        }

                }
                else {
                        rc = ENOMEM;
                }
        }
        else {
                rc = 0;
        }

        LOG_EXIT_INT(rc);
        return rc;
}

/*
 *  Called to delete the specified disk private data by pruning it from the double
 *  linked list.
 */
int delete_gpt_disk_private_data( LOGICALDISK *ld )
{
        int                rc = EINVAL;
        DISK_PRIVATE_DATA *disk_pdata = get_gpt_disk_private_data(ld );


        LOG_ENTRY();

        // make sure list exists
        if ( Disk_PrivateData_List == NULL ) {
                LOG_EXIT_INT(rc);
                return rc;
        }

        // delete object from list and free its memory
        if (disk_pdata) {

                EngFncs->remove_thing( Disk_PrivateData_List, disk_pdata );

                free(disk_pdata);

                rc = 0;
        }
        else {
                rc = EINVAL;
        }

        LOG_EXIT_INT(rc);
        return rc;
}

/*
 *  Returns the private data area for the specified disk.
 */
DISK_PRIVATE_DATA * get_gpt_disk_private_data( LOGICALDISK *ld )
{
        DISK_PRIVATE_DATA  *disk_pdata = NULL;
        list_element_t   iter;


        if ( Disk_PrivateData_List != NULL ) {

                LIST_FOR_EACH( Disk_PrivateData_List, iter, disk_pdata ) {

                        // compare storage object ptr to key field in private data area
                        if ( disk_pdata->key == ld ) {
                                return disk_pdata;
                        }
                }

        }

        return NULL;
}


/*
 *  Called by cleanup code to toss disk private data.
 */
void  delete_all_gpt_disk_private_data( void )
{
        list_element_t iter, iter2;
        DISK_PRIVATE_DATA *disk_pdata;

        if ( Disk_PrivateData_List != NULL ) {

                LIST_FOR_EACH_SAFE(Disk_PrivateData_List,iter,iter2, disk_pdata ) {
                        free(disk_pdata);
                        EngFncs->delete_element(iter);
                }

                EngFncs->destroy_list( Disk_PrivateData_List );
                Disk_PrivateData_List = NULL;

        }

}


/*
 *  A freespace has a name that has 3 parts to it.
 *  (1) a parent object name like hda
 *  (2) the word freespace
 *  (3) a number used to keep freespace seg names different
 *
 *  Examples:  hda_freespace1
 *             hda_freespace2
 *             dasdh3_freespace4
 *
 *  This routine returns the number found at the end
 *  of the freespace segment name, ignoring any differences
 *  that may be found in other parts of the name.
 *
 *  Returns: success: integer > 0
 *           failure: -1
 *
 */
static int get_freespace_number(DISKSEG *freespace)
{
        int number=-1;
        int i;

        if (freespace) {

                if (freespace->name) {

                        // pretty liberal here...
                        // minimum string has length of 2 ... freespac[eN]
                        if (strlen(freespace->name)>=2) {

                                // walk backwards
                                for (i=strlen(freespace->name)-1; i>0; i--) {

                                        // first e we find ends the search
                                        if ( freespace->name[i-1] == 'e' ) {

                                                // convert ascii number to integer
                                                return(  atoi( &freespace->name[i] ) );

                                        }

                                }

                        }

                }

        }

        return number;
}

/*
 *  Called by the get_name_for_disk_segment() routine to come up
 *  with the next available number to use when naming freespace
 *  segments on this drive.
 */
static int  get_next_avail_freespace_number(LOGICALDISK *ld)
{
        int freespace_number=0;
        DISKSEG *seg;
        list_element_t iter;

        LOG_ENTRY();

        LIST_FOR_EACH( ld->parent_objects, iter, seg ) {

                if (seg->data_type == FREE_SPACE_TYPE) {
                        if ( get_freespace_number(seg) > freespace_number ) {
                                freespace_number = get_freespace_number(seg);
                        }
                }

        }

        ++freespace_number;

        LOG_EXIT_INT(freespace_number);
        return freespace_number;
}


/*
 *  Called to get the segment storage object corresponding to the specified
 *  device minor number.  This information is kept in the private data area
 *  of gpt segment storage objects.
 *
 *  Returns: DISKSEG * if successful
 *           NULL ptr if minor is not found
 */
DISKSEG * get_gpt_segment_from_minor( LOGICALDISK *ld, int minor )
{
        DISKSEG    *seg=NULL;
        list_element_t iter;

        LOG_ENTRY();

        LIST_FOR_EACH( ld->parent_objects, iter, seg ) {
                if (seg->data_type == DATA_TYPE) {
                        if ( ((SEG_PRIVATE_DATA *)seg->private_data)->minor == minor ) {
                                LOG_EXIT_PTR(seg);
                                return seg;
                        }
                }
        }

        LOG_EXIT_PTR(NULL);
        return NULL;
}


/*
 *  Called to obtain the next available device minor
 *  number for the specified drive. This is an UNUSED
 *  minor.
 *
 *  Note: this routine should not be called during discovery
 *        because it expects all segments to be created for
 *        the drive and not in the middle of adding existing
 *        drive partitions.
 *
 *  Returns: device minor number if successful
 *           -1 if not successful
 */
int get_next_gpt_minor( LOGICALDISK *ld )
{
        int                 i;
        DISKSEG            *md=NULL;
        gpt_header         *gh=NULL;
        DISK_PRIVATE_DATA  *disk_pdata=NULL;

        LOG_ENTRY();

        disk_pdata = get_gpt_disk_private_data(ld);
        if (disk_pdata) {

                md = disk_pdata->md1;
                if (md) {

                        gh = ((SEG_PRIVATE_DATA *)md->private_data)->gh;
                        if (gh) {

                                // for each minor number ... if there are no disk segments
                                // using the minor ... then return it as the next available.
                                for (i=1; i<=gh->ptable_count; i++) {
                                        if ( get_gpt_segment_from_minor( ld, i)==NULL ) {
                                                return(i);
                                        }
                                }

                        }

                }

        }


        LOG_EXIT_INT(-1);
        return -1;
}


/*
 *  Called by insert_DiskSeg_Into_List() to get a name
 *  for the segment storage object being created.
 */
DISKSEG * allocate_gpt_freespace_disk_segment( LOGICALDISK *ld )
{
        DISKSEG  *freeseg=NULL;
        char devname[EVMS_VOLUME_NAME_SIZE+1];

        if (ld) {

                freeseg = allocate_gpt_disk_segment(ld);

                if (freeseg) {

                        get_device_name(ld, devname);

                        if ( isa_disk_object(ld) == FALSE ) {
                                sprintf(freeseg->name, "%s.freespace%d", devname, get_next_avail_freespace_number(ld));
                        }
                        else {
                                if (devname[strlen(devname)-1] == '/') {
                                        sprintf(freeseg->name, "%sfreespace%d", devname, get_next_avail_freespace_number(ld) );
                                }
                                else {
                                        sprintf(freeseg->name, "%s_freespace%d", devname, get_next_avail_freespace_number(ld) );
                                }
                        }

                }

        }

        return freeseg;
}


/*
 *   Returns the DISKSEG struct ptr to the freespace seg following the
 *   specified DISKSEG.  Returns NULL if freespace is not found.
 */
DISKSEG * get_freespace_following_gpt_disk_segment(DISKSEG *seg)
{
        DISKSEG     *prev = NULL;
        DISKSEG     *this = NULL;
        LOGICALDISK  *ld=NULL;
        list_element_t  iter;

        // get logical disk and the entire segment list for the disk
        ld = get_logical_disk( seg );
        if ( ld ) {

                prev = EngFncs->first_thing( ld->parent_objects, &iter );
                if (prev==NULL) return NULL;

                do {

                        this = EngFncs->next_thing( &iter );
                        if (this != NULL) {

                                if (prev == seg) {

                                        if ( this->data_type==FREE_SPACE_TYPE ) {
                                                return this;
                                        }
                                        else {
                                                return NULL;
                                        }

                                }

                                prev=this;
                        }

                } while (this != NULL);

                return NULL;
        }
        else {
                return NULL;
        }

}



/*
 *   Called to try and join neighboring free segments together.
 *
 *   NOTE:  Assumes that the DISKSEG structs are kept in an ordered list
 *
 */
static int  merge_freespace_segments( list_anchor_t seglist )
{
        DISKSEG            *prev;
        DISKSEG            *this;
        LOGICALDISK        *ld;
        list_element_t     iter;

        LOG_ENTRY();

        prev = EngFncs->first_thing( seglist, &iter );
        if ( prev != NULL ) {

                // need logical disk
                ld = get_logical_disk(prev);
                if (ld) {

                        do {
                                this = EngFncs->next_thing( &iter );
                                if (this != NULL) {

                                        if ( ( prev==NULL ) ||
                                             ( this->data_type != FREE_SPACE_TYPE )||
                                             ( prev->data_type != FREE_SPACE_TYPE )) {
                                                prev = this;
                                                continue;
                                        }

                                        if (get_freespace_number(prev) > get_freespace_number(this) ) {
                                                EngFncs->remove_thing( seglist, prev );
                                                this->start -= prev->size;
                                                this->size  += prev->size;
                                                free_gpt_disk_segment( prev );
                                                LOG_DEBUG("        kept seg: %s  start: %"PRIu64"  size: %"PRIu64"\n", this->name, this->start, this->size );
                                                return 0;
                                        }
                                        else {
                                                EngFncs->remove_thing( seglist, this );
                                                prev->size  += this->size;
                                                free_gpt_disk_segment( this );
                                                LOG_DEBUG("        kept seg: %s  start: %"PRIu64"  size: %"PRIu64"\n", prev->name, prev->start, prev->size );
                                                return 0;
                                        }

                                }

                        } while (this != NULL);
                }
        }


        LOG_EXIT_INT(ENODATA);
        return ENODATA;
}


/*
 *   Called to try and join neighboring free segments together.
 *
 *   NOTE:  Assumes that the DISKSEG structs are kept in an ordered list
 *
 */
int  merge_adjacent_freedisksegs_in_gpt_seglist( list_anchor_t seglist )
{
        int rc=0;

        LOG_ENTRY();

        do {
                rc = merge_freespace_segments(seglist);
        } while (rc==0);


        rc = 0;

        LOG_EXIT_INT(rc);
        return rc;
}





/*
 *   Called to try and find free space between disk data segments, returning a
 *   new free space DISKSEG struct if a gap between existing segments is found.
 */

DISKSEG * find_freespace_in_seglist( list_anchor_t seglist )
{
        DISKSEG            *prev = NULL;
        DISKSEG            *this = NULL;
        DISKSEG            *freeseg  = NULL;
        LOGICALDISK        *ld;
        int64_t             gap;
        list_element_t      iter;

        LOG_ENTRY();

        if (seglist) {

                prev = EngFncs->first_thing( seglist, &iter );
                if ( prev != NULL ) {

                        // need logical disk
                        ld = get_logical_disk( prev );
                        if (ld==NULL) return NULL;

                        do {
                                this = EngFncs->next_thing( &iter );
                                if (this != NULL) {

                                        gap = this->start - ( prev->start + prev->size );

                                        if ( gap > 0 ) {

                                                // allocate a segment storage object from the engine
                                                freeseg = allocate_gpt_freespace_disk_segment(ld);

                                                if (freeseg ) {

                                                        freeseg->data_type    = FREE_SPACE_TYPE;
                                                        freeseg->size         = gap;
                                                        freeseg->start        = prev->start + prev->size;
                                                        freeseg->flags       &= ~SOFLAG_DIRTY;

                                                        LOG_EXIT_PTR(freeseg);
                                                        return freeseg;

                                                }
                                                else {
                                                        LOG_EXIT_PTR(NULL);
                                                        return NULL;
                                                }

                                        }
                                        else {
                                                prev = this;
                                        }
                                }

                        } while ( this != NULL );
                }

        }


        LOG_EXIT_PTR(NULL);
        return NULL;
}




/*
 *  Called to look for free space on the logical disk and to create
 *  segments which will expose the free space on the disk.
 */
int  find_freespace_on_gpt_disk( LOGICALDISK *ld )
{
        DISKSEG            *freeseg              = NULL;
        DISKSEG            *seg                  = NULL;
        DISKSEG            *segaddr              = NULL;
        sector_count_t      sectors_left_on_disk = 0;
        int                 rc;
        lba_t               freespace_start_lba;
        list_anchor_t       seglist = ld->parent_objects;


        LOG_ENTRY();

        if ( EngFncs->list_count( seglist ) > 0 ) {                /* IF ... we have disk segments */

                do {                                               /* THEN ... look for gaps between them */
                        seg = find_freespace_in_seglist( seglist );

                        if (seg != NULL) {

                                segaddr = insert_gpt_segment_into_list( seglist, seg);

                                if (segaddr==NULL) {

                                        int i;

                                        for (i=0; i<10 && segaddr==NULL; i++) {
                                                segaddr = insert_gpt_segment_into_list( seglist, seg);
                                        }

                                        if (segaddr==NULL) {

                                                free_gpt_disk_segment( seg );
                                                rc = ENOMEM;
                                                LOG_EXIT_INT(rc);
                                                return rc;
                                        }

                                }

                        }

                } while (seg != NULL);

                seg = EngFncs->last_thing( seglist, NULL );

                sectors_left_on_disk = (ld->size - 1) - ( seg->start + seg->size -1 );

                if ( sectors_left_on_disk > 0 ) {
                        freespace_start_lba  =  seg->start + seg->size;
                }
                else {
                        sectors_left_on_disk = 0;
                        freespace_start_lba  = 0;
                }
        }
        else {  // there are no segments on the disk at all so just create a
                // single free space segment for the disk!
                freespace_start_lba  = 0;
                sectors_left_on_disk = ld->size;
        }

        if ( sectors_left_on_disk > 0 ) {

                freeseg = allocate_gpt_freespace_disk_segment(ld);
                if (freeseg) {

                        freeseg->data_type  = FREE_SPACE_TYPE;
                        freeseg->size       = sectors_left_on_disk;
                        freeseg->start      = freespace_start_lba;
                        freeseg->flags     &= ~SOFLAG_DIRTY;


                        segaddr = insert_gpt_segment_into_list( seglist, freeseg);
                        if (segaddr==NULL) {

                                int i;

                                for (i=1; i<10; i++) {

                                        segaddr = insert_gpt_segment_into_list( seglist, freeseg);
                                        if (segaddr==NULL) {
                                                LOG_DEBUG("error, insert_DiskSeg_Into_List returned an error\n");
                                                free_gpt_disk_segment(freeseg);
                                                LOG_EXIT_INT(EIO);
                                                return EIO;
                                        }
                                        else {
                                                break;
                                        }
                                }

                        }

                }
                else {
                        LOG_EXIT_INT(EIO);
                        return EIO;
                }
        }

        // lastly, merge adjacent free areas
        merge_adjacent_freedisksegs_in_gpt_seglist( ld->parent_objects );

        LOG_EXIT_INT(0);
        return 0;
}




/*
 *  Called to allocate a segment storage object
 */
storage_object_t *  allocate_gpt_disk_segment( storage_object_t *object )
{
        int   rc;
        storage_object_t  *seg=NULL;
        SEG_PRIVATE_DATA  *pdata=NULL;


        LOG_ENTRY();

        rc = EngFncs->allocate_segment( NULL, &seg );
        if (rc==0) {

                // insert logical disk into new segment's child list
                if (EngFncs->insert_thing(seg->child_objects,object,INSERT_AFTER,NULL)) {
                        rc = 0;
                }
                else {
                        rc = EPERM;
                }

                if (rc == 0) {

                        seg->plugin      = gpt_plugin_record_ptr;
                        seg->object_type = SEGMENT;

                        memcpy(&seg->geometry, &object->geometry, sizeof(geometry_t) );

                        seg->private_data = calloc(1, sizeof(SEG_PRIVATE_DATA) );
                        if (seg->private_data) {

                                pdata = (SEG_PRIVATE_DATA *)seg->private_data;

                                pdata->signature    = GPT_SEG_MGR_PDATA_SIGNATURE ;
                                pdata->logical_disk = object;


                        }
                        else {
                                LOG_ERROR("call to malloc segment private storage area failed\n");
                                EngFncs->free_segment( seg );
                                seg = NULL;
                        }
                }
                else {
                        LOG_ERROR("call to insert DISK storage object in segment child_objects list failed, RC= %d\n", rc );
                        EngFncs->free_segment( seg );
                        seg = NULL;
                }

        }
        else {
                LOG_ERROR("call to engine_allocate_segment failed, RC= %d\n", rc);
                seg = NULL;
        }

        LOG_EXIT_PTR(seg);
        return seg;
}


/*
 * Called to free a segment storage object
 */
void free_gpt_disk_segment( storage_object_t *seg )
{
        LOG_ENTRY();
        LOG_DEBUG("segment name= %s\n", seg->name );

        if (seg->private_data) free(seg->private_data);

        EngFncs->free_segment( seg );

        LOG_EXIT_VOID();
}



/*
 *  Called to add a segment to the logical disk's segment list. First, convert the
 *  partition record to a DISKSEG struct.  Then, insert the DISKSEG struct into
 *  the list hanging off the logical disk.
 */
storage_object_t * create_gpt_metadata_segment( storage_object_t   *object,
                                                lba_t               start,
                                                sector_count_t      size,
                                                char               *name )
{
        storage_object_t      *metadata=NULL;
        char devname[EVMS_VOLUME_NAME_SIZE+1];

        LOG_ENTRY();

        metadata = allocate_gpt_disk_segment( object );
        if (metadata) {

                metadata->size        = size;
                metadata->start       = start;
                metadata->data_type   = META_DATA_TYPE;


                get_device_name(object, devname);

                if ( isa_disk_object(object) == FALSE ) {
                        sprintf(metadata->name, "%s.%s", devname, name);
                }
                else {
                        if (name[strlen(devname)-1] == '/') {
                                sprintf(metadata->name, "%s%s", devname, name );
                        }
                        else {
                                sprintf(metadata->name, "%s_%s", devname, name );
                        }
                }

        }

        LOG_EXIT_PTR(metadata);
        return metadata;
}



/*
 *  Called to build an evms segment storage object from a GPT partition
 *  record.
 */
storage_object_t * build_gpt_segment_from_partition_record(  storage_object_t *object,
                                                             gpt_partition    *part,
                                                             int               minor )
{
        storage_object_t      *seg = NULL;
        SEG_PRIVATE_DATA  *pdata=NULL;
        char devname[EVMS_VOLUME_NAME_SIZE+1];

        LOG_ENTRY();

        // allocate a new disk segment storage object
        seg = allocate_gpt_disk_segment( object );
        if (seg==NULL) {
                LOG_EXIT_PTR(NULL);
                return NULL;
        }

        // get device name
        get_device_name(object, devname);

        // get gpt seg private data area
        pdata = (SEG_PRIVATE_DATA *)seg->private_data;

        // record what kind of partition it is
        if ( isa_legacy_mbr_gpt_partition_record( part ) == TRUE ) {
                pdata->type = legacy_mbr_partition;
        }
        else if ( isa_esp_gpt_partition_record( part ) == TRUE ) {
                pdata->type = efi_system_partition;
        }
        else if ( isa_basic_data_gpt_partition_record( part ) == TRUE ) {
                pdata->type = basic_data_partition;
        }
        else if ( isa_gpt_swap_partition_record( part ) == TRUE ) {
                pdata->type = swap_partition;
        }
        else if ( isa_gpt_ms_reserved_partition_record( part ) == TRUE ) {
                pdata->type = ms_reserved_partition;
        }
        else if ( isa_gpt_ms_ldm_data_partition_record( part ) == TRUE ) {
                pdata->type = ms_reserved_partition;
        }
        else if ( isa_gpt_ms_ldm_metadata_partition_record( part ) == TRUE ) {
                pdata->type = ms_reserved_partition;
        }

        // build segment name
        if ( pdata->type == legacy_mbr_partition ) {

                if ( isa_disk_object(object) == FALSE ) {
                        sprintf(seg->name, "%s.legacy_mbr_partition", devname );
                }
                else {
                        if (devname[strlen(devname)-1] == '/') {
                                sprintf(seg->name, "%slegacy_mbr_partition", devname);
                        }
                        else {
                                sprintf(seg->name, "%s_legacy_mbr_partition", devname);
                        }
                }

                seg->data_type = DATA_TYPE;

        }
        else {

                if ( isa_disk_object(object) == FALSE ) {
			char last_char = devname[strlen(devname)-1];

			if ((last_char >= '1') && (last_char <= '9')) {
				// Must be an embedded GPT partition.
				strcat(devname, ".");
			}
                }
                else {
                        if (devname[strlen(devname)-1] == '/') {
                                strcat(devname, "part");
                        }
                }
		sprintf(seg->name, "%s%d", devname, minor);

                seg->data_type = DATA_TYPE;
                pdata->minor   = (u_int32_t) minor;
        }


        // seg info
        seg->start = DISK_TO_CPU64(part->start);
        seg->size  = DISK_TO_CPU64(part->end) - DISK_TO_CPU64(part->start) + 1;

        // seg private data info
        memcpy(&pdata->guid_type, &part->type, sizeof(guid_t));
        disk_guid_to_cpu( &pdata->guid_type );

        memcpy(&pdata->guid_id,   &part->id, sizeof(guid_t));
        disk_guid_to_cpu( &pdata->guid_id );

        memcpy(&pdata->name, &part->name, sizeof(utf16_char_t)*GPT_PNAME_SIZE );

        pdata->attributes   = DISK_TO_CPU64(part->attributes);


        // look for system partition record (ESP) and mark segment as non divideable
        if ( pdata->type == efi_system_partition ) {
                pdata->cflags |= SEG_CFLAG_TOP_SEGMENT;
        }


        LOG_EXIT_PTR(seg);
        return seg;
}


/*
 *  Wrapper routine that is called to remove a segment storage object from
 *  the parent object list, owned by a logical disk.
 */
int  remove_gpt_segment_from_list( list_anchor_t seglist, storage_object_t *seg )
{
        LOG_ENTRY();
        LOG_DEBUG("segment name= %s\n",  seg->name );

        EngFncs->remove_thing( seglist, seg );

        EngFncs->unregister_name( seg->name );

        // DATA segments need their uuid identifiers unregistered
        // ... convert the uuid to an ascii string and unregister it.
        if (seg->data_type == DATA_TYPE) {

                char *uuid_string;

                uuid_string = guid_to_string( &((SEG_PRIVATE_DATA *)seg->private_data)->guid_id );
                if (uuid_string) {
                        EngFncs->unregister_name( uuid_string );
                        free(uuid_string);
                }

        }

        LOG_EXIT_INT(0);
        return 0;
}


/*
 *   Called to name a new disk segment and insert it into the
 *   ordered segment list for the logical disk.
 *
 *   Returns the list handle if successful.
 */
void *  insert_gpt_segment_into_list( list_anchor_t seglist, storage_object_t *seg)
{
        int    rc=EINVAL;
        void  *result=NULL;


        LOG_ENTRY();
        LOG_DEBUG("seg start= %"PRIu64"   size= %"PRIu64"   name= %s\n", seg->start, seg->size, seg->name);

        rc = EngFncs->register_name( seg->name );
        if (rc == 0) {

                // DATA segments need their uuid identifiers registered
                // ... convert the uuid to an ascii string and register it.
                if (seg->data_type == DATA_TYPE) {

                        char *uuid_string;

                        uuid_string = guid_to_string( &((SEG_PRIVATE_DATA *)seg->private_data)->guid_id );
                        if (uuid_string) {
                                rc = EngFncs->register_name( uuid_string );
                                free(uuid_string);
                        }

                }

                if (rc == 0) {
                        result = insert_gpt_segment_into_ordered_list( seglist, seg );
                }

        }
        else {
                LOG_ERROR("error, registering name for disk segment failed, RC= %d\n", rc);
                LOG_EXIT_PTR(NULL);
                return NULL;
        }


        LOG_DEBUG("returning %p\n", result);
        LOG_EXIT_PTR(result);
        return result;
}


/*
 *   Called to insert a disk segment into a segment list that we
 *   are trying to keep ordered by the segments starting LBA
 *   number.
 *
 *   Returns the list handle if successful.
 */
void *  insert_gpt_segment_into_ordered_list( list_anchor_t seglist, storage_object_t *seg )
{
        int                rc;
        storage_object_t  *seg2;
        lba_t              seg2_end_lba;
        boolean            overlapping=FALSE;
        list_element_t     iter,e;


        LOG_ENTRY();
        LOG_DEBUG("seg name= %s   seg start= %"PRIu64"  ends= %"PRIu64"  size= %"PRIu64"\n", seg->name, seg->start, seg->start+seg->size-1, seg->size );

        rc = -1;

        LIST_FOR_EACH( seglist, iter, seg2 ) {

                seg2_end_lba = seg2->start + seg2->size - 1;

                // test and set ... overlapping segments flag
                if ( (  seg->start >= seg2->start )&&
                     (  seg->start <= seg2_end_lba)) {
                        overlapping = TRUE;
                }
                else if ( ( seg->start  <  seg2->start ) &&
                          ( seg2->start <= (seg->start + seg->size - 1)) ) {
                        overlapping = TRUE;
                }
                else {
                        overlapping = FALSE;
                }

                if ( overlapping == TRUE ) {

                        LOG_DEBUG("Error ... Overlapping Segments ...\n");
                        LOG_DEBUG("seg2:   name: %s\n", seg2->name );
                        LOG_DEBUG("       start: %"PRIu64"\n", seg2->start );
                        LOG_DEBUG("        size: %"PRIu64"\n", seg2->size );
                        LOG_DEBUG("         end: %"PRIu64"\n", seg2_end_lba );
                        LOG_DEBUG(" overlap lba: %"PRIu64"\n", seg->start );

                        rc = EINVAL;    // must be genuine partition overlap
                        break;          // break out of loop ... looking for insertion pt

                }

                // test for ... valid insertion point
                if (seg2->start > seg->start ) {
                        rc = 0;
                        break;
                }
        }


        switch (rc) {

        case 0:  /* Ok, found a segment we should insert in front of */
                e = EngFncs->insert_thing( seglist,
                                           seg,
                                           INSERT_BEFORE|EXCLUSIVE_INSERT,
                                           EngFncs->find_in_list(seglist, seg2, NULL, NULL) );
                if (e != NULL) {
                        rc = 0;
                }
                else {
                        rc = EPERM;
                }
                break;


        case -1: /* This new segment lays out on the disk at a higher */
                /* LBA than any other existing segment.  So, just    */
                /* insert at end of the segment list.                */
                e = EngFncs->insert_thing( seglist,
                                           seg,
                                           INSERT_AFTER|EXCLUSIVE_INSERT,
                                           NULL );
                if (e != NULL) {
                        rc = 0;
                }
                else {
                        rc = EPERM;
                }
                break;

        default:     /* REAL ERROR ... return NULL ptr */
                LOG_ERROR("error, insertion failed ... RC= %d\n",  rc);
                break;
        }

        if (rc) {
                LOG_EXIT_PTR(NULL);
                return NULL;
        }
        else {
                LOG_EXIT_PTR(seg);
                return seg;
        }
}


/*
 *  Called because the disk doesn't have an MBR yet.  So, we create one, hang
 *  it off the disk, and fill in a DLAT sector.
 */
int  create_protective_mbr_segment( LOGICALDISK *ld )
{
        storage_object_t  *pmbr;
        int                rc=EINVAL;

        LOG_ENTRY();

        /* hang an PMBR meta data segment off the logical drive */
        pmbr = allocate_gpt_disk_segment( ld );
        if (pmbr) {

                pmbr->size        = 1;
                pmbr->start       = 0;
                pmbr->data_type   = META_DATA_TYPE;

                if ( isa_disk_object(ld) == TRUE ) {
                        sprintf( pmbr->name, "%s_pmbr", ld->name );
                }
                else {
                        sprintf( pmbr->name, "%s.pmbr", ld->name );
                }

                if ( insert_gpt_segment_into_list( ld->parent_objects, pmbr) != NULL ) {
                        rc = 0;
                }
                else {
                        free_gpt_disk_segment(pmbr);
                        LOG_ERROR("error, cant create PMBR, call to insert PMBR storage object into disk list failed\n");
                        rc = ENODEV;
                }

        }
        else {
                LOG_ERROR("error, failed to allocate a segment object\n");
                rc = ENOMEM;
        }


        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called to convert a Globally Unique Identifier to
 *  char string format.
 *
 *  Returns: char ptr is successful
 *           null ptr is unusccessful
 */
char * guid_to_string( guid_t *id )
{
        char *guid_string=NULL;

        if (id) {

                guid_string = (char *) malloc(37);
                if (guid_string) {

                        sprintf( guid_string, "%08X-%04X-%04X-%02X%02X-%02X%02X%02X%02X%02X%02X",
                                 id->time_low,
                                 id->time_mid,
                                 id->time_high,
                                 id->clock_seq_high,
                                 id->clock_seq_low,
                                 id->node[0],
                                 id->node[1],
                                 id->node[2],
                                 id->node[3],
                                 id->node[4],
                                 id->node[5] );

                }

        }

        return guid_string;
}
