/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: move.c
 */

#include <frontend.h>
#include <gtk/gtk.h>

#include "support.h"
#include "thing.h"
#include "move.h"
#include "readable.h"
#include "logging.h"
#include "help.h"

/*
 *
 *   void on_move_thing_button_clicked (GtkButton *, object_handle_t, gchar *, gchar *, gboolean)
 *
 *   Description:
 *      This routine initiates the move API call using the handle of the
 *      the given source object and the target object associated with the
 *      last row selected.
 *
 *   Entry:
 *      button        - address of the GtkButton widget
 *      source        - the object handle of the source thing to move
 *      error_msg     - string to use for additional error message on failure
 *      success_msg   - string to use as additional message on success
 *
 *   Exit:
 *      evms_replace() is invoked and results window is displayed
 *
 */
void on_move_thing_button_clicked(GtkButton * button, object_handle_t source, gchar * error_msg,
				  gchar * success_msg)
{
	GtkCList *clist;
	object_handle_t target;

	clist = GTK_CLIST(lookup_widget(GTK_WIDGET(button), "selection_window_clist"));

	target = GPOINTER_TO_UINT(get_single_select_current_row_data(clist));

	if (target != 0) {
		gint rc;

		rc = evms_replace(source, target);

		display_selection_window_results(GTK_WIDGET(button), rc, error_msg, success_msg);
	} else {
		log_error("%s: The handle of the target object for the replace was zero!\n",
			  __FUNCTION__);
	}
}

/**
 *	can_replace_object - can a given object be replaced with some other object
 *	@handle: the object that we want to replace
 *
 *	This routine answers the question "Can the given object be replaced with some
 *	candidate object?". Since a call to evms_can_replace(handle, 0) simply answers
 *	the question "Can the given object be replaced?" is insufficient since it does
 *	not look whether there are likely candidates for the replace. Therefore, we
 *	make the additional checks ourselves.
 */
int can_replace_object(object_handle_t handle)
{
	int rc;

	rc = evms_can_replace(handle, 0);
	if (rc == 0) {
		handle_array_t *objects;

		rc = evms_get_object_list(0, DATA_TYPE, 0, 0, TOPMOST, &objects);
		if (rc == 0) {
			rc = ENOENT;

			if (objects->count > 0) {
				int i = 0;

				while (i < objects->count && rc != 0) {
					rc = evms_can_replace(handle, objects->handle[i]);
					i++;
				}
			}
			evms_free(objects);
		}
	}
	return rc;
}

/*
 *
 *   void on_move_thing_source_clist_realize (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of things that can be used as source objects for a move operation.
 *
 *   Entry:
 *      widget    - address of the selections GtkCList widget
 *      user_data - contains type of source thing to move (feature, segment, etc.)
 *
 *   Exit:
 *      Selection list populated with acceptable things
 *
 */
void on_move_thing_source_clist_realize(GtkWidget * widget, gpointer user_data)
{
	gint rc = 0;
	object_type_t type = GPOINTER_TO_UINT(user_data);
	GtkCList *clist = GTK_CLIST(widget);
	handle_array_t *things;

	switch (type) {
	case 0:
	case EVMS_OBJECT:
	case REGION:
	case SEGMENT:
	case DISK:
		rc = evms_get_object_list(type, DATA_TYPE, 0, 0, 0, &things);
		if (rc != SUCCESS)
			log_error("%s: evms_get_object_list() returned error code %d.\n",
				  __FUNCTION__, rc);
		break;

	default:
		log_error("%s: Unsupported selection type %d.\n", __FUNCTION__, type);
		rc = EINVAL;
		break;
	}

	if (rc == SUCCESS) {
		guint i;
		gboolean is_selected = (things->count == 1);

		set_selection_window_clist_column_titles(clist, _("Size"), _("Source"), NULL);

		for (i = 0; i < things->count; i++) {
			if (can_replace_object(things->handle[i]) == 0)
				add_thing_to_selection_list(clist, things->handle[i], is_selected);
		}

		if (clist->rows == 1)
			gtk_clist_select_row(clist, 0, 0);

		evms_free(things);
	}
}

/*
 *
 *   void on_move_thing_target_clist_realize (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of things that can be used as target objects for a move operation.
 *
 *   Entry:
 *      widget - address of the selections GtkCList widget
 *      user_data - contains handle of source thing to move
 *
 *   Exit:
 *      Selection list populated with acceptable things
 *
 */
void on_move_thing_target_clist_realize(GtkWidget * widget, gpointer user_data)
{
	gint rc;
	object_handle_t source = GPOINTER_TO_UINT(user_data);
	GtkCList *clist = GTK_CLIST(widget);
	handle_array_t *things;

	rc = evms_get_object_list(0, DATA_TYPE, 0, 0, TOPMOST, &things);

	if (rc != SUCCESS) {
		log_error("%s: evms_get_object_list() returned error code %d.\n", __FUNCTION__, rc);
	} else {
		guint i;
		gboolean is_selected = (things->count == 1);

		set_selection_window_clist_column_titles(clist, _("Size"), _("Target"), NULL);

		for (i = 0; i < things->count; i++) {
			if (evms_can_replace(source, things->handle[i]) == 0)
				add_thing_to_selection_list(clist, things->handle[i], is_selected);
		}

		if (clist->rows == 1)
			gtk_clist_select_row(clist, 0, 0);

		evms_free(things);
	}
}

/*
 *
 *   void on_move_thing_generic_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine handles initiating the move of a "thing".
 *      The handle to the thing is received as the user data. We
 *      select the proper messages for the results screen and invoke
 *      the routine that actually moves the data located on the source
 *      to the target object.
 *
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - object handle
 *
 *   Exit:
 *      See description.
 *
 */
void on_move_thing_generic_button_clicked(GtkButton * button, gpointer user_data)
{
	object_type_t type;
	object_handle_t source;

	source = GPOINTER_TO_UINT(user_data);

	if (evms_get_handle_object_type(source, &type) == SUCCESS) {
		gchar *error_msg;
		gchar *success_msg;

		switch (type) {
		case EVMS_OBJECT:
			error_msg =
			    _("An error was encountered attempting to replace the feature object.");
			success_msg = _("The feature object is scheduled to be replaced.");
			break;

		case REGION:
			error_msg =
			    _("An error was encountered attempting to replace the storage region.");
			success_msg = _("The storage region is scheduled to be replaced.");
			break;

		case SEGMENT:
			error_msg =
			    _("An error was encountered attempting to replace the disk segment.");
			success_msg = _("The disk segment is scheduled to be replaced.");
			break;

		case DISK:
			error_msg =
			    _("An error was encountered attempting to replace the logical disk.");
			success_msg = _("The the logical disk is scheduled to be replaced.");
			break;

		default:
			error_msg = _("The replace operation encountered an error.");
			success_msg = _("The the storage object is scheduled to be replaced.");
			break;
		}

		on_move_thing_button_clicked(button, source, error_msg, success_msg);
	}
}

/*
 *
 *   inline GtkWidget *create_move_target_selection_window (object_handle_t, gboolean)
 *
 *   Description:
 *      This routine will create the selection window of target objects available
 *      to move the source object's data to.
 *
 *   Entry:
 *      source        - handle of source object
 *      single_window - TRUE if don't need to link previous button handler
 *                      to a previous selection window
 *
 *   Exit:
 *      Returns id of selection window for target objects of move
 *
 */
inline GtkWidget *create_move_target_selection_window(object_handle_t source,
						      gboolean single_window)
{
	gchar *name;
	gchar *window_title;
	GtkWidget *window;

	name = get_object_name(source);

	if (name)
		window_title = g_strdup_printf(_("Replace  %s with Target Storage Object"), name);
	else
		window_title = g_strdup(_("Select Target Storage Object for Replace"));

	window = create_standard_selection_window(window_title,
						  _("Replace"),
						  NULL,
						  on_move_thing_target_clist_realize,
						  on_move_thing_generic_button_clicked,
						  single_window ? NULL :
						  on_button_clicked_display_prev_window,
						  on_button_clicked_destroy_window_list, NULL, NULL,
						  GUINT_TO_POINTER(source));
	g_free(name);
	g_free(window_title);

	return window;
}

/*
 *
 *   void on_move_thing_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine will display the target objects available
 *      to move the source object's data to.
 *
 *   Entry:
 *      menuitem  - the menuitem that initiated the action
 *      user_data - source object handle
 *
 *   Exit:
 *      See description.
 *
 */
void on_move_thing_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_move_target_selection_window(GPOINTER_TO_UINT(user_data), TRUE);

	gtk_widget_show(window);
}

/*
 *
 *   void on_move_thing_source_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine handles initiating the move of a "thing".
 *      The handle to the source thing is gotten from the selection
 *      list. We then display the list of possible target objects
 *      for the move operation.
 *
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - not used
 *
 *   Exit:
 *      See description.
 *
 */
void on_move_thing_source_button_clicked(GtkButton * button, gpointer user_data)
{
	GList *window_list;
	GtkWidget *clist;
	GtkWidget *next_window;
	GtkWidget *selection_window;
	object_handle_t source;
	object_handle_t prev_source;

	selection_window = gtk_widget_get_toplevel(GTK_WIDGET(button));
	clist = lookup_widget(GTK_WIDGET(button), "selection_window_clist");
	source = GPOINTER_TO_UINT(get_single_select_current_row_data(GTK_CLIST(clist)));
	window_list = get_window_list(selection_window);
	next_window = gtk_object_get_data(GTK_OBJECT(selection_window), "next_window_id");
	prev_source =
	    GPOINTER_TO_UINT(gtk_object_get_data(GTK_OBJECT(selection_window), "previous_source"));

	if (window_list == NULL)
		window_list = g_list_append(window_list, selection_window);

	if (source != prev_source) {
		if (next_window != NULL)
			destroy_window_list(g_list_find(window_list, next_window));

		next_window = create_move_target_selection_window(source, FALSE);

		window_list = g_list_append(window_list, next_window);

		set_window_list(next_window, window_list);
		set_window_list(selection_window, window_list);

		gtk_object_set_data(GTK_OBJECT(selection_window), "next_window_id", next_window);
		gtk_object_set_data(GTK_OBJECT(selection_window), "previous_source",
				    GUINT_TO_POINTER(source));
	}

	gtk_widget_show(next_window);
	gtk_widget_hide(selection_window);
}

/*
 *
 *   GtkWidget *create_move_source_selection_window (object_type_t)
 *
 *   Description:
 *      This routine will display the source objects available
 *      to move the data on it.
 *
 *   Entry:
 *      type - type of source thing to move (feature, segment, etc.)
 *
 *   Exit:
 *      See description.
 *
 */
GtkWidget *create_move_source_selection_window(object_type_t type)
{
	gchar *title;
	GtkWidget *window;

	switch (type) {
	case EVMS_OBJECT:
		title = _("Replace Feature Object");
		break;

	case REGION:
		title = _("Replace Storage Region");
		break;

	case SEGMENT:
		title = _("Replace Disk Segment");
		break;

	case DISK:
		title = _("Replace Logical Disk");
		break;

	default:
		title = _("Replace Storage Object");
		break;
	}

	window = create_standard_selection_window(title,
						  NULL,
						  NULL,
						  on_move_thing_source_clist_realize,
						  on_move_thing_source_button_clicked,
						  NULL,
						  on_button_clicked_destroy_window_list,
						  NULL, NULL, GUINT_TO_POINTER(type));

	return window;
}
