/*
 * ece.c EVMS Plugin for heartbeart2
 *
 * (C) Copyright IBM Corp. 2002, 2003
 * Copyright (C) Novell, Inc. 2005, 2006
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/utsname.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <stdarg.h>
#include <hb_api.h>
#include <heartbeat.h>

#include <string.h>
#include <pthread.h>

#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/un.h>

#undef	LOG_WARNING
#undef LOG_DEBUG

#include <ece.h>
#include <plugin.h>

#define MAX_NODES		(32)
#define NODE_NUM_MASK		(0xf)

#define	IPC_MSG_EVMS_MSG_TYPE	(1)
#define	IPC_MSG_MEMBERSHIP_TYPE	(2)

#define ECE_BROADCAST_MASK	(0x40000000)
#define ECE_DAEMON_MASK		(0x20000000)

#ifndef HA_VARRUNDIR
	#define HA_VARRUNDIR	"/var/run"
#endif

#define ADDRESS     		VAR_RUN_D"/heartbeat/evmsd"  /* addr to connect */

/* macro-like declarations */
static ece_nodeid_t             ECE_BROADCASE_ADDR = ECE_ALL_NODES;

static char                     EVMS_GROUP_NAME[] = "evms";
static char                     EVMS_CORROLATOR[] = "corrolator";
static char                     EVMS_CMD[] = "evms_cmd";
static char                     EVMS_DATA[] = "evms_data";
static char                     EVMS_FROM[] = "evms_from";

/* types */
typedef struct Node_Info_S
{
	cl_uuid_t               nodeID;
	char                    nodeName[SYS_NMLN];
} Node_Info_T;

typedef struct Cluster_Info_S
{
	u_int32_t               membership;
	int                     clusterNodeNum;
} Cluster_Info_T;

typedef struct ece_ha_ipc_msg_hdr_s
{
	u_int32_t       msgType;
	u_int32_t       corrolator;
	u_int32_t       cmd;
	size_t          size;
	u_int32_t       fromTo;
	char            data[0];
} ece_ha_ipc_msg_hdr_t;

/* variables required by ECE */
extern  plugin_record_t         HB2_plugin_record;
#define my_plugin_record 	(&HB2_plugin_record)

/* global needed by evms engine functions */
static engine_functions_t      *EngFncs=NULL;

static engine_mode_t		EngineMode = 0;

/* heartbeat handle */
static ll_cluster_t*            pEceHbHandle=NULL;

/* locks */
static pthread_mutex_t          EngineIPCMutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t          EceCallbackMutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t          EceHaApiMutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t          ClusterInfoMutex =PTHREAD_MUTEX_INITIALIZER;

/* callback */
static ece_cb_t                 EceCallbackFunction = NULL;

static pthread_t                PluginMainThread = 0;
static pthread_t                MsgHandlerThread = 0;

/* cluster dynamic information */
static Cluster_Info_T           ClusterInfo = {0, 0};

/* cluster static info */
static Node_Info_T              ClusterNodes[MAX_NODES];
static unsigned char            MyNodeNumber = 0xff;

/* IPC channels */
static int                      PluginSocket = -1;
static int                      PluginIPCFd = -1;

static u_int32_t                MessageIndex = 0;

static void notify_membership_change( u_int32_t new_membership, u_int32_t old_membership )
{
	LOG_ENTRY();

	/* only if they are different */
	if ( new_membership != old_membership )
	{
		u_int32_t	left = ( old_membership & (~( old_membership & new_membership)));
		u_int32_t	joined = ( new_membership & (~( old_membership & new_membership)));

		LOG_DEBUG("mode=%d: ECE Membership: new_membership=%08x\n", EngineMode, new_membership );
		LOG_DEBUG("mode=%d: ECE Membership: membership=%08x\n", EngineMode, old_membership );
		LOG_DEBUG("mode=%d: ECE Membership: joined=%08x\n", EngineMode, joined );
		LOG_DEBUG("mode=%d: ECE Membership: left=%08x\n", EngineMode, left );

		/* get the lock */
		if ( 0 == pthread_mutex_lock( &EceCallbackMutex) )
		{
        		if ( EceCallbackFunction )
			{
				int             i;
				ece_event_t     eceevent;

				/* one at a time */
				eceevent.num_entries = 1;

				/* once here, we always have quorum with regard to membership changes */
				eceevent.quorum_flag = TRUE;

				for ( i = 0; i < MAX_NODES; i++ )
				{
					if ( left & (1 << i ) )
					{
						eceevent.type = DELTA_LEAVE;

						memset(eceevent.node, 0, sizeof(ece_nodeid_t));
						eceevent.node[0].bytes[0] = (unsigned char)i;
						eceevent.node[0].bytes[1] = 0x86;

						EceCallbackFunction( CALLBACK_MEMBERSHIP, sizeof(ece_event_t), &eceevent);
					}
					else if ( joined & (1 << i ) )
					{
						eceevent.type = DELTA_JOIN;

						memset(eceevent.node, 0, sizeof(ece_nodeid_t));
						eceevent.node[0].bytes[0] = (unsigned char)i;
						eceevent.node[0].bytes[1] = 0x86;

						EceCallbackFunction( CALLBACK_MEMBERSHIP, sizeof(ece_event_t), &eceevent);
					}
				}

			}
			pthread_mutex_unlock( &EceCallbackMutex );

			LOG_DEBUG( "Callback done.\n" );
		}

		/* also notify Engine through the IPC channel */
		if ( ENGINE_DAEMON == EngineMode )
		{
			ece_ha_ipc_msg_hdr_t        ipcmsg;

			/* construct the event msg */
			ipcmsg.msgType = IPC_MSG_MEMBERSHIP_TYPE;
			ipcmsg.corrolator = old_membership;

			/* get the lock */
			if ( 0 == pthread_mutex_lock( &EngineIPCMutex ) )
			{
				int     len = sizeof( ece_ha_ipc_msg_hdr_t );

				/* valid connection? */
				if ( 0 <= PluginIPCFd )
				{
					/* write the fixed header */
					write( PluginIPCFd, &len, sizeof( int ));

					/* write the msg */
					write( PluginIPCFd, &ipcmsg, sizeof( ece_ha_ipc_msg_hdr_t ));

					LOG_DEBUG( "IPC done.\n" );
				}

				/* release the lock */
				pthread_mutex_unlock( &EngineIPCMutex );
			}
		}
	}

	LOG_EXIT_VOID();
	return;
}

static void get_config_and_membership()
{
	Cluster_Info_T  update = { 0, 0 };

	LOG_ENTRY();

	/* retrieve configuration information from heartbeat */
	if ( HA_OK == pEceHbHandle->llc_ops->init_nodewalk(pEceHbHandle) )
	{
		const char *    node;

		/* get all the nodes here */
		while ( (node = pEceHbHandle->llc_ops->nextnode(pEceHbHandle))!= NULL )
		{
			const char*     status = pEceHbHandle->llc_ops->node_status(pEceHbHandle, node);

			LOG_DEBUG( "Cluster node: %s: status: %s.\n", node, status );

			/* is it active? */
			if ( 0 == strcmp( ACTIVESTATUS, status ) )
			{
				update.membership |= (1 << update.clusterNodeNum );
			}

			/* who am i? */
			if ( 0xff == MyNodeNumber )
			{
				if ( 0 == strcmp( pEceHbHandle->llc_ops->get_mynodeid( pEceHbHandle ), node ) )
				{
					MyNodeNumber = update.clusterNodeNum;
				}
			}

			if ( update.clusterNodeNum >= ClusterInfo.clusterNodeNum )
			{
				cl_uuid_t       uuid;

				/* get the ID */
				if ( HA_OK == pEceHbHandle->llc_ops->get_uuid_by_name(pEceHbHandle, node, &uuid ) )
				{
					char    str_id[38];

					cl_uuid_unparse(&uuid, str_id);
					LOG_DEBUG( "New node %s/uuid : [%s]\n", node, str_id );

					/* record node information */
					cl_uuid_copy( &ClusterNodes[update.clusterNodeNum].nodeID, &uuid );
					strncpy( ClusterNodes[update.clusterNodeNum].nodeName, node, SYS_NMLN );
					ClusterNodes[update.clusterNodeNum].nodeName[SYS_NMLN-1] = 0;
				}

			}

			update.clusterNodeNum++;
		}

		if ( pEceHbHandle->llc_ops->end_nodewalk(pEceHbHandle) != HA_OK )
		{
			LOG_WARNING( "Cannot end ifwalk. REASON: %s\n", pEceHbHandle->llc_ops->errmsg(pEceHbHandle));
		}

		LOG_DEBUG("New membership=%08x(%d), old membership=%08x(%d), and MyNodeNumber=%d\n",
			  update.membership, update.clusterNodeNum,
			  ClusterInfo.membership, ClusterInfo.clusterNodeNum,
			  MyNodeNumber );

		/* start with the current info */
		ClusterInfo = update;
	}

	LOG_EXIT_VOID();
	return;
}


/* node status callback */
static void node_status_callback(const char * node, const char * status, void * private)
{

	LOG_ENTRY();
	LOG_DEBUG( "Status update (%08x): Node %s now has status %s.\n", ClusterInfo.membership, node, status);

	if ( 0 == pthread_mutex_lock(&ClusterInfoMutex) )
	{
		int             i;
		u_int32_t       oldMembership = ClusterInfo.membership;

		/* get current membership */
		get_config_and_membership();

		/* just in case the previous call failed to get full membership */
		for ( i = 0; i < ClusterInfo.clusterNodeNum; i++ )
		{
			/* which node? */
			if ( 0 == strcmp( ClusterNodes[i].nodeName, node ) )
			{
				/* active? */
				if ( 0 == strcmp( ACTIVESTATUS, status ) )
				{
					/* mark the bit */
					ClusterInfo.membership |= (1 << i );
				}
				else
				{
					/* clean the bit */
					ClusterInfo.membership &= (~(1 << i ));
				}

				break;
			}
		}

		LOG_DEBUG("New membership=%08x, old membership=%08x.\n", ClusterInfo.membership, oldMembership );

		/* any changes?  */
		if ( oldMembership != ClusterInfo.membership )
		{
			/* notify others if the changes */
			notify_membership_change( ClusterInfo.membership, oldMembership );
		}

		pthread_mutex_unlock(&ClusterInfoMutex);
	}

	LOG_EXIT_VOID();
	return;
}

static void* HB2_msg_handler( void* args )
{
	struct  ha_msg* msg = NULL;

	LOG_ENTRY();

	while ( TRUE )
	{
		const char *    type;

		/* delete the message before read */
		if ( msg )
		{
			/* grab the lock just in case */
			if ( 0 == pthread_mutex_lock( &EceHaApiMutex) )
			{
				LOG_DEBUG( "deleting msg.....\n" );
				ha_msg_del( msg );
				LOG_DEBUG( "msg deleted.\n" );

				/* release the lock */
				pthread_mutex_unlock( &EceHaApiMutex);
			}

			msg = NULL;
		}

		pthread_testcancel();

		if ( 0 == pthread_mutex_lock( &EceHaApiMutex) )
		{
			msg = pEceHbHandle->llc_ops->readmsg( pEceHbHandle, FALSE );
			pthread_mutex_unlock( &EceHaApiMutex);
		}

		/* no message? */
		if ( NULL == msg )
		{
			pthread_testcancel();

			/* delay for a while */
			usleep( 200000 );

			pthread_testcancel();

			/* start over */
			continue;
		}

		/* get the message type */
		if ( NULL == (type = ha_msg_value(msg, F_TYPE)) )
		{
			type = "?";
		}

		LOG_DEBUG( "Start processing a message (%p) of type <%s>......\n", msg, type );

		/* meant for daemon/or Engine */
		if ( 0 == strcmp(type, EVMS_GROUP_NAME ) )
		{
			const char *    orig;
			int             fromTo = 0;

			/* get the origin */
			if ( NULL == (orig = ha_msg_value(msg, F_ORIG)) )
			{
				orig = "?";
			}

			/* get EVMS address field in the payload */
			ha_msg_value_int( msg, EVMS_FROM, &fromTo );
			LOG_DEBUG( "Got message (seq:%s) of type [%s] from [%s/%08x].\n", cl_get_string(msg, F_SEQ), type, orig, fromTo );

			/* from daemon or Engine? */
			if ( ECE_DAEMON_MASK & fromTo )
			{
				/* deliver to Engine */
				int                     rc, len;
				const char*             data;
				ece_ha_ipc_msg_hdr_t    ipcmsg;

				/* construct the ece message */
				if ( HA_OK != ha_msg_value_int( msg, EVMS_CORROLATOR, (int*)&ipcmsg.corrolator ) ||
				     HA_OK != ha_msg_value_int( msg, EVMS_CMD, (int*)&ipcmsg.cmd ) ) continue;

				/* must be a message */
				ipcmsg.msgType = IPC_MSG_EVMS_MSG_TYPE;

				/* copy this field */
				ipcmsg.fromTo = fromTo;

				/* get the data if any */
				data = cl_get_binary( msg, EVMS_DATA, &ipcmsg.size );
				if ( NULL == data ) ipcmsg.size = 0;

				len = sizeof( ece_ha_ipc_msg_hdr_t )  + ipcmsg.size;

				/* get the lock */
				if ( 0 == pthread_mutex_lock( &EngineIPCMutex ) )
				{
					/* write the header */
					if ( 0 <= PluginIPCFd )
					{
						rc = write( PluginIPCFd, &len, sizeof( int ));
						if ( sizeof( int ) != rc )
						{
							LOG_ERROR( "mode=%d: failed to write header. rc = %d.\n", EngineMode, rc );

							/* release the lock */
							pthread_mutex_unlock( &EngineIPCMutex );

							continue;
						}

						LOG_DEBUG( "mode=%d: header with length of %d written.\n", EngineMode, len );

						/* write the to-fields */
						rc = write( PluginIPCFd, &ipcmsg, sizeof( ece_ha_ipc_msg_hdr_t ));

						if ( (sizeof( ece_ha_ipc_msg_hdr_t )) != rc )
						{
							LOG_ERROR( "mode=%d: failed to write ipcmsg. rc = %d.\n", EngineMode, rc );

							/* release the lock */
							pthread_mutex_unlock( &EngineIPCMutex );

							continue;
						}

						LOG_DEBUG( "mode=%d: wrote ipcmsg with length of %d.\n", EngineMode, sizeof( ece_ha_ipc_msg_hdr_t ) );

						if ( ipcmsg.size > 0 )
						{
							/* write the to-fields */
							rc = write( PluginIPCFd, data, ipcmsg.size );

							if ( ipcmsg.size != rc )
							{
								LOG_ERROR( "mode=%d: failed to write msg. rc = %d.\n", EngineMode, rc );

								/* release the lock */
								pthread_mutex_unlock( &EngineIPCMutex );

								continue;
							}

							LOG_DEBUG( "mode=%d: wrote ece msg [%d] with length of %d.\n", EngineMode, data[0], ipcmsg.size );
						}
					}

					/* release the lock */
					pthread_mutex_unlock( &EngineIPCMutex );

					LOG_DEBUG( "mode=%d: Daemon->Engine message corrolator=%08x\n", EngineMode, ipcmsg.corrolator );
					LOG_DEBUG( "mode=%d: Daemon->Engine message cmd=%08x\n", EngineMode, ipcmsg.cmd );
					LOG_DEBUG( "mode=%d: Daemon->Engine message size=%d\n", EngineMode, ipcmsg.size );
					LOG_DEBUG( "mode=%d: Daemon->Engine message fromTo=%08x\n", EngineMode, ipcmsg.fromTo );
				}
			}
			else
			{
				/* deliver to daemon */
				ece_msg_t       ecemsg;

				/* construct the ece message */
				if ( HA_OK != ha_msg_value_int( msg, EVMS_CORROLATOR, (int*)&ecemsg.corrolator ) ||
				     HA_OK != ha_msg_value_int( msg, EVMS_CMD, (int*)&ecemsg.cmd ) ) continue;

				/* get the data if any */
				ecemsg.msg = (void*) cl_get_binary( msg, EVMS_DATA, &ecemsg.size );
				if ( NULL == ecemsg.msg ) ecemsg.size = 0;

				/* set the node ID */
				memset(&ecemsg.node, 0, sizeof(ece_nodeid_t));
				ecemsg.node.bytes[0] = (unsigned char) (fromTo & NODE_NUM_MASK );
				ecemsg.node.bytes[1] = 0x86;

				LOG_DEBUG("mode=%d: Engine->Daemon message: from=%s.\n", EngineMode, ClusterNodes[ecemsg.node.bytes[0]].nodeName );
				LOG_DEBUG("mode=%d: Engine->Daemon message: cmd=%08x\n", EngineMode, ecemsg.cmd );
				LOG_DEBUG("mode=%d: Engine->Daemon message: size=%d\n", EngineMode, ecemsg.size );
				LOG_DEBUG("mode=%d: Engine->Daemon message: corrolator=%08x\n", EngineMode, ecemsg.corrolator );

				/* callback function is valid? */
				if ( EceCallbackFunction && 0 == pthread_mutex_lock( &EceCallbackMutex ) )
				{
					if ( EceCallbackFunction )
					{
						/* delivery to callback and eventually to Engine */
						EceCallbackFunction( CALLBACK_MESSAGE, ecemsg.size+sizeof(ece_msg_t)-sizeof(void*), &ecemsg);
					}

					pthread_mutex_unlock( &EceCallbackMutex );
				}
			}
		}
	}

	return NULL;
}

static void* HB2_main_daemon( void* args )
{
	int                  len, rc;
	char*      buffer = NULL;

	/* main loop */
	while ( TRUE )
	{
		socklen_t			fromlen=sizeof(struct sockaddr_un);
		int                     initNS;
		struct sockaddr_un      fsaun;

		/* accept any connections */
		pthread_testcancel();
		initNS = accept(PluginSocket, (struct sockaddr *)&fsaun, &fromlen);
		pthread_testcancel();

		LOG_DEBUG( "Accepted one connect %d.\n", initNS );

		/* valid connection? */
		if ( 0 <= initNS )
		{
			if ( 0 == pthread_mutex_lock( &ClusterInfoMutex ) )
			{
				int		i;

				/* send current cluster information
				 * before others know about this fd
				 */
				write( initNS, &ClusterInfo.clusterNodeNum, sizeof(int));
				write( initNS, &MyNodeNumber, sizeof(unsigned char));
				write( initNS, &ClusterInfo.membership, sizeof(u_int32_t));

				/* send the initialization information */
				for ( i = 0; i < ClusterInfo.clusterNodeNum; i++ )
				{
					write( initNS, &ClusterNodes[i], sizeof(Node_Info_T));
				}

				pthread_mutex_unlock( &ClusterInfoMutex );
			}

			if ( 0 == pthread_mutex_lock( &EngineIPCMutex ) )
			{
				/* record the fd # */
				PluginIPCFd = initNS;
				pthread_mutex_unlock( &EngineIPCMutex );
			}

			LOG_DEBUG( "Configuration written.\n" );

			/* loop to serve one instance of Engine */
			while ( TRUE )
			{
				LOG_DEBUG( "Preparing to read a header.....\n" );
				/* read header */
				rc =  recv( PluginIPCFd, (char*)&len, sizeof( int ), MSG_WAITALL );
				LOG_DEBUG( "header=%d.\n", rc );

				/* broken? */
				if ( 0 >= rc ) break;

				/* get the header */
				if ( sizeof( int ) ==  rc )
				{
					/* allocate the buffer */
					LOG_DEBUG( "Preparing to engine_alloc (%d).....\n", len );
					if ( 0 == pthread_mutex_lock( &EceHaApiMutex) )
					{
						buffer = EngFncs->engine_alloc( len );
						pthread_mutex_unlock( &EceHaApiMutex);
						LOG_DEBUG( "engine_alloc = %08x.\n", (int)buffer );
					}

					/* got the buffer */
					if ( buffer )
					{
						LOG_DEBUG( "Preparing to read data (%d).....\n", len );
						rc = recv( PluginIPCFd, buffer, len, MSG_WAITALL );
						LOG_DEBUG( "read data = %d.\n", rc );

						/* read data */
						if ( rc == len )
						{
							struct ha_msg*  hamsg = NULL;

							/* construct the message */
							if ( 0 == pthread_mutex_lock( &EceHaApiMutex) )
							{
								hamsg = ha_msg_new(0);
								pthread_mutex_unlock( &EceHaApiMutex);
							}

							if ( hamsg )
							{
								ece_ha_ipc_msg_hdr_t*       ipcmsg = (ece_ha_ipc_msg_hdr_t *) buffer;

								if( HA_OK == ha_msg_add(hamsg, F_TYPE, EVMS_GROUP_NAME ) &&
									HA_OK == ha_msg_add_int( hamsg, EVMS_CORROLATOR, ipcmsg->corrolator ) &&
									HA_OK == ha_msg_add_int( hamsg, EVMS_CMD, ipcmsg->cmd ) &&
									HA_OK == ha_msg_addbin( hamsg, EVMS_DATA, &buffer[sizeof(ece_ha_ipc_msg_hdr_t)], ipcmsg->size ) &&
									HA_OK == ha_msg_add_int( hamsg, EVMS_FROM, MyNodeNumber ))
								{
									/* delivery to heartbeat and eventually to daemon */
									if ( ECE_BROADCAST_MASK & ipcmsg->fromTo )
									{
										if ( 0 == pthread_mutex_lock( &EceHaApiMutex) )
										{
											/* send to everybody */
											rc = pEceHbHandle->llc_ops->sendclustermsg(pEceHbHandle, hamsg);
											pthread_mutex_unlock( &EceHaApiMutex);
										}
									}
									else
									{
										if ( (ipcmsg->fromTo & NODE_NUM_MASK ) == MyNodeNumber )
										{
											/* deliver to daemon */
											ece_msg_t       ecemsg;

											/* construct the ece message */
											ecemsg.corrolator = ipcmsg->corrolator;
											ecemsg.cmd = ipcmsg->cmd;
											ecemsg.size = ipcmsg->size;

											/* set the node ID */
											memset(&ecemsg.node, 0, sizeof(ece_nodeid_t));
											ecemsg.node.bytes[0] = MyNodeNumber;
											ecemsg.node.bytes[1] = 0x86;

											/* callback function is valid? */
											if ( EceCallbackFunction && 0 == pthread_mutex_lock( &EceCallbackMutex ) )
											{
												if ( EceCallbackFunction )
												{
													/* delivery to callback and eventually to Engine */
													EceCallbackFunction( CALLBACK_MESSAGE, ecemsg.size+sizeof(ece_msg_t)-sizeof(void*), &ecemsg);
												}
												pthread_mutex_unlock( &EceCallbackMutex );
											}
										}
										else
										{
											if ( 0 == pthread_mutex_lock( &EceHaApiMutex) )
											{
												/* deliver to one node */
												rc = pEceHbHandle->llc_ops->sendnodemsg_byuuid(pEceHbHandle, hamsg, &ClusterNodes[(ipcmsg->fromTo & NODE_NUM_MASK)].nodeID );
												pthread_mutex_unlock( &EceHaApiMutex);
											}
										}
									}
								}
								else
								{
									LOG_ERROR( "mode=%d: Engine->Daemon message too long!\n", EngineMode );
								}

								LOG_DEBUG( "mode=%d: Engine->Daemon message corrolator=%08x\n", EngineMode, ipcmsg->corrolator );
								LOG_DEBUG( "mode=%d: Engine->Daemon message cmd=%08x\n", EngineMode, ipcmsg->cmd );
								LOG_DEBUG( "mode=%d: Engine->Daemon message size=%d\n", EngineMode, ipcmsg->size );
								LOG_DEBUG( "mode=%d: Engine->Daemon message fromTo=%08x(hamsg %d)\n", EngineMode, ipcmsg->fromTo, MyNodeNumber );
								LOG_DEBUG( "mode=%d: Engine->Daemon message rc=%d\n", EngineMode, rc );

								if ( 0 == pthread_mutex_lock( &EceHaApiMutex) )
								{
									/* delete the message */
									ha_msg_del( hamsg );
									pthread_mutex_unlock( &EceHaApiMutex);
								}
							}
						}

						if ( 0 == pthread_mutex_lock( &EceHaApiMutex) )
						{
							/* free the buffer */
							EngFncs->engine_free( buffer );

							pthread_mutex_unlock( &EceHaApiMutex);
						}
					}
				}
			}

			if ( 0 == pthread_mutex_lock( &EngineIPCMutex ) )
			{
				close( PluginIPCFd );
				PluginIPCFd = -1;
				pthread_mutex_unlock( &EngineIPCMutex );
			}
		}
	}
}

static void* HB2_main_engine( void* args )
{
	int                  len, rc;
	char*      buffer = NULL;

	while ( TRUE )
	{
		/* read header */
		pthread_testcancel();
		rc =  recv( PluginSocket, (char*)&len, sizeof( int ), MSG_WAITALL);
		pthread_testcancel();

		if ( 0 >= rc ) break;

		if ( sizeof( int ) == rc )
		{
			/* one final check, don't allow pthread_cancel
			 * once we got hold of the buffer
			 */
			pthread_testcancel();

			/* allocate the buffer */
			buffer = EngFncs->engine_alloc( len );

			/* got the buffer */
			if ( buffer )
			{
				/* read data */
				if ( len == recv( PluginSocket, buffer, len, MSG_WAITALL ) )
				{
					ece_ha_ipc_msg_hdr_t*       ipcmsg = (ece_ha_ipc_msg_hdr_t *) buffer;

					if ( IPC_MSG_EVMS_MSG_TYPE == ipcmsg->msgType )
					{
						ece_msg_t       ece_msg;

						/* construct the ece message */
						ece_msg.corrolator =  ipcmsg->corrolator;
						ece_msg.cmd =  ipcmsg->cmd;
						ece_msg.size =  ipcmsg->size;
						ece_msg.msg =  &buffer[sizeof(ece_ha_ipc_msg_hdr_t)];

						/* set the node ID */
						memset(&ece_msg.node, 0, sizeof(ece_nodeid_t));
						ece_msg.node.bytes[0] = (unsigned char) (ipcmsg->fromTo & NODE_NUM_MASK );
						ece_msg.node.bytes[1] = 0x86;

						LOG_DEBUG("mode=%d: Daemon->Engine message: from=%s.\n", EngineMode, ClusterNodes[ece_msg.node.bytes[0]].nodeName );
						LOG_DEBUG("mode=%d: Daemon->Engine message: cmd=%08x\n", EngineMode, ece_msg.cmd );
						LOG_DEBUG("mode=%d: Daemon->Engine message: size=%d(len=%d)\n", EngineMode, ece_msg.size, len );
						LOG_DEBUG("mode=%d: Daemon->Engine message: corrolator=%08x\n", EngineMode, ece_msg.corrolator );
						LOG_DEBUG("mode=%d: Daemon->Engine message: ece_msg.msg=%08x\n", EngineMode, (int)ece_msg.msg );

						/* callback function is valid? */
						if ( EceCallbackFunction && 0 == pthread_mutex_lock( &EceCallbackMutex ) )
						{
							/* delivery to callback and eventually to Engine */
							if ( EceCallbackFunction )
							{
								EceCallbackFunction( CALLBACK_MESSAGE, len, &ece_msg);
							}

							pthread_mutex_unlock( &EceCallbackMutex );
						}
					}
					else
						if ( IPC_MSG_MEMBERSHIP_TYPE == ipcmsg->msgType )
					{
						u_int32_t       newMembership = ipcmsg->corrolator;

						if ( ClusterInfo.membership != newMembership )
						{
							if ( 0 == pthread_mutex_lock( &ClusterInfoMutex ) )
							{
								notify_membership_change( newMembership, ClusterInfo.membership );

								/* record the membership */
								ClusterInfo.membership = newMembership;
								pthread_mutex_unlock( &ClusterInfoMutex );
							}
						}
					}
				}

				/* free the buffer */
				EngFncs->engine_free( buffer );
			}
		}
	}

	return NULL;
}

static void HB2_cleanup(void)
{
	if ( MsgHandlerThread )
	{
		LOG_DEBUG( "Cancelling message handling thread.\n" );

		/* cancel the thread */
		pthread_cancel( MsgHandlerThread );

		/* wait until it returns */
		pthread_join( MsgHandlerThread, NULL );

		MsgHandlerThread = 0;
	}

	if ( PluginMainThread )
	{
		LOG_DEBUG( "Cancelling main thread.\n" );

		/* cancel the thread */
		pthread_cancel( PluginMainThread );

		/* wait until it returns */
		pthread_join( PluginMainThread, NULL );

		PluginMainThread = 0;
	}

	/* destroy */
	if ( NULL != pEceHbHandle )
	{
		LOG_DEBUG( "Signing off.\n" );

		if ( 0 == pthread_mutex_lock(&EceHaApiMutex ) )
		{
			/* sign off */
			if ( HA_OK != pEceHbHandle->llc_ops->signoff(pEceHbHandle, TRUE) )
			{
				LOG_ERROR( "Cannot sign off from heartbeat.");
				LOG_ERROR( "REASON: %s\n", pEceHbHandle->llc_ops->errmsg(pEceHbHandle));
			}

			LOG_DEBUG( "deleting hb.\n" );

			if ( HA_OK != pEceHbHandle->llc_ops->delete(pEceHbHandle) )
			{
				LOG_ERROR( "Cannot delete API object.");
				LOG_ERROR( "REASON: %s\n", pEceHbHandle->llc_ops->errmsg(pEceHbHandle));
			}
			pthread_mutex_unlock(&EceHaApiMutex );
		}

		pEceHbHandle = NULL;
	}

	LOG_DEBUG( "closing socket.\n" );

	/* close IPC */
	if ( 0 <= PluginIPCFd )
	{
		close( PluginIPCFd );

		PluginIPCFd = -1;
	}

	/* close socket */
	if ( 0 <= PluginSocket )
	{
		close( PluginSocket );

		PluginSocket = -1;
	}

	/* destroy mutex */
	pthread_mutex_destroy( &EngineIPCMutex );
	pthread_mutex_destroy( &EceCallbackMutex );
	pthread_mutex_destroy( &EceHaApiMutex );
	pthread_mutex_destroy( &ClusterInfoMutex );

	LOG_EXIT_VOID();
	return;
}

#define CLEANUP_AND_RETURN( rc ) { \
		HB2_cleanup(); \
		LOG_EXIT_INT(rc); \
		return(rc); \
		}

static int HB2_init(engine_functions_t *func)
{
	int     rc = 0;
	struct  sockaddr_un saun;

	/* have to do this first */
	EngFncs = func;

	LOG_ENTRY();

	/* initialize globals */
	pEceHbHandle=NULL;
	EceCallbackFunction = NULL;
	PluginMainThread = 0;
	MsgHandlerThread = 0;
	ClusterInfo.membership = 0;
	ClusterInfo.clusterNodeNum = 0;
	MyNodeNumber = 0xff;
	PluginSocket = -1;
	PluginIPCFd = -1;

	pthread_mutex_init(&EngineIPCMutex, NULL);
	pthread_mutex_init(&EceCallbackMutex, NULL);
	pthread_mutex_init(&EceHaApiMutex, NULL);
	pthread_mutex_init(&ClusterInfoMutex, NULL);

	/* sanity check */
	if ( !EngFncs || !EngFncs->get_engine_mode )
	{
		CLEANUP_AND_RETURN( EINVAL );
	}

	/* record the mode */
	EngineMode = EngFncs->get_engine_mode();

	/* mode? */
	if ( ENGINE_DAEMON == EngineMode )
	{
		int     retry = 0;

		/* subscribe to heartbeat */
		/* 1.1 create the new hb service */
		if ( NULL == (pEceHbHandle = ll_cluster_new("heartbeat")) )
		{
			LOG_SERIOUS("Failed to create new heartbeat instance!\n" );

			CLEANUP_AND_RETURN( ENOMEM );
		}

		LOG_DEBUG("New heartbeat instance created.\n" );

		/* 1.2 sign on to the heartbeat daemon */
		while ( retry++ < 24 )
		{
			rc = pEceHbHandle->llc_ops->signon(pEceHbHandle, EVMS_GROUP_NAME );
			if ( HA_OK == rc ) break;

			sleep( 5 );
		}

		if ( HA_OK != rc )
		{
			LOG_SERIOUS( "Cannot sign on with heartbeat. REASON: %s\n", pEceHbHandle->llc_ops->errmsg(pEceHbHandle));
			CLEANUP_AND_RETURN( EACCES );
		}

		LOG_DEBUG("Signed on with heartbeat, retry=%d.\n", retry );

		get_config_and_membership();

		/* initialization OK? */
		/* did we get my node num? */
		if ( 0xff == MyNodeNumber )
		{
			LOG_SERIOUS( "Cannot retrieve cluster configuration");
			CLEANUP_AND_RETURN( ENOENT );
		}

		/* 1.3 set up the callbacks */
		if ( pEceHbHandle->llc_ops->set_nstatus_callback(pEceHbHandle, node_status_callback, NULL) !=HA_OK )
		{
			LOG_SERIOUS( "Cannot set node status callback. REASON: %s\n", pEceHbHandle->llc_ops->errmsg(pEceHbHandle));
			CLEANUP_AND_RETURN( ECANCELED );
		}

		LOG_DEBUG("Status callback set up.\n" );

		/*
		 * Get a socket to work with.  This socket will
		 * be in the UNIX domain, and will be a
		 * stream socket.
		 */
		if ( (PluginSocket = socket(AF_UNIX, SOCK_STREAM, 0)) < 0 )
		{
			LOG_SERIOUS( "Cannot create socket.\n");
			CLEANUP_AND_RETURN( PluginSocket );
		}

		/*
		 * Create the address we will be binding to.
		 */
		saun.sun_family = AF_UNIX;
		strcpy(saun.sun_path, ADDRESS);

		/*
		 * Try to bind the address to the socket.  We
		 * unlink the name first so that the bind won't
		 * fail.
		 *
		 * The third argument indicates the "length" of
		 * the structure, not just the length of the
		 * socket name.
		 */
		rc = sizeof(saun.sun_family) + strlen(saun.sun_path);

		/* cleanup any leftovers */
		unlink(ADDRESS);

		if ( 0 > ( rc = bind(PluginSocket, (struct sockaddr *)&saun, rc )) )
		{
			LOG_SERIOUS( "Cannot bind socket.\n");
			CLEANUP_AND_RETURN( rc );
		}

		/*
		 * Listen on the socket.
		 */
		if ( 0 > ( rc = listen(PluginSocket, 5)) )
		{
			LOG_SERIOUS( "Cannot listen to socket.\n");
			CLEANUP_AND_RETURN( rc );
		}

		LOG_DEBUG( "Started to listen...\n");

		/* create thread to process messages IPC */
		rc = pthread_create(&PluginMainThread, NULL, HB2_main_daemon, NULL );
		if ( 0 != rc )
		{
			LOG_SERIOUS("Failed to spawn a thread, try again\n");
			CLEANUP_AND_RETURN( EAGAIN );
		}

		rc = pthread_create(&MsgHandlerThread, NULL, HB2_msg_handler, NULL );
		if ( 0 != rc )
		{
			LOG_SERIOUS("Failed to spawn a thread, try again\n");
			CLEANUP_AND_RETURN( EAGAIN );
		}
	}
	else /* Engine mode */
	{
		/* 1. connect to evms daemon */
		if ( (PluginSocket = socket(AF_UNIX, SOCK_STREAM, 0)) < 0 )
		{
			LOG_ERROR( "Cannot create socket.\n" );
			CLEANUP_AND_RETURN( PluginSocket );
		}

		/*
		 * Create the address we will be connecting to.
		 */
		saun.sun_family = AF_UNIX;
		strcpy(saun.sun_path, ADDRESS);

		/*
		 * Try to connect to the address.  For this to
		 * succeed, the server must already have bound
		 * this address, and must have issued a listen()
		 * request.
		 *
		 * The third argument indicates the "length" of
		 * the structure, not just the length of the
		 * socket name.
		 */
		rc = sizeof(saun.sun_family) + strlen(saun.sun_path);

		if ( 0 > (rc = connect(PluginSocket, (struct sockaddr *)&saun, rc)) )
		{
			LOG_ERROR( "Failed to connect.\n" );
			CLEANUP_AND_RETURN( PluginSocket );
		}

		/* 2. retrieve configuration information from evms daemon */
		/* get cluster node number */
		recv( PluginSocket, (void*) &ClusterInfo.clusterNodeNum, sizeof(int), MSG_WAITALL);
		LOG_DEBUG( "clusterNodeNum=%d.\n", ClusterInfo.clusterNodeNum );
		recv( PluginSocket, &MyNodeNumber, sizeof(unsigned char), MSG_WAITALL);
		LOG_DEBUG( "MyNodeNumber=%d.\n", MyNodeNumber );
		recv( PluginSocket, (void*) &ClusterInfo.membership, sizeof(u_int32_t), MSG_WAITALL);
		LOG_DEBUG( "membership=%d.\n", ClusterInfo.membership );

		for ( rc = 0; rc < ClusterInfo.clusterNodeNum; rc++ )
		{
			recv( PluginSocket, &ClusterNodes[rc], sizeof(Node_Info_T), MSG_WAITALL);
			LOG_DEBUG( "node #%02d, name=%s.\n", rc, ClusterNodes[rc].nodeName );
		}

		/* 3. create thread to process messages from evms daemon */
		/* create thread to process messages IPC */
		rc = pthread_create(&PluginMainThread, NULL, HB2_main_engine, NULL );
		if ( 0 != rc )
		{
			LOG_SERIOUS("Failed to spawn a thread, try again\n");
			CLEANUP_AND_RETURN( EAGAIN );
		}
	}

	/* try to get a different number for every instance of
	 * the plugin.
	 * it will be easier to debug this way
	 */
	srand( time( NULL ));
	MessageIndex = rand();

//	usleep(100000);//give sometime for initialization

	rc = 0;
	LOG_EXIT_INT(rc);
	return rc;
}

static int HB2_register_callback(ece_callback_type_t type, ece_cb_t cb)
{
	int     rc = 0;

	LOG_ENTRY();

	/* sanity check */
	if ( NULL == cb )
	{
		rc = EINVAL;
		LOG_EXIT_INT(rc);
		return rc;
	}

	switch ( type )
	{
	case DELTAS:
		if ( 0 == pthread_mutex_lock( &ClusterInfoMutex ) )
		{
			LOG_DEBUG( "mode=%d: membership=%08x\n", EngineMode, ClusterInfo.membership );

			/* deliver the first membership if we are active */
			if ( 0 != ClusterInfo.membership )
			{
				ece_event_t     eceevent;
				int             i;

				/* set up the header */
				eceevent.transid =
				eceevent.quorum_flag = TRUE;

				/* one at a time */
				eceevent.num_entries = 1;

				/* join only */
				eceevent.type = DELTA_JOIN;

				for ( i = 0; i < MAX_NODES; i++ )
				{
					if ( ClusterInfo.membership & (1 << i ) )
					{
						memset(eceevent.node, 0, sizeof(ece_nodeid_t));
						eceevent.node[0].bytes[0] = (unsigned char) i;
						eceevent.node[0].bytes[1] = 0x86;	 //just to avoid all 0s.

						cb( CALLBACK_MEMBERSHIP, sizeof(ece_event_t), &eceevent);

						LOG_DEBUG("mode=%d: ECE Membership: type=%d\n", EngineMode, eceevent.type );
						LOG_DEBUG("mode=%d: ECE Membership: num_entries=%d\n", EngineMode, eceevent.num_entries );
						LOG_DEBUG("mode=%d: ECE Membership: node=%d\n", EngineMode, eceevent.node[0].bytes[0] );
					}
				}

				EceCallbackFunction = cb;
				pthread_mutex_unlock( &ClusterInfoMutex );
			}
			LOG_DEBUG( "mode=%d: callback registered\n", EngineMode );
		}
		break;
	case FULL_MEMBERSHIP:
		rc = ENOSYS;
	default:
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static  int HB2_unregister_callback(ece_cb_t cb)
{
	int     rc = 0;

	LOG_ENTRY();

	if ( 0 == pthread_mutex_lock( &EceCallbackMutex ) )
	{
		EceCallbackFunction = NULL;
		pthread_mutex_unlock( &EceCallbackMutex );
	}

	LOG_DEBUG( "mode=%d: callback unregistered\n", EngineMode );

	LOG_EXIT_INT(rc);
	return rc;
}

static int HB2_send_msg(ece_msg_t * msg)
{
	int     rc = EAGAIN;

	LOG_ENTRY();

	/* sanity check */
	if ( NULL == msg )
	{
		rc = EINVAL;
		LOG_EXIT_INT(rc);
		return rc;
	}

	/* generate the number if needed */
	if ( 0 == msg->corrolator )
	{
		msg->corrolator = (++MessageIndex << 8 ) + MyNodeNumber + 1;
	}

	if ( ENGINE_DAEMON == EngineMode )
	{
		u_int32_t       fromTo = (ECE_DAEMON_MASK | MyNodeNumber);
		struct ha_msg*  hamsg = NULL;

		if ( 0 == pthread_mutex_lock( &EceHaApiMutex) )
		{
			/* get a new message */
			hamsg = ha_msg_new(0);
			pthread_mutex_unlock( &EceHaApiMutex);
		}

		if ( NULL == hamsg )
		{
			LOG_ERROR( "mode=%d: no memory\n", EngineMode );

			rc = ENOMEM;
			LOG_EXIT_INT(rc);
			return rc;
		}

		/* construct the message */
		ha_msg_add(hamsg, F_TYPE, EVMS_GROUP_NAME );
		ha_msg_add_int( hamsg, EVMS_CORROLATOR, msg->corrolator );
		ha_msg_add_int( hamsg, EVMS_CMD, msg->cmd );
		ha_msg_add_int( hamsg, EVMS_FROM, fromTo );

		/* only if we have some data */
		if ( 0 < msg->size )
		{
			rc = ha_msg_addbin( hamsg, EVMS_DATA, msg->msg, msg->size );
		}

		/* no message constructed? */
		if( HA_OK != rc )
		{
			LOG_ERROR( "mode=%d: Message too long!\n", EngineMode );
		}
		else
		{
			/* delivery to heartbeat and eventually to daemon */
			if ( 0 == memcmp( &ECE_BROADCASE_ADDR, msg->node.bytes, sizeof(ece_nodeid_t)) )
			{
				if ( 0 == pthread_mutex_lock( &EceHaApiMutex) )
				{
					/* send to everybody */
					rc = pEceHbHandle->llc_ops->sendclustermsg(pEceHbHandle, hamsg);
					pthread_mutex_unlock( &EceHaApiMutex);
				}
				else
				{
					LOG_ERROR( "failed to obtain HA API lock!\n" );
				}
			}
			else
			{
				/* to the local Engine? */
				if ( msg->node.bytes[0] == MyNodeNumber )
				{
					/* don't send it to HA */
					if ( 0 == pthread_mutex_lock( &EngineIPCMutex ) )
					{
						ece_ha_ipc_msg_hdr_t    ipcmsg;
						int                     len = sizeof( ece_ha_ipc_msg_hdr_t ) + msg->size;

						/* copy the int fields */
						ipcmsg.msgType = IPC_MSG_EVMS_MSG_TYPE;
						ipcmsg.corrolator = msg->corrolator;
						ipcmsg.cmd = msg->cmd;
						ipcmsg.size = msg->size;
						ipcmsg.fromTo = MyNodeNumber;

						/* write the fixed header */
						write( PluginIPCFd, &len, sizeof( int ));
						write( PluginIPCFd, &ipcmsg, sizeof( ece_ha_ipc_msg_hdr_t ));

						/* any data? */
						if ( 0 < msg->size )
						{
							/* dump it here */
							write( PluginIPCFd, msg->msg, msg->size );
						}

						pthread_mutex_unlock( &EngineIPCMutex);

						rc = HA_OK;
					}
					else
					{
						LOG_ERROR( "failed to obtain IPC lock!\n" );
					}
				}
				else
				{
					if ( 0 == pthread_mutex_lock( &EceHaApiMutex) )
					{
						/* deliver to one node */
						rc = pEceHbHandle->llc_ops->sendnodemsg_byuuid(pEceHbHandle, hamsg, &ClusterNodes[msg->node.bytes[0]].nodeID );
						pthread_mutex_unlock( &EceHaApiMutex);
					}
					else
					{
						LOG_ERROR( "failed to obtain HA API lock!\n" );
					}
				}
			}
		}

		LOG_DEBUG( "mode=%d: Daemon->Engine message corrolator=%08x\n", EngineMode, msg->corrolator );
		LOG_DEBUG( "mode=%d: Daemon->Engine message cmd=%08x\n", EngineMode, msg->cmd );
		LOG_DEBUG( "mode=%d: Daemon->Engine message size=%d\n", EngineMode, msg->size );
		LOG_DEBUG( "mode=%d: Daemon->Engine message fromTo=%08x (hamsg %d) \n", EngineMode, fromTo, MyNodeNumber );
		LOG_DEBUG( "mode=%d: Daemon->Engine message rc=%d\n", EngineMode, rc );

		if ( 0 == pthread_mutex_lock( &EceHaApiMutex) )
		{
			/* delete the message */
			ha_msg_del( hamsg );
			pthread_mutex_unlock( &EceHaApiMutex);
		}
		else
		{
			LOG_ERROR( "failed to obtain HA API lock!\n" );
		}

		/* translate the return code */
		rc = HA_OK == rc ? 0 : EAGAIN;
	}
	else
	{
		ece_ha_ipc_msg_hdr_t    ipcmsg;
		int                     len = sizeof( ece_ha_ipc_msg_hdr_t ) + msg->size;

		/* copy the int fields */
		ipcmsg.corrolator = msg->corrolator;
		ipcmsg.cmd = msg->cmd;
		ipcmsg.size = msg->size;
		ipcmsg.fromTo = 0;

		if ( 0 == pthread_mutex_lock( &EngineIPCMutex ) )
		{
			/* write the header */
			rc = write( PluginSocket, &len, sizeof( int ));
			if ( sizeof( int ) != rc )
			{
				/* release the lock */
				pthread_mutex_unlock( &EngineIPCMutex );

				rc = EAGAIN;
				LOG_EXIT_INT(rc);
				return rc;
			}

			/* broadcast message? */
			if ( 0 == memcmp( &ECE_BROADCASE_ADDR, msg->node.bytes, sizeof(ece_nodeid_t)) )
			{
				ipcmsg.fromTo |= ECE_BROADCAST_MASK;
			}
			else /* send to the node */
			{
				ipcmsg.fromTo |= msg->node.bytes[0];
			}

			/* write the to-fields */
			rc = write( PluginSocket, &ipcmsg, sizeof( ece_ha_ipc_msg_hdr_t ));

			if ( (sizeof( ece_ha_ipc_msg_hdr_t )) != rc )
			{
				/* release the lock */
				pthread_mutex_unlock( &EngineIPCMutex );

				rc = EAGAIN;
				LOG_EXIT_INT(rc);
				return rc;
			}

			if ( msg->size > 0 )
			{
				/* write the to-fields */
				rc = write( PluginSocket, msg->msg, msg->size );

				if ( msg->size != rc )
				{
					/* release the lock */
					pthread_mutex_unlock( &EngineIPCMutex );

					rc = EAGAIN;
					LOG_EXIT_INT(rc);
					return rc;
				}
			}

			/* release the lock */
			pthread_mutex_unlock( &EngineIPCMutex );

			LOG_DEBUG( "mode=%d: Engine->Daemon message corrolator=%08x\n", EngineMode, ipcmsg.corrolator );
			LOG_DEBUG( "mode=%d: Engine->Daemon message cmd=%08x\n", EngineMode, ipcmsg.cmd );
			LOG_DEBUG( "mode=%d: Engine->Daemon message size=%d\n", EngineMode, ipcmsg.size );
			LOG_DEBUG( "mode=%d: Engine->Daemon message fromTo=%08x\n", EngineMode, ipcmsg.fromTo );

			rc = 0;
		}
		else
		{
			LOG_ERROR( "failed to obtain IPC lock!\n" );
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static  int HB2_get_clusterid(ece_clusterid_t * clusterid)
{
	int     rc = 0;

	LOG_ENTRY();

	/* sanity check */
	if ( NULL == clusterid )
	{
		rc = EINVAL;
		LOG_EXIT_INT(rc);
		return rc;
	}

	strcpy( (void*)clusterid, "linuxha" );

	LOG_EXIT_INT(rc);
	return rc;
}

static  int HB2_get_my_nodeid(ece_nodeid_t * nodeid)
{
	int     rc = 0;

	LOG_ENTRY();

	/* sanity check */
	if ( NULL == nodeid )
	{
		rc = EINVAL;
		LOG_EXIT_INT(rc);
		return rc;
	}

	/* set the id */
	memset( nodeid->bytes, 0, sizeof(ece_nodeid_t) );
	nodeid->bytes[0]  = MyNodeNumber;
	nodeid->bytes[1]  = 0x86;

	LOG_EXIT_INT(rc);
	return rc;
}

static  int HB2_get_num_config_nodes(uint * num_nodes)
{
	int     rc = 0;

	LOG_ENTRY();

	/* sanity check */
	if ( NULL == num_nodes )
	{
		rc = EINVAL;
		LOG_EXIT_INT(rc);
		return rc;
	}

	/* set the number */
	*num_nodes = ClusterInfo.clusterNodeNum;

	LOG_DEBUG("mode=%d(%d): rc = %d.\n", EngineMode, *num_nodes, rc );

	LOG_EXIT_INT(rc);
	return rc;
}

static  int HB2_get_all_nodes(uint * num_nodes, ece_nodeid_t * nodes)
{
	int     i, rc = 0;

	LOG_ENTRY();

	/* sanity check */
	if ( NULL == nodes )
	{
		rc = EINVAL;
		LOG_EXIT_INT(rc);
		return rc;
	}

	/* fill as much as we can */
	for ( i = 0; i < ClusterInfo.clusterNodeNum && i < *num_nodes; i++ )
	{
		memset( nodes[i].bytes, 0, sizeof(ece_nodeid_t) );
		nodes[i].bytes[0]  = i;
		nodes[i].bytes[1]  = 0x86;
	}

	if ( num_nodes ) *num_nodes = ClusterInfo.clusterNodeNum;

	LOG_EXIT_INT(rc);
	return rc;
}

static  int HB2_get_membership(ece_event_t * membership_event)
{
	int             rc = 0;
	int             i, member;
	u_int32_t       localCachedMembership = ClusterInfo.membership;

	LOG_ENTRY();

	/* sanity check */
	if ( NULL == membership_event )
	{
		rc = EINVAL;
		LOG_EXIT_INT(rc);
		return rc;
	}

	/* ask the caller to retry */
	if ( 0 == localCachedMembership )
	{
		rc = EAGAIN;
		LOG_EXIT_INT(rc);
		return rc;
	}

	/* get the number of current members */
	for ( i = 0, member = 0; i < MAX_NODES; i++ )
	{
		if ( localCachedMembership & (1 << i) )
		{
			member++;
		}
	}

	membership_event->type = MEMBERSHIP;
	membership_event->transid = 0;
	membership_event->quorum_flag = TRUE;

	if ( membership_event->num_entries < member )
	{
		membership_event->num_entries = member;
		rc = ENOSPC;
		LOG_EXIT_INT(rc);
		return rc;
	}

	/* set the num of members */
	membership_event->num_entries = member;

	/* one more round */
	for ( i = 0, member = 0; i < MAX_NODES; i++ )
	{
		if ( localCachedMembership & (1 << i) )
		{
			/* set the node ID */
			memset(&membership_event->node[member], 0, sizeof(ece_nodeid_t));
			membership_event->node[member].bytes[0] = (unsigned char) i;
			membership_event->node[member].bytes[1] = 0x86;

			member++;
		}
	}

	LOG_DEBUG("mode=%d(%08x): rc = %d.\n", EngineMode, localCachedMembership, rc );

	LOG_EXIT_INT(rc);
	return rc;
}

static  int HB2_nodeid_to_string(const ece_nodeid_t * nodeid, char* string, uint* len)
{
	int     rc = 0;
	int     name_len;

	LOG_ENTRY();

	if ( NULL == len || NULL == nodeid )
	{
		rc = EINVAL;
		LOG_EXIT_INT(rc);
		return rc;
	}

	if ( ClusterInfo.clusterNodeNum > nodeid->bytes[0] )
	{
		name_len = strlen( ClusterNodes[nodeid->bytes[0]].nodeName )+1;
	}
	else
	{
		/* node #02 */
		name_len = 10;
	}

	/* not enough room? */
	if ( *len < name_len )
	{
		rc = ENOSPC;
	}
	else if ( string )
	{
		if ( ClusterInfo.clusterNodeNum > nodeid->bytes[0] )
		{
			strcpy(string, ClusterNodes[nodeid->bytes[0]].nodeName );
		}
		else
		if ( MAX_NODES > nodeid->bytes[0] )
		{
			sprintf(string, "Node #%02d", nodeid->bytes[0]+1 );
		}
		else
		{
			rc = EINVAL;
		}

		LOG_DEBUG("mode=%d: node #%02d = \"%s\", rc = %d.\n", EngineMode,
			  nodeid->bytes[0], string, rc );
	}

	*len = name_len;
	LOG_EXIT_INT(rc);
	return rc;
}

static  int HB2_string_to_nodeid(const char   * string,   ece_nodeid_t * nodeid)
{
	int     rc = ENOENT;
	int     i;

	LOG_ENTRY();

	/* sanity check */
	if ( NULL == nodeid || NULL == string )
	{
		rc = EINVAL;
		LOG_EXIT_INT(rc);
		return rc;
	}

	/* go through all the nodes */
	for ( i = 0; i < ClusterInfo.clusterNodeNum; i++ )
	{
		/* compare the name */
		if ( 0 == strcmp( ClusterNodes[nodeid->bytes[0]].nodeName, string ) )
		{
			/* set the node ID */
			memset( nodeid->bytes, 0, sizeof(ece_nodeid_t));
			nodeid->bytes[0] = (unsigned char) i;
			nodeid->bytes[1] = 0x86;

			rc = 0;
			break;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

#ifndef SET_STRING
	#define SET_STRING( d, s )	d = EngFncs->engine_strdup( s )
#endif

static int HB2_get_plugin_info(char  * info_name, extended_info_array_t** info_array )
{
	int                     rc = 0;
	extended_info_array_t * info = NULL;
	char                    buffer[51] = {0};
	int                     i = 0;

	LOG_ENTRY();

	/* Parameter check */
	if ( NULL == info_array )
	{
		rc = EINVAL;
		LOG_EXIT_INT(rc);
		return rc;
	}


	if ( NULL == info_name )
	{
		info = EngFncs->engine_alloc( sizeof(extended_info_array_t) +  sizeof(extended_info_t)*5);

		if ( NULL == info )
		{
			rc = ENOMEM;
			LOG_EXIT_INT(rc);
			return(rc);
		}

		SET_STRING(info->info[i].name, "ShortName");
		SET_STRING(info->info[i].title, "Short Name");
		SET_STRING(info->info[i].desc, "A short name given to this plugin");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, HB2_plugin_record.short_name);
		i++;

		SET_STRING(info->info[i].name, "LongName");
		SET_STRING(info->info[i].title, "Long Name");
		SET_STRING(info->info[i].desc, "A long name given to this plugin");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, HB2_plugin_record.long_name);
		i++;

		SET_STRING(info->info[i].name, "Type");
		SET_STRING(info->info[i].title, "Plugin Type");
		SET_STRING(info->info[i].desc, "There are various "
			   "types of plugins; each responsible for "
			   "some kind of storage object.");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, "Cluster Manager");
		i++;

		SET_STRING(info->info[i].name, "Version");
		SET_STRING(info->info[i].title, "Plugin Version");
		SET_STRING(info->info[i].desc,
			   "This is the version number of the plugin.");
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", MAJOR_VERSION,
			 MINOR_VERSION, PATCH_LEVEL);
		SET_STRING(info->info[i].value.s, buffer);
		i++;

		SET_STRING(info->info[i].name, "Required_Engine_Version");
		SET_STRING(info->info[i].title, "Required Engine Services Version");
		SET_STRING(info->info[i].desc, "This is the version of the Engine "
			   "services that this plug-in requires.  It will not "
			   "run on older versions of the Engine services.");
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d",
			 HB2_plugin_record.required_engine_api_version.major,
			 HB2_plugin_record.required_engine_api_version.minor,
			 HB2_plugin_record.required_engine_api_version.patchlevel);
		SET_STRING(info->info[i].value.s, buffer);
		i++;

		SET_STRING(info->info[i].name, "Required_Cluster_Version");
		SET_STRING(info->info[i].title, "Required Cluster API Version");
		SET_STRING(info->info[i].desc, "This is the version of the "
			   "Engine's cluster plug-in API that this plug-in "
			   "requires.  It will not run on older versions of "
			   "the Engine's cluster plug-in API.");
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d",
			 HB2_plugin_record.required_plugin_api_version.cluster.major,
			 HB2_plugin_record.required_plugin_api_version.cluster.minor,
			 HB2_plugin_record.required_plugin_api_version.cluster.patchlevel);
		SET_STRING(info->info[i].value.s, buffer);
		i++;
	}
	else
	{
		LOG_ERROR("No support for extra plugin information about \"%s\"\n", info_name);

		rc = EINVAL;
		LOG_EXIT_INT(rc);
		return(rc);
	}

	info->count = i;
	*info_array = info;

	LOG_EXIT_INT(0);
	return(0);
}

static int HB2_get_plugin_functions(ece_nodeid_t * nodeid,
					  function_info_array_t * * actions)
{
	int     rc = ENOSYS;

	LOG_ENTRY();
	LOG_ERROR("Error: Not supported Yet\n");
	LOG_EXIT_INT(rc);
	return rc;
}

static int
HB2_plugin_function(ece_nodeid_t * nodeid,
			  task_action_t  action,
			  list_anchor_t  objects,
			  option_array_t * options)
{
	int     rc = ENOSYS;

	LOG_ENTRY();
	LOG_ERROR("Error: Not supported Yet\n");
	LOG_EXIT_INT(rc);
	return rc;
}

static cluster_functions_t ft={
	setup_evms_plugin:      HB2_init,
	cleanup_evms_plugin:    HB2_cleanup,
	register_callback:      HB2_register_callback,
	unregister_callback:    HB2_unregister_callback,
	send_msg:               HB2_send_msg,
	get_clusterid:          HB2_get_clusterid,
	get_my_nodeid:          HB2_get_my_nodeid,
	get_num_config_nodes:   HB2_get_num_config_nodes,
	get_all_nodes:          HB2_get_all_nodes,
	get_membership:         HB2_get_membership,
	nodeid_to_string:       HB2_nodeid_to_string,
	string_to_nodeid:       HB2_string_to_nodeid,
	get_plugin_info:        HB2_get_plugin_info,
	get_plugin_functions:   HB2_get_plugin_functions,
	plugin_function:        HB2_plugin_function
};


plugin_record_t HB2_plugin_record = {

	id:                     EVMS_HB2_PLUGIN_ID,

	version:
		{major:         MAJOR_VERSION,
		minor:          MINOR_VERSION,
		patchlevel:     PATCH_LEVEL},

	required_engine_api_version:
		{major:         ENGINE_SERVICES_API_MAJOR_VERION,
		minor:          ENGINE_SERVICES_API_MINOR_VERION,
		patchlevel:     ENGINE_SERVICES_API_PATCH_LEVEL},

	required_plugin_api_version: {cluster:
		{major:         ENGINE_CLUSTER_API_MAJOR_VERION,
		minor:          ENGINE_CLUSTER_API_MINOR_VERION,
		patchlevel:     ENGINE_CLUSTER_API_PATCH_LEVEL}},

	short_name:             EVMS_HB2_PLUGIN_SHORT_NAME,
	long_name:              EVMS_HB2_PLUGIN_LONG_NAME,
	oem_name:               EVMS_NOVELL_OEM_NAME,

	functions:              {cluster:   &ft},

	container_functions:    NULL
};

plugin_record_t * evms_plugin_records[] = {&HB2_plugin_record, NULL};
