/*
 *   (C) Copyright IBM Corp. 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: LVM2 Plugin
 * File: evms2/engine/plugins/lvm2/info.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "lvm2.h"

/**
 * get_region_info
 **/
int get_region_info(storage_object_t *region,
		    extended_info_array_t **info_array)
{
	region_data_t *r_data = region->private_data;
	extended_info_array_t *info;
	char region_uuid[LVM2_UUID_LEN+10];
	int i = 0;

	LOG_ENTRY();
	LOG_DEBUG("Getting basic info for region %s.\n", region->name);

	info = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
				     sizeof(extended_info_t) * 5);
	if (!info) {
		LOG_ERROR("Error allocating memory for info array.\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	/* Region Name */
	info->info[i].name = EngFncs->engine_strdup("Name");
	info->info[i].title = EngFncs->engine_strdup(_("Region Name"));
	info->info[i].desc = EngFncs->engine_strdup(_("Name of this LVM2 region (LV)."));
	info->info[i].type = EVMS_Type_String;
	info->info[i].value.s = EngFncs->engine_strdup(region->name);
	i++;

	/* Region UUID */
	format_uuid(r_data->uuid, region_uuid);
	info->info[i].name = EngFncs->engine_strdup("UUID");
	info->info[i].title = EngFncs->engine_strdup(_("Region UUID"));
	info->info[i].desc = EngFncs->engine_strdup(_("Unique identifier for this region."));
	info->info[i].type = EVMS_Type_String;
	info->info[i].value.s = EngFncs->engine_strdup(region_uuid);
	i++;

	/* Container Name */
	info->info[i].name = EngFncs->engine_strdup("Container");
	info->info[i].title = EngFncs->engine_strdup(_("Container Name"));
	info->info[i].desc = EngFncs->engine_strdup(_("Name of the LVM2 container (VG) that produces this region."));
	info->info[i].type = EVMS_Type_String;
	info->info[i].value.s = EngFncs->engine_strdup(region->producing_container->name);
	i++;

	/* Region Size */
	info->info[i].name = EngFncs->engine_strdup("Size");
	info->info[i].title = EngFncs->engine_strdup(_("Region Size"));
	info->info[i].desc = EngFncs->engine_strdup(_("Space allocated for this region."));
	info->info[i].type = EVMS_Type_Unsigned_Int64;
	info->info[i].unit = EVMS_Unit_Sectors;
	info->info[i].value.ui64 = region->size;
	i++;

	/* Region Mappings */
	info->info[i].name = EngFncs->engine_strdup("Mappings");
	info->info[i].title = EngFncs->engine_strdup(_("Region Mappings"));
	info->info[i].desc = EngFncs->engine_strdup(_("Number of mappings for this region."));
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].value.ui32 = EngFncs->list_count(r_data->mappings);
	info->info[i].flags = EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE;
	i++;
	
	info->count = i;
	*info_array = info;

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * get_region_mappings_info
 *
 * Get info about the mappings for this region. This simply provides a list
 * of all mappings, with only their starting-LE. Each entry in this list has
 * extra information available.
 **/
int get_region_mappings_info(storage_object_t *region,
			     extended_info_array_t **info_array)
{
	region_data_t *r_data = region->private_data;
	extended_info_array_t *info;
	region_mapping_t *r_map;
	list_element_t iter;
	u_int32_t mapping_count, i = 0, j = 0;
	u_int64_t extents_per_stripe, k;
	char buffer[256];

	LOG_ENTRY();
	LOG_DEBUG("Getting mappings list for region %s.\n", region->name);

	mapping_count = EngFncs->list_count(r_data->mappings);

	info = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
				     sizeof(extended_info_t) * mapping_count * 6);
	if (!info) {
		LOG_ERROR("Error allocating memory for info array.\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	LIST_FOR_EACH(r_data->mappings, iter, r_map) {

		/* Start LE */
		snprintf(buffer, 256, "mapping%u_start_le", j);
		info->info[i].name = EngFncs->engine_strdup(buffer);
		snprintf(buffer, 256, _("Mapping %u: Starting Extent"), j);
		info->info[i].title = EngFncs->engine_strdup(buffer);
		info->info[i].desc = EngFncs->engine_strdup(_("Extent within the region where this mapping begins."));
		info->info[i].type = EVMS_Type_Unsigned_Int64;
		info->info[i].value.ui64 = r_map->start_le;
		i++;

		/* LE Count */
		snprintf(buffer, 256, "mapping%u_le_count", j);
		info->info[i].name = EngFncs->engine_strdup(buffer);
		snprintf(buffer, 256, _("Mapping %u: Extent Count"), j);
		info->info[i].title = EngFncs->engine_strdup(buffer);
		info->info[i].desc = EngFncs->engine_strdup(_("Number of extents in this mapping."));
		info->info[i].type = EVMS_Type_Unsigned_Int64;
		info->info[i].value.ui64 = r_map->le_count;
		i++;

		/* Type */
		snprintf(buffer, 256, "mapping%u_type", j);
		info->info[i].name = EngFncs->engine_strdup(buffer);
		snprintf(buffer, 256, _("Mapping %u: Mapping Type"), j);
		info->info[i].title = EngFncs->engine_strdup(buffer);
		info->info[i].type = EVMS_Type_String;
		if (r_map->stripe_count > 1) {
			info->info[i].value.s = EngFncs->engine_strdup(_("Striped"));
		} else {
			info->info[i].value.s = EngFncs->engine_strdup(_("Linear"));
		}
		i++;

		if (r_map->stripe_count > 1) {
			/* Stripe Count */
			snprintf(buffer, 256, "mapping%u_stripes", j);
			info->info[i].name = EngFncs->engine_strdup(buffer);
			snprintf(buffer, 256, _("Mapping %u: Stripe Count"), j);
			info->info[i].title = EngFncs->engine_strdup(buffer);
			info->info[i].desc = EngFncs->engine_strdup(_("Number of stripes in this mapping."));
			info->info[i].type = EVMS_Type_Unsigned_Int64;
			info->info[i].value.ui64 = r_map->stripe_count;
			i++;

			/* Stripe Size */
			snprintf(buffer, 256, "mapping%u_stripe_size", j);
			info->info[i].name = EngFncs->engine_strdup(buffer);
			snprintf(buffer, 256, _("Mapping %u: Stripe Size"), j);
			info->info[i].title = EngFncs->engine_strdup(buffer);
			info->info[i].desc = EngFncs->engine_strdup(_("Size of each stripe \"chunk\"."));
			info->info[i].type = EVMS_Type_Unsigned_Int64;
			info->info[i].unit = EVMS_Unit_Sectors;
			info->info[i].value.ui64 = r_map->stripe_size;
			i++;
		}

		/* List of PVs and PEs */
		snprintf(buffer, 256, "mapping%u_pvs", j);
		info->info[i].name = EngFncs->engine_strdup(buffer);
		snprintf(buffer, 256, _("Mapping %u: PVs and PEs"), j);
		info->info[i].title = EngFncs->engine_strdup(buffer);
		info->info[i].desc = EngFncs->engine_strdup(_("The target objects and physical-extents for this mapping."));
		info->info[i].type = EVMS_Type_String;
		info->info[i].collection_type = EVMS_Collection_List;
		info->info[i].collection.list = EngFncs->engine_alloc(sizeof(value_list_t) +
								      sizeof(value_t) * (r_map->stripe_count + 1));
		info->info[i].collection.list->count = r_map->stripe_count + 1;

		snprintf(buffer, 256, "%-20s : %-15s : %s", _("Object"), _("Start PE"), _("PE Count"));
		info->info[i].collection.list->value[0].s = EngFncs->engine_strdup(buffer);

		extents_per_stripe = r_map->le_count / r_map->stripe_count;
		for (k = 0; k < r_map->stripe_count; k++) {
			snprintf(buffer, 256, "%-20s : %-15"PRIu64" : %"PRIu64,
				 r_map->le_maps[k].map[0].pe->pv_data->object->name,
				 r_map->le_maps[k].map[0].pe->number,
				 extents_per_stripe);
			info->info[i].collection.list->value[k+1].s = EngFncs->engine_strdup(buffer);
		}
		i++;

		/* FIXME: Add mappings for "new" LE-map. */

		j++;
	}

	info->count = i;
	*info_array = info;

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * get_container_info
 **/
int get_container_info(storage_container_t *container,
		       extended_info_array_t **info_array)
{
	container_data_t *c_data = container->private_data;
	storage_object_t *freespace;
	extended_info_array_t *info;
	char container_uuid[LVM2_UUID_LEN+10];
	u_int32_t i = 0;

	LOG_ENTRY();
	LOG_DEBUG("Getting basic info for container %s.\n", container->name);

	info = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
				     sizeof(extended_info_t) * 11);
	if (!info) {
		LOG_ERROR("Error allocating memory for info array.\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	/* Container Name */
	info->info[i].name = EngFncs->engine_strdup("Name");
	info->info[i].title = EngFncs->engine_strdup(_("Container Name"));
	info->info[i].desc = EngFncs->engine_strdup(_("Name of LVM2 Container (VG)."));
	info->info[i].type = EVMS_Type_String;
	info->info[i].value.s = EngFncs->engine_strdup(container->name);
	i++;

	/* Container UUID */
	format_uuid(c_data->uuid, container_uuid);
	info->info[i].name = EngFncs->engine_strdup("UUID");
	info->info[i].title = EngFncs->engine_strdup(_("Container UUID"));
	info->info[i].desc = EngFncs->engine_strdup(_("Unique identifier for this container."));
	info->info[i].type = EVMS_Type_String;
	info->info[i].value.s = EngFncs->engine_strdup(container_uuid);
	i++;

	/* Container Size */
	info->info[i].name = EngFncs->engine_strdup("Size");
	info->info[i].title = EngFncs->engine_strdup(_("Container Size"));
	info->info[i].desc = EngFncs->engine_strdup(_("Total accumulated space in this container."));
	info->info[i].type = EVMS_Type_Unsigned_Int64;
	info->info[i].unit = EVMS_Unit_Sectors;
	info->info[i].value.ui64 = container->size;
	i++;

	/* Available Container Size */
	freespace = get_freespace_region(container->objects_produced);
	info->info[i].name = EngFncs->engine_strdup("Freespace");
	info->info[i].title = EngFncs->engine_strdup(_("Available Space"));
	info->info[i].desc = EngFncs->engine_strdup(_("Amount of space currently available for allocating to regions."));
	info->info[i].type = EVMS_Type_Unsigned_Int64;
	info->info[i].unit = EVMS_Unit_Sectors;
	info->info[i].value.ui64 = freespace->size;
	i++;

	/* Percent Allocated */
	info->info[i].name = EngFncs->engine_strdup("Percent_Allocated");
	info->info[i].title = EngFncs->engine_strdup(_("Percent Allocated"));
	info->info[i].desc = EngFncs->engine_strdup(_("Percentage of space currently allocated to regions"));
	info->info[i].type = EVMS_Type_Real32;
	info->info[i].unit = EVMS_Unit_Percent;
	info->info[i].value.r32 = ((double)(container->size - freespace->size) /
				   (double)(container->size)) * 100.0;
	i++;

	/* Extent Size */
	info->info[i].name = EngFncs->engine_strdup("Extent_Size");
	info->info[i].title = EngFncs->engine_strdup(_("Extent Size"));
	info->info[i].desc = EngFncs->engine_strdup(_("Size of each extent available for allocating to regions."));
	info->info[i].type = EVMS_Type_Unsigned_Int64;
	info->info[i].unit = EVMS_Unit_Sectors;
	info->info[i].value.ui64 = c_data->pe_size;
	i++;

	/* Total Extents */
	info->info[i].name = EngFncs->engine_strdup("Extents");
	info->info[i].title = EngFncs->engine_strdup(_("Total Extents"));
	info->info[i].desc = EngFncs->engine_strdup(_("Total number of extents in the container."));
	info->info[i].type = EVMS_Type_Unsigned_Int64;
	info->info[i].value.ui64 = container->size / c_data->pe_size;
	i++;

	/* Available Extents */
	info->info[i].name = EngFncs->engine_strdup("Available_Extents");
	info->info[i].title = EngFncs->engine_strdup(_("Available Extents"));
	info->info[i].desc = EngFncs->engine_strdup(_("Number of extents available for allocating to regions"));
	info->info[i].type = EVMS_Type_Unsigned_Int64;
	info->info[i].value.ui64 = freespace->size / c_data->pe_size;
	i++;

	/* Sequence Number */
	info->info[i].name = EngFncs->engine_strdup("Sequence");
	info->info[i].title = EngFncs->engine_strdup(_("Sequence Number"));
	info->info[i].desc = EngFncs->engine_strdup(_("Number of times that this container's metadata has been written to disk."));
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].value.ui32 = c_data->sequence;
	i++;

	/* Number of Objects */
	info->info[i].name = EngFncs->engine_strdup("Objects");
	info->info[i].title = EngFncs->engine_strdup(_("Number of Objects (PVs)"));
	info->info[i].desc = EngFncs->engine_strdup(_("Number of objects consumed by this container"));
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].value.ui32 = EngFncs->list_count(container->objects_consumed);
	info->info[i].flags = EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE;
	i++;

	/* Number of regions */
	info->info[i].name = EngFncs->engine_strdup("Regions");
	info->info[i].title = EngFncs->engine_strdup(_("Number of Regions (LVs)"));
	info->info[i].desc = EngFncs->engine_strdup(_("Number of regions produced by this container"));
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].value.ui32 = EngFncs->list_count(container->objects_produced) - 1;
	info->info[i].flags = EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE;
	i++;

	info->count = i;
	*info_array = info;

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * get_container_regions_info
 **/
int get_container_regions_info(storage_container_t *container,
			       extended_info_array_t **info_array)
{
	extended_info_array_t *info;
	storage_object_t *region;
	list_element_t iter;
	char buffer[64];
	u_int32_t i = 0;

	LOG_ENTRY();
	LOG_DEBUG("Getting list of regions in container %s.\n", container->name);

	info = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
				     sizeof(extended_info_t) *
				     EngFncs->list_count(container->objects_produced));
	if (!info) {
		LOG_ERROR("Error allocating memory for info array.\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	LIST_FOR_EACH(container->objects_produced, iter, region) {
		if (region->data_type != DATA_TYPE) {
			continue;
		}

		snprintf(buffer, 64, "Region%u", i);
		info->info[i].name = EngFncs->engine_strdup(buffer);
		snprintf(buffer, 64, _("Region (LV) %u"), i);
		info->info[i].title = EngFncs->engine_strdup(buffer);
		info->info[i].type = EVMS_Type_String;
		info->info[i].value.s = EngFncs->engine_strdup(region->name);
		i++;
	}

	info->count = i;
	*info_array = info;

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * get_container_objects_info
 **/
int get_container_objects_info(storage_container_t *container,
			       extended_info_array_t **info_array)
{
	extended_info_array_t *info;
	storage_object_t *object;
	list_element_t iter;
	char buffer[64];
	u_int32_t i = 0;

	LOG_ENTRY();
	LOG_DEBUG("Getting list of objects in container %s.\n", container->name);

	info = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
				     sizeof(extended_info_t) *
				     EngFncs->list_count(container->objects_consumed));
	if (!info) {
		LOG_ERROR("Error allocating memory for info array.\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	LIST_FOR_EACH(container->objects_consumed, iter, object) {
		snprintf(buffer, 64, "Object%u", i);
		info->info[i].name = EngFncs->engine_strdup(buffer);
		snprintf(buffer, 64, _("Object (PV) %u"), i);
		info->info[i].title = EngFncs->engine_strdup(buffer);
		info->info[i].type = EVMS_Type_String;
		info->info[i].value.s = EngFncs->engine_strdup(object->name);
		info->info[i].flags = EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE;
		i++;
	}

	info->count = i;
	*info_array = info;

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * get_container_object_info
 **/
int get_container_object_info(storage_container_t *container,
			      extended_info_array_t **info_array,
			      u_int32_t object_index)
{
	container_data_t *c_data = container->private_data;
	extended_info_array_t *info;
	storage_object_t *object;
	list_element_t iter;
	pv_data_t *pv_data;
	physical_extent_t *pe_map;
	char pv_uuid[LVM2_UUID_LEN+10];
	char buffer[256];
	u_int64_t j, k, l = 0;
	u_int32_t i = 0;

	LOG_ENTRY();
	LOG_DEBUG("Getting info for object %u in container %s.\n",
		  object_index, container->name);

	/* Find the desired object. */
	LIST_FOR_EACH(container->objects_consumed, iter, object) {
		if (i == object_index) {
			break;
		}
		i++;
	}
	if (!object) {
		LOG_ERROR("Specified object %u, but only %u objects exist.\n",
			  object_index, i);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}
	pv_data = object->consuming_private_data;
	pe_map = pv_data->pe_map;
	i = 0;

	info = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
				     sizeof(extended_info_t) * 8);
	if (!info) {
		LOG_ERROR("Error allocating memory for info array.\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	/* Object Name */
	info->info[i].name = EngFncs->engine_strdup("Name");
	info->info[i].title = EngFncs->engine_strdup(_("Object Name"));
	info->info[i].desc = EngFncs->engine_strdup(_("Name of LVM2 object (PV)."));
	info->info[i].type = EVMS_Type_String;
	info->info[i].value.s = EngFncs->engine_strdup(object->name);
	i++;

	/* Object UUID */
	format_uuid(pv_data->uuid, pv_uuid);
	info->info[i].name = EngFncs->engine_strdup("UUID");
	info->info[i].title = EngFncs->engine_strdup(_("Object UUID"));
	info->info[i].desc = EngFncs->engine_strdup(_("Unique identifier for this object."));
	info->info[i].type = EVMS_Type_String;
	info->info[i].value.s = EngFncs->engine_strdup(pv_uuid);
	i++;

	/* Object Size */
	info->info[i].name = EngFncs->engine_strdup("Size");
	info->info[i].title = EngFncs->engine_strdup(_("Object Size"));
	info->info[i].type = EVMS_Type_Unsigned_Int64;
	info->info[i].unit = EVMS_Unit_Sectors;
	info->info[i].value.ui64 = object->size;
	i++;

	/* Available Size */
	info->info[i].name = EngFncs->engine_strdup("Freespace");
	info->info[i].title = EngFncs->engine_strdup(_("Available Space"));
	info->info[i].desc = EngFncs->engine_strdup(_("Total space currently available for allocating to new regions."));
	info->info[i].type = EVMS_Type_Unsigned_Int64;
	info->info[i].unit = EVMS_Unit_Sectors;
	info->info[i].value.ui64 = count_available_extents_in_pv(object) *
				   c_data->pe_size;
	i++;

	/* Total Extents */
	info->info[i].name = EngFncs->engine_strdup("Extents");
	info->info[i].title = EngFncs->engine_strdup(_("Total Extents"));
	info->info[i].desc = EngFncs->engine_strdup(_("Total number of extents in the object."));
	info->info[i].type = EVMS_Type_Unsigned_Int64;
	info->info[i].value.ui64 = pv_data->pe_count;
	i++;

	/* Extent Start */
	info->info[i].name = EngFncs->engine_strdup("Extent_Start");
	info->info[i].title = EngFncs->engine_strdup(_("Extent Start"));
	info->info[i].desc = EngFncs->engine_strdup(_("Starting sector of the data extents area on this object."));
	info->info[i].type = EVMS_Type_Unsigned_Int64;
	info->info[i].value.ui64 = pv_data->pe_start;
	i++;

	/* Index */
	info->info[i].name = EngFncs->engine_strdup("Index");
	info->info[i].title = EngFncs->engine_strdup(_("Object Index"));
	info->info[i].desc = EngFncs->engine_strdup(_("Index of this object in the container metadata."));
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].value.ui32 = pv_data->pv_index;
	i++;

	/* Extent-Map */
	info->info[i].name = EngFncs->engine_strdup("Extent_Map");
	info->info[i].title = EngFncs->engine_strdup(_("Extent Map"));
	info->info[i].type = EVMS_Type_String;
	info->info[i].collection_type = EVMS_Collection_List;
	info->info[i].collection.list = EngFncs->engine_alloc(sizeof(value_list_t) +
							      sizeof(value_t) *
							      (pv_data->pe_count + 1));
	snprintf(buffer, 256, "%-6s : %-10s : %-30s : %s",
		 _("PE #"), _("Sector"), _("Region"), _("PE Count"));
	info->info[i].collection.list->value[l++].s = EngFncs->engine_strdup(buffer);

	for (j = 0; j < pv_data->pe_count; j++) {
		if (pe_map[j].le) {
			for (k = j+1; k < pv_data->pe_count; k++) {
				if (!pe_map[k].le ||
				    (pe_map[j].le->le_map->r_map->r_data !=
				     pe_map[k].le->le_map->r_map->r_data)) {
					break;
				}
			}
			snprintf(buffer, 256, "%-6"PRIu64" : %-10"PRIu64" : %-30s : %"PRIu64,
				 pe_map[j].number,
				 pe_map[j].number * c_data->pe_size + pv_data->pe_start,
				 pe_map[j].le->le_map->r_map->r_data->region->name, k - j);
			info->info[i].collection.list->value[l++].s = EngFncs->engine_strdup(buffer);
			j = k-1;
		}
	}
	info->info[i].collection.list->count = l;
	i++;

	info->count = i;
	*info_array = info;

	LOG_EXIT_INT(0);
	return 0;
}

