/*************************************************************
*  This file is part of the Surface Evolver source code      *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/

/******************************************************
*
*  File: fixvol.c
*
*  Contents: Routines for projecting force on vertices
*        to perpendicular of all body volume gradients.
*        Does constrained quantities also.
*
*/

#include "include.h"

static  int maxquants;  /* total number of constraints */
static  REAL *vpressures = NULL; /* so global pressures not messed */
/* for numbering various types of constraints */
static  int bodystart = 0;
static  int quantity_start;
static  int gen_quant_start;

REAL **vgev = NULL;  /* vector vol grads, for approx curvature */
REAL **vgef = NULL;  /* form vol grads, for approx curvature */

/*************************************************************************
*
*  Function:  calc_volgrads()
*
*  purpose: calculate gradients of constrained volumes and quantities.
*          Leaves them in vgptr... array.
*        
*          Also does variable_parameter constraint gradients.
*/

void calc_volgrads(mode)
int mode; /* DO_OPTS or NO_OPTS */
{
  body_id bi_id;  /* identifier for body i */
  vertex_id v_id;
  int i,k,n;
  struct boundary *bdry;
  int qfixed = 0;
  struct gen_quant *gq;
  MAT2D(a,MAXCOORD,MAXCOORD);

  vgrad_end();  /* in case anything left over from a previous time */


  /* for numbering various types of constraints */
  quantity_start = web.skel[BODY].max_ord + 1;
  gen_quant_start = quantity_start + web.quantity_count;

  /* see if anything needs to be done */
  fixed_constraint_flag = 0;
  if ( !web.pressure_flag && !everything_quantities_flag )
    FOR_ALL_BODIES(bi_id)
     if ( get_battr(bi_id) & (FIXEDVOL|PRESSURE) ) fixed_constraint_flag = 1;
  for ( i = 0 ; i < web.quantity_count ; i++ )
     if ( get_quant(i)->attr & QFIXED ) { fixed_constraint_flag = 1; qfixed++; }
  for ( k = n = 0, gq = GEN_QUANTS ; n < gen_quant_count ; gq++,n++ )
     if ( gq->flags & Q_FIXED )
     { if ( !valid_id(gq->b_id) || !web.pressure_flag ) 
       { fixed_constraint_flag = 1; qfixed++;
         gq->vol_number = gen_quant_start + k++;
       }
       else if ( web.pressure_flag )
       gq->vol_number = gen_quant_start + k++;
     }
  maxquants = gen_quant_start + k;
  if ( !web.pressure_flag && !fixed_constraint_flag ) return;

  /* allocate space to hold vertex body volume gradients */
  vgrad_init(qfixed);

  /* calculate body volume gradients at all control points 
      due to free surfaces */
  if ( !quantities_only_flag )
  { if ( web.representation == SIMPLEX ) 
    simplex_grad_l();
     else if ( web.representation == STRING )
      (*string_grad)();
     else /* web.representation == SOAPFILM */
      (*film_grad)();
  }
  calc_quant_grads(Q_FIXED);

  /* calculate optimizing_parameter gradients by finite differences */
  if ( (mode == DO_OPTS) && (optparamcount > 0) )
  { REAL **convalues; 
     struct oldcoord osaved;

     if ( optparam_congrads ) free_matrix(optparam_congrads);
     optparam_congrads = dmatrix(0,optparamcount,0,maxquants);
     convalues = dmatrix(0,maxquants,0,2);
     osaved.coord = NULL;
     save_coords(&osaved);
     /* save values */
     if ( !web.pressure_flag && !everything_quantities_flag )
      FOR_ALL_BODIES(bi_id)
        if ( get_battr(bi_id) & FIXEDVOL ) 
          convalues[ordinal(bi_id)][0] = get_body_volume(bi_id);
     for ( k = n = 0, gq = GEN_QUANTS ; n < gen_quant_count ; gq++,n++ )
      if ( gq->flags & Q_FIXED )
       if ( !valid_id(gq->b_id) || !web.pressure_flag ) 
         convalues[gq->vol_number][0] = gq->value;

     for ( i = 0 ; i < optparamcount ; i++ )
     { REAL dp;

       dp = globals[optparam[i].pnum].delta;

       /* right difference */
       globals[optparam[i].pnum].value.real += dp;
       project_all(0, TEST_MOVE);
       calc_content(Q_FIXED);
       /* save values */
       if (!web.pressure_flag &&  !everything_quantities_flag )
         FOR_ALL_BODIES(bi_id)
           if ( get_battr(bi_id) & FIXEDVOL ) 
             convalues[ordinal(bi_id)][1] = get_body_volume(bi_id);
       for ( k = n = 0, gq = GEN_QUANTS ; n < gen_quant_count ; gq++,n++ )
        if ( gq->flags & Q_FIXED )
         if ( !valid_id(gq->b_id) || !web.pressure_flag )
           convalues[gq->vol_number][1] = gq->value;
       restore_coords(&osaved);  /* also fixes opt params */

       /* left difference */
       globals[optparam[i].pnum].value.real -= dp;
       project_all(0, TEST_MOVE);
       if ( fixed_constraint_flag || web.pressure_flag || web.pressflag )
       calc_content(Q_FIXED);
       /* save values */
       if (!web.pressure_flag &&  !everything_quantities_flag )
         FOR_ALL_BODIES(bi_id)
           if ( get_battr(bi_id) & FIXEDVOL ) 
             convalues[ordinal(bi_id)][2] = get_body_volume(bi_id);
       for ( k = n = 0, gq = GEN_QUANTS ; n < gen_quant_count ; gq++,n++ )
        if ( gq->flags & Q_FIXED )
         if ( !valid_id(gq->b_id) || !web.pressure_flag )
           convalues[gq->vol_number][2] = gq->value;
       restore_coords(&osaved);  /* also fixes opt params */
   
       /* calculate gradients */
       if (!web.pressure_flag &&  !everything_quantities_flag )
         FOR_ALL_BODIES(bi_id)
           if ( get_battr(bi_id) & FIXEDVOL ) 
             optparam_congrads[i][ordinal(bi_id)] =
             (convalues[ordinal(bi_id)][1]-convalues[ordinal(bi_id)][2])/2/dp;
       for ( k = n = 0, gq = GEN_QUANTS ; n < gen_quant_count ; gq++,n++ )
        if ( gq->flags & Q_FIXED )
         if ( !valid_id(gq->b_id) || !web.pressure_flag )
           optparam_congrads[i][gq->vol_number] =
             (convalues[gq->vol_number][1]-convalues[gq->vol_number][2])/2/dp;
      }
      /* restore values */
      if (!web.pressure_flag &&  !everything_quantities_flag )
      FOR_ALL_BODIES(bi_id)
        if ( get_battr(bi_id) & FIXEDVOL ) 
           set_body_volume(bi_id, convalues[ordinal(bi_id)][0]);
      for ( k = n = 0, gq = GEN_QUANTS ; n < gen_quant_count ; gq++,n++ )
        if ( gq->flags & Q_FIXED )
      if ( !valid_id(gq->b_id) || !web.pressure_flag )
          gq->value = convalues[gq->vol_number][2];
      free_matrix(convalues);
      unsave_coords(&osaved);
  } /* end optimizing parameters */

  /* add on gradients due to constraint integrals */
  if ( !quantities_only_flag )
  { if ( web.representation == STRING )
     { 
       string_constr_grad();
       film_constr_grad();  /* for quantities */
     }
     else /* web.representation == SOAPFILM */
     { 
       film_constr_grad();
     }
  } 
  /* project to parameter space for boundary points */
  FOR_ALL_VERTICES(v_id)
  { REAL  dummy;
    REAL temp[MAXCOORD];
    int pcount,j;
    volgrad *vgptri;

    if ( get_vattr(v_id) & FIXED ) continue;
    if ( !(get_vattr(v_id) & BOUNDARY) ) continue;
    bdry = get_boundary(v_id);
    pcount = bdry->pcount;
    for ( j = 0 ; j < SDIM ; j++ )
    { eval_all(bdry->coordf[j],get_param(v_id),pcount,&dummy,temp,v_id);
      for ( i = 0 ; i < pcount ; i++ )
        a[i][j] = temp[i];
    }

    vgptri = get_vertex_vgrad(v_id);
    while ( vgptri )
      { REAL tmp[MAXCOORD];
         int m;
         matvec_mul(a,vgptri->grad,tmp,pcount,SDIM);
         for ( m = 0 ; m < pcount ; m++ ) vgptri->grad[m] = tmp[m];
         for ( m = pcount ; m < SDIM ; m++ )
           vgptri->grad[m] = 0.0;
         vgptri = vgptri->chain;
      }
  }

} /* end calc_volgrads() */

/**********************************************************************
*
* function: pressure_forces()
*
* purpose: add forces due to dynamic pressure
*/

void pressure_forces()
{ vertex_id v_id;
  int k;
  body_id b_id;
  volgrad *vgptr;
  int to_do = 0;

  if ( everything_quantities_flag ) return; 

  if ( web.pressure_flag )
  {
    /* add forces due to dynamic pressure */
    FOR_ALL_VERTICES(v_id)
    { REAL *f;
     
      if ( get_vattr(v_id) & FIXED ) continue;
      f = get_force(v_id);
      for ( vgptr = get_vertex_vgrad(v_id) ; vgptr ; vgptr = vgptr->chain )
        {
          if ( valid_id(vgptr->bb_id) ) 
          if ( (get_battr(vgptr->bb_id) & FIXEDVOL) ) 
          { REAL p = get_body_pressure(vgptr->bb_id);
             for ( k = 0 ; k < SDIM ; k++ )
                f[k] += (p - web.pressure)*vgptr->grad[k];
          }
        }
     
    }
    return;
  }

  /* add prescribed pressure forces */
  /* first, see if there are any */
  FOR_ALL_BODIES(b_id)
     if ( get_battr(b_id) & PRESSURE ) to_do = 1;
  
  if ( to_do == 0 ) return;

  FOR_ALL_VERTICES(v_id)
  { REAL *f;

    if ( get_vattr(v_id) & FIXED ) continue;
    f = get_force(v_id);
    for ( vgptr = get_vertex_vgrad(v_id) ; vgptr ; vgptr = vgptr->chain )
      {
         if ( valid_id(vgptr->bb_id) )  /* check for real bodies */
         if ( get_battr(vgptr->bb_id) & PRESSURE )
         for ( k = 0 ; k < SDIM ; k++ )
          f[k] += get_body_pressure(vgptr->bb_id)*vgptr->grad[k];
      }

  }
} /* end pressure_forces() */

/************************************************************************
*
* function: one_sided_adjust()
*
* purpose: see which vertices need to have velocity and vgrads
*         restricted by one-sided constraints. Uses old
*         Lagrange multipliers to calculate force, then sees
*         which constraints violated, and projects tangent
*         to violated constraints and regular constraints.
*/

void one_sided_adjust(mode)
int mode; /* CALC_FORCE and/or CALC_VOLGRADS */
{ vertex_id v_id;
  int i,j,k;
  REAL vel[MAXCOORD];
  int flag;

  /* see if we need to do anything */
  if ( !fixed_constraint_flag ) return;
  for ( i = 0, flag = 0 ; i < MAXCON ; i++ )
    { struct constraint *con = get_constraint(i);
      if ( con->attr & (NONNEGATIVE|NONPOSITIVE) )
        { flag = 1;  break; }
    }
  if ( flag == 0 ) return;
  
  if ( !pressure_set_flag ) calc_lagrange(); /* for first time only */

  FOR_ALL_VERTICES(v_id)
  {
    volgrad *vgptr;
    int ord = ordinal(v_id);
    REAL *f;
    conmap_t * conmap = get_v_constraint_map(v_id);
    int oncount = 0;
    struct constraint *con[MAXCONPER];
    REAL *x;
    REAL perp[MAXCOORD];
    REAL fval;
    REAL grad[MAXCOORD];
    REAL fp;

    if ( get_vattr(v_id) & FIXED ) continue;
    if ( !(get_vattr(v_id) & CONSTRAINT) ) continue;
    x = get_coord(v_id);

    f = get_velocity(v_id);
    for ( i = 0 ; i < SDIM ; i++ ) vel[i] = f[i];
    for ( vgptr = get_vertex_vgrad(v_id) ; vgptr ; vgptr = vgptr->chain )
    { REAL p; /* Lagrange multiplier */
      if ( valid_id(vgptr->bb_id) )
      p = get_body_pressure(vgptr->bb_id);
      else  /* for quantities */
      p = GEN_QUANTS[vgptr->qnum].pressure;
      if ( approx_curve_flag )
        for ( k = 0 ; k < SDIM ; k++ )
            vel[k] += p*vgev[vgptr->fixnum][SDIM*ord + k];
      else
        for ( k = 0 ; k < SDIM ; k++ )
            vel[k] += p*vgptr->velocity[k];
    }

    for ( j = 1,oncount = 0 ; j <= (int)conmap[0] ; j++ )
      {
         if ( conmap[j] & CON_HIT_BIT )
         { struct constraint *cc = get_constraint(conmap[j]);
           if ( cc->attr & (NONNEGATIVE | NONPOSITIVE) )
           { /* check for violation */
             eval_all(cc->formula,x,SDIM,&fval,grad,v_id);
             fp = SDIM_dot(vel,grad);
             if ( (cc->attr & NONNEGATIVE) && (fp < 0.0) )
                con[oncount++] = cc;
             else if ( (cc->attr & NONPOSITIVE) && (fp > 0.0) )
                con[oncount++] = cc;
           }
         }
      }
    if ( mode & CALC_VOLGRADS )
    for ( vgptr = get_vertex_vgrad(v_id) ; vgptr ; vgptr = vgptr->chain )
    { constr_proj(TANGPROJ,oncount,con,x,vgptr->velocity,perp,NULL,NO_DETECT,NULLID);
      for ( j = 0 ; j < SDIM ; j++ )
          vgptr->velocity[j] -= perp[j]; 
    }
    if ( mode & CALC_FORCE )
    { constr_proj(TANGPROJ,oncount,con,x,f,perp,NULL,NO_DETECT,NULLID);
      for ( j = 0 ; j < SDIM ; j++ )
          f[j] -= perp[j]; 
    }
  }
}  /* one_sided_adjust() */

/**************************************************************************
*
* function: calc_leftside()
*
* purpose: set up DV^T DV
*/

void calc_leftside()
{ int i,j,k;
  body_id bi_id;
  int bi,bj;
  struct gen_quant *gq;
  vertex_id v_id;

  /* set up matrices for DV^T DV */
  rleftside = dmatrix(0,maxquants,0,maxquants);
  for ( k = 0 ; k <= maxquants ; k++ )
    rleftside[k][k] = 1.0; /* for invertibility */

  if (!web.pressure_flag &&  !everything_quantities_flag )
    FOR_ALL_BODIES(bi_id)
     { bi =  bodystart + ordinal(bi_id);
       if ( !(get_battr(bi_id) & FIXEDVOL) ) continue;
       rleftside[bi][bi] = 0.0;  /* to undo safety move above */
     }

  for ( k = 0 ; k < web.quantity_count ; k++ )
  { struct quantity *quan = get_quant(k);
    if ( !(quan->attr & QFIXED) ) continue;
    bi = quantity_start + k;
    rleftside[bi][bi] = 0.0;  /* to undo safety move above */
  }
  for ( k = 0, gq = GEN_QUANTS ; k < gen_quant_count ; gq++,k++ )
  { 
    if ( !(gq->flags & Q_FIXED) ) continue;
    if ( valid_id(gq->b_id) && web.pressure_flag ) continue;
    bi = gq->vol_number;
    rleftside[bi][bi] = 0.0;  /* to undo safety move above */
  }

  /* generate  DV^T DV */
  if ( !approx_curve_flag )
    FOR_ALL_VERTICES(v_id)
    {
      volgrad *vgptri,*vgptrj;
      ATTR attr = get_vattr(v_id);

      if ( attr & FIXED ) continue;

      for ( vgptri = get_vertex_vgrad(v_id); vgptri ; vgptri = vgptri->chain )
      { REAL tmp;
        bi = vgptri->fixnum;
        if (valid_id(vgptri->bb_id) && !web.pressure_flag && !everything_quantities_flag )
        { if ( !(get_battr(vgptri->bb_id)&FIXEDVOL) ) continue;
        }
        tmp = SDIM_dot(vgptri->velocity,vgptri->grad);
/*      if ( !(attr & HIT_WALL) )    */
        { rleftside[bi][bi] += tmp;
        }
         for ( vgptrj = vgptri->chain ; vgptrj ; vgptrj = vgptrj->chain )
        { tmp = SDIM_dot(vgptri->grad,vgptrj->velocity);
          bj = vgptrj->fixnum;
/*        if ( !(attr & HIT_WALL) ) */
          {
            rleftside[bi][bj] += tmp;
            rleftside[bj][bi] += tmp;
          }
        }
      }
    }

  if ( approx_curve_flag )
  { int NV = SDIM*(1+web.skel[VERTEX].max_ord);

    /*  dot products */
    for ( bi = 0 ; bi <= maxquants ; bi++ )
     { rleftside[bi][bi] += dot(vgev[bi],vgef[bi],NV);  /* self product */ 
       for ( bj = bi+1 ; bj <= maxquants ; bj++ ) /* other products */
          { REAL tmp = dot(vgev[bi],vgef[bj],NV);
            rleftside[bi][bj] += tmp;
            rleftside[bj][bi] += tmp;
          }
     }

  }  /* end approx_curve_flag */

  /* variable_parameter contributions */
  if ( optparamcount )
  for ( i = 0 ; i < maxquants ; i++ )
    for ( j = 0 ; j < maxquants ; j++ )
      for ( k = 0 ; k < optparamcount ; k++ )
    rleftside[i][j] += optparam_congrads[k][i]*globals[optparam[k].pnum].scale
                            *optparam_congrads[k][j];

  /* solve for coefficients */
  if ( web.full_flag )    /* to prevent singular matrix */
  { bi = -1;
    FOR_ALL_BODIES(bi_id) /* find a fixed one */
     { bi =  bodystart + ordinal(bi_id);
       if ( !(get_battr(bi_id) & FIXEDVOL) ) continue;
       rleftside[bi][bi] = 1.0;
       for ( i = 0 ; i <= maxquants ; i++ )
            rleftside[bi][i] = rleftside[i][bi] = 0.0;
       break;
     }
  }
  mat_inv(rleftside,maxquants+1);
  if ( web.full_flag && (bi>=0) )  rleftside[bi][bi] = 0.0;
} /* end calc_leftside() */

/************************************************************************
*
* function: calc_lagrange()
*
* purpose: calculate Lagrange multipliers
*
*/

void calc_lagrange()
{ int i,k;
  body_id b_id;
  int bi;
  struct gen_quant *gq;
  vertex_id v_id;

  if ( !fixed_constraint_flag ) return;

  calc_leftside();  /* set up rleftside inverse */

  rightside = vector(0,maxquants);
  vpressures = vector(0,maxquants);

  /* generate right side of matrix equation */
  if ( !approx_curve_flag )
    FOR_ALL_VERTICES(v_id)
    {
      volgrad *vgptri;
      ATTR attr = get_vattr(v_id);
      REAL *f;

      if ( attr & FIXED ) continue;
      f = get_velocity(v_id);

      for ( vgptri = get_vertex_vgrad(v_id); vgptri ; vgptri = vgptri->chain )
      { 
        bi = vgptri->fixnum;
        if (valid_id(vgptri->bb_id) &&!web.pressure_flag &&  !everything_quantities_flag )
        { if ( !(get_battr(vgptri->bb_id)&FIXEDVOL) ) continue;
        }
        rightside[bi] += SDIM_dot(f,vgptri->grad);
      }
    }

  if ( approx_curve_flag )
  { REAL *f;

    /* calculate right side */
    FOR_ALL_VERTICES(v_id)
     {
        volgrad *vgptri;
        ATTR attr = get_vattr(v_id);

        if ( attr & FIXED ) continue;

        f = get_velocity(v_id);
        for ( vgptri=get_vertex_vgrad(v_id); vgptri ; vgptri = vgptri->chain )
        {
          bi = vgptri->fixnum;
          if (valid_id(vgptri->bb_id) &&!web.pressure_flag &&  !everything_quantities_flag )
            { if ( !(get_battr(vgptri->bb_id)&FIXEDVOL) ) continue;
            }
          rightside[bi] += SDIM_dot(f,&vgef[bi][SDIM*ordinal(v_id)]);
        }
      }
  }  /* end approx_curve_flag */

  /* optimizing_parameter contributions */
  if ( optparamcount )
  for ( i = 0 ; i < maxquants ; i++ )
     for ( k = 0 ; k < optparamcount ; k++ )
    rightside[i] -= optparam_congrads[k][i]*optparam[k].velocity;
  /* negative since using grad, not force */ 

  /* solve for coefficients */
  matvec_mul(rleftside,rightside,vpressures,maxquants+1,maxquants+1);

  /* install pressures into body structures */
  if ( !web.pressure_flag )
    FOR_ALL_BODIES(b_id)
     {
        if ( get_battr(b_id) & FIXEDVOL )
        {
          if ( everything_quantities_flag )
          { gq = GEN_QUANTS+get_body_volquant(b_id);
            set_body_pressure(b_id,-vpressures[gq->vol_number]);
          }
          else
          {
            bi = bodystart + ordinal(b_id);
            set_body_pressure(b_id,-vpressures[bi]);
          }
        }
     }
    for ( k = 0 ; k < web.quantity_count ; k++ )
      { struct quantity *quan = get_quant(k);
         if ( !(quan->attr & QFIXED) ) continue;
         bi = quantity_start  + k;
         quan->pressure = -vpressures[bi];
      }
    for ( k = 0, gq = GEN_QUANTS ; k < gen_quant_count ; gq++,k++ )
     if ( gq->flags & Q_FIXED )
      if ( !valid_id(gq->b_id) || !web.pressure_flag )
        gq->pressure = -vpressures[gq->vol_number];

} /* end calc_lagrange() */

/*********************************************************************
*
* function: lagrange_adjust()
*
* purpose: adjust forces by Lagrange multipliers of constraint grads.
*
*/

void lagrange_adjust()
{ vertex_id v_id;
  int i,k;
  
  if ( !fixed_constraint_flag ) return;
  /* subtract multiples of volume gradients from force */

  FOR_ALL_VERTICES(v_id)
  {
    volgrad *vgptr;
    int ord = ordinal(v_id);
    REAL *f;
    int bi; /* row number for constraint */

    if ( get_vattr(v_id) & FIXED ) continue;
    f = get_velocity(v_id);
    for ( vgptr = get_vertex_vgrad(v_id) ; vgptr ; vgptr = vgptr->chain )
    { bi = vgptr->fixnum;
      if ( approx_curve_flag )
         for ( k = 0 ; k < SDIM ; k++ )
            f[k] -= vpressures[bi]*vgev[bi][SDIM*ord +k];
      else
         for ( k = 0 ; k < SDIM ; k++ )
            f[k] -= vpressures[bi]*vgptr->velocity[k];
    }
  }

  /* optimizing parameter adjust */
  if ( optparamcount )
    for ( i = 0 ; i < optparamcount ; i++ )
      for ( k = 0 ; k < maxquants ; k++ )
     optparam[i].velocity += vpressures[k]*globals[optparam[i].pnum].scale
                              *optparam_congrads[i][k];

} /* end lagrange_adjust() */
        
/*******************************************************************
*
* function: volume_restore()
*
* purpose: adjust surface to global constraints.
*         Assumes quantities and gradients already calculated.
*
*/

void volume_restore(stepsize,mode)
REAL stepsize; /* multiplier for motion; usually 1 */
int mode;/* TEST_MOVE or ACTUAL_MOVE */
{ int bi;
  body_id bi_id;
  struct gen_quant *gq;
  int i,k;
  vertex_id v_id;

  if ( !fixed_constraint_flag ) return;

  vol_deficit = vector(0,maxquants);
  vol_restore = vector(0,maxquants);

  /* gather differences from targets */
  if ( !web.pressure_flag && !everything_quantities_flag )
    FOR_ALL_BODIES(bi_id)
     {
       bi =  bodystart + ordinal(bi_id);
       if ( !(get_battr(bi_id) & FIXEDVOL) ) continue;
       vol_deficit[bi] = get_body_fixvol(bi_id) - get_body_volume(bi_id);
     }
  for ( k = 0 ; k < web.quantity_count ; k++ )
     { struct quantity *quan = get_quant(k);
       if ( !(quan->attr & QFIXED) ) continue;
       bi = quantity_start + k;
       vol_deficit[bi] = quan->target - quan->value;
     }
  for ( k = 0, gq = GEN_QUANTS ; k < gen_quant_count ; gq++,k++ )
     { 
       if ( !(gq->flags & Q_FIXED) ) continue;
       if ( valid_id(gq->b_id) && web.pressure_flag ) continue;
       bi = gq->vol_number;
       vol_deficit[bi] = gq->target - gq->value;
     }

  /* solve for volume restoration coefficients */
  if ( rleftside == NULL ) calc_leftside();
  matvec_mul(rleftside,vol_deficit,vol_restore,maxquants+1,maxquants+1);
  
  /* subtract multiples of volume gradients from force */
  /* and combine multiples of gradients for restoring motion */

  /* set restoring motion */

  /* optimizing parameter adjust */
  if ( optparamcount )
    for ( i = 0 ; i < optparamcount ; i++ )
     for ( k = 0 ; k < maxquants ; k++ )
      globals[optparam[i].pnum].value.real += 
         vol_restore[k]*optparam_congrads[i][k];

  /* vertices */
  FOR_ALL_VERTICES(v_id)
  { REAL *x;
    volgrad *vgptr;
    int ord = ordinal(v_id);
    int attr = get_vattr(v_id);

    x = get_coord(v_id);
    if ( attr & BOUNDARY )
    { REAL *p = get_param(v_id);
      struct boundary *boundary = get_boundary(v_id);
      if ( !(attr & FIXED) ) 
        for ( vgptr = get_vertex_vgrad(v_id) ; vgptr ; vgptr = vgptr->chain )
        { bi = vgptr->fixnum;
          if ( approx_curve_flag )
          for ( k = 0 ; k < SDIM ; k++ )
             p[k] += stepsize*vol_restore[bi]*vgev[bi][SDIM*ord+k];
          else
          for ( k = 0 ; k < SDIM ; k++ )
             p[k] += stepsize*vol_restore[bi]*vgptr->velocity[k];
        }
      for ( i = 0 ; i < SDIM ; i++ )
        x[i] = eval(boundary->coordf[i],p,v_id);

    }
    else
    { 
      if ( !(attr & FIXED) ) 
      for ( vgptr = get_vertex_vgrad(v_id) ; vgptr ; vgptr = vgptr->chain )
      { bi = vgptr->fixnum;
         if ( approx_curve_flag )
          for ( k = 0 ; k < SDIM ; k++ )
             x[k] += stepsize*vol_restore[bi]*vgev[bi][SDIM*ord+k];
         else
          for ( k = 0 ; k < SDIM ; k++ )
             x[k] += stepsize*vol_restore[bi]*vgptr->velocity[k];
      }
      if ( attr & CONSTRAINT ) project_v_constr(v_id,mode);
    }
  }

  if ( vol_deficit ) myfree((char *)vol_deficit);  vol_deficit = NULL;
  if ( vol_restore ) myfree((char *)vol_restore);  vol_restore = NULL;

} /* end volume_restore() */

  

/*******************************************************************
*
*  function: vgrad_init()
*
*  purpose:  allocates storage for vertex volume gradients
*
*/

volgrad *vgradbase;    /* allocated list of structures */
volgrad **vgptrbase;    /* allocated list of chain start pointers */
int    vgradtop;     /* number of first free  structure */
long      vgradmax;     /* number allocated */

void vgrad_init(qfixed)
int qfixed; /* how many fixed quantities (aside from body volumes)*/
{
  long allocsize;

  allocsize = (long)web.skel[VERTEX].maxcount*sizeof(volgrad  *);
  if ( allocsize < MAXALLOC )
    vgptrbase = (volgrad **)temp_calloc(web.skel[VERTEX].maxcount,
                                    sizeof(volgrad *));
  if ( vgptrbase == NULL )
  {
    sprintf(errmsg,"Cannot allocate %ld volgrad pointers of size %d\n",
      web.skel[VERTEX].maxcount, sizeof(volgrad  *));
    kb_error(1039,errmsg,RECOVERABLE);

  }

  if ( web.representation == SIMPLEX )  /* may be gross overestimate, but safe */
     vgradmax = web.skel[VERTEX].count*(qfixed + web.skel[BODY].count);
  else if ( web.representation == STRING )
  { int crude;
     vgradmax = 2*web.skel[EDGE].count
         + web.skel[BODY].count + qfixed*web.skel[VERTEX].count + 10;
     crude = (web.skel[BODY].count + qfixed)*web.skel[VERTEX].count + 10;
     if ( crude < vgradmax ) vgradmax = crude;
  }
  else /* SOAPFILM */
  { /* could save factor of 2 on facets if could guarantee bodies
     next to an edge are separated by a facet, which we can't
     if we permit incomplete bodies. 
      */
     int crude;  
     vgradmax = web.skel[FACETEDGE].count - web.skel[FACET].count
         + web.skel[BODY].count + qfixed*web.skel[VERTEX].count + 10;
     crude = (web.skel[BODY].count + qfixed)*web.skel[VERTEX].count + 10;
     if ( crude < vgradmax ) vgradmax = crude;
  }

  if ( web.modeltype == QUADRATIC )
     vgradmax += web.skel[FACETEDGE].count;  /* for midpoints */
  else if ( web.modeltype == LAGRANGE )
  { vgradmax += web.skel[FACETEDGE].count*(web.skel[EDGE].ctrlpts-2);  /* for midpoints */
    vgradmax += web.skel[FACET].count*(web.lagrange_order-1)
                *(web.lagrange_order-2)/2;  /* for midpoints */
  }
  allocsize = (long)vgradmax*sizeof(struct volgrad);
  if ( allocsize < MAXALLOC )
     vgradbase = (struct volgrad *)temp_calloc(vgradmax,sizeof(struct volgrad));
  if ( vgradbase == NULL )
  {
    sprintf(errmsg,"Cannot allocate %ld volgrad structures of size %d\n",
      vgradmax,sizeof(struct volgrad));
    vgrad_end();
    kb_error(1040,errmsg,RECOVERABLE);

  }
  
  vgradtop = 0;

} /* end vgrad_init() */

/*********************************************************************
*
* function: vgrad_end()
*
* purpose: Deallocate constraint gradients structures.
*
*/
void vgrad_end()
{
  if ( vgradbase ) { temp_free((char *)vgradbase); vgradbase = NULL; }
  if ( vgptrbase ) { temp_free((char *)vgptrbase); vgptrbase = NULL; }
  vgradmax = 0;
  if ( rleftside ) free_matrix(rleftside);  rleftside = NULL;
  if ( rightside ) myfree((char *)rightside);  rightside = NULL;
  if ( vpressures ) myfree((char *)vpressures); vpressures = NULL;
  if ( optparam_congrads ) free_matrix(optparam_congrads);
  optparam_congrads = NULL;
} /* end vgrad_end() */

/********************************************************************
*
* function: get_vertex_vgrad()
*
* purpose: return pointer to first vgrad structure in vertex's chain
*/

volgrad *get_vertex_vgrad(v_id)
vertex_id v_id;
{
  return vgptrbase ? vgptrbase[ordinal(v_id)] : NULL;
}

/**********************************************************************
*
* function: new_vgrad()
*
* purpose: allocate a new vgrad structure from pool.
*
*/

volgrad  *new_vgrad()
{ struct volgrad *vg;

  if ( vgradtop >= vgradmax ) 
  { /* error */
    vgrad_end();
    sprintf(errmsg,"Internal error: Too many volgrad structures requested: %d\n", vgradtop);
    kb_error(1041,errmsg,RECOVERABLE);
  }

  vg = vgradbase + vgradtop++;

  return vg;
}

/******************************************************************
*
* function: get_bv_vgrad()
*
* purpose: return pointer to vgrad structure of given body at
*         given vertex. NULL if none.
*/
    
volgrad *get_bv_vgrad(fixnum,v_id)
int fixnum; /* which constrained quantity */
vertex_id v_id;
{
  volgrad  *vgptr;

  vgptr = get_vertex_vgrad(v_id);
  while ( vgptr )
    if ( vgptr->fixnum == fixnum ) break;
    else vgptr = vgptr->chain;

  return vgptr;    /* null if not found */
}
    
/******************************************************************
*
* function: get_bv_new_vgrad()
*
* purpose: return pointer to vgrad structure of given body at
*         given vertex. Allocates if none.
*/

volgrad *get_bv_new_vgrad(fixnum,v_id)
int fixnum;
vertex_id v_id;
{
  volgrad  **ptrptr;  /* pointer to pointer so can update if need be */

  ptrptr = vgptrbase + ordinal(v_id);
  while ( *ptrptr )
    if ( (*ptrptr)->fixnum == fixnum ) return *ptrptr;
    else ptrptr = &(*ptrptr)->chain;

  /* need to get new structure */
  *ptrptr = new_vgrad();
  (*ptrptr)->fixnum = fixnum;

  return *ptrptr;
}


/**********************************************************************
*
*  function: approx_curv_calc()
*
*  purpose: converts energy gradients to approximate curvature
*
*/

void approx_curv_calc(mode)
int mode; /* bits for CALC_FORCE and CALC_VOLGRADS */
{
  int j;
  REAL *B;
  vertex_id v_id;

  if ( mode & CALC_FORCE )
  { B  = (REAL *)temp_calloc(SDIM*(web.skel[VERTEX].max_ord+1),
             sizeof(REAL));  

    /* each coordinate gives a right side */
    FOR_ALL_VERTICES(v_id)
      { 
        int vnum = SDIM*ordinal(v_id);
        for ( j = 0 ; j < SDIM ; j++ )
           B[vnum+j] = get_force(v_id)[j];
      }
    mobility_mult(B);
    FOR_ALL_VERTICES(v_id)
      { REAL *vel = get_velocity(v_id);
        int vnum = SDIM*ordinal(v_id);
        for ( j = 0 ; j < SDIM ; j++ )
          vel[j] = B[vnum+j];
      }

     temp_free((char *)B);
  }

  if ( mode & CALC_VOLGRADS )
  /* constraint gradients */
  { int NV = SDIM*(1+web.skel[VERTEX].max_ord);
    int bi;

    vgef = dmatrix(0,maxquants+1,0,NV);
    vgev = dmatrix(0,maxquants+1,0,NV);
    /* load volume gradients */
    FOR_ALL_VERTICES(v_id)
     {
        volgrad *vgptri;
        ATTR attr = get_vattr(v_id);
        int ord = ordinal(v_id);

        if ( attr & FIXED ) continue;

        for ( vgptri=get_vertex_vgrad(v_id); vgptri ; vgptri = vgptri->chain )
        { bi = vgptri->fixnum;
          if (valid_id(vgptri->bb_id) && !web.pressure_flag && !everything_quantities_flag )
            { if ( !(get_battr(vgptri->bb_id)&FIXEDVOL) ) continue;
            }
          for ( j = 0 ; j < SDIM ; j++ )
            vgef[bi][SDIM*ord+j] = vgptri->grad[j];
        }
     }

      /* convert gradients to vectors */
     for ( bi = 0 ; bi <= maxquants ; bi++ )
     { memcpy((char*)vgev[bi],(char*)vgef[bi],NV*sizeof(REAL));
       mobility_mult(vgev[bi]); /* to vector */
     }

  }  

} /* end approx_curv_calc() */

