/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/

/**********************************************************************
*
*  File: userio.c
*
*  Purpose: Handles interaction with user.  All interaction goes
*              through this file for easy porting.
*/

#include "include.h"
#ifdef WIN32
#undef FIXED
#undef DOUBLE
#include <windows.h>
#endif

/***********************************************************************
*
*  Function: outstring()
*
*  Purpose: Prints string for user.  Does not wait for response.
*              Does not add newline.
*/

void outstring(outmsg)
char *outmsg;
{
  if ( quiet_flag || !outmsg ) return;
  if ( broken_pipe_flag )
  { broken_pipe_flag = 0; kb_error(1349,"Broken pipe.\n",RECOVERABLE); }

  if ( outfd == NULL ) outfd = stdout;    /* maybe after error */

  if ( logfile_flag && (outfd==stdout) )
      fprintf(logfilefd,outmsg);

#if defined(MAC_APP) || defined(WIN32S) || defined(MAC_CW)
  if ( outfd == stdout )
     write_to_console(outmsg);
  else
     fputs(outmsg,outfd);
#else
  fputs(outmsg,outfd);
  fflush(outfd);
#endif
}

/***********************************************************************
*
*  Function: erroutstring()
*
*  Purpose: Prints error string for user.  Does not wait for response.
*              Does not add newline.
*/

void erroutstring(outmsg)
char *outmsg;
{
#if defined(MAC_APP) || defined(WIN32S) || defined(MAC_CW)
  write_to_console(outmsg);
#else
  fputs(outmsg,stderr);
  fflush(stderr);
#endif
  if ( logfile_flag )
      fprintf(logfilefd,outmsg);

}

/***********************************************************************
*
* function: my_fgets()
*
* purpose: replacement for fgets() that recognizes various line end
* formats: CR, NL, CR NL, and converts all to NL.  Returned string
* ends with null, included in max size.
*/

char *my_fgets(s,n,stream)
char *s;
int n;  /* max size of string */
FILE *stream;
{ char *p;
  int k;
  for ( k = 0, p = s ; k < n-1 ; k++,p++ )
  { int c;
    c = fgetc(stream);
    if ( (c == EOF) && ( k == 0 ) ) return NULL;
    if ( c == EOF ) { *p = 0; return s; }
    *p = c;
    if ( c == '\n' ) { *(++p) = 0; return s; }
    if ( c == '\r' )
     { *p = '\n'; *(++p) = 0;
        c = fgetc(stream);
        if ( (c != '\n') && (c != EOF) ) ungetc(c, stream); 
        return s; 
     }
  }
  *p = 0;  /* null terminator */
  return s;
}

/***********************************************************************
*
*  Function: getstring()
*
*  Purpose: Gets string from user. 
*/

void getstring(inmsg,max)
char *inmsg;
int max;
{
  while ( commandfd && (commandfd != stdin) )
  { /* from command input file */
    if (my_fgets(inmsg,max,commandfd) == NULL) 
    { pop_commandfd();
      continue;  /* will fall thru to stdin eventually */
    }
    else 
    {
      outstring(inmsg); /* echo */
      if ( (int)strlen(inmsg) == max-1 ) 
        inmsg[max-1] = MOREIN; 
      return;
    }
  }

  /* from stdin */
  broken_pipe_flag = 0; /* in case left over */
#ifdef NeXTappX
  if ( tty_flag )
    { 
        fgets(inmsg,max,stdin);
    }
  else
    { NeXT_prompt(NULL,inmsg);
      outstring(inmsg);
    }
#else 
#if defined(MAC_APP) || defined(WIN32S) || defined(MAC_CW)
  read_line_from_console(inmsg);
#else
  if ( gets(inmsg) == NULL )
     my_exit(0);
  if ( echo_flag ) { outstring(inmsg);  outstring("\n"); }
#endif
#endif
}

#ifdef OOGL
/* Geomview picking stuff */
/***************************************************************************
*
* function: read_pick()
*
* purpose: Read pick message from geomview. 
*
* return: 0 for no pick, 1 for pick
*/
int read_pick()
{
  char pickbuf[500],*ptr;
  int len;
  int vnum=-1,ednum=-1,fnum=-1;
  int prim;

  len = read(gv_pipe[0],pickbuf,sizeof(pickbuf));
  if ( len > 0 )
   { 
     pickbuf[len] = 0;
     ptr = strchr(pickbuf+1,'('); /* coord of picked pt */
     if ( !ptr ) goto pick_fail_exit;
     ptr = strchr(ptr+1,'('); /* coord of picked vertex */
     if ( !ptr ) goto pick_fail_exit;
     ptr = strchr(ptr+1,'('); /* coord of picked edge */
     if ( !ptr ) goto pick_fail_exit;
     ptr = strchr(ptr+1,'('); /* coord of picked face */
     if ( !ptr ) goto pick_fail_exit;
     ptr = strchr(ptr+1,'('); /* path to picked primitive */
     if ( !ptr ) goto pick_fail_exit;
     prim = atoi(ptr+1);
     if ( setting_backcull ) prim--;
     ptr = strchr(ptr+1,')')+1; /* index of picked primitive */
     if ( !ptr ) goto pick_fail_exit;
     vnum = atoi(ptr);
     ptr = strchr(ptr+1,'('); /* endpoints of picked edge */
     if ( !ptr ) goto pick_fail_exit;
     if ( ptr[1] != ')' && (web.representation != SIMPLEX) )
     { /* figure edge from endpoints */
        int v1,v2;
        vertex_id v_id1,v_id2;
        edge_id e_id,e1;

        if ( vedge_timestamp < top_timestamp ) make_vedge_lists();
        sscanf(ptr+1,"%d %d",&v1,&v2);
        if ( prim > 0 ) 
        { v_id1 = vpicklist[gv_vect_start+v1]; 
           v_id2 = vpicklist[gv_vect_start+v2]; 
        }
        else { v_id1 = vpicklist[v1]; v_id2 = vpicklist[v2]; }
        if ( valid_id(v_id1) && valid_id(v_id2) &&
               !(get_vattr(v_id1) & Q_MIDFACET)
                    && !(get_vattr(v_id2) & Q_MIDFACET) )
        {
           if ( web.modeltype != LINEAR )
           { edge_id e2,e_id2;
             e1 = e_id = get_vertex_edge(v_id1);
             do 
             {
                e2 = e_id2 = get_vertex_edge(v_id2);
                do
                { 
                   if ( equal_element(e_id,e_id2) )
                   { ednum = ordinal(e_id) + 1; break; }
                   if ( get_vattr(v_id2) & (Q_MIDPOINT | Q_MIDEDGE) )
                    break;
                   e_id2 = get_next_tail_edge(e_id2);
                } while ( !equal_id(e2,e_id2) );

                if ( get_vattr(v_id1) & (Q_MIDPOINT | Q_MIDEDGE) )
                   break;
                e_id = get_next_tail_edge(e_id);
             } while ( (ednum < 0) &&  !equal_id(e1,e_id) );
           }
           else  /* LINEAR */
           { e1 = e_id = get_vertex_edge(v_id1);
             do 
             { if ( equal_id(v_id2,get_edge_headv(e_id)) )
                { ednum = ordinal(e_id) + 1; break; }
                e_id = get_next_tail_edge(e_id);
             } while ( !equal_id(e1,e_id) );
           }
        }
     }
     ptr = strchr(ptr+1,')')+1; /* index of picked face */
     if ( !ptr ) goto pick_fail_exit;
     fnum = atoi(ptr);
     /* puts(pickbuf); */
     strcpy(msg,"Picked ");
     if ( vnum >= 0 )
     { if ( prim > 0 ) vnum += gv_vect_start;
       pickvnum = ordinal(vpicklist[vnum]) + 1;
       sprintf(msg+strlen(msg),"vertex %d  ",pickvnum);
       if ( geomview_bug_flag == 1 )
       { strcat(msg,
        "\nGeomview version 1.6.1 before 1.6.1p7 does not do picking correctly.\n");
          strcat(msg,
               "Get fixed version of gvx from ftp://www.susqu.edu   /priv/slevy.\n");
         geomview_bug_flag = 2;
       }
       if ( geomview_bug_flag == 2 )
          strcat(msg," (probably wrong)");
     } 
     if ( ednum > 0 )
     { pickenum = ednum;
       sprintf(msg+strlen(msg),"edge %d  ",pickenum);
       if ( geomview_bug_flag == 1 )
       { strcat(msg,
        "\nGeomview version 1.6.1 before 1.6.1p7 does not do picking correctly.\n");
         strcat(msg,
         "Get fixed version of gvx from ftp://www.susqu.edu   /priv/slevy.\n");
         geomview_bug_flag = 2;
       }
       if ( geomview_bug_flag == 2 )
         strcat(msg," (probably wrong)");
     } 
     if ( fnum >= 0 )
     { pickfnum = ordinal(fpicklist[fnum]) + 1;
        sprintf(msg+strlen(msg),"facet %d  ",pickfnum);
     } 
     strcat(msg,"\n");
     outstring(msg);
   }
  return (len > 0) ? 1 : 0;

pick_fail_exit:
  outstring("\nMessage from geomview:\n");
  outstring(pickbuf);
  if ( current_prompt ) outstring(current_prompt);
  return 0;
}

/***************************************************************************
*
* function: check_pick()
*
* purpose: Check for pick message from geomview.  Waits for input on
*             either stdin or gv_pipe.
*
* return: 0 for stdin, 1 for pick
*/
#ifdef POLLIN
extern int poll ARGS((struct pollfd *, unsigned long, int));
#endif

int check_pick()
{ 
#ifdef POLLIN
  struct pollfd pp[2];

  /* see if anything ready to read */
  pp[0].fd = 0; /* stdin */
  pp[0].events = POLLIN;
  pp[1].fd = gv_pipe[0]; /* geomview pick */
  pp[1].events = POLLIN;
  poll(pp,2,-1);
  if ( pp[1].revents )  return read_pick();
#else
/* try using select() from time.h */
  fd_set fdset;
  struct timeval timeout;

  timeout.tv_sec = 5; timeout.tv_usec = 0;
  FD_ZERO(&fdset);
  do
  { FD_SET(0,&fdset);
     FD_SET(gv_pipe[0],&fdset);
  } while (  select(FD_SETSIZE,&fdset,NULL,NULL,&timeout) == 0 );
  if ( FD_ISSET(gv_pipe[0],&fdset) ) return read_pick();
  else return 0;
#endif
  return 0;
}
#endif

/***********************************************************************
*
*  Function: prompt()
*
*  Purpose: Displays prompt and gets response from user.
*/

int prompt(promptmsg,inmsg,max)
char *promptmsg; /* to user */
char *inmsg;    /* from user, space already allocated */
int max; /* max char, including terminal null */
{ char *c,*ptr;

  inmsg[0] = 0;  /* in case no message */

  if ( commandfd && (commandfd != stdin) )
  { /* from command input file */
    if (my_fgets(inmsg,max,commandfd) == NULL) 
    { return EOF; }
    else 
    {
      outstring(promptmsg);  /* so appears on screen also */
      outstring(inmsg); /* echo */
      if ( (c = strchr(inmsg,'\n')) != NULL ) 
        {}
      else if ( (int)strlen(inmsg) == max-1 ) 
        inmsg[max-1] = MOREIN; 
      return 1; /* not EOF */         
    }
  }

  breakflag = 0;  /* back to user input */
  current_prompt = promptmsg; /* for those who want to redisplay prompt */

#ifdef NeXTappX
  /* from attention panel */
  outstring(promptmsg);
  NeXT_prompt(promptmsg,inmsg);
#else
#if defined(MAC_APP) || defined(WIN32S) || defined(MAC_CW)
  write_to_console(promptmsg);
  read_line_from_console(inmsg);
#else
  /* from stdin */
  outstring(promptmsg);
#ifdef OOGL
  /* check for geomview pick */
  if ( geomview_flag && !geompipe_flag )
    while ( check_pick() == 1 ) 
        outstring(promptmsg);
#endif
#ifdef WIN32
  signal(SIGINT,SIG_IGN); /* no interrupt during input (esp. WIN32) */
#endif
  if ( gets(inmsg) == NULL )
     return EOF;
  if ( echo_flag ) { outstring(inmsg); outstring("\n"); }
  signal(SIGINT,catcher);
#endif
#endif

  current_prompt = NULL;
  /* strip whitespace from start of inmsg */
  ptr = inmsg;
  while ( (*ptr==' ') || (*ptr=='\t') ) ptr++;
  if ( ptr != inmsg )
  { for ( c = inmsg ; *ptr ; ) *(c++) = *(ptr++);
     *c = 0;
  }
  if ( logfile_flag ) fprintf(logfilefd,"%s\n",inmsg);

  return 1; /* not EOF */
}


/***********************************************************************
*
*  function: push_datafd()
*
*  purpose:  push new datafile include file on stack.
*/

void push_datafd(cfd,name)
FILE *cfd; /* NULL if not opened yet */
char *name;  /* name of file */
{
  if ( include_depth >= NESTDEPTH-1 )
     kb_error(1352,"INCLUDEs nested too deeply.\n",DATAFILE_ERROR);

  if ( cfd == NULL )
     cfd = path_open(name);
  if ( cfd == NULL )
  { sprintf(errmsg,"Cannot open %s.",name);
     kb_error(1353,errmsg,DATAFILE_ERROR);

     return;
  }
  if ( include_depth > 0 )
      datafile_stack[include_depth-1].line = line_no;
  line_no = 1;
  strncpy(datafile_stack[include_depth].filename,name,PATHSIZE-1);
  datafile_stack[include_depth].line = 0;
  datafile_stack[include_depth++].fd = data_fd = cfd;
  yylex_init();  /* reset lex */
}

/***********************************************************************
*
*  function: pop_datafd()
*
*  purpose:  Exit out of one level of datafile include nest.
*/

void pop_datafd()
{
  if ( include_depth <= 1 )
     { include_depth = 0; data_fd = NULL;}
  else fclose(data_fd);
  if ( include_depth > 0 ) 
    { data_fd = datafile_stack[--include_depth-1].fd;
      line_no = datafile_stack[include_depth-1].line;
    } 
  else data_fd = NULL;
}

/***********************************************************************
*
*  function: push_commandfd()
*
*  purpose:  push new command file on stack.
*/

void push_commandfd(cfd,name)
FILE *cfd; /* NULL if not opened yet */
char *name;  /* name of file */
{
  if ( (cfd==NULL) && (name==NULL) ) return;
  if ( read_depth >= NESTDEPTH-1 )
     kb_error(1354,"READs nested too deeply.\n",RECOVERABLE);

  if ( cfd == NULL )
     cfd = path_open(name);
  if ( cfd == NULL )
  { sprintf(errmsg,"Cannot open %s.",name);
     kb_error(1355,errmsg,RECOVERABLE);

  }
  if ( (cfd == stdin) && (read_depth == 1) ) read_depth = 0;

  strncpy(cmdfile_stack[read_depth].filename,name,PATHSIZE-1);
  cmdfile_stack[read_depth].datafile_flag = datafile_flag;
  cmdfile_stack[read_depth].line = line_no;
  line_no = 1;
  cmdfile_stack[read_depth].fd = commandfd = cfd;
  read_depth++;
  yylex_init();  /* reset lex */
}

/***********************************************************************
*
*  function: pop_commandfd()
*
*  purpose:  Exit out of one level of command file nest.
*/

void pop_commandfd()
{
  if ( in_comment )
          { kb_error(3857,"End of file in comment\n",WARNING );  }
#ifdef WIN32
  if ( (commandfd == stdin) && !echo_flag ) return; /* Ctrl-C gives spurious EOF */
#endif
  if ( datafile_flag & !cmdfile_stack[read_depth-1].datafile_flag )
     { datafile_flag = 0;
        return;  /* lex needs to keep reading EOF for datafile */
     }
  if ( read_depth <= 1 )
     { read_depth = 0; commandfd = NULL;}
  else fclose(commandfd);
  if ( read_depth > 0 ) 
  {  commandfd = cmdfile_stack[--read_depth-1].fd;
      line_no = cmdfile_stack[read_depth].line;
  }
  if ( read_depth <= 1 ) cmdfilename = NULL;
}

/***********************************************************************
*
*  Purpose: Error handling.  Prints error message.  If error is
*     recoverable, longjmps to pre-set spot.  If not, exits.
*
*  May be system dependent for message display.
*/

void kb_error(errnum,emsg,mode)
int errnum; /* error identifier */
char *emsg; /* might be msg or errmsg */
int mode;
{
  extern int line_no;
  char *fullmsg;
  int size;

 
  if ( emsg == NULL ) emsg = "";  /* just in case */
  if ( emsg == errmsg ) { fullmsg = msg; size = msgmax; }
  else { fullmsg = errmsg; size = sizeof(errmsg); }

  last_error = errnum;

  if ( read_depth > 1 ) { sprintf(fullmsg,"%s Line %d - ",
      cmdfile_stack[read_depth-1].filename,line_no); }
  else fullmsg[0] = 0;

  reading_comp_quant_flag = 0;  /* just in case */

  switch ( mode )
  {
     case UNRECOVERABLE:
        sprintf(fullmsg+strlen(fullmsg),"FATAL ERROR %d: ",errnum);
        strncat(fullmsg,emsg,size); strncat(fullmsg,"\n",size);
        if ( datafile_flag )
          dump_buff(fullmsg+strlen(fullmsg),size-strlen(fullmsg));
        erroutstring(fullmsg);
        my_exit(1);  

     case RECOVERABLE:
        sprintf(fullmsg+strlen(fullmsg),"ERROR %d: ",errnum);
        strncat(fullmsg,emsg,size); strncat(fullmsg,"\n",size);
        /* pop stack of command files */
        if ( read_depth > 0 ) cmdfile_stack[read_depth-1].line = line_no;
        while ( commandfd && (commandfd  != stdin) )
          {
/*
            sprintf(fullmsg+strlen(fullmsg),"file %s at line %d\n",
                  cmdfile_stack[read_depth-1].filename,
                  cmdfile_stack[read_depth-1].line);
*/
             pop_commandfd();
          }
        erroutstring(fullmsg);
        goto bailout;

     case RECOVERABLE_QUIET:
        /* pop stack of command files */
        while ( commandfd && (commandfd  != stdin) )
             pop_commandfd();
        goto bailout;

     case Q_ERROR:
        strcat(fullmsg,emsg); 
        /* pop stack of command files */
        if ( read_depth > 0 ) cmdfile_stack[read_depth-1].line = line_no;
        while ( commandfd && (commandfd  != stdin) )
          { sprintf(fullmsg+strlen(fullmsg),"file %s at line %d\n",
                  cmdfile_stack[read_depth-1].filename,
                  cmdfile_stack[read_depth-1].line);             
             if ( datafile_flag )
          {  parse_error_flag = 1;
                 recovery_flag = 1; 
             erroutstring(fullmsg);
                 if ( exit_after_error ) my_exit(1);
                 return;
          }
          else 
              pop_commandfd();
          }
        erroutstring(fullmsg);
        goto bailout;

     case WARNING:
        sprintf(fullmsg+strlen(fullmsg),"WARNING %d: ",errnum);
        strncat(fullmsg,emsg,size); strncat(fullmsg,"\n",size);
        erroutstring(fullmsg);
        if ( exit_after_warning ) my_exit(1);
        break;

    case EXPRESSION_ERROR:
        sprintf(fullmsg+strlen(fullmsg),"SYNTAX ERROR %d: ",errnum);
        strncat(fullmsg,emsg,size); strncat(fullmsg,"\n",size);
        dump_buff(fullmsg+strlen(fullmsg),size-strlen(fullmsg));
        erroutstring(fullmsg);
        if ( ++parse_errors >= 5 ) 
          {
             erroutstring("Too many errors. Aborting.\n");
             /* pop stack of command files */
             while ( commandfd && (commandfd  != stdin) ) pop_commandfd();
             goto bailout;
          }
        parse_error_flag = 1;
        recovery_flag = 1;
        if ( exit_after_error ) my_exit(1);
        break;

     case SYNTAX_ERROR:
        strncat(fullmsg,emsg,size); strncat(fullmsg,"\n",size);
        dump_buff(fullmsg+strlen(fullmsg),size-strlen(fullmsg));
        web.global_count = old_global_count;
        /* pop stack of command files */
        if ( commandfd && (commandfd  != stdin) ) pop_commandfd();
        while ( commandfd && (commandfd  != stdin) )
          { char c[100];
             sprintf(c,"called from file %s at line %d\n",
                  cmdfile_stack[read_depth-1].filename,
                  cmdfile_stack[read_depth-1].line);
             strncat(fullmsg,c,size-strlen(fullmsg));
             pop_commandfd();
          }
        erroutstring(fullmsg);
        const_expr_flag = 0;
        if ( exit_after_error ) my_exit(1);
        break; /* return to parser for its message */

     case COMMAND_ERROR:
        sprintf(fullmsg+strlen(fullmsg),"SYNTAX ERROR %d: ",errnum);
        strncat(fullmsg,emsg,size); strncat(fullmsg,"\n",size);
        dump_buff(fullmsg+strlen(fullmsg),size-strlen(fullmsg));
        parse_error_flag = 1;
        web.global_count = old_global_count;
        /* pop stack of command files */
        if ( read_depth > 0 ) cmdfile_stack[read_depth-1].line = line_no;
        while ( commandfd && (commandfd  != stdin) )
          {
/*
             char stuff[200];
             sprintf(stuff,"file %s at line %d\n",
                  cmdfile_stack[read_depth-1].filename,
                  cmdfile_stack[read_depth-1].line);
                 strncat(fullmsg,stuff,size); 
*/
             pop_commandfd();
          }
        erroutstring(fullmsg);
        temp_free_all();
        if ( exit_after_error ) my_exit(1);
        FPRESET;
        longjmp(cmdbuf,1);
        break;

     case PARSE_ERROR:
        sprintf(fullmsg+strlen(fullmsg),"SYNTAX ERROR %d: ",errnum);
        strncat(fullmsg,emsg,size); strncat(fullmsg,"\n",size);
        dump_buff(fullmsg+strlen(fullmsg),size-strlen(fullmsg));
        erroutstring(fullmsg);
        if ( ++parse_errors >= 5 ) 
          {
             erroutstring("Too many errors in datafile. Aborting.\n");
             /* pop stack of command files */
             while ( commandfd && (commandfd  != stdin) ) pop_commandfd();
             goto bailout;
          }
        parse_error_flag = 1;
        const_expr_flag = 0;
        recovery_flag = 1;
        if ( exit_after_error ) my_exit(1);
        break;
        
     case DATAFILE_ERROR:
        sprintf(fullmsg+strlen(fullmsg),"DATAFILE ERROR %d: ",errnum);
        strncat(fullmsg,emsg,size); strncat(fullmsg,"\n",size);
        dump_buff(fullmsg+strlen(fullmsg),size-strlen(fullmsg));
        erroutstring(fullmsg);
        if ( ++parse_errors >= 5 ) 
          {
             erroutstring("Too many errors in datafile. Aborting.\n");
             /* pop stack of command files */
             while ( commandfd && (commandfd  != stdin) ) pop_commandfd();
             goto bailout;
          }
        parse_error_flag = 1;
        recovery_flag = 1;
        if ( exit_after_error ) my_exit(1);
        break;
     }

     return;

bailout:
  if ( outfd != stdout )
    { if ( outfd != NULL )
      {
         fclose(outfd);
      }
      outfd = stdout;    /* in case of previous piping */
    }
  breakflag = 0; iterate_flag = 0;
  if ( saved.coord ) 
     { restore_coords(&saved); unsave_coords(&saved); recalc();}
  clear_symtable();     /* clear symbol table */
  temp_free_all();     
  if ( list && (list != permlist) ) { myfree((char*)list); list = NULL; } /* plug memory leak */
  if ( exit_after_error ) my_exit(1);
  FPRESET;
#ifdef MSC
  if ( draw_thread_id == GetCurrentThreadId() )
      longjmp(graphjumpbuf,1);
#endif
  longjmp(jumpbuf,1);
}
 
/**************************************************************************
*
* function: start_logfile()
*
* purpose: open logfile and start logging user input and screen output
*
* arguments: char * filename : Name of file to open, append mode.
*                                        If null, use logfilename[].
*
*/

void start_logfile(filename)
char *filename;
{
  char *name = filename ? filename : logfilename;

  if ( name[0] == 0 )
     kb_error(2002,"Missing log file name.\n",RECOVERABLE);

  stop_logfile();

  logfilefd = fopen(name,"a");
  if ( !logfilefd ) { perror(name); return; }

  logfile_flag = 1;
  if ( filename ) strncpy(logfilename,filename,sizeof(logfilename)-1);

}

/***************************************************************************
*
* function: stop_logfile()
*
* purpose: end logging input and output.
*/

void stop_logfile()
{
  if ( logfilefd ) fclose(logfilefd);
  logfilefd = NULL;
  logfile_flag = 0;
}
