/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1999 */
/* See the file NOTICE for conditions of use and distribution. */

/* Many thanks to Stuart Lynne for contributing the original code for this
driver. */


#include "../exim.h"
#include "ldap.h"


/* We can't just compile this code and allow the library mechanism to omit the
functions if they are not wanted, because we need to have the LDAP headers
available for compiling. Therefore, compile these functions only if LOOKUP_LDAP
is defined. However, some compilers don't like compiling empty modules, so keep
them happy with a dummy when skipping the rest. Make it reference itself to
stop picky compilers complaining that it is unused, and put in a dummy argument
to stop even pickier compilers complaining about infinite loops. */

#ifndef LOOKUP_LDAP
static void dummy(int x) { dummy(x-1); }
#else


/* Include LDAP headers */

#include <lber.h>
#include <ldap.h>


/* Annoyingly, the different LDAP libraries handle errors in different ways.
There doesn't seem to be an automatic way of distinguishing between them. Check
if the user set LDAP_LIB_TYPE (which causes a specific macro to be set), and if
not set, use the original heuristic that distinguishes UMich LDAP from the rest
and guess Netscape if not UMich. */

#ifndef LDAP_LIB_DEFINED
  #ifdef LDAP_OPT_SIZELIMIT        /* UMich does not have this */
  #define LDAP_NETSCAPE
  #endif
#endif


/* Structure and anchor for caching connections. */

typedef struct ldap_connection {
  struct ldap_connection *next;
  char *host;
  char *user;
  char *password;
  BOOL bound;
  int   port;
  LDAP *ld;
} LDAP_CONNECTION;

static LDAP_CONNECTION *ldap_connections = NULL;



/*************************************************
*         Internal search function               *
*************************************************/

/* This is the function that actually does the work. It is called (indirectly
via control_ldap_search) from both eldap_find() and eldapm_find(), with a
difference in the "single" argument.

Arguments:
  ldap_url      the URL to be looked up
  server        server host name, when URL contains none
  s_port        server port, used when URL contains no name
  single        TRUE if a maximum of 1 result value is permitted
                FALSE otherwise
  res           set to point at the result
  errmsg        set to point a message if result is not OK
  defer_break   set TRUE if no more servers to be tried after a DEFER
  user          user name for authentication, or NULL
  password      password for authentication, or NULL
  sizelimit     max number of entries returned, or -1 for no limit
  timelimit     max time to wait, or -1 for default

Returns:        OK or FAIL or DEFER
                FAIL is given only if a lookup was performed successfully, but
                returned no data.
*/

static int
perform_ldap_search(char *ldap_url, char *server, int s_port, BOOL single,
  char **res, char **errmsg, BOOL *defer_break, char *user, char *password,
  int sizelimit, int timelimit)
{
LDAPURLDesc  *ludp;
LDAPMessage  *result;
LDAPMessage  *e;
BerElement   *ber;
LDAP_CONNECTION *lcp;

char  *attr;
char  *data = NULL;
char  *host;
char  **attrp;
char  **values;
char  **firstval;
int   attr_count = 0;
int   rc;
int   port;
int   ptr = 0;
int   size = 0;
BOOL  add_newline = FALSE;
BOOL  attribute_found = FALSE;

DEBUG(9)
  debug_printf("perform_ldap_search: ldap URL =\"%s\" server=%s port=%d\n",
    ldap_url, server, s_port);

/* Check if LDAP thinks the URL is a valid LDAP URL */

if (!ldap_is_ldap_url(ldap_url))
  {
  *errmsg = string_sprintf("ldap_is_ldap_url: not an LDAP url \"%s\"\n",
    ldap_url);
  DEBUG(9) debug_printf("%s\n", *errmsg);
  *defer_break = TRUE;
  return DEFER;
  }

/* Parse the URL to get host and port, too bad there isn't an
ldap_url_[init|open] */

if ((rc = ldap_url_parse(ldap_url, &ludp)) != 0)
  {
  *errmsg = string_sprintf("ldap_url_parse: (error %d) parsing \"%s\"\n", rc,
    ldap_url);
  DEBUG(9) debug_printf("%s\n", *errmsg);
  *defer_break = TRUE;
  return DEFER;
  }

/* If the host name is empty, take it from the separate argument, if
one is given. */

if (ludp->lud_host == NULL && server != NULL)
  {
  host = server;
  port = s_port;
  }
else
  {
  host = ludp->lud_host;
  port = ludp->lud_port;
  }

/* Count the attributes; we need this later to tell us how to format results */

for (attrp = ludp->lud_attrs; attrp != NULL && *attrp != NULL; attrp++)
  attr_count++;

/* See if we can find a cached connection to this host. The host name pointer
is set to NULL if no host was given, rather than to the empty string. */

for (lcp = ldap_connections; lcp != NULL; lcp = lcp->next)
  {
  if (lcp->port != port) continue;
  if (lcp->host == NULL)
    {
    if (host == NULL) break;
    }
  else
    {
    if (host != NULL && strcmpic(lcp->host, host) == 0) break;
    }
  }

/* If no cached connection found, we must open a connection to the server. */

if (lcp == NULL)
  {
  LDAP *ld = ldap_open(host, (port != 0)? port : LDAP_PORT);
  if (ld == NULL)
    {
    ldap_free_urldesc(ludp);
    *errmsg = string_sprintf("failed to open connection to LDAP server %s:%d "
      "- %s", host, port, strerror(errno));
    DEBUG(9) debug_printf("%s\n", *errmsg);
    *defer_break = FALSE;
    return DEFER;
    }

  DEBUG(9) debug_printf("Opened connection to LDAP server %s:%d\n", host, port);

  lcp = store_malloc(sizeof(LDAP_CONNECTION));
  lcp->host = (host == NULL)? NULL : string_copy_malloc(host);
  lcp->bound = FALSE;
  lcp->port = port;
  lcp->ld = ld;
  lcp->next = ldap_connections;
  ldap_connections = lcp;
  }

/* Found cached connection */

else
  {
  DEBUG(9) debug_printf("Re-using cached connection to LDAP server %s:%d\n",
    host, port);
  }

/* Finished with the broken-down URL */

ldap_free_urldesc(ludp);

/* Bind with the user/password supplied, or an anonymous bind if these values
are NULL, unless a cached connection is already bound with the same values. */

if (!lcp->bound ||
    (lcp->user == NULL && user != NULL) ||
    (lcp->user != NULL && user == NULL) ||
    (lcp->user != NULL && user != NULL && strcmp(lcp->user, user) != 0) ||
    (lcp->password == NULL && password != NULL) ||
    (lcp->password != NULL && password == NULL) ||
    (lcp->password != NULL && password != NULL &&
      strcmp(lcp->password, password) != 0))
  {
  DEBUG(9) debug_printf("Binding with user=%s password=%s\n", user, password);
  if ((rc = ldap_bind_s(lcp->ld, user, password, LDAP_AUTH_SIMPLE))
       != LDAP_SUCCESS)
    {
    *errmsg = string_sprintf("failed to bind the LDAP connection to server "
      "%s:%d - LDAP error %d: %s", host, port, rc, ldap_err2string(rc));
    DEBUG(9) debug_printf("%s\n", *errmsg);
    *defer_break = FALSE;
    return DEFER;
    }
  lcp->bound = TRUE;
  lcp->user = (user == NULL)? NULL : string_copy_malloc(user);
  lcp->password = (password == NULL)? NULL : string_copy_malloc(password);
  }

/* Before doing the search, set the time and size limits (if given). Here again
the different implementations of LDAP have chosen to do things differently. */

#if defined LDAP_LIB_NETSCAPE
if (sizelimit > 0)
  ldap_set_option(lcp->ld, LDAP_OPT_SIZELIMIT, (void *)&sizelimit);
if (timelimit > 0)
  ldap_set_option(lcp->ld, LDAP_OPT_TIMELIMIT, (void *)&timelimit);
#else
if (sizelimit > 0) lcp->ld->ld_sizelimit = sizelimit;
if (timelimit > 0) lcp->ld->ld_timelimit = timelimit;
#endif

/* Use the LDAP URL search function; no need to do our own parsing. If it
succeeds, wait for results - this will time out if no results are available
in a reasonable time. */

if (ldap_url_search(lcp->ld, ldap_url, 0) == -1 ||
    (rc = ldap_result(lcp->ld, LDAP_RES_ANY, 1, NULL, &result)) == -1)
  {
  char *matched, *error;

  /* Annoyingly, the different implementations of LDAP have gone for different
  methods of handling error codes and generating error messages. */

  #if defined LDAP_LIB_SOLARIS7     /* Solaris 7 LDAP */
    *errmsg = string_sprintf("ldap search failed: %s",
      ldap_err2string(ldap_result2error(lcp->ld, result, 0)));
  #elif defined LDAP_LIB_NETSCAPE   /* Netscape SDK */
    (void)ldap_get_lderrno(lcp->ld, &matched, &error);
    *errmsg = string_sprintf("ldap search failed: %s (%s)", error, matched);
  #else                             /* UMich LDAP */
    matched = lcp->ld->ld_matched;
    error = lcp->ld->ld_error;
    *errmsg = string_sprintf("ldap search failed: %s (%s)", error, matched);
  #endif

  DEBUG(9) debug_printf("%s\n", *errmsg);
  *defer_break = TRUE;
  return DEFER;
  }

if (rc == 0)
  {
  *errmsg = string_sprintf("LDAP search timed out");
  DEBUG(9) debug_printf("%s\n", *errmsg);
  *defer_break = FALSE;
  return DEFER;
  }

/* Check if we have too many entries */

rc = ldap_count_entries(lcp->ld, result);
if (single && rc > 1)
  {
  *errmsg = string_sprintf("LDAP search: too many (%d) results "
    "(filter not specific enough)", rc);
  DEBUG(9) debug_printf("%s\n", *errmsg);
  *defer_break = TRUE;
  return DEFER;
  }

/* Check if we have too few (zero) entries */

if (rc < 1)
  {
  *errmsg = string_sprintf("LDAP search: no results");
  DEBUG(9) debug_printf("%s\n", *errmsg);
  return FAIL;
  }

/* Initialize chunk of store in which to return the answer. */

size = 100;
ptr = 0;
data = store_get(size);

/* Loop through returned entries; we have already checked above for zero
entries. */

for(e = ldap_first_entry(lcp->ld, result);
    e != NULLMSG;
    e = ldap_next_entry(lcp->ld, e))
  {
  BOOL add_comma = FALSE;

  DEBUG(9) debug_printf("LDAP entry loop\n");

  /* Results for multiple entries values are separated by newlines. */

  if (add_newline)
    data = string_cat(data, &size, &ptr, "\n", 1);
  else
    add_newline = TRUE;

  /* Loop through the entry, grabbing attribute values. Multiple attribute
  values are separated by commas. */

  for (attr = ldap_first_attribute(lcp->ld, e, &ber);
       attr != NULL;
       attr = ldap_next_attribute(lcp->ld, e, ber))
    {
    if (attr[0])
      {
      /* Get array of values for this attribute. */

      if ((firstval = values = ldap_get_values(lcp->ld, e, attr)) != NULL)
        {
        while (*values)
          {
          DEBUG(9) debug_printf("LDAP attr loop %s:%s\n", attr, *values);

          if (add_comma)
            data = string_cat(data, &size, &ptr, ", ", 2);
          else
            add_comma = TRUE;

          if (attr_count > 1)
            {
            data = string_cat(data, &size, &ptr, attr, strlen(attr));
            data = string_cat(data, &size, &ptr, "=", 1);
            }

          data = string_cat(data, &size, &ptr, *values, strlen(*values));
          data[ptr] = '\0';
          values++;
          attribute_found = TRUE;
          }
        ldap_value_free(firstval);
        }
      }

#ifdef HAVE_NETSCAPE_LDAP_SDK

    /* Netscape LDAP's attr's are dynamically allocated and need to be freed.
    UMich LDAP stores them in static storage and does not require this. */

    ldap_memfree(attr);
#endif
    }

#if HAVE_NETSCAPE_LDAP_SDK_AND_CHECK_IF_BER_FREE_IS_REALLY_REQUIRED

  /* TODO UMich ldap_next_attribute() calls ber_free().
  Need to check if Netscape LDAP SDK does or does not.... */

  if (ber != NULL) ber_free(ber, 0);
#endif
  }

/* If an entry was found, but it had no attributes, we behave as if no entries
were found. */

if (!attribute_found)
  {
  *errmsg = "LDAP search: found no attributes";
  DEBUG(9) debug_printf("%s", *errmsg);
  return FAIL;
  }

/* Otherwise, it's all worked */

else
  {
  DEBUG(9) debug_printf("LDAP search: returning: %s\n", data);
  *res = data;
  return OK;
  }
}



/*************************************************
*        Internal search control function        *
*************************************************/

/* This function is called from eldap_find() and eldapm_find() with a
difference in the "single" argument. It controls calls to perform_ldap_search()
which actually does the work. We call that repeatedly for certain types of
defer in the case when the URL contains no host name and ldap_default_servers
is set to a list of servers to try. This gives more control than just passing
over a list of hosts to ldap_open() because it handles other kinds of defer as
well as just a failure to open. Note that the URL is defined to contain either
zero or one "hostport" only.

Parameter data in addition to the URL can be passed as preceding text in the
string, as items of the form XXX=yyy. The URL itself can be detected because it
must begin "ldap:///".

Arguments:
  ldap_url    the URL to be looked up, optionally preceded by other parameter
                settings
  single      TRUE if a maximum of 1 result value is permitted; FALSE otherwise
  res         set to point at the result
  errmsg      set to point a message if result is not OK

Returns:      OK or FAIL or DEFER
*/

static int
control_ldap_search(char *ldap_url, BOOL single, char **res, char **errmsg)
{
BOOL defer_break;
int timelimit = -1;
int sizelimit = -1;
char *url = ldap_url;
char *user = NULL;
char *password = NULL;
char *server, *list;
char buffer[512];

/* Until the string begins "ldap://", search for the other parameter settings
that are recognized. They are of the form NAME=VALUE, with the value being
optionally double-quoted. There must still be a space after it, however. */

while (strncmp(url, "ldap://", 7) != 0)
  {
  char *name = url;
  while (*url != 0 && *url != '=') url++;
  if (*url == '=')
    {
    int namelen = ++url - name;
    char *value = string_dequote(&url);
    if (isspace(*url))
      {
      if (strncmpic(name, "USER=", namelen) == 0) user = value;
      else if (strncmpic(name, "PASS=", namelen) == 0) password = value;
      else if (strncmpic(name, "SIZE=", namelen) == 0) sizelimit = atoi(value);
      else if (strncmpic(name, "TIME=", namelen) == 0) timelimit = atoi(value);
      else
        {
        *errmsg =
          string_sprintf("unknown parameter \"%.*s\" precedes LDAP URL",
          namelen, name);
        return DEFER;
        }
      while (isspace(*url)) url++;
      continue;
      }
    }
  *errmsg = "malformed parameter setting precedes LDAP URL";
  return DEFER;
  }

DEBUG(9) debug_printf("LDAP parameters: user=%s pass=%s size=%d time=%d\n",
  user, password, sizelimit, timelimit);

/* No default servers, or URL contains a server name */

if (ldap_default_servers == NULL || strncmpic(ldap_url, "ldap:///", 8) != 0)
  {
  return perform_ldap_search(url, NULL, 0, single, res, errmsg, &defer_break,
    user, password, sizelimit, timelimit);
  }

/* Loop through the default servers until OK or FAIL */

list = ldap_default_servers;
while ((server = string_nextinlist(&list, ':', buffer, sizeof(buffer))) != NULL)
  {
  int rc;
  int port = 0;
  char *colon = strchr(server, ':');
  if (colon != NULL)
    {
    *colon = 0;
    port = atoi(colon+1);
    }
  rc = perform_ldap_search(url, server, port, single, res, errmsg, &defer_break,
    user, password, sizelimit, timelimit);
  if (rc != DEFER || defer_break) return rc;
  }

return DEFER;
}



/*************************************************
*               Find entry point                 *
*************************************************/

/* See local README for interface description. The two different searches are
handled by a common function, with a flag to differentiate between them. The
handle and filename arguments are not used. */

int
eldap_find(void *handle, char *filename, char *ldap_url, int length,
  char **result, char **errmsg)
{
return(control_ldap_search(ldap_url, TRUE, result, errmsg));
}

int
eldapm_find(void *handle, char *filename, char *ldap_url, int length,
  char **result, char **errmsg)
{
return(control_ldap_search(ldap_url, FALSE, result, errmsg));
}



/*************************************************
*              Open entry point                  *
*************************************************/

/* See local README for interface description. */

void *
eldap_open(char *filename, char **errmsg)
{
return (void *)(1);    /* Just return something non-null */
}



/*************************************************
*               Tidy entry point                 *
*************************************************/

/* See local README for interface description. */

void
eldap_tidy(void)
{
LDAP_CONNECTION *lcp = NULL;
while ((lcp = ldap_connections) != NULL)
  {
  DEBUG(9) debug_printf("unbind LDAP connection to %s:%d\n", lcp->host,
    lcp->port);
  ldap_unbind(lcp->ld);
  ldap_connections = lcp->next;
  if (lcp->host != NULL) store_free(lcp->host);
  store_free(lcp);
  }
}



/*************************************************
*               Quote entry point                *
*************************************************/

/* Two levels of quoting have to be done, LDAP quoting, and URL quoting. LDAP
quoting uses a backslash, and requires the quoting of #,+"\<>; for
distinguished names, and *() for filters (and spaces at the start and end
of things). It does no harm to quote when not strictly needed.

For URL quoting, the only characters that need not be quoted are the
alphamerics and !$'()*+-._ - all the others must be hexified and preceded by %.
This includes the backslashes used for LDAP quoting.

Argument: the string to be quoted
Returns:  the processed string
*/

#define LDAP_QUOTE   "#,+\"\\<>;*()"
#define URL_NONQUOTE "!$'()*+-._"

char *
eldap_quote(char *s)
{
register int c;
int count = 0;
char *t = s;
char *quoted;

while ((c = *t++) != 0)
  {
  if (!isalnum(c))
    {
    if (strchr(URL_NONQUOTE, c) == NULL) count += 2;
    if (strchr(LDAP_QUOTE, c) != NULL) count += 3;
    }
  }

if (count == 0) return s;
t = quoted = store_get((int)strlen(s) + count + 1);

while ((c = *s++) != 0)
  {
  if (!isalnum(c))
    {
    if (strchr(LDAP_QUOTE, c) != NULL)
      {
      strncpy(t, "%5C", 3);
      t += 3;
      }
    if (strchr(URL_NONQUOTE, c) == NULL)
      {
      sprintf(t, "%%%02X", c);
      t += 3;
      continue;
      }
    }
  *t++ = c;
  }

*t = 0;
return quoted;
}

#endif  /* LOOKUP_LDAP */

/* End of lookups/ldap.c */
