/*
    ext2.h -- ext2 header
    Copyright (C) 1998, 1999, 2000 Lennert Buytenhek <buytenh@gnu.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef _EXT2_H
#define _EXT2_H

static const char _ext2_h[] = "$Id: ext2.h,v 1.23 2002/12/11 00:05:12 rjwalsh Exp $";

#include <sys/types.h>
#include <endian.h>
#include "tune.h"

#ifndef howmany
#define howmany(x,y)	(((x)+((y)-1))/(y))
#endif

#ifndef max
#define max(a,b)	(((a)>(b))?(a):(b))
#endif

#ifndef min
#define min(a,b)	(((a)<(b))?(a):(b))
#endif

#if __BYTE_ORDER__ == __BIG_ENDIAN
#define le16_to_cpu(a)	((((a) & 0x00ffU) << 8) | (((a) & 0xff00) >> 8))
#define le32_to_cpu(a)	((((a) & 0x000000ffU) << 24)|(((a) & 0x0000ff00U) << 8)\
			|(((a) & 0xff000000U) >> 24)|(((a) & 0x00ff0000U) >> 8))
#define cpu_to_le16(a)	le16_to_cpu(a)
#define cpu_to_le32(a)	le32_to_cpu(a)
#else
#define le16_to_cpu(a)	(a)
#define le32_to_cpu(a)	(a)
#define cpu_to_le16(a)	(a)
#define cpu_to_le32(a)	(a)
#endif
typedef u_int32_t blk_t;

#include "ext2_fs.h"

extern unsigned char _bitmap[8];

struct ext2_buffer_cache
{
	struct ext2_buffer_head	 *cache;
	struct ext2_buffer_head  *heads;
	struct ext2_buffer_head **hash;
	struct ext2_fs		 *fs;

	int			  size;
	int			  numalloc;
	unsigned char		 *buffermem;
};

struct ext2_buffer_head
{
	struct ext2_buffer_head  *next;
	struct ext2_buffer_head  *prev;
	unsigned char		 *data;
	blk_t			  block;

	int			  usecount;
	int			  dirty;

	struct ext2_buffer_cache *bc;
	int			  alloc;
};

struct ext2_dev_ops
{
	void	(*close)(void *cookie);
	blk_t	(*get_size)(void *cookie);
	void	(*read)(void *cookie, void *ptr, blk_t block, blk_t num);
	void	(*set_blocksize)(void *cookie, int logsize);
	void	(*sync)(void *cookie);
	void	(*write)(void *cookie, void *ptr, blk_t block, blk_t num);
	void	(*direct_read)(void *cookie, void *ptr, loff_t offset, size_t numbytes);
	void	(*direct_write)(void *cookie, void *ptr, loff_t offset, size_t numbytes);
	int	(*ioctl)(void *cookie, int ioc, void *ptr);
};

struct ext2_dev_handle
{
	struct ext2_dev_ops	*ops;
	char			*prog;
	void			*cookie;
};

struct ext2_fs
{
	char			*prog;
	struct ext2_dev_handle	*devhandle;

	struct ext2_super_block	 sb;
	struct ext2_group_desc	*gd;
	struct ext2_buffer_cache *bc;
	int			 metadirty;	/* See EXT2_META defines */
	int			 flags;		/* See FL_* defines */

	int			 blocksize;
	int			 u32perblock;	/* #blocks per indirect block */
	int			 logsize;	/* base 2 log of blocksize */
	blk_t			 newblocks;	/* blocks in resized fs */
	int			 numgroups;	/* groups in current fs */
	int			 newgroups;	/* groups in resized fs */
	blk_t			 gdblocks;	/* group descriptor blocks */
	blk_t			 resgdblocks;	/* reserved GD blocks */
	blk_t			 newgdblocks;	/* GD blocks in resized fs */
	int			 itoffset;	/* offset of inode table */
	blk_t			 inodeblocks;	/* blocks in each inode table */
	int			 r_frac;	/* percentage reserved blocks */
	int			 stride;	/* per-group bitmap offset */
	struct ext2_inode	 resize;

	unsigned char		*relocator_pool;
	unsigned char		*relocator_pool_end;

	void			*journal;
};


#define EXT2_ACTION_ADD		1
#define EXT2_ACTION_DELETE	2
#define EXT2_ACTION_FIND	3

#define EXT2_META_CLEAN		0
#define EXT2_META_PRIMARY_SB	1
#define EXT2_META_BACKUP_SB	2
#define EXT2_META_PRIMARY_GD	4
#define EXT2_META_BACKUP_GD	8

#define EXT2_META_PRIMARY	(EXT2_META_PRIMARY_SB | EXT2_META_PRIMARY_GD)
#define EXT2_META_BACKUP	(EXT2_META_BACKUP_SB | EXT2_META_BACKUP_GD)
#define EXT2_META_SB		(EXT2_META_PRIMARY_SB | EXT2_META_BACKUP_SB)
#define EXT2_META_GD		(EXT2_META_PRIMARY_GD | EXT2_META_BACKUP_GD)

#define FL_VERBOSE		0x0001
#define FL_DEBUG		0x0002
#define FL_VERSION		0x0004
#define FL_KB_BLOCKS		0x0008
#define FL_QUIET		0x0010
#define FL_SAFE			0x0020
#define FL_FORCE		0x0040
#define FL_SPARSE		0x0100
#define FL_ONLINE		0x0200
#define FL_PREPARE		0x0400
#define FL_IOCTL		0x0800
#define FL_COMPLETION		0x1000

/* generic stuff */
void ext2_print_version(FILE *outfile, char *prog);
void ext2_copy_block(struct ext2_fs *fs, blk_t from, blk_t to);
void ext2_close(struct ext2_fs *fs);
void ext2_commit_metadata(struct ext2_fs *fs, int copies);
blk_t ext2_find_free_block(struct ext2_fs *fs);
ino_t ext2_find_free_inode(struct ext2_fs *fs);
off_t ext2_get_inode_offset(struct ext2_fs *fs, ino_t inode, blk_t *block);
int ext2_get_inode_state(struct ext2_fs *fs, ino_t inode);
int ext2_bg_has_super(struct ext2_fs *fs, int group);
unsigned int ext2_list_backups(struct ext2_fs *fs, unsigned int *three,
			       unsigned int *five, unsigned int *seven);
void ext2_move_blocks(struct ext2_fs *fs, blk_t src, blk_t num, blk_t dest);
struct ext2_fs *ext2_open(struct ext2_dev_handle *handle, blk_t newblocks,
			  int kb_blocks);
void ext2_read_blocks(struct ext2_fs *fs, void *ptr, blk_t block,
		      blk_t numblocks);
void ext2_read_inode(struct ext2_fs *fs, ino_t inode,struct ext2_inode *inodep);
void ext2_set_inode_state(struct ext2_fs *fs, ino_t inode, int state,
			  int updatemetadata);
int ext2_block_iterate(struct ext2_fs *fs, struct ext2_inode *inode,
		       blk_t block, int action);
void ext2_sync(struct ext2_fs *fs);
void ext2_write_blocks(struct ext2_fs *fs, void *ptr, blk_t block,
		       blk_t numblocks);
void ext2_write_inode(struct ext2_fs *fs, ino_t inode,
		      const struct ext2_inode *inodep);
void ext2_zero_blocks(struct ext2_fs *fs, blk_t block, blk_t num);
void ext2_zero_inode(struct ext2_fs *fs, ino_t inode);
int ext2_ioctl(struct ext2_fs *fs, int ioc, void *ptr);

/* block related */
void ext2_bgbitmap_cache_deinit(struct ext2_fs *fs);
void ext2_bgbitmap_cache_flush(struct ext2_fs *fs);
void ext2_bgbitmap_cache_init(struct ext2_fs *fs);
int ext2_get_block_state(struct ext2_fs *, blk_t block);
void ext2_set_block_state(struct ext2_fs *, blk_t block, int state,
			  int updatemetadata);

/* block relocator */
int ext2_block_relocate(struct ext2_fs *fs);

/* buffer */
void ext2_bcache_deinit(struct ext2_fs *fs);
void ext2_bcache_dump(struct ext2_fs *fs);
void ext2_bcache_flush(struct ext2_fs *fs, blk_t block);
void ext2_bcache_flush_range(struct ext2_fs *fs, blk_t first, blk_t last);
int ext2_bcache_init(struct ext2_fs *fs);
void ext2_bcache_sync(struct ext2_fs *fs);
struct ext2_buffer_head *ext2_bcreate(struct ext2_fs *fs, blk_t block);
struct ext2_buffer_head *ext2_bread(struct ext2_fs *fs, blk_t block);
void ext2_brelse(struct ext2_buffer_head *bh, int forget);

/* inode relocator */
int ext2_inode_relocate(struct ext2_fs *fs);

/* journalling */
void ext2_journal_deinit(struct ext2_fs *fs);
int ext2_journal_init(struct ext2_fs *fs);

/* metadata mover */
int ext2_block_bitmap_push(struct ext2_fs *fs, int group, int diff);
int ext2_inode_bitmap_push(struct ext2_fs *fs, int group, int diff);
int ext2_metadata_push(struct ext2_fs *fs);

/* fs creation */
struct ext2_fs *ext2_mkfs(struct ext2_dev_handle *handle, blk_t numblocks,
			  int log_block_size, blk_t blocks_per_group,
			  int inodes_per_group, int sparse_sb,
			  int reserved_block_percentage);

/* resize */
int ext2_resize_fs(struct ext2_fs *fs);

/* unix I/O */
loff_t ext2_llseek(unsigned int fd, loff_t offset, unsigned int whence);
struct ext2_dev_handle *ext2_make_dev_handle_from_file(char *dev, char *dir,
						       char *prog);

#define set_bit(buf, offset)	buf[(offset)>>3] |= _bitmap[(offset)&7]
#define clear_bit(buf, offset)	buf[(offset)>>3] &= ~_bitmap[(offset)&7]
#define check_bit(buf, offset)	(buf[(offset)>>3] & _bitmap[(offset)&7])

#ifdef USE_EXT2_IS_DATA_BLOCK
static int __inline__ ext2_is_data_block(struct ext2_fs *fs, blk_t block)
{
	blk_t blk;
	int   group;

	if (block >= fs->sb.s_blocks_count)
		fprintf(stderr, "is_data_block(%i): block number invalid!\n", block);

	blk = block - fs->sb.s_first_data_block;

	group = blk / fs->sb.s_blocks_per_group;
	blk %= fs->sb.s_blocks_per_group;

	if (ext2_bg_has_super(fs, group) && blk <= fs->gdblocks)
		return 0;

	if (block == fs->gd[group].bg_block_bitmap ||
	    block == fs->gd[group].bg_inode_bitmap)
		return 0;

	if (block >= fs->gd[group].bg_inode_table &&
	    block < fs->gd[group].bg_inode_table + fs->inodeblocks)
		return 0;

	return 1;
}
#endif /* USE_EXT2_IS_DATA_BLOCK */

#endif
