// ruleset.h
// part of ezbounce

#ifndef __RULESET_H
#define __RULESET_H

#include "general.h"
#include "objset.h"
#include "linkedlist.h"


/* 
 *  holds information stored in
 *  allow|deny {
 *     from ...
 *     to ...
 *  }
 *  blocks in the configuration file
 */


class rule_set {
private:
    rule_set *pNext;
public:
    friend int reload_config_file(struct config_struct *, const char *);
    static const char FROM, TO;
    static const int  UNLIMITED;
    /* check if this guy is permitted to connect to the server*/
    virtual int is_allowed(const char *address, unsigned short port, 
                           char *buffer4reason, size_t) const = 0;
    /* check if this guy may connect to "to" */
    virtual int is_allowed_to(const char *from, const char *to,
                              unsigned short port, char *buffer4reason, size_t) const = 0;
    /* checks if host & port are in its list of addresses */
    virtual bool does_match(const char *host, unsigned short port, char t) const = 0;
    /* add a host name to the list of ppl that can connect */
    virtual bool add_host_from(const char *address, const char *ports,
                               const char *reason, int max) = 0;
    /* add a host name to the list of places ppl can connect to */
    virtual bool add_host_to(const char *address, const char *ports,
                             const char *reason, int max) = 0;
    /* 
     *  Call these functions to register/unregister the connection
     *  (that is to update the num counts of the rulesets).
     *  We provide a bare-bones one here that just updates
     *  the usage counters. Derived classes should call this to update the counters.
     */
    virtual int register_connection(char t, const char *, const char *, unsigned short) 
    {
    #ifdef __DEBUG__
        fprintf(stderr,"Increasing %s usage count of %p to %d\n", ((t == FROM) ? "FROM" : "TO"), this, (t == FROM ? num_registered_from + 1 : num_registered_to + 1));
    #endif 
        return (t == FROM) ? ++num_registered_from : ++num_registered_to;
    }
    virtual int unregister_connection(char t, const char *, const char *, unsigned short) 
    {
    #ifdef __DEBUG__
        fprintf(stderr, "Decreasing %s usage count of %p to %d\n", ((t == FROM) ? "FROM" : "TO"), this,(t == FROM ? num_registered_from - 1 : num_registered_to - 1));
    #endif 
        return (t == FROM) ? --num_registered_from : --num_registered_to;
    }

    rule_set();
    virtual ~rule_set();
    
    bool dead() const 
    { 
        return (obsolete && !num_registered_to && !num_registered_from);
    }
    rule_set * next() { return pNext;}
    static rule_set * first() { return pFirst;}
    static bool find_matching_sets(const char *addr, u_short port, obj_set<rule_set> *);
    bool operator == (/* const */ rule_set &);

    static rule_set *pFirst;

protected:
    struct rs_host 
    {
        char *address, *ports, *reason, type;
        int  max, num;
        ~rs_host();
        rs_host(char, const char *, const char *, const char *, int);
    };

    linkedlist<rs_host> hosts;

    /*
     * These are really only used by denied_rule_set and it might be a good idea
     *  to just put these there ..
     */
    unsigned num_froms, num_tos;

    /* these keep track of register_connection() and unregister_connection calls */
    unsigned num_registered_to, num_registered_from;

    bool obsolete;
};

class allowed_rule_set : public rule_set 
{
public:
    bool add_host_to( const char *, const char *, const char *, int);
    bool add_host_from( const char *, const char *, const char *, int);
    int is_allowed(const char *, unsigned short, char *, size_t) const;
    int is_allowed_to(const char*, const char*, unsigned short, char *, size_t) const;
    bool does_match(const char *from, unsigned short port, char t) const ;
    int register_connection(char,const char *, const char *, unsigned short);
    int unregister_connection(char, const char * , const char *, unsigned short);
};

class denied_rule_set : public rule_set 
{
public:
    bool add_host_to(const char *, const char *, const char *, int);
    bool add_host_from(const char *, const char *, const char *, int);
    int is_allowed(const char *, unsigned short, char *, size_t) const ;
    int is_allowed_to(const char*, const char *, unsigned short, char *, size_t) const;
    bool does_match(const char *from, unsigned short port, char t) const;
    int register_connection(char, const char *, const char *, unsigned short);
    int unregister_connection(char, const char * , const char *, unsigned short);      
};

#endif
