import operator
import os
import re
import string

from globals import *

rx_thread = re.compile(r'^(\d+):(\S+) \[(\d+)\] (.+)$')
rx_subject = re.compile(r'^(\S+) (.+)$')
rx_message = re.compile(r'^(\d+):(\d+):(\S+) (.+)$')
rx_references = re.compile(r'\nReferences:[^\n]+(\n\s[^\n]+)*\n',
						   re.IGNORECASE)
rx_inreplyto = re.compile(r'\nIn-Reply-To:\s*<([^>]+)>', re.IGNORECASE)
rx_endofhdr = re.compile(r'^\s*$')
rx_messageid = re.compile(r'\nMessage-ID:\s*<([^>]+)>', re.IGNORECASE)
rx_mid = re.compile(r'<([^>]+)>')
rx_ws = re.compile(r'\s+')

def _in(a, min, max):
	if a > max: return max
	if a < min: return min
	return a

month_names = {
	1: 'January',
	2: 'February',
	3: 'March',
	4: 'April',
	5: 'May',
	6: 'June',
	7: 'July',
	8: 'August',
	9: 'September',
	10: 'October',
	11: 'November',
	12: 'December'
	}

def _parse_references(str):
	list = [ ]
	match = rx_mid.search(str)
	while match:
		list.append(match.group(1))
		match = rx_mid.search(str, match.end())
	return list

#def _thread_message(id, idmap, nummap, irtmap, replymap, refmap):
def _thread_message(id, idmap, nummap, replymap):
	# result: {replies}
	#try: irt = nummap[irtmap[id]]
	#except KeyError: irt = None
	replies = { }
	for i in replymap[id]:
		num = nummap[i]
		replies[num] = _thread_message(i, idmap, nummap, replymap)
		idmap[num] = None
	#references = [ ]
	#if refmap[id]:
	#	for i in refmap[id]:
	#		if i <> irtmap[id] and nummap.has_key(i):
	#			references.append(nummap[i])
	#followups = [ ]
	#for i in replymap[id]:
	#	if not replymap.has_key(i):
	#		followups.append(nummap[i])
	#return (irt, replies, references, followups)
	return replies

def _process_thread(thread, map):
	list = [ ]
	nums = thread.keys()
	nums.sort()
	for num in nums:
		msg = map[num]
		msg['threads'] = _process_thread(thread[num], map)
		list.append(msg)
	return list
	
class StringFile:
	def __init__(self):
		self.data = [ ]
	def write(self, data):
		self.data.append(data)
	def __str__(self):
		return string.join(self.data)

def _decode_parts(msg, mf):
	import mimetools
	if msg.maintype == 'multipart':
		for item in msg.plist:
			if item[:9] == 'boundary=':
				boundary = item[9:]
				if boundary[0] == '"' and boundary[-1] == '"':
					boundary = boundary[1:-1]
		mf.push(boundary)
		parts = [ ]
		while 1:
			if not mf.next(): break
			parts.extend(_decode_parts(mimetools.Message(mf), mf))
		mf.pop()
		return parts
	elif msg.type == 'text/plain' or msg.type == 'message/rfc822':
		eh = string.lower(msg.encodingheader or '')
		if eh and eh <> '7bit' and eh <> '8bit':
			body = StringFile()
			mimetools.decode(mf, body, eh)
			body = str(body)
		else:
			body = string.join(mf.readlines(), '')
		return [ { TYPE: msg.type, BODY: body } ]
	else:
		return [ { TYPE: msg.type, BODY: None } ]

_rx_header = re.compile(r'^(\S+):\s*(.*)$')
def _parse_headers(headers):
	result = [ ]
	name = None
	value = None
	for line in headers:
		match = _rx_header.match(line)
		if match:
			(name,value) = match.groups()
			if name:
				result.append({NAME: name, VALUE: value})
		else:	
			value = "%s\n%s" % (value, line)
	if name:
		result.append({NAME: name, VALUE: value})
	return result

class EzmlmArchive:
	def __init__(self, listdir):
		self.listdir = listdir
		self.archdir = os.path.join(listdir, 'archive')
		self.months = os.listdir(os.path.join(self.archdir, 'threads'))
		self.months.sort()

	def set_months(self, form):
		firstmonth = int(self.months[0])
		lastmonth = int(self.months[-1])
		try:
			month = int(form[MONTH])
			if month > lastmonth or month < firstmonth:
				raise ValueError, "Month value is out of range."
		except KeyError:
			month = lastmonth
		nextmonth = month + 1
		prevmonth = month - 1
		if nextmonth % 100 > 12: nextmonth = month + 89
		if prevmonth % 100 == 0: prevmonth = prevmonth - 88
		elif prevmonth % 100 > 12: prevmonth = month - 100
		nextmonth = _in(nextmonth, firstmonth, lastmonth)
		prevmonth = _in(prevmonth, firstmonth, lastmonth)
		form[FIRSTMONTH] = firstmonth
		form[PREVMONTH] = prevmonth
		form[MONTH] = month
		form[NEXTMONTH] = nextmonth
		form[LASTMONTH] = lastmonth
		form[MONTHNAME] = month_names[month % 100]
		form[YEAR] = month / 100

	def open(self, num):
		num = '%03d' % num
		return open(os.path.join(self.archdir, num[:-2], num[-2:]))

	def message(self, number):
		msg = self.open(number)
		import multifile
		mf = multifile.MultiFile(msg)
		import mimetools
		msg = mimetools.Message(mf)
		headers = _parse_headers(msg.headers)
		parts = _decode_parts(msg, mf)
		return {
			SENDER: msg.getheader('from') or msg.getheader('return-path'),
			SUBJECT: msg.getheader('subject'),
			DATE: msg.getheader('date'),
			MESSAGEID: msg.getheader('message-id'),
			HEADERS: headers,
			PARTS: parts,
			}

	def month(self, month):
		messages = { }
		for thread in self.threads(month):
			tid = thread[THREADID]
			thread = self.thread(tid)
			subject = thread[SUBJECT]
			for message in thread[MESSAGES]:
				if message[MONTH] == month:
					message[SUBJECT] = subject
					message[THREADID] = tid
					messages[message[MSGNUM]] = message
		numbers = messages.keys()
		numbers.sort()
		return map(operator.getitem, [messages]*len(numbers), numbers)

	def thread(self, threadid):
		if '.' in threadid or '/' in threadid:
			raise ValueError, "Thread ID contains invalid characters"
		part1 = threadid[:2]
		part2 = threadid[2:]
		path = os.path.join(self.archdir, 'subjects', part1, part2)
		lines = map(string.strip, open(path).readlines())
		subject = rx_subject.match(lines[0]).group(2)
		list = [ ]
		for line in lines[1:]:
			match = rx_message.match(line)
			if match:
				groups = match.groups()
				msgnum = int(groups[0])
				list.append({
					MSGNUM: msgnum,
					MONTH: int(groups[1]),
					AUTHORID: groups[2],
					AUTHOR: groups[3],
					})
		return { SUBJECT: subject, MESSAGES: list }

	def threads(self, month):
		list = [ ]
		path = os.path.join(self.archdir, 'threads', str(month))
		for line in open(path).readlines():
			match = rx_thread.match(string.strip(line))
			groups = match.groups()
			list.append({
				MSGNUM: int(groups[0]),
				THREADID: groups[1],
				MSGCOUNT: int(groups[2]),
				SUBJECT: groups[3]
				})
		return list

	def search(self, terms):
		terms = rx_ws.split(terms)
		terms = map(string.lower, terms)
		dir = os.path.join(self.archdir, 'threads')
		threads = [ ]
		for month in self.months:
			for line in open(os.path.join(dir, month)).readlines():
				line = string.strip(line)
				match = rx_thread.match(line)
				if match:
					groups = match.groups()
					subject = string.lower(groups[3])
					for term in terms:
						if string.find(subject, term) < 0: subject = ''; break
					if not subject: continue
					groups = match.groups()
					threads.append({
						MSGNUM: int(groups[0]),
						THREADID: groups[1],
						MSGCOUNT: int(groups[2]),
						SUBJECT: groups[3],
						MONTH: int(month)
						})
		return threads

	def _parse_message(self, num):
		# Result: message-id, in-reply-to, references
		headers = self.open(num).read(4096)
		eoh = rx_endofhdr.search(headers)
		if eoh: headers = headers[:eoh.start()]
		messageid = rx_messageid.search(headers)
		if messageid: messageid = messageid.group(1)
		inreplyto = rx_inreplyto.search(headers)
		if inreplyto: inreplyto = inreplyto.group(1)
		refs = rx_references.search(headers)
		if refs: refs = _parse_references(refs.group(0))
		return (messageid, inreplyto, refs)

	def thread_messages(self, messages):
		messageids = { }
		messagenums = { }
		replies = { }
		inreplytos = { }
		references = { }
		for message in messages:
			msgnum = message[MSGNUM]
			( messageid,inreplyto,refs ) = self._parse_message(msgnum)
			messageids[msgnum] = messageid
			messagenums[messageid] = msgnum
			replies[messageid] = [ ]
			references[messageid] = refs
			if refs and not inreplyto:
				inreplyto = refs[-1]
			if inreplyto:
				if replies.has_key(inreplyto):
					replies[inreplyto].append(messageid)
			inreplytos[messageid] = inreplyto
		#print 'messageids:', messageids
		#print 'messagenums:', messagenums
		#print 'replies:', replies
		#print 'references:', references
		num2msg = { }
		for message in messages:
			num2msg[message[MSGNUM]] = message
		list = { }
		for message in messages:
			msgnum = message[MSGNUM]
			if messageids[msgnum] is not None:
				tmp = _thread_message(messageids[msgnum],
									  messageids, messagenums, replies)
									  #inreplytos, replies, references)
				list[msgnum] = tmp
		return _process_thread(list, num2msg)

