/***************************************************************************
                          FLDataTable.cpp  -  description
                             -------------------
    begin                : Sun Jul 1 2001
    copyright            : (C) 2001,2002 by Federico Albujer Zornoza
    email                : mail@infosial.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "FLDataTable.h"
#include "FLSqlCursor.h"
#include "FLTableMetaData.h"
#include "FLFieldMetaData.h"
#include "FLUtil.h"

static const char *ok_xpm[] = {
  "16 14 26 1",
  " 	c None",
  ".	c #E2FDBE",
  "+	c #A8FD2E",
  "@	c #86E400",
  "#	c #589600",
  "$	c #2C4E00",
  "%	c #B1FD4E",
  "&	c #B1FD3A",
  "*	c #B3FD3E",
  "=	c #90FD02",
  "-	c #7DDE00",
  ";	c #3D7200",
  ">	c #6FC600",
  ",	c #91FD06",
  "'	c #94FD0C",
  ")	c #A2FD2C",
  "!	c #7CE600",
  "~	c #508E00",
  "{	c #1B2F00",
  "]	c #3B6E00",
  "^	c #6AC400",
  "/	c #91FD04",
  "(	c #8AEC00",
  "_	c #4D8400",
  ":	c #89FE00",
  "<	c #3F7600",
  "                ",
  "                ",
  "            .+  ",
  "           .+@# ",
  "          .+@#$ ",
  "  .%     .&@#$  ",
  " .*=@   .&-#$   ",
  "$;>,'- .)!~$    ",
  " {]^'/.)(_$     ",
  "  {]>:&!_$      ",
  "   {]>-~$       ",
  "    {<#$        ",
  "     {$         ",
  "                "
};

static const char *remove_xpm[] = {
  "16 16 53 1",
  " 	c None",
  ".	c #EFB447",
  "+	c #FEB632",
  "@	c #FEA622",
  "#	c #FE9A16",
  "$	c #E89A31",
  "%	c #E9BD5F",
  "&	c #FECA52",
  "*	c #FEA723",
  "=	c #FE9014",
  "-	c #CA7E3E",
  ";	c #EEB54B",
  ">	c #FEC950",
  ",	c #FEDA7A",
  "'	c #FEE6A2",
  ")	c #FEC246",
  "!	c #FE7E12",
  "~	c #C26961",
  "{	c #FED666",
  "]	c #FEE7B2",
  "^	c #FEF6DE",
  "/	c #DF5140",
  "(	c #F3A22F",
  "_	c #FFFFFF",
  ":	c #DE4632",
  "<	c #C43C6F",
  "[	c #FA6E12",
  "}	c #D6323E",
  "|	c #CE025E",
  "1	c #FE8A0E",
  "2	c #EE621E",
  "3	c #D2224A",
  "4	c #B60262",
  "5	c #FE8E12",
  "6	c #D2026A",
  "7	c #AA025E",
  "8	c #DB743E",
  "9	c #FE860E",
  "0	c #D21252",
  "a	c #892C5E",
  "b	c #EA5A26",
  "c	c #B33C65",
  "d	c #D21E46",
  "e	c #95175C",
  "f	c #A42F6A",
  "g	c #AF1064",
  "h	c #B10862",
  "i	c #8F255E",
  "j	c #9B165E",
  "k	c #A9035D",
  "l	c #A7035C",
  "m	c #A00B5D",
  "n	c #8B1F5A",
  "                ",
  "     .+@#$      ",
  "   %&&&&+*=-    ",
  "  ;>,',,)+#!~   ",
  "  +{]^]]{+@!/   ",
  " (),__]]{__!:<  ",
  " #+&,__,__#[}|  ",
  " 1@+))___#1234  ",
  " !#@++___5[:67  ",
  " 8[9#__#__:04a  ",
  "  :2__[[b__47   ",
  "  c3}}}}d|47e   ",
  "   fgh44477i    ",
  "     jklmn      ",
  "                ",
  "                "
};

FLDataTable::FLDataTable (QWidget * parent, const char *name):
QDataTable (parent, name),
rowSelected (-1),
colSelected (-1),
cursor_ (0),
readonly_ (false),
insertonly_ (false)
{
  if (!name)
	setName ("FLDataTable");
  setPaletteBackgroundColor (QColor (220, 220, 220));
}

FLDataTable::~FLDataTable ()
{
  disconnect (this, 0, 0, 0);
}

void
FLDataTable::selectRow (int r, int c)
{
  if (!cursor_)
	return;

  if (currentRow () == rowSelected)
	return;

  if (r < 0)
	{
	  if (cursor_->isValid ())
		{
		  rowSelected = cursor_->at ();
		  colSelected = currentColumn ();
		}
	  else
		{
		  rowSelected = 0;
		  colSelected = 0;
		}
	}
  else
	{
	  rowSelected = r;
	  colSelected = c;
	}

  cursor_->seek (rowSelected);
}

void
FLDataTable::setFLSqlCursor (FLSqlCursor * c)
{
  if (!c)
	return;

  if (!cursor_)
	{
	  connect (this, SIGNAL (currentChanged (int, int)), SLOT (selectRow (int, int)));
	  connect (this, SIGNAL (clicked (int, int, int, const QPoint &)), SLOT (selectRow (int, int)));
	}

  cursor_ = c;
  QDataTable::setSqlCursor ((QSqlCursor *) c, true, false);
}

void
FLDataTable::paintCell (QPainter * p, int row, int col, const QRect & cr, bool selected, const QColorGroup & cg)
{
  QTable::paintCell (p, row, col, cr, selected, cg);

  if (!cursor_)
	return;

  FLTableMetaData *tMD = cursor_->metadata ();

  if (!tMD)
	return;

  int type = tMD->fieldType (cursor_->field (indexOf (col))->name ());

  if (type == QVariant::Pixmap && row != rowSelected)
	return;

  p->setPen (selected ? cg.highlightedText () : cg.text ());

  if (cursor_->QSqlCursor::seek (row))
	paintField (p, cursor_->field (indexOf (col)), cr, selected);

  cursor_->QSqlCursor::seek (rowSelected);
}

void
FLDataTable::paintField (QPainter * p, const QSqlField * field, const QRect & cr, bool)
{
  if (!field || !cursor_)
	return;

  FLTableMetaData *tMD = cursor_->metadata ();

  if (!tMD)
	return;

  if (!field->isNull ())
	{
	  int type = tMD->fieldType (field->name ());
	  int partDecimal = tMD->fieldPartDecimal (field->name ());
	  QString text;

	  switch (type)
		{
		case QVariant::Double:
		  text.setNum (field->value ().toDouble (), 'f', partDecimal);
		  p->drawText (2, 2, cr.width () - 4, cr.height () - 4, Qt::AlignRight, FLUtil::formatoMiles (text));
		  break;

		case FLFieldMetaData::Unlock:
		  {
			if (field->value ().toBool ())
			  {
				QPixmap ok ((const char **) ok_xpm);
				p->drawPixmap (2, 2, ok, 0, 0, cr.width () - 4, cr.height () - 4);
			  }
			else
			  {
				QPixmap no ((const char **) remove_xpm);
				p->drawPixmap (2, 2, no, 0, 0, cr.width () - 4, cr.height () - 4);
			  }
		  }
		  break;

		case QVariant::String:
		  text = field->value ().toString ();
		  p->drawText (2, 2, cr.width () - 4, cr.height () - 4, fieldAlignment (field), text);
		  break;

		case FLFieldMetaData::Serial:
		case FLFieldMetaData::Sequence:
		case QVariant::UInt:
		case QVariant::Int:
		  text = field->value ().toString ();
		  p->drawText (2, 2, cr.width () - 4, cr.height () - 4, Qt::AlignRight, FLUtil::formatoMiles (text));
		  break;

		case QVariant::Pixmap:
		  {
			QCString cs = field->value ().toCString ();
			if (cs.isEmpty ())
			  return;

			QPixmap pix;

			if (!QPixmapCache::find (cs.left (100), pix))
			  {
				pix.loadFromData (cs);
				QPixmapCache::insert (cs.left (100), pix);
			  }
			if (!pix.isNull ())
			  p->drawPixmap (2, 2, pix, 0, 0, cr.width () - 4, cr.height () - 4);
		  }
		  break;

		case QVariant::Date:
		  {
			QDate d = field->value ().toDate ();

			text = d.toString ("dd-MM-yyyy");
			p->drawText (2, 2, cr.width () - 4, cr.height () - 4, fieldAlignment (field), text);
		  }
		  break;

		case QVariant::Time:
		  {
			QTime t = field->value ().toTime ();

			text = t.toString ("hh:mm:ss");
			p->drawText (2, 2, cr.width () - 4, cr.height () - 4, fieldAlignment (field), text);
		  }
		  break;

		case QVariant::StringList:
		  text = field->value ().toString ();
		  p->drawText (2, 2, cr.width () - 4, cr.height () - 4, fieldAlignment (field), text.left (255) + "...");
		  break;

		case QVariant::Bool:
		  text = field->value ().toBool ()? tr ("S") : tr ("No");
		  p->drawText (2, 2, cr.width () - 4, cr.height () - 4, fieldAlignment (field), text);
		  break;
		}
	}
}

bool FLDataTable::eventFilter (QObject * o, QEvent * e)
{
  if (!o || !e)
	return QDataTable::eventFilter (o, e);

  int
	r =
	currentRow (),
	c =
  currentColumn (),
	nr =
  numRows (),
	nc =
  numCols ();

  switch (e->type ())
	{
	case QEvent::KeyPress:
	  {
		QKeyEvent *
		  ke = (QKeyEvent *)
		  e;

		if (ke->key () == Key_F2)
		  return true;

		if (ke->key () == Key_Up && r == 0)
		  return true;

		if (ke->key () == Key_Left && c == 0)
		  return true;

		if (ke->key () == Key_Down && r == nr - 1)
		  return true;

		if (ke->key () == Key_Right && c == nc - 1)
		  return true;

		if ((ke->key () == Key_Enter || ke->key () == Key_Return) && r > -1)
		  emit
		  recordChoosed ();

		if (ke->key () == Key_A)
		  if (cursor_ && !readonly_)
			{
			  cursor_->insertRecord ();
			  return true;
			}
		  else
			return false;

		if (ke->key () == Key_M)
		  if (insertonly_)
			return false;
		  else if (cursor_ && !readonly_)
			{
			  cursor_->editRecord ();
			  return true;
			}
		  else
			return false;

		if (ke->key () == Key_E || ke->key () == Key_Delete)
		  if (insertonly_)
			return false;
		  else if (cursor_ && !readonly_)
			{
			  cursor_->deleteRecord ();
			  return true;
			}
		  else
			return false;

		if (ke->key () == Key_V)
		  if (cursor_)
			{
			  cursor_->browseRecord ();
			  return true;
			}
	  }
	  break;

	default:
	  return QDataTable::eventFilter (o, e);
	}
  return QDataTable::eventFilter (o, e);
}

void
FLDataTable::contentsContextMenuEvent (QContextMenuEvent * e)
{
  QTable::contentsContextMenuEvent (e);
}

void
FLDataTable::contentsMouseDoubleClickEvent (QMouseEvent * e)
{
  if (e->button () != LeftButton)
	return;

  int tmpRow = rowAt (e->pos ().y ());
  int tmpCol = columnAt (e->pos ().x ());
  QTableItem *itm = item (tmpRow, tmpCol);

  if (itm && !itm->isEnabled ())
	return;

  emit doubleClicked (tmpRow, tmpCol, e->button (), e->pos ());
  emit recordChoosed ();
}

void
FLDataTable::setFocus ()
{
  setPaletteBackgroundColor (QColor (255, 255, 255));
  QWidget::setFocus ();
}

void
FLDataTable::focusOutEvent (QFocusEvent *)
{
  setPaletteBackgroundColor (QColor (230, 230, 230));
}
