/*
 * rtpstream.c - Source for RTP plugin stream implementation
 *
 * Farsight RTP/AVP/SAVP/AVPF Module
 * Copyright (C) 2005,2006 Collabora Ltd.
 * Copyright (C) 2005,2006 Nokia Corporation
 *   @author Rob Taylor <rob.taylor@collabora.co.uk>
 *   @author Philippe Kalaf <philippe.kalaf@collabora.co.uk>
 * Copyright (C) 2005 INdT
 *   @author Andre Moreira Magalhaes <andre.magalhaes@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "rtp.h"
#include "rtpstream.h"
#include "rtpsession.h"
#include "rtpgstcodecs.h"
#include "farsight.h"
#include "farsight-transport.h"
#include "farsight-transmitter.h"

#include "helpers/farsight-interfaces.h"
#ifndef G_OS_WIN32
#include <arpa/inet.h>
#endif

#ifdef HAVE_CLINKC
#include "helpers/upnpigd.h"
#endif

#include <gst/gst.h>

#include <string.h>

#define DEFAULT_CONN_TIMEOUT 45

#define DEFAULT_MIN_PTIME 0
#define DEFAULT_MAX_PTIME -1

#define GST_QUERY_JB_STATS \
        (gst_query_type_get_by_nick ("jitterbuffer-statistics"))

#define GST_CODECS_CONF_FILE "gstcodecs.conf"

enum
{
  PROP_0,
  PROP_CONN_TIMEOUT,
  PROP_TRANSMITTER,
  PROP_MIN_PTIME,
  PROP_TRANSMITTER_OBJECT,
  PROP_MAX_PTIME,
  PROP_CODEC_PREFS,
  PROP_RESERVED_PT
};

struct _FarsightRTPStreamPrivate
{
  gboolean disposed;
  gboolean on_hold;

  GList *local_codecs;
  GList *remote_codecs;
  GList *negotiated_codecs;
  GHashTable *pt_caps_table;

  GArray *codec_pref_list;
  GHashTable *negotiated_codec_associations;
  GHashTable *local_codec_associations;

  GList *reserved_pt_list;

  GList *codecs_configuration;

  FarsightTransmitter *transmitter;
  gchar *transmitter_name;

  /* send/recv pipeline */
  GstElement *main_pipeline; /*this is an optional user provided GstPipeline*/
  GstElement *pipeline;
  GstElement *rtpmuxer;
  GstElement *rtpbin;
  GstElement *send_codec_bin;
  GstElement *recv_codec_bin;

  GstElement *src;
  GstCaps *src_filter;
  GstElement *src_capsfilter;
  GstElement *src_valve;

  GstElement *rtpdemux;
  GstElement *sink;
  GstCaps *sink_filter;
  GstElement *sink_capsfilter;

  gboolean sending;

  guint bus_watch;

  GArray *pending_src_ids;

  gint recv_codec_id;
  gint send_codec_id;

  gint forced_send_codec_id;
  gint preload_recv_codec_id;

  gboolean build_send_pipeline;

  gboolean prepared; /*set if prepare has been called*/

  guint conn_timeout;
  guint timeout_src;
  guint stats_timeout;

  GList *local_candidates;
  GList *remote_candidates;

  gchar *active_native_candidate;
  gchar *active_remote_candidate;

  gboolean use_upnp;

  guint64 min_ptime;
  guint64 max_ptime;
};

typedef struct _DepayloaderChainInfo DepayloaderChainInfo;
struct _DepayloaderChainInfo
{
  FarsightStream *stream;
  GstPad *pad;
  gint pt;
};

#define FARSIGHT_RTP_STREAM_GET_PRIVATE(o)  \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), FARSIGHT_TYPE_RTP_STREAM, \
                                FarsightRTPStreamPrivate))

static gboolean g_object_has_property (GObject *object, const gchar *property);

static void farsight_rtp_stream_class_init (FarsightRTPStreamClass *klass);
static void farsight_rtp_stream_init (FarsightRTPStream *rtp_stream);

static void farsight_rtp_stream_dispose (GObject *object);
static void farsight_rtp_stream_finalize (GObject *object);

static void farsight_rtp_stream_set_property (GObject * object,
    guint prop_id, const GValue * value, GParamSpec * pspec);
static void farsight_rtp_stream_get_property (GObject * object,
    guint prop_id, GValue * value, GParamSpec * pspec);

static gboolean farsight_rtp_stream_bus_watch_cb (GstBus *bus, GstMessage
    *message, gpointer user_data);
static void farsight_rtp_stream_new_payload_type (GstElement *element,
                                                  gint pt, GstPad *pad,
                                                  gpointer user_data);
static void farsight_rtp_stream_payload_type_change (GstElement *element,
                                                     gint pt,
                                                     gpointer user_data);
static gboolean farsight_rtp_stream_create_new_pt_recv_pipeline (
    FarsightStream *stream,
    gint id, GstPad *pad);
static GstElement *farsight_rtp_stream_get_pipeline (FarsightStream *stream);
static gboolean farsight_rtp_stream_set_pipeline (FarsightStream *stream,
                                                  GstElement *pipeline);

static gboolean farsight_rtp_stream_hold (FarsightStream *stream);
static gboolean farsight_rtp_stream_unhold (FarsightStream *stream);
static gboolean farsight_rtp_stream_is_held (FarsightStream *stream);
static void farsight_rtp_stream_set_active_codec (FarsightStream *stream,
                                                  gint id);
static gboolean farsight_rtp_stream_set_active_codec_idler(GstPad *src_pad,
                                                           gpointer user_data);
static gboolean farsight_rtp_stream_clean_recv_codec_bin (
    FarsightRTPStream *self);
static gboolean farsight_rtp_stream_clean_recv_codec_bin_obj(
    FarsightRTPStream *self, GstElement *codec_bin);
static gboolean farsight_rtp_stream_clean_send_codec_bin(FarsightRTPStream *self,
                                                         GstState *old_state);
static gboolean farsight_rtp_stream_setup_send_codec_bin(FarsightRTPStream *self,
                                                         GstState new_state);
static gboolean farsight_rtp_stream_link_send_codec_bin (FarsightRTPStream *self,
                                                         GstElement *codec_bin);
static gboolean farsight_rtp_stream_unlink_send_codec_bin (FarsightRTPStream *self,
                                                      GstElement *codec_bin);
static gboolean farsight_rtp_stream_link_recv_codec_bin (FarsightRTPStream *self,
                                                    GstElement *codec_bin);
static gboolean farsight_rtp_stream_link_recv_codec_bin_ex (FarsightRTPStream *self,
                                                    GstElement *codec_bin, GstPad *pad);
static gboolean farsight_rtp_stream_right_unlink_recv_codec_bin (FarsightRTPStream *self,
                                                      GstElement *codec_bin);
static gboolean farsight_rtp_stream_unlink_recv_codec_bin (FarsightRTPStream *self,
                                                      GstElement *codec_bin, gboolean release_rtpdemux_pad);
static gboolean farsight_rtp_stream_send_codec_bin_has_sink (FarsightRTPStream *self);

static gint farsight_rtp_stream_get_active_codec (FarsightStream *stream);
static G_CONST_RETURN GList *farsight_rtp_stream_get_local_codecs (FarsightStream *stream);
static gboolean farsight_rtp_stream_set_remote_codecs (FarsightStream *stream,
                                                   const GList *codecs);
static GList *farsight_rtp_stream_get_codec_intersection (FarsightStream *stream);
static void farsight_rtp_stream_set_codec_preference_list (FarsightStream *stream,
    const GArray *codec_pref);
static gboolean farsight_rtp_stream_set_source (FarsightStream *stream,
                                                GstElement *source);
static gboolean farsight_rtp_stream_set_source_filter (FarsightStream *stream,
    GstCaps *filter);
static GstElement *farsight_rtp_stream_get_source (FarsightStream *stream);
static gboolean farsight_rtp_stream_set_sink (FarsightStream *stream,
                                              GstElement *sink);
static gboolean farsight_rtp_stream_set_sink_filter (FarsightStream *stream,
    GstCaps *filter);
static GstElement *farsight_rtp_stream_get_sink (FarsightStream *stream);

static gboolean farsight_rtp_stream_build_base_pipeline (FarsightRTPStream *self);
static gboolean farsight_rtp_stream_build_send_pipeline (FarsightRTPStream *self);
static void farsight_rtp_stream_try_set_playing (FarsightRTPStream *self);
static gboolean farsight_rtp_stream_set_playing (gpointer data);
static gboolean farsight_rtp_stream_start (FarsightStream *stream);
static void farsight_rtp_stream_stop (FarsightStream *stream);
static gboolean farsight_rtp_stream_set_sending (FarsightStream *stream,
    gboolean sending);
static CodecAssociation * farsight_rtp_stream_choose_codec (
    FarsightRTPStream *self);
static gboolean farsight_rtp_stream_candidate_exists (FarsightStream *stream,
                                                      const GList *candidate_list,
                                                      const GList *candidate);

static void
farsight_rtp_stream_new_native_candidate (gpointer transmitter,
    const FarsightTransportInfo *candidate, gpointer stream);
static void
farsight_rtp_stream_native_candidates_prepared (gpointer transmitter,
    gpointer stream);
static void
farsight_rtp_stream_new_active_candidate_pair (gpointer transmitter,
    const gchar *native_candidate_id, 
    const gchar *remote_candidate_id,
    gpointer stream);
static void 
farsight_rtp_stream_transmitter_state_changed (gpointer transmitter,
    FarsightTransmitterState state,
    gpointer stream);
static void farsight_rtp_stream_transmitter_error (gpointer transmitter,
    gpointer stream);

static void farsight_rtp_stream_upnp_send_request (FarsightRTPStream *self,
    const FarsightTransportInfo *derived_trans);
static void farsight_rtp_stream_upnp_close_ports (FarsightRTPStream *self);

static void farsight_rtp_stream_prepare_transports (FarsightStream *self);
static gboolean farsight_rtp_stream_add_remote_candidate_to_rtpbin (FarsightRTPStream *self, 
                                                                    const gchar *remote_candidate_id);
static gboolean farsight_rtp_stream_set_active_candidate_pair (FarsightStream *stream,
                                                               const gchar *native_candidate_id, 
                                                               const gchar *remote_candidate_id);
static GList *farsight_rtp_stream_get_native_candidate (FarsightStream *stream, 
                                                        const gchar *candidate_id);
static G_CONST_RETURN GList *farsight_rtp_stream_get_native_candidate_list (FarsightStream *stream);
static void farsight_rtp_stream_set_remote_candidate_list (FarsightStream *stream, 
                                                           const GList *remote_candidates);
static void farsight_rtp_stream_add_remote_candidate (FarsightStream *self,
                                                      const GList *remote_candidate);

static gboolean farsight_rtp_stream_connection_timed_out (gpointer data);
static gboolean farsight_rtp_stream_start_telephony_event (
    FarsightStream *self, guint8 ev, guint8 volume, FarsightStreamDTMFMethod method);
static gboolean farsight_rtp_stream_stop_telephony_event (
    FarsightStream *self, FarsightStreamDTMFMethod method);
static gboolean farsight_rtp_stream_preload_receive_pipeline (
    FarsightStream *self, gint payload_type);

static gboolean farsight_rtp_stream_get_jb_statistics (FarsightStream *stream,
    guint64 *total_packets,
    guint64 *late_packets,
    guint64 *duplicate_packets,
    guint   *fill_level,
    guint64 *times_overrun,
    guint64 *times_underrun);
static void farsight_rtp_stream_destroy (FarsightRTPStream *stream);


FarsightStreamClass *rtp_stream_parent_class = NULL;

typedef gboolean (*PadBlockIdlerCallback)  (GstPad *pad, gpointer user_data);

static void block_pad_and_call_idler (FarsightRTPStream *stream, GstPad *pad, PadBlockIdlerCallback function,
                                      gpointer user_data, gchar *name);
static GstPad * farsight_rtp_stream_get_source_linked_pad (FarsightRTPStream *stream);

static gchar *get_pad_full_name (GstPad *pad);

static void
blocked_cb (GstPad *pad, gboolean blocked,
    gpointer user_data);

static GType type = 0;

void farsight_rtp_stream_register_type(GTypeModule *module)
{
  static const GTypeInfo info = {
    sizeof (FarsightRTPStreamClass),
    NULL,
    NULL,
    (GClassInitFunc) farsight_rtp_stream_class_init,
    NULL,
    NULL,
    sizeof (FarsightRTPStream),
    0,
    (GInstanceInitFunc) farsight_rtp_stream_init
  };

  type = g_type_module_register_type (module, FARSIGHT_TYPE_STREAM,
      "FarsightRTPStream", &info, 0);
}

GType
farsight_rtp_stream_get_type (void)
{
  return type;
}


static void
farsight_rtp_stream_class_init (FarsightRTPStreamClass *klass)
{
  GObjectClass *gobject_class;
  FarsightStreamClass *farsight_stream_class;

  gobject_class = (GObjectClass *) klass;
  farsight_stream_class = (FarsightStreamClass *) klass;
  rtp_stream_parent_class = g_type_class_peek_parent (klass);

  gobject_class->set_property = farsight_rtp_stream_set_property;
  gobject_class->get_property = farsight_rtp_stream_get_property;

  g_object_class_install_property (gobject_class, PROP_CONN_TIMEOUT,
      g_param_spec_uint ("conn_timeout", "Connection timeout",
        "Number of secs before connection timeout", 0, G_MAXUINT, 0,
        G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_TRANSMITTER,
      g_param_spec_string ("transmitter", "Transmitter name",
        "The name of the network transmitter to use",
        NULL, G_PARAM_WRITABLE));

  g_object_class_install_property (gobject_class, PROP_MIN_PTIME,
      g_param_spec_int64 ("min-ptime", "Min packet time",
          "Minimum duration of the packet data in ns (can't go above MTU)",
          0, G_MAXINT64, DEFAULT_MIN_PTIME, 
          G_PARAM_READWRITE | G_PARAM_CONSTRUCT));

  g_object_class_install_property (gobject_class, PROP_MAX_PTIME,
      g_param_spec_int64 ("max-ptime", "Max packet time",
          "Maximum duration of the packet data in ns (-1 = unlimited up to MTU)",
          -1, G_MAXINT64, DEFAULT_MAX_PTIME, 
          G_PARAM_READWRITE | G_PARAM_CONSTRUCT));

  g_object_class_install_property (gobject_class, PROP_TRANSMITTER_OBJECT,
      g_param_spec_pointer ("transmitter-object", "Transmitter object",
        "Transmitter object used for this stream", G_PARAM_READABLE));

  g_object_class_install_property (gobject_class, PROP_CODEC_PREFS,
      g_param_spec_pointer ("codec-prefs", "Codec Preferences",
        "GList of FarsightCodecs for the preferences,"
          " just like gstcodecs.conf."
          " Except for the FARSIGHT_CODEC_ANY/DISABLED", G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_RESERVED_PT,
      g_param_spec_string ("reserved-pt-list", "List of reserved payload types",
          "A list in the form of a comma separated string with the payload types to skip for dynamic pt codecs",
          NULL, G_PARAM_WRITABLE));

  gobject_class->dispose = farsight_rtp_stream_dispose;
  gobject_class->finalize = farsight_rtp_stream_finalize;

  farsight_stream_class->prepare_transports =
      farsight_rtp_stream_prepare_transports;
  farsight_stream_class->get_native_candidate_list =
      farsight_rtp_stream_get_native_candidate_list;
  farsight_stream_class->get_native_candidate =
      farsight_rtp_stream_get_native_candidate;
  farsight_stream_class->set_remote_candidate_list =
      farsight_rtp_stream_set_remote_candidate_list;
  farsight_stream_class->add_remote_candidate =
      farsight_rtp_stream_add_remote_candidate;
  //farsight_stream_class->remove_remote_candidate =
      //farsight_rtp_stream_remove_remote_candidate;
  farsight_stream_class->set_active_candidate_pair =
      farsight_rtp_stream_set_active_candidate_pair;
  farsight_stream_class->get_local_codecs =
      farsight_rtp_stream_get_local_codecs;
  farsight_stream_class->set_remote_codecs =
      farsight_rtp_stream_set_remote_codecs;
  farsight_stream_class->get_codec_intersection =
    farsight_rtp_stream_get_codec_intersection;
  farsight_stream_class->set_codec_preference_list =
    farsight_rtp_stream_set_codec_preference_list;
  farsight_stream_class->set_active_codec =
      farsight_rtp_stream_set_active_codec;
  farsight_stream_class->get_active_codec =
      farsight_rtp_stream_get_active_codec;
  farsight_stream_class->set_sink = farsight_rtp_stream_set_sink;
  farsight_stream_class->set_sink_filter = farsight_rtp_stream_set_sink_filter;
  farsight_stream_class->get_sink = farsight_rtp_stream_get_sink;
  farsight_stream_class->set_source = farsight_rtp_stream_set_source;
  farsight_stream_class->set_source_filter =
    farsight_rtp_stream_set_source_filter;
  farsight_stream_class->get_source = farsight_rtp_stream_get_source;
  farsight_stream_class->get_pipeline = farsight_rtp_stream_get_pipeline;
  farsight_stream_class->set_pipeline = farsight_rtp_stream_set_pipeline;
  farsight_stream_class->start = farsight_rtp_stream_start;
  farsight_stream_class->stop = farsight_rtp_stream_stop;
  farsight_stream_class->set_sending = farsight_rtp_stream_set_sending;
  farsight_stream_class->hold = farsight_rtp_stream_hold;
  farsight_stream_class->unhold = farsight_rtp_stream_unhold;
  farsight_stream_class->is_held = farsight_rtp_stream_is_held;
  farsight_stream_class->start_telephony_event =
      farsight_rtp_stream_start_telephony_event;
  farsight_stream_class->stop_telephony_event =
      farsight_rtp_stream_stop_telephony_event;
  farsight_stream_class->preload_receive_pipeline =
      farsight_rtp_stream_preload_receive_pipeline;
  farsight_stream_class->get_jb_statistics =
      farsight_rtp_stream_get_jb_statistics;

  g_type_class_add_private (klass, sizeof (FarsightRTPStreamPrivate));
}

static void
farsight_rtp_stream_init (FarsightRTPStream *self)
{
  self->priv = FARSIGHT_RTP_STREAM_GET_PRIVATE (self);
  self->priv->remote_codecs = NULL;

  self->priv->codec_pref_list = NULL;

  self->priv->pipeline = NULL;
  self->priv->rtpmuxer = NULL;
  self->priv->rtpbin = NULL;
  self->priv->send_codec_bin = NULL;
  self->priv->recv_codec_bin = NULL;

  self->priv->src = NULL;
  self->priv->src_filter = NULL;
  self->priv->src_capsfilter = NULL;
  self->priv->src_valve = NULL;

  self->priv->rtpdemux = NULL;
  self->priv->sink = NULL;
  self->priv->sink_filter = NULL;
  self->priv->sink_capsfilter = NULL;

  self->priv->local_codecs = NULL;
  self->priv->remote_codecs = NULL;
  self->priv->negotiated_codecs = NULL;

  self->priv->negotiated_codec_associations = NULL;
  self->priv->local_codec_associations = NULL;
  self->priv->codecs_configuration = NULL;


  if (farsight_stream_get_direction (FARSIGHT_STREAM (self)) &
      FARSIGHT_STREAM_DIRECTION_SENDONLY)
  {
    self->priv->sending = TRUE;
  }
  else
  {
    self->priv->sending = FALSE;
  }

  self->priv->prepared = FALSE;

  self->priv->bus_watch = 0;
  self->priv->stats_timeout = 0;

  self->priv->pending_src_ids = g_array_new (FALSE, FALSE, sizeof (gint));

  self->priv->pt_caps_table = g_hash_table_new_full (g_direct_hash,
      g_direct_equal, NULL, NULL);

  self->priv->local_candidates = NULL;
  self->priv->remote_candidates = NULL;

  /* this tells _start to auto pick a codec */
  self->priv->send_codec_id = -1;
  self->priv->recv_codec_id = -1;
  self->priv->forced_send_codec_id = -1;
  self->priv->preload_recv_codec_id = -1;

  self->priv->build_send_pipeline = FALSE;

  self->priv->timeout_src = 0;
  self->priv->conn_timeout = DEFAULT_CONN_TIMEOUT;

  self->priv->min_ptime = DEFAULT_MIN_PTIME;
  self->priv->max_ptime = DEFAULT_MAX_PTIME;

  self->priv->on_hold = FALSE;

#ifdef HAVE_CLINKC
  if (!getenv ("FARSIGHT_DISABLE_UPNP"))
  {
    self->priv->use_upnp = TRUE;
    STREAM_DEBUG (self, "Setting uPnP to true");
    if (!upnp_cp_init())
    {
      STREAM_WARNING (self, "Error setting up uPnP");
      self->priv->use_upnp = FALSE;
    }
  }
#endif
}

static GList *
load_codecs_configuration (void)
{
  GList *list = NULL;
  gchar *path = NULL;

  path = g_build_filename (g_get_home_dir(), ".farsight",
      GST_CODECS_CONF_FILE, NULL);
  DEBUG ("Trying to load %s", path);
  list = farsight_codec_list_from_keyfile (path);
  g_free (path);

  if (list)
    return list;

  path = g_build_filename (SYSCONFDIR, "farsight",
          GST_CODECS_CONF_FILE, NULL);
  DEBUG ("Trying to load %s", path);
  list = farsight_codec_list_from_keyfile (path);
  g_free (path);

  return list;
}

static gboolean
ensure_local_codecs (FarsightRTPStream *self)
{
  GList *local_codecs;
  guint media_type;

  if (self->priv->local_codecs)
    return TRUE;

  g_object_get (G_OBJECT (self), "media-type", &media_type, NULL);

  STREAM_DEBUG (self, "media type is %d", media_type);

  if (!load_codecs (media_type))
  {
    STREAM_DEBUG (self, "loading codecs failed");

    farsight_stream_signal_error (FARSIGHT_STREAM (self),
        FARSIGHT_STREAM_ERROR_PIPELINE_SETUP,
        "loading codecs failed");

    return FALSE;
  }

  if (!self->priv->codecs_configuration) {
    self->priv->codecs_configuration = load_codecs_configuration ();

    if (self->priv->codecs_configuration) {
      /* Now lets remove any invalid entry */
      self->priv->codecs_configuration = validate_codecs_configuration (
          media_type, self->priv->codecs_configuration);
    } else {
      STREAM_DEBUG (self, "could not load codecs configuration file");
    }
  }

  if (self->priv->local_codec_associations) {
    g_hash_table_destroy ( self->priv->local_codec_associations);
    self->priv->local_codec_associations = NULL;
  }

  self->priv->local_codec_associations = create_local_codec_associations (
      media_type, self->priv->codecs_configuration,
      self->priv->negotiated_codec_associations, &local_codecs,
      self->priv->reserved_pt_list);

  if (self->priv->local_codec_associations) {
    self->priv->local_codecs = local_codecs;

    /* What to do with the resorting... Evil resorting done here */
    /* And only for audio.. thats even more evil */

    if (self->priv->codec_pref_list)
      sort_codecs (&self->priv->local_codecs, self->priv->codec_pref_list);

    return TRUE;
  } else {
    return FALSE;
  }
}

static void
remove_pending_mainloop_sources (FarsightRTPStreamPrivate *priv)
{
  gint i;
  if (priv->pending_src_ids->len)
  {
    for (i = 0; i < priv->pending_src_ids->len; i++)
    {
      g_source_remove (g_array_index (priv->pending_src_ids, gint, i));
    }
    g_array_remove_range (priv->pending_src_ids, 0, priv->pending_src_ids->len);
  }
}

static void
farsight_rtp_stream_dispose (GObject *object)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (object);
  guint media_type;

  if (self->priv->disposed) {
    /* If dispose did already run, return. */
    return;
  }

  self->priv->disposed = TRUE;

  farsight_rtp_stream_destroy (self);

  g_object_get (G_OBJECT (self), "media-type", &media_type, NULL);

  unload_codecs (media_type);

  remove_pending_mainloop_sources (self->priv);


  if (self->priv->sink) {
    gst_object_unref (self->priv->sink);
    self->priv->sink = NULL;
  }

  if (self->priv->sink_filter) {
    gst_caps_unref (self->priv->sink_filter);
    self->priv->sink_filter = NULL;
  }

  if (self->priv->sink_capsfilter) {
    gst_object_unref (self->priv->sink_capsfilter);
    self->priv->sink_capsfilter = NULL;
  }

  if (self->priv->src) {
    gst_object_unref (self->priv->src);
    self->priv->src = NULL;
  }

  if (self->priv->src_filter) {
    gst_caps_unref (self->priv->src_filter);
    self->priv->src_filter = NULL;
  }

  if (self->priv->src_capsfilter) {
    gst_object_unref (self->priv->src_capsfilter);
    self->priv->src_capsfilter = NULL;
  }

  if (self->priv->src_valve) {
    gst_object_unref (self->priv->src_valve);
    self->priv->src_valve = NULL;
  }


  /* chain up to parent */
  G_OBJECT_CLASS (rtp_stream_parent_class)->dispose (object);
}

static void
farsight_rtp_stream_finalize (GObject *object)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (object);

  g_return_if_fail (self != NULL);
  g_return_if_fail (FARSIGHT_IS_RTP_STREAM (self));

  if (self->priv->pending_src_ids)
    g_array_free (self->priv->pending_src_ids, TRUE);

  if (self->priv->local_candidates)
    farsight_transport_list_destroy (self->priv->local_candidates);

  if (self->priv->remote_candidates)
    farsight_transport_list_destroy (self->priv->remote_candidates);

  if (self->priv->codec_pref_list)
    g_array_free (self->priv->codec_pref_list, TRUE);

  if (self->priv->local_codecs)
    g_list_free (self->priv->local_codecs);

  if (self->priv->remote_codecs)
    farsight_codec_list_destroy (self->priv->remote_codecs);

  if (self->priv->active_native_candidate) {
    g_free (self->priv->active_native_candidate);
  }

  if (self->priv->active_remote_candidate) {
    g_free (self->priv->active_remote_candidate);
  }

  if (self->priv->pt_caps_table) {
    g_hash_table_destroy (self->priv->pt_caps_table);
  }

  if (self->priv->negotiated_codec_associations)
  {
    g_hash_table_destroy (self->priv->negotiated_codec_associations);
  }

  /* TODO free GSTelements */

  G_OBJECT_CLASS (rtp_stream_parent_class)->finalize (object);
}

static void
bin_element_set_property (GstBin *bin, const gchar *property, ...)
{
  GstIterator *it;

  if (!bin)
    return;

  it = gst_bin_iterate_elements (bin);

  if (!it)
    return;

  for(;;) {
    gpointer pt;
    GstIteratorResult res = gst_iterator_next (it, &pt);
    GstElement *elem = pt;

    if (res == GST_ITERATOR_DONE)
      break;
    else if (res == GST_ITERATOR_RESYNC) {
      gst_iterator_resync (it);
      break;
    } else if (res == GST_ITERATOR_ERROR) {
      g_error ("Error iterating contents of send_codec_bin");
      break;
    } else if (res == GST_ITERATOR_OK) {
      GObjectClass *klass;
      klass = G_OBJECT_GET_CLASS (elem);
      if (g_object_class_find_property (klass, property)) {
        va_list var_args;
        va_start (var_args, property);
        g_object_set_valist (G_OBJECT (elem), property, var_args);
        va_end (var_args);
      }
      gst_object_unref (GST_OBJECT (elem));
    }
  }
  gst_iterator_free (it);
}

static void farsight_rtp_stream_set_property (GObject * object,
    guint prop_id, const GValue * value, GParamSpec * pspec)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (object);

  switch (prop_id) {
    case PROP_CONN_TIMEOUT:
      self->priv->conn_timeout = g_value_get_uint (value);
      break;
    case PROP_TRANSMITTER:
      {
        gchar *transmitter_name = g_value_dup_string (value);

        if (self->priv->transmitter) {
          farsight_transmitter_stop (self->priv->transmitter);
          g_object_unref (G_OBJECT (self->priv->transmitter));
          self->priv->transmitter = NULL;
        }

        self->priv->transmitter = farsight_transmitter_factory_make
            (transmitter_name);

        if (!self->priv->transmitter)
        {
          STREAM_WARNING (self, "Error creating %s transmitter",
              transmitter_name);
        }
        g_free (transmitter_name);
      }
      break;
    case PROP_TRANSMITTER_OBJECT:
      STREAM_WARNING (self, "trying to manually set transmitter object");
      break;
    case PROP_MIN_PTIME:
      self->priv->min_ptime = g_value_get_int64 (value);

      bin_element_set_property (GST_BIN (self->priv->send_codec_bin),
          "min-ptime", self->priv->min_ptime, NULL);
      break;
    case PROP_MAX_PTIME:
      self->priv->max_ptime = g_value_get_int64 (value);

      bin_element_set_property (GST_BIN (self->priv->send_codec_bin),
          "max-ptime", self->priv->max_ptime, NULL);
      break;
    case PROP_CODEC_PREFS:
      {
        GList *codecs = g_value_get_pointer (value);
        guint media_type;
        GList *new_local_codecs = NULL;
        GHashTable *new_local_codec_associations = NULL;

        g_object_get (object, "media-type", &media_type, NULL);

        if (self->priv->codecs_configuration) {
          farsight_codec_list_destroy (self->priv->codecs_configuration);
          self->priv->codecs_configuration = NULL;
        }

        if (!load_codecs (media_type))
        {
          STREAM_DEBUG (self, "loading codecs failed");
          farsight_stream_signal_error (FARSIGHT_STREAM (self),
              FARSIGHT_STREAM_ERROR_PIPELINE_SETUP,
              "loading codecs failed");
          break;
        }

        self->priv->codecs_configuration = farsight_codec_list_copy (codecs);

        self->priv->codecs_configuration = validate_codecs_configuration (
            media_type, self->priv->codecs_configuration);

        new_local_codec_associations = create_local_codec_associations (
            media_type, self->priv->codecs_configuration,
            self->priv->negotiated_codec_associations, &new_local_codecs,
            self->priv->reserved_pt_list);

        if (new_local_codec_associations) {
          self->priv->local_codecs = new_local_codecs;

          if ( self->priv->local_codec_associations) {
            g_hash_table_destroy ( self->priv->local_codec_associations);
          }
          self->priv->local_codec_associations = new_local_codec_associations;

          /* What to do with the resorting... Evil resorting done here */
          /* And only for audio.. thats even more evil */

          if (self->priv->codec_pref_list)
            sort_codecs (&self->priv->local_codecs,
                self->priv->codec_pref_list);
        }

        break;
      }
    case PROP_RESERVED_PT:
      {
        gchar * pt_list = g_value_dup_string (value);
        gchar ** pt_array = NULL;
        int idx = 0;
        int size = 0;
        int pt = 0;
        gboolean valid = TRUE;

        pt_array = g_strsplit(pt_list, ",", -1);

        /* Loop, get number of pts and validate each */
        idx = 0;
        while (pt_array[idx] != NULL) {
          gchar * end_ptr = NULL;
          pt = g_ascii_strtoll (pt_array[idx], &end_ptr, 10);
          if (*end_ptr != 0 || pt < 96 || pt > 128) {
            STREAM_WARNING (self, "Invalid reserved-pt-list property value"
                " given : %s", pt_list);
            valid = FALSE;
            break;
          }
          idx++;
        }
        size = idx;

        STREAM_DEBUG (self, "Got %d reserved pts. %s", size, valid ? "valid": "");

        if (valid && size > 0) {
          if (self->priv->reserved_pt_list) {
            g_list_free (self->priv->reserved_pt_list);
          }

          for (idx = 0; idx < size; idx++) {
            pt = g_ascii_strtoll (pt_array[idx], NULL, 10);
            STREAM_DEBUG (self, "Got reserved pt : %d", pt);
            self->priv->reserved_pt_list =
                g_list_append (self->priv->reserved_pt_list, GINT_TO_POINTER (pt));
          }

          /* If no local_codec_associations, create a new one with
             the reserved PT */
          if (self->priv->local_codec_associations == NULL) {
            self->priv->local_codec_associations =
                reserved_pt_to_codec_associations (self->priv->reserved_pt_list);
          }

        }


        g_strfreev (pt_array);
        g_free (pt_list);
        break;
      }
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void farsight_rtp_stream_get_property (GObject * object,
    guint prop_id, GValue * value, GParamSpec * pspec)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (object);

  switch (prop_id) {
    case PROP_CONN_TIMEOUT:
      g_value_set_uint (value, self->priv->conn_timeout);
      break;
    case PROP_MIN_PTIME:
      g_value_set_int64 (value, self->priv->min_ptime);
      break;
    case PROP_MAX_PTIME:
      g_value_set_int64 (value, self->priv->max_ptime);
      break;
    case PROP_TRANSMITTER_OBJECT:
      g_object_ref (self->priv->transmitter);
      g_value_set_pointer (value, self->priv->transmitter);
      break;
    case PROP_CODEC_PREFS:
      g_value_set_pointer (value,
          farsight_codec_list_copy (self->priv->codecs_configuration));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static gboolean
farsight_rtp_stream_bus_watch_cb (GstBus *bus, GstMessage *message,
                                  gpointer user_data)
{
  FarsightStream *stream = FARSIGHT_STREAM (user_data);

  switch (GST_MESSAGE_TYPE (message)) {
    case GST_MESSAGE_EOS:
      {
      gchar *source_name = gst_object_get_name (message->src);
      STREAM_DEBUG (stream, "end of stream on stream pipeline from %s",
          source_name);
      g_free (source_name);
      farsight_stream_signal_error (stream,
          FARSIGHT_STREAM_ERROR_EOS, NULL);
      farsight_rtp_stream_stop (stream);
      break;
      }
    case GST_MESSAGE_ERROR:
      {
      gchar *debug = NULL;
      GError *err = NULL;

      gst_message_parse_error (message, &err, &debug);

      if (err == NULL) {
        STREAM_WARNING (stream, "gst_message_parse_error returned err == NULL");

        farsight_stream_signal_error (stream,
                                      FARSIGHT_STREAM_ERROR_UNKNOWN, NULL);
      } else {
        STREAM_WARNING (stream, "Error on stream pipeline. "
            "Error code=%d message=%s",
            err->code, err->message);
        STREAM_DEBUG (stream, "Error: %s", debug);

        g_free (debug);

        if (err->domain == GST_RESOURCE_ERROR) {
          farsight_stream_signal_error (stream,
              FARSIGHT_STREAM_ERROR_RESOURCE, err->message);
        } else {
          farsight_stream_signal_error (stream,
              FARSIGHT_STREAM_ERROR_UNKNOWN, err->message);
        }
        g_error_free (err);

      }
      farsight_rtp_stream_stop (stream);
      break;
      }
    case GST_MESSAGE_CLOCK_LOST:
      {
        FarsightRTPStream *self = FARSIGHT_RTP_STREAM (stream);
        GstStateChangeReturn ret;

        STREAM_DEBUG (self, "Clock lost, changing pipeline to PAUSED->PLAYING "
            "to find a new clock");

        /* We MUST change the pipeline's state to PAUSED then set it back to PLAYING because
         * the clock provider of the pipeline is the codec bin (in case of a dsp).
         * when we change the active codec, the clock might be lost when removing the old codec bin
         * from the pipeline. The clock is ref-ed by the pipeline and unless we change
         * the clock provider from the pipeline, it will never get destroyed. To do that, the only
         * allowed method of making the pipeline search for a new clock provider is to change state
         * from paused to playing.
         */

        STREAM_DEBUG (stream, "Changing state of the pipeline to PAUSED");
        ret = gst_element_set_state (self->priv->pipeline, GST_STATE_PAUSED);
        if (ret == GST_STATE_CHANGE_FAILURE) {
          farsight_stream_signal_error (stream,
              FARSIGHT_STREAM_ERROR_UNKNOWN, "Problem setting pipeline to paused"
              "due to lost clock");
          farsight_rtp_stream_stop (stream);
        }

        STREAM_DEBUG (self, "Changing state of the pipeline to PLAYING - %d",
            ret);
        ret = gst_element_set_state (self->priv->pipeline, GST_STATE_PLAYING);
        if (ret == GST_STATE_CHANGE_FAILURE) {
          farsight_stream_signal_error (stream,
              FARSIGHT_STREAM_ERROR_UNKNOWN, "Problem setting pipeline to playing"
              "due to lost clock");
          farsight_rtp_stream_stop (stream);
        }
      }
      break;
    case GST_MESSAGE_CLOCK_PROVIDE:
      STREAM_DEBUG (stream, "Clock provider found");
      break;
    default:
      break;
  }

  return TRUE;
}


static gboolean
farsight_rtp_stream_new_payload_type_idler (GstPad *src_pad, gpointer user_data) {
  DepayloaderChainInfo *chain_info = (DepayloaderChainInfo *) user_data;
  gboolean ret = FALSE;

  ret = farsight_rtp_stream_create_new_pt_recv_pipeline (chain_info->stream, chain_info->pt, src_pad);

  g_free (user_data);

  return ret;
}


static void
farsight_rtp_stream_new_payload_type (GstElement *element, gint pt,
                                      GstPad *pad, gpointer user_data)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (user_data);
  DepayloaderChainInfo *chain_info;

  STREAM_DEBUG (self, "Received stream with new pt %d. blocking rtpdemux", pt);

  if (gst_pad_is_blocked (pad)) {
    STREAM_WARNING (self, "We are getting a new payload type event on an "
        "already blocked pad, this shouldn't happen!");
    return;
  }

  if (self->priv->on_hold) {
    STREAM_DEBUG (self, "Received new pt, but current call is on hold."
        " Skipping codec bin creation");
    return;
  }

  chain_info = g_new0 (DepayloaderChainInfo, 1);
  chain_info->stream = (FarsightStream *)self;
  chain_info->pt = pt;

  /* We must block the rtpdemux pad before linking it*/
  block_pad_and_call_idler (self, pad, farsight_rtp_stream_new_payload_type_idler,
        chain_info, "new_payload_type");
}

static void
farsight_rtp_stream_payload_type_change (GstElement *element, gint pt,
                                         gpointer user_data)
{
  DEBUG ("received stream payload changed to %d", pt);
}

static gboolean
cleanup_unique (FarsightRTPStream *self, FarsightMediaType media_type,
    FarsightStreamDirection direction, gint unique_id)
{
  GstElement *old_codec_bin;

  old_codec_bin = get_unique_bin (media_type, direction, unique_id);

  if (old_codec_bin) {
    farsight_rtp_stream_clean_recv_codec_bin_obj (self, old_codec_bin);

    if (old_codec_bin == self->priv->recv_codec_bin) {
      self->priv->recv_codec_bin = NULL;
    }
  }

  return TRUE;
}


/*
 * This function makes sure that the codec bin has been created and returns it.
 */
static GstElement *
farsight_rtp_stream_get_or_create_recv_codec_bin (FarsightStream *stream, gint id) {

  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (stream);
  GstElement *codec_bin = NULL;
  gchar *name = NULL;
  CodecAssociation *codec_association = NULL;

  codec_association = lookup_codec_by_pt (
      self->priv->negotiated_codec_associations, id);
  if (!codec_association) {
    STREAM_WARNING (self, "Payload type %d not supported", id);
    return NULL;
  }


  if (!self->priv->pipeline) {
    STREAM_WARNING (self, "Pipeline has dissappeared, not doing anything");
    return NULL;
  }

  /* Let's see if a recv codec bin for this codec already exists */
  name = g_strdup_printf ("recv%d", id);
  codec_bin = gst_bin_get_by_name (GST_BIN(self->priv->pipeline), name);
  g_free (name);

  if (codec_bin) {
    STREAM_DEBUG (self, "Codec bin %p for PT %d exists. Returning", codec_bin,
        id);
    return codec_bin;
  } else {
    STREAM_DEBUG (self, "Codec bin for PT %d does not exist. Creating", id);

    /* Destroy any existing codec bin with the same unique id */
    if (!cleanup_unique (self, farsight_stream_get_media_type (stream),
            DIR_RECV, codec_association->codec_blueprint->receive_has_unique)) {
      STREAM_WARNING (self, "Could not unload unique codec");
      return NULL;
    }


    codec_bin = create_codec_bin (self->priv->negotiated_codec_associations,
        id, DIR_RECV, self->priv->remote_codecs);
    if (!codec_bin)
    {
      STREAM_WARNING (self, "Couldn't create elements to receive codec %d,"
          " check your gstreamer installation", id);
      return NULL;
    }

    /* This should remove the floating reference */
    if (!gst_bin_add (GST_BIN (self->priv->pipeline), codec_bin)) {
      STREAM_WARNING (self, "Could not add the codec bin to the pipeline");
      gst_object_unref (codec_bin);
      return NULL;
    }

    /* We add another reference here just to make sure the function
       behaves the same in both branches of the 'if' */
    gst_object_ref (codec_bin);

    STREAM_DEBUG (self, "Created codec bin %p. Returning", codec_bin);
    return codec_bin;
  }
}


/*
 * When called with a NULL pad, it will create the recv pipeline, but not link
 * it
 */

static gboolean
farsight_rtp_stream_create_new_pt_recv_pipeline (FarsightStream *stream,
    gint id, GstPad *pad)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (stream);
  GstElement *codec_bin = NULL;
  CodecAssociation *codec_association = NULL;
  GstPad *codec_bin_src_pad = NULL;
  GstPad *codec_bin_cnsink_pad = NULL;
  GstStateChangeReturn state_ret;

  codec_association = lookup_codec_by_pt (
      self->priv->negotiated_codec_associations, id);
  if (!codec_association) {
    STREAM_WARNING (self, "Payload type %d not supported", id);
    goto error;
  }

  STREAM_DEBUG (self, "active PT change to %d", id);

  if (!self->priv->pipeline)
  {
    STREAM_WARNING (self, "Pipeline has dissappeared, not doing anything");
    return FALSE;
  }


  if (g_strcasecmp (codec_association->codec->encoding_name, "CN") == 0) {
    STREAM_DEBUG (self, "Received CN packet on new-pt");
    if (self->priv->recv_codec_bin != NULL) {
      codec_bin_cnsink_pad = gst_element_get_static_pad (self->priv->recv_codec_bin, "cnsink");

      if (codec_bin_cnsink_pad && !gst_pad_is_linked (codec_bin_cnsink_pad)) {
        GstPadLinkReturn padlink_ret;

        STREAM_DEBUG (self, "Linking CN sink pad of codec bin to rtpdemux");
        padlink_ret = gst_pad_link (pad, codec_bin_cnsink_pad);

        if (GST_PAD_LINK_FAILED(padlink_ret)) {
          STREAM_WARNING (self, "could not link the rtpdemuxer to the "
              " codec bin CN pad");
        }
      } else {
        STREAM_DEBUG (self, "The codec_bin_cnsink_pad is already linked?");
      }
    } else {
      STREAM_DEBUG(self, "Can't link CN pad because no non-CN packet was received yet.");
    }
    return TRUE;
  }

  codec_bin = farsight_rtp_stream_get_or_create_recv_codec_bin (stream, id);

  if (codec_bin == NULL) {
    STREAM_WARNING (self, "Could not get codec bin for pt %d", id);
    goto error;
  }


  state_ret = gst_element_set_state (codec_bin, GST_STATE_READY);
  if (state_ret == GST_STATE_CHANGE_FAILURE) {
    STREAM_WARNING (self, "could not set the codec_bin to READY");
    goto error;
  }

  /* we emit now so we can set anything we need on the elements before they
   * are playing */
  farsight_stream_signal_codec_changed (stream, id);

  farsight_stream_signal_state_changed (stream,
      FARSIGHT_STREAM_STATE_CONNECTED,
      farsight_stream_get_current_direction (stream) |
      FARSIGHT_STREAM_DIRECTION_RECEIVEONLY);


  codec_bin_src_pad = gst_element_get_static_pad (codec_bin, "src");

  if (codec_bin_src_pad) {

    /* We have to create the sink_capsfilter before linking the codec bin */
    if (!self->priv->sink_capsfilter) {
      GstElement *capsfilter = gst_element_factory_make ("capsfilter", "sink_capsfilter");

      if (!capsfilter) {
        STREAM_WARNING (self, "Could not create the sink capsfilter");
        goto error;
      }

      gst_object_ref (capsfilter);

      if (!gst_bin_add (GST_BIN (self->priv->pipeline), capsfilter)) {
        STREAM_WARNING (self, "Could not add capsfilter to pipeline");
        gst_object_unref (capsfilter);
        gst_object_unref (capsfilter);
        goto error;
      }

      g_object_set (capsfilter, "caps", self->priv->sink_filter, NULL);

      self->priv->sink_capsfilter = capsfilter;

      state_ret = gst_element_set_state (self->priv->sink_capsfilter, GST_STATE_PLAYING);

      if (state_ret == GST_STATE_CHANGE_FAILURE) {
        STREAM_WARNING (self, "Setting sink capsfilter failed");
        goto error;
      }

      if (self->priv->sink) {
        GstElement *sink_parent = (GstElement *)
            gst_element_get_parent (self->priv->sink);
        if (sink_parent == NULL) {
          /* We give one ref to the bin, lets keep one */
          gst_object_ref (self->priv->sink);
          if (!gst_bin_add (GST_BIN (self->priv->pipeline), self->priv->sink)) {
            STREAM_WARNING (self, "Could not add sink to pipeline");
            goto error;
          }

          if (!gst_element_set_state (self->priv->sink, GST_STATE_PLAYING)) {
            STREAM_WARNING (self, "Sink state change to PLAYING failed!");
            goto error;
          }

          /* Now the pipeline is our parent */

          sink_parent = self->priv->pipeline;
        }

        if (sink_parent == self->priv->pipeline) {
          /* connect sink to capsfilter */
          gchar *tmp_caps = gst_caps_to_string (self->priv->sink_filter);
          STREAM_DEBUG (self, "linking sink %p to caps_filter %p with caps %s",
              self->priv->sink, self->priv->sink_capsfilter, tmp_caps);
          g_free (tmp_caps);
          if (!gst_element_link (self->priv->sink_capsfilter, self->priv->sink)) {
            STREAM_WARNING (self, "Could not link sink to capsfilter");
            goto error;
          }
        } else {

          /* capsfilter and sink have different parents, let's use our ghostpad */
          /* let's create a ghostpad for our sink */
          GstPad *ghostpad;
          GstPad *capsfilter_src_pad =
              gst_element_get_static_pad (self->priv->sink_capsfilter, "src");
          gchar *tmp_caps = NULL;

          STREAM_DEBUG (self, "Creating ghost pad for sink capsfilter");

          ghostpad = gst_ghost_pad_new ("src", capsfilter_src_pad);

          if (!ghostpad) {
            STREAM_WARNING (self, "Could not create the pipeline's src "
                "ghostpad");
            goto error;
          }

          if (!gst_pad_set_active (ghostpad, TRUE)) {
            STREAM_WARNING (self, "Could not set the src ghost pad active");
            goto error;
          }

          if (!gst_element_add_pad (self->priv->pipeline, ghostpad)) {
            STREAM_WARNING (self, "Could not add the src ghost pad to the "
                "pipeline");
            goto error;
          }
          gst_object_unref (capsfilter_src_pad);

          tmp_caps = gst_caps_to_string (self->priv->sink_filter);
          STREAM_DEBUG (self, "linking sink %p to capsfilter %p with caps %s",
              self->priv->sink, self->priv->sink_capsfilter, tmp_caps);
          g_free (tmp_caps);
          if (!gst_element_link (self->priv->pipeline, self->priv->sink)) {
            STREAM_WARNING (self, "Could not link sink to pipeline");
            goto error;
          }
        }
      }
    }

    gst_object_unref (codec_bin_src_pad);
  }



  /* If there was a previously linked codec bin, unlink it from the sink */
  if (self->priv->recv_codec_bin) {
    farsight_rtp_stream_right_unlink_recv_codec_bin (self, self->priv->recv_codec_bin);
  }

  self->priv->recv_codec_id = id;
  self->priv->recv_codec_bin = codec_bin;

  /* Then link the new codec bin to the pad */
  if ( !farsight_rtp_stream_link_recv_codec_bin_ex (self, codec_bin, pad)) {
    STREAM_DEBUG (self, "Unable to link recv codec bin");
    goto error;
  }


  state_ret = gst_element_set_state (codec_bin, GST_STATE_PLAYING);

  if (state_ret == GST_STATE_CHANGE_FAILURE) {
    STREAM_WARNING (self, "Could not set the recv codec_bin to PLAYING");
    goto error;
  }


  gst_object_unref (GST_OBJECT (codec_bin));

  /* unblock pad and continue */
  return TRUE;

error:
  {
    STREAM_WARNING (self, "PT change failed");
    farsight_rtp_stream_stop (stream);
    farsight_stream_signal_error (stream, FARSIGHT_STREAM_ERROR_PIPELINE_SETUP,
        "Error creating new recv pipeline");
    return FALSE;
  }
}

static GstElement*
farsight_rtp_stream_get_pipeline (FarsightStream *stream)
{
  FarsightRTPStream *self = (FarsightRTPStream*) stream;

  if (self->priv->main_pipeline)
  {
    return self->priv->main_pipeline;
  }
  else
  {
    return self->priv->pipeline;
  }
}

static gboolean
farsight_rtp_stream_set_pipeline (FarsightStream *stream,
    GstElement *pipeline)
{
  FarsightRTPStream *self = (FarsightRTPStream*) stream;

  if (self->priv->main_pipeline)
  {
    STREAM_WARNING (self, "Pipeline already set! Ignoring");
    return FALSE;
  }

  if (self->priv->src || self->priv->sink)
  {
    g_warning ("Can not set the main pipeline after a sink or a source"
        " has been set");
    return FALSE;
  }

  self->priv->main_pipeline = pipeline;
  return TRUE;
}

static gboolean
farsight_rtp_stream_hold (FarsightStream *stream)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (stream);
  GstState old_state;
  GstStateChangeReturn state_ret;
  GstElement *transmitter_sink = NULL;

  STREAM_DEBUG (self, "called to hold call on");

  if (self->priv->on_hold == TRUE) {
    return FALSE;
  }

  /* We must keep the transmitter src as is to allow RTCP to be
     processed/generated correctly */


  /* block the reception pipeline to avoid new codec bins being created */
  self->priv->on_hold = TRUE;


  /* We set the state of the recv codec bin and the sink to PLAYING in order to make
     sure that the sinks are not blocked in a prerolling state which will cause the
     jitterbuffer thread to wait indefinitely before being able to pause the task */
  if (self->priv->recv_codec_bin) {
    STREAM_DEBUG (self, "Setting the recv codec bin state to PLAYING");
    state_ret = gst_element_set_state (self->priv->recv_codec_bin, GST_STATE_PLAYING);
  }
  if (self->priv->sink) {
    STREAM_DEBUG (self, "Setting the sink state to PLAYING");
    state_ret = gst_element_set_state (self->priv->sink, GST_STATE_PLAYING);
  }

  /* Block the jitterbuffer thread to avoid wrong-state errors */
  STREAM_DEBUG (self, "Setting the hold property on the jitterbuffer");
  gst_child_proxy_set (GST_OBJECT (self->priv->rtpbin), "jitterbuffer::hold",
      self->priv->on_hold, NULL);

  /* clean the receive codec bin*/
  farsight_rtp_stream_clean_recv_codec_bin (self);

  /* set state GST_STATE_NULL on the sink to free resources */
  if (self->priv->sink) {
    STREAM_DEBUG (self, "Setting the sink state to NULL");
    state_ret = gst_element_set_state (self->priv->sink, GST_STATE_NULL);
    if (state_ret == GST_STATE_CHANGE_FAILURE) {
      self->priv->on_hold = FALSE;
      return FALSE;
    } else if (state_ret == GST_STATE_CHANGE_ASYNC) {
    STREAM_DEBUG (self, "Waiting");
      state_ret = gst_element_get_state (self->priv->sink, NULL, NULL,
          GST_CLOCK_TIME_NONE);
    }

    if (state_ret == GST_STATE_CHANGE_FAILURE) {
      self->priv->on_hold = FALSE;
      return FALSE;
    }
  }


  /* set state GST_STATE_NULL on the source to free resources
     and avoid a not-linked error */
  STREAM_DEBUG (self, "Setting the src state to NULL");
  if (self->priv->src && self->priv->main_pipeline == NULL) {
    /* We need to lock state of the source in case an rtcp is received
       at the same time, which will make the whole pipeline go into PLAYING
       and will revert our state change, causing a not-linked error */
    gst_element_set_locked_state (self->priv->src, TRUE);


    state_ret = gst_element_set_state (self->priv->src, GST_STATE_NULL);
    if (state_ret == GST_STATE_CHANGE_FAILURE) {
      self->priv->on_hold = FALSE;
      return FALSE;
    } else if (state_ret == GST_STATE_CHANGE_ASYNC) {
      STREAM_DEBUG (self, "Waiting");
      state_ret = gst_element_get_state (self->priv->src, NULL, NULL,
          GST_CLOCK_TIME_NONE);
    }

    if (state_ret == GST_STATE_CHANGE_FAILURE) {
      self->priv->on_hold = FALSE;
      return FALSE;
    }
  } else if (self->priv->src && self->priv->main_pipeline) {
    gst_element_unlink (self->priv->src, self->priv->pipeline);
  }

  /* Destroy the send codec bin if it exists */
  STREAM_DEBUG (self, "Cleaning the send codec bin");
  if (self->priv->send_codec_bin) {
    /* No need to pad block since we know that the source is
       now in state NULL*/
    if (farsight_rtp_stream_clean_send_codec_bin (self, &old_state) == FALSE) {
      self->priv->on_hold = FALSE;
      return FALSE;
    }
  }

  /* set state GST_STATE_READY on the sink transmitter to avoid an issue
     with the reception of the EOS */
  STREAM_DEBUG (self, "Setting the sink transmitter state to NULL");
  transmitter_sink = farsight_transmitter_get_gst_sink (self->priv->transmitter);
  if (transmitter_sink) {
    state_ret = gst_element_set_state (transmitter_sink, GST_STATE_READY);
    if (state_ret == GST_STATE_CHANGE_FAILURE) {
      self->priv->on_hold = FALSE;
      return FALSE;
    } else if (state_ret == GST_STATE_CHANGE_ASYNC) {
    STREAM_DEBUG (self, "Waiting");
      state_ret = gst_element_get_state (transmitter_sink, NULL, NULL,
          GST_CLOCK_TIME_NONE);
    }


    if (state_ret == GST_STATE_CHANGE_FAILURE) {
      self->priv->on_hold = FALSE;
      return FALSE;
    }
  }

  STREAM_DEBUG (self, "Finished holding call");

  return TRUE;
}


static gboolean
farsight_rtp_stream_unhold (FarsightStream *stream)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (stream);
  GstStateChangeReturn state_ret;
  GstElement *transmitter_sink = NULL;

  if (self->priv->on_hold == FALSE) {
    return FALSE;
  }

  if (self->priv->sink) {
    /* set sink state to GST_STATE_PLAYING */
    STREAM_DEBUG (self, "Setting the sink state to PLAYING");
    state_ret = gst_element_set_state (self->priv->sink, GST_STATE_PLAYING);
    if (state_ret == GST_STATE_CHANGE_FAILURE) {
      STREAM_WARNING (self, "Error changing the sink state to PLAYING");
      return FALSE;
    }
  }

  /* We don't need to recreate the recv codec bin. It will be created by
     the rtpdemux when the first packet arrives, by sending a new-pt signal */


  /* Restart the jitterbuffer's thread */
  STREAM_DEBUG (self, "Resetting the hold property on the jitterbuffer");
  gst_child_proxy_set (GST_OBJECT (self->priv->rtpbin), "jitterbuffer::hold",
      !self->priv->on_hold, NULL);



  /* reset state on the sink transmitter  */
  STREAM_DEBUG (self, "Setting state PLAYING on the transmitter sink");
  transmitter_sink = farsight_transmitter_get_gst_sink (self->priv->transmitter);
  if (transmitter_sink) {
    state_ret = gst_element_set_state (transmitter_sink, GST_STATE_PLAYING);
    if (state_ret == GST_STATE_CHANGE_FAILURE) {
      STREAM_WARNING (self, "Error changing the transmitter sink state"
          " to PLAYING");
      return FALSE;
    }
  }


  /* Setup the send codec bin */
  STREAM_DEBUG (self, "Setting up the send codec bin");
  if (farsight_rtp_stream_setup_send_codec_bin (self, GST_STATE_PLAYING)
      == FALSE) {
    STREAM_WARNING (self, "Error while setting up the send codec bin");
    return FALSE;
  }

  if (self->priv->src && self->priv->main_pipeline == NULL) {
    GstClockTime base_time;

    STREAM_DEBUG (self, "Setting the pipeline base_time on the source");
    base_time = gst_element_get_base_time (self->priv->pipeline);
    gst_element_set_base_time (self->priv->src, base_time);

    /* set source state to GST_STATE_PLAYING */
    STREAM_DEBUG (self, "Setting the source state to PLAYING");
    state_ret = gst_element_set_state (self->priv->src, GST_STATE_PLAYING);
    if (state_ret == GST_STATE_CHANGE_FAILURE) {
      STREAM_WARNING (self, "Error changing the source state to PLAYING");
      return FALSE;
    }

    /* Unlock state of the src */
    gst_element_set_locked_state (self->priv->src, FALSE);
  } else if (self->priv->src && self->priv->main_pipeline) {
    gst_element_link (self->priv->src, self->priv->pipeline);
  }

  /* unblock the reception pipeline and unhold the call*/
  self->priv->on_hold = FALSE;

  return TRUE;
}

static gboolean
farsight_rtp_stream_is_held (FarsightStream *stream)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (stream);

  return self->priv->on_hold;
}


static void
farsight_rtp_stream_set_active_codec (FarsightStream *stream, gint id)
{

  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (stream);
  gchar *name;
  GstElement *codec_bin = NULL;
  CodecAssociation *codec_association = NULL;
  GstState pending;
  GstState state;

  STREAM_DEBUG (self, "called to change codec from %d to %d",
      farsight_rtp_stream_get_active_codec(stream), id);

  if (!self->priv->send_codec_bin) {
    self->priv->send_codec_id = id;
    return;
  }

  if (id == farsight_rtp_stream_get_active_codec(stream))
    return;

  /* Let's check if the given codec id is valid */
  codec_association = lookup_codec_by_pt (
      self->priv->negotiated_codec_associations, id);
  if (!codec_association)
  {
    STREAM_DEBUG (self, "invalid codec id %d", id);
    return;
  }

  STREAM_DEBUG (self, "changing active send PT to %d", id);

  /* Let's see if a send codec bin for this codec already exists
   * in shouldn't really exist... */
  name = g_strdup_printf ("send%d", id);
  codec_bin = gst_bin_get_by_name (GST_BIN(self->priv->pipeline), name);
  g_free (name);
  if (codec_bin) {
    gst_object_unref (GST_OBJECT (codec_bin));
    STREAM_WARNING (self, "Send codec already exists for codec %d, "
        "this shouldn't happen", id);
    goto error;
  }

  self->priv->forced_send_codec_id = id;

  gst_element_get_state (self->priv->send_codec_bin, &state, &pending, 0);
  if (pending != GST_STATE_VOID_PENDING) {
    state = pending;
  }
  if (self->priv->src && state == GST_STATE_PLAYING) {
    GstPad *pad = farsight_rtp_stream_get_source_linked_pad (self);

    /* We must block the source pad if the source exists and is not blocked */
    if (pad) {
      block_pad_and_call_idler (self, pad,
          farsight_rtp_stream_set_active_codec_idler, self, "set_active_codec");
      gst_object_unref (GST_OBJECT (pad));
    } else {
      STREAM_DEBUG (self, "Calling idler directly without blocking source pad");
      farsight_rtp_stream_set_active_codec_idler (NULL, self);
    }

  } else {
    /* If we're not playing, then blocking the pad would block the thread forever
     * we call the idler function from here directly (since we're already in the
     * main thread) and we pass NULL as the source pad.
     */
    farsight_rtp_stream_set_active_codec_idler(NULL, stream);
  }

  return;

error:
  farsight_rtp_stream_stop (stream);
  farsight_stream_signal_error (stream, FARSIGHT_STREAM_ERROR_PIPELINE_SETUP,
      "Error while changing send codec");
}

static gboolean
farsight_rtp_stream_set_active_codec_idler(GstPad *src_pad, gpointer user_data)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (user_data);
  GstState old_state;
  guint codec_id;
  CodecAssociation *codec_association = NULL;

  DEBUG ("set_active_codec idle callback called");

  if (self->priv->forced_send_codec_id >= 0 &&
      self->priv->forced_send_codec_id == self->priv->send_codec_id) {
    STREAM_DEBUG (self, "The codec has already been changed to the requested "
        "codec %d", self->priv->forced_send_codec_id);
    goto out;
  }

  if (self->priv->forced_send_codec_id < 0) {
    codec_association = farsight_rtp_stream_choose_codec (self);
    if (!codec_association) {
      STREAM_WARNING (self, "Could not find one suitable send codec to change "
          "to");
      goto error;
    }
    codec_id = codec_association->codec->id;
  } else
    codec_id = self->priv->forced_send_codec_id;


  /* We must create the new send codec after unlinking the old one
     because a dsp can only hold one codec at a time.  */
  if (farsight_rtp_stream_clean_send_codec_bin (self, &old_state) == FALSE) {
    return FALSE;
  }

  self->priv->send_codec_id = codec_id;

  if (farsight_rtp_stream_setup_send_codec_bin (self, old_state) == FALSE) {
    return FALSE;
  }


 out:

  /* return TRUE unblocks the pad, return FALSE leaves it blocked
   * unblock the source only if the current send codec has a sink (non-dsp)
   */
  return farsight_rtp_stream_send_codec_bin_has_sink (self);

error:
  farsight_rtp_stream_stop (FARSIGHT_STREAM (self));
  farsight_stream_signal_error (FARSIGHT_STREAM (self),
      FARSIGHT_STREAM_ERROR_PIPELINE_SETUP,
      "Error while changing send codec");

  return FALSE;
}


static gint
farsight_rtp_stream_get_active_codec (FarsightStream *stream)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (stream);
  return self->priv->send_codec_id;
}

static void
blocked_cb (GstPad *pad, gboolean blocked,
    gpointer user_data)
{
  gchar *name = gst_pad_get_name (pad);
  if (blocked)
  {
    DEBUG ("Pad %s blocked successfully for %s", name, (gchar *) user_data);
  }
  else
  {
    DEBUG ("Pad %s unblocked successfully for %s", name, (gchar *) user_data);
  }
  gst_object_unref (GST_OBJECT (pad));
  g_free (name);
}

/* this function will be called when a block has succeeded, it will unlink the
 * pads and insert the new element if given in user_data.
 * Also works if the new element is a sink it implements part-block.txt on
 * dynamically switching element in PLAYING state.
 * It expects element4 to be in the correct bin
 * 
  .----------.      .----------.      .----------.
  | element1 |      | element2 |      | element3 |
 ...        src -> sink       src -> sink       ...
  '----------'      '----------'      '----------'
                    .----------.
                    | element4 |
                   sink       src
                    '----------'
*/

static gboolean
unlink_and_replace (GstPad *element1_src_pad, GstElement *element4)
{
  GstPad *element2_sink_pad = NULL;
  GstElement *element2;
  gboolean ret = FALSE;
  GstStateChangeReturn state_ret;

  DEBUG ("Blocked pad successfully, unlinking and replacing downstream");

  element2_sink_pad = gst_pad_get_peer (element1_src_pad);

  if (element2_sink_pad == NULL)
  {
    g_debug ("Pad is not linked");
    return ret;
  }

  element2 = gst_pad_get_parent_element (element2_sink_pad);

  if (element2 == NULL)
  {
    g_debug ("Linked pad does not have parent element");
    gst_object_unref (GST_OBJECT (element2_sink_pad));
    return ret;
  }

  if (!gst_pad_unlink (element1_src_pad, element2_sink_pad)) {
    WARNING ("Trying to unlink already unlinked pads %s:%s and %s:%s",
        GST_DEBUG_PAD_NAME (element1_src_pad),
        GST_DEBUG_PAD_NAME (element2_sink_pad));
  }

  if (element4)
  {
    GstElement *element1 = NULL;
    GstPad *element2_src_pad = NULL;

    element1 = gst_pad_get_parent_element (element1_src_pad);
    element2_src_pad = gst_element_get_pad (element2, "src");
    if (element2_src_pad)
    {
      GstPad *element3_sink_pad = NULL;
      GstElement *element3 = NULL;
      element3_sink_pad = gst_pad_get_peer (element2_src_pad);
      element3 = gst_pad_get_parent_element (element3_sink_pad);
      if (!gst_pad_unlink (element2_src_pad, element3_sink_pad)) {
        WARNING ("Trying to unlink already unlinked pads %s:%s and %s:%s",
            GST_DEBUG_PAD_NAME (element2_src_pad),
            GST_DEBUG_PAD_NAME (element3_sink_pad));
      }
      if (!gst_element_link (element4, element3)) {
        gchar *element4_name = gst_element_get_name (element4);
        gchar *element3_name = gst_element_get_name (element3);
        WARNING ("Could not link %s to %s", element4_name, element3_name);
        g_free (element3_name);
        g_free (element4_name);
        goto out;
      }
      gst_object_unref (GST_OBJECT (element3_sink_pad));
      gst_object_unref (GST_OBJECT (element3));
      gst_object_unref (GST_OBJECT (element2_src_pad));
    }
    if (!gst_element_link (element1, element4)) {
      gchar *element1_name = gst_element_get_name (element1);
      gchar *element4_name = gst_element_get_name (element4);
      WARNING ("Could not link %s to %s", element1_name, element4_name);
      g_free (element4_name);
      g_free (element1_name);
      goto out;
    }
    //gst_element_sync_state_with_parent (element4);
    state_ret = gst_element_set_state (element4, GST_STATE_PLAYING);
    if (state_ret == GST_STATE_CHANGE_FAILURE) {
      gchar *element4_name = gst_element_get_name (element4);
      WARNING ("Could not set %s to PLAYING", element4_name);
      g_free (element4_name);
      goto out;
    }
    gst_object_unref (GST_OBJECT (element1));
    ret = TRUE;
    gst_object_unref (GST_OBJECT (element4));
  }

 out:

  gst_object_unref (GST_OBJECT (element2_sink_pad));
  gst_object_unref (GST_OBJECT (element2));

  return ret;
}

struct _replace_sink_data
{
  FarsightRTPStream *self;
  GstElement *sink;
};

typedef struct _replace_sink_data replace_sink_data;

static gboolean
unlink_and_replace_sink (GstPad *element1_src_pad, gpointer user_data)
{
  replace_sink_data *data;
  FarsightRTPStream *self;
  GstStateChangeReturn state_ret;
  gboolean ret;

  data = (replace_sink_data *)user_data;
  self = data->self;

  DEBUG ("%p by %p", self->priv->sink, data->sink);


  ret = unlink_and_replace (element1_src_pad, data->sink);

  if (self->priv->sink)
    {
      GstElement *parent =
          (GstElement*) gst_element_get_parent (self->priv->sink);
      /* Remove first to prevent the pipeline from bringing us back up */
      if (parent == self->priv->pipeline)
      {
        if (!gst_bin_remove (GST_BIN (self->priv->pipeline), self->priv->sink)) {
          STREAM_WARNING (self, "Could not remove the sink from the pipeline");
          return ret;
        }
      }

      gst_object_unref (parent);

      /* remove old sink and update references */
      state_ret = gst_element_set_state (self->priv->sink, GST_STATE_NULL);
      if (state_ret == GST_STATE_CHANGE_ASYNC)
      {
        STREAM_DEBUG (self, "waiting for state change");
        gst_element_get_state (self->priv->sink, NULL, NULL, GST_CLOCK_TIME_NONE);
        STREAM_DEBUG (self, "done");
      }
      if (state_ret == GST_STATE_CHANGE_FAILURE) {
        STREAM_WARNING (self, "Failed while trying to set the old sink to NULL");
      }

      /* unref previous sink */
      gst_object_unref (self->priv->sink);
      self->priv->sink = NULL;
    }
  /* set new sink */
  self->priv->sink = data->sink;
  g_free (data);

  return ret;
}


static gboolean
farsight_rtp_stream_unlink_source (FarsightRTPStream *self)
{
  GstElement *src_parent = NULL;
  gboolean src_in_pipeline;

  /* we want to safely unlink the source here */
  if (!self->priv->src)
  {
    return FALSE;
  }

  /*
   * If we own the source, we stop it, then unlink it
   *
   * If the source is outside our pipeline, its owner should have stopped it
   * before killing us (through unref) or setting a new source
   */


  src_parent = (GstElement *) gst_element_get_parent (self->priv->src);
  src_in_pipeline = (src_parent == self->priv->pipeline);

  gst_object_unref (src_parent);

  if (src_in_pipeline) {
    /* In this case, we own the source */
    GstStateChangeReturn ret;

    ret = gst_element_set_state (self->priv->src, GST_STATE_NULL);

    if (ret == GST_STATE_CHANGE_ASYNC)
      ret = gst_element_get_state (self->priv->src, NULL, NULL, 2*GST_SECOND);

    if (ret == GST_STATE_CHANGE_FAILURE) {
      STREAM_WARNING (self, "Failure while setting our source to NULL");
      return FALSE;
    }
    else if (ret == GST_STATE_CHANGE_ASYNC) {
      STREAM_WARNING (self, "Setting our source to NULL returned ASYNC");
    }

    gst_element_unlink (self->priv->src, self->priv->src_valve);

  } else {
    gst_element_unlink (self->priv->src, self->priv->pipeline);
  }

  return TRUE;
}

static G_CONST_RETURN GList *
farsight_rtp_stream_get_local_codecs (FarsightStream *stream)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (stream);

  if (!ensure_local_codecs (self))
    return NULL;

  return self->priv->local_codecs;
}


static gboolean
remove_codec_idle_cb (GstPad *pad, gpointer user_data)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (user_data);
  GstStateChangeReturn ret;
  GstElement *codec_bin;
  GstPad *peer_pad;

  peer_pad = gst_pad_get_peer (pad);

  if (peer_pad) {
    codec_bin = gst_pad_get_parent_element (peer_pad);
  } else {
    codec_bin = gst_pad_get_parent_element (pad);
  }


  STREAM_DEBUG (self, "We are going to remove a recv codec bin");

  STREAM_DEBUG (self, "Setting codec bin state to NULL");
  ret = gst_element_set_state (codec_bin, GST_STATE_NULL);
  if (ret == GST_STATE_CHANGE_ASYNC)
    ret = gst_element_get_state (codec_bin, NULL, NULL, GST_CLOCK_TIME_NONE);

  if (ret == GST_STATE_CHANGE_FAILURE)
    g_error ("Error setting the codec to NULL");


  STREAM_DEBUG (self, "Unlinking codec bin");
  farsight_rtp_stream_unlink_recv_codec_bin (self, codec_bin, TRUE);

  STREAM_DEBUG (self, "Removing codec bin");
  if (!gst_bin_remove (GST_BIN(self->priv->pipeline), codec_bin)) {
    gchar *name = gst_element_get_name(codec_bin);
    gchar *cname = gst_element_get_name(self->priv->pipeline);
    g_error ("There was an error removing recv codec bin %s from container %s",
        name, cname);
    g_free (name);
    g_free (cname);
  }

  return TRUE;
}

static gboolean
remove_recv_codec (FarsightRTPStream *self, guint pt)
{
  GstElement *codec_bin;
  gchar *bin_name;

  if (!self->priv->pipeline)
    return TRUE;

  STREAM_DEBUG (self, "Remove the receive bin for pt %d because its definition"
      " has changed", pt);

  bin_name = g_strdup_printf ("recv%d", pt);
  codec_bin = gst_bin_get_by_name (GST_BIN (self->priv->pipeline), bin_name);
  g_free (bin_name);

  if (codec_bin) {
    GstPad *sink_pad;
    GstPad *peer_pad;

    sink_pad = gst_element_get_static_pad (codec_bin, "sink");

    if (!sink_pad) {
      gchar *name = gst_element_get_name (codec_bin);
      STREAM_WARNING (self, "Recv codec bin %s does not have a sink pad", name);
      g_free (name);
      return FALSE;
    }

    peer_pad = gst_pad_get_peer (sink_pad);

    if (peer_pad) {
      block_pad_and_call_idler (self, peer_pad, remove_codec_idle_cb, self,
          "remove_recv_codec_idle_cb");
    } else {
      /* Else its not linked, aka pre-loaded, lets remove it directly */
      remove_codec_idle_cb (sink_pad, self);

    }
    gst_object_unref (sink_pad);
  }

  return TRUE;
}


static void
remove_send_codec (FarsightRTPStream *self)
{
  GstElement *codec_bin;
  gchar *bin_name;

  if (!self->priv->pipeline)
    return;

  STREAM_DEBUG (self, "Scheduling the destruction of the current send codec "
      "because it's not part of the renegotiated set");

  if (self->priv->forced_send_codec_id == self->priv->send_codec_id)
    self->priv->forced_send_codec_id = -1;

  bin_name = g_strdup_printf ("send%d", self->priv->send_codec_id);

  codec_bin = gst_bin_get_by_name (GST_BIN (self->priv->pipeline), bin_name);

  g_free (bin_name);

  if (codec_bin) {
    GstPad *pad = farsight_rtp_stream_get_source_linked_pad (self);

    /* We must block the source pad if the source exists and is not blocked */
    if (pad) {
      block_pad_and_call_idler (self, pad, farsight_rtp_stream_set_active_codec_idler, self, "remove_send_codec");
      gst_object_unref (GST_OBJECT (pad));
    } else {
      STREAM_DEBUG (self, "Calling idler directly without blocking source pad");
      farsight_rtp_stream_set_active_codec_idler (NULL, self);
    }
  }
}

static gboolean
farsight_rtp_stream_set_remote_codecs (FarsightStream *stream,
    const GList *codecs)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (stream);
  GHashTable *new_negotiated_codec_associations = NULL;
  GList *new_negotiated_codecs = NULL;
  const GList *codec_list;
  GHashTable *old_pt_caps_table;
  int i;

  if (!ensure_local_codecs (self))
    return FALSE;

  codec_list = codecs;
  do {
    FarsightCodec *codec = codec_list->data;
    STREAM_DEBUG (self, "remote_codec %s %d", codec->encoding_name, codec->clock_rate);
    codec = NULL;
    codec_list = codec_list->next;
  } while (codec_list);

  new_negotiated_codec_associations = negotiate_codecs (codecs,
      self->priv->negotiated_codec_associations,
      self->priv->local_codec_associations,
      self->priv->local_codecs,
      &new_negotiated_codecs);

  if (!new_negotiated_codec_associations) {
    STREAM_WARNING (self, "Codec negociation failed, there is no intersection"
        "between remote and local codecs");
    return FALSE;
  }

  /*
   * Lets find the codecs that have gone away
   */

  /*
   * First the send codec
   */
  if (self->priv->send_codec_id >= 0) {
    CodecAssociation *old_send_codec_ca = NULL;

    old_send_codec_ca =
        lookup_codec_by_pt (self->priv->negotiated_codec_associations,
            self->priv->send_codec_id);

    if (old_send_codec_ca) {
      GList *codecitem;

      for (codecitem = g_list_first (new_negotiated_codecs);
           codecitem;
           codecitem = g_list_next (codecitem)) {
        FarsightCodec *codec = codecitem->data;
        if (codec->id == self->priv->send_codec_id) {
          if (!farsight_codec_compare (codec, old_send_codec_ca->codec)) {
            codecitem = NULL;
          }
          break;
        }
      }

      if (codecitem == NULL) {
        remove_send_codec (self);
      }
    }
  }

  /*
   * Now lets check the recv codecs
   */

  for (i = 0; i < 128; i++) {
    CodecAssociation *old_codec_ca;
    CodecAssociation *new_codec_ca;

    old_codec_ca =
        lookup_codec_by_pt (self->priv->negotiated_codec_associations, i);
    if (old_codec_ca == NULL)
      continue;

    new_codec_ca = lookup_codec_by_pt (new_negotiated_codec_associations, i);

    if (new_codec_ca == NULL ||
        !farsight_codec_compare (old_codec_ca->codec, new_codec_ca->codec)) {
      remove_recv_codec (self, i);
    }
  }


  /* we sort the negotiated codecs according to our priorities */
  /* TODO What is the correct behaviour?
   * it seems wrong to ignore the remote preferences, should they be included in
   * the reply? Should they be only kept in memory for the stream we send while
   * our reply has out preference?
   * GTalk's behaviour is to save our preference, send us what we prefer, but
   * in the intersection reply, he overwrites our preference with his. The
   * result is that we can get 2 different codecs being sent/recved. This won't
   * happen if we sort as done below. */

  /* Sorting codecs if EVIL */
  /* We should respect the remote order
   * We also have to find a way to prevent the remote guy from
   * sending us PCMA/U unless its absolutely required
   * Seems like GTalk no longer behaves like that
   */
  if (self->priv->codec_pref_list)
    sort_codecs (&self->priv->remote_codecs,self->priv->codec_pref_list);


  if (self->priv->remote_codecs)
    farsight_codec_list_destroy (self->priv->remote_codecs);
  self->priv->remote_codecs = farsight_codec_list_copy (codecs);

 if (self->priv->negotiated_codecs)
    g_list_free (self->priv->negotiated_codecs);
  self->priv->negotiated_codecs = new_negotiated_codecs;

  if (self->priv->negotiated_codec_associations)
    g_hash_table_destroy (self->priv->negotiated_codec_associations);
  self->priv->negotiated_codec_associations = new_negotiated_codec_associations;


  old_pt_caps_table = self->priv->pt_caps_table;

  self->priv->pt_caps_table =
      create_pt_caps_hashtable (self->priv->negotiated_codec_associations);

  if (self->priv->rtpbin)
  {
    g_object_set (G_OBJECT(self->priv->rtpbin), "pt-map", self->priv->pt_caps_table, NULL);
  }
  else
  {
    farsight_stream_signal_error (FARSIGHT_STREAM(self),
        FARSIGHT_STREAM_ERROR_UNKNOWN,
        "You need to run farsight_stream_prepare_transports() before setting "
        "the remote codecs");
    return FALSE;
  }


  if (old_pt_caps_table) {
    g_hash_table_destroy (old_pt_caps_table);
  }

  /* We can create the send pipeline now */
  if (!farsight_rtp_stream_build_send_pipeline (self))
    return FALSE;

  farsight_rtp_stream_set_playing (self);

  return TRUE;
}

static GList *
farsight_rtp_stream_get_codec_intersection (FarsightStream *stream)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (stream);

  if (self->priv->negotiated_codecs == NULL)
    return NULL;

  return farsight_codec_list_copy (self->priv->negotiated_codecs);
}

static void
farsight_rtp_stream_set_codec_preference_list (FarsightStream *stream,
    const GArray *codec_pref_list)
{
  FarsightRTPStream *self = (FarsightRTPStream*) stream;
  FarsightRTPStreamPrivate *priv = self->priv;

  if (codec_pref_list)
    g_return_if_fail (codec_pref_list->len);

  if (!ensure_local_codecs (self))
    return;

  /* let's free the previous list and make a copy of this one */
  if (priv->codec_pref_list)
  {
    g_array_free (priv->codec_pref_list, TRUE);
    priv->codec_pref_list = NULL;
  }

  if (codec_pref_list) {
    priv->codec_pref_list = g_array_sized_new (FALSE, FALSE, sizeof
        (FarsightCodecPreference), codec_pref_list->len);
    g_array_append_vals (priv->codec_pref_list, codec_pref_list->data,
        codec_pref_list->len);
    sort_codecs (&priv->local_codecs, priv->codec_pref_list);
  }
}

static gboolean
farsight_rtp_stream_set_source (FarsightStream *stream, GstElement *source)
{
  FarsightRTPStream *self = (FarsightRTPStream*) stream;
  gboolean ret = TRUE;
  GstStateChangeReturn state_ret;

  if (!ensure_local_codecs (self))
    return FALSE;

  if (source == self->priv->src)
    return TRUE;

  if (source)
  {
    gpointer parent = gst_element_get_parent (source);
    if (parent)
      gst_object_unref (parent);

    if (self->priv->main_pipeline && parent == NULL)
    {
      STREAM_WARNING (self, "You need to put your source in your main pipeline"
          " if you set one");
      return FALSE;
    }
    if (self->priv->main_pipeline == NULL && parent)
    {
      STREAM_WARNING (self, "Your source has a parent, but you haven't the set"
          " the main pipeline");
      return FALSE;
    }
    if (self->priv->main_pipeline && parent != self->priv->main_pipeline)
    {
      STREAM_WARNING (self, "Source has a parent, but is not the main pipeline"
          "that you have set");
      return FALSE;
    }
  }


  if (self->priv->src) {
    GstElement *src_parent = (GstElement *)gst_element_get_parent (self->priv->src);
    if (src_parent) {
      gboolean src_in_pipeline = (src_parent == self->priv->pipeline);

      if (!farsight_rtp_stream_unlink_source (self)) {
        STREAM_WARNING (self, "Could not unlink the source");
        return FALSE;
      }

      if (src_in_pipeline) {
        if (!gst_bin_remove (GST_BIN (self->priv->pipeline), self->priv->src)) {
          STREAM_WARNING (self, "Could not remove the old source from the "
              "pipeline");
          gst_object_unref (src_parent);
          return FALSE;
        }

        state_ret = gst_element_set_state (self->priv->src, GST_STATE_NULL);
        if (state_ret == GST_STATE_CHANGE_FAILURE) {
          STREAM_WARNING (self, "Could not set the source to NULL");
          gst_object_unref (src_parent);
          return FALSE;
        }
      }
      gst_object_unref (src_parent);
    }
    gst_object_unref (self->priv->src);
    self->priv->src = NULL;
  }

  STREAM_DEBUG (self, "setting src");

  if (source) {
    if (self->priv->send_codec_bin) {
      GstElement *src_parent = (GstElement *)gst_element_get_parent (source);
      if (src_parent) {
        gboolean src_in_pipeline = (src_parent == self->priv->pipeline);

        if (src_in_pipeline) {
          STREAM_WARNING (self, "Trying to add source that is already in the "
              "pipeline");
          ret = FALSE;
        } else {
          ret = gst_element_link (source, self->priv->pipeline);
        }

        gst_object_unref (src_parent);
      } else {

        gst_object_ref (source);
        if (!gst_bin_add (GST_BIN (self->priv->pipeline), source)) {
          STREAM_WARNING (self, "Could not add the source to the pipeline");
          gst_object_unref (source);
          return FALSE;
        }

        ret = gst_element_link (source, self->priv->src_valve);
      }

      if (ret)
        self->priv->src = source;

    } else {
      self->priv->src = source;
      if (self->priv->build_send_pipeline) {
        ret = farsight_rtp_stream_build_send_pipeline (self);
      }
    }
  }
  else {
    self->priv->src = NULL;
  }


  return ret;
}

static gboolean
farsight_rtp_stream_set_source_filter (FarsightStream *stream, GstCaps *filter)
{
  FarsightRTPStream *self = (FarsightRTPStream*) stream;
  STREAM_DEBUG (self, "setting source filter");

  if (self->priv->src_filter) {
    gst_caps_unref (self->priv->src_filter);
  }
  self->priv->src_filter = filter;

  if (self->priv->src_capsfilter) {
    g_object_set (self->priv->src_capsfilter, "caps", self->priv->src_filter, NULL);
  }

  if (filter) {
    gst_caps_ref(filter);
  }
  return TRUE;
}

static GstElement *
farsight_rtp_stream_get_source (FarsightStream *stream)
{
  FarsightRTPStream *self = (FarsightRTPStream*) stream;
  GstElement *codec_bin = NULL;
  GstIterator *iter;
  gchar *name;

  if (self->priv->src)
    return self->priv->src;

  g_return_val_if_fail (self->priv->pipeline != NULL, NULL);

  /* see if we have a src inside the pipeline */
  /* Let's see if a recv codec bin for this codec already exists */
  name = g_strdup_printf ("send%d", self->priv->send_codec_id);
  codec_bin = gst_bin_get_by_name (GST_BIN(self->priv->pipeline), name);
  g_free (name);

  if (codec_bin) {
    gboolean done = FALSE;
    iter = gst_bin_iterate_elements (GST_BIN(codec_bin));

    while (!done) {
      gpointer data;
      switch (gst_iterator_next (iter, &data)) {
        case GST_ITERATOR_OK:
          {
            GstElement *child;
            gboolean is_src;
            child = GST_ELEMENT_CAST (data);

            GST_OBJECT_LOCK (child);
            if (!GST_OBJECT_FLAG_IS_SET (child, GST_ELEMENT_IS_SINK) &&
                !child->numsinkpads) {
              is_src = TRUE;
            }
            else
            {
              is_src = FALSE;
            }
            GST_OBJECT_UNLOCK (child);

            gst_object_unref (child);

            if (is_src)
            {
              gst_iterator_free (iter);
              return child;
            }
            break;
          }
        case GST_ITERATOR_RESYNC:
          gst_iterator_resync (iter);
          break;
        case GST_ITERATOR_DONE:
          done = TRUE;
          break;
        case GST_ITERATOR_ERROR:
          g_assert_not_reached ();
          break;
      }
    }
    gst_iterator_free (iter);
    gst_object_unref (GST_OBJECT (codec_bin));
  }
  return NULL;
}

static GstPad *
get_sink_peer_pad (FarsightRTPStream *self, gboolean need_ghostpad)
{
  GstPad *pad;
  GstPad *ghostpad;

  /* First check if our pipeline has a ghostpad */
  pad = gst_element_get_static_pad (self->priv->pipeline, "src");
  if (pad)
    return pad;

  if (self->priv->sink_capsfilter == NULL)
    return NULL;

  /* else use the capsfilter src pad directly */
  pad = gst_element_get_static_pad ( self->priv->sink_capsfilter, "src");

  if (!need_ghostpad)
    return pad;

  ghostpad = gst_ghost_pad_new ("src", pad);

  if (!ghostpad) {
    STREAM_WARNING (self, "Could not create the pipeline's src "
        "ghostpad");
    goto error;
  }

  if (!gst_pad_set_active (ghostpad, TRUE)) {
    STREAM_WARNING (self, "Could not set the src ghost pad active");
    goto error;
  }

  if (!gst_element_add_pad (self->priv->pipeline, ghostpad)) {
    STREAM_WARNING (self, "Could not add the src ghost pad to the "
        "pipeline");
    goto error;
  }
  gst_object_unref (pad);

  return gst_object_ref (ghostpad);

 error:
  gst_object_unref (pad);
  if (ghostpad)
    gst_object_unref (ghostpad);
  return NULL;
}

static gboolean
farsight_rtp_stream_set_sink (FarsightStream *stream, GstElement *sink)
{
  FarsightRTPStream *self = (FarsightRTPStream*) stream;
  GstPad *sink_capsfilter_src_pad = NULL;
  GstStateChangeReturn state_ret;
  gchar *name = NULL;


  if (sink)
    name = gst_element_get_name (sink);
  STREAM_DEBUG (self, "setting sink %s", name ? name : "NULL");
  g_free (name);

  if (sink)
  {
    gpointer parent = gst_element_get_parent (sink);
    if (parent)
      gst_object_unref (parent);

    if (self->priv->main_pipeline && parent == NULL)
    {
      STREAM_WARNING (self, "You need to put your sink in your main pipeline"
          " if you set one");
      return FALSE;
    }
    if (self->priv->main_pipeline == NULL && parent)
    {
      STREAM_WARNING (self, "Your sink has a parent, but you haven't the set"
          " the main pipeline");
      return FALSE;
    }
    if (self->priv->main_pipeline && parent != self->priv->main_pipeline)
    {
      STREAM_WARNING (self, "Sink has a parent, but is not the main pipeline"
          " that you have set");
      return FALSE;
    }
  }

  if (!self->priv->sink_capsfilter) {
    STREAM_DEBUG (self, "No capsfilter, setting sink for future use");


    /* If there is no capsfilter, we can be sure that the sink
     * has not been added to our pipeline
     */
    if (self->priv->sink)
      gst_object_unref (self->priv->sink);

    self->priv->sink = sink;
    return TRUE;
  }

  sink_capsfilter_src_pad = get_sink_peer_pad (self,
      self->priv->main_pipeline != NULL);

  g_assert (sink_capsfilter_src_pad);

  /* no new sink */
  if (sink == NULL)
  {
    /* in this case we just want to block the recv bin and unlink the sink */
    replace_sink_data *user_data = g_new0 (replace_sink_data, 1);
    user_data->self = self;
    user_data->sink = NULL;

    STREAM_DEBUG (self, "blocking codec_bin_src_pad and removing old sink");

    block_pad_and_call_idler (self, sink_capsfilter_src_pad,
        unlink_and_replace_sink, user_data,
        "unlink_and_replace_sink for set_sink");
  }
  /* new sink provided */
  else
  {
    if (self->priv->main_pipeline == NULL)
    {
      /* Lets keep a ref before we add it */
      gst_object_ref (sink);
      if (!gst_bin_add (GST_BIN (self->priv->pipeline), sink)) {
        STREAM_WARNING (self, "Could not add the sink to the pipeline");
        gst_object_unref (sink);
        return FALSE;
      }
    }

    /* in this case we just want to unblock the recv bin and connect the new
     * sink */
    if (self->priv->sink == NULL)
    {
      GstPad *sink_pad = NULL;

      sink_pad = gst_element_get_static_pad (sink, "sink");

      if (!sink_pad)
      {
        STREAM_WARNING (self, "Could not find sink pad on sink");
        return FALSE;
      }

      gst_element_set_state (sink, GST_STATE_READY);

      if (GST_PAD_LINK_FAILED (
              gst_pad_link (sink_capsfilter_src_pad, sink_pad)))
      {
        STREAM_WARNING (self, "Could not link the sink to the capsfilter");
        gst_object_unref (sink_pad);
        return FALSE;
      }
      gst_object_unref (sink_pad);

      state_ret = gst_element_set_state (sink, GST_STATE_PLAYING);
      if (state_ret == GST_STATE_CHANGE_FAILURE) {
        STREAM_WARNING (self, "Failure while setting the sink to PLAYING");
        return FALSE;
      }

      STREAM_DEBUG (self, "unblocking sink_capsfilter_src_pad and setting new"
          " sink");
      /* Here we ignore the return value
       * If it was already unblocked, it has probably failed in some other
       * way by now. We just need to release the reference
       */
      if (!gst_pad_set_blocked_async (sink_capsfilter_src_pad, FALSE, blocked_cb,
              "set_sink new, there was none, unblock the src pad")) {
        gst_object_unref (sink_capsfilter_src_pad);
      }
      self->priv->sink = sink;
    }
    else
    {
      replace_sink_data *user_data;
      /* we want to block, remove old sink, add new sink, link and unblock */
      STREAM_DEBUG (self, "sink already present, replacing old one");
      user_data = g_new0 (replace_sink_data, 1);
      user_data->self = self;
      user_data->sink = sink;

      block_pad_and_call_idler (self, sink_capsfilter_src_pad,
          unlink_and_replace_sink, user_data,
          "unlink_and_replace_sink for set_sink");
    }
  }

  return TRUE;
}

static gboolean
farsight_rtp_stream_set_sink_filter (FarsightStream *stream, GstCaps *filter)
{
  FarsightRTPStream *self = (FarsightRTPStream*) stream;
  STREAM_DEBUG (self, "setting sink filter");

  if (self->priv->sink_filter) {
    gst_caps_unref (self->priv->sink_filter);
  }
  self->priv->sink_filter = filter;

  if (self->priv->sink_capsfilter) {
    g_object_set (self->priv->sink_capsfilter, "caps", self->priv->sink_filter, NULL);
  }

  if (filter) {
    gst_caps_ref(filter);
  }

  return TRUE;
}

static GstElement *
farsight_rtp_stream_get_sink (FarsightStream *stream)
{
  FarsightRTPStream *self = (FarsightRTPStream*) stream;
  GstElement *codec_bin = NULL;
  GstIterator *iter;
  gchar *name;

  if (self->priv->sink)
    return self->priv->sink;

  /* see if we have a sink inside the pipeline */
  /* we need a pipeline in this case */
  if (!self->priv->pipeline)
  {
    return NULL;
  }

  /* Let's see if a recv codec bin for this codec already exists */
  name = g_strdup_printf ("recv%d", self->priv->recv_codec_id);
  codec_bin = gst_bin_get_by_name (GST_BIN(self->priv->pipeline), name);
  g_free (name);

  if (codec_bin) {
    gboolean done = FALSE;
    iter = gst_bin_iterate_elements (GST_BIN(codec_bin));

    if (!iter)
      return NULL;

    while (!done) {
      gpointer data;

      switch (gst_iterator_next (iter, &data)) {
        case GST_ITERATOR_OK:
          {
            GstElement *child;
            gboolean is_sink;
            child = GST_ELEMENT_CAST (data);

            GST_OBJECT_LOCK (child);
            is_sink = GST_OBJECT_FLAG_IS_SET (child, GST_ELEMENT_IS_SINK);
            GST_OBJECT_UNLOCK (child);

            gst_object_unref (child);

            if (is_sink)
            {
              gst_iterator_free (iter);
              return child;
            }
            break;
          }
        case GST_ITERATOR_RESYNC:
          gst_iterator_resync (iter);
          break;
        case GST_ITERATOR_DONE:
          done = TRUE;
          break;
        case GST_ITERATOR_ERROR:
          g_assert_not_reached ();
          break;
      }
    }
    gst_iterator_free (iter);
    gst_object_unref (GST_OBJECT (codec_bin));
  }

  return NULL;
}

/* this build the core of the pipeline, meaning RTPBin and rtpdemux. The
 * send/recv parts are added later */
static gboolean
farsight_rtp_stream_build_base_pipeline (FarsightRTPStream *self)
{
  GstElement *rtpbin = NULL;
  GstBus *pipe_bus;

  g_return_val_if_fail (self != NULL, FALSE);

  STREAM_DEBUG (self, "creating core RTP pipeline");
  /*
   * build base pipeline
   */
  if (self->priv->pipeline == NULL) {
    GstElement *transmitter_src = NULL;

    if (self->priv->main_pipeline)
    {
      /* Create a bin and a gst bus for it and set it on the bin */
      self->priv->pipeline = gst_bin_new (NULL);
      if (!self->priv->pipeline)
      {
        goto error;
      }
      /* since our bin is independant of the parent state changes, we need to
       * make sure we manage our own async state changes inside the bin */
      /* NOTE async-handling only exists since gst 10.13 */
      if (g_object_has_property (G_OBJECT (self->priv->pipeline),
            "async-handling"))
      {
        g_object_set (G_OBJECT (self->priv->pipeline), "async-handling", TRUE,
            NULL);
      }
      /* let's make our bin independant of the parent state changes */
      gst_element_set_locked_state (self->priv->pipeline, TRUE);
      if (!gst_bin_add (GST_BIN (self->priv->main_pipeline),
              self->priv->pipeline)) {
        STREAM_WARNING (self, "Could not add our pipeline to the main pipeline");
        goto error;
      }
      pipe_bus = gst_bus_new ();
      gst_element_set_bus (GST_ELEMENT_CAST (self->priv->pipeline), pipe_bus);
    }
    else
    {
      /* Create a pipeline and listen on it's bus */
      self->priv->pipeline = gst_pipeline_new ("pipeline");
      if (!self->priv->pipeline)
      {
        goto error;
      }
      pipe_bus = gst_pipeline_get_bus (GST_PIPELINE (self->priv->pipeline));
    }
    self->priv->bus_watch = gst_bus_add_watch (pipe_bus, farsight_rtp_stream_bus_watch_cb, self);
    gst_object_unref (pipe_bus);

    /* create rtpbin element */
    rtpbin = gst_element_factory_make ("rtpbin", NULL);
    if (!rtpbin)
    {
      STREAM_WARNING (self, "Couldn't create rtpbin, "
          "check your gstreamer install");
      goto error;
    }
    self->priv->rtpbin = rtpbin;
    if (!gst_bin_add (GST_BIN (self->priv->pipeline), rtpbin)) {
      STREAM_WARNING (self, "Could not add the rtpbin to our pipeline");
      goto error;
    }

    g_object_set (G_OBJECT (rtpbin),
        "rtcp-support", FALSE,
        "pt-map", self->priv->pt_caps_table,
        NULL);

    switch(farsight_stream_get_media_type (FARSIGHT_STREAM (self))) {
      case FARSIGHT_MEDIA_TYPE_AUDIO:
        g_object_set (G_OBJECT (rtpbin), "queue-delay", 500, NULL);
        break;
      case FARSIGHT_MEDIA_TYPE_VIDEO:
        g_object_set (G_OBJECT (rtpbin), "queue-delay", 0, NULL);
        break;
    }

    /* create transmitter specific source element */
    g_object_set (G_OBJECT(rtpbin), "bypass-udp", TRUE, NULL);

    transmitter_src = farsight_transmitter_get_gst_src (self->priv->transmitter);

    if (!gst_bin_add (GST_BIN (self->priv->pipeline), transmitter_src)) {
      STREAM_WARNING (self, "Could not add the transmitter source to our "
          "pipeline");
      goto error;
    }
    STREAM_DEBUG (self, "added transmitter_src %p to pipeline %p",
        (void *)transmitter_src, (void *)self->priv->pipeline);

    /* TODO HACK ALERT HACK ALERT */
    /* just adding a boggus destination for now */
    /* so that jrtplib releases the packets to someone */
    g_object_set (G_OBJECT (rtpbin), "destinations", "64.34.23.11:5000", NULL);

    /* link rtpbin to rtpdemux to be able to discover what codec
     * should be used.
     * We don't actually connect the depayloaders decoders yet */
    self->priv->rtpdemux = gst_element_factory_make ("rtpdemux", NULL);
    if (!self->priv->rtpdemux)
    {
      STREAM_WARNING (self, "Couldn't create rtpdemux, "
          "check your gstreamer install");
      goto error;
    }
    g_signal_connect (G_OBJECT (self->priv->rtpdemux), "new-payload-type",
        G_CALLBACK (farsight_rtp_stream_new_payload_type), self);
    g_signal_connect (G_OBJECT (self->priv->rtpdemux), "payload-type-change",
        G_CALLBACK (farsight_rtp_stream_payload_type_change), self);

    if (!gst_bin_add (GST_BIN (self->priv->pipeline),
            self->priv->rtpdemux)) {
      STREAM_WARNING (self, "Could not add the rtp demuxer to our pipeline");
      goto error;
    }


    if (!gst_element_link_pads (rtpbin, "src%d", self->priv->rtpdemux,
        "sink"))
    {
      STREAM_WARNING (self, "Could not link rtpbin:src to rtpdemux:sink");
      goto error;
    }


    if (!gst_element_link_pads (transmitter_src, "src", rtpbin, "rtpsink"))
    {
      STREAM_WARNING (self, "Could not link transmitter_src:src "
          "to rtpbin:rtpsink");
      goto error;
    }
  }

  if (self->priv->preload_recv_codec_id >= 0)
    farsight_rtp_stream_preload_receive_pipeline (FARSIGHT_STREAM(self),
        self->priv->preload_recv_codec_id);

  farsight_rtp_stream_set_playing (self);

  return TRUE;

error:
  STREAM_WARNING (self, "error setting up core RTP pipeline");

  if (self->priv->pipeline) {
    if (self->priv->main_pipeline &&
        gst_element_get_parent (self->priv->pipeline))
      gst_bin_remove (GST_BIN (self->priv->main_pipeline),
          self->priv->pipeline);
    else
      gst_object_unref (GST_OBJECT (self->priv->pipeline));
    self->priv->pipeline = NULL;
  }
  farsight_stream_signal_error (FARSIGHT_STREAM(self),
          FARSIGHT_STREAM_ERROR_PIPELINE_SETUP,
          "Error setting up core RTP pipeline");
  return FALSE;
}

static gboolean
farsight_rtp_stream_has_dtmf (FarsightRTPStream *self)
{
  GList *codec_list;
  GstPluginFeature *pluginfeature = NULL;

  if (farsight_stream_get_media_type (FARSIGHT_STREAM (self)) !=
      FARSIGHT_MEDIA_TYPE_AUDIO)
    return FALSE;

  for (codec_list = self->priv->remote_codecs;
       codec_list;
       codec_list = g_list_next (codec_list)) {
    FarsightCodec *codec = codec_list->data;
    if (codec->media_type == FARSIGHT_MEDIA_TYPE_AUDIO &&
        !g_ascii_strcasecmp ("telephone-event", codec->encoding_name)) {
      STREAM_DEBUG (self, "Found audio/telephone-event for PT %d\n", codec->id);
      break;
    }
  }

  if (!codec_list) {
    /* Remote end does not adverstise telephone-event codec */
    STREAM_DEBUG (self, "Remote end does not have audio/telephone-event");
    return FALSE;
  }

  pluginfeature = gst_default_registry_find_feature("rtpdtmfsrc",
      GST_TYPE_ELEMENT_FACTORY);
  if (!pluginfeature) {
    STREAM_DEBUG (self, "The rtpdtmfsrc element is not installed");
    goto no_plugin;
  }
  gst_object_unref (pluginfeature);

  pluginfeature = gst_default_registry_find_feature("rtpdtmfmux",
      GST_TYPE_ELEMENT_FACTORY);
  if (!pluginfeature) {
    STREAM_DEBUG (self, "The rtpdtmfmux element is not installed");
    goto no_plugin;
  }
  gst_object_unref (pluginfeature);

  return TRUE;

 no_plugin:
  if (pluginfeature)
    g_object_unref (pluginfeature);

  return FALSE;
}

/*
 * Used to check PCMA/PCMU capabilities of the local and remote clients
 * arguments :
 *    remote_pcma : TRUE if the remote codecs contain PCMA
 *    remote_pcmu : TRUE if the remote codecs contain PCMU
 *    local_pcma : TRUE if the PCMA encoder and payloaders are available
 *    local_pcmu : TRUE if the PCMU encoder and payloaders are available
 *    pcma_priority : TRUE if the remote codecs have PCMA priority higher to PCMU
 *
 * return value :
 *    TRUE if local capabilities were checked
 *    FALSE if only remote capabilities were checked
 */
static gboolean
farsight_rtp_stream_check_pcm_capabilities (FarsightRTPStream *self,
                                            gboolean *remote_pcma,
                                            gboolean *remote_pcmu,
                                            gboolean *local_pcma,
                                            gboolean *local_pcmu,
                                            gboolean *pcma_priority) {
  GList *codec_list;
  GstPluginFeature *pluginfeature = NULL;

  *remote_pcma = FALSE;
  *remote_pcmu = FALSE;
  *local_pcma = FALSE;
  *local_pcmu = FALSE;
  *pcma_priority = FALSE;

  if (farsight_stream_get_media_type (FARSIGHT_STREAM (self)) !=
      FARSIGHT_MEDIA_TYPE_AUDIO)
    return FALSE;

  STREAM_DEBUG (self, "Looking for PCMA or PCMU in the remote codecs");

  for (codec_list = self->priv->remote_codecs;
       codec_list;
       codec_list = g_list_next (codec_list)) {
    FarsightCodec *codec = codec_list->data;
    if (codec->media_type == FARSIGHT_MEDIA_TYPE_AUDIO) {
      if (!g_ascii_strcasecmp ("PCMA", codec->encoding_name)) {
        *remote_pcma = TRUE;
        if (*remote_pcmu == FALSE) {
          *pcma_priority = TRUE;
        }

        STREAM_DEBUG (self, "Found PCMA codec with PT %d\n", codec->id);
      } else if (!g_ascii_strcasecmp ("PCMU", codec->encoding_name)) {
        *remote_pcmu = TRUE;
        STREAM_DEBUG (self, "Found PCMU codec with PT %d\n", codec->id);
      }
    }
  }

  if (*remote_pcma == FALSE && *remote_pcmu == FALSE) {
    /* Remote end does not adverstise PCMA/PCMU codec */
    STREAM_DEBUG (self, "Remote end does not have PCMA or PCMU codec support");
    return FALSE;
  }

  STREAM_DEBUG (self, "Found remote_pcma = %d - remote_pcmu = %d",
      *remote_pcma, *remote_pcmu);

  pluginfeature = gst_default_registry_find_feature("dtmfsrc",
      GST_TYPE_ELEMENT_FACTORY);
  if (!pluginfeature) {
    STREAM_DEBUG (self, "The dtmfsrc element is not installed");
    goto no_plugin;
  }
  gst_object_unref (pluginfeature);

  pluginfeature = gst_default_registry_find_feature("rtpdtmfmux",
      GST_TYPE_ELEMENT_FACTORY);
  if (!pluginfeature) {
    STREAM_DEBUG (self, "The rtpdtmfmux element is not installed");
    goto no_plugin;
  }
  gst_object_unref (pluginfeature);

  STREAM_DEBUG (self, "Found both the dtmfsrc and rtpdtmfmux installed");

  if (*remote_pcma) {
    pluginfeature = gst_default_registry_find_feature("alawenc",
        GST_TYPE_ELEMENT_FACTORY);
    if (!pluginfeature) {
      STREAM_DEBUG (self, "The alawenc element is not installed");
      goto no_pcma;
    }
    gst_object_unref (pluginfeature);

    pluginfeature = gst_default_registry_find_feature("rtppcmapay",
        GST_TYPE_ELEMENT_FACTORY);
    if (!pluginfeature) {
      STREAM_DEBUG (self, "The rtppcmapay element is not installed");
      goto no_pcma;
    }
    gst_object_unref (pluginfeature);
    *local_pcma = TRUE;
  }

  STREAM_DEBUG (self, "PCMA found locally? %d", *local_pcma);
 no_pcma:

  if (*remote_pcmu) {
    pluginfeature = gst_default_registry_find_feature("mulawenc",
        GST_TYPE_ELEMENT_FACTORY);
    if (!pluginfeature) {
      STREAM_DEBUG (self, "The mulawenc element is not installed");
      goto no_pcmu;
    }
    gst_object_unref (pluginfeature);

    pluginfeature = gst_default_registry_find_feature("rtppcmupay",
        GST_TYPE_ELEMENT_FACTORY);
    if (!pluginfeature) {
      STREAM_DEBUG (self, "The rtppcmupay element is not installed");
      goto no_pcmu;
    }
    gst_object_unref (pluginfeature);

    *local_pcmu = TRUE;
  }

  STREAM_DEBUG (self, "PCMU found locally? %d", *local_pcmu);
 no_pcmu:

  return TRUE;

 no_plugin:
  if (pluginfeature)
    g_object_unref (pluginfeature);

  return FALSE;
}

static gboolean
farsight_rtp_stream_has_pcm (FarsightRTPStream *self)
{
  gboolean ret = FALSE;
  gboolean remote_pcma = FALSE;
  gboolean remote_pcmu = FALSE;
  gboolean local_pcma = FALSE;
  gboolean local_pcmu = FALSE;
  gboolean pcma_priority = FALSE;

  ret = farsight_rtp_stream_check_pcm_capabilities (self,
      &remote_pcma,
      &remote_pcmu,
      &local_pcma,
      &local_pcmu,
      &pcma_priority);

  return (ret && (local_pcma || local_pcmu));

}


GstElement *
build_dtmf_tone_generator (FarsightRTPStream *self)
{
  gboolean ret = FALSE;
  gboolean remote_pcma = FALSE;
  gboolean remote_pcmu = FALSE;
  gboolean local_pcma = FALSE;
  gboolean local_pcmu = FALSE;
  gboolean pcma_priority = FALSE;
  gboolean build_pcma = FALSE;

  GstElement *dtmf_bin = NULL;
  GstElement *dtmfsrc = NULL;
  GstElement *encoder = NULL;
  GstElement *payloader = NULL;
  GstPad *bin_ghostpad = NULL;
  GstPad *payloader_src_pad = NULL;

  ret = farsight_rtp_stream_check_pcm_capabilities (self,
      &remote_pcma,
      &remote_pcmu,
      &local_pcma,
      &local_pcmu,
      &pcma_priority);

  if (ret) {
    if (pcma_priority && local_pcma) {
      build_pcma = TRUE;
    } else if (!pcma_priority && local_pcmu) {
      build_pcma = FALSE;
    } else if (!pcma_priority && local_pcma) {
      build_pcma = TRUE;
    } else if (pcma_priority && local_pcmu) {
      build_pcma = FALSE;
    }

    dtmf_bin = gst_bin_new ("dtmftonegen");
    if (!dtmf_bin) {
      STREAM_WARNING (self, "Error creating dtmfsrc element");
      goto error;
    }

    dtmfsrc = gst_element_factory_make ("dtmfsrc", "dtmfsrc");
    if (!dtmfsrc) {
      STREAM_WARNING (self, "Error creating dtmfsrc element");
      goto error;
    }

    if (build_pcma) {
      encoder = gst_element_factory_make ("alawenc", "dtmf_alawenc");
      if (!encoder) {
        STREAM_WARNING (self, "Error creating alawenc element");
        goto error;
      }

      payloader = gst_element_factory_make ("rtppcmapay", "dtmf_rtppcmapay");
      if (!payloader) {
        STREAM_WARNING (self, "Error creating rtppcmapay element");
        goto error;
      }
    } else {
      encoder = gst_element_factory_make ("mulawenc", "dtmf_mulawenc");
      if (!encoder) {
        STREAM_WARNING (self, "Error creating mulawenc element");
        goto error;
      }

      payloader = gst_element_factory_make ("rtppcmupay", "dtmf_rtppcmupay");
      if (!payloader) {
        STREAM_WARNING (self, "Error creating rtppcmupay element");
        goto error;
      }
    }

    gst_bin_add_many (GST_BIN (dtmf_bin), dtmfsrc, encoder, payloader, NULL);

    if (!gst_element_link_pads (dtmfsrc, "src", encoder, "sink")) {
      STREAM_WARNING (self, "Could not link dtmf source pad to encoder");
      goto error;
    }
    if (!gst_element_link_pads (encoder, "src", payloader, "sink")) {
      STREAM_WARNING (self, "Could not link dtmf encoder to payloader");
      goto error;
    }

    payloader_src_pad = gst_element_get_static_pad (payloader, "src");

    if (payloader_src_pad) {
      bin_ghostpad = gst_ghost_pad_new ("src", payloader_src_pad);
      if (bin_ghostpad) {
        gst_pad_set_active (bin_ghostpad, TRUE);

        if (!gst_element_add_pad (dtmf_bin, bin_ghostpad)) {
          STREAM_WARNING (self, "Could not add DTMF ghost pad to tone "
              "generator bin");
          gst_object_unref (payloader_src_pad);
          gst_object_unref (bin_ghostpad);
          goto error;
        }
        gst_object_unref (payloader_src_pad);
      } else {
        STREAM_WARNING (self, "Could not create DTMF tone generator ghost pad");
        goto error;
      }
    } else {
      STREAM_WARNING (self, "DTMF payloader has no 'src' pad");
      goto error;
    }
  }

  return dtmf_bin;
 error:

  /* Any error that might occur would happen before the elements
   * get added to the bin, so we need to remove them one by one
   */
  if(dtmf_bin)
    gst_object_unref (dtmf_bin);
  if(dtmfsrc)
    gst_object_unref (dtmfsrc);
  if(encoder)
    gst_object_unref (encoder);
  if(payloader)
    gst_object_unref (payloader);

  return NULL;
}


GstElement *
build_dtmf_rtpdtmfsrc (FarsightRTPStream *self)
{
  GList *codec_list;
  GstElement *dtmfsrc = NULL;
  guint pt;

  for (codec_list = self->priv->remote_codecs;
       codec_list;
       codec_list = g_list_next (codec_list)) {
    FarsightCodec *codec = codec_list->data;
    if (codec->media_type == FARSIGHT_MEDIA_TYPE_AUDIO &&
        !g_ascii_strcasecmp ("telephone-event", codec->encoding_name)) {
      pt = codec->id;
      break;
    }
  }

  if (!codec_list)
    return FALSE;

  dtmfsrc = gst_element_factory_make ("rtpdtmfsrc", "rtpdtmfsrc");
  if (!dtmfsrc) {
    STREAM_WARNING (self, "Error creating rtpdtmfsrc element");
    goto error;
  }

  g_object_set (dtmfsrc,
      "pt", pt,
      "interval", 30,
      "packet-redundancy", 3, NULL);

  return dtmfsrc;

 error:
  if (dtmfsrc)
    gst_object_unref (dtmfsrc);

  return NULL;

}

gboolean
g_object_has_property (GObject *object, const gchar *property)
{
  GObjectClass *klass;

  klass = G_OBJECT_GET_CLASS (object);
  return NULL != g_object_class_find_property (klass, property);
}

static gboolean
farsight_rtp_stream_build_send_pipeline (FarsightRTPStream *self)
{
  CodecAssociation *codec_association = NULL;
  GstElement *rtpmuxer = NULL;
  GstElement *rtpdtmfsrc = NULL;
  GstElement *dtmfsrc = NULL;
  GstElement *transmitter_sink = NULL;
  GstElement *src_parent = NULL;
  gboolean src_in_pipeline = FALSE;
  GstStateChangeReturn state_ret;
  GstElement *transmitter_async_handling_bin;
  GstPad *pad;

  g_return_val_if_fail (self != NULL, FALSE);
  /* let's create the base pipeline if not already done so (prepare_transports
   * not called) */
  if (!self->priv->rtpbin || !self->priv->pipeline)
  {
    farsight_rtp_stream_build_base_pipeline (self);
  }

  g_return_val_if_fail (self->priv->rtpbin != NULL, FALSE);
  g_return_val_if_fail (self->priv->pipeline != NULL, FALSE);

  if (self->priv->send_codec_bin || self->priv->on_hold)
  {
    STREAM_WARNING (self, "Send pipeline already created, will not recreate");
    return TRUE;
  }

  /* Let us automaticaly pick a codec if not set by the user */
  /* TODO farsight_rtp_stream_choose_codec() will pick the first codec from the
   * remote codec list that it finds in the local internal (rtpgstcodecs) list.
   * Should we actually pick the first codec in our list? If so, we need to use
   * the self->local_codecs list since the internal one is not sorted according
   * to codec_prefs */
  if (self->priv->send_codec_id == -1)
  {
    codec_association = farsight_rtp_stream_choose_codec (self);
    if (!codec_association)
      return FALSE;

    self->priv->send_codec_id = codec_association->codec->id;
  }
  else
  {
    /* let's get the CodecBlueprint for the selected PT */
    codec_association = lookup_codec_by_pt (
        self->priv->negotiated_codec_associations, self->priv->send_codec_id);
    if (!codec_association) {
      STREAM_WARNING (self, "Codec %d not supported",
          self->priv->send_codec_id);
      goto error;
    }
  }

  STREAM_DEBUG (self, "creating send pipeline with codec %d",
      self->priv->send_codec_id);

  if (!self->priv->src && !codec_association->codec_blueprint->has_src)
  {
    STREAM_DEBUG (self, "No source has been set yet, "
        "send pipeline built for later");
    self->priv->build_send_pipeline = TRUE;
    return FALSE;
  }

  /* add active remote candidate as destination if set */
  if (self->priv->active_remote_candidate)
  {
    farsight_rtp_stream_add_remote_candidate_to_rtpbin(self,
        self->priv->active_remote_candidate);
  }

  /*
   * sending part of pipeline
   */

  if (self->priv->src)
  {
    /*
     * Check if the source must go into our pipeline
     */
    src_parent = (GstElement *) gst_element_get_parent (self->priv->src);
    if (src_parent)
      gst_object_unref (src_parent);
    src_in_pipeline = !src_parent;

    /* let's add the source to our bin if there is no user provided pipeline or
     * if the given source is not added to the user pipeline */
    if ((self->priv->main_pipeline && src_in_pipeline)
        || !self->priv->main_pipeline) {
      /* We give one ref to the bin, lets keep one */
      gst_object_ref (self->priv->src);
      if (!gst_bin_add (GST_BIN (self->priv->pipeline), self->priv->src)) {
        STREAM_WARNING (self, "Could not add source to pipeline");
        goto error;
      }
    }
  }


  /* Lets build a Muxer in the case we have to add DTMF or some other
   * secondary codec
   */
  rtpmuxer = gst_element_factory_make ("rtpdtmfmux", NULL);
  if (!rtpmuxer) {
    STREAM_WARNING (self, "Error creating rtpdtmfmux element");
    goto error;
  }

  if (!gst_bin_add (GST_BIN (self->priv->pipeline), rtpmuxer)) {
    STREAM_WARNING (self, "Could not add RTP muxer to the pipeline");
    goto error;
  }

  if (!gst_element_link_pads (rtpmuxer, "src",
          self->priv->rtpbin, "sink%d")) {
    STREAM_WARNING (self, "Could not link rtpmuxer and rtpbin\n");
    goto error;
  }

   if (farsight_rtp_stream_has_dtmf (self)) {

     rtpdtmfsrc = build_dtmf_rtpdtmfsrc (self);
     if (!rtpdtmfsrc) {
       STREAM_WARNING (self, "Could not build rtp dtmf source element\n");
     } else {
       if (!gst_bin_add (GST_BIN (self->priv->pipeline), rtpdtmfsrc)) {
         STREAM_WARNING (self, "Could not add rtp dtmf source to pipeline");
       } else {
         if (!gst_element_link_pads (rtpdtmfsrc, "src" , rtpmuxer, "sink_%d")) {
           STREAM_WARNING (self, "Could not link rtp dtmf source to rtpmuxer");
         }
       }
     }
   }

   if (farsight_rtp_stream_has_pcm (self)) {
     dtmfsrc = build_dtmf_tone_generator (self);

     if (!dtmfsrc) {
       STREAM_WARNING (self, "Could not build dtmf source element\n");
     } else {
       if (!gst_bin_add (GST_BIN (self->priv->pipeline), dtmfsrc)) {
         STREAM_WARNING (self, "Could not add dtmf source to pipeline");
       } else {
         if (!gst_element_link_pads (dtmfsrc, "src" , rtpmuxer, "sink_%d")) {
           STREAM_WARNING (self, "Could not link dtmf source to rtpmuxer");
         }
       }
     }
   }

  self->priv->rtpmuxer = rtpmuxer;

  /* create transmitter specific sink element */

  transmitter_sink = farsight_transmitter_get_gst_sink
      (self->priv->transmitter);

  transmitter_async_handling_bin = gst_bin_new ("transmitter-sink-bin");

  g_object_set (transmitter_async_handling_bin, "async-handling", TRUE, NULL);

  pad = gst_element_get_static_pad (transmitter_sink, "sink");

  if (!pad)
  {
    g_warning ("Could not find sink pad in sink");
    goto error;
  }


  if (!gst_bin_add (GST_BIN (transmitter_async_handling_bin),
          transmitter_sink)) {
    STREAM_WARNING (self, "Could not add transmitter sink to its"
        " async-handling bin");
    goto error;
  }


  if (!gst_element_add_pad (transmitter_async_handling_bin,
          gst_ghost_pad_new ("sink", pad)))
  {
    STREAM_WARNING (self, "Could not add pad to bin");
    goto error;
  }

  gst_object_unref (pad);


  if (!gst_bin_add (GST_BIN (self->priv->pipeline),
          transmitter_async_handling_bin))
  {
    STREAM_WARNING (self, "Could not add transmitter sink to pipeline");
    goto error;
  }

  transmitter_sink = transmitter_async_handling_bin;

  if (!gst_element_link_pads (self->priv->rtpbin, "rtpsrc",
        transmitter_sink, "sink"))
  {
    STREAM_WARNING (self, "Could not link rtpbin:rtpsrc to "
        "transmitter_sink:sink");
    goto error;
  }

  /* Now we need to build our capsfilter and link it to the source */
  if (self->priv->src)
  {
    GstElement *capsfilter = gst_element_factory_make ("capsfilter", "source_capsfilter");
    GstElement *valve = NULL;

    if (!capsfilter) {
      STREAM_WARNING (self, "Could not create the source capsfilter");
      goto error;
    }

    gst_object_ref (capsfilter);

    if (!gst_bin_add (GST_BIN (self->priv->pipeline), capsfilter)) {
      STREAM_WARNING (self, "Could not add source caps filter to pipeline");
      goto error;
    }


    g_object_set (capsfilter, "caps", self->priv->src_filter, NULL);

    self->priv->src_capsfilter = capsfilter;

    valve = gst_element_factory_make ("fsvalve", NULL);

    if (!valve) {
      STREAM_WARNING (self, "Valve not installed,"
          " you need gst-plugins-farsight >= 0.12.6");
      goto error;
    }

    g_object_set (G_OBJECT (valve), "drop", !self->priv->sending, NULL);

    gst_object_ref (valve);

    if (!gst_bin_add (GST_BIN (self->priv->pipeline), valve)) {
      STREAM_WARNING (self, "Could not add valve to pipeline");
      goto error;
    }

    self->priv->src_valve = valve;

    if (!gst_element_link (valve, capsfilter)) {
      STREAM_WARNING (self, "Could not link valve to capsfilter");
      goto error;
    }

    if (src_in_pipeline)
    {
      /* connect src to valve */
      gchar *tmp_caps = gst_caps_to_string (self->priv->src_filter);
      STREAM_DEBUG (self, "linking src %p to valve %p with caps %s", self->priv->src,
               self->priv->src_valve, tmp_caps);
      g_free (tmp_caps);
      if (!gst_element_link (self->priv->src, self->priv->src_valve))
      {
        STREAM_WARNING (self, "Could not link src to valve");
        goto error;
      }
    }
    else
    {

      /* valve and source have different parents, let's use our ghostpad */
      /* let's create a ghostpad for our sink */
      GstPad *ghostpad;
      GstPad *valve_sink_pad = gst_element_get_static_pad (valve, "sink");
      gchar *tmp_caps = NULL;

      STREAM_DEBUG (self, "Creating ghost pad for vale");

      ghostpad = gst_ghost_pad_new ("sink", valve_sink_pad);
      if (!gst_pad_set_active (ghostpad, TRUE)) {
        STREAM_WARNING (self, "Could not set the pipeline's sink ghostpad "
            "active");
        goto error;
      }
      if (!gst_element_add_pad (self->priv->pipeline, ghostpad)) {
        STREAM_WARNING (self, "Could not add the sink ghostpad to our "
            "pipeline");
        goto error;
      }
      gst_object_unref (valve_sink_pad);

      tmp_caps = gst_caps_to_string (self->priv->src_filter);
      STREAM_DEBUG (self, "adding valve %p with caps %s for src",
               self->priv->src_valve, tmp_caps);
      g_free (tmp_caps);
    }
  }


  if (!self->priv->on_hold) {
    farsight_rtp_stream_setup_send_codec_bin (self, GST_STATE_NULL);
  }

  self->priv->build_send_pipeline = FALSE;

  /* if user requested to block the pad, let's do it */
  if (self->priv->sending == FALSE)
  {
    self->priv->sending = TRUE;
    farsight_rtp_stream_set_sending (FARSIGHT_STREAM (self), FALSE);
  }


  /*
   * The live source is inside our bin, let the bin manage the states.
   * Otherwise, we need to do it ourselves.
   */

  if (!src_in_pipeline && self->priv->src) {
    state_ret = gst_element_set_state (transmitter_sink, GST_STATE_PLAYING);
    if (state_ret == GST_STATE_CHANGE_FAILURE) {
      STREAM_WARNING (self, "Could not set the transmitter sink to PLAYING");
      goto error;
    }

    state_ret = gst_element_set_state (rtpmuxer, GST_STATE_PLAYING);
    if (state_ret == GST_STATE_CHANGE_FAILURE) {
      STREAM_WARNING (self, "Could not set the rtp muxer to PLAYING");
      goto error;
    }

    if (self->priv->src_capsfilter) {
      state_ret = gst_element_set_state(self->priv->src_capsfilter,
          GST_STATE_PLAYING);
      if (state_ret == GST_STATE_CHANGE_FAILURE) {
        STREAM_WARNING (self, "Could not set the source capsfilter to PLAYING");
        goto error;
      }
    }

    if (self->priv->src_valve) {
      state_ret = gst_element_set_state(self->priv->src_valve, GST_STATE_PLAYING);
      if (state_ret == GST_STATE_CHANGE_FAILURE) {
        STREAM_WARNING (self, "Could not set the valve to PLAYING");
        goto error;
      }
    }

    if (dtmfsrc) {
      if (!gst_element_sync_state_with_parent (dtmfsrc)) {
        STREAM_WARNING (self, "Could not sync the dtmfsrc's state with its "
            "parent");
        goto error;
      }
    }

    if (rtpdtmfsrc) {
      if (!gst_element_sync_state_with_parent (rtpdtmfsrc)) {
        STREAM_WARNING (self, "Could not sync the rtpdtmfsrc's state with its "
            "parent");
        goto error;
      }
   }

    if (self->priv->send_codec_bin &&
        !gst_element_sync_state_with_parent (self->priv->send_codec_bin)) {
      STREAM_WARNING (self, "Could not sync the send codec bin's state with its"
          " parent");
       goto error;
    }
  }

  farsight_rtp_stream_set_playing (self);

  return TRUE;

error:
  STREAM_WARNING (self, "%s (%d): error setting up send codec pipeline",
      __FUNCTION__, __LINE__);

  /*
   * No error signal, this always returns a sync error
   *
  farsight_stream_signal_error (FARSIGHT_STREAM (self),
      FARSIGHT_STREAM_ERROR_PIPELINE_SETUP,
      "Could not build the send pipeline");
  */

  return FALSE;
}

static gboolean
farsight_rtp_stream_start (FarsightStream *stream)
{
  FarsightRTPStream *self = (FarsightRTPStream *) stream;

  g_return_val_if_fail (self != NULL, FALSE);
  if (self->priv->pipeline == NULL ||
      self->priv->rtpbin == NULL ||
      farsight_stream_get_state (stream) != FARSIGHT_STREAM_STATE_CONNECTED)
  {
    return FALSE;
  }

  return TRUE;
}


static void
farsight_rtp_stream_try_set_playing (FarsightRTPStream *self)
{
  guint src_id = 0;

  if (farsight_stream_get_state (FARSIGHT_STREAM (self)) ==
      FARSIGHT_STREAM_STATE_CONNECTED &&
      self->priv->remote_codecs != NULL &&
      self->priv->pipeline != NULL) {
    src_id = g_idle_add_full (G_PRIORITY_HIGH, farsight_rtp_stream_set_playing, self, NULL);
    if (src_id)
      g_array_append_val (self->priv->pending_src_ids, src_id);
    else
      STREAM_WARNING (self, "Could not add idle task for "
          "farsight_rtp_stream_set_playing");
  }
}

static gboolean
farsight_rtp_stream_set_playing (gpointer data)
{
  FarsightRTPStream *stream = FARSIGHT_RTP_STREAM (data);
  GstStateChangeReturn rv;

  if (stream->priv->pipeline == NULL ||
      stream->priv->remote_codecs == NULL ||
      farsight_stream_get_state (FARSIGHT_STREAM (stream)) !=
      FARSIGHT_STREAM_STATE_CONNECTED)
    return FALSE;

  STREAM_DEBUG (stream, "We are now trying to go PLAYING");


  if (stream->priv->src && stream->priv->send_codec_bin)
  {

    if ((GstElement *)gst_element_get_parent (stream->priv->src) ==
        stream->priv->pipeline)
    {
      /* The source is in our pipeline, its already linked */

      rv = gst_element_set_state (stream->priv->pipeline, GST_STATE_PLAYING);
      if (rv == GST_STATE_CHANGE_FAILURE) {
        STREAM_WARNING (stream, "Failed to set the pipeline to PLAYING");
      }
    }
    else
    {
      GstPad *pad = gst_element_get_static_pad (stream->priv->pipeline, "sink");

      g_assert (pad);

      STREAM_DEBUG (stream, "External source: first set our bin to PLAYING");

      rv = gst_element_set_state (stream->priv->pipeline, GST_STATE_PLAYING);

      if (rv == GST_STATE_CHANGE_FAILURE) {
        WARNING ("Setting the pipeline to playing returned failure");
        return FALSE;
      }

      STREAM_DEBUG (stream, "Returned %d", rv);

      if (!gst_pad_is_linked (pad)) {
        STREAM_DEBUG (stream, "Lets now link the source");

        if (!gst_element_link_pads (stream->priv->src, NULL,
                stream->priv->pipeline, "sink")) {
          STREAM_WARNING (stream, "Could not link src to pipeline");
          return FALSE;
        }
      }

      gst_object_unref (pad);

    }
  }
  else
  {
    rv = gst_element_set_state (stream->priv->pipeline, GST_STATE_PLAYING);
    if (rv == GST_STATE_CHANGE_FAILURE) {
      STREAM_WARNING (stream, "Setting the pipeline to PLAYING returned an error");
    }
  }



  return FALSE;
}


/**
 * Stop a #FarsightRTPStream instance.
 *
 * @param stream #FarsightRTPStream instance
 */
static void
farsight_rtp_stream_stop (FarsightStream *stream)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (stream);

  g_return_if_fail (stream != NULL);

  farsight_rtp_stream_set_sending (FARSIGHT_STREAM (self), FALSE);

  if (self->priv->timeout_src)
  {
    if (g_source_remove (self->priv->timeout_src) == FALSE) {
      WARNING ("Tried to remove non-existent source %u",
          self->priv->timeout_src);
    }
    self->priv->timeout_src = 0;
  }
}



/**
 * Destroy a rtp stream instance
 *
 * This must be called once farsight as been unlinked from the
 * main pipeline if there was one
 */
static void
farsight_rtp_stream_destroy (FarsightRTPStream *self)
{
  g_return_if_fail (self != NULL);

  /* let's unlink from the source first */
  farsight_rtp_stream_unlink_source (self);

  if (self->priv->pipeline)
  {
    GstStateChangeReturn state_set_return;

    STREAM_DEBUG (self, "stopping media pipeline");

    STREAM_DEBUG (self, "Setting state to NULL");
    state_set_return = gst_element_set_state (self->priv->pipeline,
        GST_STATE_NULL);
    STREAM_DEBUG (self, "DONE Setting state to NULL returned %d", 
        state_set_return);

    if (state_set_return == GST_STATE_CHANGE_ASYNC)
    {
      STREAM_DEBUG (self, "Getting state");
      state_set_return = gst_element_get_state (self->priv->pipeline, NULL, NULL,
          5 * GST_SECOND);
      STREAM_DEBUG (self, "DONE Getting state");
    }

    switch(state_set_return)
    {
      case GST_STATE_CHANGE_FAILURE:
        STREAM_WARNING (self, "Unable to set pipeline to NULL! This could break"
            "the teardown");
        break;
      case GST_STATE_CHANGE_ASYNC:
        STREAM_WARNING (self, "State change not finished, after 5 seconds."
            " This could  break the teardown");
        break;
      case GST_STATE_CHANGE_SUCCESS:
        STREAM_DEBUG (self, "Changed pipeline state to NULL succesfully");
        break;
      default:
        break;
    }

    if (self->priv->bus_watch)
      g_source_remove (self->priv->bus_watch);

    if (!self->priv->main_pipeline)
    {
      gst_object_unref (GST_OBJECT (self->priv->pipeline));
    }
    else
    {
      if (!gst_bin_remove (GST_BIN (self->priv->main_pipeline),
              self->priv->pipeline)) {
        STREAM_WARNING (self, "Could not remove our bin from the main pipeline");
      }
    }
    self->priv->pipeline = NULL;
    self->priv->rtpbin = NULL;

    if (self->priv->stats_timeout)
      g_source_remove (self->priv->stats_timeout);
  }

  remove_pending_mainloop_sources (self->priv);

  if (self->priv->transmitter) {
    farsight_transmitter_stop (self->priv->transmitter);
    g_object_unref (G_OBJECT (self->priv->transmitter));
    self->priv->transmitter = NULL;
  }

  if (self->priv->use_upnp)
  {
    farsight_rtp_stream_upnp_close_ports (self);
#ifdef HAVE_CLINKC
    upnp_cp_shutdown();
#endif
  }

  farsight_stream_signal_state_changed (FARSIGHT_STREAM (self),
      FARSIGHT_STREAM_STATE_DISCONNECTED,
      FARSIGHT_STREAM_DIRECTION_NONE);
}

static gboolean
farsight_rtp_stream_set_sending (FarsightStream *stream, gboolean sending)
{
  FarsightRTPStream *self = (FarsightRTPStream *) stream;

  GstElement *codec_bin = NULL;
  GstPad *codec_bin_src_pad  = NULL;
  gchar *name = NULL;

  if (self->priv->sending == sending)
  {
    return TRUE;
  }

  self->priv->sending = sending;

  if (self->priv->pipeline == NULL)
  {
    STREAM_WARNING (self, "No pipeline present, will set sending later");
    return FALSE;
  }

  name = g_strdup_printf ("send%d", self->priv->send_codec_id);
  codec_bin = gst_bin_get_by_name (GST_BIN(self->priv->pipeline), name);
  g_free (name);

  if (codec_bin)
  {
    if (!self->priv->src_valve) {
      codec_bin_src_pad = gst_element_get_static_pad (codec_bin, "src");
      if (!codec_bin_src_pad)
      {
        g_error ("send codec has no source pad! This shouldn't happen");
      }
      gst_object_unref (GST_OBJECT (codec_bin));
    }
  }
  else
  {
    g_message ("send codec bin not created yet, will set sending later");
    return TRUE;
  }

  if (self->priv->src_valve) {
    g_object_set (self->priv->src_valve, "drop", !sending, NULL);
  }

  if (sending)
  {
    STREAM_DEBUG (self, "Setting sending to %d", sending);
    if (!self->priv->src_valve) {
      if (!gst_pad_set_blocked_async (codec_bin_src_pad, FALSE, blocked_cb,
              (gpointer) __FUNCTION__))
        gst_object_unref (codec_bin_src_pad);
    }

    /* only advertise sending if we're CONNECTED */
    if (farsight_stream_get_state (stream) == FARSIGHT_STREAM_STATE_CONNECTED)
      farsight_stream_signal_state_changed (stream,
          FARSIGHT_STREAM_STATE_CONNECTED,
          farsight_stream_get_current_direction (stream) |
          FARSIGHT_STREAM_DIRECTION_SENDONLY);
  }
  else
  {
    STREAM_DEBUG (self, "Setting sending on %d", sending);
    if (!self->priv->src_valve) {
      if (!gst_pad_set_blocked_async (codec_bin_src_pad, TRUE, blocked_cb,
              (gpointer) __FUNCTION__)) {
        gst_object_unref (codec_bin_src_pad);
      }
    }

    farsight_stream_signal_state_changed (stream,
        farsight_stream_get_state (stream),
        farsight_stream_get_current_direction (stream) &
        ~FARSIGHT_STREAM_DIRECTION_SENDONLY);
  }

  return TRUE;
}

CodecAssociation *
farsight_rtp_stream_choose_codec (FarsightRTPStream *self)
{
  GList *codec_item = NULL;
  CodecAssociation *ca = NULL;

  for (codec_item = self->priv->negotiated_codecs;
       codec_item;
       codec_item = g_list_next (codec_item)) {
    FarsightCodec *codec = codec_item->data;

    ca = lookup_codec_by_pt (
                self->priv->negotiated_codec_associations,
                codec->id);
    if (ca)
      return ca;
  }
  return NULL;
}

static void
farsight_rtp_stream_upnp_send_request(FarsightRTPStream *self,
    const FarsightTransportInfo *derived_trans)
{
#ifdef HAVE_CLINKC
  FarsightTransportInfo *trans = NULL;
  const GList *lp;
  GList *candidate_list = self->priv->local_candidates;


  STREAM_DEBUG (self, "Looking for local ip");

  for (lp = candidate_list; lp; lp = g_list_next (lp)) {
    trans = (FarsightTransportInfo *) lp->data;
    if (trans->type == FARSIGHT_CANDIDATE_TYPE_LOCAL)
    {
      if (trans->ip)
      {
        /* let's ignore loopback */
        if (g_ascii_strcasecmp(trans->ip, "127.0.0.1") != 0)
        {
          STREAM_DEBUG (self, "Found local_ip %s", trans->ip);
          /* open ports */
          upnp_open_port_all_igds (trans->ip, derived_trans->port, derived_trans->port,
              (derived_trans->proto == FARSIGHT_NETWORK_PROTOCOL_UDP)?
              IPPROTO_UDP:IPPROTO_TCP);
        }
      }
    }
  }
#endif
}

static void
farsight_rtp_stream_upnp_close_ports (FarsightRTPStream *self)
{
#ifdef HAVE_CLINKC
  FarsightTransportInfo *trans = NULL;
  const GList *lp;
  GList *candidate_list = self->priv->local_candidates;

  for (lp = candidate_list; lp; lp = g_list_next (lp)) {
    trans = (FarsightTransportInfo *) lp->data;
    if (trans->type == FARSIGHT_CANDIDATE_TYPE_DERIVED)
    {
      STREAM_DEBUG (self, "Found derived ip %s", trans->ip);
      /* close ports */
      upnp_close_port_all_igds (trans->port,
          (trans->proto == FARSIGHT_NETWORK_PROTOCOL_UDP)?
            IPPROTO_UDP:IPPROTO_TCP);
    }
  }
#endif
}

static gboolean
farsight_rtp_stream_candidate_exists (FarsightStream *stream,
                                      const GList *candidate_list,
                                      const GList *candidate)
{
  FarsightTransportInfo *trans = NULL;
  FarsightTransportInfo *trans2 = NULL;
  const GList *lp = NULL;
  const GList *lp2 = NULL;
  gint i = 0;

  if (candidate_list == NULL || candidate == NULL)
    return FALSE;

  /* we check ip and port */
  for (lp = candidate; lp; lp = g_list_next (lp)) {
    trans = (FarsightTransportInfo *) lp->data;

    for (lp2 = candidate_list; lp2; lp2 = g_list_next (lp2)) {
      trans2 = (FarsightTransportInfo *) lp2->data;
      if (farsight_transport_are_equal (trans, trans2))
        i++;
    }
  }

  if (i == g_list_length ((GList *)candidate))
    return TRUE;
  else if (i == 0)
    return FALSE;
  else
  {
    g_error("Candidate only partially exists (some components do), "
        "this should not happen!");
    return FALSE;
  }
}

static void
farsight_rtp_stream_new_native_candidate (gpointer transmitter,
    const FarsightTransportInfo *candidate, gpointer stream)
{
  FarsightRTPStream *self = (FarsightRTPStream *) stream;
  FarsightTransportInfo *candidate_copy = NULL;
  GList *temp_list = NULL;

  STREAM_DEBUG (self, "Called farsight_rtp_stream_new_native_candidate");

  /* FIXME let's just add it to a GList for now until the whole
 * Candidate/component/transport issue is solved */
  temp_list = g_list_append (temp_list, (gpointer) candidate);
  if (farsight_rtp_stream_candidate_exists (stream, self->priv->local_candidates,
              temp_list))
  {
    STREAM_DEBUG (self, "Native candidate already in list, not adding");
    g_list_free (temp_list);
    return;
  }
  else
  {
    g_list_free (temp_list);
    STREAM_DEBUG (self, "Native candidates found, adding to list");
    candidate_copy = farsight_transport_copy (candidate);
    self->priv->local_candidates = g_list_append(self->priv->local_candidates,
        candidate_copy);
    farsight_stream_signal_new_native_candidate (stream,
            candidate_copy->candidate_id);

    if (self->priv->use_upnp)
    {
      /* TODO this assumes the local IP is found before the derived IP */
      /* if not it won't send any request since it needs the local ip */
      if (candidate_copy->type == FARSIGHT_CANDIDATE_TYPE_DERIVED)
      {
        farsight_rtp_stream_upnp_send_request (self, candidate_copy);
      }
    }
  }
}

static void
farsight_rtp_stream_native_candidates_prepared (gpointer transmitter, gpointer stream)
{
  farsight_stream_signal_native_candidates_prepared (stream);
}


static void farsight_rtp_stream_new_active_candidate_pair (gpointer transmitter,
    const gchar *native_candidate_id,
    const gchar *remote_candidate_id,
    gpointer stream)
{
  farsight_stream_signal_new_active_candidate_pair (stream, native_candidate_id,
      remote_candidate_id);
}

static void
farsight_rtp_stream_transmitter_state_changed (gpointer transmitter,
    FarsightTransmitterState state, gpointer stream)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (stream);

  STREAM_DEBUG (self, "connect state changed to %d", state);

  g_return_if_fail(self->priv->disposed == FALSE);

  if (state == FARSIGHT_TRANSMITTER_STATE_CONNECTED)
    {
      GstStateChangeReturn rv;

      if (self->priv->timeout_src)
      {
        if (g_source_remove (self->priv->timeout_src) == FALSE) {
          WARNING ("Tried to remove non-existent source %u",
              self->priv->timeout_src);
        }
        self->priv->timeout_src = 0;
      }

      if (self->priv->sending)
        farsight_stream_signal_state_changed (stream,
          FARSIGHT_STREAM_STATE_CONNECTED,
          farsight_stream_get_current_direction (stream) |
          FARSIGHT_STREAM_DIRECTION_SENDONLY);
      else
        farsight_stream_signal_state_changed (stream,
          FARSIGHT_STREAM_STATE_CONNECTED,
          farsight_stream_get_current_direction (stream));

      if (self->priv->pipeline) {
        /* At this stage, we want to be able to receive data, so makes sure the
         * pipeline is playing */
        rv = gst_element_set_state (self->priv->pipeline, GST_STATE_PLAYING);
        STREAM_DEBUG (self, "Setting pipeline to PLAYING returned %d", rv);
      }
      farsight_rtp_stream_try_set_playing (self);
    }
  else if (state == FARSIGHT_TRANSMITTER_STATE_CONNECTING)
    {
      if (self->priv->timeout_src)
      {
        if (g_source_remove (self->priv->timeout_src) == FALSE) {
          WARNING ("Tried to remove non-existent source %u",
              self->priv->timeout_src);
        }
        self->priv->timeout_src = 0;
      }
      /* Let us set a timer for a timeout on reestablishing a connection */
      self->priv->timeout_src = g_timeout_add (self->priv->conn_timeout * 1000,
          farsight_rtp_stream_connection_timed_out,
          self);

      /* farsight_rtp_stream_stop (FARSIGHT_STREAM (stream)); */
      /* What we really want here is some sort of _pause/_resume functionality
       * in farsight */
      farsight_stream_signal_state_changed (stream,
          FARSIGHT_STREAM_STATE_CONNECTING,
          farsight_stream_get_current_direction (stream));
    }
}

static void
farsight_rtp_stream_transmitter_error (gpointer transmitter, gpointer stream)
{
  STREAM_WARNING (stream, "Error from transmitter");
  farsight_rtp_stream_stop (stream);
  farsight_stream_signal_error (stream,
          FARSIGHT_STREAM_ERROR_NETWORK,
          "Network error from the transmitter");
}

static gboolean
farsight_rtp_stream_connection_timed_out (gpointer data)
{
  FarsightRTPStream *self = (FarsightRTPStream *) data;

  self->priv->timeout_src = 0;

  /* let's check if we are connected yet, throw an error if not */
  if (farsight_stream_get_state (FARSIGHT_STREAM(self)) ==
      FARSIGHT_STREAM_STATE_CONNECTING)
  {
    STREAM_WARNING (self, "Could not establish a connection");
    farsight_stream_signal_error (FARSIGHT_STREAM(self),
        FARSIGHT_STREAM_ERROR_TIMEOUT, "Could not establish a connection");
    farsight_rtp_stream_stop (FARSIGHT_STREAM(self));
  }

  return FALSE;
}

/* This is to signal you to prepare the list of transport condidates,
 * e.g. start off any network probing, like STUN
 * */
static void
farsight_rtp_stream_prepare_transports (FarsightStream *stream)
{
  FarsightRTPStream *self = (FarsightRTPStream *) stream;
  guint media_type;

  if (!ensure_local_codecs (self))
    return;

  if (self->priv->prepared)
    return;

  if (NULL == self->priv->transmitter)
    {
      STREAM_WARNING (self, "No transmitter created yet, exiting");
      farsight_stream_signal_error (stream, FARSIGHT_STREAM_ERROR_UNKNOWN,
          "Create transmitter prior to calling prepare_transports()");
      return;
    }

  STREAM_DEBUG (self, "Preparing transmitter");

  g_object_get (G_OBJECT (self), "media-type", &media_type, NULL);
  g_object_set (G_OBJECT (self->priv->transmitter), "media-type", media_type,
      NULL);

  /* connect callbacks to signals from the transmitter */
  g_signal_connect (G_OBJECT (self->priv->transmitter), "new-native-candidate",
      G_CALLBACK (farsight_rtp_stream_new_native_candidate), self);
  g_signal_connect (G_OBJECT (self->priv->transmitter), "native-candidates-prepared",
      G_CALLBACK (farsight_rtp_stream_native_candidates_prepared), self);
  g_signal_connect (G_OBJECT (self->priv->transmitter), "new-active-candidate-pair",
      G_CALLBACK (farsight_rtp_stream_new_active_candidate_pair), self);
  g_signal_connect (G_OBJECT (self->priv->transmitter), "connection-state-changed",
      G_CALLBACK (farsight_rtp_stream_transmitter_state_changed), self);
  g_signal_connect (G_OBJECT (self->priv->transmitter), "error",
      G_CALLBACK (farsight_rtp_stream_transmitter_error), self);

  farsight_transmitter_prepare (self->priv->transmitter);

  if(self->priv->timeout_src == 0) {
    /* Let us set a timer for a timeout on establishing a connection */
    self->priv->timeout_src = g_timeout_add (self->priv->conn_timeout * 1000,
        farsight_rtp_stream_connection_timed_out,
        self);
  }

  /* We can already create the core pipeline now */
  farsight_rtp_stream_build_base_pipeline (self);
}

static gboolean
farsight_rtp_stream_add_remote_candidate_to_rtpbin (FarsightRTPStream *self,
                                                    const gchar *remote_candidate_id)
{
  GList *remote_candidate;
  gchar *addr = NULL;
  FarsightTransportInfo *trans = NULL;
  const GList *lp;

  if (self->priv->rtpbin)
  {
    remote_candidate = farsight_transport_get_list_for_candidate_id
        (self->priv->remote_candidates, remote_candidate_id);
    if (remote_candidate == NULL)
      return FALSE;

    /* Find remote candidate that is of subtype RTP */
    for (lp = remote_candidate; lp; lp = g_list_next (lp)) {
      trans = (FarsightTransportInfo *) lp->data;
      if (g_ascii_strcasecmp(trans->proto_subtype, "RTP") == 0)
      {
        break;
      }
    }
    if (trans == NULL)
      return FALSE;
    /* Set the ip:port of that transport to rtpbin */
    /* TODO there should be a way to set alternative RTCP ip:port that are
     * not simply ip:port+1 */
    addr = g_strdup_printf ("%s:%d", trans->ip, trans->port);
    g_object_set (G_OBJECT (self->priv->rtpbin),
            "destinations", addr, NULL);

    g_free(addr);
    return TRUE;
  }
  return FALSE;
}

static gboolean
farsight_rtp_stream_set_active_candidate_pair (FarsightStream *stream,
                                               const gchar *native_candidate_id,
                                               const gchar *remote_candidate_id)
{
  FarsightRTPStream *self = (FarsightRTPStream *) stream;

  self->priv->active_native_candidate = g_strdup (native_candidate_id);
  self->priv->active_remote_candidate = g_strdup (remote_candidate_id);

  /* this will not work if the pipeline has not yet been created */
  if (!farsight_rtp_stream_add_remote_candidate_to_rtpbin (self,
              remote_candidate_id))
    return FALSE;

  return TRUE;
}

static GList *
farsight_rtp_stream_get_native_candidate (FarsightStream *stream,
                                          const gchar *candidate_id)
{
  FarsightRTPStream *self = (FarsightRTPStream *) stream;
  GList *candidate = NULL;

  candidate = farsight_transport_get_list_for_candidate_id
          (self->priv->local_candidates, candidate_id);

  return candidate;
}

static G_CONST_RETURN GList *
farsight_rtp_stream_get_native_candidate_list (FarsightStream *stream)
{
  FarsightRTPStream *self = (FarsightRTPStream *) stream;
  return self->priv->local_candidates;
}

static void
farsight_rtp_stream_set_remote_candidate_list (FarsightStream *stream,
                                               const GList *remote_candidates)
{
  FarsightRTPStream *self = (FarsightRTPStream *) stream;

  self->priv->remote_candidates = farsight_transport_list_copy(remote_candidates);
}

static void
farsight_rtp_stream_add_remote_candidate (FarsightStream *stream,
                                          const GList *remote_candidate)
{
  FarsightRTPStream *self = (FarsightRTPStream *) stream;
  FarsightTransportInfo *info = (FarsightTransportInfo *)remote_candidate->data;
  GList *rc_copy;

  STREAM_DEBUG (self, "adding remote candidate %s %d", info->ip, info->port);
  info = NULL;

  rc_copy = farsight_transport_list_copy (remote_candidate);

  if (self->priv->remote_candidates == NULL)
  {
    self->priv->remote_candidates = rc_copy;
  }
  else
  {
    if (farsight_rtp_stream_candidate_exists (stream,
                self->priv->remote_candidates, rc_copy))
    {
      STREAM_DEBUG (self, "Remote candidate already in list, not adding");
      return;
    }
    else
    {
      self->priv->remote_candidates = g_list_concat(self->priv->remote_candidates,
                rc_copy);
      STREAM_DEBUG (self, "Added remote candidate");
    }
  }

  if (self->priv->transmitter)
  {
    farsight_transmitter_add_remote_candidates (self->priv->transmitter,
        (const GList *)rc_copy);
  }
}


static gboolean
farsight_rtp_stream_start_telephony_event (FarsightStream *self,
                                           guint8 ev,
                                           guint8 volume,
                                           FarsightStreamDTMFMethod method)
{
  GstStructure *structure = NULL;
  GstEvent *event = NULL;
  FarsightRTPStream *rtpself = FARSIGHT_RTP_STREAM (self);
  FarsightStreamDTMFMethod chosen_method;
  gchar *method_str;

  g_return_val_if_fail (rtpself->priv->pipeline != NULL, FALSE);

  structure = gst_structure_new ("dtmf-event",
                                 "number", G_TYPE_INT, ev,
                                 "volume", G_TYPE_INT, volume,
                                 "type", G_TYPE_INT, 1,
                                 "start", G_TYPE_BOOLEAN, TRUE, NULL);

  if (method == FARSIGHT_DTMF_METHOD_AUTO) {
    GstElement *dtmfsrc = gst_bin_get_by_name (GST_BIN (rtpself->priv->pipeline),
                                               "rtpdtmfsrc");
    if (dtmfsrc) {
      chosen_method = FARSIGHT_DTMF_METHOD_RTP_RFC4733;
      gst_object_unref (dtmfsrc);
    } else {
       chosen_method = FARSIGHT_DTMF_METHOD_SOUND;
    }
  } else {
      chosen_method = method;
  }

  gst_structure_set (structure, "method", G_TYPE_INT, chosen_method, NULL);

  switch (chosen_method) {
    case FARSIGHT_DTMF_METHOD_AUTO:
      method_str = "default";
      break;
    case FARSIGHT_DTMF_METHOD_SOUND:
      method_str="sound";
      break;
    case FARSIGHT_DTMF_METHOD_RTP_RFC4733:
      method_str="RFC4733";
      break;
    default:
      method_str="other";
  }
  STREAM_DEBUG (self, "sending telephony event %d using method=%s",
      ev, method_str);

  event = gst_event_new_custom (GST_EVENT_CUSTOM_UPSTREAM, structure);

  return gst_element_send_event (rtpself->priv->pipeline, event);
}

static gboolean
farsight_rtp_stream_stop_telephony_event (FarsightStream *self,
                                          FarsightStreamDTMFMethod method)
{
  GstStructure *structure = NULL;
  GstEvent *event = NULL;
  FarsightRTPStream *rtpself = FARSIGHT_RTP_STREAM (self);

  g_return_val_if_fail (rtpself->priv->pipeline != NULL, FALSE);

  structure = gst_structure_new ("dtmf-event",
                                 "start", G_TYPE_BOOLEAN, FALSE,
                                 "type", G_TYPE_INT, 1, NULL);

  if (method == FARSIGHT_DTMF_METHOD_AUTO) {
    GstElement *dtmfsrc = gst_bin_get_by_name (GST_BIN (rtpself->priv->pipeline),
                                               "rtpdtmfsrc");
    if (dtmfsrc) {
      gst_structure_set (structure,
          "method", G_TYPE_INT, FARSIGHT_DTMF_METHOD_RTP_RFC4733, NULL);
      gst_object_unref (dtmfsrc);
    } else {
      gst_structure_set (structure,
          "method", G_TYPE_INT, FARSIGHT_DTMF_METHOD_SOUND, NULL);
    }
  } else {
    gst_structure_set (structure, "method", G_TYPE_INT, method, NULL);
  }

  event = gst_event_new_custom (GST_EVENT_CUSTOM_UPSTREAM, structure);

  return gst_element_send_event (rtpself->priv->pipeline, event);
}


static gboolean
farsight_rtp_stream_preload_receive_pipeline (FarsightStream *stream,
    gint payload_type)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (stream);
  CodecAssociation *ca = NULL;
  GstElement *codec_bin = NULL;
  gchar *name = NULL;

  STREAM_DEBUG (self, "Trying to preload codec %d", payload_type);

  if (self->priv->negotiated_codec_associations == NULL) {
    STREAM_WARNING (self, "You have to set the remote codecs before being able"
        " to preload a receive pipeline");
    return FALSE;
  }

  if (self->priv->recv_codec_id >= 0) {
    STREAM_WARNING (self, "Tried to preload codec while receive codec already "
        "loaded");
    return FALSE;
  }
  self->priv->preload_recv_codec_id = payload_type;

  if (!self->priv->pipeline) {
    STREAM_DEBUG (self, "Pipeline not created yet, will preload later");
    return TRUE;
  }

  name = g_strdup_printf ("recv%d", payload_type);
  codec_bin = gst_bin_get_by_name (GST_BIN(self->priv->pipeline), name);
  g_free (name);

  if (codec_bin) {
    gst_object_unref (codec_bin);
    return TRUE;
  }

  ca = lookup_codec_by_pt (self->priv->negotiated_codec_associations,
      payload_type);

  if (!ca) {
    STREAM_WARNING (self, "Tried to preload Codec that does not exist");
    return FALSE;
  }

  /* Create the preloaded codec bin */
  codec_bin = farsight_rtp_stream_get_or_create_recv_codec_bin (stream, payload_type);

  /* We store those values here for later use */
  self->priv->recv_codec_id = payload_type;
  self->priv->recv_codec_bin = codec_bin;

  /* Make the codec READY to make it load faster afterwards */
  gst_element_set_state (codec_bin, GST_STATE_READY);

  return TRUE;
}

static gboolean
farsight_rtp_stream_get_jb_statistics (FarsightStream *stream,
    guint64 *total_packets,
    guint64 *late_packets,
    guint64 *duplicate_packets,
    guint   *fill_level,
    guint64 *times_overrun,
    guint64 *times_underrun)
{
  FarsightRTPStream *rtpstream = FARSIGHT_RTP_STREAM (stream);
  GstQuery *query;
  GstStructure *structure;
  gboolean ret;

  if (!rtpstream->priv->pipeline)
    return FALSE;

  if (GST_QUERY_JB_STATS == GST_QUERY_NONE)
    return FALSE;

  structure = gst_structure_new ("jb-stats", NULL);
  query = gst_query_new_application (GST_QUERY_JB_STATS, structure);

  if (!query) {
    gst_structure_free (structure);
    return FALSE;
  }

  ret = gst_element_query (rtpstream->priv->pipeline, query);

  if (ret) {
    const GValue *val;

    val = gst_structure_get_value (structure, "total-packets");
    if (val)
      *total_packets = g_value_get_uint64 (val);
    else
      ret = FALSE;

    val = gst_structure_get_value (structure, "late-packets");
    if (val)
      *late_packets = g_value_get_uint64 (val);
    else
      ret = FALSE;

    val = gst_structure_get_value (structure, "duplicate-packets");
    if (val)
      *duplicate_packets = g_value_get_uint64 (val);
    else
      ret = FALSE;

    val = gst_structure_get_value (structure, "times-overrun");
    if (val)
      *times_overrun = g_value_get_uint64 (val);
    else
      ret = FALSE;

    val = gst_structure_get_value (structure, "times-underrun");
    if (val)
      *times_underrun = g_value_get_uint64 (val);
    else
      ret = FALSE;

    val = gst_structure_get_value (structure, "fill-level");
    if (val)
      *fill_level = g_value_get_uint (val);
    else
      ret = FALSE;
  }

  gst_query_unref (query);

  return ret;
}

/****************************************************/
/*                HELPER FUNCTIONS                  */
/****************************************************/



static void block_pad_and_call_idler_cb (GstPad *pad, gboolean blocked, gpointer user_data);
static gboolean block_pad_and_call_idler_idle (gpointer user_data);

/*
 * farsight_rtp_stream_get_source_linked_pad (FarsightRTPStream *self)
 *
 * This helper function will return the GstPad of the source which is
 * linked to that our pipeline.
 *
 */
static GstPad *
farsight_rtp_stream_get_source_linked_pad (FarsightRTPStream *stream) {
  GstPad *valve_sink_pad = NULL;
  GstPad *pipeline_ghostpad = NULL;
  GstPad *source_pad = NULL;

  if (stream->priv->src) {

    /* Get the pad of the valve and of the pipeline */
    if (stream->priv->src_valve) {
      valve_sink_pad = gst_element_get_static_pad (stream->priv->src_valve,
          "sink");
    }
    if (stream->priv->pipeline) {
      pipeline_ghostpad = gst_element_get_static_pad (stream->priv->pipeline,
          "sink");
    }

    /* If the pipeline has a sink pad, then we have a ghostpad */
    if (pipeline_ghostpad) {
      source_pad = gst_pad_get_peer (pipeline_ghostpad);
    } else if (valve_sink_pad) {
      source_pad = gst_pad_get_peer (valve_sink_pad);
    }

    /* Unref the used pads */
    if (valve_sink_pad) {
      gst_object_unref (GST_OBJECT (valve_sink_pad));
    }
    if (pipeline_ghostpad) {
      gst_object_unref (GST_OBJECT (pipeline_ghostpad));
    }
  }

  return source_pad;
}



struct _block_pad_data
{
  FarsightRTPStream *stream;
  PadBlockIdlerCallback function;
  GstPad *pad;
  gpointer user_data;
  gchar *name;
};

typedef struct _block_pad_data block_pad_data;


/*
 * block_pad_and_call_idler (FarsightRTPStream *self, GstPad *pad,
 *                           PadBlockIdlerCallback function,
 *                           gpointer user_data, gchar *name)
 *
 * This helper function will block a pad and call the specified function
 * once the pad is blocked. The function is called on an idle timer so
 * it will get called from the main loop.
 *
 * Input :
 * * stream : The RTP stream
 * * pad : The pad to block
 * * function : The callback function to call on the idle timer.
 *              **** The callback MUST return TRUE in order to unblock the pad,
 *              **** or FALSE if you want the pad to stay blocked (in case of error)
 * * user_data : a gpointer to pass to the function
 * * name : an optional name describing the function. Only used in debug messages.
 */
static void
block_pad_and_call_idler (FarsightRTPStream *stream, GstPad *pad,
                          PadBlockIdlerCallback function, gpointer user_data, gchar *name)
{
  gchar *pad_name = get_pad_full_name (pad);

  block_pad_data *block_data = g_new0 (block_pad_data, 1);
  block_data->stream = stream;
  block_data->function = function;
  block_data->pad = pad;
  block_data->user_data = user_data;
  block_data->name = g_strdup (name);

  STREAM_DEBUG (stream, "Blocking pad %s in order to call idler function <%s>",
      pad_name, name);

  g_free (pad_name);

  gst_object_ref (GST_OBJECT (pad));
  if (!gst_pad_set_blocked_async (pad, TRUE, block_pad_and_call_idler_cb, block_data)) {
    STREAM_WARNING (stream, "Trying to block an already blocked pad! Running "
        "the function directly for %s", name);
    if (function (pad, user_data))
    {
      if (!gst_pad_set_blocked_async (block_data->pad, FALSE, block_pad_and_call_idler_cb, block_data))  {
        gst_object_unref (pad);
        g_free (block_data);
        WARNING ("Trying to unblock a non blocked pad!");
      } else {
        DEBUG ("Waiting for pad to unblock");
      }
    }
  } else {
    STREAM_DEBUG (stream, "Waiting for pad to block");
  }

}

/* Function called when a pad is blocked */
static void
block_pad_and_call_idler_cb (GstPad *pad, gboolean blocked, gpointer user_data)
{
  gchar *pad_name = get_pad_full_name (pad);
  guint src_id = 0;
  block_pad_data *block_data = (block_pad_data *) user_data;

  if (blocked){
    DEBUG ("Pad %s blocked successfully", pad_name);

    gst_object_ref (GST_OBJECT (pad));

    DEBUG ("Adding idle callback");
    src_id = g_idle_add(block_pad_and_call_idler_idle, user_data);
    if (src_id)
      g_array_append_val (block_data->stream->priv->pending_src_ids, src_id);
    else
      WARNING ("Could not add idle task for block_pad_and_call_idler_idle");

  } else {
    DEBUG ("Pad %s unblocked successfully", pad_name);
  }

  gst_object_unref (GST_OBJECT (pad));

  g_free (pad_name);
}


/* Function called on the idle timer from the main loop after a pad has been blocked*/
static gboolean
block_pad_and_call_idler_idle (gpointer user_data)
{
  block_pad_data *block_data = (block_pad_data *) user_data;
  gchar *pad_name = get_pad_full_name (block_data->pad);

  DEBUG ("Idle callback called");

  DEBUG ("Calling function callback <%s> on blocked pad %s", block_data->name,
      pad_name);
  if (block_data->function (block_data->pad, block_data->user_data) ) {
    gst_object_ref (GST_OBJECT (block_data->pad));
    if (!gst_pad_set_blocked_async (block_data->pad, FALSE, block_pad_and_call_idler_cb, block_data))  {
      gst_object_unref (GST_OBJECT (block_data->pad));
      WARNING ("Trying to unblock a non blocked pad!");
    } else {
      DEBUG ("Waiting for pad to unblock");
    }
  } else {
    WARNING ("Function callback %s returned FALSE. "
        "Pad not being unblocked", block_data->name);
  }


  gst_object_unref (GST_OBJECT (block_data->pad));

  g_free (pad_name);
  g_free (block_data->name);
  g_free (block_data);


  return FALSE;
}

gboolean
farsight_rtp_stream_recv_codec_bin_has_src (FarsightRTPStream *self)
{
  gboolean has_src = FALSE;

  if (self->priv->recv_codec_bin) {
    GstPad *pad = gst_element_get_static_pad (self->priv->recv_codec_bin,
        "src");
    if (pad) {
      has_src = TRUE;
      gst_object_unref (pad);
    }else {
      has_src = FALSE;
    }
  }

  return has_src;
}


static gboolean
farsight_rtp_stream_send_codec_bin_has_sink (FarsightRTPStream *self)
{
  gboolean has_sink = FALSE;

  if (self->priv->send_codec_bin) {
    GstPad *pad = gst_element_get_static_pad (self->priv->send_codec_bin,
        "sink");
    if (pad) {
      has_sink = TRUE;
      gst_object_unref (pad);
    }else {
      has_sink = FALSE;
    }
  }

  return has_sink;
}


/* farsight_rtp_stream_clean_recv_codec_bin(FarsightRTPStream *)
 *
 * This function will clean the recv codec bin of the pipeline by setting its
 * state to NULL, unlinking it and removing it from the pipeline, thus
 * destroying it.
 */
static gboolean
farsight_rtp_stream_clean_recv_codec_bin(FarsightRTPStream *self)
{
  gchar *name;
  GstElement *codec_bin = NULL;
  CodecAssociation *codec_association = NULL;
  gboolean ret = FALSE;

  if (self->priv->recv_codec_bin == NULL) {
    STREAM_DEBUG (self, "No recv codec bin. nothing to clean");
    return TRUE;
  }

  name = g_strdup_printf ("recv%d", self->priv->recv_codec_id);
  codec_bin = gst_bin_get_by_name (GST_BIN(self->priv->pipeline), name);
  g_free (name);
  if (!codec_bin || codec_bin != self->priv->recv_codec_bin)
  {
    STREAM_WARNING (self,
        "Couldn't find current recv codec for codec %d or %p != %p",
        self->priv->recv_codec_id, codec_bin, self->priv->recv_codec_bin);
    goto error;
  }

  ret = farsight_rtp_stream_clean_recv_codec_bin_obj (self, codec_bin);

  /* The codec bin is unusable now so it shouldn't hold any 'unique' slot */
  codec_association = lookup_codec_by_pt (self->priv->negotiated_codec_associations,
      self->priv->recv_codec_id);
  if (codec_association->codec_blueprint->receive_unique_bin == codec_bin) {
    g_object_remove_weak_pointer (G_OBJECT (codec_bin),
        (gpointer) &codec_association->codec_blueprint->receive_unique_bin);
    codec_association->codec_blueprint->receive_unique_bin = NULL;
  }


  if (ret == TRUE) {
    self->priv->recv_codec_bin = NULL;
    return ret;
  }

 error:

  farsight_rtp_stream_stop (FARSIGHT_STREAM (self));
  farsight_stream_signal_error (FARSIGHT_STREAM (self),
      FARSIGHT_STREAM_ERROR_PIPELINE_SETUP,
      "Error while cleaning the recv codec bin");

  return FALSE;

}


/* farsight_rtp_stream_clean_recv_codec_bin_obj(FarsightRTPStream *, GstElement*)
 *
 * This function will clean a codec bin of the reception pipeline by setting its
 * state to NULL, unlinking it and removing it from the pipeline, thus
 * destroying it.
 */
static gboolean
farsight_rtp_stream_clean_recv_codec_bin_obj(FarsightRTPStream *self,
    GstElement *codec_bin)
{
  GstStateChangeReturn state_ret;
  GstState state;

  STREAM_DEBUG (self, "clean recv codec bin called");

  /* Let's unlink the existing codec bin and delete it */
  STREAM_DEBUG (self, "removing recv codec bin %p from pipeline %p",
      codec_bin, self->priv->pipeline);

  /* We need to lock state of the codec bin in case an rtcp is received
     at the same time, which will make the whole pipeline go into PLAYING
     and will revert our state change, causing us to try and remove an
     element not in the NULL state, which will crash the whole pipeline */
  gst_element_set_locked_state (codec_bin, TRUE);

  STREAM_DEBUG (self, "Setting codec bin state to NULL");
  state_ret = gst_element_set_state (codec_bin, GST_STATE_NULL);
  if (state_ret == GST_STATE_CHANGE_ASYNC)
    gst_element_get_state (codec_bin, &state, NULL, GST_CLOCK_TIME_NONE);

  if (state_ret == GST_STATE_CHANGE_FAILURE) {
    STREAM_WARNING (self, "Could not set the codec bin to NULL");
    goto error;
  }

  STREAM_DEBUG (self, "Unlinking old codec bin");
  farsight_rtp_stream_unlink_recv_codec_bin(self, codec_bin, TRUE);

  STREAM_DEBUG (self, "Removing bin");
  if (!gst_bin_remove (GST_BIN(self->priv->pipeline), codec_bin)) {
    gchar *name = gst_element_get_name(codec_bin);
    STREAM_WARNING (self, "There was an error removing codec bin %s "
        "from pipeline", name);
    g_free (name);
    gst_object_unref (GST_OBJECT (codec_bin));
    goto error;
  }

  gst_object_unref (GST_OBJECT (codec_bin));

  return TRUE;

error:
  return FALSE;
}

/* farsight_rtp_stream_clean_send_codec_bin(FarsightRTPStream *, GstState *)
 *
 * This function will clean the send codec bin of the pipeline by setting its
 * state to NULL, unlinking it and removing it from the pipeline, thus
 * destroying it.
 *
 * It will return the state of the codec bin before being set to NULL in the
 * argument old_state
 */
static gboolean
farsight_rtp_stream_clean_send_codec_bin(FarsightRTPStream *self,
    GstState *old_state)
{
  gchar *name;
  GstElement *codec_bin = NULL;
  CodecAssociation *codec_association = NULL;
  GstStateChangeReturn state_ret;
  GstState state;
  GstState pending;

  STREAM_DEBUG (self, "clean send codec bin called");

  if (self->priv->send_codec_bin == NULL) {
    STREAM_DEBUG (self, "No send codec bin. nothing to clean");
    return TRUE;
  }

  /* Let's unlink the existing codec bin and delete it */
  name = g_strdup_printf ("send%d", self->priv->send_codec_id);
  codec_bin = gst_bin_get_by_name (GST_BIN(self->priv->pipeline), name);
  STREAM_DEBUG (self, "removing send codec bin %p on pipeline %p",
      codec_bin, self->priv->pipeline);
  g_free (name);
  if (!codec_bin || codec_bin != self->priv->send_codec_bin)
  {
    STREAM_WARNING (self, "Couldn't find current send codec for codec %d"
        " or %p != %p!",
        self->priv->send_codec_id, codec_bin, self->priv->send_codec_bin);
    goto error;
  }

  gst_element_get_state (codec_bin, old_state, &pending, 0);
  if (pending != GST_STATE_VOID_PENDING) {
    *old_state = pending;
  }

  /* We need to lock state of the codec bin in case an rtcp is received
     at the same time, which will make the whole pipeline go into PLAYING
     and will revert our state change, causing us to try and remove an
     element not in the NULL state, which will crash the whole pipeline */
  gst_element_set_locked_state (codec_bin, TRUE);


  STREAM_DEBUG (self, "clean send codec bin : Setting codec bin state to NULL");
  state_ret = gst_element_set_state (codec_bin, GST_STATE_NULL);
  if (state_ret == GST_STATE_CHANGE_ASYNC)
    gst_element_get_state (codec_bin, &state, NULL, GST_CLOCK_TIME_NONE);

  if (state_ret == GST_STATE_CHANGE_FAILURE) {
    STREAM_WARNING (self, "Could not set the codec bin to NULL");
    goto error;
  }

  STREAM_DEBUG (self, "Unlinking old codec bin");
  farsight_rtp_stream_unlink_send_codec_bin(self, codec_bin);

  STREAM_DEBUG (self, "Removing bin");
  if (!gst_bin_remove (GST_BIN(self->priv->pipeline), codec_bin)) {
    gchar *name = gst_element_get_name(codec_bin);
    gchar *cname = gst_element_get_name(self->priv->pipeline);
    STREAM_WARNING (self, "There was an error removing unique codec bin %s "
        "from container %s", name, cname);
    g_free (name);
    g_free (cname);
    gst_object_unref (GST_OBJECT (codec_bin));
    goto error;
  }

  /* The codec bin is unusable now so it shouldn't hold any 'unique' slot */
  codec_association = lookup_codec_by_pt (self->priv->negotiated_codec_associations,
      self->priv->send_codec_id);
  if (codec_association->codec_blueprint->send_unique_bin == codec_bin) {
    g_object_remove_weak_pointer (G_OBJECT (codec_bin),
        (gpointer) &codec_association->codec_blueprint->send_unique_bin);
    codec_association->codec_blueprint->send_unique_bin = NULL;
  }

  gst_object_unref (GST_OBJECT (codec_bin));

  self->priv->send_codec_bin = NULL;

  return TRUE;

error:
  farsight_rtp_stream_stop (FARSIGHT_STREAM (self));
  farsight_stream_signal_error (FARSIGHT_STREAM (self),
      FARSIGHT_STREAM_ERROR_PIPELINE_SETUP,
      "Error while cleaning the send codec bin");

  return FALSE;
}


/* farsight_rtp_stream_setup_send_codec_bin(FarsightRTPStream *)
 *
 * This function will setup the send codec bin of the pipeline by creating it,
 * adding it to the pipeline,  linking it and setting its state to the value
 * of new_state.
 */
static gboolean
farsight_rtp_stream_setup_send_codec_bin(FarsightRTPStream *self,
    GstState new_state)
{
  GstElement *codec_bin = NULL;
  GstStateChangeReturn state_ret;

  if (self->priv->send_codec_bin != NULL) {
    STREAM_DEBUG (self, "Send codec bin already setup");
    return TRUE;
  }

  codec_bin = create_codec_bin (self->priv->negotiated_codec_associations,
      self->priv->send_codec_id, DIR_SEND, self->priv->remote_codecs);
  if (!codec_bin)
  {
    STREAM_WARNING (self, "Couldn't create elements, "
        "check your gstreamer installation");
    goto error;
  }

  STREAM_DEBUG (self, "adding send codec bin %p on pipeline %p", codec_bin,
      self->priv->pipeline);
  if (!gst_bin_add (GST_BIN (self->priv->pipeline), codec_bin)) {
    STREAM_WARNING (self, "Could not add the new send codec bin to the "
        "pipeline");
    goto error;
  }

  if (!farsight_rtp_stream_link_send_codec_bin(self, codec_bin) ) {
    goto error;
  }
  self->priv->send_codec_bin = codec_bin;

  bin_element_set_property (GST_BIN (self->priv->send_codec_bin),
      "min-ptime", self->priv->min_ptime, NULL);
  bin_element_set_property (GST_BIN (self->priv->send_codec_bin),
      "max-ptime", self->priv->max_ptime, NULL);

  if (new_state != GST_STATE_NULL) {
    state_ret = gst_element_set_state (codec_bin, new_state);
    if (state_ret == GST_STATE_CHANGE_FAILURE) {
      STREAM_WARNING (self, "Failure while changing the state of the new send "
          "codec bin to %s",
          gst_element_state_get_name (new_state));
      goto error;
    }
  }

  return TRUE;

error:
  farsight_rtp_stream_stop (FARSIGHT_STREAM (self));
  farsight_stream_signal_error (FARSIGHT_STREAM (self),
      FARSIGHT_STREAM_ERROR_PIPELINE_SETUP,
      "Error while setting up send codec bin");

  return FALSE;
}


/* farsight_rtp_stream_unlink_recv_codec_bin (FarsightRTPStream *, GstElement *)
 *
 * This function will unlink a codec bin from the receive pipeline to anything it is linked to.
 * It assumes it has a single (optional) src pad named "src" and multiple sink pads.
 *
 */
static gboolean
farsight_rtp_stream_unlink_recv_codec_bin (FarsightRTPStream *self,  GstElement *codec_bin, gboolean release_rtpdemux_pad)
{
  gboolean ret = FALSE;
  GstPad *pad1 = NULL;

  ret = farsight_rtp_stream_right_unlink_recv_codec_bin (self, codec_bin);

  if (ret) {
    /* Unlink the sink pad of the codec bin */
    pad1 = gst_element_get_static_pad (codec_bin, "sink");
    if (pad1 && gst_pad_is_linked (pad1)) {
      GstPad *pad2 = gst_pad_get_peer (pad1);

      gst_pad_unlink (pad2, pad1);

      if (release_rtpdemux_pad) {
        gst_element_release_request_pad(self->priv->rtpdemux, pad2);
      }

      gst_object_unref (pad2);
    }
    if (pad1) {
      gst_object_unref (pad1);
    }

    /* TODO link/unlink the CN source */
  }

  return ret;
}

/* farsight_rtp_stream_unlink_recv_codec_bin (FarsightRTPStream *, GstElement *)
 *
 * This function will unlink a recvcodec bin from the capsfilter
 * It assumes it has a single (optional) src pad named "src" and multiple sink pads.
 *
 */
static gboolean
farsight_rtp_stream_right_unlink_recv_codec_bin (FarsightRTPStream *self,  GstElement *codec_bin)
{

  GstPad *pad1 = NULL;

  /* Unlink the codec bin if it has a src pad and it was linked */
  pad1 = gst_element_get_static_pad (codec_bin, "src");
  if (pad1 && gst_pad_is_linked (pad1)) {
    GstPad *pad2 = gst_pad_get_peer (pad1);

    gst_pad_unlink (pad1, pad2);
    gst_object_unref (pad2);
  }
  if (pad1) {
    gst_object_unref (pad1);
  }



  return TRUE;
}



/* farsight_rtp_stream_unlink_send_codec_bin (FarsightRTPStream *, GstElement *)
 *
 * This function will unlink a codec bin from the send pipeline to anything it is linked to.
 * It assumes it has a single (optional) sink pad named "sink" and multiple source pads.
 *
 */
static gboolean
farsight_rtp_stream_unlink_send_codec_bin (FarsightRTPStream *self,  GstElement *codec_bin)
{

  GstPad *pad1 = NULL;
  GstIterator *srciter = NULL;
  GstPad *codec_srcpad;
  gboolean done = FALSE;
  gpointer item;

  /* Unlink the codec bin if it has a sink pad and it was linked */
  pad1 = gst_element_get_static_pad (codec_bin, "sink");
  if (pad1 && gst_pad_is_linked (pad1)) {
    GstPad *pad2 = gst_pad_get_peer (pad1);

    gst_pad_unlink (pad2, pad1);

    gst_object_unref (pad2);
  }
  if (pad1) {
    gst_object_unref (pad1);
  }


  /* Unlink all the src pads of the codec bin */
  srciter = gst_element_iterate_src_pads (codec_bin);
  done = FALSE;
  while (!done) {
    switch (gst_iterator_next (srciter, &item)) {
    case GST_ITERATOR_OK:
      codec_srcpad = item;
      if (gst_pad_is_linked (codec_srcpad)) {
        GstPad *pad2 = gst_pad_get_peer (codec_srcpad);
        gchar *padname = gst_pad_get_name (codec_srcpad);
        gchar *rtpmuxpadname = gst_pad_get_name (pad2);

        gst_element_unlink_pads (codec_bin, padname , self->priv->rtpmuxer, rtpmuxpadname);
        gst_element_release_request_pad(self->priv->rtpmuxer, pad2);

        gst_object_unref (GST_OBJECT (pad2));
        g_free (padname);
        g_free (rtpmuxpadname);
      }
      gst_object_unref (GST_OBJECT (item));
      break;
    case GST_ITERATOR_RESYNC:
      // ...rollback changes to items... ???
      gst_iterator_resync (srciter);
      break;
    case GST_ITERATOR_ERROR:
      g_error ("Something is wrong, can't iterate src pads");
      done = TRUE;
      break;
    case GST_ITERATOR_DONE:
      done = TRUE;
      break;
    }
  }
  gst_iterator_free (srciter);

  return TRUE;
}

/* farsight_rtp_stream_link_recv_codec_bin (FarsightRTPStream *, GstElement *)
 *
 * This function will link a recv codec bin to the capsfilter and to the rtpdemuxer
 * The codec bin must not be linked to anything.
 *
 */
static gboolean
farsight_rtp_stream_link_recv_codec_bin_ex (FarsightRTPStream *self,
    GstElement *codec_bin, GstPad *pad)
{
  gboolean ret = FALSE;
  ret = farsight_rtp_stream_link_recv_codec_bin (self, codec_bin);
  if (ret) {
    GstPad *codec_bin_sink_pad = gst_element_get_static_pad (codec_bin, "sink");

    if (codec_bin_sink_pad) {
      if (!gst_pad_is_linked (codec_bin_sink_pad)) {
        GstPadLinkReturn padlink_ret;

        STREAM_DEBUG (self, "Linking codec bin to rtpdemux");
        padlink_ret = gst_pad_link (pad, codec_bin_sink_pad);

        if (GST_PAD_LINK_FAILED(padlink_ret)) {
          STREAM_WARNING (self, "could not link the rtpdemuxer to the new recv"
              " codec bin");
          ret = FALSE;
        }
      } else {
        STREAM_DEBUG (self, "The codec_bin_sink_pad is already linked?");
      }
      gst_object_unref (codec_bin_sink_pad);
    } else {
      g_error ("The codec bin does not have a sink pad!");
    }

  }
  return ret;

}

/* farsight_rtp_stream_link_recv_codec_bin (FarsightRTPStream *, GstElement *)
 *
 * This function will link a recv codec bin to the capsfilter
 * The codec bin must not be linked to anything.
 *
 */
static gboolean
farsight_rtp_stream_link_recv_codec_bin (FarsightRTPStream *self,
    GstElement *codec_bin)
{

  GstPad *bin_pad = gst_element_get_static_pad (codec_bin, "src");
  gchar *tmp_caps = NULL;

  if (!bin_pad) {
      STREAM_DEBUG (self, "Codec bin has no 'src' pad. Not linking with "
          "capsfilter");
      return TRUE;
  }

  g_assert (self->priv->sink_capsfilter);

  tmp_caps = gst_caps_to_string (self->priv->sink_filter);
  STREAM_DEBUG (self, "linking sink capsfilter %p to codec bin %p with caps %s",
      self->priv->sink_capsfilter, codec_bin, tmp_caps);
  g_free (tmp_caps);

  if (!gst_element_link (codec_bin, self->priv->sink_capsfilter))  {
    gst_object_unref (bin_pad);
    STREAM_WARNING (self, "Could not link recv codec bin to sink capsfilter");
    goto error;
  }

  if (!self->priv->sink) {
    GstPad *sink_capsfilter_src_pad =
        gst_element_get_static_pad (self->priv->sink_capsfilter, "src");

    g_assert (sink_capsfilter_src_pad);

    STREAM_DEBUG (self, "Codec bin requires a sink but sink is not set, "
        "lets block the capsfilter's source pad");

    if (!gst_pad_set_blocked_async (sink_capsfilter_src_pad, TRUE, blocked_cb,
            (void*) __FUNCTION__)) {
      STREAM_DEBUG (self, "The src pad of the sink_capsfilter is already "
          "blocked");
    }
  }

  gst_object_unref (bin_pad);

  return TRUE;
 error:
  if (bin_pad)
    gst_object_unref (bin_pad);

  return FALSE;

}



/* farsight_rtp_stream_link_send_codec_bin (FarsightRTPStream *, GstElement *)
 *
 * This function will link a codec bin to the capsfilter and to the rtpmuxer
 * The codec bin must not be linked to anything.
 *
 */
static gboolean
farsight_rtp_stream_link_send_codec_bin (FarsightRTPStream *self,
    GstElement *codec_bin)
{

  GstPad *bin_pad = gst_element_get_static_pad (codec_bin, "sink");
  GstIterator *sinkiter = NULL;
  gboolean done = FALSE;
  gpointer item;
  GstPad *codec_sinkpad;

  if (self->priv->src && self->priv->src_capsfilter)
  {
    /* We must first check to see if the codec bin has a sink pad or not
     * A DSP bin could be used on a device where no source is needed.
     * In which case, we shouldn't link the capsfilter with the codec bin.
     * Otherwise, it will just fail and crash the app.
     */
    if (bin_pad) {
      /* connect src capsfilter to codec_bin */
      gchar *tmp_caps = gst_caps_to_string (self->priv->src_filter);
      STREAM_DEBUG (self, "linking src capsfilter %p to codec bin %p with caps"
          " %s", self->priv->src_capsfilter, codec_bin, tmp_caps);
      g_free (tmp_caps);
      if (!gst_element_link (self->priv->src_capsfilter, codec_bin))  {
        gst_object_unref (bin_pad);
        STREAM_WARNING (self, "Could not link src to codec bin");
        goto error;
      }
      gst_object_unref (bin_pad);
    } else {
      STREAM_DEBUG (self, "Codec bin has no 'sink' pad. "
          "Not linking with capsfilter");
    }

  } else if(bin_pad) {
    gst_object_unref (bin_pad);
    /* Here we check the case where we don't have a source but the codec bin
     * needs a source */
    STREAM_WARNING (self, "Codec bin requires a source but no source is set");
    goto error;
  }

  /* Iterate through the src pads of the codec bin and link them to the rtpmuxer */
  sinkiter = gst_element_iterate_src_pads (codec_bin);
  done = FALSE;
  while (!done) {
    switch (gst_iterator_next (sinkiter, &item)) {
       case GST_ITERATOR_OK:
         codec_sinkpad = item;
         if (!gst_pad_is_linked (codec_sinkpad)) {
           gchar *padname = gst_pad_get_name (codec_sinkpad);
           if (!gst_element_link_pads (codec_bin, padname , self->priv->rtpmuxer, "sink_%d")) {
             g_error ("Can't link pad %s from codec_bin to rtpmuxer", padname);
           }
           g_free (padname);
         }
         gst_object_unref (GST_OBJECT (item));
         break;
       case GST_ITERATOR_RESYNC:
         /* No need to rollback, if they are already linked
          * its all fine
          */
         gst_iterator_resync (sinkiter);
         break;
       case GST_ITERATOR_ERROR:

         g_error ("Something is wrong, can't iterate sink pads");
         done = TRUE;
         break;
       case GST_ITERATOR_DONE:

         done = TRUE;
         break;
     }
   }
   gst_iterator_free (sinkiter);



  return TRUE;
 error:
  return FALSE;

}

/* get_pad_full_name (GstPad *pad)
 *
 * This function returns the full name of a pad in the format of
 * <parent_element_name:pad_name>
 * You must free the gchar* once you used it.
 *
 */
static gchar *
get_pad_full_name (GstPad *pad) {
  gchar *pad_name = NULL;
  GstElement *pad_element = NULL;
  gchar *element_name = NULL;
  gchar *fullname = NULL;

  if (pad) {
    pad_name = gst_pad_get_name (pad);
    pad_element = gst_pad_get_parent_element (pad);
    if (pad_element) {
      element_name = gst_element_get_name (pad_element);
    }
  }
  if (!pad_name) {
    pad_name = g_strdup ("(nil)");
  }
  if (!element_name) {
    element_name = g_strdup ("(nil)");
  }
  fullname = g_strdup_printf ("<%s:%s>", element_name, pad_name);

  g_free (pad_name);
  g_free (element_name);
  if ( pad_element ) {
    gst_object_unref (GST_OBJECT (pad_element));
  }

  return fullname;
}
