/*
 * test7.c - Farsight tests
 *
 * Farsight Voice+Video library test suite
 *  Copyright 2005,2006 Collabora Ltd.
 *  Copyright 2005,2006 Nokia Corporation
 *   @author: Rob Taylor <rob.taylor@collabora.co.uk>.
 *   @author: Philippe Khalaf <philippe.khalaf@collabora.co.uk>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <glib.h>
#include <unistd.h>
#include <gst/gst.h>
#include <farsight/farsight-session.h>
#include <farsight/farsight-stream.h>
#include <farsight/farsight-transport.h>

GMainLoop *mainloop = NULL;
gboolean change_codec = FALSE;

gboolean hold_cb (gpointer data) {
  FarsightStream * stream = (FarsightStream *) data;
  gboolean ret = FALSE;
  if (farsight_stream_is_held (stream)) {
    g_debug("Unholding call");
    ret = farsight_stream_unhold (stream);
  } else {
    g_debug ("Holding call");
    ret = farsight_stream_hold (stream);
  }

  g_debug ("Hold/Unhold returned : %d", ret);
  return TRUE;
}

gboolean hold_io_cb (GIOChannel *source, GIOCondition condition, gpointer data) {
  GString *buf = g_string_new (NULL);
  gsize length = 0;
  GError *error = NULL;

  /* Free what was entered */
  g_io_channel_read_line_string (source, buf, &length, &error);
  g_string_free (buf, TRUE);
  g_free (error);

  hold_cb (data);

  return TRUE;
}

gboolean change_codec_cb (gpointer data) {
  FarsightStream * stream = (FarsightStream *) data;
  static int current_id = 0;
  int id = -1;
  int next = 0;
  FarsightCodec *codec;

  GList *lp;
  GList * codecs = farsight_codec_list_copy (farsight_stream_get_codec_intersection (stream));

  for (lp = codecs; lp; lp = g_list_next (lp)) {
    codec = (FarsightCodec*) lp->data;
    g_message ("codec %s: %d: %s/%d found", codec->media_type == FARSIGHT_MEDIA_TYPE_VIDEO ? "video" : "audio", codec->id, codec->encoding_name, codec->clock_rate);

    g_assert (codec->media_type == FARSIGHT_MEDIA_TYPE_AUDIO);

    if (next == 1) {
      id = codec->id;
      break;
    }
    if (codec->id == current_id) {
#if 0
       g_free (codec->encoding_name);
       codec->encoding_name = g_strdup ("asdas");
#endif
       next++;
    }
  }

  g_assert (next);

  if (id == -1) {
    /* We've gone around */
    lp = g_list_first (codecs);
    codec = lp->data;
    id = codec->id;
  }

  current_id = id;

  g_print("timeout reached - Changing codec to %d\n", id);
  farsight_stream_set_active_codec (stream, id);
#if 0
  farsight_stream_set_remote_codecs (stream, codecs);
#endif

  farsight_codec_list_destroy (codecs);

  return TRUE;
}

gboolean change_codec_io_cb (GIOChannel *source, GIOCondition condition, gpointer data) {
  GString *buf = g_string_new (NULL);
  gsize length = 0;
  GError *error = NULL;

  /* Free what was entered */
  g_io_channel_read_line_string (source, buf, &length, &error);
  g_string_free (buf, TRUE);
  g_free (error);

  change_codec_cb (data);

  return TRUE;
}

static void
stream_error (FarsightStream *stream,
       FarsightStreamError error,
       const gchar *debug)
{
    g_print ("%s: stream error: stream=%p error=%s\n", __FUNCTION__, stream, debug);
}

static void
session_error (FarsightSession *stream,
       FarsightSessionError error,
       const gchar *debug)
{
    g_print ("%s: session error: session=%p error=%s\n", __FUNCTION__, stream, debug);
}


static void
new_active_candidate_pair (FarsightStream *stream, gchar* native_candidate, gchar *remote_candidate)
{
    g_print ("%s: new-native-candidate-pair: stream=%p\n", __FUNCTION__, stream);
}

static void
codec_changed (FarsightStream *stream, gint codec_id)
{
    g_print ("%s: codec-changed: codec_id=%d, stream=%p\n", __FUNCTION__, codec_id, stream);

}

static void
native_candidates_prepared (FarsightStream *stream)
{
    const GList *transport_candidates, *lp;
    FarsightTransportInfo *info;

    g_print ("%s: preparation-complete: stream=%p\n", __FUNCTION__, stream);

    transport_candidates = farsight_stream_get_native_candidate_list (stream);
    for (lp = transport_candidates; lp; lp = g_list_next (lp))
    {
      info = (FarsightTransportInfo*)lp->data;
      g_message ("Local transport candidate: %s %d %s %s %s:%d, pref %f",
          info->candidate_id, info->component, (info->proto == FARSIGHT_NETWORK_PROTOCOL_TCP)?"TCP":"UDP",
          info->proto_subtype, info->ip, info->port, (double) info->preference);
    }
    //g_main_loop_quit(mainloop);
}

static void
state_changed (FarsightStream *stream,
               FarsightStreamState state,
               FarsightStreamDirection dir)
{
    switch (state) {
      case FARSIGHT_STREAM_STATE_CONNECTING:
        g_message ("%s: %p connecting\n", __FUNCTION__, stream);
        break;
      case FARSIGHT_STREAM_STATE_CONNECTED:
        g_message ("%s: %p connected\n", __FUNCTION__, stream);
        break;
      case FARSIGHT_STREAM_STATE_DISCONNECTED:
            g_message ("%s: %p disconnected\n", __FUNCTION__, stream);
            break;
    }
}

FarsightSession *setup_rtp_session();
FarsightStream *setup_rtp_stream(FarsightSession *session, GstElement * pipeline);

FarsightCodecPreference codec_pref_list[] = {
  { "AMR", 8000 },
  { "GSM", 8000 },
  { "iLBC", 8000 },
  { "PCMU", 8000 },
  { "PCMA", 8000 },
  { "speex", 8000 },
  { "speex", 16000 },
};



static gboolean
bus_watch_cb (GstBus *bus, GstMessage *message,
    gpointer user_data)
{

  switch (GST_MESSAGE_TYPE (message)) {
    case GST_MESSAGE_EOS:
      g_debug ("%s (%d): end of stream on stream pipeline",
          __FUNCTION__, __LINE__);
      break;
    case GST_MESSAGE_ERROR:
      {
      gchar *debug;
      GError *err;

      gst_message_parse_error (message, &err, &debug);
      g_free (debug);

      g_warning ("%s (%d): error on stream pipeline. "
                 "Error code=%d message=%s",
                 __FUNCTION__, __LINE__, err->code, err->message);
      g_error_free (err);

      g_main_loop_quit (mainloop);

      break;
      }
    default:
      break;
  }

  return TRUE;
}


void usage (char *prog) {

  g_print("Usage : %s <remoteip> <remoteport> <type>\n", prog);
  g_print("Where <type> is :\n");
  g_print("\t1 :  send source\n");
  g_print("\t2 :  send source and change codec\n");
  g_print("\t3 :  send source with filter\n");
  g_print("\t4 :  send source with filter and change codec\n");
  g_print("\t5 :  send pipeline\n");
  g_print("\t6 :  send pipeline and change codec\n");
  g_print("\t7 :  send pipeline with filter\n");
  g_print("\t8 :  send pipeline with filter and change codec\n");
  g_print("\t9 :  send with hold/unhold\n");
  exit (-1);
}


/* Read usage */
int main(int argc, char **argv)
{
    gboolean use_hold = FALSE;
    gboolean use_pipeline = FALSE;
    gboolean use_filter = FALSE;
    FarsightSession *session;
    FarsightStream *stream1;
    GList *candidate_glist = NULL;
    FarsightTransportInfo *trans = NULL;
    GArray *codec_pref_array;
    GstElement *alsasrc, *alsasink, *tee = NULL;
    GstElement *pipeline = NULL;
    GIOChannel *ioc = g_io_channel_unix_new (0);
    const GList *codec_list = NULL;
    const GList *codec_item = NULL;
    FarsightCodec *codec = NULL;


    gst_init (&argc, &argv);
    if (argc != 4)
    {
      usage (argv[0]);
    }
    switch (atoi(argv[3])) {
    case 1:
      use_pipeline = FALSE;
      use_filter = FALSE;
      change_codec = FALSE;
      use_hold = FALSE;
      break;
    case 2:
      use_pipeline = FALSE;
      use_filter = FALSE;
      change_codec = TRUE;
      use_hold = FALSE;
      break;
    case 3:
      use_pipeline = FALSE;
      use_filter = TRUE;
      change_codec = FALSE;
      use_hold = FALSE;
      break;
    case 4:
      use_pipeline = FALSE;
      use_filter = TRUE;
      change_codec = TRUE;
      use_hold = FALSE;
      break;
    case 5:
      use_pipeline = TRUE;
      use_filter = FALSE;
      change_codec = FALSE;
      use_hold = FALSE;
      break;
    case 6:
      use_pipeline = TRUE;
      use_filter = FALSE;
      change_codec = TRUE;
      use_hold = FALSE;
      break;
    case 7:
      use_pipeline = TRUE;
      use_filter = TRUE;
      change_codec = FALSE;
      use_hold = FALSE;
      break;
    case 8:
      use_pipeline = TRUE;
      use_filter = TRUE;
      change_codec = TRUE;
      use_hold = FALSE;
      break;
    case 9:
      use_pipeline = FALSE;
      use_filter = FALSE;
      change_codec = FALSE;
      use_hold = TRUE;
      break;
    default:
      usage (argv[0]);
    }

    mainloop = g_main_loop_new (NULL, FALSE);

    session = setup_rtp_session();

    if (getenv ("FS_AUDIOSRC") == NULL) {
      alsasrc = gst_element_factory_make("audiotestsrc", "alsasrc");
    } else {
      alsasrc = gst_element_factory_make(getenv ("FS_AUDIOSRC"), "alsasrc");
    }
    alsasink = gst_element_factory_make("alsasink", "alsasink");

    //g_assert (alsasrc && alsasink);

    if (alsasrc) {
      g_object_set(G_OBJECT(alsasrc), "is-live", TRUE, NULL);
      g_object_set(G_OBJECT(alsasrc), "blocksize", 320, NULL);
      //  g_object_set(G_OBJECT(alsasrc), "latency-time", G_GINT64_CONSTANT (20000), NULL);
    }
    if (alsasink) {
      g_object_set(G_OBJECT(alsasink), "sync", FALSE, NULL);
      g_object_set(G_OBJECT(alsasink), "latency-time", G_GINT64_CONSTANT (20000), NULL);
      g_object_set(G_OBJECT(alsasink), "buffer-time", G_GINT64_CONSTANT (80000), NULL);
    }


    if (use_pipeline) {
      GstBus *pipe_bus;
      GstElement *fakesink;

      pipeline = gst_pipeline_new ("main_pipeline");
      pipe_bus = gst_pipeline_get_bus (GST_PIPELINE (pipeline));
      gst_bus_add_watch (pipe_bus, bus_watch_cb, NULL);

      tee = gst_element_factory_make ("tee", NULL);
      fakesink = gst_element_factory_make ("fakesink", NULL);

      if (alsasrc) {
        gst_bin_add_many (GST_BIN (pipeline), alsasrc, tee, fakesink,  NULL);
      } else {
        gst_bin_add_many (GST_BIN (pipeline), tee, fakesink,  NULL);
      }
      gst_element_link (tee, fakesink);
      //      gst_element_link (alsasrc, tee);

      stream1 = setup_rtp_stream(session, pipeline);
    } else {
      stream1 = setup_rtp_stream(session, NULL);
    }

    codec_pref_array =
      g_array_new (FALSE, FALSE, sizeof (FarsightCodecPreference));
    g_array_append_vals (codec_pref_array, codec_pref_list,
        sizeof (codec_pref_list)/sizeof (FarsightCodecPreference));

    /*
    farsight_stream_set_codec_preference_list (stream1,
        (const GArray *)codec_pref_array);
    */

    g_array_free (codec_pref_array, TRUE);

    /* it's better to set the active codec before the remote_codecs to avoid a
     * send pipeline change from the start */
    farsight_stream_set_active_codec (stream1, 0);


    codec_list = farsight_stream_get_local_codecs(stream1);

 for (codec_item = codec_list;
         codec_item;
        codec_item = g_list_next (codec_item)) {
      codec = codec_item->data;

      g_debug("************* %s codec : %s:%d - PT %d", codec->media_type == 0 ? "Audio" : "Video", codec->encoding_name, codec->clock_rate, codec->id);

   }

    /* this assumes both hosts have the same codecs */
    farsight_stream_set_remote_codecs(stream1,
            farsight_stream_get_local_codecs(stream1));

    /* let's create our candidate from ip:port given on command line */
    trans = g_new0 (FarsightTransportInfo,1);
    trans->candidate_id = g_strdup_printf ("L1");
    trans->component = 1;
    trans->ip = g_strdup (argv[1]);
    trans->port = atoi(argv[2]);
    trans->proto = FARSIGHT_NETWORK_PROTOCOL_UDP;
    trans->proto_subtype = "RTP";
    trans->proto_profile = "AVP";
    trans->preference = 1.0;
    trans->type = FARSIGHT_CANDIDATE_TYPE_LOCAL;

    candidate_glist = g_list_append(candidate_glist, trans);

    farsight_stream_set_remote_candidate_list(stream1, candidate_glist);

    g_free(trans);
    g_list_free(candidate_glist);

    farsight_stream_set_active_candidate_pair(stream1, "L1", "L1");


    if (use_filter) {
      GstCaps *src_filter = gst_caps_new_simple ("audio/x-raw-int",
                                                 "rate", G_TYPE_INT, 8000,
                                                 NULL);
      farsight_stream_set_source_filter (stream1, src_filter);
      gst_caps_unref (src_filter);

    }
    if (use_pipeline) {
      // farsight_stream_set_source(stream1, tee);
      farsight_stream_set_source(stream1, alsasrc);
    } else {
      farsight_stream_set_source(stream1, alsasrc);
    }

    farsight_stream_set_sink(stream1, alsasink);

    farsight_stream_start(stream1);
    if (pipeline) {
      gst_element_set_state (pipeline, GST_STATE_PLAYING);
    }


    if (change_codec) {
      //g_timeout_add_full(G_PRIORITY_DEFAULT_IDLE, 10000, change_codec_cb, (gpointer) stream1, NULL);
      g_io_add_watch (ioc, G_IO_IN, change_codec_io_cb, (gpointer) stream1);
    } else if (use_hold) {
      //g_timeout_add_full(G_PRIORITY_DEFAULT_IDLE, 10000, hold_cb, (gpointer) stream1, NULL);
      g_io_add_watch (ioc, G_IO_IN, hold_io_cb, (gpointer) stream1);
    }
    g_main_loop_run(mainloop);
    return 0;
}

FarsightSession *setup_rtp_session()
{
    FarsightSession *session;
    g_print ("farsight_session_factory_make");
    session = farsight_session_factory_make ("rtp");

    if (!session) {
      g_error("RTP plugin not found");
      exit(1);
    }
    g_print ("protocol details:\n name: %s\n description: %s\n author: %s\n",
             farsight_plugin_get_name (session->plugin),
             farsight_plugin_get_description (session->plugin),
             farsight_plugin_get_author (session->plugin));
    g_signal_connect (G_OBJECT (session), "error",
                      G_CALLBACK (session_error), NULL);

    return session;
}

FarsightStream *setup_rtp_stream(FarsightSession *session, GstElement *pipeline)
{
    FarsightStream *stream;
    const GList *possible_codecs, *lp;
    FarsightCodec *codec;

    stream = farsight_session_create_stream (session,
            FARSIGHT_MEDIA_TYPE_AUDIO, FARSIGHT_STREAM_DIRECTION_BOTH);
    g_object_set (G_OBJECT (stream), "transmitter", "rawudp", NULL);
    g_object_set (G_OBJECT (stream), "reserved-pt-list", "96,98,99,100,102", NULL);


    g_signal_connect (G_OBJECT (stream), "error",
                      G_CALLBACK (stream_error), NULL);
    g_signal_connect (G_OBJECT (stream), "new-active-candidate-pair",
                      G_CALLBACK (new_active_candidate_pair), NULL);
    g_signal_connect (G_OBJECT (stream), "codec-changed",
                      G_CALLBACK (codec_changed), NULL);
    g_signal_connect (G_OBJECT (stream), "native-candidates-prepared",
                      G_CALLBACK (native_candidates_prepared), NULL);
    g_signal_connect (G_OBJECT (stream), "state-changed",
                      G_CALLBACK (state_changed), NULL);

    possible_codecs = farsight_stream_get_local_codecs (stream);

    for (lp = possible_codecs; lp; lp = g_list_next (lp))
    {
      codec = (FarsightCodec*) lp->data;
      g_message ("codec: %d: %s/%d found", codec->id, codec->encoding_name,
                                       codec->clock_rate);
    }

    if (pipeline != NULL) {
      farsight_stream_set_pipeline (stream, pipeline);
    }

    farsight_stream_prepare_transports(stream);

    return stream;
}
