/*
 *  FauBackup - Backup System, using a Filesystem for Storage
 *  Copyright (C) 2000-2002 Martin Waitz, Dr. Volkmar Sieh
 *  $Id: vbackup-gather.c,v 1.6.2.1 2001/05/21 13:41:09 mnwaitz Exp $
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */



#include <sys/types.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <assert.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <utime.h>

#include "faubackup.h"


int error;
char* progname;


static void
send_buf(void *buf, int len)
{
	int ret;

	while( len > 0 ) {
		ret = write( 1, buf, len );
		if( ret < 0 ) {
			fprintf( stderr, "%s: write(1, ...): %s\n",
					progname, strerror(errno));
			exit(1);
		}
		buf += ret;
		len -= ret;
	}
}

static void
send_string(char *str)
{
	unsigned int tmp;
	tmp = htonl( strlen(str) );

	send_buf( &tmp, sizeof(tmp) );
	send_buf( str, strlen(str) );
}

static void
send_inode(struct stat *st)
{
	unsigned int tmp;

	/*
	 * send contents of inode
	 */
	assert( sizeof(tmp) == 4 );
	tmp = htonl(st->st_dev); send_buf( &tmp, sizeof(tmp) );
	tmp = htonl(st->st_ino); send_buf( &tmp, sizeof(tmp) );
	tmp = st->st_mode & 07777;
	switch( st->st_mode & ~07777 ) {
	case S_IFBLK: tmp |= B_IFBLK; break;
	case S_IFCHR: tmp |= B_IFCHR; break;
	case S_IFDIR: tmp |= B_IFDIR; break;
	case S_IFIFO: tmp |= B_IFIFO; break;
	case S_IFLNK: tmp |= B_IFLNK; break;
	case S_IFREG: tmp |= B_IFREG; break;
	case S_IFSOCK: tmp |= B_IFSOCK; break;
	default: assert(0); break;
	}
	tmp = htonl(tmp);
	send_buf( &tmp, sizeof(tmp) );
	tmp = htonl(st->st_uid); send_buf( &tmp, sizeof(tmp) );
	tmp = htonl(st->st_gid); send_buf( &tmp, sizeof(tmp) );
	tmp = htonl(st->st_rdev); send_buf( &tmp, sizeof(tmp) );
	tmp = htonl(st->st_size); send_buf( &tmp, sizeof(tmp) );
/*	tmp = htonl(st->st_atime); send_buf( &tmp, sizeof(tmp) );*/
	tmp = htonl(st->st_mtime); send_buf( &tmp, sizeof(tmp) );
}


static void
gather_blk(char *path, struct stat *st)
{
	assert( S_ISBLK(st->st_mode) );

	send_inode( st );
	send_string( path );
}


static void
gather_chr(char *path, struct stat *st)
{
	assert( S_ISCHR(st->st_mode) );

	send_inode( st );
	send_string( path );
}


static void
gather_reg(char *path, struct stat *st)
{
	int fd;
	int ret;
	int pos;
	struct utimbuf times;

	assert( S_ISREG(st->st_mode) );

	fd = open( path, O_RDONLY );
	if( fd < 0) {
		fprintf( stderr, "%s: open(%s, O_RDONLY): %s\n",
				progname, path, strerror(errno));
		error++;
		return;
	}

	send_inode( st );
	send_string( path );

	for( pos = 0; pos < st->st_size; ) {
		char buf[1024*1024];
		int count;
		int len;

		count = st->st_size - pos;
		if( sizeof(buf) < count ) {
			count = sizeof(buf);
		}
		len = read(fd, buf, count);
		if( len < 0) {
			fprintf(stderr, "%s: read: %s\n",
					progname, strerror(errno));
			error++;

			/* ignore any error and send `count' bytes */
			/* otherwise we will get a protocol error */
			len = count;
		}

		send_buf( buf, len );

		pos += len;
	}

	ret = close(fd);
	if( ret < 0) {
		fprintf(stderr, "%s: close(%d): %s\n",
				progname, fd, strerror(errno));
		error++;
		return;
	}

	/* reset atime of file */
	times.actime = st->st_atime;
	times.modtime = st->st_mtime;
	ret = utime( path, &times );
	if( ret<0 && errno != EROFS ) {
		fprintf(stderr, "%s: utime(%s, ...): %s\n",
				progname, path, strerror(errno));
		return;
	}
}


static void
gather_lnk(char *path, struct stat *st)
{
	char link[MAXLEN];
	int ret;

	assert( S_ISLNK(st->st_mode) );

	ret = readlink(path, link, sizeof(link) - 1);
	if( ret < 0) {
		fprintf(stderr, "%s: readlink(%s, link, sizeof(link) - 1): %s\n",
				progname, path, strerror(errno));
		error++;
		return;
	}
	link[ret] = '\0';

	send_inode( st );
	send_string( path );
	send_string( link );
}


static void
gather_dir(char *path, struct stat *st)
{
	assert( S_ISDIR(st->st_mode) );

	send_inode( st );
	send_string( path );
}


static void
gather(char *path)
{
	int ret;
	struct stat st;

	ret = stat(".", &st);
	if( ret < 0) {
		fprintf(stderr, "%s: stat(\".\", &st): %s\n",
				progname, strerror(errno));
		error++;
		exit( 1 );
	}

	ret = lstat( path, &st );
	if( ret < 0) {
		fprintf(stderr, "%s: lstat(%s, &st): %s\n",
				progname, path, strerror(errno));
		error++;
		return;
	}

	if( S_ISBLK(st.st_mode) ) {
		gather_blk(path, &st);
	} else if( S_ISCHR(st.st_mode) ) {
		gather_chr(path, &st);
	} else if( S_ISREG(st.st_mode) ) {
		gather_reg(path, &st);
	} else if( S_ISLNK(st.st_mode) ) {
		gather_lnk(path, &st);
	} else if( S_ISDIR(st.st_mode) ) {
		gather_dir(path, &st);
	} else if( S_ISFIFO(st.st_mode)
		|| S_ISSOCK(st.st_mode) ) {
		/* skip fifo/socket */
		/* nothing to do */
	} else {
		fprintf( stderr, "%s: %s: unknown mode 0%o\n",
				progname, path, (int) st.st_mode );
		error++;
		return;
	}

}

int
main(int argc, char* argv[])
{
	char path[MAXLEN];
	int verbose;
	int c, len;

	error=0;
	progname=argv[0];
	verbose=0;

	if( argc==2 && strcmp(argv[1], "-v")==0 ) {
		verbose=1;
	} else if( argc>1 ) {
		fprintf(stderr, "Usage: %s [-v]\n", progname);
		exit( 1 );
	}

	len = 0;
	while( (c=fgetc(stdin)) != EOF ) {
		path[len++] = c;

		if( len==sizeof(path) ) {
			/* didn't fit in here -> skip */
			path[sizeof(path)-1] = '\0';
			fprintf(stderr, "path too long: %s...!\n", path);
			while( fgetc(stdin)>0 );
			len = 0;
			continue;
		}

		if( c!='\0' ) continue;
		/* one filename read, process it */

		if( verbose) {
			fprintf(stderr, "%s\n", path);
		}

		gather( path );
		len = 0;
	}

	return( error!=0 );
}
