/* polygon.c  */ 
/* COPYRIGHT (C) 2000 THE VICTORIA UNIVERSITY OF MANCHESTER and John Levon
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details. 
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA. 
 */
/* contains functions assoc. with regular polygon composition */
/*
 * $Log: polygon.c,v $
 * Revision 1.2  2000/12/06 20:56:03  moz
 * GPL stuff.
 *
 * Revision 1.1.1.1  2000/08/21 01:05:31  moz
 *
 *
 * Revision 1.1.1.1  2000/07/19 22:45:30  moz
 * CVS Import
 *
 * Revision 1.7  2000/03/07 21:46:35  moz
 * Compile fixes.
 *
 * Revision 1.6  2000/03/06 01:43:32  moz
 * Compile fixes.
 *
 * Revision 1.5  2000/02/18 21:18:39  moz
 * polyline.pic usage changed.
 *
 * Revision 1.4  2000/01/26 18:20:24  moz
 * GC temporary points correctly.
 *
 * Revision 1.3  1999/11/15 02:05:53  moz
 * Use rounded trig.
 *
 * Revision 1.2  1999/05/19 17:08:33  moz
 * 1.0 Checkin.
 *
 * Revision 1.1  1999/03/30 00:05:44  moz
 * Initial revision
 *
 */    

/* see mouse_button.c for general behaviour */  
 
#include <math.h> 
#include "include/figurine.h"  
#include "include/extern.h"
 
static long cx=0;
static long cy=0;
static long r=0;
static List points; /* this is for rubbering - freed when commiting */  
static double angle_offset = 0.0; 

static Boolean held=FALSE;

List 
alloc_points(int num)
{
	uint i; 
	List l=NULL; 
	VPoint *p; 
	 
	for (i=0; i<(uint)num; i++)
		{
		p = (VPoint *)malloc(sizeof(VPoint));
		l = add_to_list(l,i,0,(void *)p);
		};

	return l;
}
 
void kill_points(void);

void 
kill_points()
{
	List l=points;

	while (l!=NULL)
		{
		free(l->data);
		l=l->next;
		};
	delete_list(points);
	points=NULL; 
}

void unselect_polygon(void);

void
unselect_polygon()
{
	kill_points();
}


/* convert a number of sides into actual points  */  
List 
polygon_into_points(View *view, List tpoints)
{
	int i;
	List l; 
	double cum_angle=0.0;
	
	l = tpoints;

	for (i=0;i<view->sides;i++)
		{
		POINT(l)->x = r*cosround(cum_angle+angle_offset) + cx;
		POINT(l)->y = r*sinround(cum_angle+angle_offset) + cy;
		POINT(l)->derried = FALSE; 
		cum_angle += view->side_angle; 
		l = l->next; 
		};
	
	return tpoints;
}

void 
commit_polygon(View *view, long x, long y)
{
	Object *ob; 
	List l; 
	 
	if (!state.shift_down)
		{
		if (view->gridon)
			r = max(abs(GXP2D(x, view) - cx),abs(GYP2D(y, view) - cy));
		else
			r = max(abs(XP2D(x, view) - cx),abs(YP2D(y, view) - cy));
		};

	/* no constrain for polygon */ 

	ob = (Object *)malloc(sizeof(Object));
	if (ob==NULL)
		return;
	
	ob->type = POLYGON;
	ob->colour = view->colour; 
	ob->fillcolour = view->fillcolour; 
	ob->ticket = ob_ticket++;
	ob->derries = NULL; 
	ob->depth = view->doc->ob_depth--; 
	ob->ls = view->linestyle;
	ob->lw = view->linewidth;
	ob->js = view->joinstyle;
	ob->es = view->endstyle;
	if (view->fillon)
		ob->fs = view->fillstyle; 
	else
		ob->fs = NONE;
		 
	ob->dash = 5;
	ob->gap = 4;
	ob->bbox.x1 = cx - r;
	ob->bbox.y1 = cy - r;
	ob->bbox.x2 = cx + r;
	ob->bbox.y2 = cy + r;
	ob->farrow = NULL;
	ob->barrow = NULL;
	
	/* convert poly description into lines and place in object */  
	ob->ob.polyline.points = polygon_into_points(view,alloc_points(view->sides));
	ob->ob.polyline.pic = NULL;
	l = ob->ob.polyline.points;

	while (l!=NULL)
		{
		POINT(l)->x -= ob->bbox.x1;
		POINT(l)->y -= ob->bbox.y1;
		l = l->next;
		};

 	normalise_rectangle(&ob->bbox.x1, &ob->bbox.y1, &ob->bbox.x2, &ob->bbox.y2); 
	ob->bbox.x2 = cx + r;
	ob->bbox.y2 = cy + r;
	 
	recalc_polyline_bbox(ob, FALSE);  
	view->doc->o = add_object(view->doc->o, &view->doc->lo, ob);
	 
	register_undo(UNDO_PASTE,ob,view->doc); 
	send_redraw_object(view,ob); 
}

void 
polygon_button(BEvent *bev, View *view)
{
	 
	if (!view->regular)
		{
		polyline_button(bev,view);
		return;
		};

	if (((held && bev->button==Button2) || bev->button==Button3) && state.busy_drawing )
		{
		/* cancel the current polygon */ 
		if (!held) 
			XUngrabPointer(display,CurrentTime);
		
		toggle_polygon(view, bev->x, bev->y);
			 
		kill_points();
		state.busy_drawing = FALSE;
		held = FALSE;
		}
	else if (bev->button==Button1)
		{
		if (!P_IN_DOC(bev->x, bev->y, view) && state.busy_drawing)
			{
			/* ok, cancel the polygon  we're drawing  */
			state.busy_drawing = FALSE;
			 
			if (!held)
				XUngrabPointer(display,CurrentTime);
			else 
				held = FALSE;
			
			toggle_polygon(view, bev->x, bev->y);
			kill_points(); 
			}
		else
			{
			switch (bev->type)
				{
				case BUTTON_HELD:
					
					if (state.busy_drawing)
						{
						}
					else
						{
						/* start drawing */
						state.busy_drawing = TRUE;
						held = TRUE;
						points = alloc_points(view->sides);
						angle_offset = 0.0; 
						 
						if (view->gridon)
							{
							cx = GXP2D(bev->x,view);
							cy = GYP2D(bev->y,view);
							}
						else
							{
							cx = XP2D(bev->x,view);
							cy = YP2D(bev->y,view);
							};	
						
						r = 1;
							
 						toggle_polygon(view, bev->x, bev->y);  
						}; 

					break; /* BUTTON_HELD  */ 

				case BUTTON_CLICKED:

					if (held)
						{
						XGrabPointer(display, view->draw_window->win, False, ButtonPressMask | ButtonReleaseMask | PointerMotionMask | ButtonMotionMask 
										 | Button1MotionMask, GrabModeAsync, GrabModeAsync, None, None, CurrentTime); 
						}
					else
						{
						XUngrabPointer(display,CurrentTime);
						toggle_polygon(view, bev->x, bev->y);
					
						commit_polygon(view, bev->x, bev->y);

						kill_points(); 
						state.busy_drawing=FALSE;
						}; 
					 
					held = FALSE; 
					break;
					 
					  
				case BUTTON_RELEASED:
					
					toggle_polygon(view, bev->x, bev->y);
					 
					commit_polygon(view, bev->x, bev->y);
					held = FALSE;
					kill_points(); 
					state.busy_drawing=FALSE;

					break;
				};
			}; 
		}; 
}

void 
toggle_polygon(View *view, int x, int y)
{
	List l;
	VLine line; VLine *pl; 
	Boolean clipped; 

	if (!view->regular)
		{
		toggle_polyline(view,x,y);
		return;
		};

	if (state.control_down && !state.shift_down) 
		{
		/* place one side on "bottom" */ 
		angle_offset = 0.5*(PI-view->side_angle);
		}
	else if (state.shift_down)
		{
		double f; 
 		f =  (double)max(abs(XP2D(x, view) - cx),abs(YP2D(y, view) - cy)); 
		f = f/r; 
		/* 1.2 == fudge factor to give more natural feel  */  
 		angle_offset = view->side_angle*f*1.2;
		};

	if (!state.shift_down)
		{
		if (view->gridon)
			r = max(abs(GXP2D(x, view) - cx),abs(GYP2D(y, view) - cy));
		else
			r = max(abs(XP2D(x, view) - cx),abs(YP2D(y, view) - cy));

		r = max(r,1); 
		}; 
	 
	polygon_into_points(view,points);
	
	l = points; 
	while (l->next!=NULL)
		{
		line.x1 = XD2P(POINT(l)->x,view);
		line.y1 = YD2P(POINT(l)->y,view);
		line.x2 = XD2P(POINT(l->next)->x,view);
		line.y2 = YD2P(POINT(l->next)->y,view);
		
		pl = clip_line(0,0,(signed long)view->draw_window->w, (signed long)view->draw_window->h, &line, &clipped, 0);

		if (pl!=NULL)
			XDrawLine(display,view->draw_window->win, blackxorgc, pl->x1, pl->y1, pl->x2, pl->y2);
		
		l = l->next;
		};
	
	/* closed, compute final line  */ 
	
	line.x1 = XD2P(POINT(l)->x,view);
	line.y1 = YD2P(POINT(l)->y,view);
	line.x2 = XD2P(POINT(points)->x,view);
	line.y2 = YD2P(POINT(points)->y,view);

	pl = clip_line(0,0,(signed long)view->draw_window->w, (signed long)view->draw_window->h, &line, &clipped, 0);

	if (pl!=NULL)
		XDrawLine(display,view->draw_window->win, blackxorgc, pl->x1, pl->y1, pl->x2, pl->y2);
		 
}
