/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  File-Roller
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <glib.h>
#include <gdk/gdk.h>
#include <gtk/gtkmain.h>
#include <gtk/gtksignal.h>
#include <libgnome/libgnome.h>
#include "fr-process.h"

#define REFRESH_RATE 100


enum {
	START,
	DONE,
	LAST_SIGNAL
};

static GtkObjectClass *parent_class;
static guint fr_process_signals[LAST_SIGNAL] = { 0 };


static FRCommandInfo *
fr_command_info_new ()
{
	FRCommandInfo *c_info;

	c_info = g_new (FRCommandInfo, 1);
	c_info->args = NULL;
	c_info->dir = NULL;
	c_info->sticky = FALSE;

	return c_info;
}


static void
fr_command_info_free (FRCommandInfo * c_info)
{
	if (c_info == NULL)
		return;

	if (c_info->args != NULL) {
		g_list_foreach (c_info->args, (GFunc) g_free, NULL);
		g_list_free (c_info->args);
		c_info->args = NULL;
	}

	if (c_info->dir != NULL) {
		g_free (c_info->dir);
		c_info->dir = NULL;
	}

	g_free (c_info);
}


static void
fr_process_destroy (GtkObject *object)
{
	FRProcess *fr_proc;

	g_return_if_fail (object != NULL);
        g_return_if_fail (FR_IS_PROCESS (object));
  
	fr_proc = FR_PROCESS (object);

	fr_process_stop (fr_proc);
	fr_process_clear (fr_proc);
	g_ptr_array_free (fr_proc->comm, FALSE);

	if (fr_proc->row_output != NULL) {
		g_list_foreach (fr_proc->row_output, (GFunc) g_free, NULL);
		g_list_free (fr_proc->row_output);
		fr_proc->row_output = NULL;
	}

	/* Chain up */

        if (GTK_OBJECT_CLASS (parent_class)->destroy)
                (* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}


static void
fr_process_class_init (FRProcessClass *class)
{
        GtkObjectClass *object_class;

        object_class = (GtkObjectClass *) class;
        parent_class = gtk_type_class (gtk_object_get_type ());

	fr_process_signals[START] =
		gtk_signal_new ("start",
                                GTK_RUN_LAST,
                                object_class->type,
                                GTK_SIGNAL_OFFSET (FRProcessClass, start),
                                gtk_marshal_NONE__NONE,
                                GTK_TYPE_NONE, 0);
	fr_process_signals[DONE] =
		gtk_signal_new ("done",
                                GTK_RUN_LAST,
                                object_class->type,
                                GTK_SIGNAL_OFFSET (FRProcessClass, done),
                                gtk_marshal_NONE__POINTER,
                                GTK_TYPE_NONE, 1,
				GTK_TYPE_POINTER);
        gtk_object_class_add_signals (object_class, fr_process_signals, 
                                      LAST_SIGNAL);

	object_class->destroy = fr_process_destroy;

        class->start = NULL;
        class->done  = NULL;
}


static void
fr_process_init (FRProcess *fr_proc)
{
	fr_proc->term_on_stop = TRUE;

	fr_proc->comm = g_ptr_array_new ();
	fr_proc->n_comm = -1;

	fr_proc->command_pid = 0;
	fr_proc->output_fd = 0;

	fr_proc->log_timeout = -1;
	fr_proc->not_processed = 0;
	fr_proc->row_output = NULL;

	fr_proc->proc_line_func = NULL;
	fr_proc->proc_line_data = NULL;

	fr_proc->running = FALSE;
	fr_proc->stopping = FALSE;

	fr_proc->use_standard_locale = FALSE;
}


GtkType
fr_process_get_type (void)
{
        static GtkType fr_process_type = 0;

        if (! fr_process_type) {
                GtkTypeInfo fr_process_info = {
                        "FRProcess",
                        sizeof (FRProcess),
                        sizeof (FRProcessClass),
                        (GtkClassInitFunc) fr_process_class_init,
                        (GtkObjectInitFunc) fr_process_init,
                        NULL, /* reserved_1 */
                        NULL, /* reserved_2 */
                        (GtkClassInitFunc) NULL
                };

		fr_process_type = gtk_type_unique (gtk_object_get_type (), 
						   &fr_process_info);
        }

        return fr_process_type;
}


FRProcess * 
fr_process_new ()
{
	FRProcess *fr_proc;
	fr_proc = FR_PROCESS (gtk_type_new (fr_process_get_type ()));
	return fr_proc;
}


void
fr_process_begin_command (FRProcess *fr_proc, 
			  char *arg)
{
	FRCommandInfo * c_info;

	g_return_if_fail (fr_proc != NULL);

	c_info = fr_command_info_new ();
	c_info->args = g_list_prepend (NULL, g_strdup (arg));
	g_ptr_array_add (fr_proc->comm, c_info);
	fr_proc->n_comm++;
}


void
fr_process_set_working_dir (FRProcess *fr_proc, 
			    char *dir)
{
	FRCommandInfo *c_info;

	g_return_if_fail (fr_proc != NULL);

	c_info = g_ptr_array_index (fr_proc->comm, fr_proc->n_comm);
	if (c_info->dir != NULL)
		g_free (c_info->dir);
	c_info->dir = g_strdup (dir);
}


void
fr_process_set_sticky (FRProcess *fr_proc, 
		       gboolean sticky)
{
	FRCommandInfo *c_info;

	g_return_if_fail (fr_proc != NULL);

	c_info = g_ptr_array_index (fr_proc->comm, fr_proc->n_comm);
	c_info->sticky = sticky;
}


void
fr_process_add_arg (FRProcess *fr_proc, 
		    char *arg)
{
	FRCommandInfo *c_info;

	g_return_if_fail (fr_proc != NULL);

	c_info = g_ptr_array_index (fr_proc->comm, fr_proc->n_comm);
	c_info->args = g_list_prepend (c_info->args, g_strdup (arg));
}


void
fr_process_end_command (FRProcess *fr_proc)
{
	FRCommandInfo *c_info;

	g_return_if_fail (fr_proc != NULL);

	c_info = g_ptr_array_index (fr_proc->comm, fr_proc->n_comm);
	c_info->args = g_list_reverse (c_info->args);
}


void
fr_process_clear (FRProcess *fr_proc)
{
	gint i;

	g_return_if_fail (fr_proc != NULL);

	for (i = 0; i <= fr_proc->n_comm; i++) {
		FRCommandInfo *c_info;

		c_info = g_ptr_array_index (fr_proc->comm, i);
		fr_command_info_free (c_info);
		g_ptr_array_index (fr_proc->comm, i) = NULL;
	}

	for (i = 0; i <= fr_proc->n_comm; i++) 
		g_ptr_array_remove_index_fast (fr_proc->comm, 0);

	fr_proc->n_comm = -1;
}


void
fr_process_set_proc_line_func (FRProcess *fr_proc, 
			       ProcLineFunc func,
			       gpointer data)
{
	g_return_if_fail (fr_proc != NULL);
	fr_proc->proc_line_func = func;
	fr_proc->proc_line_data = data;
}


static gboolean
process_output (FRProcess *fr_proc)
{
	int n, i;
	char *line, *eol;

 again:
	n = read (fr_proc->output_fd, 
		  fr_proc->buffer + fr_proc->not_processed, 
		  BUFFER_SIZE - fr_proc->not_processed);

	if ((n < 0) && (errno == EINTR))
		goto again;

	if (n <= 0)
		return FALSE;

	fr_proc->buffer[fr_proc->not_processed + n] = 0;

	line = fr_proc->buffer;
	while ((eol = strchr (line, '\n')) != NULL) {
		*eol = 0;

		fr_proc->row_output = g_list_prepend (fr_proc->row_output, 
						      g_strdup (line));

		if (fr_proc->proc_stdout && (fr_proc->proc_line_func != NULL))
			(*fr_proc->proc_line_func) (line, 
						    fr_proc->proc_line_data);
		line = eol + 1;
	}
	
	/* shift unprocessed text to the beginning. */

	fr_proc->not_processed = strlen (line);
	for (i = 0; *line != 0; line++, i++)
		fr_proc->buffer[i] = *line;

	return TRUE;
}


static gint check_child (gpointer data);


static void
start_current_command (FRProcess *fr_proc)
{
	pid_t pid;
	int pipe_fd[2];

#ifdef DEBUG
	g_print ("%d/%d) ", fr_proc->current_command, fr_proc->n_comm);
#endif

	if (pipe (pipe_fd) < 0) {
		fr_proc->error.type = FR_PROC_ERROR_PIPE;
		gtk_signal_emit (GTK_OBJECT (fr_proc), 
				 fr_process_signals[DONE],
				 &fr_proc->error);
		return;
	}

	pid = fork ();

	if (pid < 0) {
		close (pipe_fd[0]);
		close (pipe_fd[1]);

		fr_proc->error.type = FR_PROC_ERROR_FORK;
		gtk_signal_emit (GTK_OBJECT (fr_proc), 
				 fr_process_signals[DONE],
				 &fr_proc->error);

		return;
	}

	if (pid == 0) {
		FRCommandInfo * c_info;
		GList *arg_list, *scan;
		GString *command;
		char *dir;
		char ** argv;
		int i = 0;

		c_info = g_ptr_array_index (fr_proc->comm, 
					    fr_proc->current_command);
		arg_list = c_info->args;
		dir = c_info->dir;

		if (dir != NULL) {
#ifdef DEBUG
			g_print ("cd %s\n", dir); 
#endif
			if (chdir (dir) != 0)
				_exit (FR_PROC_ERROR_CANNOT_CHDIR);
		}

		command = NULL;

		argv = g_new (char *, 4);
		argv[i++] = "/bin/sh";
		argv[i++] = "-c";

		command = g_string_new ("");
		for (scan = arg_list; scan; scan = scan->next) {
			g_string_append (command, scan->data);
			g_string_append_c (command, ' ');
		}

		argv[i++] = command->str;
		argv[i] = NULL;

#ifdef DEBUG
		{
			int j;

			g_print ("/bin/sh "); 
			for (j = 0; j < i; j++)
				g_print ("%s ", argv[j]);
			g_print ("\n"); 
		}
#endif

		close (pipe_fd[0]);
		dup2 (pipe_fd[1], STDOUT_FILENO);
		/* dup2 (pipe_fd[1], STDERR_FILENO); FIXME */

		if (fr_proc->use_standard_locale)
			putenv ("LC_ALL=C");

		execvp ("/bin/sh", argv);

		_exit (255);
	}

	close (pipe_fd[1]);

	fr_proc->command_pid = pid;
	fr_proc->output_fd = pipe_fd[0];
	fcntl (fr_proc->output_fd, F_SETFL, O_NONBLOCK);

	fr_proc->not_processed = 0;
	fr_proc->log_timeout = gtk_timeout_add (REFRESH_RATE, 
						check_child,
						fr_proc);
}


static gboolean
command_is_sticky (FRProcess *fr_proc, int i)
{
	FRCommandInfo *c_info;
	c_info = g_ptr_array_index (fr_proc->comm, i);
	return c_info->sticky;
}


static gint
check_child (gpointer data)
{
	FRProcess *fr_proc = data;
	pid_t pid;
	int status;

	/* Remove check. */
	gtk_timeout_remove (fr_proc->log_timeout);	
	fr_proc->log_timeout = -1;

	process_output (fr_proc);

	pid = waitpid (fr_proc->command_pid, &status, WNOHANG);
	if (pid != fr_proc->command_pid) {
		/* Add check again. */
		fr_proc->log_timeout = gtk_timeout_add (REFRESH_RATE, 
							check_child,
							fr_proc);
		return TRUE;
	}

	if (fr_proc->error.type != FR_PROC_ERROR_STOPPED) {
		if (WIFEXITED (status)) {
			if (WEXITSTATUS (status) == 0)
				fr_proc->error.type = FR_PROC_ERROR_NONE;
			else if (WEXITSTATUS (status) == 255) 
				fr_proc->error.type = FR_PROC_ERROR_COMMAND_NOT_FOUND;
			else if (WEXITSTATUS (status) == FR_PROC_ERROR_CANNOT_CHDIR)
				fr_proc->error.type = FR_PROC_ERROR_CANNOT_CHDIR;
			else {
				fr_proc->error.type = FR_PROC_ERROR_GENERIC;
				fr_proc->error.status = WEXITSTATUS (status);
			}
		} else
			fr_proc->error.type = FR_PROC_ERROR_EXITED_ABNORMALLY;
	}

	fr_proc->command_pid = 0;

	/* Read all pending output. */

	if (fr_proc->error.type == FR_PROC_ERROR_NONE)
		while (process_output (fr_proc)) ;

	close (fr_proc->output_fd);
	fr_proc->output_fd = 0;

	/* Execute next command. */

	if (fr_proc->error.type != FR_PROC_ERROR_NONE) {
		if (! fr_proc->sticky_only) {
			/* Remember the first error. */
			fr_proc->first_error.type = fr_proc->error.type;
			fr_proc->first_error.status = fr_proc->error.status;
		}

		fr_proc->sticky_only = TRUE;
	}

	if (fr_proc->sticky_only) {
		do {
			fr_proc->current_command++;
		} while ((fr_proc->current_command <= fr_proc->n_comm)
			 && ! command_is_sticky (fr_proc, 
						 fr_proc->current_command));
	} else
		fr_proc->current_command++;

	if (fr_proc->current_command <= fr_proc->n_comm) {
		start_current_command (fr_proc);
		return FALSE;
	}

	fr_proc->current_command = -1;

	if (fr_proc->row_output != NULL)
		fr_proc->row_output = g_list_reverse (fr_proc->row_output);

	fr_proc->running = FALSE;
	fr_proc->stopping = FALSE;

	if (fr_proc->sticky_only) {
		/* Restore the first error. */
		fr_proc->error.type = fr_proc->first_error.type;
		fr_proc->error.status = fr_proc->first_error.status;
	}

	gtk_signal_emit (GTK_OBJECT (fr_proc), 
			 fr_process_signals[DONE],
			 &fr_proc->error);

	return FALSE;
}


void
fr_process_use_standard_locale (FRProcess *fr_proc,
				gboolean use_stand_locale)
{
	g_return_if_fail (fr_proc != NULL);
	fr_proc->use_standard_locale = use_stand_locale;
}


void
fr_process_start (FRProcess *fr_proc,
		  gboolean proc_stdout)
{
	g_return_if_fail (fr_proc != NULL);
	g_return_if_fail (fr_proc->n_comm != -1);

	if (fr_proc->running)
		return;

	fr_proc->proc_stdout = proc_stdout;

	if (fr_proc->row_output != NULL) {
		g_list_foreach (fr_proc->row_output, (GFunc) g_free, NULL);
		g_list_free (fr_proc->row_output);
		fr_proc->row_output = NULL;
	}

	gtk_signal_emit (GTK_OBJECT (fr_proc), 
			 fr_process_signals[START]);

	fr_proc->sticky_only = FALSE;
	fr_proc->current_command = 0;
	fr_proc->error.type = FR_PROC_ERROR_NONE;

	fr_proc->running = TRUE;
	fr_proc->stopping = FALSE;

	start_current_command (fr_proc);
}


void
fr_process_stop (FRProcess *fr_proc)
{
	g_return_if_fail (fr_proc != NULL);

	if (! fr_proc->running)
		return;

	if (fr_proc->stopping)
		return;
	fr_proc->stopping = TRUE;

	fr_proc->error.type = FR_PROC_ERROR_STOPPED;

	if (fr_proc->term_on_stop) 
		kill (fr_proc->command_pid, SIGTERM);
	else {
		if (fr_proc->log_timeout != -1) {
			gtk_timeout_remove (fr_proc->log_timeout);
			fr_proc->log_timeout = -1;
		}
		
		fr_proc->command_pid = 0;
		close (fr_proc->output_fd);
		fr_proc->output_fd = 0;
		
		fr_proc->running = FALSE;
		
		gtk_signal_emit (GTK_OBJECT (fr_proc), 
				 fr_process_signals[DONE],
				 &fr_proc->error);
	}
}
