/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  File-Roller
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */


/* Based upon gide-document.c from the ximian-utils package. 
 * The original copyright note follows :
 *
 *
 * gIDE
 * Copyright (C) 1998-2000 Steffen Kern
 *               2000-2001 Dave Camp <dave@ximian.com>
 *               2001 Ximian, Inc.
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <config.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <gnome.h>
#include <libgnomeui/gnome-window-icon.h>
#include <liboaf/liboaf.h>
#include <bonobo.h>
#include <libgnomevfs/gnome-vfs-mime.h>
#include <libgnomevfs/gnome-vfs-mime-handlers.h>


typedef struct {
        char *name;
        char *iid;
	gboolean zoomable;
	gint  menu_index;
} AvailableComponent;


typedef struct {
	GtkWidget *          win;
	BonoboUIContainer *  uic;
	BonoboUIComponent *  ui_comp;

	char *               file_name;
	const char *         mime_type;

	GList *              component_list;
	AvailableComponent * current_component;

	GtkWidget *          comp_container;
	GtkWidget *          comp_widget;
} Document;


Document *__Doc;


static void
destroy_available_components (GList *components)
{
        GList *i;
        for (i  = components; i != NULL; i = i->next) {
                AvailableComponent *c = i->data;
                g_free (c->name);
                g_free (c->iid);
                g_free (c);
        }
        g_list_free (components);
}


static void
app_destroy_cb (GtkWidget *app, Document *doc)
{
	if (doc->file_name != NULL)
		g_free (doc->file_name);
	destroy_available_components (doc->component_list);
	bonobo_object_unref (BONOBO_OBJECT (doc->ui_comp));
	bonobo_object_unref (BONOBO_OBJECT (doc->uic));
	g_free (doc);

	gtk_main_quit ();
}


static int
app_delete_cb (GtkWidget *widget, GdkEvent *event, gpointer dummy)
{
	gtk_widget_destroy (GTK_WIDGET (widget));
	return FALSE;
}


static void
load_document (Document *doc)
{
	BonoboObjectClient *o_client;
	Bonobo_PersistStream persist_stream;
	Bonobo_PersistFile persist_file;
	BonoboStream *stream;
	CORBA_Environment ev;

	if (!doc->file_name) 
		return;

	CORBA_exception_init (&ev);

	/* Try IDL:Bonobo/PersistFile:1.0 */

	o_client = bonobo_widget_get_server (BONOBO_WIDGET (doc->comp_widget));
	persist_file = bonobo_object_client_query_interface (o_client, "IDL:Bonobo/PersistFile:1.0", NULL);

	if (persist_file != CORBA_OBJECT_NIL) {
		Bonobo_PersistFile_load (persist_file, doc->file_name, &ev);
		bonobo_object_release_unref (persist_file, &ev);
		CORBA_exception_free (&ev);

		return;
	}

	/* Try IDL:Bonobo/PersistStream:1.0 */

	stream = bonobo_stream_open ("fs", 
				     doc->file_name, 
				     Bonobo_Storage_READ, 
				     O_RDONLY);
	if (!stream) {
		char *error_msg;
		error_msg = g_strdup_printf (_("Could not open file %s"), 
					     doc->file_name);
		gnome_warning_dialog (error_msg);
		g_free (error_msg);

		CORBA_exception_free (&ev);

		return;
	}

	o_client = bonobo_widget_get_server (BONOBO_WIDGET (doc->comp_widget));
	persist_stream = bonobo_object_client_query_interface (o_client, "IDL:Bonobo/PersistStream:1.0", &ev);

	Bonobo_PersistStream_load (persist_stream,
				   (Bonobo_Stream) BONOBO_OBJREF (stream),
				   "", &ev);

	bonobo_object_release_unref (persist_stream, &ev);
	bonobo_object_unref (BONOBO_OBJECT (stream));

        CORBA_exception_free (&ev);
}


static GSList *
get_lang_list (void)
{
        GSList *retval;
        char *lang;
        char * equal_char;

        retval = NULL;

        lang = g_getenv ("LANGUAGE");

        if (!lang) 
                lang = g_getenv ("LANG");

        if (lang) {
                equal_char = strchr (lang, '=');
                if (equal_char != NULL) 
                        lang = equal_char + 1;

                retval = g_slist_prepend (retval, lang);
        }
        
        return retval;
}


static gboolean
has_zoomable_interface (char *iid)
{
        CORBA_Environment ev;
        OAF_ServerInfoList *oaf_result;
        CORBA_char *query;
	gboolean is_zoomable;

	CORBA_exception_init (&ev);
        query = g_strdup_printf ("(iid == '%s') AND repo_ids.has ('IDL:Bonobo/Zoomable:1.0')", iid);
        
        oaf_result = oaf_query (query, NULL, &ev);
        g_free (query);

	is_zoomable = ((ev._major == CORBA_NO_EXCEPTION)
		       && (oaf_result != NULL) 
		       && (oaf_result->_length >= 1)); 

	if (oaf_result != NULL) 
                CORBA_free (oaf_result);
        CORBA_exception_free (&ev);

	return is_zoomable;
}


static GList *
get_available_components (const char *mime_type)
{
        CORBA_Environment ev;
        OAF_ServerInfoList *oaf_result;
        CORBA_char *query;
        GList *ret = NULL;
        char *generic;
        char *p;
        GSList *langs = get_lang_list ();

        generic = g_strdup (mime_type);
        p = strchr (generic, '/');
        g_assert (p);
        *(++p) = '*';
        *(++p) = 0;

	CORBA_exception_init (&ev);

        query = g_strdup_printf ("repo_ids.has ('IDL:Bonobo/Control:1.0') AND (bonobo:supported_mime_types.has ('%s') OR bonobo:supported_mime_types.has ('%s')) AND (repo_ids.has ('IDL:Bonobo/PersistFile:1.0') OR repo_ids.has ('IDL:Bonobo/PersistStream:1.0'))", mime_type, generic);
        
        oaf_result = oaf_query (query, NULL, &ev);

        g_free (generic);              
        g_free (query);

	/* if no component was found use the text viewer component. */
        if ((ev._major != CORBA_NO_EXCEPTION)
	    || (oaf_result == NULL) 
	    || (oaf_result->_length < 1)) {
		if (oaf_result != NULL) 
			CORBA_free (oaf_result);

		query = g_strdup ("repo_ids.has ('IDL:Bonobo/Control:1.0') AND bonobo:supported_mime_types.has ('text/*') AND (repo_ids.has ('IDL:Bonobo/PersistFile:1.0') OR repo_ids.has ('IDL:Bonobo/PersistStream:1.0'))");
		oaf_result = oaf_query (query, NULL, &ev);
		g_free (query);
	}

	if ((ev._major == CORBA_NO_EXCEPTION)
	    && (oaf_result != NULL) 
	    && (oaf_result->_length >= 1)) {
                int i;

                for (i = 0; i < oaf_result->_length; i++) {
			AvailableComponent *v = g_new (AvailableComponent, 1);
			OAF_ServerInfo *s = &oaf_result->_buffer[i];
			v->name = g_strdup (oaf_server_info_prop_lookup (s, "name", langs));
			v->iid = g_strdup (s->iid);
			v->zoomable = has_zoomable_interface (v->iid);
			v->menu_index = oaf_result->_length - i - 1;

			ret = g_list_prepend (ret, v);
                }
        } 
        
        if (oaf_result != NULL) 
                CORBA_free (oaf_result);
	g_slist_free (langs);

        CORBA_exception_free (&ev);

	return ret;
}


static AvailableComponent *
choose_default_component (Document *doc)
{
	GList *available_components = doc->component_list;
	const char *mime_type = doc->mime_type;
        OAF_ServerInfo *server_info;
        AvailableComponent *ret = NULL;
        
        g_return_val_if_fail (available_components != NULL, NULL);

        /* First try to match the gnome-vfs default component */
     
	server_info = gnome_vfs_mime_get_default_component (mime_type);
        if (server_info) {
                GList *i;
                for (i = available_components; i != NULL; i = i->next) {
                        AvailableComponent *component = i->data;

			if ((component->iid == NULL) 
			    || (server_info->iid == NULL))
				continue;

                        if (strcmp (component->iid, server_info->iid) == 0) {
                                ret = component;
                                break;
                        }
                }

		CORBA_free (server_info);
        }

	/* Then just pick the first one */
        if (!ret) 
                ret = available_components->data;

        return ret;
}


static void
unload_component (Document *doc)
{
	CORBA_Environment ev;

	CORBA_exception_init (&ev);

	if (doc->current_component->zoomable)
		bonobo_ui_component_rm (doc->ui_comp, 
					"/menu/View", 
					NULL);

	Bonobo_Control_activate (bonobo_widget_get_objref (BONOBO_WIDGET (doc->comp_widget)), FALSE, &ev);

        gtk_container_remove (GTK_CONTAINER (doc->comp_container), 
                              doc->comp_widget);

        doc->comp_widget = NULL;
        doc->current_component = NULL;

        CORBA_exception_free (&ev);
}


const char zoom_ui [] =
"<submenu name=\"View\" _label=\"_View\">\n"
"<placeholder name=\"Zoom Items Placeholder\" delimit=\"none\">\n"
"	<menuitem name=\"Zoom In\"\n"
"	  _label=\"Zoom _In\" accel=\"*Control*plus\"\n"
"	  verb=\"Zoom In\"/>\n"
"	<menuitem name=\"Zoom Out\"\n"
"	  _label=\"Zoom _Out\" accel=\"*Control*minus\"\n"
"	  verb=\"Zoom Out\"/>\n"
"	<menuitem name=\"Zoom Normal\"\n"
"	  _label=\"_Normal Size\" accel=\"1\"\n"
"	  verb=\"Zoom Normal\"/>\n"
"	<menuitem name=\"Zoom Fit\"\n"
"	  _label=\"Zoom to _Fit\" accel=\"x\"\n"
"	  verb=\"Zoom Fit\"/>\n"
"	<separator/>\n"
"</placeholder>\n"
"</submenu>\n";


static void
activate_component (Document *doc, AvailableComponent *comp)
{
	if (doc->current_component) 
                unload_component (doc);

	doc->comp_widget = bonobo_widget_new_control (comp->iid, BONOBO_OBJREF (doc->uic));

	if (doc->comp_widget) { 
		CORBA_Environment ev;

		CORBA_exception_init (&ev);

		gtk_container_add (GTK_CONTAINER(doc->comp_container),
				   doc->comp_widget);
		gtk_widget_show (doc->comp_widget);
		doc->current_component = comp;

		/* Add zoom items if the control is zoomable. */

		if (comp->zoomable) 
			bonobo_ui_component_set_translate (doc->ui_comp, 
							   "/menu", 
							   zoom_ui, 
							   &ev);
		load_document (doc);
		Bonobo_Control_activate (bonobo_widget_get_objref (BONOBO_WIDGET (doc->comp_widget)), TRUE, &ev);

		CORBA_exception_free (&ev);
	}	
}


static void
view_activated_cb (GtkWidget *widget, Document *doc)
{
        AvailableComponent *comp;

	comp = gtk_object_get_data (GTK_OBJECT (widget), "Component");
        if (comp) 
		activate_component (doc, comp);
}


static GtkWidget *
build_component_menu (Document *doc)
{
        GtkWidget *menu;
        GList *i;
        GtkWidget *item;
	GList *available_components = doc->component_list;

        menu = gtk_menu_new ();
        for (i = available_components; i != NULL; i = i->next) {
                AvailableComponent *v = i->data;

                item = gtk_menu_item_new_with_label (v->name);
                gtk_object_set_data (GTK_OBJECT (item), "Component", v);

                gtk_signal_connect (GTK_OBJECT (item), "activate", 
                                    GTK_SIGNAL_FUNC (view_activated_cb), 
                                    doc);
		gtk_widget_show (item);

                gtk_menu_append (GTK_MENU (menu), item);
        }

	return menu;
}


void
verb_exit (BonoboUIComponent *component, 
	   gpointer callback_data, 
	   const char *cname)
{
        Document *doc = callback_data;
	app_destroy_cb (NULL, doc);
}


void
verb_zoom_in (BonoboUIComponent *component, 
	      gpointer callback_data, 
	      const char *cname)
{
        Document *doc = callback_data;
	BonoboObjectClient *o_client;
        Bonobo_Zoomable zoomable;
        CORBA_Environment ev;

        CORBA_exception_init (&ev);

        o_client = bonobo_widget_get_server (BONOBO_WIDGET (doc->comp_widget));
        zoomable = bonobo_object_client_query_interface (o_client, "IDL:Bonobo/Zoomable:1.0", &ev);

        if (zoomable == CORBA_OBJECT_NIL)
                return;

        Bonobo_Zoomable_zoomIn (zoomable, &ev);

        bonobo_object_release_unref (zoomable, &ev);
        CORBA_exception_free (&ev);

}


void
verb_zoom_out (BonoboUIComponent *component, 
	       gpointer callback_data, 
	       const char *cname)
{
       Document *doc = callback_data;
	BonoboObjectClient *o_client;
        Bonobo_Zoomable zoomable;
        CORBA_Environment ev;

        CORBA_exception_init (&ev);

        o_client = bonobo_widget_get_server (BONOBO_WIDGET (doc->comp_widget));
        zoomable = bonobo_object_client_query_interface (o_client, "IDL:Bonobo/Zoomable:1.0", &ev);

        if (zoomable == CORBA_OBJECT_NIL)
                return;

        Bonobo_Zoomable_zoomOut (zoomable, &ev);

        bonobo_object_release_unref (zoomable, &ev);
        CORBA_exception_free (&ev);
}


void
verb_zoom_normal (BonoboUIComponent *component, 
		  gpointer callback_data, 
		  const char *cname)
{
       Document *doc = callback_data;
	BonoboObjectClient *o_client;
        Bonobo_Zoomable zoomable;
        CORBA_Environment ev;

        CORBA_exception_init (&ev);

        o_client = bonobo_widget_get_server (BONOBO_WIDGET (doc->comp_widget));
        zoomable = bonobo_object_client_query_interface (o_client, "IDL:Bonobo/Zoomable:1.0", &ev);

        if (zoomable == CORBA_OBJECT_NIL)
                return;

        Bonobo_Zoomable_zoomDefault (zoomable, &ev);

        bonobo_object_release_unref (zoomable, &ev);
        CORBA_exception_free (&ev);
}


void
verb_zoom_fit (BonoboUIComponent *component, 
	       gpointer callback_data, 
	       const char *cname)
{
       Document *doc = callback_data;
	BonoboObjectClient *o_client;
        Bonobo_Zoomable zoomable;
        CORBA_Environment ev;

        CORBA_exception_init (&ev);

        o_client = bonobo_widget_get_server (BONOBO_WIDGET (doc->comp_widget));
        zoomable = bonobo_object_client_query_interface (o_client, "IDL:Bonobo/Zoomable:1.0", &ev);

        if (zoomable == CORBA_OBJECT_NIL)
                return;

        Bonobo_Zoomable_zoomFit (zoomable, &ev);

        bonobo_object_release_unref (zoomable, &ev);
        CORBA_exception_free (&ev);
}


static BonoboUIVerb verbs [] = {
        BONOBO_UI_VERB ("FileExit", verb_exit),
        BONOBO_UI_VERB ("Zoom In", verb_zoom_in),
        BONOBO_UI_VERB ("Zoom Out", verb_zoom_out),
        BONOBO_UI_VERB ("Zoom Normal", verb_zoom_normal),
        BONOBO_UI_VERB ("Zoom Fit", verb_zoom_fit),
        BONOBO_UI_VERB_END
};


const char my_ui1 [] =
"<status/>\n";


const char my_ui2 [] =
"<commands>\n"
"	<cmd name=\"FileExit\" _label=\"E_xit\" _tip=\"Exit the program\"\n"
"	  pixtype=\"stock\" pixname=\"Quit\" accel=\"*Control*q\"/>\n"
"	<cmd name=\"Zoom In\" _label=\"Zoom In\"\n"
"         _tip=\"Show the contents in more detail\"/>\n"
"       <cmd name=\"Zoom Out\" _label=\"Zoom Out\"\n"
"         _tip=\"Show the contents in less detail\"/>\n"
"       <cmd name=\"Zoom Normal\" _label=\"Normal Size\"\n"
"         _tip=\"Show the contents at the normal size\"/>\n"
"       <cmd name=\"Zoom Fit\" _label=\"Zoom to Fit\"\n"
"         _tip=\"Show the contents at maximum size\"/>\n"
"</commands>\n";


const char my_ui3 [] =
"<menu>\n"
"	<submenu name=\"File\" _label=\"_File\">\n"
"		<placeholder name=\"FileOperations\" delimit=\"bottom\" />"
"		<placeholder name=\"PrintOperations\" delimit=\"bottom\" />"
"       	<menuitem name=\"FileExit\" verb=\"\" accel=\"*Control*q\"/>\n"
"	</submenu>\n"
"</menu>\n";



static void 
set_default_clicked_cb (GtkWidget *btn, Document *document)
{
        gnome_vfs_mime_set_default_component (document->mime_type, document->current_component->iid);
}


static void
create_container (char *file_name)
{
	GtkWidget *box;
	BonoboUIContainer *uic;
	GtkWindow *window;
	GtkWidget *menu;
	GtkWidget *opt_menu;
	GtkWidget *frame;
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *btn;
	GtkWidget *separator;
	Document *doc; 
	AvailableComponent *default_comp;
	CORBA_Environment ev;
	gchar *pixmap_file;
	gchar *title;

	__Doc = doc = g_new (Document, 1);
	doc->file_name = g_strdup (file_name);
	doc->mime_type = gnome_vfs_get_file_mime_type (file_name, NULL, FALSE);
	doc->component_list = get_available_components (doc->mime_type);
	doc->current_component = NULL;

	if (doc->component_list == NULL) {
		/* no component viewer for this mime type. */

		char *msg = g_strdup_printf ("Could not find a viewer for %s", 
					     doc->mime_type);
                GtkWidget *dlg = gnome_error_dialog (msg);
                gnome_dialog_run_and_close (GNOME_DIALOG (dlg));
		g_free (msg);
		g_free (doc);

                return;
	}

	title = g_strconcat (_("Document Viewer"),
			     " - ",
			     g_basename (file_name),
			     NULL);
	doc->win = bonobo_window_new ("fr-document-viewer", title);
	g_free (title);
	window = GTK_WINDOW (doc->win);
	gtk_window_set_policy (window, TRUE, TRUE, TRUE);

	pixmap_file = gnome_pixmap_file ("file-roller.png");
        gnome_window_icon_set_from_file (window, pixmap_file);
        g_free (pixmap_file);

	doc->uic = uic = bonobo_ui_container_new ();
	bonobo_ui_container_set_win (uic, BONOBO_WINDOW (doc->win));

	/**/
	doc->ui_comp = bonobo_ui_component_new ("F_R_DocViewer");
	bonobo_ui_component_set_container (doc->ui_comp, 
					   BONOBO_OBJREF (doc->uic));

	CORBA_exception_init (&ev);

	bonobo_ui_component_freeze (doc->ui_comp, &ev);
	bonobo_ui_component_set_translate (doc->ui_comp, "/", my_ui1, NULL);
	bonobo_ui_component_set_translate (doc->ui_comp, "/", my_ui2, NULL);
	bonobo_ui_component_set_translate (doc->ui_comp, "/", my_ui3, NULL);
        bonobo_ui_component_add_verb_list_with_data (doc->ui_comp, verbs, doc);
	bonobo_ui_component_thaw (doc->ui_comp, &ev);

        CORBA_exception_free (&ev);

	gtk_signal_connect (GTK_OBJECT (window), "delete_event",
			    GTK_SIGNAL_FUNC (app_delete_cb), 
			    NULL);

	gtk_signal_connect (GTK_OBJECT (window), "destroy",
			    GTK_SIGNAL_FUNC (app_destroy_cb), 
			    doc);

	box = gtk_vbox_new (FALSE, 5);
	gtk_container_set_border_width (GTK_CONTAINER (box), 5);
	bonobo_window_set_contents (BONOBO_WINDOW (doc->win), box);

	doc->comp_container = frame = gtk_frame_new (NULL);
	gtk_widget_set_usize (frame, 460, 400);
	gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_NONE);
	gtk_box_pack_start (GTK_BOX (box), frame, TRUE, TRUE, 0);

	separator = gtk_hseparator_new ();
	gtk_box_pack_start (GTK_BOX (box), separator, FALSE, FALSE, 0);

	/**/

	hbox = gtk_hbox_new (FALSE, 0);	
	gtk_box_pack_start (GTK_BOX (box), hbox, FALSE, FALSE, 0);

	label = gtk_label_new (_("View with:"));
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

	opt_menu = gtk_option_menu_new ();
	menu = build_component_menu (doc);
        gtk_option_menu_set_menu (GTK_OPTION_MENU (opt_menu), menu);
	gtk_widget_show (opt_menu);
	gtk_box_pack_start (GTK_BOX (hbox), opt_menu, FALSE, FALSE, 5);

	btn = gtk_button_new_with_label (_("Make the default viewer"));
        gtk_box_pack_end (GTK_BOX (hbox), btn, FALSE, FALSE, 0);
        gtk_signal_connect (GTK_OBJECT (btn), "clicked", 
                            GTK_SIGNAL_FUNC (set_default_clicked_cb), 
                            doc);

	gtk_widget_show_all (GTK_WIDGET (window));

	/* activate default component. */

	default_comp = choose_default_component (doc);
	gtk_option_menu_set_history (GTK_OPTION_MENU (opt_menu),
				     default_comp->menu_index);
	activate_component (doc, default_comp);
}


static void
kill_signal (gint sig)
{
        app_destroy_cb (NULL, __Doc);
}


int
main (int argc, char **argv)
{
	char *file;
	CORBA_ORB orb;

	bindtextdomain (PACKAGE, LOCALEDIR);  
        textdomain (PACKAGE);

        gnome_init_with_popt_table ("FR-Document-Viewer", "1.0",
				    argc, argv,
				    oaf_popt_options, 0, NULL); 
	orb = oaf_init (argc, argv);

	if (bonobo_init (orb, NULL, NULL) == FALSE)
		g_error ("Could not initialize Bonobo");

	file = argv[1];
	if ((file == NULL) || !g_file_exists (file)) {
                char *msg;
                GtkWidget *dlg;

		if (file != NULL)
			msg = g_strdup_printf (_("Could not find file %s"), 
					       file);
		else
			msg = g_strdup_printf (_("You must specify a file"));
		dlg = gnome_error_dialog (msg);
                gnome_dialog_run_and_close (GNOME_DIALOG (dlg));
                return 1;
        }

	bonobo_activate ();
	create_container (file);

	signal (SIGINT, kill_signal);
        signal (SIGTERM, kill_signal);

	gtk_main ();

	return 0;
}
