(* $Id: metacache_unix.ml,v 1.4 2001/10/12 20:17:00 gerd Exp $
 * ----------------------------------------------------------------------
 *
 *)


open Metacache;;
open Split;;

(* The following requires that the Unix library is linked in *)

let list_dir d =
  let rec list fd =
    try
      let f = Unix.readdir fd in
      if f = Filename.current_dir_name || f = Filename.parent_dir_name then
	list fd
      else
	f :: list fd
    with
	End_of_file -> []
  in
  let fd = Unix.opendir d in
  try
    let l = list fd in
    Unix.closedir fd;
    l
  with
      exc -> Unix.closedir fd; raise exc
;;


let init() =
  (* Ensures that the cache is completely filled with every package
   * of the system
   *)
  let list_directory d =
    try
      list_dir d
    with
	Unix.Unix_error (code,_,_) ->
	  prerr_endline ("WARNING: cannot read directory " ^ d ^ ": " ^
			 Unix.error_message code);
	  []
  in

  let rec run_ocamlpath path =
    match path with
      [] -> ()
    | dir :: path' ->
	let files = list_directory dir in
	List.iter
	  (fun f ->
	     (* If f/META exists: Add package f *)
	     let package_dir = Filename.concat dir f in
	     let meta_file_1 = Filename.concat package_dir "META" in
	     if Sys.file_exists meta_file_1 then begin
	       try
		 let entry = get_entry f package_dir meta_file_1 in
		 Metastore.add store entry;   (* or Inconsistent_ordering *)
	       with
		   Topo.Inconsistent_ordering ->
		     ()
		 | Failure s ->
		     prerr_endline ("WARNING: " ^ s)
	     end
	     else
	       (* If f is META.pkgname: Add package pkgname *)
	       if String.length f >= 6 && String.sub f 0 5 = "META." then begin
		 let name = String.sub f 5 (String.length f - 5) in
		 try
		   let meta_file_2 = package_dir in
		   let entry = get_entry name dir meta_file_2 in
		   Metastore.add store entry;   (* or Inconsistent_ordering *)
		 with
		     Topo.Inconsistent_ordering ->
		       ()
		   | Failure s ->
		       prerr_endline ("WARNING: " ^ s)
	       end;
	  )
	  files;
	run_ocamlpath path'
  in
  let add_relations() =
    let rels = ref [] in
    Metastore.iter_up
      (fun entry ->
	 let r =
	   try Metascanner.lookup "requires" [] entry.meta_file
	   with Not_found -> ""
	 in
	 let ancestors = Split.in_words r in
	 List.iter
	   (fun p ->
	      rels := (p, entry.package_name) :: !rels
	   )
	   ancestors
      )
      store;
    List.iter
      (fun (p, p') ->
	 try
	   Metastore.let_le store p p'
	 with
	     Not_found ->
               prerr_endline ("WARNING: package " ^ p ^ 
			      " requires package " ^ p' ^ ": not found")
      )
      !rels
  in

  run_ocamlpath !ocamlpath;
  add_relations()
;;


let list_packages() =
  init();

  let l = ref [] in

  Metastore.iter_up
    (fun m ->
      l := m.package_name :: !l)
    store;

  !l
;;


let users pl =
  (* Get the descendants of pl *)

  init();

  let l = ref [] in

  Metastore.iter_down_at
    (fun m ->
      l := m.package_name :: !l)
    store
    pl;

  !l
;;


let module_conflict_report incpath =
  (* Find any *.cmi files occurring twice in (incpath @ package directories).
   *)
  let dir_of_module = Hashtbl.create 100 in
  let dirs = ref [] in

  let examine_dir d = 
    (* If d ends with a slash: remove it *)
    let d' = norm_dir d in

    (* Is d' new? *)
    if not (List.mem d' !dirs) then begin
      dirs := d' :: !dirs;
      (* Yes: Get all files ending in .cmi *)
      try
	let d_all = list_dir d' in   (* or Unix_error *)
	let d_cmi = List.filter 
		      (fun n -> Filename.check_suffix n ".cmi") 
		      d_all in
	(* Add the modules to dir_of_module: *)
	List.iter
	  (fun m ->
	     try
	       let entry = Hashtbl.find dir_of_module m in (* or Not_found *)
	       entry := d' :: !entry
	     with
		 Not_found ->
		   Hashtbl.add dir_of_module m (ref [d'])
	  )
	  d_cmi
      with
	Unix.Unix_error (code,_,_) ->
	  prerr_endline ("WARNING: cannot read directory " ^ d' ^ ": " ^
			 Unix.error_message code);
    end
  in

  let print_report() =
    Hashtbl.iter
      (fun m dlist ->
	 match !dlist with
	     []
	   | [_] ->
	       ()
	   | _ ->
	       Printf.eprintf "WARNING: Interface %s occurs in several directories: %s\n"
		 m
		 (String.concat ", " !dlist)
      )
      dir_of_module
  in

  List.iter examine_dir incpath;
  Metastore.iter_up 
    (fun pkg -> examine_dir pkg.package_dir)
    store;

  print_report();
  flush stderr
;;


(* ======================================================================
 * History:
 * 
 * $Log: metacache_unix.ml,v $
 * Revision 1.4  2001/10/12 20:17:00  gerd
 * 	When directory names are compared, they are normalized before that.
 *
 * Revision 1.3  2001/07/24 20:00:59  gerd
 * 	Bugfix: init() initializes now the relations, too. Because of
 * this, users() works now
 *
 * Revision 1.2  2001/03/03 19:28:34  gerd
 * 	Added conflict reports.
 *
 * Revision 1.1  2001/02/24 20:21:58  gerd
 * 	Initial revision.
 *
 * 
 *)
